/*

    File: xfs.c

    Copyright (C) 2004 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 59
    Temple Place - Suite 330, Boston MA 02111-1307, USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include "types.h"
#include "common.h"
#include "xfs.h"
#include "fnctdsk.h"
#include "intrf.h"
static int set_xfs_info(const t_param_disk *disk_car, const struct xfs_sb *sb,t_partition *partition, const int debug, const int dump_ind);
static int test_xfs(const t_param_disk *disk_car, const struct xfs_sb *sb,t_partition *partition,const int debug, const int dump_ind);

int check_xfs(t_param_disk *disk_car,t_partition *partition,const int debug)
{
  unsigned char buffer[8*DEFAULT_SECTOR_SIZE];
  if(disk_car->read(disk_car,sizeof(buffer), &buffer, partition->part_offset)!=0)
  { return 1; }
  if(test_xfs(disk_car,(struct xfs_sb*)&buffer,partition,debug,0)!=0)
    return 1;
  set_xfs_info(disk_car,(struct xfs_sb*)&buffer,partition,debug,0);
  return 0;
}

static int test_xfs(const t_param_disk *disk_car, const struct xfs_sb *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if (sb->sb_magicnum==be32(XFS_SB_MAGIC))
  {
    switch(be16(sb->sb_versionnum) & XFS_SB_VERSION_NUMBITS)
    {
      case XFS_SB_VERSION_1:
	partition->upart_type = UP_XFS;
	break;
      case XFS_SB_VERSION_2:
	partition->upart_type = UP_XFS2;
	break;
      case XFS_SB_VERSION_3:
	partition->upart_type = UP_XFS3;
	break;
      case XFS_SB_VERSION_4:
	partition->upart_type = UP_XFS4;
	break;
      default:
	ecrit_rapport("Unknown XFS version %x\n",be16(sb->sb_versionnum)& XFS_SB_VERSION_NUMBITS);
	partition->upart_type = UP_XFS4;
	break;
    }
  }
  else
    return 1;
  if(debug!=0)
    ecrit_rapport("\nXFS Marker at %u/%u/%u\n", offset2cylinder(disk_car,partition->part_offset),offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset));
  return 0;
}

int recover_xfs(t_param_disk *disk_car, const struct xfs_sb *sb,t_partition *partition,const int debug, const int dump_ind)
{
  if(test_xfs(disk_car,sb,partition,debug,dump_ind)!=0)
    return 1;
  if(debug!=0 || dump_ind!=0)
  {
    ecrit_rapport("\nrecover_xfs\n");
    if(dump_ind!=0)
    {
      dump(stdscr,sb,DEFAULT_SECTOR_SIZE);
    }
  }

  partition->part_size = (uint64_t)be64(sb->sb_dblocks) * be32(sb->sb_blocksize);
  partition->part_type = (unsigned char)0x83;
  set_xfs_info(disk_car,sb,partition,debug,dump_ind);
  return 0;
}

static int set_xfs_info(const t_param_disk *disk_car,const struct xfs_sb *sb,t_partition *partition, const int debug, const int dump_ind)
{
  partition->info[0]='\0';
  switch(partition->upart_type)
  {
    case UP_XFS:
      strncpy(partition->info,"XFS <=6.1",sizeof(partition->info));
      break;
    case UP_XFS2:
      strncpy(partition->info,"XFS 6.2 - attributes",sizeof(partition->info));
      break;
    case UP_XFS3:
      strncpy(partition->info,"XFS 6.2 - new inode version",sizeof(partition->info));
    case UP_XFS4:
	strncpy(partition->info,"XFS 6.2+ - bitmap version",sizeof(partition->info));
      break;
    default:
      return 1;
  }
  set_part_name(partition,sb->sb_fname,12);
  return 0;
}
