/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file IsoLabel.h
    \brief Definition of the Template class IsoLabel.
    
    Magics Team - ECMWF 2004
    
    Started: Tue 9-Mar-2004
    
    Changes:
    
*/

#ifndef IsoLabel_H
#define IsoLabel_H

#include "magics.h"
#include "Factory.h"
#include "Translator.h"
#include "LevelSelection.h"


#include "IsoLabelAttributes.h"

namespace magics {

class IsoLabel: public map<double, double>, public IsoLabelAttributes {

public:
   IsoLabel();
   virtual ~IsoLabel();

   virtual IsoLabel* clone() const {
     IsoLabel* object = new IsoLabel();
     object->copy(*this);
     return object;
   }

   virtual void operator()(Polyline&, double) const;

   virtual void prepare(const LevelSelection& levels, const string& colour)
   {
        int label = 0;
        if ( colour_ == "contour_line_colour" ) colour_ = colour;

        const LevelSelection::const_iterator levelsEnd = levels.end();
        for (LevelSelection::const_iterator level = levels.begin(); level != levelsEnd; ++level) {
            if (label++ == 0) {
                (*this)[*level] = *level;
                Log::dev() << "label--->" << *level << "\n";
            }
            if (label == frequency_) label = 0;
        }
    }

protected:
    //! Method to print string about this class on to a stream of type ostream (virtual).
    virtual void print(ostream&) const;

    typedef string (IsoLabel::*Method)(double) const;
    std::map<string, Method> methods_;
    string number(double) const;
    string text(double) const;
    string both(double) const;

private:
    //! Copy constructor - No copy allowed
    IsoLabel(const IsoLabel&);
    //! Overloaded << operator to copy - No copy allowed
    IsoLabel& operator=(const IsoLabel&);

// -- Friends
    //! Overloaded << operator to call print().
    friend ostream& operator<<(ostream& s,const IsoLabel& p)
        { p.print(s); return s; }
};


class NoIsoLabel: public IsoLabel {

public:
	NoIsoLabel() {}
	virtual ~NoIsoLabel() {}
	virtual IsoLabel* clone() const {
		IsoLabel* object = new NoIsoLabel();	
		return object;
	}
	virtual void operator()(Polyline&, double) const; 
	virtual void prepare(const LevelSelection&) {}
}; 


template<>
class Translator<string, IsoLabel> { 
public:
    IsoLabel* operator()(const string& val ) {
       return SimpleObjectMaker<IsoLabel>::create(val);
    }
    IsoLabel* magics(const string& param)
    {
        IsoLabel* object;
		ParameterManager::update(param, object);
		return object;
    }
};
} // namespace magics
#endif
