/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file RootSceneNode.h
    \brief Definition of the Template class RootSceneNode.
    
    Magics Team - ECMWF 2007
    
    Started: Thu 1-Mar-2007
    
    Changes:
    
*/

#ifndef RootSceneNode_H
#define RootSceneNode_H

#include "magics.h"

#include "BasicSceneObject.h"
#include "FortranRootSceneNodeAttributes.h"
#include "XmlRootNodeAttributes.h"
#include "WrepRootNodeAttributes.h"

namespace magics {


class RootSceneNode;

class RootScenePage: public BasicSceneNode 
{
public:
	RootScenePage();
	~RootScenePage();

	BasicSceneNode* clone() 
	{
	  return new  RootScenePage();
	}

	BasicSceneNode* newNode(BasicPositionalObject*);
	void root(RootSceneNode* root) { root_ = root; }

protected:
	RootSceneNode* root_;
};

class RootSceneNode: public BasicSceneNode {

public:
	RootSceneNode();
	virtual ~RootSceneNode();
/*
	virtual double rootPercentX() const  { return 0.; } // expressed in percent 
	virtual double absoluteX() const     { return 0.; } // expressed in device unit in root node coordinates 
	virtual double x() const             { return 0.; }   // expressed in percentage from the parent 
	
	virtual double rootPercentY() const  { return 0.; } // expressed in percent 
	virtual double absoluteY() const     { return 0.; } // expressed in device unit in root node coordinates
	virtual double y() const             { return 0.; } // expressed in percentage from the parent 
	
	virtual double rootPercentWidth() const  { return 100.; } // expressed in percent 
	virtual double absoluteWidth() const { return absoluteWidth_; } // expressed in device unit in root node coordinates
	virtual double width()          { return 100.; } // expressed in percentage from the parent 
	
	virtual double rootPercentHeight() const  { return 100.; } // expressed in percent 
	virtual double absoluteHeight() const { return absoluteHeight_; } // expressed in device unit in root node coordinates
	virtual double height()  const       { return 100.; } // expressed in percentage from the parent 
*/
	
	double absoluteWidth() const { return absoluteWidth_; }
	double absoluteHeight() const { return absoluteHeight_; }
	virtual void  absoluteRootWidth(double width)  { scale_ = width/absoluteWidth_; if ( scale_ < 1 ) scale_ = 1; absoluteWidth_ = width; }
	void absoluteRootHeight(double height) { absoluteHeight_ = height; }
	double scalingFactor() const { return scale_; } // For Magml and wrep ... 
	BasicSceneNode* clone();
	Layout* root() { assert(layout_); return layout_; }
	virtual void getReady() {}
	void execute();
	void newpage() { current_->newpage(); }
	BasicSceneNode* insert(BasicPositionalObject*);
	void current(RootScenePage* current) { current_ = current; }
	
	//void push_back(BasicSceneObject* item);
	
	virtual void text(TextVisitor*) { }
	virtual void legend(LegendVisitor*) {  }
	virtual MagicsMode mode() { return mode_; }
	void mode(MagicsMode mode) { mode_ = mode; }

protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream&) const; 
	 double absoluteWidth_;
	 double absoluteHeight_;
	
	 RootScenePage* current_;
	 double scale_;
	 MagicsMode mode_;

private:
    //! Copy constructor - No copy allowed
	RootSceneNode(const RootSceneNode&);
    //! Overloaded << operator to copy - No copy allowed
	RootSceneNode& operator=(const RootSceneNode&);

// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const RootSceneNode& p)
		{ p.print(s); return s; }
};


class FortranRootSceneNode : public RootSceneNode, public FortranRootSceneNodeAttributes
{
public:
	FortranRootSceneNode();
	~FortranRootSceneNode();
	void getReady();
protected:
	void print(ostream&) const;
};


class XmlRootSceneNode : public RootSceneNode, public XmlRootNodeAttributes
{
public:
	XmlRootSceneNode();
	~XmlRootSceneNode();

	void set(const map<string, string>& map)
	{
		XmlRootNodeAttributes::set(map);
	}

	void set(const XmlNode& node)
	{
		XmlRootNodeAttributes::set(node);
	}

	void getReady();

protected:
	void print(ostream&) const; 
};

class WrepRootSceneNode : public XmlRootSceneNode, public WrepRootNodeAttributes
{
public:
	WrepRootSceneNode();
	~WrepRootSceneNode();

	void set(const map<string, string>& map)
	{
		WrepRootNodeAttributes::set(map);
	}

	void set(const XmlNode& node)
	{ 
		WrepRootNodeAttributes::set(node);
	}

	void  absoluteRootWidth(double width);
	void getReady();

protected:
	void print(ostream&) const; 
};

} // namespace magics
#endif
