// T1M_MM2PARAM.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include <time.h>
#include <stdlib.h>

#include <ctype.h>

#include "mpi++.h"

#include "v3d.h"
#include "matrix.h"
#include "notice.h"

#include <fstream>
#include <strstream>
#include <iomanip>

#include <algorithm>

/*################################################################################################*/

const char symbols[20 + 1] = "ARNDCQEGHILKMFPSTWYV";

#define TYPES 37
const i32s types[TYPES] =
{
	0x0100,
	0x0200, 0x0201, 0x0202,
	0x0300, 0x0301,
	0x0400, 0x0401,
	0x0500, 0x0501,
	0x0600, 0x0601,
	0x0700, 0x0701,
	0x0800,
	0x0900, 0x0901,
	0x0A00, 0x0A01,
	0x0B00, 0x0B01,
	0x0C00, 0x0C01, 0x0C02,
	0x0D00, 0x0D01,
	0x0E00, 0x0E01,
	0x0F00,
	0x1000,
	0x1100,
	0x1200, 0x1201, 0x1202,
	0x1300, 0x1301,
	0x1400
};

#define RAND_f64 ((f64) rand() / (f64) RAND_MAX - 0.5)

int main(int argc, char * argv[])
{
	print_full_copyright_notice(cout);
	
/*##############################################*/
/*##############################################*/
/*

	// 230800 : conversion of models from PDB-imported MM1 to MM2.
	
	// secondary-structural elements are built automatically here.
	// there can be overlapping secondary-structural elements?!?!?!
	// that is, ones not separated by a loop residue?!?!?!?!?!
	
	ifstream if1("../../psel_dec_1998/files.txt", ios::in);
	ofstream of2("../../psel_dec_1998/output.txt", ios::out);
	
	mm1_mdl * mdl1;
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		if (if1.peek() == '/') { if1.getline(buffer2, sizeof(buffer2)); continue; }
		
		i32s tmp1; if1 >> tmp1 >> buffer1;
		char chain[2]; chain[0] = chain[1] = buffer1[4]; if (chain[1] == '_') chain[1] = ' ';
		for (i32s n1 = 0;n1 < 4;n1++) buffer1[n1] = (char) tolower(buffer1[n1]); buffer1[4] = 0;
		
		if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str1(buffer2, sizeof(buffer2));
		str1 << "../../psel_dec_1998/pdb" << buffer1 << ".ent" << ends;
		
		cout << buffer2 << " " << chain[0] << endl;
		
		mdl1 = new mm1_mdl(& cout, * console_class_factory::GetInstance());
		mdl1->ImportENT_OLD(buffer2, chain[1]);
		
		mdl1->GatherGroups();
		if (mdl1->nmol > 0)
		{
			mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
			mdl2->Reduce(mdl1);
			
			ostrstream str2(buffer2, sizeof(buffer2));
			str2 << "../../pconv2_dec_1998/" << buffer1 << chain[0] << ".mm2" << ends;
			
			ofstream of1(buffer2, ios::out); mdl2->WriteStream(of1); of1.close();
			
			// an additional check for broken chains...
			// an additional check for broken chains...
			// an additional check for broken chains...
			
			i32s bad_bonds = 0;
			
			for (i32s n1 = 0;n1 < mdl2->chn_vector.size();n1++)
			{
				for (i32s n2 = 0;n2 < mdl2->chn_vector.size() - 1;n2++)
				{
					fGL * c1 = mdl2->chn_vector[n1].res_vector[n2 + 0].crd_vector[0][0].data;
					fGL * c2 = mdl2->chn_vector[n1].res_vector[n2 + 1].crd_vector[0][0].data;
					
					v3d<fGL> v1 = v3d<fGL>(c1, c2);
					if (v1.len() > 0.65) bad_bonds++;
				}
			}
			
			of2 << buffer2 << "   ";
			if (bad_bonds) of2 << "bad_bonds = " << bad_bonds << endl;
			else of2 << "OK" << endl;
			
			delete mdl2;
		}
		
		delete mdl1;
	}
		
	if1.close();
	of2.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 300800 : conversion of decoy sets from PDB-imported MM1 to MM2.
	
	// secondary-structural elements are built automatically here.
	// there can be overlapping secondary-structural elements?!?!?!
	// that is, ones not separated by a loop residue?!?!?!?!?!
	
// decoy-S-misfold
// decoy-M-4state_reduced
// decoy-M-hg_structal
// decoy-M-lmds
// decoy-M-lattice_ssfit
// decoy-M-fisa_all

	ifstream if1("../../decoy-S-misfold/convert.txt", ios::in);
	ofstream of2("../../decoy-S-misfold/output.txt", ios::out);
	
	mm1_mdl * mdl1;
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		i32s nn = strlen(buffer1); buffer1[nn - 4] = 0;		// clip ".pdb" from the filename...
		
		ostrstream str1(buffer2, sizeof(buffer2));
		str1 << "../../decoy-S-misfold/" << buffer1 << ".pdb" << ends;
		
		cout << buffer2 << endl;
		
		mdl1 = new mm1_mdl(& cout, * console_class_factory::GetInstance());
		mdl1->ImportENT_OLD(buffer2, NOT_DEFINED);
		
		mdl1->GatherGroups();
		if (mdl1->nmol > 0)
		{
			mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
			mdl2->Reduce(mdl1);
			
			ostrstream str2(buffer2, sizeof(buffer2));
			str2 << "../../decoy-S-misfold/" << buffer1 << ".mm2" << ends;
			
			ofstream of1(buffer2, ios::out); mdl2->WriteStream(of1); of1.close();
			
			// an additional check for broken chains...
			// an additional check for broken chains...
			// an additional check for broken chains...
			
			i32s bad_bonds = 0;
			
			for (i32s n1 = 0;n1 < mdl2->chn_vector.size();n1++)
			{
				for (i32s n2 = 0;n2 < mdl2->chn_vector.size() - 1;n2++)
				{
					fGL * c1 = mdl2->chn_vector[n1].res_vector[n2 + 0].crd_vector[0][0].data;
					fGL * c2 = mdl2->chn_vector[n1].res_vector[n2 + 1].crd_vector[0][0].data;
					
					v3d<fGL> v1 = v3d<fGL>(c1, c2);
					if (v1.len() > 0.65) bad_bonds++;
				}
			}
			
			of2 << buffer2 << "   ";
			if (bad_bonds) of2 << "bad_bonds = " << bad_bonds << endl;
			else of2 << "OK" << endl;
			
			delete mdl2;
		}
		
		delete mdl1;
	}
		
	if1.close();
	of2.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 050799 : gather information from the original all-atom PDB-entries.
	
	ifstream if1("../../psel_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output1.txt", ios::out);
	
	mm1_mdl * mdl1;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20	// leave gly and pro out...
	
	i32s beta_data[SIZE1]; f64 beta_range[2] = { 0.0, 90.0 };
	for (i32s n1 = 0;n1 < SIZE1;n1++) beta_data[n1] = 0;
	
	i32s rdir_data[SIZE1]; f64 rdir_range[2] = { 0.0, 60.0 };
	for (i32s n1 = 0;n1 < SIZE1;n1++) rdir_data[n1] = 0;
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		i32s tmp1; if1 >> tmp1 >> buffer1;
		char chain[2]; chain[0] = chain[1] = buffer1[4]; if (chain[1] == '_') chain[1] = ' ';
		for (i32s n1 = 0;n1 < 4;n1++) buffer1[n1] = (char) tolower(buffer1[n1]); buffer1[4] = 0;
		
		if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str1(buffer2, sizeof(buffer2));
		str1 << "../../psel_dec_1998/pdb" << buffer1 << ".ent" << ends;
		
		cout << buffer2 << " " << chain[0] << endl;
		
		mdl1 = new mm1_mdl(& cout, * console_class_factory::GetInstance());
		mdl1->ImportENT_OLD(buffer2, chain[1]);
		
		mdl1->GatherGroups();
		mm1_mdl::amino_builder.Identify(mdl1);
		if (mdl1->nmol > 0)
		{
			vector<mm1_chn_info> & ci_vector = (* mdl1->ref_civ);
			for (i32s t1 = 0;t1 < ci_vector.size();t1++)
			{
				iter_mm1al range1[2];
				mdl1->GetRange(1, t1, range1);
				
				for (i32s t2 = 1;t2 < ci_vector[t1].length - 1;t2++)
				{
					if (ci_vector[t1].sequence[t2] == 'G') continue;	// glycine...
					if (ci_vector[t1].sequence[t2] == 'P') continue;	// proline...
					
					iter_mm1al range2a[2];
					mdl1->GetRange(2, range1, t2 - 1, range2a);
					
					iter_mm1al prev = range2a[0];
					while (prev != range2a[1] && ((* prev).res_id & 0xFF) != 0x01) prev++;
					if (prev == range2a[1]) continue;
					
					iter_mm1al range2b[2];
					mdl1->GetRange(2, range1, t2, range2b);
					
					iter_mm1al curr = range2b[0];
					while (curr != range2b[1] && ((* curr).res_id & 0xFF) != 0x01) curr++;
					if (curr == range2b[1]) continue;
					
					iter_mm1al beta = range2b[0];
					while (beta != range2b[1] && ((* beta).res_id & 0xFF) != 0x20) beta++;
					if (beta == range2b[1]) continue;
					
					iter_mm1al range2c[2];
					mdl1->GetRange(2, range1, t2 + 1, range2c);
					
					iter_mm1al next = range2c[0];
					while (next != range2c[1] && ((* next).res_id & 0xFF) != 0x01) next++;
					if (next == range2c[1]) continue;
					
					v3d<fGL> v1 = v3d<fGL>((* curr).crd_vector[0].data, (* prev).crd_vector[0].data);
					v3d<fGL> v2 = v3d<fGL>((* curr).crd_vector[0].data, (* next).crd_vector[0].data);
					v3d<fGL> v3 = v1.vpr(v2);
					
					v3d<fGL> v4 = v3d<fGL>((* curr).crd_vector[0].data, (* beta).crd_vector[0].data);
					f64 t1 = v3.ang(v4); t1 = 180.0 * t1 / M_PI;
					
					if (t1 < beta_range[0]) continue;
					if (t1 > beta_range[1]) continue;
					
					f64 t2 = (t1 - beta_range[0]) / (beta_range[1] - beta_range[0]);
					if (t2 < 1.0) beta_data[((i32s) (SIZE1 * t2))]++;
					
					v3d<fGL> v5 = v1.vpr(v2); v3d<fGL> v6 = v1 + v2;
					v5 = v5 * (cos(M_PI * BETA / 180.0) / v5.len());	// c
					v6 = v6 * (sin(M_PI * BETA / 180.0) / v6.len());	// d
					v3d<fGL> v7 = v5 - v6;		// e
					
					f64 t3 = v7.ang(v4); t3 = 180.0 * t3 / M_PI;
					
					if (t3 < rdir_range[0]) continue;
					if (t3 > rdir_range[1]) continue;
					
					f64 t4 = (t3 - rdir_range[0]) / (rdir_range[1] - rdir_range[0]);
					if (t4 < 1.0) rdir_data[((i32s) (SIZE1 * t4))]++;
				}
			}
		}
		
		delete mdl1;
	}
	
	if1.close();
	
	of1 << "beta-carbon directions: " << beta_range[0] << " -> " << beta_range[1] << endl;
	for (i32s n1 = 0;n1 < SIZE1;n1++)
	{
		f64 zzz = beta_range[0] + (n1 + 0.5) * (beta_range[1] - beta_range[0]) / SIZE1;
		of1 << n1 << " " << zzz << " " << beta_data[n1] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "error in directions: " << rdir_range[0] << " -> " << rdir_range[1] << endl;
	for (i32s n1 = 0;n1 < SIZE1;n1++)
	{
		f64 zzz = rdir_range[0] + (n1 + 0.5) * (rdir_range[1] - rdir_range[0]) / SIZE1;
		of1 << n1 << " " << zzz << " " << rdir_data[n1] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "END";
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 050799 : gather information from the VA bond lengths.
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output2.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20
	
	i32s mc_dist_data[SIZE1]; f64 mc_dist_range[2] = { 0.33, 0.43 };	// 0.33 or 0.25...
	for (i32s init = 0;init < SIZE1;init++) mc_dist_data[init] = 0;
	
	i32s sc1_dist_data[20][SIZE1];
	for (i32s init1 = 0;init1 < 20;init1++)
	{
		for (i32s init2 = 0;init2 < SIZE1;init2++)
		{
			sc1_dist_data[init1][init2] = 0;
		}
	}
	
	f64 sc1_dist_range[20][2] =
	{
		{ 0.000, 1.000 },	// a
		{ 0.225, 0.350 },	// r
		{ 0.230, 0.280 },	// n
		{ 0.230, 0.280 },	// d
		{ 0.250, 0.300 },	// c
		{ 0.245, 0.350 },	// q
		{ 0.245, 0.350 },	// e
		{ 0.000, 1.000 },	// g
		{ 0.315, 0.390 },	// h
		{ 0.225, 0.275 },	// i
		{ 0.225, 0.295 },	// l
		{ 0.220, 0.285 },	// k
		{ 0.245, 0.445 },	// m
		{ 0.340, 0.410 },	// f
		{ 0.000, 1.000 },	// p
		{ 0.000, 1.000 },	// s
		{ 0.000, 1.000 },	// t
		{ 0.310, 0.370 },	// w
		{ 0.340, 0.410 },	// y
		{ 0.000, 1.000 }	// v
	};
	
	i32s sc2_dist_data[20][SIZE1];
	for (i32s init1 = 0;init1 < 20;init1++)
	{
		for (i32s init2 = 0;init2 < SIZE1;init2++)
		{
			sc2_dist_data[init1][init2] = 0;
		}
	}
	
	f64 sc2_dist_range[20][2] =
	{
		{ 0.000, 1.000 },	// a
		{ 0.250, 0.325 },	// r
		{ 0.000, 1.000 },	// n
		{ 0.000, 1.000 },	// d
		{ 0.000, 1.000 },	// c
		{ 0.000, 1.000 },	// q
		{ 0.000, 1.000 },	// e
		{ 0.000, 1.000 },	// g
		{ 0.000, 1.000 },	// h
		{ 0.000, 1.000 },	// i
		{ 0.000, 1.000 },	// l
		{ 0.225, 0.275 },	// k
		{ 0.000, 1.000 },	// m
		{ 0.000, 1.000 },	// f
		{ 0.000, 1.000 },	// p
		{ 0.000, 1.000 },	// s
		{ 0.000, 1.000 },	// t
		{ 0.200, 0.220 },	// w
		{ 0.000, 1.000 },	// y
		{ 0.000, 1.000 }	// v
	};
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 0;d2 < mdl2->chn_vector[d1].res_vector.size() - 1;d2++)
			{
////////////////////////////////////////////////////////////
//if (mdl2->chn_vector[d1].res_vector[d2 + 1].symbol != 'P') continue;		// X-pro only!!!
////////////////////////////////////////////////////////////

				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				v3d<fGL> v1 = v3d<fGL>(c1, c2); f64 t1 = v1.len();
				
				if (t1 < mc_dist_range[0]) continue;
				if (t1 > mc_dist_range[1]) continue;
				
				f64 t2 = (t1 - mc_dist_range[0]) / (mc_dist_range[1] - mc_dist_range[0]);
				if (t2 < 1.0) mc_dist_data[((i32s) (SIZE1 * t2))]++;
			}
		}
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 0;d2 < mdl2->chn_vector[d1].res_vector.size();d2++)
			{
				i32s ind = 0; while (ind < 20 && symbols[ind] != mdl2->chn_vector[d1].res_vector[d2].symbol) ind++;
				if (ind == 20) { cout << "FATAL ERROR: unknown residue symbol !!!" << endl; exit(EXIT_FAILURE); }
				
				if (mdl2->chn_vector[d1].res_vector[d2].natm > 1)
				{
					fGL * c1 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
					fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[1][0].data;
					v3d<fGL> v1 = v3d<fGL>(c1, c2); f64 t1 = v1.len();
					
					if (t1 < sc1_dist_range[ind][0]) continue;
					if (t1 > sc1_dist_range[ind][1]) continue;
					
					f64 t2 = (t1 - sc1_dist_range[ind][0]) / (sc1_dist_range[ind][1] - sc1_dist_range[ind][0]);
					if (t2 < 1.0) sc1_dist_data[ind][((i32s) (SIZE1 * t2))]++;
				}
				
				if (mdl2->chn_vector[d1].res_vector[d2].natm > 2)
				{
					fGL * c1 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[1][0].data;
					fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[2][0].data;
					v3d<fGL> v1 = v3d<fGL>(c1, c2); f64 t1 = v1.len();
					
					if (t1 < sc2_dist_range[ind][0]) continue;
					if (t1 > sc2_dist_range[ind][1]) continue;
					
					f64 t2 = (t1 - sc2_dist_range[ind][0]) / (sc2_dist_range[ind][1] - sc2_dist_range[ind][0]);
					if (t2 < 1.0) sc2_dist_data[ind][((i32s) (SIZE1 * t2))]++;
				}
			}
		}
		
		delete mdl2;
	}
	
	if1.close();
	
	of1 << "main-chain VA bond lengths: " << mc_dist_range[0] << " -> " << mc_dist_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = mc_dist_range[0] + (out + 0.5) * (mc_dist_range[1] - mc_dist_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << mc_dist_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		of1 << "sidechain #1: " << sc1_dist_range[out1][0] << " -> " << sc1_dist_range[out1][1] << " ";
		of1 << "TYPE IS " << out1 << " WHICH MEANS " << symbols[out1] << endl;
		for (i32s out2 = 0;out2 < SIZE1;out2++)
		{
f64 zzz = sc1_dist_range[out1][0] + (out2 + 0.5) * (sc1_dist_range[out1][1] - sc1_dist_range[out1][0]) / SIZE1;
of1 << out2 << " " << zzz << " " << sc1_dist_data[out1][out2] << endl;
		} of1 << "READY" << endl << endl;
	}
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		of1 << "sidechain #2: " << sc2_dist_range[out1][0] << " -> " << sc2_dist_range[out1][1] << " ";
		of1 << "TYPE IS " << out1 << " WHICH MEANS " << symbols[out1] << endl;
		for (i32s out2 = 0;out2 < SIZE1;out2++)
		{
f64 zzz = sc2_dist_range[out1][0] + (out2 + 0.5) * (sc2_dist_range[out1][1] - sc2_dist_range[out1][0]) / SIZE1;
of1 << out2 << " " << zzz << " " << sc2_dist_data[out1][out2] << endl;
		} of1 << "READY" << endl << endl;
	}
	
	of1 << "END";
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 050799 : gather information from VA bond angles and BT4 angles...
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output3.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20
	
	i32s sc2_ang_data[20][SIZE1];
	for (i32s init1 = 0;init1 < 20;init1++)
	{
		for (i32s init2 = 0;init2 < SIZE1;init2++)
		{
			sc2_ang_data[init1][init2] = 0;
		}
	}
	
	f64 sc2_ang_range[20][2] =
	{
		{ 0.0, 180.0 },		// a
		{ 0.0, 180.0 },		// r
		{ 0.0, 180.0 },		// n
		{ 0.0, 180.0 },		// d
		{ 0.0, 180.0 },		// c
		{ 0.0, 180.0 },		// q
		{ 0.0, 180.0 },		// e
		{ 0.0, 180.0 },		// g
		{ 0.0, 180.0 },		// h
		{ 0.0, 180.0 },		// i
		{ 0.0, 180.0 },		// l
		{ 0.0, 180.0 },		// k
		{ 0.0, 180.0 },		// m
		{ 0.0, 180.0 },		// f
		{ 0.0, 180.0 },		// p
		{ 0.0, 180.0 },		// s
		{ 0.0, 180.0 },		// t
		{ 0.0, 180.0 },		// w
		{ 0.0, 180.0 },		// y
		{ 0.0, 180.0 }		// v
	};
	
	i32s sc5_ang_data[20][SIZE1];
	for (i32s init1 = 0;init1 < 20;init1++)
	{
		for (i32s init2 = 0;init2 < SIZE1;init2++)
		{
			sc5_ang_data[init1][init2] = 0;
		}
	}
	
	f64 sc5_ang_range[20][2] =
	{
		{ 0.0, 90.0 },		// a
		{ 0.0, 90.0 },		// r
		{ 0.0, 90.0 },		// n
		{ 0.0, 90.0 },		// d
		{ 0.0, 90.0 },		// c
		{ 0.0, 90.0 },		// q
		{ 0.0, 90.0 },		// e
		{ 0.0, 90.0 },		// g
		{ 0.0, 90.0 },		// h
		{ 0.0, 90.0 },		// i
		{ 0.0, 90.0 },		// l
		{ 0.0, 90.0 },		// k
		{ 0.0, 90.0 },		// m
		{ 0.0, 90.0 },		// f
		{ 0.0, 90.0 },		// p
		{ 0.0, 90.0 },		// s
		{ 0.0, 90.0 },		// t
		{ 0.0, 90.0 },		// w
		{ 0.0, 90.0 },		// y
		{ 0.0, 90.0 }		// v
	};
	
	i32s sc6h_ang_data[20][SIZE1];
	i32s sc6s_ang_data[20][SIZE1];
	i32s sc6l_ang_data[20][SIZE1];
	for (i32s init1 = 0;init1 < 20;init1++)
	{
		for (i32s init2 = 0;init2 < SIZE1;init2++)
		{
			sc6h_ang_data[init1][init2] = 0;
			sc6s_ang_data[init1][init2] = 0;
			sc6l_ang_data[init1][init2] = 0;
		}
	}
	
	f64 sc6_ang_range[20][2] =
	{
		{ -180.0, +180.0 },	// a
		{ -180.0, +180.0 },	// r
		{ -180.0, +180.0 },	// n
		{ -180.0, +180.0 },	// d
		{ -180.0, +180.0 },	// c
		{ -180.0, +180.0 },	// q
		{ -180.0, +180.0 },	// e
		{ -180.0, +180.0 },	// g
		{ -180.0, +180.0 },	// h
		{ -180.0, +180.0 },	// i
		{ -180.0, +180.0 },	// l
		{ -180.0, +180.0 },	// k
		{ -180.0, +180.0 },	// m
		{ -180.0, +180.0 },	// f
		{ -180.0, +180.0 },	// p
		{ -180.0, +180.0 },	// s
		{ -180.0, +180.0 },	// t
		{ -180.0, +180.0 },	// w
		{ -180.0, +180.0 },	// y
		{ -180.0, +180.0 } 	// v
	};
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 0;d2 < mdl2->chn_vector[d1].res_vector.size();d2++)
			{
				i32s ind = 0; while (ind < 20 && symbols[ind] != mdl2->chn_vector[d1].res_vector[d2].symbol) ind++;
				if (ind == 20) { cout << "FATAL ERROR: unknown residue symbol !!!" << endl; exit(EXIT_FAILURE); }
				
				if (mdl2->chn_vector[d1].res_vector[d2].natm > 2)
				{
					fGL * c1 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
					fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[1][0].data;
					fGL * c3 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[2][0].data;
					
					v3d<fGL> v1 = v3d<fGL>(c2, c1);
					v3d<fGL> v2 = v3d<fGL>(c2, c3);
					
					f64 t1 = v1.ang(v2);
					t1 = 180.0 * t1 / M_PI;
					
					if (t1 < sc2_ang_range[ind][0]) continue;
					if (t1 > sc2_ang_range[ind][1]) continue;
					
					f64 t2 = (t1 - sc2_ang_range[ind][0]) / (sc2_ang_range[ind][1] - sc2_ang_range[ind][0]);
					if (t2 < 1.0) sc2_ang_data[ind][((i32s) (SIZE1 * t2))]++;
				}
			}
			
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 1;d2++)
			{
				i32s ind = 0; while (ind < 20 && symbols[ind] != mdl2->chn_vector[d1].res_vector[d2].symbol) ind++;
				if (ind == 20) { cout << "FATAL ERROR: unknown residue symbol !!!" << endl; exit(EXIT_FAILURE); }
				
				if (mdl2->chn_vector[d1].res_vector[d2].natm > 1)
				{
					fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
					fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
					fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
					fGL * c4 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[1][0].data;
					
					bool skip = false;
					v3d<fGL> v1 = v3d<fGL>(c2, c1); if (v1.len() > 0.75) skip = true;
					v3d<fGL> v2 = v3d<fGL>(c2, c3); if (v2.len() > 0.75) skip = true;
					
					if (!skip)
					{
						v1 = v1 / v1.len();		// a
						v2 = v2 / v2.len();		// b
						
						v3d<fGL> v3 = v1.vpr(v2); v3d<fGL> v4 = v1 + v2;
						v3 = v3 * (cos(M_PI * BETA / 180.0) / v3.len());	// c
						v4 = v4 * (sin(M_PI * BETA / 180.0) / v4.len());	// d
						
						v3d<fGL> v5 = v3 - v4;			// e
						v3d<fGL> v6 = v3d<fGL>(c2, c4);		// f
						
						//////////
						
						f64 t1A = v5.ang(v6); t1A = 180.0 * t1A / M_PI;
						if (!(t1A < sc5_ang_range[ind][0]) && !(t1A > sc5_ang_range[ind][1]))
						{
							f64 tt = sc5_ang_range[ind][1] - sc5_ang_range[ind][0];
							f64 t2 = (t1A - sc5_ang_range[ind][0]) / tt;
							
							if (t2 < 1.0) sc5_ang_data[ind][((i32s) (SIZE1 * t2))]++;
						}
						
						//////////
						
						v3 = v3 * (sin(M_PI * BETA / 180.0) / v3.len());
						v4 = v4 * (cos(M_PI * BETA / 180.0) / v4.len());
						v3d<fGL> v7 = v3 + v4;		// g
						
						v3d<fGL> v8 = v6 - (v5 * (v5.spr(v6) / v5.spr(v5)));	// h
						v3d<fGL> v9 = v4.vpr(v3);				// k = d*c !!!
						
						//////////
						
						f64 t1B = v7.ang(v8); t1B = 180.0 * t1B / M_PI;
						if (v9.ang(v8) > M_PI * 0.5) t1B = -t1B;
						
						if (!(t1B < sc6_ang_range[ind][0]) && !(t1B > sc6_ang_range[ind][1]))
						{
							f64 tt = sc6_ang_range[ind][1] - sc6_ang_range[ind][0];
							f64 t2 = (t1B - sc6_ang_range[ind][0]) / tt;
							
							if (t2 < 1.0)
							{
								switch (mdl2->chn_vector[d1].res_vector[d2].state)
								{
									case STATE_HELIX:
									sc6h_ang_data[ind][((i32s) (SIZE1 * t2))]++;
									break;
									
									case STATE_STRAND:
									sc6s_ang_data[ind][((i32s) (SIZE1 * t2))]++;
									break;
									
									default:
									sc6l_ang_data[ind][((i32s) (SIZE1 * t2))]++;
								}
							}
						}
					}
				}
			}
		}
		
		delete mdl2;
	}
	
	if1.close();
	
	// calculate first BT4-parameters directly...
	
	f64 arotbarr = 0.0; i32s rotbarrc = 0;
	
	for (i32s loop = 0;loop < 3;loop++)
	{
		switch (loop)
		{
			case 0:
			of1 << "HELIX-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			of1 << "HELIX-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			of1 << "HELIX-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			break;
			
			case 1:
			of1 << "STRAND-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			of1 << "STRAND-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			of1 << "STRAND-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			break;
			
			case 2:
			of1 << "LOOP-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			of1 << "LOOP-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			of1 << "LOOP-RESULTS !!!!!!!!!!!!!!!!!!" << endl;
			break;
		}
		
		for (i32s n1 = 0;n1 < 20;n1++)
		{
			f64 xdata[SIZE1];
			f64 ydata[SIZE1]; f64 max = 0.0;
			for (i32s n2 = 0;n2 < SIZE1;n2++)
			{
				xdata[n2] = -M_PI + M_PI / ((f64) SIZE1) + 2.0 * ((f64) n2) * M_PI / ((f64) SIZE1);
				
				switch (loop)
				{
					case 0: ydata[n2] = sc6h_ang_data[n1][n2]; break;
					case 1: ydata[n2] = sc6s_ang_data[n1][n2]; break;
					case 2: ydata[n2] = sc6l_ang_data[n1][n2]; break;
				}
				
				if (ydata[n2] < 0.25) ydata[n2] = 0.25;		// mark possible zeros as 1/4!!!
				if (max < ydata[n2]) max = ydata[n2];		// find the maximum...
			}
			
			const f64 rt = 8.31451 * (273.15 + 25.0) / 1000.0;
			for (i32s i = 0;i < SIZE1;i++)
			{
				ydata[i] = -rt * log(ydata[i] / max);		// natural logarithm!!!
			}
			
			// build the normal equations...
			
			f64 m1[7 * 7]; f64 v2[7];
			for (i32s i1 = 0;i1 < 7;i1++)
			{
				v2[i1] = 0.0;
				for (i32s i2 = 0;i2 < 7;i2++) m1[i1 * 7 + i2] = 0.0;
			}
			
			for (i32s i1 = 0;i1 < SIZE1;i1++)
			{
				f64 tmp[7]; tmp[6] = 1.0;
				tmp[0] = cos(xdata[i1]); tmp[1] = cos(2.0 * xdata[i1]); tmp[2] = cos(3.0 * xdata[i1]);
				tmp[3] = sin(xdata[i1]); tmp[4] = sin(2.0 * xdata[i1]); tmp[5] = sin(3.0 * xdata[i1]);
				
				for (i32s i2 = 0;i2 < 7;i2++)
				{
					v2[i2] += tmp[i2] * ydata[i1];
					for (i32s i3 = 0;i3 < 7;i3++) m1[i2 * 7 + i3] += tmp[i2] * tmp[i3];
				}
			}
			
			f64 c1, c2, c3, s1, s2, s3, k, error;
			gauss_elimination<f64> ge(7, m1, v2); ge.solve();
			c1 = ge.getvalue(0); c2 = ge.getvalue(1); c3 = ge.getvalue(2);
			s1 = ge.getvalue(3); s2 = ge.getvalue(4); s3 = ge.getvalue(5);
			k = ge.getvalue(6); error = 0.0;
			
			for (i32s i1 = 0;i1 < SIZE1;i1++)
			{
				f64 x = xdata[i1];
				f64 f1 = c1 * cos(x) + c2 * cos(2.0 * x) + c3 * cos(3.0 * x);
				f64 f2 = s1 * sin(x) + s2 * sin(2.0 * x) + s3 * sin(3.0 * x);
				error += pow(ydata[i1] - (f1 + f2 + k), 2.0);
			}
			
			of1 << n1 << " WHICH IS " << symbols[n1] << " ::: ";
			of1 << c1 << " " << c2 << " " << c3 << " " << s1 << " " << s2 << " " << s3;
			of1 << " ||| " << k << " ||| " << error << endl;
			
			f64 mine = +1.0e+99;
			f64 maxe = -1.0e+99;
			
			for (i32s rb = 0;rb < 50;rb++)
			{
				f64 zz = 2.0 * M_PI * rb / 50.0;
				
				f64 f1 = c1 * cos(zz) + c2 * cos(2.0 * zz) + c3 * cos(3.0 * zz);
				f64 f2 = s1 * sin(zz) + s2 * sin(2.0 * zz) + s3 * sin(3.0 * zz);
				f64 f3 = f1 + f2;
				
				if (f3 < mine) mine = f3;
				if (f3 > maxe) maxe = f3;
			}
			
			arotbarr += (maxe - mine);
			rotbarrc++;
		}
	}
	
	//////////
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		of1 << "sc angle #2: " << sc2_ang_range[out1][0] << " -> " << sc2_ang_range[out1][1] << " ";
		of1 << "TYPE IS " << out1 << " WHICH IS " << symbols[out1] << endl;
		for (i32s out2 = 0;out2 < SIZE1;out2++)
		{
			f64 zzz = sc2_ang_range[out1][0] + (out2 + 0.5) * (sc2_ang_range[out1][1] - sc2_ang_range[out1][0]) / SIZE1;
			of1 << out2 << " " << zzz << " " << sc2_ang_data[out1][out2] << endl;
		} of1 << "READY" << endl << endl;
	}
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		of1 << "BT4 angle A: " << sc5_ang_range[out1][0] << " -> " << sc5_ang_range[out1][1] << " ";
		of1 << "TYPE IS " << out1 << " WHICH IS " << symbols[out1] << endl;
		for (i32s out2 = 0;out2 < SIZE1;out2++)
		{
			f64 zzz = sc5_ang_range[out1][0] + (out2 + 0.5) * (sc5_ang_range[out1][1] - sc5_ang_range[out1][0]) / SIZE1;
			of1 << out2 << " " << zzz << " " << sc5_ang_data[out1][out2] << endl;
		} of1 << "READY" << endl << endl;
	}
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		of1 << "BT4 angle B (H/S/L): " << sc6_ang_range[out1][0] << " -> " << sc6_ang_range[out1][1] << " ";
		of1 << "TYPE IS " << out1 << " WHICH IS " << symbols[out1] << endl;
		for (i32s out2 = 0;out2 < SIZE1;out2++)
		{
			f64 zzz = sc6_ang_range[out1][0] + (out2 + 0.5) * (sc6_ang_range[out1][1] - sc6_ang_range[out1][0]) / SIZE1;
			
			of1 << out2 << " " << zzz << " ";
			of1 << sc6h_ang_data[out1][out2] << " ";
			of1 << sc6s_ang_data[out1][out2] << " ";
			of1 << sc6l_ang_data[out1][out2] << " " << endl;
		} of1 << "READY" << endl << endl;
	}
	
	of1 << "END";
	of1.close();
	
	cout << "average sc barrier = " << (arotbarr / rotbarrc) << " kJ/mol without weighting..." << endl;
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 050799 : gather information from alpha/beta-params and constraints.
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output4.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20
	
	i32s ah_ang_data[SIZE1]; f64 ah_ang_range[2] = { 1.0, 2.5 };
	for (i32s init = 0;init < SIZE1;init++) ah_ang_data[init] = 0;
	
	i32s ah_tor_data[SIZE1]; f64 ah_tor_range[2] = { -M_PI, +M_PI };
	for (i32s init = 0;init < SIZE1;init++) ah_tor_data[init] = 0;
	
	i32s ah_13d_data[SIZE1]; f64 ah_13d_range[2] = { 0.45, 0.70 };
	for (i32s init = 0;init < SIZE1;init++) ah_13d_data[init] = 0;
	
	i32s ah_14d_data[SIZE1]; f64 ah_14d_range[2] = { 0.35, 0.75 };
	for (i32s init = 0;init < SIZE1;init++) ah_14d_data[init] = 0;
	
	i32s ah_15d_data[SIZE1]; f64 ah_15d_range[2] = { 0.40, 0.80 };
	for (i32s init = 0;init < SIZE1;init++) ah_15d_data[init] = 0;
	
	i32s bs_ang_data[SIZE1]; f64 bs_ang_range[2] = { 1.2, 3.0 };
	for (i32s init = 0;init < SIZE1;init++) bs_ang_data[init] = 0;
	
	i32s bs_tor_data[SIZE1]; f64 bs_tor_range[2] = { -M_PI, +M_PI };
	for (i32s init = 0;init < SIZE1;init++) bs_tor_data[init] = 0;
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		mdl2->UpdateConstraints();
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 1;d2++)	// ang
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
				fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				
				v3d<fGL> v1 = v3d<fGL>(c2, c1);
				v3d<fGL> v2 = v3d<fGL>(c2, c3);
				
				if (v1.len() < 0.75 && v2.len() < 0.75)
				{
					f64 t1 = v1.ang(v2);
					
					if (mdl2->chn_vector[d1].res_vector[d2].state == STATE_HELIX)
					{
						if (t1 < ah_ang_range[0]) continue;
						if (t1 > ah_ang_range[1]) continue;
						
						f64 t2 = (t1 - ah_ang_range[0]) / (ah_ang_range[1] - ah_ang_range[0]);
						if (t2 < 1.0) ah_ang_data[((i32s) (SIZE1 * t2))]++;
					}
					
					if (mdl2->chn_vector[d1].res_vector[d2].state == STATE_STRAND)
					{
						if (t1 < bs_ang_range[0]) continue;
						if (t1 > bs_ang_range[1]) continue;
						
						f64 t2 = (t1 - bs_ang_range[0]) / (bs_ang_range[1] - bs_ang_range[0]);
						if (t2 < 1.0) bs_ang_data[((i32s) (SIZE1 * t2))]++;
					}
				}
			}
			
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 2;d2++)	// tor
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
				fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				fGL * c4 = mdl2->chn_vector[d1].res_vector[d2 + 2].crd_vector[0][0].data;
				
				v3d<fGL> v1 = v3d<fGL>(c2, c1);
				v3d<fGL> v2 = v3d<fGL>(c2, c3);
				v3d<fGL> v3 = v3d<fGL>(c3, c4);
				
				if (v1.len() < 0.75 && v2.len() < 0.75 && v3.len() < 0.75)
				{
					f64 t1 = v1.tor(v2, v3);
					
					bool htest1 = (mdl2->chn_vector[d1].res_vector[d2].state == STATE_HELIX);
					bool htest2 = (mdl2->chn_vector[d1].res_vector[d2 + 1].state == STATE_HELIX);
					if (htest1 && htest2)
					{
						if (t1 < ah_tor_range[0]) continue;
						if (t1 > ah_tor_range[1]) continue;
						
						f64 t2 = (t1 - ah_tor_range[0]) / (ah_tor_range[1] - ah_tor_range[0]);
						if (t2 < 1.0) ah_tor_data[((i32s) (SIZE1 * t2))]++;
					}
					
					bool stest1 = (mdl2->chn_vector[d1].res_vector[d2].state == STATE_STRAND);
					bool stest2 = (mdl2->chn_vector[d1].res_vector[d2 + 1].state == STATE_STRAND);
					if (stest1 && stest2)
					{
						if (t1 < bs_tor_range[0]) continue;
						if (t1 > bs_tor_range[1]) continue;
						
						f64 t2 = (t1 - bs_tor_range[0]) / (bs_tor_range[1] - bs_tor_range[0]);
						if (t2 < 1.0) bs_tor_data[((i32s) (SIZE1 * t2))]++;
					}
				}
			}
			
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 1;d2++)
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				
				v3d<fGL> v1 = v3d<fGL>(c1, c2);
				f64 t1 = v1.len();
				
				if (mdl2->chn_vector[d1].res_vector[d2].state == STATE_HELIX)
				{
					if (t1 < ah_13d_range[0]) continue;
					if (t1 > ah_13d_range[1]) continue;
					
					f64 t2 = (t1 - ah_13d_range[0]) / (ah_13d_range[1] - ah_13d_range[0]);
					if (t2 < 1.0) ah_13d_data[((i32s) (SIZE1 * t2))]++;
				}
			}
			
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 2;d2++)
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2 + 2].crd_vector[0][0].data;
				
				v3d<fGL> v1 = v3d<fGL>(c1, c2);
				f64 t1 = v1.len();
				
				bool htest1 = (mdl2->chn_vector[d1].res_vector[d2].state == STATE_HELIX);
				bool htest2 = (mdl2->chn_vector[d1].res_vector[d2 + 1].state == STATE_HELIX);
				if (htest1 && htest2)
				{
					if (t1 < ah_14d_range[0]) continue;
					if (t1 > ah_14d_range[1]) continue;
					
					f64 t2 = (t1 - ah_14d_range[0]) / (ah_14d_range[1] - ah_14d_range[0]);
					if (t2 < 1.0) ah_14d_data[((i32s) (SIZE1 * t2))]++;
				}
			}
			
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 3;d2++)
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2 + 3].crd_vector[0][0].data;
				
				v3d<fGL> v1 = v3d<fGL>(c1, c2);
				f64 t1 = v1.len();
				
				bool htest1 = (mdl2->chn_vector[d1].res_vector[d2].state == STATE_HELIX);
				bool htest2 = (mdl2->chn_vector[d1].res_vector[d2 + 1].state == STATE_HELIX);
				bool htest3 = (mdl2->chn_vector[d1].res_vector[d2 + 2].state == STATE_HELIX);
				if (htest1 && htest2 && htest3)
				{
					if (t1 < ah_15d_range[0]) continue;
					if (t1 > ah_15d_range[1]) continue;
					
					f64 t2 = (t1 - ah_15d_range[0]) / (ah_15d_range[1] - ah_15d_range[0]);
					if (t2 < 1.0) ah_15d_data[((i32s) (SIZE1 * t2))]++;
				}
			}
		}
		
		delete mdl2;
	}
	
	if1.close();
	
	of1 << "ALPHA-angles: " << ah_ang_range[0] << " -> " << ah_ang_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ah_ang_range[0] + (out + 0.5) * (ah_ang_range[1] - ah_ang_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ah_ang_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "ALPHA-torsions: " << ah_tor_range[0] << " -> " << ah_tor_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ah_tor_range[0] + (out + 0.5) * (ah_tor_range[1] - ah_tor_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ah_tor_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "ALPHA 13-dist: " << ah_13d_range[0] << " -> " << ah_13d_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ah_13d_range[0] + (out + 0.5) * (ah_13d_range[1] - ah_13d_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ah_13d_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "ALPHA 14-dist: " << ah_14d_range[0] << " -> " << ah_14d_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ah_14d_range[0] + (out + 0.5) * (ah_14d_range[1] - ah_14d_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ah_14d_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "ALPHA 15-dist: " << ah_15d_range[0] << " -> " << ah_15d_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ah_15d_range[0] + (out + 0.5) * (ah_15d_range[1] - ah_15d_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ah_15d_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "BETA-angles: " << bs_ang_range[0] << " -> " << bs_ang_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = bs_ang_range[0] + (out + 0.5) * (bs_ang_range[1] - bs_ang_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << bs_ang_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "BETA-torsions: " << bs_tor_range[0] << " -> " << bs_tor_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = bs_tor_range[0] + (out + 0.5) * (bs_tor_range[1] - bs_tor_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << bs_tor_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "END";
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 270800 : gather information from loop-params and constraints.
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output5.txt", ios::out);
	ofstream of2("/tmp/looptor.txt", ios::out);
	of2.setf(ios::scientific);
	
	// on some combinations there are only few observed values -> bad statistics.
	// add average distribututions for all with weight of 20 % !!!!!!!!!!!
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20
	
	i32s loop_ang_data[SIZE1]; f64 loop_ang_range[2] = { 0.75, M_PI };
	for (i32s init = 0;init < SIZE1;init++) loop_ang_data[init] = 0;
	
	#define SIZE2 12	// 30 degrees...
	
	f64 average_1st[20][SIZE2];
	f64 average_2nd[20][SIZE2];
	
	f64 loop_tor_data[20][20][SIZE2];
	f64 loop_tor_range[2] = { -M_PI, +M_PI };
	
	for (i32s i1 = 0;i1 < 20;i1++)
	{
		for (i32s i3 = 0;i3 < SIZE2;i3++)
		{
			average_1st[i1][i3] = 0.0;
			average_2nd[i1][i3] = 0.0;
		}
		
		for (i32s i2 = 0;i2 < 20;i2++)
		{
			for (i32s i3 = 0;i3 < SIZE2;i3++) loop_tor_data[i1][i2][i3] = 0.0;
		}
	}
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 1;d2++)
			{
////////////////////////////////////////////////////////////
//if (mdl2->chn_vector[d1].res_vector[d2 + 0].symbol != 'P') continue;		// X-pro only!!!
//if (mdl2->chn_vector[d1].res_vector[d2 + 1].symbol != 'P') continue;		// X-pro only!!!
////////////////////////////////////////////////////////////

				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
				fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				
				v3d<fGL> v1 = v3d<fGL>(c2, c1);
				v3d<fGL> v2 = v3d<fGL>(c2, c3);
				
				if (v1.len() < 0.75 && v2.len() < 0.75)
				{
					f64 t1 = v1.ang(v2);
					
					if (mdl2->chn_vector[d1].res_vector[d2].state == STATE_LOOP)
					{
						if (t1 < loop_ang_range[0]) continue;
						if (t1 > loop_ang_range[1]) continue;
						
						f64 t2 = (t1 - loop_ang_range[0]) / (loop_ang_range[1] - loop_ang_range[0]);
						if (t2 < 1.0) loop_ang_data[((i32s) (SIZE1 * t2))]++;
					}
				}
			}
			
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 2;d2++)
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
				fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				fGL * c4 = mdl2->chn_vector[d1].res_vector[d2 + 2].crd_vector[0][0].data;
				
				v3d<fGL> v1 = v3d<fGL>(c2, c1);
				v3d<fGL> v2 = v3d<fGL>(c2, c3);
				v3d<fGL> v3 = v3d<fGL>(c3, c4);
				
				if (v1.len() < 0.75 && v2.len() < 0.75 && v3.len() < 0.75)
				{
					f64 t1 = v1.tor(v2, v3);
					
					// which criteria should be used here, and/or ?!?!?!
					// or would be strictly correct, but and will give more "pure" dataset ?!?!?!?!
					
					bool ltest1 = (mdl2->chn_vector[d1].res_vector[d2].state == STATE_LOOP);
					bool ltest2 = (mdl2->chn_vector[d1].res_vector[d2 + 1].state == STATE_LOOP);
					if (ltest1 && ltest2)
					{
						char tp1a = mdl2->chn_vector[d1].res_vector[d2].symbol;
						i32s tp2a = 0; while (tp2a < 20 && symbols[tp2a] != tp1a) tp2a++;
						if (tp2a == 20) { cout << "error 1" << endl; exit(false); }
						
						char tp1b = mdl2->chn_vector[d1].res_vector[d2 + 1].symbol;
						i32s tp2b = 0; while (tp2b < 20 && symbols[tp2b] != tp1b) tp2b++;
						if (tp2b == 20) { cout << "error 2" << endl; exit(false); }
						
						if (t1 < loop_tor_range[0]) continue;
						if (t1 > loop_tor_range[1]) continue;
						
						f64 t2 = (t1 - loop_tor_range[0]) / (loop_tor_range[1] - loop_tor_range[0]);
						if (t2 < 1.0)
						{
							average_1st[tp2a][((i32s) (SIZE2 * t2))] += 1.0;
							average_2nd[tp2b][((i32s) (SIZE2 * t2))] += 1.0;
							
							loop_tor_data[tp2a][tp2b][((i32s) (SIZE2 * t2))] += 1.0;
						}
					}
				}
			}
		}
		
		delete mdl2;
	}
	
	if1.close();
	
	// normalize the averages...
	// normalize the averages...
	// normalize the averages...
	
	for (i32s i1 = 0;i1 < 20;i1++)
	{
		f64 sum1 = 0.0;
		f64 sum2 = 0.0;
		
		for (i32s i3 = 0;i3 < SIZE2;i3++)
		{
			sum1 += average_1st[i1][i3];
			sum2 += average_2nd[i1][i3];
		}
		
		for (i32s i3 = 0;i3 < SIZE2;i3++)
		{
			average_1st[i1][i3] /= sum1;
			average_2nd[i1][i3] /= sum2;
		}
	}
	
	// sum the averages to the distributions with weight 20% !!!!!!!!!!
	// sum the averages to the distributions with weight 20% !!!!!!!!!!
	// sum the averages to the distributions with weight 20% !!!!!!!!!!
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		for (i32s out2 = 0;out2 < 20;out2++)
		{
			f64 sum = 0.0;
			for (i32s i = 0;i < SIZE2;i++)
			{
				sum += loop_tor_data[out1][out2][i];
			}
			
			for (i32s i = 0;i < SIZE2;i++)
			{
				loop_tor_data[out1][out2][i] += 0.10 * sum * average_1st[out1][i];
				loop_tor_data[out1][out2][i] += 0.10 * sum * average_2nd[out2][i];
			}
		}
	}
	
	f64 arotbarr = 0.0; i32s rotbarrc = 0;
	
	of1 << "LOOP-angles: " << loop_ang_range[0] << " -> " << loop_ang_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = loop_ang_range[0] + (out + 0.5) * (loop_ang_range[1] - loop_ang_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << loop_ang_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		for (i32s out2 = 0;out2 < 20;out2++)
		{
			f64 xdata[SIZE2]; f64 ydata[SIZE2]; f64 max = 0.0;
			for (i32s i = 0;i < SIZE2;i++)
			{
				xdata[i] = -M_PI + M_PI / 12.0 + 2.0 * ((f64) i) * M_PI / 12.0;
				
				ydata[i] = loop_tor_data[out1][out2][i];
				if (ydata[i] < 0.0) ydata[i] += 0.0;		// this is obsolete, averages should work now...
				if (max < ydata[i]) max = ydata[i];		// find the maximum...
			}
						
			const f64 rt = 8.31451 * (273.15 + 25.0) / 1000.0;
			for (i32s i = 0;i < SIZE2;i++)
			{
				ydata[i] = -rt * log(ydata[i] / max);		// natural logarithm!!!
			}
			
			// build the normal equations...
			
			f64 m1[7 * 7]; f64 v2[7];
			for (i32s i1 = 0;i1 < 7;i1++)
			{
				v2[i1] = 0.0;
				for (i32s i2 = 0;i2 < 7;i2++) m1[i1 * 7 + i2] = 0.0;
			}
			
			for (i32s i1 = 0;i1 < SIZE2;i1++)
			{
				f64 tmp[7]; tmp[6] = 1.0;
				tmp[0] = cos(xdata[i1]); tmp[1] = cos(2.0 * xdata[i1]); tmp[2] = cos(3.0 * xdata[i1]);
				tmp[3] = sin(xdata[i1]); tmp[4] = sin(2.0 * xdata[i1]); tmp[5] = sin(3.0 * xdata[i1]);
				
				for (i32s i2 = 0;i2 < 7;i2++)
				{
					v2[i2] += tmp[i2] * ydata[i1];
					for (i32s i3 = 0;i3 < 7;i3++) m1[i2 * 7 + i3] += tmp[i2] * tmp[i3];
				}
			}
			
			gauss_elimination<f64> ge(7, m1, v2); ge.solve();
			f64 c1 = ge.getvalue(0); f64 c2 = ge.getvalue(1); f64 c3 = ge.getvalue(2);
			f64 s1 = ge.getvalue(3); f64 s2 = ge.getvalue(4); f64 s3 = ge.getvalue(5);
			f64 k = ge.getvalue(6);
			
			f64 error = 0.0;
			for (i32s i1 = 0;i1 < SIZE2;i1++)
			{
				f64 x = xdata[i1];
				f64 f1 = c1 * cos(x) + c2 * cos(2.0 * x) + c3 * cos(3.0 * x);
				f64 f2 = s1 * sin(x) + s2 * sin(2.0 * x) + s3 * sin(3.0 * x);
				error += pow(ydata[i1] - (f1 + f2 + k), 2.0);
			}
			
			of1 << out1 << " " << out2 << " ::: ";
			of1 << c1 << " " << c2 << " " << c3 << " ";
			of1 << s1 << " " << s2 << " " << s3 << " ";
			of1 << "||| " << k << " ||| " << error << endl;
			
			of2 << symbols[out1] << symbols[out2] << " ";
			of2 << c1 << " " << c2 << " " << c3 << " ";
			of2 << s1 << " " << s2 << " " << s3 << " ";
			of2 << endl;
			
			f64 mine = +1.0e+99;
			f64 maxe = -1.0e+99;
			
			for (i32s rb = 0;rb < 50;rb++)
			{
				f64 zz = 2.0 * M_PI * rb / 50.0;
				
				f64 f1 = c1 * cos(zz) + c2 * cos(2.0 * zz) + c3 * cos(3.0 * zz);
				f64 f2 = s1 * sin(zz) + s2 * sin(2.0 * zz) + s3 * sin(3.0 * zz);
				f64 f3 = f1 + f2;
				
				if (f3 < mine) mine = f3;
				if (f3 > maxe) maxe = f3;
			}
			
			arotbarr += (maxe - mine);
			rotbarrc++;
		}
	}
	
	of2 << "end";
	of2.close();
		
	of1 << "LOOP-torsions: " << loop_tor_range[0] << " -> " << loop_tor_range[1] << endl;
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		for (i32s out2 = 0;out2 < 20;out2++)
		{
			of1 << out1 << " " << out2 << " -> ";
			for (i32s out3 = 0;out3 < SIZE2;out3++)
			{
				of1 << loop_tor_data[out1][out2][out3] << " ";
			}
			
			of1 << endl;
		}
	}
	
	of1 << "END";
	of1.close();
			
	cout << "average barrier = " << (arotbarr / rotbarrc) << " kJ/mol without weighting..." << endl;
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 050799 : gather information from terminals and disulphide bridges.
	
	// are terminal torsions needed??? then it would be necessary to include
	// angle directions properly in the torsion terms...
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output6.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20
	
	i32s dsb_dist_data[SIZE1]; f64 dsb_dist_range[2] = { 0.15, 0.25 };
	for (i32s init = 0;init < SIZE1;init++) dsb_dist_data[init] = 0;
	
	i32s dsb_ang_data[SIZE1]; f64 dsb_ang_range[2] = { 1.0, 3.0 };
	for (i32s init = 0;init < SIZE1;init++) dsb_ang_data[init] = 0;
	
	i32s nt_ang_data[SIZE1]; f64 nt_ang_range[2] = { 0.0, M_PI };
	for (i32s init = 0;init < SIZE1;init++) nt_ang_data[init] = 0;
	
	i32s nt_tor_data[SIZE1]; f64 nt_tor_range[2] = { -M_PI, +M_PI };
	for (i32s init = 0;init < SIZE1;init++) nt_tor_data[init] = 0;
	
	i32s ct_ang_data[SIZE1]; f64 ct_ang_range[2] = { 0.0, M_PI };
	for (i32s init = 0;init < SIZE1;init++) nt_ang_data[init] = 0;
	
	i32s ct_tor_data[SIZE1]; f64 ct_tor_range[2] = { -M_PI, +M_PI };
	for (i32s init = 0;init < SIZE1;init++) nt_tor_data[init] = 0;
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		for (i32s d1 = 0;d1 < mdl2->dsb_vector.size();d1++)
		{
			i32s chn1 = mdl2->dsb_vector[d1].chn[0]; i32s res1 = mdl2->dsb_vector[d1].res[0];
			i32s chn2 = mdl2->dsb_vector[d1].chn[1]; i32s res2 = mdl2->dsb_vector[d1].res[1];
			
			fGL * c1 = mdl2->chn_vector[chn1].res_vector[res1].crd_vector[0][0].data;
			fGL * c2 = mdl2->chn_vector[chn1].res_vector[res1].crd_vector[1][0].data;
			fGL * c3 = mdl2->chn_vector[chn2].res_vector[res2].crd_vector[1][0].data;
			fGL * c4 = mdl2->chn_vector[chn2].res_vector[res2].crd_vector[0][0].data;
			
			//////////
			
			v3d<fGL> v1 = v3d<fGL>(c2, c3); f64 t1 = v1.len();
			
			if (t1 < dsb_dist_range[0]) continue;
			if (t1 > dsb_dist_range[1]) continue;
			
			f64 t2 = (t1 - dsb_dist_range[0]) / (dsb_dist_range[1] - dsb_dist_range[0]);
			if (t2 < 1.0) dsb_dist_data[((i32s) (SIZE1 * t2))]++;
			
			//////////
			
			v3d<fGL> v2a = v3d<fGL>(c2, c1);
			v3d<fGL> v2b = v3d<fGL>(c2, c3);
			f64 t3 = v2a.ang(v2b);
			
			if (t3 < dsb_ang_range[0]) continue;
			if (t3 > dsb_ang_range[1]) continue;
			
			f64 t4 = (t3 - dsb_ang_range[0]) / (dsb_ang_range[1] - dsb_ang_range[0]);
			if (t4 < 1.0) dsb_ang_data[((i32s) (SIZE1 * t4))]++;
			
			v3d<fGL> v3a = v3d<fGL>(c3, c2);
			v3d<fGL> v3b = v3d<fGL>(c3, c4);
			f64 t5 = v3a.ang(v3b);
			
			if (t5 < dsb_ang_range[0]) continue;
			if (t5 > dsb_ang_range[1]) continue;
			
			f64 t6 = (t5 - dsb_ang_range[0]) / (dsb_ang_range[1] - dsb_ang_range[0]);
			if (t6 < 1.0) dsb_ang_data[((i32s) (SIZE1 * t6))]++;
		}
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			if (mdl2->chn_vector[d1].res_vector.size() < 3) continue;
			if (mdl2->chn_vector[d1].res_vector[0].natm < 2) continue;
			
			fGL * c1 = mdl2->chn_vector[d1].res_vector[0].crd_vector[1][0].data;
			fGL * c2 = mdl2->chn_vector[d1].res_vector[0].crd_vector[0][0].data;
			fGL * c3 = mdl2->chn_vector[d1].res_vector[1].crd_vector[0][0].data;
			fGL * c4 = mdl2->chn_vector[d1].res_vector[2].crd_vector[0][0].data;
			
			//////////
			
			v3d<fGL> v1 = v3d<fGL>(c2, c1);
			v3d<fGL> v2 = v3d<fGL>(c2, c3);
			f64 t1 = v1.ang(v2);
			
			if (t1 < nt_ang_range[0]) continue;
			if (t1 > nt_ang_range[1]) continue;
			
			f64 t2 = (t1 - nt_ang_range[0]) / (nt_ang_range[1] - nt_ang_range[0]);
			if (t2 < 1.0) nt_ang_data[((i32s) (SIZE1 * t2))]++;
			
			//////////
			
			v3d<fGL> v3 = v3d<fGL>(c3, c4);
			f64 t3 = v1.tor(v2, v3);
			
			if (t3 < nt_tor_range[0]) continue;
			if (t3 > nt_tor_range[1]) continue;
			
			f64 t4 = (t3 - nt_tor_range[0]) / (nt_tor_range[1] - nt_tor_range[0]);
			if (t4 < 1.0) nt_tor_data[((i32s) (SIZE1 * t4))]++;
		}
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			i32s sz = mdl2->chn_vector[d1].res_vector.size();
			
			if (sz < 3) continue;
			if (mdl2->chn_vector[d1].res_vector[sz - 1].natm < 2) continue;
			
			fGL * c1 = mdl2->chn_vector[d1].res_vector[sz - 1].crd_vector[1][0].data;
			fGL * c2 = mdl2->chn_vector[d1].res_vector[sz - 1].crd_vector[0][0].data;
			fGL * c3 = mdl2->chn_vector[d1].res_vector[sz - 2].crd_vector[0][0].data;
			fGL * c4 = mdl2->chn_vector[d1].res_vector[sz - 3].crd_vector[0][0].data;
			
			//////////
			
			v3d<fGL> v1 = v3d<fGL>(c2, c1);
			v3d<fGL> v2 = v3d<fGL>(c2, c3);
			f64 t1 = v1.ang(v2);
			
			if (t1 < ct_ang_range[0]) continue;
			if (t1 > ct_ang_range[1]) continue;
			
			f64 t2 = (t1 - ct_ang_range[0]) / (ct_ang_range[1] - ct_ang_range[0]);
			if (t2 < 1.0) ct_ang_data[((i32s) (SIZE1 * t2))]++;
			
			//////////
			
			v3d<fGL> v3 = v3d<fGL>(c3, c4);
			f64 t3 = v1.tor(v2, v3);
			
			if (t3 < ct_tor_range[0]) continue;
			if (t3 > ct_tor_range[1]) continue;
			
			f64 t4 = (t3 - ct_tor_range[0]) / (ct_tor_range[1] - ct_tor_range[0]);
			if (t4 < 1.0) ct_tor_data[((i32s) (SIZE1 * t4))]++;
		}
		
		delete mdl2;
	}
	
	if1.close();
	
	of1 << "DSB-distances: " << dsb_dist_range[0] << " -> " << dsb_dist_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = dsb_dist_range[0] + (out + 0.5) * (dsb_dist_range[1] - dsb_dist_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << dsb_dist_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "DSB-angles: " << dsb_ang_range[0] << " -> " << dsb_ang_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = dsb_ang_range[0] + (out + 0.5) * (dsb_ang_range[1] - dsb_ang_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << dsb_ang_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "N-terminal ANG: " << nt_ang_range[0] << " -> " << nt_ang_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = nt_ang_range[0] + (out + 0.5) * (nt_ang_range[1] - nt_ang_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << nt_ang_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "N-terminal TOR: " << nt_tor_range[0] << " -> " << nt_tor_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = nt_tor_range[0] + (out + 0.5) * (nt_tor_range[1] - nt_tor_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << nt_tor_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "C-terminal ANG: " << ct_ang_range[0] << " -> " << ct_ang_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ct_ang_range[0] + (out + 0.5) * (ct_ang_range[1] - ct_ang_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ct_ang_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "C-terminal TOR: " << ct_tor_range[0] << " -> " << ct_tor_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ct_tor_range[0] + (out + 0.5) * (ct_tor_range[1] - ct_tor_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ct_tor_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "END";
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 051199 : gather information from the dipole term #1.
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output7.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20
	
	const f64 rrange[2] = { 0.0, 100.0 };
	
	i32s ch_ld_angp_data[SIZE1]; f64 ch_ld_angp_range[2] = { -1.0, +1.0 };
	for (i32s init = 0;init < SIZE1;init++) ch_ld_angp_data[init] = 0;
	
	i32s ah_ddir_data[SIZE1]; f64 ah_ddir_range[2] = { -M_PI, +M_PI };
	for (i32s init = 0;init < SIZE1;init++) ah_ddir_data[init] = 0;
	
	i32s bs_ddir_data[SIZE1]; f64 bs_ddir_range[2] = { -M_PI, +M_PI };
	for (i32s init = 0;init < SIZE1;init++) bs_ddir_data[init] = 0;
	
	i32s lp_ddir_data[SIZE1]; f64 lp_ddir_range[2] = { -M_PI, +M_PI };
	for (i32s init = 0;init < SIZE1;init++) lp_ddir_data[init] = 0;
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 2;d2++)
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
				fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				
				v3d<fGL> v1a = v3d<fGL>(c2, c1);
				v3d<fGL> v2a = v3d<fGL>(c2, c3);
				
				v3d<fGL> v3a = v1a - v2a * (v1a.spr(v2a) / v2a.spr(v2a)); v3a = v3a / v3a.len();
				v3d<fGL> v4a = v1a.vpr(v2a); v4a = v4a / v4a.len();
				
				f64 k1a = cos(mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0]);
				f64 k2a = sin(mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0]);
				v3d<fGL> v5a = v3a * k1a - v4a * k2a;
				
				v3d<fGL> pv1 = v3d<fGL>(c2); pv1 = pv1 + v2a / 2.0;
				
				bool test1a = (mdl2->chn_vector[d1].res_vector[d2+0].state == STATE_HELIX);
				bool test1b = (mdl2->chn_vector[d1].res_vector[d2+1].state == STATE_HELIX);
				bool test2a = (mdl2->chn_vector[d1].res_vector[d2+0].state == STATE_STRAND);
				bool test2b = (mdl2->chn_vector[d1].res_vector[d2+1].state == STATE_STRAND);
				bool test3a = (mdl2->chn_vector[d1].res_vector[d2+0].state == STATE_LOOP);
				bool test3b = (mdl2->chn_vector[d1].res_vector[d2+1].state == STATE_LOOP);
				
				if (test1a && test1b)
				{
					f64 t1 = mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0];
					if (t1 < ah_ddir_range[0]) continue; if (t1 > ah_ddir_range[1]) continue;
					f64 t2 = (t1 - ah_ddir_range[0]) / (ah_ddir_range[1] - ah_ddir_range[0]);
					if (t2 < 1.0) ah_ddir_data[((i32s) (SIZE1 * t2))]++;
				}
				else if (test2a && test2b)
				{
					f64 t1 = mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0];
					if (t1 < bs_ddir_range[0]) continue; if (t1 > bs_ddir_range[1]) continue;
					f64 t2 = (t1 - bs_ddir_range[0]) / (bs_ddir_range[1] - bs_ddir_range[0]);
					if (t2 < 1.0) bs_ddir_data[((i32s) (SIZE1 * t2))]++;
				}
				else
				{
					if (test3a || test3b)
					{
						fGL * c4 = mdl2->chn_vector[d1].res_vector[d2 + 2].crd_vector[0][0].data;
						
						v3d<fGL> v3t = v3d<fGL>(c3, c4);
						
						bool gly = (mdl2->chn_vector[d1].res_vector[d2 + 1].symbol == 'G');
						
//	if (gly) continue;	// MAJOR ONLY
//	if (!gly) continue;	// MINOR ONLY

						f64 tor = v1a.tor(v2a, v3t);
						
//	if (tor < -180.0 * M_PI / 180.0) continue; if (tor > -135.0 * M_PI / 180.0) continue;
//	if (tor < -135.0 * M_PI / 180.0) continue; if (tor > -90.0 * M_PI / 180.0) continue;
//	if (tor < -90.0 * M_PI / 180.0) continue; if (tor > -45.0 * M_PI / 180.0) continue;
//	if (tor < -45.0 * M_PI / 180.0) continue; if (tor > 0.0 * M_PI / 180.0) continue;
//	if (tor < 0.0 * M_PI / 180.0) continue; if (tor > +45.0 * M_PI / 180.0) continue;
//	if (tor < +45.0 * M_PI / 180.0) continue; if (tor > +90.0 * M_PI / 180.0) continue;
//	if (tor < +90.0 * M_PI / 180.0) continue; if (tor > +135.0 * M_PI / 180.0) continue;
//	if (tor < +135.0 * M_PI / 180.0) continue; if (tor > +180.0 * M_PI / 180.0) continue;
						
						f64 t1 = mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0];
						if (t1 < lp_ddir_range[0]) continue; if (t1 > lp_ddir_range[1]) continue;
						f64 t2 = (t1 - lp_ddir_range[0]) / (lp_ddir_range[1] - lp_ddir_range[0]);
						if (t2 < 1.0) lp_ddir_data[((i32s) (SIZE1 * t2))]++;
					}
					
					if (test3a && test3b)
					{
						for (i32s d3 = 0;d3 < mdl2->chn_vector.size();d3++)
						{
							for (i32s d4 = 4;d4 < mdl2->chn_vector[d3].res_vector.size();d4++)
							{
								for (i32s d5 = 0;d5 < mdl2->chn_vector[d3].res_vector[d4].natm;d5++)
								{
					bool ttt1 = (mdl2->chn_vector[d3].res_vector[d4].type[d5] == TYPE_CPOS);
					bool ttt2 = (mdl2->chn_vector[d3].res_vector[d4].type[d5] == TYPE_CNEG);
					if (!ttt1 && !ttt2) continue;
					
					fGL * c9 = mdl2->chn_vector[d3].res_vector[d4].crd_vector[d5][0].data;
					v3d<fGL> v9 = v3d<fGL>(pv1.data, c9); if (ttt2) v9 = v9 * -1.0;
					
					if (v9.len() < rrange[0]) continue;
					if (v9.len() > rrange[1]) continue;
					
					f64 t1 = cos(v5a.ang(v9));
					if (t1 < ch_ld_angp_range[0]) continue; if (t1 > ch_ld_angp_range[1]) continue;
					
					f64 t2 = (t1 - ch_ld_angp_range[0]) / (ch_ld_angp_range[1] - ch_ld_angp_range[0]);
					if (t2 < 1.0) ch_ld_angp_data[((i32s) (SIZE1 * t2))]++;
								}
							}
						}
					}
				}
			}
		}
		
		delete mdl2;
	}
	
	if1.close();
	
	of1 << "ch-ld angles when rrange is " << rrange[0] << " -> " << rrange[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ch_ld_angp_range[0] + (out + 0.5) * (ch_ld_angp_range[1] - ch_ld_angp_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ch_ld_angp_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "ah directions: " << ah_ddir_range[0] << " -> " << ah_ddir_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = ah_ddir_range[0] + (out + 0.5) * (ah_ddir_range[1] - ah_ddir_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << ah_ddir_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "bs directions: " << bs_ddir_range[0] << " -> " << bs_ddir_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = bs_ddir_range[0] + (out + 0.5) * (bs_ddir_range[1] - bs_ddir_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << bs_ddir_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "loop-directions (on the selected range): " << lp_ddir_range[0] << " -> " << lp_ddir_range[1] << endl;
	for (i32s out = 0;out < SIZE1;out++)
	{
		f64 zzz = lp_ddir_range[0] + (out + 0.5) * (lp_ddir_range[1] - lp_ddir_range[0]) / SIZE1;
		of1 << out << " " << zzz << " " << lp_ddir_data[out] << endl;
	} of1 << "READY" << endl << endl;
	
	of1 << "END";
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 270800 : gather information from the dipole term #2 PARAMETERS.
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of2("/tmp/loopdip.txt", ios::out);
	of2.setf(ios::scientific);
	
	// on some combinations there are only few observed values -> bad statistics.
	// add average distribututions for all with weight of 20 % !!!!!!!!!!!
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE2 36	// 10 degrees...
	#define SIZE3 120	// 3 degrees...
	
	float average_1st[20][SIZE2][SIZE3];		// must use float here due to a compiler problem?!?!?!
	float average_2nd[20][SIZE2][SIZE3];		// must use float here due to a compiler problem?!?!?!
	
	float loop_dip_data[20][20][SIZE2][SIZE3];	// must use float here due to a compiler problem?!?!?!
	f64 radian_range[2] = { -M_PI, +M_PI };
	
	for (i32s i1 = 0;i1 < 20;i1++)
	{
		for (i32s i3 = 0;i3 < SIZE2;i3++)
		{
			for (i32s i4 = 0;i4 < SIZE3;i4++)
			{
				average_1st[i1][i3][i4] = 0.0;
				average_2nd[i1][i3][i4] = 0.0;
			}
		}
		
		for (i32s i2 = 0;i2 < 20;i2++)
		{
			for (i32s i3 = 0;i3 < SIZE2;i3++)
			{
				for (i32s i4 = 0;i4 < SIZE3;i4++) loop_dip_data[i1][i2][i3][i4] = 0.0;
			}
		}
	}
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 2;d2++)
			{
				fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
				fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
				fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
				
				v3d<fGL> v1a = v3d<fGL>(c2, c1);
				v3d<fGL> v2a = v3d<fGL>(c2, c3);
				
				v3d<fGL> v3a = v1a - v2a * (v1a.spr(v2a) / v2a.spr(v2a)); v3a = v3a / v3a.len();
				v3d<fGL> v4a = v1a.vpr(v2a); v4a = v4a / v4a.len();
				
				f64 k1a = cos(mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0]);
				f64 k2a = sin(mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0]);
				v3d<fGL> v5a = v3a * k1a - v4a * k2a;
				
				v3d<fGL> pv1 = v3d<fGL>(c2); pv1 = pv1 + v2a / 2.0;
				
				bool test3a = (mdl2->chn_vector[d1].res_vector[d2+0].state == STATE_LOOP);
				bool test3b = (mdl2->chn_vector[d1].res_vector[d2+1].state == STATE_LOOP);
				if (test3a && test3b)		// and operator is needed here...
				{
					fGL * c4 = mdl2->chn_vector[d1].res_vector[d2 + 2].crd_vector[0][0].data;
					v3d<fGL> v3t = v3d<fGL>(c3, c4);
					
					f64 tor = v1a.tor(v2a, v3t);
					f64 dip = mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0];
					
					if (v1a.len() < 0.75 && v2a.len() < 0.75 && v3t.len() < 0.75)
					{
						char tp1a = mdl2->chn_vector[d1].res_vector[d2].symbol;
						i32s tp2a = 0; while (tp2a < 20 && symbols[tp2a] != tp1a) tp2a++;
						if (tp2a == 20) { cout << "error 1" << endl; exit(false); }
						
						char tp1b = mdl2->chn_vector[d1].res_vector[d2 + 1].symbol;
						i32s tp2b = 0; while (tp2b < 20 && symbols[tp2b] != tp1b) tp2b++;
						if (tp2b == 20) { cout << "error 2" << endl; exit(false); }
						
						if (tor < radian_range[0]) { cout << "tor -"; continue; }
						if (tor > radian_range[1]) { cout << "tor +"; continue; }
						
						if (dip < radian_range[0]) { cout << "dip -"; continue; }
						if (dip > radian_range[1]) { cout << "dip +"; continue; }
						
						f64 t2 = (tor - radian_range[0]) / (radian_range[1] - radian_range[0]);
						f64 t3 = (dip - radian_range[0]) / (radian_range[1] - radian_range[0]);
						if (t2 < 1.0 && t3 < 1.0)
						{
							average_1st[tp2a][((i32s) (SIZE2 * t2))][((i32s) (SIZE3 * t3))] += 1.0;
							average_2nd[tp2b][((i32s) (SIZE2 * t2))][((i32s) (SIZE3 * t3))] += 1.0;
							
							loop_dip_data[tp2a][tp2b][((i32s) (SIZE2 * t2))][((i32s) (SIZE3 * t3))] += 1.0;
						}
					}
				}
			}
		}
		
		delete mdl2;
	}
	
	if1.close();
		
	// normalize the averages...
	// normalize the averages...
	// normalize the averages...
	
	for (i32s i1 = 0;i1 < 20;i1++)
	{
		for (i32s i3 = 0;i3 < SIZE2;i3++)
		{
			f64 sum1 = 0.0;
			f64 sum2 = 0.0;
			
			for (i32s i4 = 0;i4 < SIZE3;i4++)
			{
				sum1 += average_1st[i1][i3][i4];
				sum2 += average_2nd[i1][i3][i4];
			}
			
			for (i32s i4 = 0;i4 < SIZE3;i4++)
			{
				average_1st[i1][i3][i4] /= sum1;
				average_2nd[i1][i3][i4] /= sum2;
			}
		}
	}
	
	// sum the averages to the distributions with weight 20% !!!!!!!!!!
	// sum the averages to the distributions with weight 20% !!!!!!!!!!
	// sum the averages to the distributions with weight 20% !!!!!!!!!!
	
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		for (i32s out2 = 0;out2 < 20;out2++)
		{
			for (i32s i1 = 0;i1 < SIZE2;i1++)
			{
				f64 sum = 0.0;
				for (i32s i2 = 0;i2 < SIZE3;i2++)
				{
					sum += loop_dip_data[out1][out2][i1][i2];
				}
				
				for (i32s i2 = 0;i2 < SIZE3;i2++)
				{
					loop_dip_data[out1][out2][i1][i2] += 0.10 * sum * average_1st[out1][i1][i2];
					loop_dip_data[out1][out2][i1][i2] += 0.10 * sum * average_2nd[out2][i1][i2];
				}
			}
		}
	}
	
	// compute...
	// compute...
	// compute...
	
	f64 xdata[SIZE2];
	for (i32s i1 = 0;i1 < SIZE2;i1++)
	{
		xdata[i1] = -M_PI + M_PI / (f64) SIZE2 + 2.0 * ((f64) i1) * M_PI / (f64) SIZE2;
	}
	
	f64 ydata[SIZE3];
	for (i32s i2 = 0;i2 < SIZE3;i2++)
	{
		ydata[i2] = -M_PI + M_PI / (f64) SIZE3 + 2.0 * ((f64) i2) * M_PI / (f64) SIZE3;
	}
	
	f64 ycount[SIZE2 * SIZE3];
	for (i32s out1 = 0;out1 < 20;out1++)
	{
		for (i32s out2 = 0;out2 < 20;out2++)
		{
			for (i32s i1 = 0;i1 < SIZE2;i1++)
			{
				for (i32s i2 = 0;i2 < SIZE3;i2++)
				{
					ycount[i1 * SIZE3 + i2] = loop_dip_data[out1][out2][i1][i2];
				}
			}
			
		//	ddsolver dds1(-2.0, (f64 *) xdata, (f64 *) ydata, (f64 *) ycount);	// NOT USED!!!
		//	ddsolver dds2(-1.0, (f64 *) xdata, (f64 *) ydata, (f64 *) ycount);	// NOT USED!!!
			ddsolver dds3(0.0, (f64 *) xdata, (f64 *) ydata, (f64 *) ycount);
			ddsolver dds4(+1.0, (f64 *) xdata, (f64 *) ydata, (f64 *) ycount);
		//	ddsolver dds5(+2.0, (f64 *) xdata, (f64 *) ydata, (f64 *) ycount);	// NOT USED!!!
			
			for (i32s opt = 0;opt < 100;opt++)
			{
				cout << opt << " ";
			//	dds1.TakeCGStep(conjugate_gradient::Newton2Num); cout << dds1.GetValue() << " ";
			//	dds2.TakeCGStep(conjugate_gradient::Newton2Num); cout << dds2.GetValue() << " ";
				dds3.TakeCGStep(conjugate_gradient::Newton2Num); cout << dds3.GetValue() << " ";
				dds4.TakeCGStep(conjugate_gradient::Newton2Num); cout << dds4.GetValue() << " ";
			//	dds5.TakeCGStep(conjugate_gradient::Newton2Num); cout << dds5.GetValue() << " ";
				cout << endl;
			}
			
			ddsolver * mref = NULL;
			bool mt1; bool mt2; bool mt3; bool mt4;
			
		//	mt1 = (dds1.GetValue() < dds2.GetValue()); mt2 = (dds1.GetValue() < dds3.GetValue());
		//	mt3 = (dds1.GetValue() < dds4.GetValue()); mt4 = (dds1.GetValue() < dds5.GetValue());
		//	if (mt1 && mt2 && mt3 && mt4) mref = & dds1;
		//	
		//	mt1 = (dds2.GetValue() < dds1.GetValue()); mt2 = (dds2.GetValue() < dds3.GetValue());
		//	mt3 = (dds2.GetValue() < dds4.GetValue()); mt4 = (dds2.GetValue() < dds5.GetValue());
		//	if (mt1 && mt2 && mt3 && mt4) mref = & dds2;
		//	
		//	mt1 = (dds3.GetValue() < dds1.GetValue()); mt2 = (dds3.GetValue() < dds2.GetValue());
		//	mt3 = (dds3.GetValue() < dds4.GetValue()); mt4 = (dds3.GetValue() < dds5.GetValue());
		//	if (mt1 && mt2 && mt3 && mt4) mref = & dds3;
		//	
		//	mt1 = (dds4.GetValue() < dds1.GetValue()); mt2 = (dds4.GetValue() < dds2.GetValue());
		//	mt3 = (dds4.GetValue() < dds3.GetValue()); mt4 = (dds4.GetValue() < dds5.GetValue());
		//	if (mt1 && mt2 && mt3 && mt4) mref = & dds4;
		//	
		//	mt1 = (dds5.GetValue() < dds1.GetValue()); mt2 = (dds5.GetValue() < dds2.GetValue());
		//	mt3 = (dds5.GetValue() < dds3.GetValue()); mt4 = (dds5.GetValue() < dds4.GetValue());
		//	if (mt1 && mt2 && mt3 && mt4) mref = & dds5;
			
			mt3 = (dds3.GetValue() < dds4.GetValue());
			if (mt3) mref = & dds3; else mref = & dds4;
			
			f64 c1 = mref->var[0]; f64 c2 = mref->var[1]; f64 c3 = mref->var[2];
			f64 s1 = mref->var[3]; f64 s2 = mref->var[4]; f64 s3 = mref->var[5];
			f64 kk = mref->var[6]; f64 k = mref->var[7];
			
			cout << out1 << " " << out2 << " ::: ";
			cout << c1 << " " << c2 << " " << c3 << " ";
			cout << s1 << " " << s2 << " " << s3 << " ";
			cout << kk << " " << k << endl;
			
			of2 << symbols[out1] << symbols[out2] << " ";
			of2 << c1 << " " << c2 << " " << c3 << " ";
			of2 << s1 << " " << s2 << " " << s3 << " ";
			of2 << kk << " " << k << endl;
		}
	}
			
	of2 << "end"; of2.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 270800 : gather information from the dipole term #3 ERRORS.
	
	ifstream if1;
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	f64 total_error1 = 0.0; f64 total_error2 = 0.0;
	i32s total_count = 0; i32s bad_count = 0;
	
	//////////
	
	for (i32s lll = 0;lll < 1;lll++)
	{
		if1.open("../../pconv2_dec_1998/files.txt", ios::in);
		
		while (true)
		{
			if (if1.peek() == 'e') break;
			
			for (i32s n1 = 0;n1 < 55;n1++) if1.get();
			if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
			
			ostrstream str(buffer2, sizeof(buffer2));
			str << "../../pconv2_dec_1998/" << buffer1 << ends;
			
			cout << buffer2 << endl;
			
			mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
			ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
			
			for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
			{
				for (i32s d2 = 1;d2 < mdl2->chn_vector[d1].res_vector.size() - 2;d2++)
				{
					fGL * c1 = mdl2->chn_vector[d1].res_vector[d2 - 1].crd_vector[0][0].data;
					fGL * c2 = mdl2->chn_vector[d1].res_vector[d2].crd_vector[0][0].data;
					fGL * c3 = mdl2->chn_vector[d1].res_vector[d2 + 1].crd_vector[0][0].data;
					
					v3d<fGL> v1a = v3d<fGL>(c2, c1);
					v3d<fGL> v2a = v3d<fGL>(c2, c3);
					
					v3d<fGL> v3a = v1a - v2a * (v1a.spr(v2a) / v2a.spr(v2a)); v3a = v3a / v3a.len();
					v3d<fGL> v4a = v1a.vpr(v2a); v4a = v4a / v4a.len();
					
					f64 k1a = cos(mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0]);
					f64 k2a = sin(mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0]);
					v3d<fGL> v5a = v3a * k1a - v4a * k2a;
					
					v3d<fGL> pv1 = v3d<fGL>(c2); pv1 = pv1 + v2a / 2.0;
					
					bool test3a = (mdl2->chn_vector[d1].res_vector[d2+0].state == STATE_LOOP);
					bool test3b = (mdl2->chn_vector[d1].res_vector[d2+1].state == STATE_LOOP);
					if (test3a && test3b)		// and operator is needed here...
					{
						fGL * c4 = mdl2->chn_vector[d1].res_vector[d2 + 2].crd_vector[0][0].data;
						v3d<fGL> v3t = v3d<fGL>(c3, c4);
						
						f64 tor = v1a.tor(v2a, v3t);
						f64 dip = mdl2->chn_vector[d1].res_vector[d2].peptide_vector[0];
						
						ifstream file;		// common...
						
						f64 wzz1[8];
						file.open("/tmp/loopdip.txt", ios::in);
						if (file.fail()) cout << "can't open loopdip.txt!!!" << endl;
						while (true)
						{
							if (file.peek() == 'e') exit(EXIT_FAILURE);
							
							char buffer[256]; char tp1; char tp2; file >> tp1 >> tp2;
							bool test1 = (tp1 != mdl2->chn_vector[d1].res_vector[d2].symbol);
							bool test2 = (tp2 != mdl2->chn_vector[d1].res_vector[d2 + 1].symbol);
							if (test1 || test2) file.getline(buffer, sizeof(buffer));
							else
							{
								f64 value;
								file >> value; wzz1[0] = value;
								file >> value; wzz1[1] = value;
								file >> value; wzz1[2] = value;
								file >> value; wzz1[3] = value;
								file >> value; wzz1[4] = value;
								file >> value; wzz1[5] = value;
								file >> value; wzz1[6] = value;
								file >> value; wzz1[7] = value;
								break;
							}
						}
						
						file.close();		// loopdip.txt
						
						f64 tzz1 = wzz1[0] * cos(tor) + wzz1[1] * cos(2.0 * tor) + wzz1[2] * cos(3.0 * tor);
						tzz1 += wzz1[3] * sin(tor) + wzz1[4] * sin(2.0 * tor) + wzz1[5] * sin(3.0 * tor);
						tzz1 += tor * wzz1[6] + wzz1[7];
						
						f64 compare = tzz1;
						
						f64 diff = dip - compare;
						if (diff > +M_PI) diff = 2.0 * M_PI - diff;
						else if (diff < -M_PI) diff = 2.0 * M_PI + diff;
						
						total_error1 += fabs(diff);
						total_error2 += diff * diff;
						total_count++;
						
						if (fabs(diff) > M_PI / 2.0) bad_count++;
					}
				}
			}
			
			delete mdl2;
		}
		
		if1.close();
	}
	
	cout << "total_error1 = " << total_error1 << endl;
	cout << "total_error2 = " << total_error2 << endl;
	cout << "total_count = " << total_count << endl << endl;
	
	cout << "rms-error = " << sqrt(total_error2 / total_count) << " rad" << endl;
	cout << "average error = " << (total_error1 / total_count) << " rad" << endl;
	cout << "there are " << (100.0 * bad_count / total_count) << " % bad predictions" << endl;

//*/
/*##############################################*/
/*##############################################*/
/*

	// 060799 : gather information from the VDW-radii...

////////////////////////////////////////////////////////////
// will eat tons of memory -> skip some NB-terms!!! append the following to mm2_eng::InitNBT1():
//fGL * c1 = GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1]].crd_vector[atm[0][2]][0].data;
//fGL * c2 = GetModel()->chn_vector[atm[1][0]].res_vector[atm[1][1]].crd_vector[atm[1][2]][0].data;
//v3d<fGL> v1 = v3d<fGL>(c1, c2); if (v1.len() > 1.5) return false;
////////////////////////////////////////////////////////////

	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output8.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 20
	
	const f64 vdwr_dist_range[2] = { 0.2, 1.0 };	// do not edit!!! has dependencies later...
	
	i32s * vdwr_dist_data = new i32s[SIZE1 * TYPES * TYPES];
	for (i32s init = 0;init < SIZE1 * TYPES * TYPES;init++) vdwr_dist_data[init] = 0;
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		mm2_eng * eng = new mm2_eng(* mdl2);
		
		for (i32s d1 = 0;d1 < eng->nbt1_vector.size();d1++)
		{
			mm2_atm & at1 = eng->index_vector[eng->nbt1_vector[d1].atmi[0]];
			
			char tmp1a = mdl2->chn_vector[at1[0]].res_vector[at1[1]].symbol;
			i32s tmp1b = 0; while (tmp1b < 20 && symbols[tmp1b] != tmp1a) tmp1b++;
			if (tmp1b == 20) { cout << "symbol error #1"; exit(EXIT_FAILURE); }
			
			tmp1b++; tmp1b = tmp1b << 8; tmp1b += at1[2];
			i32s tmp1c = 0; while (tmp1c < TYPES && types[tmp1c] != tmp1b) tmp1c++;
			if (tmp1c == TYPES) { cout << "type error #1"; exit(EXIT_FAILURE); }
			
			mm2_atm & at2 = eng->index_vector[eng->nbt1_vector[d1].atmi[1]];
			
			char tmp2a = mdl2->chn_vector[at2[0]].res_vector[at2[1]].symbol;
			i32s tmp2b = 0; while (tmp2b < 20 && symbols[tmp2b] != tmp2a) tmp2b++;
			if (tmp2b == 20) { cout << "symbol error #2"; exit(EXIT_FAILURE); }
			
			tmp2b++; tmp2b = tmp2b << 8; tmp2b += at2[2];
			i32s tmp2c = 0; while (tmp2c < TYPES && types[tmp2c] != tmp2b) tmp2c++;
			if (tmp2c == TYPES) { cout << "type error #2"; exit(EXIT_FAILURE); }
			
			fGL * c1 = mdl2->chn_vector[at1[0]].res_vector[at1[1]].crd_vector[at1[2]][0].data;
			fGL * c2 = mdl2->chn_vector[at2[0]].res_vector[at2[1]].crd_vector[at2[2]][0].data;
			v3d<fGL> v1 = v3d<fGL>(c1, c2); f64 t1 = v1.len();
			
			if (t1 < vdwr_dist_range[0]) continue;
			if (t1 > vdwr_dist_range[1]) continue;
			
			f64 t2 = (t1 - vdwr_dist_range[0]) / (vdwr_dist_range[1] - vdwr_dist_range[0]);
			if (t2 < 1.0) vdwr_dist_data[SIZE1 * (tmp1c * TYPES + tmp2c) + ((i32s) (SIZE1 * t2))]++;
			if (t2 < 1.0) vdwr_dist_data[SIZE1 * (tmp2c * TYPES + tmp1c) + ((i32s) (SIZE1 * t2))]++;
		}
		
		delete eng;
		delete mdl2;
	}
	
	if1.close();
	
	// convert the observed data cumulative...
	// convert the observed data cumulative...
	// convert the observed data cumulative...
	
	for (i32s t1 = 0;t1 < TYPES;t1++)
	{
		for (i32s t2 = t1;t2 < TYPES;t2++)
		{
			of1 << t1 << " " << t2 << " ::: ";
			
			i32s sigma = 0;
			for (i32s t3 = 0;t3 < SIZE1;t3++)
			{
				sigma += vdwr_dist_data[SIZE1 * (t1 * TYPES + t2) + t3];
				vdwr_dist_data[SIZE1 * (t1 * TYPES + t2) + t3] = sigma;
				
				of1 << sigma << " ";
			} of1 << endl;
		}
	}
	
	// radius fitting: search the distance where we have more than X observations,
	// fit a line to five nearest points, and get the radius using this line.
	
	const i32s limit = 100;
	
	f64 result[TYPES][TYPES];
	for (i32s t1 = 0;t1 < TYPES;t1++)
	{
		for (i32s t2 = t1;t2 < TYPES;t2++)
		{
			i32s count = 0;
			while (count < SIZE1)
			{
				if (vdwr_dist_data[SIZE1 * (t1 * TYPES + t2) + count] > limit) break;
				else count++;
			}
			
			if (count < 1 || count > 18) continue;
			
			// here the formula for "xdata" is dependent on "vdwr_dist_range[2]" above!!!
			// here the formula for "xdata" is dependent on "vdwr_dist_range[2]" above!!!
			// here the formula for "xdata" is dependent on "vdwr_dist_range[2]" above!!!
			
			f64 xdata[3]; f64 ydata[3];
			for (i32s i = 0;i < 3;i++)
			{
				i32s tmp1 = count + i - 1;
				xdata[i] = 0.22 + ((f64) tmp1) / 19.0 * 0.76;
				ydata[i] = vdwr_dist_data[SIZE1 * (t1 * TYPES + t2) + tmp1];
			}
			
			f64 sx = 0.0; f64 sy = 0.0;
			for (i32s i = 0;i < 3;i++)
			{
				sx += xdata[i];
				sy += ydata[i];
			}
			
			sx /= 3.0; sy /= 3.0;
			f64 sxx = 0.0; f64 sxy = 0.0; f64 syy = 0.0;
			for (i32s i = 0;i < 3;i++)
			{
				sxx += (xdata[i] - sx) * (xdata[i] - sx);
				sxy += (xdata[i] - sx) * (ydata[i] - sy);
				syy += (ydata[i] - sy) * (ydata[i] - sy);
			}
			
			f64 a = sxy / sxx;
			f64 b = sy - a * sx;
			
			result[t1][t2] = result[t2][t1] = (((f64) limit) - b) / a;
		}
	}
	
	for (i32s t1 = 0;t1 < TYPES;t1++)
	{
		for (i32s t2 = 0;t2 < TYPES;t2++)
		{
			of1 << result[t1][t2] << " ";
		} of1 << endl;
	}
	
	// build the normal equations...
	
	f64 m1[TYPES * TYPES]; f64 v2[TYPES];
	for (i32s i1 = 0;i1 < TYPES;i1++)
	{
		v2[i1] = 0.0;
		for (i32s i2 = 0;i2 < TYPES;i2++) m1[i1 * TYPES + i2] = 0.0;
	}
	
	for (i32s t1 = 0;t1 < TYPES;t1++)
	{
		for (i32s t2 = 0;t2 < TYPES;t2++)
		{
			f64 tmp[TYPES];
			for (i32s t3 = 0;t3 < TYPES;t3++) tmp[t3] = 0.0;
			tmp[t1] += 1.0; tmp[t2] += 1.0;
			
			for (i32s i2 = 0;i2 < TYPES;i2++)
			{
				v2[i2] += tmp[i2] * result[t1][t2];
				for (i32s i3 = 0;i3 < TYPES;i3++) m1[i2 * TYPES + i3] += tmp[i2] * tmp[i3];
			}
		}
	}
	
	// here we add some constraints to get more stable and reasonable distribution...
	// here we add some constraints to get more stable and reasonable distribution...
	// here we add some constraints to get more stable and reasonable distribution...
	
	const i32s nc1 = 14;
	for (i32s c1 = 0;c1 < 50;c1++)		// weight = 50
	{
		i32s t1[nc1] =
		{
			1, 4, 6, 8, 10, 12, 15,
			17, 19, 21, 24, 26, 31, 34
		};
		
		for (i32s c2 = 0;c2 < nc1 - 1;c2++)
		{
			for (i32s c3 = c2 + 1;c3 < nc1;c3++)
			{
				f64 tmp[TYPES];
				for (i32s t3 = 0;t3 < TYPES;t3++) tmp[t3] = 0.0;
				tmp[t1[c2]] += 1.0; tmp[t1[c3]] -= 1.0;
				
				for (i32s i2 = 0;i2 < TYPES;i2++)
				{
					v2[i2] += tmp[i2] * 0.0;
					for (i32s i3 = 0;i3 < TYPES;i3++) m1[i2 * TYPES + i3] += tmp[i2] * tmp[i3];
				}
			}
		}
	}
	
	const i32s nc2 = 19;
	for (i32s c1 = 0;c1 < 15;c1++)		// weight = 15
	{
		i32s t1[nc2] =
		{
			1, 4, 6, 8, 9, 10, 12, 15, 16, 17,
			19, 21, 24, 25, 26, 31, 32, 33, 34
		};
		
		for (i32s c2 = 0;c2 < nc2;c2++)
		{
			f64 tmp[TYPES];
			for (i32s t3 = 0;t3 < TYPES;t3++) tmp[t3] = 0.0;
			tmp[t1[c2]] += 1.0;
			
			for (i32s i2 = 0;i2 < TYPES;i2++)
			{
				v2[i2] += tmp[i2] * 0.0;
				for (i32s i3 = 0;i3 < TYPES;i3++) m1[i2 * TYPES + i3] += tmp[i2] * tmp[i3];
			}
		}
	}
	
	gauss_elimination<f64> ge(TYPES, m1, v2); ge.solve();
	for (i32s i1 = 0;i1 < TYPES;i1++)
	{
		of1 << "type " << i1 << " VDWR = " << ge.getvalue(i1) << endl;
	}
	
	f64 error = 0.0; i32s cnt = 0;
	for (i32s t1 = 0;t1 < TYPES;t1++)
	{
		for (i32s t2 = 0;t2 < TYPES;t2++)
		{
			f64 value = ge.getvalue(t1) + ge.getvalue(t2);
			error += pow(result[t1][t2] - value, 2.0); cnt++;
		}
	}
	
	of1 << "RMS-error = " << (sqrt(error) / (f64) cnt) << endl;
	
	of1 << "END";
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 130100 : gather information from the surface areas...
	
// here you have to check some things from mm2_eng::ComputeBT3() that are normally commented out!!!!
// here you have to check some things from mm2_eng::ComputeBT3() that are normally commented out!!!!
// here you have to check some things from mm2_eng::ComputeBT3() that are normally commented out!!!!

////////////////////////////////////////////////////////////
// will eat tons of memory -> skip some NB-terms!!! append the following to mm2_eng::InitNBT1():
//fGL limit = vdwr[ref->atmi[0]] + vdwr[ref->atmi[1]] + 2.0 * GetModel()->prm.solvrad + 0.5;
//fGL * c1 = GetModel()->chn_vector[atm[0][0]].res_vector[atm[0][1]].crd_vector[atm[0][2]][0].data;
//fGL * c2 = GetModel()->chn_vector[atm[1][0]].res_vector[atm[1][1]].crd_vector[atm[1][2]][0].data;
//v3d<fGL> v1 = v3d<fGL>(c1, c2); if (v1.len() > limit) return false;
////////////////////////////////////////////////////////////

	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	ofstream of1("/tmp/output9.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
	#define SIZE1 50
	
	const f64 sasa_range[2] = { 0.0, 2.5 };		// 2.5
	
	i32s * sasa_data = new i32s[SIZE1 * TYPES];
	for (i32s init = 0;init < SIZE1 * TYPES;init++) sasa_data[init] = 0;
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str(buffer2, sizeof(buffer2));
		str << "../../pconv2_dec_1998/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		ifstream if2(buffer2, ios::in); mdl2->ReadStream(if2); if2.close();
		
		mm2_eng * eng = new mm2_eng(* mdl2);
		CopyCRD(mdl2, eng, 0); eng->Compute(0);
		
		for (i32s n1 = 0;n1 < eng->index_vector.size();n1++)
		{
			mm2_atm & at1 = eng->index_vector[n1];
			
			char tmp1 = mdl2->chn_vector[at1[0]].res_vector[at1[1]].symbol;
			i32s tmp2 = 0; while (tmp2 < 20 && symbols[tmp2] != tmp1) tmp2++;
			if (tmp2 == 20) { cout << "symbol error #2"; exit(EXIT_FAILURE); }
			
			tmp2++; tmp2 = tmp2 << 8; tmp2 += at1[2];
			i32s tmp3 = 0; while (tmp3 < TYPES && types[tmp3] != tmp2) tmp3++;
			if (tmp3 == TYPES) { cout << "type error #2"; exit(EXIT_FAILURE); }
			
			// we just assume that the areas are written in mm2_eng::sasa[]!!!!
			// we just assume that the areas are written in mm2_eng::sasa[]!!!!
			// we just assume that the areas are written in mm2_eng::sasa[]!!!!
			
			f64 t1 = eng->sasa[n1] * eng->vdwr2[n1];
			
			if (t1 < sasa_range[0]) continue;
			if (t1 > sasa_range[1]) continue;
			
			f64 t2 = (t1 - sasa_range[0]) / (sasa_range[1] - sasa_range[0]);
			if (t2 < 1.0) sasa_data[tmp3 * SIZE1 + ((i32s) (SIZE1 * t2))]++;
		}
		
		delete eng;
		delete mdl2;
	}
	
	if1.close();
	
	of1 << "SASA-distribution: " << sasa_range[0] << " -> " << sasa_range[1] << endl;
	for (i32s out1 = 0;out1 < TYPES;out1++)
	{
		of1 << out1 << " which is 0x" << hex << types[out1] << dec << " -> ";
		for (i32s out2 = 0;out2 < SIZE1;out2++)
		{
			of1 << sasa_data[out1 * SIZE1 + out2] << " ";
		} of1 << endl;
	}
	
	of1 << "END";
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 300800 : run through the decoy sets test...
	
// decoy-S-misfold
// decoy-M-4state_reduced
// decoy-M-hg_structal
// decoy-M-lmds
// decoy-M-lattice_ssfit
// decoy-M-fisa_all

	ifstream if1("../../decoy-M-4state_reduced/files1.txt", ios::in);
	ofstream of1("../../decoy-M-4state_reduced1.txt", ios::out);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		char typeflag1 = if1.peek();
		char typeflag2 = (typeflag1 != '-' ? 'n' : 'd');
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		ostrstream str1(buffer2, sizeof(buffer2));
		str1 << "../../decoy-M-4state_reduced/" << buffer1 << ends;
		
		cout << buffer2 << endl;
		
		mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		
		ifstream in_f(buffer2, ios::in);
		mdl2->ReadStream(in_f); in_f.close();
		
		mm2_eng * eng = new mm2_eng(* mdl2); CopyCRD(mdl2, eng, 0);
		mm2_geomopt * opt = new mm2_geomopt(eng, 50, 0.025);		// optimal settings?!?!?
		
		for (i32s n2 = 0;n2 < 500;n2++)
		{
			opt->TakeCGStep(conjugate_gradient::Newton2An);
			if (!(n2 % 10)) cout << "*" << flush;
		}
		
		delete opt;
		
		eng->Compute(0);
		
		of1 << typeflag2 << " ";
		
		of1 << eng->energy_bt1 << " ";			// should be ok...
		of1 << eng->energy_bt2 << " ";			// should be ok...
		of1 << eng->energy_bt3 << " ";			// should be ok...
		of1 << eng->energy_bt4 << " ";			// should be ok...
		of1 << eng->energy_nbt1a << " ";		// should be ok...
		of1 << eng->energy_nbt1b << " ";		// should be ok...
		of1 << eng->energy_nbt2a << " ";		// should be ok...
		of1 << eng->energy_nbt2b << " ";		// should be ok...
		of1 << eng->energy_nbt3 << " ";			// should be ok...
		
		i32s hdata[20]; i32s sdata[20]; i32s ldata[20];
		
		for (i32s ii = 0;ii < 20;ii++)
		{
			hdata[ii] = 0;
			sdata[ii] = 0;
			ldata[ii] = 0;
		}
		
		for (i32s d1 = 0;d1 < mdl2->chn_vector.size();d1++)
		{
			for (i32s d2 = 0;d2 < mdl2->chn_vector[d1].res_vector.size();d2++)
			{
				i32s ind = 0; while (ind < 20 && symbols[ind] != mdl2->chn_vector[d1].res_vector[d2].symbol) ind++;
				if (ind == 20) { cout << "FATAL ERROR: unknown residue symbol !!!" << endl; exit(EXIT_FAILURE); }
				
				switch (mdl2->chn_vector[d1].res_vector[d2].state)
				{
					case STATE_HELIX:	hdata[ind]++; break;
					case STATE_STRAND:	sdata[ind]++; break;
					default:		ldata[ind]++;
				}
			}
		}
		
		for (i32s ii = 0;ii < 20;ii++) of1 << hdata[ii] << " ";
		for (i32s ii = 0;ii < 20;ii++) of1 << sdata[ii] << " ";
		for (i32s ii = 0;ii < 20;ii++) of1 << ldata[ii] << " ";
		
		of1 << buffer2 << endl;
		
		delete eng;
		delete mdl2;
	}
	
	if1.close();
	of1.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 310800 : secondary structure energetics (2nd generation)...
	
// decoy-S-misfold
// decoy-M-4state_reduced
// decoy-M-hg_structal
// decoy-M-lmds
// decoy-M-lattice_ssfit
// decoy-M-fisa_all

	decoys_opt opt;
		
	opt.AddFile("../../decoy-S-misfold.txt");
	opt.AddFile("../../decoy-M-4state_reduced.txt");
	opt.AddFile("../../decoy-M-hg_structal.txt");
	opt.AddFile("../../decoy-M-lmds.txt");
	opt.AddFile("../../decoy-M-lattice_ssfit.txt");
	opt.AddFile("../../decoy-M-fisa_all.txt");
	opt.UpdateRMS();
	
	//opt.Check(); exit(-1);
	
	cout << opt.GetValue() << endl;
	for (i32s n1 = 0;n1 < 1000;n1++)
//////////////
//////////////
//ofstream ddd; ddd.open("/home/thassine/PLOT-sqrt.TXT", ios::out);
//plottingloop:
//for (i32s ii = 0;ii < 20;ii++) decoys_dset::w_helix[ii] = 0.0;
//for (i32s ii = 0;ii < 20;ii++) decoys_dset::w_strand[ii] = 0.0;
//for (i32s ii = 0;ii < 20;ii++) decoys_dset::w_loop[ii] = 0.0;
//for (i32s n1 = 0;n1 < 500;n1++)
//////////////
	{
		opt.TakeCGStep(conjugate_gradient::Newton2An);
		cout << n1 << " " << opt.GetValue() << " " << opt.optstp << endl;
	}
//////////////
//opt.GetValue();
//ddd << "ccc = " << ccc << " badcount = " << decoys_dset::badcount << " ";
//ddd << "helix "; for (i32s ii = 0;ii < 20;ii++) ddd << decoys_dset::w_helix[ii] << " ";
//ddd << "strand "; for (i32s ii = 0;ii < 20;ii++) ddd << decoys_dset::w_strand[ii] << " ";
//ddd << "loop "; for (i32s ii = 0;ii < 20;ii++) ddd << decoys_dset::w_loop[ii] << " ";
//ddd << endl;
//ccc += 1.0e-05;
//if (ccc < 0.001) goto plottingloop;
//else { ddd.close(); return 0; }
//////////////
//////////////

	cout << endl;
	cout << "learning set RESULTS:" << endl;
	
	opt.PrintResults(cout);
	
	cout << "give an int to continue... ";
//int qqq1; cin >> qqq1;

//	opt.ClearAll();
//	
//	opt.AddFile("../../decoy-S-misfold.txt");
//	opt.AddFile("../../decoy-M-4state_reduced.txt");
//	opt.AddFile("../../decoy-M-hg_structal.txt");
//	opt.AddFile("../../decoy-M-lmds.txt");
//	opt.AddFile("../../decoy-M-lattice_ssfit.txt");
//	opt.AddFile("../../decoy-M-fisa_all.txt");
//	opt.UpdateRMS();
//	
//	cout << opt.GetValue() << endl;
//	
//	cout << "testing set RESULTS:" << endl;
//	
//	opt.PrintResults(cout);
//	
//	cout << "give an int to continue... ";
//int qqq2; cin >> qqq2;

	// make a scatterplot (relative to native); approximation - assume geomopt won't change geometry much...
	// make a scatterplot (relative to native); approximation - assume geomopt won't change geometry much...
	// make a scatterplot (relative to native); approximation - assume geomopt won't change geometry much...
	
	ofstream scatterfile;
	scatterfile.open("/wrk/thassine/decoy-tmp/final5.txt", ios::out);
	opt.WriteScatter(scatterfile);
	scatterfile.close();
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 160300 : massive testing for the full training set...
	
	ifstream if1("../../pconv2_dec_1998/files.txt", ios::in);
	
	mm2_mdl * mdl2;
	
	char buffer1[256];
	char buffer2[65536];
	
	//////////
	
//OSMesaContext ctx; ctx = OSMesaCreateContext(GL_RGBA, NULL);
//char * fb = new char[320 * 240 * 4]; OSMesaMakeCurrent(ctx, fb, GL_UNSIGNED_BYTE, 320, 240);
	
	rms_param prm1;
	prm1.osteps = 500; prm1.cfrq = 1; prm1.csize = 100;
	prm1.start = 10000; prm1.equil = 10000; prm1.simul = 10000;
	prm1.temp = 300.0; prm1.timestep = 5.0;
	
	//////////
	
	while (true)
	{
		if (if1.peek() == 'e') break;
		
		for (i32s n1 = 0;n1 < 55;n1++) if1.get();
		if1 >> buffer1; if1.getline(buffer2, sizeof(buffer2));
		
		i32s zzz = 0;
		while (buffer1[zzz] != '.') zzz++;
		buffer1[zzz] = 0;
		
		// check the path in getRMSdistance() !!!!!
		// check the path in getRMSdistance() !!!!!
		// check the path in getRMSdistance() !!!!!
		
		getRMSdistance(buffer1, & prm1, NULL);
	}
	
	if1.close();
	
//OSMesaDestroyContext(ctx); delete[] fb;
	
//*/
/*##############################################*/
/*##############################################*/
/*

	// 250699 : calculate RMS-errors for the small training set.
	
//OSMesaContext ctx; ctx = OSMesaCreateContext(GL_RGBA, NULL);
//char * fb = new char[320 * 240 * 4]; OSMesaMakeCurrent(ctx, fb, GL_UNSIGNED_BYTE, 320, 240);
	
	// 10000 equil-steps is not 100% sure... 30000 ?!?!?! WILL BE NEEDED FOR REFINEMENT!!!
	// 10000 equil-steps is not 100% sure... 30000 ?!?!?! WILL BE NEEDED FOR REFINEMENT!!!
	// 10000 equil-steps is not 100% sure... 30000 ?!?!?! WILL BE NEEDED FOR REFINEMENT!!!
	
	rms_param prm1;
	prm1.osteps = 500; prm1.cfrq = 1; prm1.csize = 100;
	prm1.start = 10000; prm1.equil = 10000; prm1.simul = 10000;
	prm1.temp = 300.0; prm1.timestep = 5.0;
	
	getRMSdistance("1abv", & prm1, NULL);		// all alpha
	getRMSdistance("2asr", & prm1, NULL);
	
	getRMSdistance("1aac", & prm1, NULL);		// all beta
	getRMSdistance("4fgf", & prm1, NULL);
	
	getRMSdistance("2trx", & prm1, NULL);		// small alpha&beta
	getRMSdistance("4rhn", & prm1, NULL);
	
	getRMSdistance("1bpi", & prm1, NULL);		// small protein
	
	getRMSdistance("1gar", & prm1, NULL);		// the big ones...
	getRMSdistance("1hcl", & prm1, NULL);
	getRMSdistance("2act", & prm1, NULL);
	
	//////////
	
//	getRMSdistance("bad-1azo_", & prm1, NULL);
//	getRMSdistance("bad-1bd8_", & prm1, NULL);
//	getRMSdistance("bad-1ois_", & prm1, NULL);
	
//OSMesaDestroyContext(ctx); delete[] fb;
	
	return 0;
	
//*/
/*##############################################*/
/*##############################################*/
///*

	// 250699 : random-search parameter refinement for the small test set...
	
	// originally this was used to randomly test parameters and to so optimize the parameter sets.
	// however, it seems that when a certain level is reached, it is very difficult to get any
	// better results because the results are very badly dependent on parameters; the target
	// function is very very noisy!!!
	
	// later this is modified to calculate average energies of slightly different parameter sets,
	// to overcome the above problem.
	
//OSMesaContext ctx; ctx = OSMesaCreateContext(GL_RGBA, NULL);
//char * fb = new char[320 * 240 * 4]; OSMesaMakeCurrent(ctx, fb, GL_UNSIGNED_BYTE, 320, 240);
	
	MPI::Init(argc, argv);
	int mpi_rank = MPI::COMM_WORLD.Get_rank();
	int mpi_size = MPI::COMM_WORLD.Get_size();
	
	rms_param prm1;
	prm1.osteps = 500; prm1.cfrq = 1; prm1.csize = 100;
	prm1.start = 10000; prm1.equil = 10000; prm1.simul = 10000;
	prm1.temp = 300.0; prm1.timestep = 5.0;
	
	const i32s popsize = mpi_size * 1;	// must be a multiple of mpi_size!!!!!!
	i32s count = popsize / mpi_size; i32s nbytes = count * sizeof(mm2_eng_param);
	
	mm2_eng_param * gpop;
	mm2_eng_param * lpop = new mm2_eng_param[count];
	
	ofstream * mlog;
	
	if (!mpi_rank)
	{
		srand(time(NULL));	// initialize the random numbers!!!
		mlog = new ofstream("/home/thassine/master.log", ios::out);
		
		gpop = new mm2_eng_param[popsize + 1];
		
gpop[popsize].rms = 0.0;

// this is the initial parameter set...
// usually copied from mm2_mdl-ctor.

gpop[popsize].wtmp1 = 0.875;			// 0.875	// bt2 fc weight (in strands and loops)
gpop[popsize].wtmp2 = 1.000;			// 1.000	// bt4 fc weight
gpop[popsize].wtmp3 = 0.00025;			// 0.00025	// angle rep fc

gpop[popsize].wtor1 = 2.75;			// 2.75
gpop[popsize].wtor2 = 3.25;			// 3.25
gpop[popsize].lenjon = 1.2550;			// 1.2550	// DOMINATING LJ-TERM -> LOST SENSITIVITY ?!?!?!?
gpop[popsize].vdwrad = 1.35;			// 1.35		// will give an average radius of 0.275 nm...
gpop[popsize].screen1 = -20.929;			// -20.929
gpop[popsize].screen2 = 78.4 - gpop[popsize].screen1;	// 78.4 minus above...
gpop[popsize].screen3 = 3.4781;			// 3.4781
gpop[popsize].screen4 = -0.01787;			// -0.01787
gpop[popsize].screen5 = 0.001;			// 0.001	// about 4 ...
gpop[popsize].dipole1 = 3.0;			// 3.0		// will give 27 kJ/mol for HB's in SSE's. (27+7)/2 = 17
gpop[popsize].dipole2 = 4.0;			// 4.0		// EASY TO CHECK USING A SINGLE ALPHA-HELIX, LIKE ALA12!!!
gpop[popsize].pbility1 = 0.0025;			// 0.0025	// should be about 0.0015 : CRC handbook of Chem & Phys
gpop[popsize].pbility2 = 0.0005;			// 0.0005	// 1st Student Edition, 1988, CRC Press Inc. (page E-62)
gpop[popsize].sasa1 = -0.5100;			// -0.5100	// near -0.5 -> in the middle, both effects?!?!?!
gpop[popsize].sasa2 = -52.00;			// -52.00	// somewhere near -50.0 !??!?!?!?!
gpop[popsize].solvrad = 0.15;			// 0.15		// H2O -> 0.15
	}
	
	while (true)
	{
		if (!mpi_rank)
		{
			(* mlog) << "starting a new cycle!!!" << endl;
			
			for (i32s n1 = 0;n1 < popsize;n1++)
			{
				gpop[n1].wtmp1 = gpop[popsize].wtmp1		;//+ RAND_f64 * 1.0;
				gpop[n1].wtmp2 = gpop[popsize].wtmp2		;//+ RAND_f64 * 1.0;
				gpop[n1].wtmp3 = gpop[popsize].wtmp3		;//+ RAND_f64 * 0.0;	// difficult!!!
				
				gpop[n1].wtor1 = gpop[popsize].wtor1		;//+ RAND_f64 * 0.5;
				gpop[n1].wtor2 = gpop[popsize].wtor2		;//+ RAND_f64 * 0.5;
				gpop[n1].lenjon = gpop[popsize].lenjon		+ RAND_f64 * 0.025;	// difficult!!!
				gpop[n1].vdwrad = gpop[popsize].vdwrad		;//+ RAND_f64 * 0.0;	// difficult!!!
				gpop[n1].screen1 = gpop[popsize].screen1	;//+ RAND_f64 * 0.0;	// difficult!!!
				gpop[n1].screen2 = 78.4 - gpop[n1].screen1;
				gpop[n1].screen3 = gpop[popsize].screen3	;//+ RAND_f64 * 0.0;	// difficult!!!
				gpop[n1].screen4 = gpop[popsize].screen4	;//+ RAND_f64 * 0.0;	// difficult!!!
				gpop[n1].screen5 = gpop[popsize].screen5	;//+ RAND_f64 * 0.0;	// difficult!!!
				gpop[n1].dipole1 = gpop[popsize].dipole1	;//+ RAND_f64 * 0.0;	// difficult!!!
				gpop[n1].dipole2 = gpop[popsize].dipole2	;//+ RAND_f64 * 0.0;	// difficult!!!
				gpop[n1].pbility1 = gpop[popsize].pbility1	;//+ RAND_f64 * 0.0030;
				gpop[n1].pbility2 = gpop[popsize].pbility2	;//+ RAND_f64 * 0.0010;
				gpop[n1].sasa1 = gpop[popsize].sasa1		+ RAND_f64 * 0.025;
				gpop[n1].sasa2 = gpop[popsize].sasa2		+ RAND_f64 * 2.5;
				gpop[n1].solvrad = gpop[popsize].solvrad	;//+ RAND_f64 * 0.001;
			}
		}
		
		MPI::COMM_WORLD.Scatter(gpop, nbytes, MPI::CHAR, lpop, nbytes, MPI::CHAR, 0);
		
		// those barriers are not actually needed...
		// create some network traffic using "ping -q -i 60 z?? &" ?!?!?!?!
		// create some network traffic using "ping -q -i 60 z?? &" ?!?!?!?!
		// create some network traffic using "ping -q -i 60 z?? &" ?!?!?!?!
		
		for (i32s n1 = 0;n1 < count;n1++)
		{
			lpop[n1].rms = 0.0; f64 tmp1;
			
			tmp1 = getRMSdistance("1abv", & prm1, & lpop[n1]);		// all alpha
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			tmp1 = getRMSdistance("2asr", & prm1, & lpop[n1]);
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			
			tmp1 = getRMSdistance("1aac", & prm1, & lpop[n1]);		// all beta
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			tmp1 = getRMSdistance("4fgf", & prm1, & lpop[n1]);
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			
			tmp1 = getRMSdistance("2trx", & prm1, & lpop[n1]);		// small alpha&beta
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			tmp1 = getRMSdistance("4rhn", & prm1, & lpop[n1]);
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			
			tmp1 = getRMSdistance("1bpi", & prm1, & lpop[n1]);		// small protein
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			
			tmp1 = getRMSdistance("1gar", & prm1, & lpop[n1]);		// the big ones...
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			tmp1 = getRMSdistance("1hcl", & prm1, & lpop[n1]);
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			tmp1 = getRMSdistance("2act", & prm1, & lpop[n1]);
			lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
			
			//////////
			
		//	tmp1 = getRMSdistance("bad-1azo_", & prm1, & lpop[n1]);
		//	lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
		//	tmp1 = getRMSdistance("bad-1bd8_", & prm1, & lpop[n1]);
		//	lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
		//	tmp1 = getRMSdistance("bad-1ois_", & prm1, & lpop[n1]);
		//	lpop[n1].rms += tmp1 * tmp1; MPI::COMM_WORLD.Barrier();
		}
		
		MPI::COMM_WORLD.Gather(lpop, nbytes, MPI::CHAR, gpop, nbytes, MPI::CHAR, 0);
		
		if (!mpi_rank)
		{
			ofstream dump("/tmp/dumpfile.txt", ios::out);
			
			for (i32s n1 = 0;n1 < popsize;n1++)
			{
				dump << "RMS = " << setw(12) << setprecision(8) << gpop[n1].rms << endl;
				
				dump << "wtmp1 = " << setw(12) << setprecision(8) << gpop[n1].wtmp1 << endl;
				dump << "wtmp2 = " << setw(12) << setprecision(8) << gpop[n1].wtmp2 << endl;
				dump << "wtmp3 = " << setw(12) << setprecision(8) << gpop[n1].wtmp3 << endl;
				
				dump << "wtor1 = " << setw(12) << setprecision(8) << gpop[n1].wtor1 << endl;
				dump << "wtor2 = " << setw(12) << setprecision(8) << gpop[n1].wtor2 << endl;
				dump << "lenjon = " << setw(12) << setprecision(8) << gpop[n1].lenjon << endl;
				dump << "vdwrad = " << setw(12) << setprecision(8) << gpop[n1].vdwrad << endl;
				dump << "screen1 = " << setw(12) << setprecision(8) << gpop[n1].screen1 << endl;
				dump << "screen2 = " << setw(12) << setprecision(8) << gpop[n1].screen2 << endl;
				dump << "screen3 = " << setw(12) << setprecision(8) << gpop[n1].screen3 << endl;
				dump << "screen4 = " << setw(12) << setprecision(8) << gpop[n1].screen4 << endl;
				dump << "screen5 = " << setw(12) << setprecision(8) << gpop[n1].screen5 << endl;
				dump << "dipole1 = " << setw(12) << setprecision(8) << gpop[n1].dipole1 << endl;
				dump << "dipole2 = " << setw(12) << setprecision(8) << gpop[n1].dipole2 << endl;
				dump << "pbility1 = " << setw(12) << setprecision(8) << gpop[n1].pbility1 << endl;
				dump << "pbility2 = " << setw(12) << setprecision(8) << gpop[n1].pbility2 << endl;
				dump << "sasa1 = " << setw(12) << setprecision(8) << gpop[n1].sasa1 << endl;
				dump << "sasa2 = " << setw(12) << setprecision(8) << gpop[n1].sasa2 << endl;
				dump << "solvrad = " << setw(12) << setprecision(8) << gpop[n1].solvrad << endl;
				
				dump << endl;
			}
			
			dump.close();
			
			for (i32s n1 = 0;n1 < popsize;n1++)
			{
				if (!(gpop[n1].rms != gpop[n1].rms)) continue;
				(* mlog) << "param set " << n1 << " failed: " << gpop[n1].rms << endl;
				gpop[n1].rms = 1.0e+15;		// clean up the NaN-mess!!!
			}
			
// optimize...
// optimize...
// optimize...

sort(gpop, gpop + popsize);

if (gpop[0].rms < gpop[popsize].rms)
{
	(* mlog) << "found the best solution so far!!!!!!!!!!" << endl;
	(* mlog) << "found the best solution so far!!!!!!!!!!" << endl;
	(* mlog) << "found the best solution so far!!!!!!!!!!" << endl;
	
	gpop[popsize].rms = gpop[0].rms;
	
	gpop[popsize].wtmp1 = gpop[0].wtmp1;
	gpop[popsize].wtmp2 = gpop[0].wtmp2;
	gpop[popsize].wtmp3 = gpop[0].wtmp3;
	
	gpop[popsize].wtor1 = gpop[0].wtor1;
	gpop[popsize].wtor2 = gpop[0].wtor2;
	gpop[popsize].lenjon = gpop[0].lenjon;
	gpop[popsize].vdwrad = gpop[0].vdwrad;
	gpop[popsize].screen1 = gpop[0].screen1;
	gpop[popsize].screen2 = gpop[0].screen2;
	gpop[popsize].screen3 = gpop[0].screen3;
	gpop[popsize].screen4 = gpop[0].screen4;
	gpop[popsize].screen5 = gpop[0].screen5;
	gpop[popsize].dipole1 = gpop[0].dipole1;
	gpop[popsize].dipole2 = gpop[0].dipole2;
	gpop[popsize].pbility1 = gpop[0].pbility1;
	gpop[popsize].pbility2 = gpop[0].pbility2;
	gpop[popsize].sasa1 = gpop[0].sasa1;
	gpop[popsize].sasa2 = gpop[0].sasa2;
	gpop[popsize].solvrad = gpop[0].solvrad;
}

			for (i32s n1 = 0;n1 < popsize;n1++)
			{
				(* mlog) << "RMS = " << setw(12) << setprecision(8) << gpop[n1].rms << endl;
				
				(* mlog) << "wtmp1 = " << setw(12) << setprecision(8) << gpop[n1].wtmp1 << endl;
				(* mlog) << "wtmp2 = " << setw(12) << setprecision(8) << gpop[n1].wtmp2 << endl;
				(* mlog) << "wtmp3 = " << setw(12) << setprecision(8) << gpop[n1].wtmp3 << endl;
				
				(* mlog) << "wtor1 = " << setw(12) << setprecision(8) << gpop[n1].wtor1 << endl;
				(* mlog) << "wtor2 = " << setw(12) << setprecision(8) << gpop[n1].wtor2 << endl;
				(* mlog) << "lenjon = " << setw(12) << setprecision(8) << gpop[n1].lenjon << endl;
				(* mlog) << "vdwrad = " << setw(12) << setprecision(8) << gpop[n1].vdwrad << endl;
				(* mlog) << "screen1 = " << setw(12) << setprecision(8) << gpop[n1].screen1 << endl;
				(* mlog) << "screen2 = " << setw(12) << setprecision(8) << gpop[n1].screen2 << endl;
				(* mlog) << "screen3 = " << setw(12) << setprecision(8) << gpop[n1].screen3 << endl;
				(* mlog) << "screen4 = " << setw(12) << setprecision(8) << gpop[n1].screen4 << endl;
				(* mlog) << "screen5 = " << setw(12) << setprecision(8) << gpop[n1].screen5 << endl;
				(* mlog) << "dipole1 = " << setw(12) << setprecision(8) << gpop[n1].dipole1 << endl;
				(* mlog) << "dipole2 = " << setw(12) << setprecision(8) << gpop[n1].dipole2 << endl;
				(* mlog) << "pbility1 = " << setw(12) << setprecision(8) << gpop[n1].pbility1 << endl;
				(* mlog) << "pbility2 = " << setw(12) << setprecision(8) << gpop[n1].pbility2 << endl;
				(* mlog) << "sasa1 = " << setw(12) << setprecision(8) << gpop[n1].sasa1 << endl;
				(* mlog) << "sasa2 = " << setw(12) << setprecision(8) << gpop[n1].sasa2 << endl;
				(* mlog) << "solvrad = " << setw(12) << setprecision(8) << gpop[n1].solvrad << endl;
			}
			
			(* mlog) << "this cycle is now done..." << endl << endl;
			
// average...
// average...
// average...

//f64 sss = 0.0; i32s ccc = 0;
//for (i32s n1 = 0;n1 < popsize;n1++)
//{
//	if (gpop[n1].rms > 1.0e+12) continue;
//	sss += gpop[n1].rms; ccc++;
//}
//
//(* mlog) << "AVERAGE = " << sss << " / " << ccc << " = " << (sss / (f64) ccc) << endl;

		}
		
// average...
// average...
// average...

//MPI::COMM_WORLD.Barrier(); break;
	}
	
	// this is practically unreachable... IN OPTIMIZATION!!!
	// this is practically unreachable... IN OPTIMIZATION!!!
	// this is practically unreachable... IN OPTIMIZATION!!!
	
	MPI::Finalize();
	
//OSMesaDestroyContext(ctx); delete[] fb;
	
	return 0;
	
//*/
/*##############################################*/
/*##############################################*/

}

/*################################################################################################*/

ddsolver::ddsolver(i32s vakio, f64 * xd, f64 * yd, f64 * yc) : conjugate_gradient(25, 0.001)
{
	var[0] = 0.0; AddVar(& var[0], & dvar[0]);
	var[1] = 0.0; AddVar(& var[1], & dvar[1]);
	var[2] = 0.0; AddVar(& var[2], & dvar[2]);
	
	var[3] = 0.0; AddVar(& var[3], & dvar[3]);
	var[4] = 0.0; AddVar(& var[4], & dvar[4]);
	var[5] = 0.0; AddVar(& var[5], & dvar[5]);
	
	var[6] = vakio;
	var[7] = 0.0; AddVar(& var[7], & dvar[7]);
	
	xdata = xd;
	ydata = yd;
	ycount = yc;
}

ddsolver::~ddsolver(void)
{
}

f64 ddsolver::GetValue(void)
{
	f64 err = 0.0;
	
	const i32s size2 = 36;		// 10 degrees...
	const i32s size3 = 120;		// 3 degrees...
	
	for (i32s i1 = 0;i1 < size2;i1++)
	{
		f64 tmp1 = var[0] * cos(xdata[i1]) + var[1] * cos(2.0 * xdata[i1]) + var[2] * cos(3.0 * xdata[i1]);
		tmp1 += var[3] * sin(xdata[i1]) + var[4] * sin(2.0 * xdata[i1]) + var[5] * sin(3.0 * xdata[i1]);
		tmp1 += var[6] * xdata[i1] + var[7];
		
		for (i32s zz = 0;zz < size3;zz++)
		{
			f64 diff = tmp1 - ydata[zz]; check1:
			if (diff > +M_PI) { diff = 2.0 * M_PI - diff; goto check1; }
			else if (diff < -M_PI) { diff = 2.0 * M_PI + diff; goto check1; }
			err += diff * diff * (f64) ycount[i1 * size3 + zz];
		}
	}
	
	return err;
}

/*################################################################################################*/

const f64 decoys_ddata::scale = 1.0;	// this is not needed -> set to 1.0 !!!!!

//i32s count_h[20] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
//i32s count_s[20] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
//i32s count_l[20] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

decoys_ddata::decoys_ddata(istream & str)
{
	str >> type;
	
	for (i32s nn = 0;nn < 4;nn++) str >> energy_bt[nn];
	for (i32s nn = 0;nn < 5;nn++) str >> energy_nbt[nn];
	for (i32s nn = 0;nn < 20;nn++) str >> hdata[nn];
	for (i32s nn = 0;nn < 20;nn++) str >> sdata[nn];
	for (i32s nn = 0;nn < 20;nn++) str >> ldata[nn];
	
//for (i32s nn = 0;nn < 20;nn++) count_h[nn] += hdata[nn];
//for (i32s nn = 0;nn < 20;nn++) count_s[nn] += sdata[nn];
//for (i32s nn = 0;nn < 20;nn++) count_l[nn] += ldata[nn];

// DO SOME SCALING DOWN TO IMPROVE NUMERICAL ACCURACY?!?!?!
// DO SOME SCALING DOWN TO IMPROVE NUMERICAL ACCURACY?!?!?!
// DO SOME SCALING DOWN TO IMPROVE NUMERICAL ACCURACY?!?!?!
for (i32s nn = 0;nn < 4;nn++) energy_bt[nn] /= scale;
for (i32s nn = 0;nn < 5;nn++) energy_nbt[nn] /= scale;

	char buffer[1024]; str >> buffer;
	name = new char[strlen(buffer) + 1]; strcpy(name, buffer);
	
	RMSdistance = NOT_DEFINED;
	
	str.getline(buffer, sizeof(buffer));
}

decoys_ddata::~decoys_ddata(void)
{
	delete[] name;
}

/*################################################################################################*/

f64 decoys_dset::f;

i32u decoys_dset::badcount;
i32u decoys_dset::totcount;

f64 decoys_dset::df_bterms[4];
f64 decoys_dset::df_nbterms[5];
f64 decoys_dset::df_helix[20];
f64 decoys_dset::df_strand[20];
f64 decoys_dset::df_loop[20];

f64 decoys_dset::w_bterms[4] = { 1.0, 1.0, 1.0, 1.0 };
f64 decoys_dset::w_nbterms[5] = { 1.0, 1.0, 1.0, 1.0, 1.0 };

f64 decoys_dset::w_helix[20] =
{
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

f64 decoys_dset::w_strand[20] =
{
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

f64 decoys_dset::w_loop[20] =
{
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
	0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
};

decoys_dset::decoys_dset(void)
{
}

decoys_dset::~decoys_dset(void)
{
	for (i32s n1 = 0;n1 < dd_vector.size();n1++)
	{
		delete dd_vector[n1];
	}
}

void decoys_dset::AddData(decoys_ddata * ref)
{
	dd_vector.push_back(ref);
}

// this has become a quite messy system since we began to use RMS-distances at the target function.
// save the RMS-distances at the same stage where we optimize and compute that data!!!!!!!!!!!!!!!!

ifstream ifile2;	// QUICK!!!

void decoys_dset::UpdateRMS(void)
{
	char * nname = dd_vector[0]->name; //cout << "nname = " << nname << endl;
	mm2_mdl * nmdl = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
	
	ifstream ifile1;
	ifile1.open(nname, ios::in);
	nmdl->ReadStream(ifile1);
	ifile1.close();
	
	nmdl->PushCRDSets(1);
	
/*
////////////////////
////////////////////
	for (i32s n1 = 1;n1 < dd_vector.size();n1++)	// calculate...
	{
		char * dname = dd_vector[n1]->name; //cout << "dname = " << dname << endl;
		mm2_mdl * dmdl = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
		
		ifstream ifile2; ifile2.open(dname, ios::in);
		dmdl->ReadStream(ifile2); ifile2.close();
		
		// now nmdl (native) and dmdl (decoy) should be similar.
		// we just copy the coordinates into nmdl and superimpose!!!
		
		// practically, there seem to be some different residues, so we have to check natm...
		
		for (i32s c1 = 0;c1 < nmdl->chn_vector.size();c1++)
		{
			for (i32s c2 = 0;c2 < nmdl->chn_vector[c1].res_vector.size();c2++)
			{
				i32s nmdl_natm = nmdl->chn_vector[c1].res_vector[c2].natm;
				i32s dmdl_natm = dmdl->chn_vector[c1].res_vector[c2].natm;
				
				i32s natm = (nmdl_natm < dmdl_natm ? nmdl_natm : dmdl_natm);
				nmdl->chn_vector[c1].res_vector[c2].natm = natm;
				
				for (i32s c3 = 0;c3 < nmdl->chn_vector[c1].res_vector[c2].natm;c3++)
				{
					fGL xcrd = dmdl->chn_vector[c1].res_vector[c2].crd_vector[c3][0].data[0];
					fGL ycrd = dmdl->chn_vector[c1].res_vector[c2].crd_vector[c3][0].data[1];
					fGL zcrd = dmdl->chn_vector[c1].res_vector[c2].crd_vector[c3][0].data[2];
					
					nmdl->chn_vector[c1].res_vector[c2].crd_vector[c3][1].data[0] = xcrd;
					nmdl->chn_vector[c1].res_vector[c2].crd_vector[c3][1].data[1] = ycrd;
					nmdl->chn_vector[c1].res_vector[c2].crd_vector[c3][1].data[2] = zcrd;
				}
			}
		}
		
		delete dmdl; cout << "dname = " << dname << endl;
		
		// and now we superimpose...
		
		mm2_superimpose si(nmdl, 0, 1);
		for (i32s n2 = 0;n2 < 50;n2++) si.TakeCGStep(conjugate_gradient::Newton2An);
		f64 rms = si.GetRMS();
		
		cout << " RMS = " << rms << endl << flush;
		
		// also save to the arrays...
		
		dd_vector[n1]->RMSdistance = rms;
	}
////////////////////
////////////////////
*/

////////////////////
////////////////////
	for (i32s n1 = 1;n1 < dd_vector.size();n1++)	// read from file...
	{
		char * dname = dd_vector[n1]->name;
		
	//	ifstream ifile2;
	//	ifile2.open("/tmp/decoy-RMSdistances.txt", ios::in);
	//	if (ifile2.fail()) { cout << "ERROR : could not open" << endl; exit(-1); }
		
		dd_vector[n1]->RMSdistance = NOT_DEFINED;
		
		while (!ifile2.eof())
		{
			char buffer[1024];
			
			ifile2.getline(buffer, sizeof(buffer), '=');
			ifile2 >> buffer;
			
			if (!strcmp(dname, buffer))
			{
				ifile2.getline(buffer, sizeof(buffer));
				ifile2.getline(buffer, sizeof(buffer), '=');
				
				f64 qqq; ifile2 >> qqq; dd_vector[n1]->RMSdistance = qqq;
				
				cout << "found RMS = " << dd_vector[n1]->RMSdistance << " for " << dname << endl;
				break;
			}
			else
			{
				ifile2.getline(buffer, sizeof(buffer));
				ifile2.getline(buffer, sizeof(buffer));
			}
		}
		
		if (dd_vector[n1]->RMSdistance < 0.0) { cout << "ERROR : could not find" << endl; exit(-1); }
	//	ifile2.close();
	}
////////////////////
////////////////////
	
	delete nmdl;
	
//cout << "COUNTS:" << endl; cout << "^^^^^^^" << endl;
//for (i32s nn = 0;nn < 20;nn++) cout << count_h[nn] << " "; cout << endl;
//for (i32s nn = 0;nn < 20;nn++) cout << count_s[nn] << " "; cout << endl;
//for (i32s nn = 0;nn < 20;nn++) cout << count_l[nn] << " "; cout << endl;
//int kkk; cin >> kkk;

}

void decoys_dset::WriteScatter(ofstream & file)
{
	for (i32s n1 = 1;n1 < dd_vector.size();n1++)
	{
		char * dname = dd_vector[n1]->name; //cout << "dname = " << dname << endl;
		
		// calculate energies...
		// calculate energies...
		// calculate energies...
		
		f64 en = 0.0;
		f64 ed = 0.0;
		
		for (i32s n2 = 0;n2 < 4;n2++) en += w_bterms[n2] * dd_vector[0]->energy_bt[n2];
		for (i32s n2 = 0;n2 < 5;n2++) en += w_nbterms[n2] * dd_vector[0]->energy_nbt[n2];
		
		for (i32s n2 = 0;n2 < 4;n2++) ed += w_bterms[n2] * dd_vector[n1]->energy_bt[n2];
		for (i32s n2 = 0;n2 < 5;n2++) ed += w_nbterms[n2] * dd_vector[n1]->energy_nbt[n2];
		
		for (i32s n2 = 0;n2 < 20;n2++)
		{
			en += w_helix[n2] * dd_vector[0]->hdata[n2];
			en += w_strand[n2] * dd_vector[0]->sdata[n2];
			//en += w_loop[n2] * dd_vector[0]->ldata[n2];
			
			ed += w_helix[n2] * dd_vector[n1]->hdata[n2];
			ed += w_strand[n2] * dd_vector[n1]->sdata[n2];
			//ed += w_loop[n2] * dd_vector[n1]->ldata[n2];
		}
		
		// calculate energies...
		// calculate energies...
		// calculate energies...
		
		cout << "rms = " << dd_vector[n1]->RMSdistance << " rel_E = " << (ed - en) * decoys_ddata::scale << endl;
		
		// also write to the file...
		
		file << dd_vector[n1]->RMSdistance << " " << (ed - en) * decoys_ddata::scale << " " << dname << endl;
	}
}

void decoys_dset::Compute(i32s p1)
{
	for (i32s n1 = 1;n1 < dd_vector.size();n1++)
	{
		f64 en = 0.0;
		f64 ed = 0.0;
		
		for (i32s n2 = 0;n2 < 4;n2++) en += w_bterms[n2] * dd_vector[0]->energy_bt[n2];
		for (i32s n2 = 0;n2 < 5;n2++) en += w_nbterms[n2] * dd_vector[0]->energy_nbt[n2];
		
		for (i32s n2 = 0;n2 < 4;n2++) ed += w_bterms[n2] * dd_vector[n1]->energy_bt[n2];
		for (i32s n2 = 0;n2 < 5;n2++) ed += w_nbterms[n2] * dd_vector[n1]->energy_nbt[n2];
		
		for (i32s n2 = 0;n2 < 20;n2++)
		{
			en += w_helix[n2] * dd_vector[0]->hdata[n2];
			en += w_strand[n2] * dd_vector[0]->sdata[n2];
			//en += w_loop[n2] * dd_vector[0]->ldata[n2];
			
			ed += w_helix[n2] * dd_vector[n1]->hdata[n2];
			ed += w_strand[n2] * dd_vector[n1]->sdata[n2];
			//ed += w_loop[n2] * dd_vector[n1]->ldata[n2];
		}
		
		if (ed < en) badcount++;
		totcount++;
		
		// "smooth" functions like sqrt(x) seem to have an advantage over "sharp" ones
		// like x^2 in this case, since few bad cases should not dominate the fitting!!!
		
		f64 tmp1 = en - ed; if (tmp1 < 0.0) continue;
		f += sqrt(tmp1); f64 tmp2 = 0.5 / sqrt(tmp1);		// f = sqrt(N-D) || if (N-D) < 0
	//	f += tmp1 * tmp1; f64 tmp2 = 2.0 * tmp1;		// f = (N-D)^2 || if (N-D) < 0
	
// here we try to maximize the difference between native and decoys,
// but this seems to lead too large energies (and high badcount if constrained).
////////////////////////////////////////////////////////////////////////////////
//f64 rms = dd_vector[n1]->RMSdistance;		// when using the current dataset,
//if (rms > 10.0) continue;			// rms > 10.0 is probably an error...
//f64 aaa = 0.001; const f64 bbb = 0.0325;		// f = a*(exp(b*(N-D))-1)
//f64 tmp1 = bbb * (en - ed);				// f = a*(exp(b*(N-D))-1)
//f += aaa * exp(tmp1) - aaa;				// f = a*(exp(b*(N-D))-1)
//f64 tmp2 = aaa * bbb * exp(tmp1);			// f = a*(exp(b*(N-D))-1)

		if (p1 > 0)
		{
			for (i32s n2 = 0;n2 < 4;n2++)
			{
				df_bterms[n2] += tmp2 * (dd_vector[0]->energy_bt[n2] - dd_vector[n1]->energy_bt[n2]);
			}
			
			for (i32s n2 = 0;n2 < 5;n2++)
			{
				df_nbterms[n2] += tmp2 * (dd_vector[0]->energy_nbt[n2] - dd_vector[n1]->energy_nbt[n2]);
			}
			
			for (i32s n2 = 0;n2 < 20;n2++)
			{
				df_helix[n2] += tmp2 * (dd_vector[0]->hdata[n2] - dd_vector[n1]->hdata[n2]);
				df_strand[n2] += tmp2 * (dd_vector[0]->sdata[n2] - dd_vector[n1]->sdata[n2]);
				//df_loop[n2] += tmp2 * (dd_vector[0]->ldata[n2] - dd_vector[n1]->ldata[n2]);
			}
		}
	}
	
	// try to keep the values near each other... quadric equation!!!
	// try to keep the values near each other... quadric equation!!!
	// try to keep the values near each other... quadric equation!!!
	
	const f64 ccc = 0.00029;	// 0.00029 for sqrt(x) will bring the average range to about 21 kJ/mol...

	for (i32s n1 = 0;n1 < 19;n1++)
	{
		for (i32s n2 = n1 + 1;n2 < 20;n2++)
		{
			f64 tmp5a = w_helix[n1] - w_helix[n2]; f += ccc * tmp5a * tmp5a;
			f64 tmp5b = w_strand[n1] - w_strand[n2]; f += ccc * tmp5b * tmp5b;
			f64 tmp5c = w_loop[n1] - w_loop[n2]; f += ccc * tmp5c * tmp5c;
			
			if (p1 > 0)
			{
				f64 tmp6a = 2.0 * ccc * tmp5a; df_helix[n1] += tmp6a; df_helix[n2] -= tmp6a;
				f64 tmp6b = 2.0 * ccc * tmp5b; df_strand[n1] += tmp6b; df_strand[n2] -= tmp6b;
				f64 tmp6c = 2.0 * ccc * tmp5c; df_loop[n1] += tmp6c; df_loop[n2] -= tmp6c;
			}
		}
	}
}

/*################################################################################################*/

decoys_opt::decoys_opt(void) : conjugate_gradient(50, 0.001)
{
//	for (i32s n1 = 0;n1 < 4;n1++) AddVar(& decoys_dset::w_bterms[n1], & decoys_dset::df_bterms[n1]);
//	for (i32s n1 = 0;n1 < 5;n1++) AddVar(& decoys_dset::w_nbterms[n1], & decoys_dset::df_nbterms[n1]);
	
	for (i32s n1 = 0;n1 < 20;n1++) AddVar(& decoys_dset::w_helix[n1], & decoys_dset::df_helix[n1]);
	for (i32s n1 = 0;n1 < 20;n1++) AddVar(& decoys_dset::w_strand[n1], & decoys_dset::df_strand[n1]);
//	for (i32s n1 = 0;n1 < 20;n1++) AddVar(& decoys_dset::w_loop[n1], & decoys_dset::df_loop[n1]);
}

decoys_opt::~decoys_opt(void)
{
	ClearAll();
}

void decoys_opt::AddFile(const char * fn)
{
	ifstream ifile(fn, ios::in);
	decoys_dset * ds = NULL;
	
	// added here is a system that checks validity of the decoys; the sequence must be same
	// as in the native structure. IT WOULD HAVE BEEN EASIER TO CHECK THIS IN EARLIER STAGE!!!
	
	const i32s maxs = 16; char * seqtab[maxs];		// this (16 seq) should be enough;
	for (i32s n9 = 0;n9 < maxs;n9++) seqtab[n9] = NULL;	// usually these are single chains!
	
	while (ifile.peek() != 'e')
	{
		decoys_ddata * dd = new decoys_ddata(ifile);
		
		if (dd->type == 'n')
		{
			//////////////////// begin...
			//////////////////// begin...
			
			for (i32s n9 = 0;n9 < maxs;n9++)
			{
				delete[] seqtab[n9];
				seqtab[n9] = NULL;
			}
			
			mm2_mdl * tmpmdl = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
			
			ifstream if2; if2.open(dd->name, ios::in);
			tmpmdl->ReadStream(if2); if2.close();
			
			for (i32s n1 = 0;n1 < tmpmdl->chn_vector.size();n1++)
			{
				i32s chnlen = tmpmdl->chn_vector[n1].res_vector.size();
				seqtab[n1] = new char[chnlen + 1];
				
				for (i32s n2 = 0;n2 < chnlen;n2++)
				{
					seqtab[n1][n2] = tmpmdl->chn_vector[n1].res_vector[n2].symbol;
				}
				
				seqtab[n1][chnlen] = 0;
			}
			
			delete tmpmdl;
			
			//////////////////// end...
			//////////////////// end...
		
			if (ds != NULL)
			{
				cout << "saving old decoy set..." << endl;
				ds_vector.push_back(ds);
			}
			
			ds = new decoys_dset();
		}
		else
		{
			//////////////////// begin...
			//////////////////// begin...
			
			bool this_is_ok_sequence = true;
			
			mm2_mdl * tmpmdl = new mm2_mdl(& cout, * console_class_factory::GetInstance(), NULL);
			
			ifstream if2; if2.open(dd->name, ios::in);
			tmpmdl->ReadStream(if2); if2.close();
			
			for (i32s n1 = 0;n1 < tmpmdl->chn_vector.size();n1++)
			{
				i32s chnlen = tmpmdl->chn_vector[n1].res_vector.size();
				char * tmpseq = new char[chnlen + 1];
				
				for (i32s n2 = 0;n2 < chnlen;n2++)
				{
					tmpseq[n2] = tmpmdl->chn_vector[n1].res_vector[n2].symbol;
				}
				
				tmpseq[chnlen] = 0;
				
				// compare the sequences; if it differs, make it bad!!!
				
				if (strcmp(seqtab[n1], tmpseq)) this_is_ok_sequence = false;
				delete[] tmpseq;
			}
			
			delete tmpmdl;
			
			// if this is not an ok sequence, report about it and skip it:
			
			if (!this_is_ok_sequence)
			{
				cout << "SKIPPED BAD DECOY " << dd->name << endl;
				delete dd; continue;
			}
			
			//////////////////// end...
			//////////////////// end...
			
			if (ds == NULL) { cout << "bad dset!!!" << endl; exit(EXIT_FAILURE); }
		}
		
		cout << "adding data " << dd->type << " " << dd->name << endl;
		ds->AddData(dd);
	}
	
	if (ds != NULL)
	{
		cout << "saving the last decoy set..." << endl;
	//	int zzz; cin >> zzz;
		
		ds_vector.push_back(ds);
	}
}

void decoys_opt::ClearAll(void)
{
	for (i32s n1 = 0;n1 < ds_vector.size();n1++) delete ds_vector[n1];
	while (!ds_vector.empty()) ds_vector.pop_back();
}

void decoys_opt::UpdateRMS(void)
{
ifile2.open("/tmp/decoy-RMSdistances.txt", ios::in);
	for (i32s n1 = 0;n1 < ds_vector.size();n1++) ds_vector[n1]->UpdateRMS();
ifile2.close();
}

void decoys_opt::WriteScatter(ofstream & file)
{
	for (i32s n1 = 0;n1 < ds_vector.size();n1++) ds_vector[n1]->WriteScatter(file);
}

void decoys_opt::Compute(i32s p1)
{
	decoys_dset::f = 0.0;
	
	decoys_dset::badcount = 0;
	decoys_dset::totcount = 0;
	
	if (p1 > 0)
	{
		for (i32s n1 = 0;n1 < 4;n1++) decoys_dset::df_bterms[n1] = 0.0;
		for (i32s n1 = 0;n1 < 5;n1++) decoys_dset::df_nbterms[n1] = 0.0;
		
		for (i32s n1 = 0;n1 < 20;n1++)
		{
			decoys_dset::df_helix[n1] = 0.0;
			decoys_dset::df_strand[n1] = 0.0;
			decoys_dset::df_loop[n1] = 0.0;
		}
	}
	
	for (i32s n1 = 0;n1 < ds_vector.size();n1++)
	{
		ds_vector[n1]->Compute(p1);
	}
}

#define DELTA 0.000001

void decoys_opt::Check(void)
{
	Compute(1); f64 tmp1 = decoys_dset::f; i32s zzz;
	
	for (i32u n1 = 0;n1 < 4;n1++)
	{
		f64 old = decoys_dset::w_bterms[n1];
		decoys_dset::w_bterms[n1] = old + DELTA; Compute(0);
		f64 tmp2 = (decoys_dset::f - tmp1) / DELTA;
		decoys_dset::w_bterms[n1] = old;
		
		cout << "bt" << n1 << " : a = " << decoys_dset::df_bterms[n1] << " n = " << tmp2 << endl;
	} cin >> zzz;
	
	for (i32u n1 = 0;n1 < 5;n1++)
	{
		f64 old = decoys_dset::w_nbterms[n1];
		decoys_dset::w_nbterms[n1] = old + DELTA; Compute(0);
		f64 tmp2 = (decoys_dset::f - tmp1) / DELTA;
		decoys_dset::w_nbterms[n1] = old;
		
		cout << "nbt" << n1 << " : a = " << decoys_dset::df_nbterms[n1] << " n = " << tmp2 << endl;
	} cin >> zzz;
	
	for (i32u n1 = 0;n1 < 20;n1++)
	{
		f64 old = decoys_dset::w_helix[n1];
		decoys_dset::w_helix[n1] = old + DELTA; Compute(0);
		f64 tmp2 = (decoys_dset::f - tmp1) / DELTA;
		decoys_dset::w_helix[n1] = old;
		
		cout << "h" << n1 << " : a = " << decoys_dset::df_helix[n1] << " n = " << tmp2 << endl;
	} cin >> zzz;
	
	for (i32u n1 = 0;n1 < 20;n1++)
	{
		f64 old = decoys_dset::w_strand[n1];
		decoys_dset::w_strand[n1] = old + DELTA; Compute(0);
		f64 tmp2 = (decoys_dset::f - tmp1) / DELTA;
		decoys_dset::w_strand[n1] = old;
		
		cout << "s" << n1 << " : a = " << decoys_dset::df_strand[n1] << " n = " << tmp2 << endl;
	} cin >> zzz;
	
	for (i32u n1 = 0;n1 < 20;n1++)
	{
		f64 old = decoys_dset::w_loop[n1];
		decoys_dset::w_loop[n1] = old + DELTA; Compute(0);
		f64 tmp2 = (decoys_dset::f - tmp1) / DELTA;
		decoys_dset::w_loop[n1] = old;
		
		cout << "l" << n1 << " : a = " << decoys_dset::df_loop[n1] << " n = " << tmp2 << endl;
	} cin >> zzz;
}

f64 decoys_opt::GetValue(void)
{
	Compute(0);
	return decoys_dset::f;
}

f64 decoys_opt::GetGradient(void)
{
	Compute(1);
	return decoys_dset::f;
}

void decoys_opt::PrintResults(ostream & str)
{
	str << "badcount = " << decoys_dset::badcount << endl;
	str << "totcount = " << decoys_dset::totcount << endl;
	
	for (i32s nn = 0;nn < 4;nn++) str << "w_bt " << nn << " " << decoys_dset::w_bterms[nn] << endl;
	for (i32s nn = 0;nn < 5;nn++) str << "w_nbt " << nn << " " << decoys_dset::w_nbterms[nn] << endl;
	
	for (i32s nn = 0;nn < 4;nn++) str << "df_bt " << nn << " " << decoys_dset::df_bterms[nn] << endl;
	for (i32s nn = 0;nn < 5;nn++) str << "df_nbt " << nn << " " << decoys_dset::df_nbterms[nn] << endl;
	
	for (i32s nn = 0;nn < 20;nn++) str << "H " << nn << " " << (decoys_dset::w_helix[nn] * decoys_ddata::scale) << endl;
	for (i32s nn = 0;nn < 20;nn++) str << "S " << nn << " " << (decoys_dset::w_strand[nn] * decoys_ddata::scale) << endl;
	for (i32s nn = 0;nn < 20;nn++) str << "L " << nn << " " << (decoys_dset::w_loop[nn] * decoys_ddata::scale) << endl;
	
	// also print them in the final form...
	
	for (i32s nn = 0;nn < 20;nn++) str << (decoys_dset::w_helix[nn] * decoys_ddata::scale) << " "; str << endl;
	for (i32s nn = 0;nn < 20;nn++) str << (decoys_dset::w_strand[nn] * decoys_ddata::scale) << " "; str << endl;
	for (i32s nn = 0;nn < 20;nn++) str << (decoys_dset::w_loop[nn] * decoys_ddata::scale) << " "; str << endl;
}

/*################################################################################################*/

f64 getRMSdistance(char * fn, rms_param * prm1, mm2_eng_param * prm2)
{
	char lfn[256];
	ostrstream lfns(lfn, sizeof(lfn));
	
//////////////////////////////
i32s mpi_rank = MPI::COMM_WORLD.Get_rank();
lfns << "/tmp/p" << mpi_rank << "_" << fn << ".log" << ends;
//////////////////////////////
//lfns << "/tmp/" << fn << ".log" << ends;	// "/tmp" is not safe for large tests!!!
//////////////////////////////

	ofstream log_f(lfn, ios::out);
	
	char ifn[256];
	ostrstream ifns(ifn, sizeof(ifn));
	ifns << "/home/thassine/test_b/" << fn << ".mm2" << ends;		// small test set...
//	ifns << "../../pconv2_dec_1998/" << fn << ".mm2" << ends;		// large test set...

	mm2_mdl * mdl2 = new mm2_mdl(& cout, * console_class_factory::GetInstance(), prm2);
	
	ifstream in_f(ifn, ios::in);
	mdl2->ReadStream(in_f); in_f.close();

////////// optional for the large test... optional for the large test...
//if (mdl2->chn_vector[0].res_vector.size() > 300) { log_f.close(); return -1.0; }
////////// optional for the large test... optional for the large test...
	
	mm2_eng * eng = new mm2_eng(* mdl2); CopyCRD(mdl2, eng, 0);
	mm2_geomopt * opt = new mm2_geomopt(eng, 50, 0.025);		// optimal settings?!?!?
	
	for (i32s n2 = 0;n2 < prm1->osteps;n2++)
	{
		opt->TakeCGStep(conjugate_gradient::Newton2An);
		log_f << n2 << " " << opt->optval << " " << opt->optstp << " " << endl;
	}
	
	delete opt;
	
	mm2_moldyn * dyn = new mm2_moldyn(eng, prm1->temp, prm1->timestep);
	
	i32s ccnt = 0; f64 control[prm1->csize]; f64 sum; f64 average;
	for (i32s n2 = 0;n2 < prm1->start + prm1->equil + prm1->simul;n2++)
	{
		dyn->TakeMDStep();
		if (!(n2 % 250))
		{
			log_f << n2 << " ";
			log_f << "T = " << dyn->ConvEKinTemp(dyn->GetEKin()) << " ";
			log_f << "Ep = " << dyn->GetEPot() << " ";
			log_f << "Et = " << (dyn->GetEKin() + dyn->GetEPot()) << endl;
			
	//		if (dyn->pot != dyn->pot) goto nan_panic;	// skip if problems...
	//		if (dyn->kin != dyn->kin) goto nan_panic;	// skip if problems...
		}
		
		if (n2 < prm1->start)
		{
			if (!(n2 % prm1->cfrq))
			{
				control[ccnt++] = dyn->GetEKin();
			}
			
			if (ccnt == prm1->csize)
			{
				f64 sum = 0.0;
				for (i32s n9 = 0;n9 < ccnt;n9++)
				{
					sum += control[n9];
				}
				
				sum /= (f64) ccnt; ccnt = 0; f64 rtemp = (f64) (n2 + prm1->cfrq) / (f64) prm1->start;
				log_f << "average = " << sum << " kJ/mol = " << dyn->ConvEKinTemp(sum) << " K." << endl;
				dyn->SetEKin(dyn->GetEKin() + (dyn->ConvTempEKin(dyn->GetTemp() * rtemp) - sum));
			}
		}
		else if (!(n2 < prm1->start + prm1->equil) && !(n2 % 100))
		{
			mdl2->PushCRDSets(1);
			CopyCRD(eng, mdl2, mdl2->GetCRDSetCount() - 1);
		}
	}
	
	delete dyn; delete eng; sum = 0.0;
	for (i32s n1 = 1;n1 < (i32s) mdl2->GetCRDSetCount();n1++)
	{
		mm2_superimpose si(mdl2, 0, n1);
		for (i32s n2 = 0;n2 < 100;n2++) si.TakeCGStep(conjugate_gradient::Newton2An);
		
		f64 rms = si.GetRMS(); sum += rms;
		log_f << n1 << " RMS-distance = " << rms << endl;
	}
	
	average = sum / ((f64) mdl2->GetCRDSetCount() - 1);
	
//	log_f << "length of this chn = " << mdl2->chn_vector[0].res_vector.size() << endl;
	
	log_f << "average RMS-error = " << average << endl;
	log_f.close(); delete mdl2;
	
	return average;
	
	nan_panic:
	delete dyn; delete eng; delete mdl2;
	return 1.0e+12;
}

/*################################################################################################*/

// eof
