// MM1DOCV.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm1docv.h"	// config.h is here -> we get ENABLE-macros here...

#ifdef ENABLE_GRAPHICS
#include "mm1alg.h"
#include "mm1eng1.h"
#include "mm1eng9.h"

#include "mm1rbn.h"

#include "plane.h"
#include "surface.h"

#include "color.h"
#include "views.h"

#include <fstream>
#include <strstream>
using namespace std;

/*################################################################################################*/

mm1_cm_element mm1_docv::cm_element = mm1_cm_element();
mm1_cm_state mm1_docv::cm_state = mm1_cm_state();

mm1_docv::mm1_docv(ostream * p1, graphics_class_factory & p2) :
	docview(p1, p2), mm1_mdl(p1, p2), model_simple(p1, p2)
{
}

mm1_docv::~mm1_docv(void)
{
}

fGL mm1_docv::GetDefaultFocus(void)
{
	return 2.0;
}

const char * mm1_docv::GetType(void)
{
	return "MM, all-atoms model";
}

color_mode * mm1_docv::GetDefaultColorMode(void)
{
	return & mm1_docv::cm_element;
}

void mm1_docv::SelectAll(void)
{
	if (selected_object != NULL)
	{
		selected_object = NULL;
		event_SelectedObjectChanged();
	}
	
	iter_mm1al it1 = atom_list.begin();
	while (it1 != atom_list.end()) (* it1++).selected = true;
	
	UpdateAllGraphicsViews();
}

void mm1_docv::InvertSelection(void)
{
	if (selected_object != NULL)
	{
		selected_object = NULL;
		event_SelectedObjectChanged();
	}
	
	iter_mm1al it1 = atom_list.begin();
	while (it1 != atom_list.end())
	{
		bool flag = (* it1).selected;
		(* it1++).selected = !flag;
	}
	
	UpdateAllGraphicsViews();
}

//void mm1_docv::SelectNone(void)
//{
//	if (selected_object != NULL)
//	{
//		selected_object = NULL;
//		event_SelectedObjectChanged();
//	}
//	
//	iter_mm1al it1 = atom_list.begin();
//	while (it1 != atom_list.end()) (* it1++).selected = false;
//	
//	UpdateAllGraphicsViews();
//}

bool mm1_docv::TestAtom(mm1_atom * ref, rmode rm)
{
	// what about the "visible"-flag???
	
	if (rm == Transform1 && ref->selected) return false;
	if (rm == Transform2 && !ref->selected) return false;
	
	return true;
}

bool mm1_docv::TestBond(mm1_bond * ref, rmode rm)
{
	// what about the "visible"-flag???
	
	if (rm == Transform1 && ref->atmr[0]->selected) return false;
	if (rm == Transform2 && !ref->atmr[0]->selected) return false;
	
	bool test = (ref->atmr[0]->selected != ref->atmr[1]->selected);
	if (rm != Normal && test) return false;
	
	return true;
}

void mm1_docv::SetColor(color_mode * cm, mm1_atom * ref)
{
	const fGL default_sel_color[3] = { 1.0, 0.0, 1.0 };
	fGL *select_color = model_prefs->ColorRGB("Graphics/SelectColor", default_sel_color);

	if (ref->selected) glColor3f(select_color[0], select_color[1], select_color[2]);
	else
	{
		fGL_a4 color;
		cm->GetColor(ref, color, model_prefs);
		glColor3fv(color);
	}

	delete [] select_color;
}

// it seems that there is something common for all models (like the sel-buffer) -> move to the docview?!?!
// it seems that there is something common for all models (like the sel-buffer) -> move to the docview?!?!
// it seems that there is something common for all models (like the sel-buffer) -> move to the docview?!?!

void mm1_docv::Render(graphics_view * gv, rmode rm)
{
	const fGL default_label_color[3] = { 0.0, 1.0, 1.0 };	// looks bad but won't fade easily into other colours...
	fGL *label_color = model_prefs->ColorRGB("Graphics/LabelColor", default_label_color);
	
	bool accum = gv->accumulate; if (rm != Normal) accum = false;
//if (accum) { glClear(GL_ACCUM_BUFFER_BIT); UpdateAccumValues(); }
//else if (rm != Transform2) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	if (periodic && rm == Normal)
	{
		glLineWidth(1.0);
		glColor3f(1.0, 0.0, 1.0);
		glBegin(GL_LINES);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], +box_hdim[2]);
		glVertex3f(+box_hdim[0], -box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], +box_hdim[1], +box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], -box_hdim[1], -box_hdim[2]);
		
		glVertex3f(-box_hdim[0], +box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], -box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(-box_hdim[0], -box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], +box_hdim[1], -box_hdim[2]);
		glVertex3f(-box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(+box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], -box_hdim[1], +box_hdim[2]);
		
		glVertex3f(+box_hdim[0], +box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(-box_hdim[0], +box_hdim[1], -box_hdim[2]);
		
		glVertex3f(-box_hdim[0], -box_hdim[1], +box_hdim[2]);
		glVertex3f(-box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glVertex3f(+box_hdim[0], -box_hdim[1], -box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], -box_hdim[2]);
		
		glVertex3f(+box_hdim[0], -box_hdim[1], +box_hdim[2]);
		glVertex3f(+box_hdim[0], +box_hdim[1], +box_hdim[2]);
		
		glEnd();
	}
	
	if (gv->enable_fog) glEnable(GL_FOG);
	
	i32s layers = 0;
	if (periodic && rm == Normal) layers = 1;
	
	for (i32s r1 = -layers;r1 < (layers + 1);r1++)
	{
		for (i32s r2 = -layers;r2 < (layers + 1);r2++)
		{
			for (i32s r3 = -layers;r3 < (layers + 1);r3++)
			{
				glPushMatrix();
				
				fGL trans1 = r1 * box_fdim[0];
				fGL trans2 = r2 * box_fdim[1];
				fGL trans3 = r3 * box_fdim[2];
				
				glTranslated(trans1, trans2, trans3);
				
				RenderScene(gv, rm, accum);
				
				glPopMatrix();
			}
		}
	}
	
	if (accum) glAccum(GL_RETURN, 1.0);
	else if (rm != Transform2) gv->cam->RenderObjects(gv);
	
	if (gv->label == LABEL_INDEX)
	{
		char string[32]; i32s tmp1 = 0;
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << tmp1++ << ends;
			
			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_CHARGE)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str.setf(ios::fixed | ios::showpos); str.precision(4); str << (* it1).charge << ends;

			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_ELEMENT)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << (* it1).el.GetSymbol() << ends;

			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_ATOMTYPE)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << "0x" << hex << (* it1).atmtp << ends;

			fGL x = (* it1).crd_vector[0].data[0];
			fGL y = (* it1).crd_vector[0].data[1];
			fGL z = (* it1).crd_vector[0].data[2];
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	else if (gv->label == LABEL_BONDTYPE)
	{
		char string[32];
		
		glColor3f(label_color[0], label_color[1], label_color[2]);
		for (iter_mm1bl it1 = bond_list.begin();it1 != bond_list.end();it1++)
		{
			ostrstream str(string, sizeof(string));
			str << (* it1).bt.GetSymbol1() << ends;

			fGL x = ((* it1).atmr[0]->crd_vector[0][0] + (* it1).atmr[1]->crd_vector[0][0]) / 2.0;
			fGL y = ((* it1).atmr[0]->crd_vector[0][1] + (* it1).atmr[1]->crd_vector[0][1]) / 2.0;
			fGL z = ((* it1).atmr[0]->crd_vector[0][2] + (* it1).atmr[1]->crd_vector[0][2]) / 2.0;
			
			gv->WriteGlutString3D(string, x, y, z, gv->cam->GetLocData(), GLUT_BITMAP_9_BY_15);
		}
	}
	
	if (gv->enable_fog) glDisable(GL_FOG);
	
	// finally call this to handle transparency...
	// finally call this to handle transparency...
	// finally call this to handle transparency...
	
	RenderAllTPs(gv, rm);

	delete [] label_color;
}

void mm1_docv::RenderScene(graphics_view * gv, rmode rm, bool accum)
{
	for (i32u n1 = 0;n1 < cs_vector.size();n1++)
	{
		if (!cs_vector[n1].visible) continue;
if (accum) glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);		// FIXME!!!

		if (gv->render == RENDER_WIREFRAME)
		{
			glPointSize(3.0); glLineWidth(1.0);
			for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)		// wireframe atoms
			{
				if (!TestAtom(& (* it1), rm)) continue;
				glPushName(GLNAME_MD_TYPE1); glPushName((i32u) & (* it1));
				
				glBegin(GL_POINTS);
				SetColor(gv->colormode, & (* it1));
				glVertex3fv((* it1).crd_vector[n1].data);
				glEnd();
				
				glPopName(); glPopName();
			}
			
			glEnable(GL_LINE_STIPPLE);
			for (iter_mm1bl it2 = bond_list.begin();it2 != bond_list.end();it2++)		// wireframe bonds
			{
				if (!TestBond(& (* it2), rm)) continue;
				
				switch ((* it2).bt.GetSymbol1())
				{
					case 'S': glLineStipple(1, 0xFFFF); break;
					case 'C': glLineStipple(1, 0x3FFF); break;
					case 'D': glLineStipple(1, 0x3F3F); break;
					case 'T': glLineStipple(1, 0x3333); break;
				}
				
				glBegin(GL_LINES);
				SetColor(gv->colormode, (* it2).atmr[0]);
				glVertex3fv((* it2).atmr[0]->crd_vector[n1].data);
				SetColor(gv->colormode, (* it2).atmr[1]);
				glVertex3fv((* it2).atmr[1]->crd_vector[n1].data);
				glEnd();
			}
			glDisable(GL_LINE_STIPPLE);
		}
		
		if (gv->render != RENDER_WIREFRAME && gv->render != RENDER_NOTHING)
		{
			glEnable(GL_LIGHTING);
			
			for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)		// atoms as spheres
			{
				if (!TestAtom(& (* it1), rm)) continue;
				
				SetColor(gv->colormode, & (* it1));
				
				float rad = 0.0; int res = 0;
				switch (gv->render)
				{
					case RENDER_BALL_AND_STICK:
					rad = model_prefs->Double("MM1Graphics/BallSize", 0.035);
					if (model_prefs->Boolean("MM1Graphics/BallVdWScale", false))
					    rad *= (* it1).el.GetVDWRadius() * 4.0;
					res = model_prefs->Value("MM1Graphics/BallResolution", 12);
					break;
					
					case RENDER_VAN_DER_WAALS:
					rad = (* it1).el.GetVDWRadius();
					res = model_prefs->Value("MM1Graphics/VdWResolution", 22);
					break;
					
					case RENDER_CYLINDERS:
					rad = model_prefs->Double("MM1Graphics/CylinderSize", 0.035);
					res = model_prefs->Value("MM1Graphics/CylinderResolution", 12);
					break;
				}
				
				glPushName(GLNAME_MD_TYPE1); glPushName((i32u) & (* it1));
				
				GLUquadricObj * qo = gluNewQuadric();
				gluQuadricDrawStyle(qo, (GLenum) GLU_FILL);
				
				glPushMatrix();
				glTranslated((* it1).crd_vector[n1][0], (* it1).crd_vector[n1][1], (* it1).crd_vector[n1][2]);
				gluSphere(qo, rad, res, res / 2);
				glPopMatrix();
				gluDeleteQuadric(qo);
				
				glPopName(); glPopName();
			}
			
			glDisable(GL_LIGHTING);
		}
		
		if (gv->render == RENDER_BALL_AND_STICK || gv->render == RENDER_CYLINDERS)
		{
			glEnable(GL_LIGHTING);
			
			for (iter_mm1bl it1 = bond_list.begin();it1 != bond_list.end();it1++)		// bonds as cylinders
			{
				if (!TestBond(& (* it1), rm)) continue;
				
				fGL vdwr[2] =
				{
					(* it1).atmr[0]->el.GetVDWRadius(),
					(* it1).atmr[1]->el.GetVDWRadius()
				};
				
				fGL vdwrsum = vdwr[0] + vdwr[1];
				
				for (i32s n2 = 0;n2 < 2;n2++)
				{
					fGL * crd1 = (* it1).atmr[n2]->crd_vector[n1].data;
					fGL * crd2 = (* it1).atmr[!n2]->crd_vector[n1].data;
					v3d<fGL> crt1 = v3d<fGL>(crd1);
					v3d<fGL> crt2 = v3d<fGL>(crd2);
					v3d<fGL> crt = crt2 - crt1;
					
					fGL pol[3]; crt2pol(crt.data, pol);
					
					SetColor(gv->colormode, (* it1).atmr[n2]);
					
					float trans, rad = 0.0; int res = 0;
					switch (gv->render)
					{
						case RENDER_BALL_AND_STICK:
						rad = model_prefs->Double("MM1Graphics/StickSize", 0.01);
						res = model_prefs->Value("MM1Graphics/StickResolution", 6);
						break;
						
						case RENDER_CYLINDERS:
						rad = model_prefs->Double("MM1Graphics/CylinderSize", 0.035);
						res = model_prefs->Value("MM1Graphics/CylinderResolution", 12);
						break;
					}
					
					glPushName(GLNAME_MD_TYPE1); glPushName((i32u) (* it1).atmr[n2]);
					
					GLUquadricObj * qo = gluNewQuadric();
					gluQuadricDrawStyle(qo, (GLenum) GLU_FILL);
					glPushMatrix();
					
					glTranslated(crd1[0], crd1[1], crd1[2]);
					
					glRotated(180.0 * pol[1] / M_PI, 0.0, 1.0, 0.0);
					glRotated(180.0 * pol[2] / M_PI, sin(-pol[1]), 0.0, cos(-pol[1]));
					
					// any chance to further define the orientation of, for example, double bonds???
					// one more rotation would be needed. but what is the axis, and how much to rotate???
					
					fGL length = crt.len() * vdwr[n2] / vdwrsum;
					
					if (gv->render == RENDER_BALL_AND_STICK)
					switch ((* it1).bt.GetValue())
					{
						case BONDTYPE_DOUBLE:
						trans = rad;
						rad = rad / 1.5;
						
						if (n2)
							glTranslated(0.0, trans, 0.0);
						else
							glTranslated(0.0, -trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);					
						if (n2)
							glTranslated(0.0, -2.0 * trans, 0.0);
						else
							glTranslated(0.0, 2.0 * trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						break;
						
						case BONDTYPE_CNJGTD:
						trans = rad;
						rad = rad / 1.5;
						
						if (n2)
							glTranslated(0.0, trans, 0.0);
						else
							glTranslated(0.0, -trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						if (n2)
							glTranslated(0.0, -2.0 * trans, 0.0);
						else
							glTranslated(0.0, 2.0 * trans, 0.0);
						
						glEnable(GL_LINE_STIPPLE);
						glLineStipple(1, 0x3F3F);
						gluQuadricDrawStyle(qo, (GLenum) GLU_LINE);
						gluCylinder(qo, rad, rad, length, res, 1);
						glDisable(GL_LINE_STIPPLE);
						break;
						
						case BONDTYPE_TRIPLE:
						trans = rad;
						rad = rad / 2.0;
						
						if (n2)
							glTranslated(0.0, trans, 0.0);
						else
							glTranslated(0.0, -trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						if (n2)
							glTranslated(0.0, -trans, 0.0);
						else
							glTranslated(0.0, trans, 0.0);
						gluCylinder(qo, rad, rad, length, res, 1);
						if (n2)
							glTranslated(0.0, -trans, 0.0);
						else
							glTranslated(0.0, trans, 0.0);
						gluCylinder(qo, rad, rad, length, res,1);
						break;
						
						default:
						gluCylinder(qo, rad, rad, length, res, 1);
					}
					else
						gluCylinder(qo, rad, rad, length, res, 1);
					
					glPopMatrix();
					gluDeleteQuadric(qo);
					
					glPopName(); glPopName();
				}
			}
			
			glDisable(GL_LIGHTING);
		}
		
		if (accum)
		{
			gv->cam->RenderObjects(gv);
			glAccum(GL_ACCUM, cs_vector[n1].accum_value);
		}
	}
}

void mm1_docv::Center(transformer * p1)
{
	i32s sum = 0;
	p1->GetLocDataRW()->crd[0] = 0.0;
	p1->GetLocDataRW()->crd[1] = 0.0;
	p1->GetLocDataRW()->crd[2] = 0.0;
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			sum++;
			p1->GetLocDataRW()->crd[0] += (* it1).crd_vector[n1][0];
			p1->GetLocDataRW()->crd[1] += (* it1).crd_vector[n1][1];
			p1->GetLocDataRW()->crd[2] += (* it1).crd_vector[n1][2];
		}
	}
	
	if (!sum) return;
	
	p1->GetLocDataRW()->crd[0] /= (fGL) sum;
	p1->GetLocDataRW()->crd[1] /= (fGL) sum;
	p1->GetLocDataRW()->crd[2] /= (fGL) sum;
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			(* it1).crd_vector[n1][0] -= p1->GetLocData()->crd[0];
			(* it1).crd_vector[n1][1] -= p1->GetLocData()->crd[1];
			(* it1).crd_vector[n1][2] -= p1->GetLocData()->crd[2];
		}
	}
}

void mm1_docv::Transform(transformer * p1)
{
	fGL matrix[16]; p1->GetMatrix(matrix);
	
	for (iter_mm1al it1 = atom_list.begin();it1 != atom_list.end();it1++)
	{
		if (!(* it1).selected) continue;
		
		for (i32u n1 = 0;n1 < cs_vector.size();n1++)
		{
			v3d<fGL> posv = v3d<fGL>((* it1).crd_vector[n1].data);
			TransformVector(posv, matrix);
			
			(* it1).crd_vector[n1][0] = posv[0];
			(* it1).crd_vector[n1][1] = posv[1];
			(* it1).crd_vector[n1][2] = posv[2];
		}
	}
}

void mm1_docv::DrawEvent(graphics_view * gv, vector<iGLu> & names)
{
	if (ogl_view::button == mouse_tool::Right) return;	// the right button is for popup menus...
	
	i32s mouse[2] =
	{
		gv->current_tool->latest_x,
		gv->current_tool->latest_y
	};
	
	if (ogl_view::state == mouse_tool::Down)
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[0] = (mm1_atom *) names[1];
		}
		else
		{
			fGL tmp1[3]; gv->GetCRD(mouse, tmp1);
			mm1_atom newatom(element::current_element, tmp1, cs_vector.size());
			AddAtom(newatom); draw_data[0] = & atom_list.back();
		}
	}
	else
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[1] = (mm1_atom *) names[1];
		}
		else
		{
			fGL tmp1[3]; gv->GetCRD(mouse, tmp1);
			mm1_atom newatom(element::current_element, tmp1, cs_vector.size());
			AddAtom(newatom); draw_data[1] = & atom_list.back();
		}
		
		// if different: update bondtype or add a new bond.
		// if not different: change atom to different element.
		
		if (draw_data[0] != draw_data[1])
		{
			mm1_bond newbond(draw_data[0], draw_data[1], bondtype::current_bondtype);
			iter_mm1bl it1 = find(bond_list.begin(), bond_list.end(), newbond);
			if (it1 != bond_list.end()) (* it1).bt = bondtype::current_bondtype;
			else AddBond(newbond);
		}
		else
		{
			draw_data[0]->el = element::current_element;
		}
		
		UpdateAllGraphicsViews();
	}
}

void mm1_docv::EraseEvent(graphics_view * gv, vector<iGLu> & names)
{
	if (ogl_view::button == mouse_tool::Right) return;	// the right button is for popup menus...
	
	if (ogl_view::state == mouse_tool::Down)
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[0] = (mm1_atom *) names[1];
		}
		else
		{
			draw_data[0] = NULL;
		}
	}
	else
	{
		if (names.size() > 1 && names[0] == GLNAME_MD_TYPE1)
		{
			draw_data[1] = (mm1_atom *) names[1];
		}
		else
		{
			draw_data[1] = NULL;
		}
		
		if (!draw_data[0] || !draw_data[1]) return;
		
		// if different: try to find and remove a bond.
		// if not different: remove atom.
		
		if (draw_data[0] != draw_data[1])
		{
			mm1_bond tmpbond(draw_data[0], draw_data[1], bondtype::current_bondtype);
			iter_mm1bl it1 = find(bond_list.begin(), bond_list.end(), tmpbond);
			if (it1 != bond_list.end()) RemoveBond(it1); else return;
		}
		else
		{
			iter_mm1al it1 = find(atom_list.begin(), atom_list.end(), (* draw_data[0]));
			if (it1 != atom_list.end()) RemoveAtom(it1); else exit(EXIT_FAILURE);
		}
		
		UpdateAllGraphicsViews();
	}
}

void mm1_docv::SelectEvent(graphics_view *, vector<iGLu> & names)
{
	if (names[0] == GLNAME_MD_TYPE1)
	{
		mm1_atom * ref = (mm1_atom *) names[1];
		ref->selected = !ref->selected;
		UpdateAllGraphicsViews();
		
		//		cout << "atomtype is " << hex << ref->atmtp << dec << endl;
	}
}

void mm1_docv::MeasureEvent(graphics_view *, vector<iGLu> & names)
{
        static mm1_atom *a1 = NULL;
	static mm1_atom *a2 = NULL;
	static mm1_atom *a3 = NULL;
// this function is quite similar in mm1_docv and qm1_docv. could this be shorter as a loop???

	if (names[0] == GLNAME_MD_TYPE1)
	{
		mm1_atom * ref = (mm1_atom *) names[1];
		ref->selected = !ref->selected;
		UpdateAllGraphicsViews();
		
		if (a1 == NULL)
		  {
		    a1 = ref;
		    cout << "charge: " << ref->charge << endl;
		  }
		else if (a1 != NULL && a2 == NULL)
		  {
		    if (a1 == ref) { a1->selected = false; a1 = NULL; return; }
		    
		    a2 = ref;
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    
		    float len = measure_len(p1, p2);
		    cout << "distance: " << len << " nm" << endl;
		  }
		else if (a1 != NULL && a2 != NULL && a3 == NULL)
		  {
		    if (a1 == ref) { a1->selected = false; a1 = a2; a2 = NULL; return; }
		    else if (a2 == ref) { a2->selected = false; a2 = NULL; return; }
		    
		    a3 = ref;
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    float * p3 = a3->crd_vector[0].data;
		    
		    float ang = measure_ang(p1, p2, p3);
		    cout << "angle: " << ang << " deg" << endl;
		  }
		else
		  {
		    if (a1 == ref) { a1->selected = false; a1 = a2; a2 = a3; a3 = NULL; return; }
		    else if (a2 == ref) { a2->selected = false; a2 = a3; a3 = NULL; return; }
		    else if (a3 == ref) { a3->selected = false; a3 = NULL; return; }
		    
		    float * p1 = a1->crd_vector[0].data;
		    float * p2 = a2->crd_vector[0].data;
		    float * p3 = a3->crd_vector[0].data;
		    float * p4 = ref->crd_vector[0].data;
		    
		    float tor = measure_tor(p1, p2, p3, p4);
		    cout << "torsion: " << tor << " deg " << endl;
		    
		    a1->selected = false; a1 = NULL;
		    a2->selected = false; a2 = NULL;
		    a3->selected = false; a3 = NULL;
		    ref->selected = false;
		    
		    UpdateAllGraphicsViews();
		  }
	}
}

void mm1_docv::DoEnergyPlot1D(void)
{
	char question; i32s osteps = 100;
	i32s qm_eng_index = NOT_DEFINED; i32s qm_total_charge = 0;
	
	i32s molnum = 0; i32s in_crdset = 0;
	
	f64 range1[2] = { 0.0, 2.0 * M_PI }; i32s steps1 = 36;
	i32s atmi1[4]; mm1_atom * atmr1[4];
cout << "give the atom ID numbers (separated by spaces) for torsion : ";
cin >> atmi1[0] >> atmi1[1] >> atmi1[2] >> atmi1[3];
cout << "do you want to change the default settings (y=yes)? "; cin >> question;
if (question == 'y' || question == 'Y')
{
	cout << "give the number of steps (-1 = cancel, default = " << steps1 << ")? "; cin >> steps1;
	if (steps1 < 0) return;
	
	f64 range[2] = { 180.0 * range1[0] / M_PI, 180.0 * range1[0] / M_PI };
	cout << "give the starting angle (degrees, default = " << range[0] << ")? "; cin >> range[0];
	cout << "give the ending angle (degrees, default = " << range[1] << ")? "; cin >> range[1];
	
	if (range[0] >= range[1]) { cout << "error: starting angle must be smaller than ending angle." << endl; return; }
	if (range[1] - range[0] > 360.0) { cout << "error: the range is now more than full circle 360 degrees." << endl; return; }
	
	range1[0] = M_PI * range[0] / 180.0;
	range1[1] = M_PI * range[1] / 180.0;
}

cout << "give the optimization steps (-1 to skip optimization, default = " << osteps << ")? "; cin >> osteps;
cout << "should the final energy be calculated using a QM method (y=yes)? "; cin >> question;
if (question == 'y' || question == 'Y')
{
	i32s index = 0;
	while (true)
	{
		if (qm1_mdl::engtab1[index] == NULL) break;
		
		cout << index << " " << qm1_mdl::engtab1[index] << endl;
		index++;
	}
	
	cout << "give the method from the above list (or -1 to cancel)? "; cin >> qm_eng_index;
	if (qm_eng_index >= index) qm_eng_index = NOT_DEFINED;	// check the index...
	
	cout << "give the total charge for the QM model (default = " << qm_total_charge << ")? "; cin >> qm_total_charge;
}

	for (i32s n1 = 0;n1 < 4;n1++)
	{
		iter_mm1al it1;
		
		it1 = FindAtomByIndex(atmi1[n1]);
		if (it1 == GetAtomsEnd()) cout << "atom not found!" << endl;
		atmr1[n1] = & (* it1);
	}
	
	GatherGroups();		// for internal coordinates...
	mm1_intcrd * ic = new mm1_intcrd((* this), molnum, in_crdset);
	i32s ict1 = ic->FindTorsion(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
if (ict1 < 0) { cout << "could not find ic tor1." << endl; return; }
	
	v3d<fGL> v1a(atmr1[1]->crd_vector[in_crdset].data, atmr1[0]->crd_vector[in_crdset].data);
	v3d<fGL> v1b(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
	v3d<fGL> v1c(atmr1[2]->crd_vector[in_crdset].data, atmr1[3]->crd_vector[in_crdset].data);
	f64 oldt1 = v1a.tor(v1b, v1c);
	
	mm1_eng * eng1 = CreateDefaultEngine();
	i32s fft1 = eng1->FindTorsion(atmr1[0], atmr1[1], atmr1[2], atmr1[3]);
if (ict1 < 0) { cout << "could not find ff tor1." << endl; return; }

	mm1_eng * eng2 = CreateDefaultEngine();
	
	plot1d_view * plot = graphics_factory->ProducePlot1DView(this, PLOT_USERDATA_STRUCTURE, 0);
	
	f64 tor1 = range1[0];
	for (i32s s1 = 0;s1 < (steps1 + 1);s1++)
	{
		ic->SetTorsion(ict1, tor1 - oldt1);
		ic->UpdateCartesian();
		
		CopyCRD(this, eng1, 0);		// lock_local_structure needs coordinates!!!
		eng1->SetTorsionConstraint(fft1, tor1, 5000.0, true);
		
		// optimize...
		
		mm1_geomopt * opt = new mm1_geomopt(eng1, 100, 0.025);		// optimal settings?!?!?
		
		for (i32s n1 = 0;n1 < osteps;n1++)
		{
			opt->TakeCGStep(conjugate_gradient::Newton2An);
			cout << n1 << " " << opt->optval << " " << opt->optstp << endl;
			
			if (!(n1 % 10))
			{
				CopyCRD(eng1, this, 0); CenterCRDSet(0);
				UpdateAllGraphicsViews(true);
			}
		}
		
		CopyCRD(eng1, this, 0); CenterCRDSet(0);
		delete opt;
		
		// compute energy for final structure...
		
		f64 value;
		if (qm_eng_index < 0)
		{
			CopyCRD(this, eng2, 0);
			eng2->Compute(0);
			
			value = eng2->energy;
		}
		else
		{
			qm1_mdl * mdl9 = new qm1_mdl(& cout, * console_class_factory::GetInstance());
			mdl9->aai_MakeCopy(this);
			
			mdl9->default_eng = qm_eng_index;
			mdl9->SetTotalCharge(qm_total_charge);
			qm1_eng * eng9 = mdl9->CreateDefaultEngine();
			
			if (eng9 != NULL)
			{
				CopyCRD(mdl9, eng9, 0);
				eng9->Compute(0);
				
				value = eng9->energy;
				
				delete eng9;
				delete mdl9;
			}
			else
			{
				value = 0.0;	// if something went wrong...
				delete mdl9;	// ...we just use zero!!!
			}
		}
		
		// ...and add it to the plot.
		
		void * udata = mm1_convert_cset_to_plotting_udata(this, 0);
		plot->AddData(180.0 * tor1 / M_PI, value, udata);
		
		tor1 += (range1[1] - range1[0]) / (f64) steps1;
	}
	
	plotting_view_vector.push_back(plot);
	plot->SetCenterAndScale();
	plot->Update();
	
	delete eng2;
	delete eng1;
	delete ic;
}

void mm1_docv::DoEnergyPlot2D(void)
{
	char question; i32s osteps = 100;
	i32s qm_eng_index = NOT_DEFINED; i32s qm_total_charge = 0;
	
	i32s molnum = 0; i32s in_crdset = 0;
	
	f64 range1[2] = { 0.0, 2.0 * M_PI }; i32s steps1 = 18;
	i32s atmi1[4]; mm1_atom * atmr1[4];
cout << "give the atom ID numbers (separated by spaces) for torsion1 : ";
cin >> atmi1[0] >> atmi1[1] >> atmi1[2] >> atmi1[3];
cout << "do you want to change the default settings (y=yes)? "; cin >> question;
if (question == 'y' || question == 'Y')
{
	cout << "give the number of steps (-1 = cancel, default = " << steps1 << ")? "; cin >> steps1;
	if (steps1 < 0) return;
	
	f64 range[2] = { 180.0 * range1[0] / M_PI, 180.0 * range1[0] / M_PI };
	cout << "give the starting angle (degrees, default = " << range[0] << ")? "; cin >> range[0];
	cout << "give the ending angle (degrees, default = " << range[1] << ")? "; cin >> range[1];
	
	if (range[0] >= range[1]) { cout << "error: starting angle must be smaller than ending angle." << endl; return; }
	if (range[1] - range[0] > 360.0) { cout << "error: the range is now more than full circle 360 degrees." << endl; return; }
	
	range1[0] = M_PI * range[0] / 180.0;
	range1[1] = M_PI * range[1] / 180.0;
}

	f64 range2[2] = { 0.0, 2.0 * M_PI }; i32s steps2 = 18;
	i32s atmi2[4]; mm1_atom * atmr2[4];
cout << "give the atom ID numbers (separated by spaces) for torsion2 : ";
cin >> atmi2[0] >> atmi2[1] >> atmi2[2] >> atmi2[3];
cout << "do you want to change the default settings (y=yes)? "; cin >> question;
if (question == 'y' || question == 'Y')
{
	cout << "give the number of steps (-1 = cancel, default = " << steps2 << ")? "; cin >> steps2;
	if (steps2 < 0) return;
	
	f64 range[2] = { 180.0 * range2[0] / M_PI, 180.0 * range2[0] / M_PI };
	cout << "give the starting angle (degrees, default = " << range[0] << ")? "; cin >> range[0];
	cout << "give the ending angle (degrees, default = " << range[1] << ")? "; cin >> range[1];
	
	if (range[0] >= range[1]) { cout << "error: starting angle must be smaller than ending angle." << endl; return; }
	if (range[1] - range[0] > 360.0) { cout << "error: the range is now more than full circle 360 degrees." << endl; return; }
	
	range2[0] = M_PI * range[0] / 180.0;
	range2[1] = M_PI * range[1] / 180.0;
}

cout << "give the optimization steps (-1 to skip optimization, default = " << osteps << ")? "; cin >> osteps;
cout << "should the final energy be calculated using a QM method (y=yes)? "; cin >> question;
if (question == 'y' || question == 'Y')
{
	i32s index = 0;
	while (true)
	{
		if (qm1_mdl::engtab1[index] == NULL) break;
		
		cout << index << " " << qm1_mdl::engtab1[index] << endl;
		index++;
	}
	
	cout << "give the method from the above list (or -1 to cancel)? "; cin >> qm_eng_index;
	if (qm_eng_index >= index) qm_eng_index = NOT_DEFINED;	// check the index...
	
	cout << "give the total charge for the QM model (default = " << qm_total_charge << ")? "; cin >> qm_total_charge;
}

	for (i32s n1 = 0;n1 < 4;n1++)
	{
		iter_mm1al it1;
		
		it1 = FindAtomByIndex(atmi1[n1]);
		if (it1 == GetAtomsEnd()) cout << "atom not found #1!" << endl;
		atmr1[n1] = & (* it1);
		
		it1 = FindAtomByIndex(atmi2[n1]);
		if (it1 == GetAtomsEnd()) cout << "atom not found #2!" << endl;
		atmr2[n1] = & (* it1);
	}
	
	GatherGroups();		// for internal coordinates...
	mm1_intcrd * ic = new mm1_intcrd((* this), molnum, in_crdset);
	i32s ict1 = ic->FindTorsion(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
if (ict1 < 0) { cout << "could not find ic tor1." << endl; return; }
	i32s ict2 = ic->FindTorsion(atmr2[1]->crd_vector[in_crdset].data, atmr2[2]->crd_vector[in_crdset].data);
if (ict2 < 0) { cout << "could not find ic tor2." << endl; return; }
	
	v3d<fGL> v1a(atmr1[1]->crd_vector[in_crdset].data, atmr1[0]->crd_vector[in_crdset].data);
	v3d<fGL> v1b(atmr1[1]->crd_vector[in_crdset].data, atmr1[2]->crd_vector[in_crdset].data);
	v3d<fGL> v1c(atmr1[2]->crd_vector[in_crdset].data, atmr1[3]->crd_vector[in_crdset].data);
	f64 oldt1 = v1a.tor(v1b, v1c);
	
	v3d<fGL> v2a(atmr2[1]->crd_vector[in_crdset].data, atmr2[0]->crd_vector[in_crdset].data);
	v3d<fGL> v2b(atmr2[1]->crd_vector[in_crdset].data, atmr2[2]->crd_vector[in_crdset].data);
	v3d<fGL> v2c(atmr2[2]->crd_vector[in_crdset].data, atmr2[3]->crd_vector[in_crdset].data);
	f64 oldt2 = v2a.tor(v2b, v2c);
	
	mm1_eng * eng1 = CreateDefaultEngine();
	i32s fft1 = eng1->FindTorsion(atmr1[0], atmr1[1], atmr1[2], atmr1[3]);
if (ict1 < 0) { cout << "could not find ff tor1." << endl; return; }
	i32s fft2 = eng1->FindTorsion(atmr2[0], atmr2[1], atmr2[2], atmr2[3]);
if (ict2 < 0) { cout << "could not find ff tor2." << endl; return; }

	mm1_eng * eng2 = CreateDefaultEngine();
	
	plot2d_view * plot = graphics_factory->ProducePlot2DView(this, PLOT_USERDATA_STRUCTURE, 0);
	
	f64 tor1 = range1[0];
	for (i32s s1 = 0;s1 < (steps1 + 1);s1++)
	{
		f64 tor2 = range2[0];
		for (i32s s2 = 0;s2 < (steps2 + 1);s2++)
		{
			ic->SetTorsion(ict1, tor1 - oldt1);
			ic->SetTorsion(ict2, tor2 - oldt2);
			ic->UpdateCartesian();
			
			CopyCRD(this, eng1, 0);		// lock_local_structure needs coordinates!!!
			eng1->SetTorsionConstraint(fft1, tor1, 5000.0, true);
			eng1->SetTorsionConstraint(fft2, tor2, 5000.0, true);
			
			// optimize...
			
			mm1_geomopt * opt = new mm1_geomopt(eng1, 100, 0.025);		// optimal settings?!?!?
			
			for (i32s n1 = 0;n1 < osteps;n1++)
			{
				opt->TakeCGStep(conjugate_gradient::Newton2An);
				cout << n1 << " " << opt->optval << " " << opt->optstp << endl;
				
				if (!(n1 % 10))
				{
					CopyCRD(eng1, this, 0); CenterCRDSet(0);
					UpdateAllGraphicsViews(true);
				}
			}
			
			CopyCRD(eng1, this, 0); CenterCRDSet(0);
			delete opt;
			
			// compute energy for final structure...
			
			f64 value;
			if (qm_eng_index < 0)
			{
				CopyCRD(this, eng2, 0);
				eng2->Compute(0);
				
				value = eng2->energy;
			}
			else
			{
				qm1_mdl * mdl9 = new qm1_mdl(& cout, * console_class_factory::GetInstance());
				mdl9->aai_MakeCopy(this);
				
				mdl9->default_eng = qm_eng_index;
				mdl9->SetTotalCharge(qm_total_charge);
				qm1_eng * eng9 = mdl9->CreateDefaultEngine();
				
				if (eng9 != NULL)
				{
					CopyCRD(mdl9, eng9, 0);
					eng9->Compute(0);
					
					value = eng9->energy;
					
					delete eng9;
					delete mdl9;
				}
				else
				{
					value = 0.0;	// if something went wrong...
					delete mdl9;	// ...we just use zero!!!
				}
			}
			
			// ...and add it to the plot.
			
			void * udata = mm1_convert_cset_to_plotting_udata(this, 0);
			plot->AddData(180.0 * tor1 / M_PI, 180.0 * tor2 / M_PI, value, udata);
			
			tor2 += (range2[1] - range2[0]) / (f64) steps2;
		}
		
		tor1 += (range1[1] - range1[0]) / (f64) steps1;
	}
	
	plotting_view_vector.push_back(plot);
	plot->SetCenterAndScale();
	plot->Update();
	
	delete eng2;
	delete eng1;
	delete ic;
}

/*################################################################################################*/

void mm1_cm_element::GetColor(const void * p1, fGL_a4 & p2, prefs *p3)
{
	mm1_atom * ref = (mm1_atom *) p1;
	const fGL * color = ref->el.GetColor(p3);
	p2[0] = color[0]; p2[1] = color[1]; p2[2] = color[2];
}

void mm1_cm_state::GetColor(const void * p1, fGL_a4 & p2, prefs *p3)
{
//	mm1_atom * ref = (mm1_atom *) p1;
//	mm1_mdl * mdl = ref->mdl;		// CURRENTLY BROKEN...
	
	p2[0] = 0.0; p2[1] = 0.0; p2[2] = 1.0;		// loop
	
/*	if (mdl->chn_info == NULL) return;
	vector<mm1_chn_info *> & ci = * mdl->chn_info;
	
	if (ref->id[1] < 0 || ref->id[2] < 0) return;
	if (ci[ref->id[1]]->state == NULL) return;
	
	char state = ci[ref->id[1]]->state[ref->id[2]];
	
	switch (state)
	{
		case '4':
		p2[0] = 1.0; p2[1] = 0.0; p2[2] = 0.0;		// helix
		return;
		
		case 'S':
		p2[0] = 0.0; p2[1] = 1.0; p2[2] = 0.0;		// strand
		return;
	}	*/
}

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS

// eof
