// CONJGRAD.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "conjgrad.h"

#include <math.h>

/*################################################################################################*/

conjugate_gradient::conjugate_gradient(i32s p1, f64 p2, f64 p3)
{
	step = 0; reset = p1;
	defstp = p2; delta = p3;
}

conjugate_gradient::~conjugate_gradient(void)
{
}

void conjugate_gradient::AddVar(f64 * p1, f64 * p2)
{
	cgvar newvar; newvar.ref1 = p1; newvar.ref2 = p2;
	cgvar_vector.push_back(newvar);
}

void conjugate_gradient::TakeCGStep(ls_mode mode)
{
	optstp = 0.0;
	optval = GetGradient();
	
	// compute inner product of the latest gradient to newip.
	
	newip = 0.0;
	for (i32u n1 = 0;n1 < cgvar_vector.size();n1++)
	{
		f64 tmp1 = (* cgvar_vector[n1].ref2);
		newip += tmp1 * tmp1;
	}
	
	// either reset beta or update it, using the Fletcher-Reeves method.
	
	if (!(step++ % reset) || oldip == 0.0)
	{
		beta = 0.0;
		for (i32u n1 = 0;n1 < cgvar_vector.size();n1++)
		{
			cgvar_vector[n1].data1 = 0.0;
		}
	}
	else
	{
		beta = newip / oldip;
	}
	
	oldip = newip;
	
	// store the current search direction and the current value for a while...
	
	f64 sum = 0.0;
	for (i32u n1 = 0;n1 < cgvar_vector.size();n1++)
	{
		cgvar_vector[n1].data2 = (* cgvar_vector[n1].ref1);
		f64 tmp1 = cgvar_vector[n1].data1 * beta - (* cgvar_vector[n1].ref2);
		cgvar_vector[n1].data1 = tmp1; sum += tmp1 * tmp1;
	}
	
	f64 scale = sqrt(sum);
	f64 stpln = defstp / scale;
	
	// do the line search...
	
	if (mode == Simple)
	{
		i32s bad_cycle_count = 0;
		bool second_good_cycle = false;
		
		while (bad_cycle_count < 10)
		{
			InitLineSearch(stpln);
			f64 tmp1 = GetValue();
			
			if (tmp1 < optval)
			{
				optstp = stpln;
				optval = tmp1;
				
				if (second_good_cycle) break;
				
				stpln *= 1.1;
				second_good_cycle = true;
			}
			else
			{
				stpln *= 0.5;
				bad_cycle_count++;
			}
		}
	}
	
	if (mode == Newton2An)
	{
		i32s newton = 0;
		
		while (true)
		{
			InitLineSearch(stpln);
			f64 tmp1 = GetGradient();
			
			if (tmp1 < optval)
			{
				optstp = stpln;
				optval = tmp1;
			}
			
			if (newton++ > 3) break;
			f64 delta = stpln * 0.0001;
			
			f64 tmp2 = 0.0; f64 tmp3 = 0.0;
			for (i32u n1 = 0;n1 < cgvar_vector.size();n1++)
			{
				tmp2 += cgvar_vector[n1].data1 * (* cgvar_vector[n1].ref2) / scale;
			}
			
			InitLineSearch(stpln + delta);
			GetGradient();
			
			for (i32u n1 = 0;n1 < cgvar_vector.size();n1++)
			{
				tmp3 += cgvar_vector[n1].data1 * (* cgvar_vector[n1].ref2) / scale;
			}
			
			f64 denom = tmp3 - tmp2;
			if (denom != 0.0) stpln = fabs(stpln - delta * tmp2 / denom);
			else break;
		}
	}
	
	if (mode == Newton2Num)
	{
		i32s newton = 0;
		
		while (true)
		{
			InitLineSearch(stpln);
			f64 tmp1 = GetValue();
			
			if (tmp1 < optval)
			{
				optstp = stpln;
				optval = tmp1;
			}
			
			if (newton++ > 3) break;
			f64 delta = stpln * 0.0001;
			
			InitLineSearch(stpln + delta);
			f64 tmp2 = GetValue();
			
			InitLineSearch(stpln + delta * 2.0);
			f64 tmp3 = GetValue();
			
			f64 denom = tmp3 - 2.0 * tmp2 + tmp1;
			if (denom != 0.0) stpln = fabs(stpln - delta * (tmp2 - tmp1) / denom);
			else break;
		}
	}
		
	if (mode == Newton1Num)
	{
		i32s newton = 0;
		
		while (true)
		{
			InitLineSearch(stpln);
			f64 tmp1 = GetValue();
			
			if (fabs(tmp1) < fabs(optval))
			{
				optstp = stpln;
				optval = tmp1;
			}
			
			if (newton++ > 3) break;
			f64 delta = stpln * 0.0001;
			
			InitLineSearch(stpln + delta);
			f64 tmp2 = GetValue();
			
			f64 denom = tmp2 - tmp1;
			if (denom != 0.0) stpln = fabs(stpln - delta * tmp1 / denom);
			else break;
		}
	}
	
	// finally take the optimal step and scale back.
	
	InitLineSearch(optstp);
	optstp *= scale;
}

void conjugate_gradient::InitLineSearch(f64 p1)
{
	for (i32u n1 = 0;n1 < cgvar_vector.size();n1++)
	{
		f64 tmp1 = cgvar_vector[n1].data2 + cgvar_vector[n1].data1 * p1;
		(* cgvar_vector[n1].ref1) = tmp1;
	}
}

// how to set optimal value for delta??? now we set a default value in ctor...
// how to set optimal value for delta??? now we set a default value in ctor...
// how to set optimal value for delta??? now we set a default value in ctor...

f64 conjugate_gradient::GetGradient(void)
{
	f64 value = GetValue();
	for (i32u n1 = 0;n1 < cgvar_vector.size();n1++)
	{
		f64 old = (* cgvar_vector[n1].ref1);
		(* cgvar_vector[n1].ref1) = old + delta;
		(* cgvar_vector[n1].ref2) = (GetValue() - value) / delta;
		(* cgvar_vector[n1].ref1) = old;
	}
	
	return value;
}

/*################################################################################################*/

// eof
