/*
 * $Id: targetarray.c,v 1.3 2006/04/04 00:19:37 lorenzo Exp $
 * 
 * Copyright (C) 2006 RIPE NCC
 * 
 * Original developer: Lorenzo Colitti <lorenzo@ripe.net>
 * Contributor(s):
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include "pinger.h"

#define LINELEN 255
#define BLOCKSIZE 512 * 1024

/**
 * Reads an line from the file
 */
char *getline(FILE *f) {
	static char line[LINELEN]; // 15 characters for an IP, [\r], \n, \0

	if(fgets(line, sizeof(line), f) == NULL)
		return NULL;

	/* Strip newlines */
	if(line[strlen(line) - 1] == '\n')
		line[strlen(line) - 1] = '\0';
	if(line[strlen(line) - 1] == '\r')
		line[strlen(line) - 1] = '\0';

	return line;
}

/**
 * @param filename the name of the file to open, or "-" for stdin
 * @return a FILE, or NULL if the open fails
 */
FILE *openfile(char *filename) {
	FILE *f;

	if(! strcmp(filename, "-")) {
		f = stdin;
	} else {
		f = fopen(filename, "r");
	}

	return f;
}

void dumptargets(struct targetlist targetlist) {
	int i;
	for(i = 0; i < targetlist.len; i++) {
		printf("%s\n", ip2str((struct sockaddr *) &targetlist.targets[i].addr));
	}
}

/**
 * Reads a list of IP addresses from a file
 * @param file the file to open, or "-" for standard input
 * @return a pointer to the target array
 */
struct targetlist readlist(struct options *options) {
	struct targetlist targetlist = { 0, NULL };
	struct target *array = NULL;
	FILE *f;
	int i, numblocks;
	struct target *target;
	char *line;

	/* Open file */
	f = openfile(options->filename);
	if(!f)
		die(options->filename);

	/* Initially allocate space for BLOCKSIZE targets */
	numblocks = 1;
	array = calloc(numblocks * BLOCKSIZE, sizeof(struct target));
	if(! array)
		die("calloc");

	for(i = 0; (line = getline(f)) != NULL; i++) {
		/* Allocate more space in array if necessary */
		if( (i + 1) > numblocks * BLOCKSIZE) {
			numblocks++;
			array = realloc(array, numblocks * BLOCKSIZE * sizeof(struct target));
			if(!array)
				die("realloc");
		}

		/* Initialize target */
		target = array + i;
		if(parse_ip(line, options->port, (struct sockaddr *) &target->addr, sizeof(target->addr))) {
			fprintf(stderr, "Invalid IP address %s\n", line);
			free(array);
			return(targetlist);
		}
		if(pthread_mutex_init(&target->lock, NULL)) {
			perror("pthread_mutex_init");
			free(array);
			return targetlist;
		}
		target->id = i;
		target->nsent = 0;
		target->nreceived = 0;
		target->replies = NULL;
	}

	targetlist.len = i;
	targetlist.targets = array;

	return targetlist;
}
