# Copyright (c) 1996, 1997    The TERENA Association
# Copyright (c) 1998, 1999    RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Filename          :   Whois.pm
# Purpose           :   Provide routines for interfacing with the 'whois' client.
# Author            :   Lee Wilmot
# Date              :   19971117
# Language Version  :   Perl 5.003_07
# OSs Tested        :   BSD
# Input Files       :   NONE
# Output Files      :   NONE
# External Programs :   $RobotConfig::WHOIS_COM
# Comments          :
#------------------------------------------------------------------------------

######################### PACKAGE INTERFACE ###########################
#
# Please see the package concerned for descriptions of imported symbols.

package Whois;

use strict;

BEGIN {

    use vars qw ( @ISA @EXPORT_OK );

    use Exporter ();

    @ISA = qw(Exporter);
    
    @EXPORT_OK = qw(
        &indb &handle_in_arin_db
    );

}

use RobotConfig qw(
    :WHOIS_VARS :DEBUG_FLAGS
);

use Misc qw(
    &eprint &dprint &fatal &strip_space
);

use Check qw( 
    &prob
);

######################### PACKAGE BODY ###########################

#   Purpose :   Decide whether an object of a specified type is in
#               the RIPE Database.
#   In      :   $:  type of object to check for ('person', 'role' etc).
#               $:  the string to search on.
#               $%: a pointer to the meta template.
#   Out     :   $:  the KEY of the object found, or "" if an object of the
#                   type requested was not found.
#
sub indb {

    # Type of object to search for, 'person', 'role' etc

    my $object_type = shift @_;

    # The value to return if we had a problem with 'whois'. This means
    # the calling routine can deal with this however it wants.

    my $whois_problem_return_value = shift @_;

    # String to search on

    my $search_string = shift @_;

    # We need a pointer to the meta template so we can give warnings if there's
    # something goes wrong.

    my $meta = shift @_;    

    # Compose the external command to use

    my $whois_com = "$WHOIS_COM $WHOIS_RECURSEOFF_OPT $WHOIS_OBJECTSELECT_OPT $object_type $search_string 2>&1 |";

    my $count = 0;      # track number of retry times

    my $success_flag = -1;  # track whether we need to try again
                            # -1 = nothing found due to a problem
                            # 0 = got negative response to query
                            # 1 = got positive response to query

    # $success_flag tracks the state of THIS subroutine. $return_val tracks the
    # value which the outside world sees. It's either a key (success), 
    # or "" (failure).
        
    my $return_val = "";    

    &dprint("LOOKUP in RIPE DB with '$whois_com'");

    # Until we've got some sort of result, loop

    while ( $success_flag == -1 ) {

        if  ( open ( WHOIS, $whois_com ) ) {
    
            # If there were no problems opening the pipe,
            # scan through the results...
    
            while ( <WHOIS> ) {
    
                # Success condition is finding a line with
                # field name of the type we're looking for. This
                # should be OK, since we turned recursion off
        
                if ( /$object_type\s*:(.*)$/i ) {
        
                    # Mark 'success', set the key as the return value and stop
        
                    $success_flag = 1;
                    $return_val = $1;
                    last;
                }
        
                # Failure condition is a regular expression being found which
                # the database uses to indicate 'no match found'.
        
                if (/ $WHOIS_NOTFOUND_REG/i ) {
        
                    # Mark 'failure', leave the $return_value as "", stop.
        
                    $success_flag = 0;
                    last;
                }
    
            }
        }
    
        close WHOIS;

        &eprint("$? from '$whois_com', but continuing.")
            if ( $? );
    
        # If we failed to get either a positive or negative result...   
    
        if ( $success_flag == -1 ) {
    
            # If we've already tried the specified number of times, note the
            # problem and return.
    
            if ( $count++ >= $WHOIS_RETRY_TIMES ) {
    
                &prob($meta, "WHOIS_PROBLEM", "$! while running '$whois_com', even after $count tries at $WHOIS_RETRY_PERIOD second intervals");
    
                # Return with the failure value requested by the calling routine
        
                return $whois_problem_return_value;
            }

            # otherwise, print an error to STDERR, sleep for a bit and try again.
    
            &eprint("warning: problem with '$whois_com' $!. Trying again in $WHOIS_RETRY_PERIOD seconds.");              
    
                sleep $WHOIS_RETRY_PERIOD;
        }
    }

    # Return either the key or "", with white space removed.

    return ( &strip_space($return_val) );
}

#   Purpose :   Decide whether a search key is in the ARIN DB
#   In      :   $:  the string to search with.
#               $%: a pointer to the meta template.
#   Out     :   $:  boolean: success or failure
#
sub handle_in_arin_db {

    # String to search on

    my $search_string = shift @_;

    # We need a pointer to the meta template so we can give warnings if there's
    # something goes wrong.

    my $meta = shift @_;

    # Add the -ARIN suffix to the query if it's not already there

    $search_string .= $ARINDB_CONTACT_HANDLE_SUFFIX
        if ( $search_string !~ /$ARINDB_CONTACT_HANDLE_SUFFIX\s*$/i );

    # Compose the external command to use

    my $whois_com = "$WHOIS_COM $ARINDB_USE_OPT $ARINDB_SEARCH_OPT $search_string 2>&1 |";

    my $count = 0;      # track number of retry times

    my $success_flag = -1;  # Track whether we need to try again
                            # -1 = nothing found due to a problem
                            # 0 = got negative response
                            # 1 = got positive response

    # Check flag for checking ARIN DB is on.

    &dprint("LOOKUP in ARIN DB with '$whois_com'");

    if ( ! $CHECK_ARIN_DB_FLAG ) {
        &dprint("STOPPED: CHECK_ARIN_DB_FLAG = $CHECK_ARIN_DB_FLAG");
        return 1;
    }

    # Until we've got some sort of result, loop

    while ( $success_flag == -1 ) {

        if  ( open WHOIS, $whois_com ) {
    
            # If there were no problems opening the pipe,
            # scan through the results...
    
            while ( <WHOIS> ) {
    
                # Success condition is a piece of text to that effect
                # put out by the ARIN DB
        
                if ( /$ARINDB_FOUND_REG/i ) {
        
                    # Mark 'success' and stop
        
                    $success_flag = 1;
                    last;
                }
        
                # Same for failure condition
        
                if ( /$ARINDB_NOTFOUND_REG/i ) {
        
                    # Mark 'failure' and stop
        
                    $success_flag = 0;
                    last;
                }
            }
        }
        
        close WHOIS;

        &eprint("$? from '$whois_com', but continuing.")
            if ( $? );
    
        # If we failed to get either a positive or negative result...   
        
        if ( $success_flag == -1 ) {
    
            # If we've already tried the specified number of times, note the
            # problem and return.
    
            if ( $count++ >= $ARINDB_RETRY_TIMES ) {
    
                &prob($meta, "WHOIS_PROBLEM", "$! while running '$whois_com', even after $count tries at $ARINDB_RETRY_PERIOD second intervals");
        
                # Return a 'found' value, so that we don't generate further problems
                # about this
        
                return 1;
            }
    
            # Warn there's a problem on STDERR and sleep for a bit.
    
            &eprint("warning: problem with '$whois_com' $!. Trying again in $ARINDB_RETRY_PERIOD seconds.");              
   
            sleep $ARINDB_RETRY_PERIOD;
        }
    }
    
    return $success_flag;
}

1;
