# Copyright (c) 1996, 1997                     The TERENA Association
# Copyright (c) 1998, 1999                     RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
#   Filename            :   RTT.pm
#   Purpose             :   Provide routines for interfacing with the RTT program
#   Author              :   Lee Wilmot
#   Date                :   1997/12/03
#   Language Version    :   Perl 5.003_07
#   OSs Tested          :   BSD
#   Command Line        :   Nothing executable from command line
#   Input Files         :   NONE
#   Output Files        :   NONE
#   External Programs   :   $RobotConfig::RTT_COM
#   Problems            :   
#
# Had some problems with RTT return codes. Often an error code is
# returned if, for instance, a ticket number doesn't exist. This makes it
# difficult to check for problems with closing the pipe etc, since 'close'
# is sensitive to the error code of the program being run.
# Therefore the following plan was adopted:
#
# 1) If we have something definite to look for, the error condition is that we didnt
# find it.
# 2) If we don't have something to look for, the error condition is 
# '! close PIPE || $RTT_ERROR_OUTPUT_REG'
# Since conceivably RTT will return error code '0', if there's an error.
# In this case the regular expression will catch the error output.
#

######################### PACKAGE INTERFACE ############################
#
# Please see the package concerned for descriptions of imported symbols.

package RTT;

use strict;

BEGIN {

    use vars qw( @ISA @EXPORT_OK);
    use Exporter ();
    @ISA = qw(Exporter);
    
    @EXPORT_OK = qw(
        &make_new_ticket &change_ticket &add_log_message_to_ticket
        &add_message_to_ticket &get_list_of_duplicate_tickets
        &number_of_messages_on_ticket &old_to_new_ticket_format
        &ticket_exists &find_regid_in_ticket
    );
}

use Misc qw(
        &dprint &eprint &fatal &strip_space &preprocess_line 
);

use RobotConfig qw(
        :RTT_VARS :DEBUG_FLAGS :REGULAR_EXPRESSIONS
        %F $UNIQUE_AFFIX
        $ROBOT_NAME
);

######################### PACKAGE BODY ############################

# Purpose       :       Make a new ticket.
# In            :       $: registry ID to enter on ticket
#           $: origin to enter on ticket
# Out           :       $: ticket number of new ticket
#
sub make_new_ticket {

    my ( $regid, $origin ) =  @_;

    if ( $TICKETIZE_FLAG < 2 ) {
        &dprint("MAKETICKET: TICKETIZE_FLAG = $TICKETIZE_FLAG, returning dummy $RTT_DEBUG_TICKET");
        return ( $RTT_DEBUG_TICKET );
    };


}

# Purpose       :   Add a message from a file to a ticket
# In            :   $: ticket number
#                   $: location of file to add
#                   $: new status to set on ticket
# Out           :   VOID
#
sub add_message_to_ticket {

    my ( $tickno, $to_add_file, $new_status_of_ticket) = @_;

    if ( $TICKETIZE_FLAG == 1 ) {
        $tickno = $RTT_DEBUG_TICKET;
        &dprint("ADDING MESSAGE TO TICKET: using dummy tickno $tickno");
    }


}

# Purpose       :   Return a list of tickets which contain a netname field
#                   the same as that given.
# In            :   $: The ticket number to check.
#                   $: The netname to look for.
#                   $: The regid to use in the search.
# Out           :   $: A string holding ticket numbers, or ""
# Method        : 
#
# 1) rtt scan to get tickets matching o=a r=REGID s=a
# 2) For each of these, check if the netname in the first message is the same as
# the one were looking for (i.e. the one in the current request).
# We can't use, e.g. 'diff', because a) the mail headers change b) the thing is
# going to look different anyway because they've changed it to correct the
# problems that were pointed out!
#
sub get_list_of_duplicate_tickets {

    my ( $tickno, $netname, $regid ) = @_;
    
    &dprint("DUPLICATE TICKETS checking");

    if ( ! $CHECK_DUPLICATES_FLAG ) {
        &dprint("STOPPED: CHECK_DUPLICATES_FLAG = $CHECK_DUPLICATES_FLAG");
        return "";
    }


}

# Purpose       :       Count the number of messages on a ticket.
# In            :       $: the ticket to count
# Out           :       $: the number of messages on the ticket
#
sub number_of_messages_on_ticket {

    my $ticket_to_examine = shift @_;

    # Compose the command to run

    my $rtt_com =   
        $RTT_COM.                           " ".
        $RTT_FLAGS{'show ticket details'}.  " ". 
        $ticket_to_examine.                 " ".
        "2>&1 |";

    &dprint("COUNTING MESSAGES: '$rtt_com'");

    if ( $NO_TICKETS_PERIOD_FLAG ) {
        &dprint("STOPPED: NO_TICKETS_PERIOD_FLAG= $NO_TICKETS_PERIOD_FLAG");
        return 0;
    }


}

# Purpose       :       Count the number of messages on a ticket.
# In            :       $: the ticket to count
# Out           :       $: the number of messages on the ticket
#
sub find_regid_in_ticket {

    my $ticket_to_examine = shift @_;

    # Compose the command to run

    my $rtt_com =   
        $RTT_COM.                           " ".
        $RTT_FLAGS{'show ticket details'}.  " ". 
        $ticket_to_examine.                 " ".
        "2>&1 |";

    &dprint("FINDING REGID IN TICKET: '$rtt_com'");

    if ( $NO_TICKETS_PERIOD_FLAG ) {
        &dprint("STOPPED: NO_TICKETS_PERIOD_FLAG= $NO_TICKETS_PERIOD_FLAG");
        return 0;
    }


}

# Purpose       :       Set an attribute on a ticket to a value
# In            :       $: the number of the ticket
#           $: the attribute to set
# Out           :       $: the value to set it to
#
sub change_ticket {

    my ( $tickno, $attribute, $value ) = ( @_ );

    if ($TICKETIZE_FLAG == 1) {
        $tickno = $RTT_DEBUG_TICKET;
        &dprint("DEBUG: using tickno $tickno");
    }

    # Compose the command to run

    my $rtt_com = 
        $RTT_COM.                           " ".
        $RTT_FLAGS{'change ticket'}.        " ". 
        $RTT_OPTS{'tickno'}.        $tickno." ". 
        $RTT_OPTS{$attribute}.      $value. " ". 
        "2>&1 |";

    &dprint("CHANGING TICKET: '$rtt_com'");

    if ($TICKETIZE_FLAG == 0) {
        &dprint("STOPPED: TICKETIZE_FLAG = $TICKETIZE_FLAG");
        return;
    };


}

# Purpose       :       Add a log message to a ticket
# In            :       $: ticket numebr
#           $: message to add
# Out       :   VOID
#
sub add_log_message_to_ticket {

    my ( $tickno, $message_to_add ) = @_;

    if ($TICKETIZE_FLAG == 1) {
        $tickno = $RTT_DEBUG_TICKET;
        &dprint("DEBUG: using tickno $tickno");
    }

    # Compose the command to run

    my $rtt_com =   
    $RTT_COM.                   " ".
    $RTT_FLAGS{'add to log'}.   " ". 
    $tickno.                    " ".
    $message_to_add.            " ".
    "2>&1 |";

    &dprint("ADDING LOG MESSAGE: '$rtt_com'");

    if ($TICKETIZE_FLAG == 0) {
        &dprint("STOPPED: TICKETIZE_FLAG = $TICKETIZE_FLAG");
        return;
    };


}

# Purpose       :       Convert a ticket number from old to new format if necessary
# In            :       $: the ticket number
# Out           :       $: the converted ticket number, or the same ticket number if
#           it is already a new style ticket number
#
sub old_to_new_ticket_format {


}

# Purpose       :       Decide whether a ticket number corresonds to an existing ticket.
# In            :       $: ticket number
# Out           :       $: boolean: yes/no
#
sub ticket_exists {

    my ( $tickno ) = @_;

    my $rtt_com =   
        $RTT_COM.                           " ".
        $RTT_FLAGS{'show ticket details'}.  " ". 
        $tickno.                            " ".
        "2>&1 |";

    if ( $NO_TICKETS_PERIOD_FLAG ) {
        return $RTT_DEBUG_TICKET;
    }

}

1;
