# Copyright (c) 1998, 1999    RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Filename          :   Misc.pm
# Purpose           :   Provide miscellaneous functions for use by the Robot packages
# Author            :   Lee Wilmot
# Date              :   971212
# Language Version  :   Perl5, version 5.003_07
# OSs Tested        :   BSD
# Command Line      :   Nothing executable from command line.
# Input Files       :   $RobotConfig::CCODE_FILE            The country codes files
#                       $RobotConfig::FROMCUST_MAIL_FILE    The mail received from customer
# Output Files      :   $RobotConfig::DIAGNOSTIC_FILE       Where to store diagnostic output
#                   :   $RobotConfig::FATAL_FILE            Where to store fatal error
#                                                           report (in some modes)
# External Programs :   $RobotConfig::SENDMAIL_COM $RobotConfig::HOSTNAME_COM

# Problems          :
# To Do             :
# Comments          :
#------------------------------------------------------------------------------

######################## PACKAGE INTERFACE #############################
#
# Please see the package concerned for descriptions of imported symbols.

package Misc;

use strict;

BEGIN {

    use vars qw( @ISA @EXPORT_OK );

    use Exporter ();

    @ISA = qw( Exporter );
    
    use Extract qw(
           @TEMPLATES
    );
    
    @EXPORT_OK = qw(

        &dprint &eprint &fatal &exit_gracefully
       
        &ismask &quad2int &int2quad &slash2int &isnichandle
        &isquad &mask2int &subnetsize2mask
        &isemail &iscountry &country_to_code
        &is_internal_field_name 

        &get_date &add_to_log &send &make_filename

        &absent &strip_space &preprocess_line &ljustify &centre_pad
        
        &parse_arguments &clr_tmp_dir &chk_tmp_dir &check_debug_flags

        &add_to_field
       
        &get_values &key_of_field

        &find_regid_in_message &find_tickno_in_message
    );
}

use RTT qw( 
        &add_message_to_ticket &make_new_ticket &old_to_new_ticket_format
);

use ExternalMisc qw(
        &get_regids &get_ticketnumbers
);

# MGTM added, to decide if should show a given problem: REPORT_TYPE_FLAG,
#      %REPORT_TYPE
use RobotConfig qw(        
        :DEBUG_FLAGS :REGULAR_EXPRESSIONS

        @SPECIAL_COUNTRIES

        %F $FORM_FS @F_INTERNAL

        $SENDMAIL_COM
        @SENDMAIL_FLAGS
        $DEBUG_REGID

        $DOMAINNAME_REG $QUOTED_LINE_START_REG $TICKNO_REG $DOSCONVERSION_REG

        $CCODE_FILE $FROMCUST_MAIL_FILE $DIAGNOSTIC_FILE $TOCUST_MAIL_FILE
        $TMP_DIR $FROMCUST_LOG_DIR $FATAL_FILE

        $ROBOT_NAME $ROBOT_TITLE

        $EMAIL_REPORT_NO_OF_COLUMNS 
        
        $TMP_FILE_MAX_AGE $UNIQUE_AFFIX

        $LOCK_EX $LOCK_UN 
       
        $FATAL_ERROR_EXIT_TXT $VERSION_NUMBER

        $REPORT_TYPE_FLAG

	%REPORT_TYPE
);

use TemplateAux qw(
    &rough_show_template
);

######################## PACKAGE INTERFACE #############################

#   Purpose :   Read & interpret arguments given to an invocation of the robot
#   In      :   VOID
#   Out     :   VOID
#
sub parse_arguments {

    my $arg;

    while ( defined ($arg = shift @ARGV) ) {

        # The test mode is intended to allow testing on the command line
        # without needing to know about the config file settings

        if ( $arg eq "-debug" || $arg eq "-d" ) {

            $MODE_FLAG = 0;

        }

        elsif ($arg eq "-test" || $arg eq "-t") {

            $MODE_FLAG = 1;

            $VERBOSE_FLAG = 0;
            $EMAIL_FLAG = 1;
            $LOGGING_FLAG = 0;
            $TICKETIZE_FLAG = 0;
            $REGID_FLAG = 0;
            $NO_TICKETS_PERIOD_FLAG = 1;
            $CLR_TMPFILES_FLAG = 1;
            $CHECK_DUPLICATES_FLAG = 0;
            $CHECK_ARIN_DB_FLAG = 0;
            $SHOW_INTERNAL_FIELDS_FLAG = 0;
            $DETAIL_EXTRACT_FLAG = 0;
            
            print STDOUT <<"            EOF";
        
            auto-hm (v$VERSION_NUMBER)
    
            This mode sends no emails, does nothing 
            ticket-related, doesn't log anything, doesn't 
            check ARIN DB, doesn't check duplicate tickets.
    
            Play away!
    
            ---------------------------------------------------
    
            EOF
        }

        # The debug mode is so the maintainer can be sure that we are in debug mode
        # and using the actual settings of the debug flags, even
        # if it's set differently in the config file.
    
        elsif ( $arg eq "-standalone" || $arg eq "-s" ) {

            $MODE_FLAG = 2;

            $VERBOSE_FLAG = 1;
            $EMAIL_FLAG = 2;
            $LOGGING_FLAG = 1;
            $TICKETIZE_FLAG = 0;
            $REGID_FLAG = 0;
            $NO_TICKETS_PERIOD_FLAG = 1;
            $CLR_TMPFILES_FLAG = 1;
            $CHECK_DUPLICATES_FLAG = 0;
            $CHECK_ARIN_DB_FLAG = 1;
            $SHOW_INTERNAL_FIELDS_FLAG = 0;
            $DETAIL_EXTRACT_FLAG = 0;

        }
        elsif ( $arg eq "-real" || $arg eq "-r" ) {

            $MODE_FLAG = 3;

            $VERBOSE_FLAG = 0;
            $EMAIL_FLAG = 2;
            $LOGGING_FLAG = 1;
            $TICKETIZE_FLAG = 2;
            $REGID_FLAG = 1;
            $NO_TICKETS_PERIOD_FLAG = 0;
            $CLR_TMPFILES_FLAG = 1;
            $CHECK_DUPLICATES_FLAG = 1;
            $CHECK_ARIN_DB_FLAG = 1;
            $SHOW_INTERNAL_FIELDS_FLAG = 0;
            $DETAIL_EXTRACT_FLAG = 0;

        }

	# MGTM
        elsif ($arg eq "-web" || $arg eq "-w") {

            $MODE_FLAG = 4;

            $REPORT_TYPE_FLAG = $REPORT_TYPE{HTML};

            $VERBOSE_FLAG = 0;
            $EMAIL_FLAG = 1;
            $LOGGING_FLAG = 0;
            $TICKETIZE_FLAG = 0;
            $REGID_FLAG = 0;
            $NO_TICKETS_PERIOD_FLAG = 1;
            $CLR_TMPFILES_FLAG = 1;
            $CHECK_DUPLICATES_FLAG = 0;
            $CHECK_ARIN_DB_FLAG = 0;
            $SHOW_INTERNAL_FIELDS_FLAG = 0;
            $DETAIL_EXTRACT_FLAG = 0;

            # MGTM on STDERR not STDOUT
            print STDERR <<"            EOF";

            auto-hm (v$VERSION_NUMBER)

            This mode just displays html of results of syntax checks
            It sends no emails, does nothing
            ticket-related, doesn't log anything, doesn't
            check ARIN DB, doesn't check duplicate tickets.
            ---------------------------------------------------

            EOF
        }

        elsif ( $arg eq "-help" || $arg eq "-h" ) {
            &usage;
            exit (0);
        }
        else {
            &eprint( "Unrecognized argument: $arg" );   

            select STDERR;
            &usage;
            select STDOUT;
        }
    }
}

#   Purpose :   Output a usage description
#   In      :   VOID
#   Out     :   VOID
#
sub usage {

    print<<"EOF";
    usage:  | auto-hm.pl (-d[ebug] | -t[est] | -s[tandalone] | -r[eal]) [-h[elp]]
    flags:
        -d:     debug mode
        -t:     command line mode   - for RIPE NCC use.
        -s:     standalone mode     - answering queries sent to a mail box.
        -r:     'real' mode         - sitting behind distribution software.
                                      full ticketizing
        -h:     print this usage text
EOF
}

#   Purpose :   Check debug flags have sensible values, set them if necessary
#   In      :   VOID
#   Out     :   VOID
#
sub check_debug_flags {

    # Set flags to normal operation defaults if the master debug flag is off

    &flag_range_check( 'MODE_FLAG', $MODE_FLAG, 0, 4 );
    &flag_range_check( 'VERBOSE_FLAG', $VERBOSE_FLAG, 0, 1 );
    &flag_range_check( 'EMAIL_FLAG', $EMAIL_FLAG, 0, 2 );
    &flag_range_check( 'LOGGING_FLAG', $LOGGING_FLAG, 0, 1 );
    &flag_range_check( 'REGID_FLAG', $REGID_FLAG, 0, 1 );
    &flag_range_check( 'TICKETIZE_FLAG', $TICKETIZE_FLAG, 0, 2 );
    &flag_range_check( 'NO_TICKETS_PERIOD_FLAG', $NO_TICKETS_PERIOD_FLAG, 0, 1 );
    &flag_range_check( 'CHECK_DUPLICATES_FLAG', $CHECK_DUPLICATES_FLAG, 0, 2 );
    &flag_range_check( 'CHECK_ARIN_DB_FLAG', $CHECK_ARIN_DB_FLAG, 0, 1 );
    &flag_range_check( 'CLR_TMPFILES_FLAG', $CLR_TMPFILES_FLAG, 0, 1 );
    &flag_range_check( 'SHOW_INTERNAL_FIELDS_FLAG', $SHOW_INTERNAL_FIELDS_FLAG, 0, 1 );

    # If we don't want ANY ticket operations, turn off the relevant 
    # associated flags

    if ( $NO_TICKETS_PERIOD_FLAG ) {

        $TICKETIZE_FLAG = 0;
        $CHECK_DUPLICATES_FLAG = 0;
    }

    &dprint("Running in mode $MODE_FLAG");
}

#   Purpose :   Test a flag's value is within specified limits
#   In      :   $:  Flag name
#               $:  Flag value
#               $:  Minimum valid value for this flag
#               $:  Maximum valid value for this flag
#   Out     :   VOID
#
sub flag_range_check {

    my ( $flag_name, $flag_value, $min, $max ) = @_;

    &fatal("Bad setting for flag in $flag_name: $flag_value. Please check config file!")
        if ( $flag_value < $min || $flag_value > $max );
}


#   Purpose :   Routine to do diagnostic output.
#   In      :   $:  The diagnostic message.
#   Out     :   VOID
#
sub dprint {

    my $message = shift @_;
   
    # Put the message on STDOUT if required

    print STDOUT "$message\n"
        if ( $VERBOSE_FLAG );

    # Store it also in the diagnostic file for possible later perusal

    &append_message_to_file( &make_filename($DIAGNOSTIC_FILE), "$message\n" );

}

#   Purpose :   Routine to do STDERR output.
#   In      :   $:  The error message.
#               $:  numeric: an exit code [OPTIONAL]
#   Out     :   VOID
#
sub eprint {

    my ( $message ) = @_;
   
    my ($package, $filename, $line, $caller, $hasargs, $wantarray) = caller 1;

    # Get the name of the diagnostic file for this invocation of the robot.

    my $file_to_write = &make_filename($DIAGNOSTIC_FILE);

    # Add a couple of things to the message

    my $message_to_write = "$ROBOT_NAME: $message\t[$caller]\n";

    # Output it to STDERR if required by the relevant flag

    print STDERR $message_to_write
        if ( $VERBOSE_FLAG );

    # Store it also in the diagnostic file for possible later perusal

    &append_message_to_file($file_to_write, $message_to_write);
}

#   Purpose :   Exit the program in a nice way :-)
#   In      :   $:  The exit code to leave with.
#   Out     :   VOID
#
sub exit_gracefully {

    my $exit_code = shift @_;
    my $exit_message = shift @_;

    my ($package, $filename, $line, $caller, $hasargs, $wantarray) = caller 1;

    $caller = "main" if ( ! $caller );

    print STDERR "$0: exiting with code $exit_code from $caller.\n";
    print STDERR $exit_message;
    
    exit $exit_code;
}

#   Purpose :   Given a message, append it to the end of the specified file.
#   In      :   $:  The file to write.
#               $:  The message to append.
#   Out     :   VOID
#
sub append_message_to_file {

    my ( $file_to_write, $message_to_write ) = @_;

    &fatal("$! while trying to open $file_to_write for writing the message '$message_to_write'")
        if ( ! open MESSAGEFILE, ">> $file_to_write");

    print MESSAGEFILE $message_to_write;

    &fatal("$! while trying to close $file_to_write after writing the message '$message_to_write'")
        if ( ! close MESSAGEFILE);

}

#   Purpose :   Perform appropriate actions when a fatal error occurs
#               during processing.
#   In      :   $:  A message from the calling routine giving the problem.
#   Out     :   VOID
#
#   The error output consists of 4 sections
#       1) The details: problem message, failed routine etc
#       2) The contents of $DIAGNOSTIC_FILE
#       3) The contents of the meta template.   [if this template defined   ]
#       4) The original customer mail.
#   The final action is to exit the program via &exit_gracefully
#
sub fatal {

    my $error_message = shift @_;

    # Set some variables for the report

    my $meta = $TEMPLATES[0];

    # Make a stack trace

    my ( $package, $file, $line, $caller, $hasargs, $wantarray );
    my ($i, $stack_trace ) = ( 0, "" );
    while (($package, $file, $line, $caller, $hasargs, $wantarray) = caller($i++)) {
        $stack_trace .= "\tPackage $package($file:line $line) called &$caller\n";
    }

    # Get details of the calling routine

    ( $package, $file, $line, $caller, $hasargs, $wantarray ) = caller 1;

    # Date and time

    my ( $date, $time ) = ( &get_date, &get_time );

    # A nicely formatted error message

    my ($justified_error_message) = &ljustify($error_message);

    # Select where the output goes dependent on the mode.
    # We have to select one of these because this routine
    # could be called while we have another output pipe
    # selected.

    my $fatal_file = &make_filename($FATAL_FILE); 

    if ( $MODE_FLAG == 0 || $MODE_FLAG == 1 )  { 
        select STDOUT;
    }

    # In standalone mode, write it to a file
    elsif ( $MODE_FLAG == 2 ) {
        
        if ( ! open FATAL, ">$fatal_file" ) {
            print STDERR "\n\n$! opening $fatal_file after fatal error.";
            &exit_gracefully( -1, $FATAL_ERROR_EXIT_TXT );
        }
        
        select FATAL;

    }
    else { 
        select STDERR; 
    }

    # Print out the first section of the failure report

    print <<"EOFEOFEOF";

Fatal Error

***
$justified_error_message
***

Helpful information follows :) ...

1) relevant files can probably be found in:

    $TMP_DIR/*.$UNIQUE_AFFIX

2) if all else fails, the customer mail is hopefully in

    $FROMCUST_LOG_DIR/$date, 

    Details: PID=$$, logged shortly before $time

3) program error info:

    PID: $$
    Subroutine generating fatal error: \&$caller

    Call stack: 
    $stack_trace

EOFEOFEOF

    # 19981208 Lee
    # Added printout of environment to aid in problem finding

    print "\n\n".&centre_pad("PROGRAM ENVIRONMENT")."\n\n";

    print 'Effective UID: ', $>, "\n";
    print 'Real UID: ', $<, "\n\n";

    foreach ( keys %ENV ) {
        print "$_: ", $ENV{$_}, "\n";
    }

    ### Include diagnostic file contents ###

    print "\n".&centre_pad("DIAGNOSTIC FILE CONTENTS")."\n\n";

    # Get the filename for this invocation of the robot.

    my $diagnostic_file = &make_filename($DIAGNOSTIC_FILE);

    print "[$diagnostic_file]\n\n";

    if ( open DIAGNOSTIC, $diagnostic_file ) {
        print
            while (<DIAGNOSTIC>);

        print "$! while closing $diagnostic_file, but continuing...\n"
            if (! close DIAGNOSTIC);
    }
    else {
        print "Couldn't open $diagnostic_file.\n";
    }

    ### Include meta template contents ###

    print "\n".&centre_pad("EMAIL TEMPLATE CONTENTS")."\n\n";

    if ( defined $meta ) {
        &rough_show_template( $meta );
    }
    else {
        print "Template is not defined.\n";
    }

    ### Include message from customer ###

    print "\n\n".&centre_pad("MESSAGE FROM CUSTOMER")."\n\n";
    
    my $fromcust_file = &make_filename($FROMCUST_MAIL_FILE);

    print "[$fromcust_file]\n\n";

    if ( open ( FROMCUST, $fromcust_file ) ) {

        print
            while (<FROMCUST>);

        print "$! while closing $fromcust_file, but continuing...\n"
            if (! close FROMCUST);
    }
    else {
        print "Couldn't open $fromcust_file.\n";
    }

    if ( $MODE_FLAG == 2 ) {
        if ( ! close FATAL ) {
            print STDERR "\n\n$! closing $fatal_file after fatal error.\n";
            &exit_gracefully( -1, $FATAL_ERROR_EXIT_TXT );
        }
    }

    select STDOUT;

    # Exit the program with an error code.

    &exit_gracefully( -1, $FATAL_ERROR_EXIT_TXT );           
}

#   Purpose :   Return the current date in yrmndy format
#   In      :   VOID
#   Out     :   $:  The date.
#
sub get_date {

    my ( $day, $month, $year ) = ( (localtime) [3], (localtime) [4], (localtime) [5] );

    # Make sure it's a 2 digit day
    $day = "0".$day
        if ( length ($day) == 1 );

    # Make sure it's a 2 digit month
    $month = "0".$month
        if ( length (++$month) == 1 );

    # Correct for yr2000

    $year += 1900;

    return $year.$month.$day;
}

#   Purpose :   Return the current time in HH:MM:SS format.
#   In      :   VOID
#   Out     :   $:  The time.
#
sub get_time {

    my ($sec, $min, $hour, @rest) = localtime(time);

    return "$hour:$min:$sec";
}

#   Purpose :   centre a string, surround it with padding
#   In      :   $:  the string
#   Out     :   $:  the centred, padded string
#
sub centre_pad {

    my $old_string = shift @_;

    # it doesn't make sense to do this to things with \n in

    $old_string =~ s/\n//g;     

    my $new_string = "";

    my $old_string_length = length ($old_string) + 2;

    my $diff = $EMAIL_REPORT_NO_OF_COLUMNS - $old_string_length;

    # if it's too big to pad, leave it as-is

    return $old_string
        if ( $diff <= 0 );

    $new_string = "#" x ($diff/2);                          # prefix it with #'s
    $new_string .= " ".$old_string." ";                     # add the old string
    $new_string .= "#" x ($EMAIL_REPORT_NO_OF_COLUMNS - length ($new_string)); # affix #'s

    return $new_string;
}

#   Purpose :   Send an email from a specified file.
#   In      :   $:  A filename
#   Out     :   VOID
#   Comments:   The file is expected to have a 'To:' line.
#
#
sub send {

    my $tosend_file = shift @_;

    &dprint("SENDING from $tosend_file");

    # Take different actions depending on the debug flag setting

    ### Don't do anything.... ###

    if ( $EMAIL_FLAG == 0 ) {
        &dprint("STOPPED: EMAIL_FLAG = $EMAIL_FLAG");
    }

    ### Print it to STDOUT... ###

    elsif ( $EMAIL_FLAG == 1 ) {

        # don't need an &fatal call here, this just test mode

        &eprint("$! while opening $tosend_file.")
            if ( ! ( open FILETOSEND,  $tosend_file ) );
        
        # This lot prints the mail, and highlights who it would
        # would normally go to

        my $found_to_flag = 0;

        while ( <FILETOSEND> ) {

            # Highlight the first 'To:' we find. But only the first.

            if ( ! $found_to_flag  && /$F{EMAILTO}:/ ) {

                chop;

                print STDOUT "******", $_, "********\n";

                $found_to_flag = 1;

            }
            else { 
                print STDOUT;
            }

        }

        &eprint("$! while closing $tosend_file after reading. Not fatal, continuing...")
            if ( ! close FILETOSEND );
    }

    # Send it as an email...

    elsif ( $EMAIL_FLAG == 2 ) {

        my $send_flags = $SENDMAIL_FLAGS[$MODE_FLAG];

        # Set command depending on whether we're debugging or not

        my $send_com = "$SENDMAIL_COM $send_flags 1>/dev/null 2>/dev/null <$tosend_file";

        &dprint("SYSTEM: $send_com");   
      
        # Send the mail, check for problems
        # sendmail returns error code 67 for addressee unknown
        # This is no reason to produce a fatal error in the robot
        # however.

        my $send_result = system("$send_com");

        $send_result >>= 8;

        &fatal("$send_com failed $!")
            if ( $send_result != 67 && $send_result != 0 );
    }

}

#   Purpose :   add a file to the end of a log file
#   In      :   $:  the directory holding the logfile
#               $:  the file to add
#   Out     :   VOID
#   Comments:   The name of the log file is simply the date, that's why
#               we only need to be passed the name of the directory to put
#               it in.
#
sub add_to_log {

    my ( $dir, $toadd_file ) =  @_;

    # Make up the filename, based on the directory and date

    my $log_file = "$dir/".&get_date;

    &dprint("LOGGING $toadd_file to $log_file");

    if (! $LOGGING_FLAG) {
        &dprint("STOPPED: LOGGING_FLAG = $LOGGING_FLAG");
        return;
    }

    &fatal("$! while opening logfile $log_file to append new request.")
        if ( ! ( open LOGFILE, ">>$log_file" ) );

    # Obtain exclusive lock on the file: multiple robot invocations could
    # attempt to write it at once

    flock(LOGFILE, $LOCK_EX);
                        
    # Compensate for possible in-between appends

    seek(LOGFILE, 0, 2);

    &fatal("$! while opening file $toadd_file to add to logfile.")
        if ( ! ( open TOADD, $toadd_file ) );

    # Add an identifier before appending the file

    print LOGFILE "<<< ", &get_time, " PID=$$", ">>>\n";
    
    print LOGFILE
        while (<TOADD>);

    flock( LOGFILE, $LOCK_UN );

    &fatal("$! while closing logfile $log_file.")
        if ( ! close LOGFILE);

    &fatal("$! while closing pipe from $toadd_file. Continuing.")
        if ( ! close TOADD );
    
}

#   Purpose :   Given a basename, returns a unique (WARNING: only for this program!)
#               filename based on the PID to prevent multiple robots interfering with
#               each other.
#   In      :   $:  The basename of the file.
#   Out     :   $:  The unique name ( i.e. with affix appended ).
#
sub make_filename {

    my $base = shift @_;

    return ( sprintf("$base%s.$UNIQUE_AFFIX") );
}

#   Purpose :   Clear all temporary files which have been created in the working
#               directory (by this process only)
#   In      :   VOID
#   Out     :   VOID
#
sub clr_tmp_dir {

    &dprint("CLEAR TMP FILES");

    if (! $CLR_TMPFILES_FLAG) {
        &dprint("STOPPED: CLR_TMPFILES_FLAG = $CLR_TMPFILES_FLAG");
        return;
    }

    my $filepath;
    foreach $filepath ( &make_filename( $FROMCUST_MAIL_FILE ),
                        &make_filename( $TOCUST_MAIL_FILE ), 
                        &make_filename( $DIAGNOSTIC_FILE )
    ) {
        &eprint("failed to clear temp file $filepath. Continuing.")
            if ( -e $filepath && ! unlink $filepath );
    }
}

#   Purpose :   Remove any files in the problem directory which are above 
#               a certain age.
#   In      :   VOID
#   Out     :   VOID
#
sub chk_tmp_dir {

    my $secs_in_day = 86400;

    my $time_now = time;

    # Get all the files in $TMP_DIR

    opendir ( TMPDIR, $TMP_DIR ) 
        or &fatal( "$! opening directory $TMP_DIR" );

    my @all_tmp_files = readdir TMPDIR;

    closedir TMPDIR;

    # Check them all

    my $file;
    foreach $file ( @all_tmp_files  ) {

        next if ( $file eq '.' || $file eq '..' );

        my $path = join '/', $TMP_DIR, $file;

        my $file_time = ( stat($path) ) [9];
    
        my $age_in_secs = $time_now - $file_time;

        my $age_in_days = $age_in_secs / $secs_in_day;
    
        if ( $age_in_days > $TMP_FILE_MAX_AGE ) {

            &eprint("warning, couldn't delete out of date file $path. Continuing.")
                if ( ! unlink $path );
        }
    }
}

#   Purpose :   Left justify a piece of text in a string  by inserting newlines 
#               as appropriate for $EMAIL_REPORT_NO_OF_COLUMNS.
#   In      :   $:  The text to justify.
#   Out     :   $:  The justified text.
#
sub ljustify {

    my $old_string = shift @_;
    
    my ( $new_string, $current_column, $word, $wordlength ) = ( "", 1, undef, undef );

    foreach $word (split / +/, $old_string ) {

        # If we find a /n already in the text, honour it and reset the
        # column count

        if ( $word =~ /\n/ ) {
            $current_column = 1;
            $new_string .= "$word";
            next;
        }

        $wordlength = length ($word);

        # If we would go over the right column, insert \n and reset the
        # column counter

        if ( $wordlength + $current_column > $EMAIL_REPORT_NO_OF_COLUMNS ) {
            $new_string .= "\n";
            $current_column = 1;
        }

        $new_string .= "$word ";                # Add the word

        $current_column += ($wordlength + 1);   # Update the column counter
    }
    
    return $new_string;
}

#   Purpose :   Say whether a scalar holds a valid dotted quad, or
#               a partial dotted quad. If so, return the number of bytes.
#   In      :   $:  The scalar to check.
#   Out     :   $:  numeric:    '3' if three bytes valid
#               $:              '4' if 4 bytes valid
#                               '0' if no bytes valid
#
sub isquad {

    my $quad = shift @_;

    return 0
        if ( $quad =~ /\.\./ || $quad =~ /\.$/ );

    my ( @bytes ) = split(/\./, $quad);   
	
    my $no_valid = scalar ( grep { $_ =~ /^\d+$/ && $_>= 0 && $_<= 255} @bytes ); 
   
    return 4 if ( @bytes == 4 && $no_valid == 4 );
    return 3 if ( @bytes == 3 && $no_valid == 3 );
    return 2 if ( @bytes == 2 && $no_valid == 2 );
    return 1 if ( @bytes == 1 && $no_valid == 1 );

    return 0;
}

#   Purpose :   Say whether a scalar holds a valid subnet mask.
#   In      :   $:  The scalar to check.
#   Out     :   $:
#   Comments:   This is horribly inelegant, sorry.
#
sub ismask {

    my $mask_to_test = shift @_;

    my ( @bytes ) = split(/\./, $mask_to_test);   

    # Reject if it's not a fully-valid dotted quad
 
    return 
        if ( &isquad($mask_to_test) < 4 );

    # It may be techincally valid, but not for the purposes of this program :)
    # We want to avoid dividing by zero.

    return 0
        if ( $mask_to_test eq "0.0.0.0" );

    # Decide whether it's a valid mask or not.
    # Method: start at $left = 4294967295. Continually reduce it by a power
    # of 2, and check each time if it equals the integer version of the mask
    # we're testing.

    # Work the integer version of the mask out only once, not every time the 
    # loop iterates

    my ( $power, $left, $integer_mask) = ( 1, 4294967295, &quad2int($mask_to_test) );

    while ( $left > 0 ) {

        # Success!

        return 1
            if ( $left == $integer_mask );

        $left -= $power;

        $power *= 2;
    }
    
    return 0;
}

#   Purpose :   Convert a subnet mask to an integer value
#   In      :   $:  The mask.
#   Out     :   $:  The integer version of the mask.
#
sub mask2int {

    my $mask = shift @_;

    return ( 4294967296 - &quad2int($mask) ); 
}

#   Purpose :   Convert an integer representing a subnet size
#               to a subnet mask.
#   In      :   $:  The subnetsize to convert.
#   Out     :   $:  The subnet mask version.
#
sub subnetsize2mask {

    return ( &int2quad( 4294967296 - $_[0] ) );
}

#   Purpose :   Convert a dotted quad into an integer.
#   In      :   $:  the dotted quad.
#   Out     :   $:  numeric:    the integer version.
#   Comments:   Taken from dbase version, in file misc.pl
#
sub quad2int {

    my( $string ) = @_;

    my( @bytes ) = split(/\./, $string);

    return 0
		if ( &isquad($string) < 4 ); 

    if ( @bytes == 4 && scalar(grep { $_>=0 && $_<=255 } @bytes) == 4 ) {
        return unpack( 'N',pack( 'C4', @bytes ) )
    }
    else {
        return(0);
    }
}

#quad int2quad(int)

#   Purpose :   
#   In      :   
#   Out     :   
#   Comments: taken from dbase software version in fiel misc.pl
#
sub int2quad {

    return join(".", unpack('C4',pack('N',$_[0])));

}

#   Purpose :   Check a scalar holds a valid RFC822 email address
#   In      :   $:  the address to check.
#   Out     :   $:  boolean: yes/no
#   Comments:   Taken from dbase software version in field misc.pl
#               Don't ask me how it works please.
#
sub isemail {
    my($str)=@_;
    
    if ( $str=~ /^\s*\<.*\>\s*$/ ) {
        $str=~ s/^\s*\<//;
        $str=~ s/\>\s*$//;
    }
    
    if ( ($str!~ /\@[^\@]*\@/) &&
        ($str=~ /\@$DOMAINNAME_REG\s*$/o) &&
        (($str=~ /^\s*[^\(\)\<\>\,\;\:\\\"\.\[\]\s]+(\.[^\(\)\<\>\,\;\:\\\"\.\[\]\s]+)*\@/) ||
          ($str=~ /^\s*\"[^\"\@\\\r\n]+\"\@/)) ) {
    
        return 1;
    
    }
    else {
        return 0;
    }
}


#   Purpose :   Say whether a scalar holds a valid country code.
#   In      :   $:  the scalar to check
#   Out     :   $:  boolean:    yes/no
#   Comments:   A 'valid' country code is one either in the ISO 3166
#               file, or listed in @SPECIAL_COUNTRIES.
#
sub iscountry {

    # Lee 19980917

    # added quotemeta because we don't want to do a match using a
    # user-submitted string with metacharacters in it

    my $tocheck = uc quotemeta (shift @_);

    # Things like 'eu' should be accepted straight away

    return 1
        if ( scalar (grep {(uc $_) eq $tocheck} @SPECIAL_COUNTRIES) > 0 );

    &fatal("$! while opening $CCODE_FILE.")
        if ( ! ( open CODES, $CCODE_FILE ) );

    # Go through the codes file looking for this one

    while( <CODES> ) {
    
        if ( /^.*\s{5,}$tocheck\s{4,}[A-Z]{3}\s{3,}[0-9]{3}/ ) {
    
            &eprint("$1 while closing $CCODE_FILE, but got data so continuing.")
                if (! close CODES);
    
            return 1;
        }
    }
          
    &fatal("$! while closing $CCODE_FILE.")
        if ( ! close CODES );

    return 0;
}

#   Purpose :   Given a country code, have a guess at which country it's
#               supposed to be indicating.
#   In      :   $:  the code to check.
#   Out     :   $:  a list of suggestions. "" indicates no suggestions.
#
sub country_to_code {

    my $tocheck = uc (shift @_);

    # we dont want to check misspelled country codes, or return
    # hundreds of matches if they use a really short entry
    
    return ("") 
        if ( length $tocheck < 4 );

    &fatal("$! while opening $CCODE_FILE.")
        if ( ! ( open CODES, $CCODE_FILE ) );

    # This variable holds a string detailing the possibles
    
    my $possible_matches = "";

    my ( $possible_code, $possible_country );

    while( <CODES> ) {

        if ( /^(.*)\s{5,}([A-Z]{2})\s{4,}[A-Z]{3}\s{3,}[0-9]{3}/ ) {
    
            $possible_country = &strip_space($1); 
            $possible_code = $2;

            $possible_matches .= "$possible_code: '$possible_country' "
                if ($possible_country =~ /$tocheck/i);
        }
    }
          
    &fatal("$! while closing $CCODE_FILE")
        if ( ! close CODES );

    return $possible_matches;
}

#   Purpose :   Say whether a value is undefined or holds only whitespace.   
#   In      :   $:  the value to test
#   Out     :   $:  booelan:    1= it's undefined or holds only whitespace
#                               0= the opposite
#
sub absent {

    my $value = shift @_;

    return 1 
        if ( ( ! defined $value ) || $value =~ /^\s*$/ );

    return 0;
}

#   Purpose  :  Provide a comparison function which sorts a list of alphanumeric
#               items, pure numeric items rank higher than non-pure-numeric.
#   In       :  VOID
#   Out      :  $:  boolean: see below for description.
#   Comments :  This would be better placed in in Misc.pm, but Perl stipulates 
#               that sort functions must be in the same package as where they
#               are used.
#               Example sort:
#               Before: ( 1, 6, ab, 10, a, 5, g, c )
#               After:  ( 1, 5, 6, 10, a, ab, c, g )
#
sub mixsort {
    
    # Examine both items to check whether they're numeric or non-numeric
        
    my $a_is_numeric_flag = ($a =~ /^\d+$/);
    my $b_is_numeric_flag = ($b =~ /^\d+$/);

    # If both numeric, do <=>

    return $a <=> $b 
        if ( $a_is_numeric_flag && $b_is_numeric_flag );

    # If both non-numeric, do cmp

    return $a cmp $b 
        if ( ! $a_is_numeric_flag && ! $b_is_numeric_flag );

    # In a comparison between numeric and non-numeric, numeric wins

    return -1 
        if ( $a_is_numeric_flag );  

    return 1; 
}

#
#   Purpose :   Strip off preceding and trailing white space from a string
#   In      :   $:  a string to strip.
#   Out     :   $:  the stripped string :-)
#
sub strip_space {

    my $to_strip = shift @_;

    $to_strip =~ s/^\s*//;      #preceeding
    $to_strip =~ s/\s*$//;      #trailing

    return $to_strip;
}

#   Purpose :   Massage $_ to remove the kind of debris at the
#               start of it caused by forwarding characters etc
#               Also weird characters at end.
#   In      :   VOID
#   Out     :   VOID
#
sub preprocess_line {

        # Remove quote chracters at beginning of line. These are sometimes
        # present because requests have been forwarded to us. If not removed
        # they would ruin the field analysis

        s/$QUOTED_LINE_START_REG//;   

        s/$DOSCONVERSION_REG//;
}

#   Purpose :   Add a value to a scalar, with a field separator if necessary.
#   In      :   
#   Out     :   
#   Comments:   The main point of this routine is to avoid continually having
#               if (defined X)      { X = Y }
#               else                { X .= $FORM_FS . Y; }
#
sub add_to_field {
    
    my ( $pointer_to_field, $to_add ) = @_;

    # If it's already defined, insert a field separator and THEN
    # add the new value.

    if ( defined $$pointer_to_field ) {
        $$pointer_to_field = $$pointer_to_field.$FORM_FS.$to_add;
    }
    else {
        $$pointer_to_field = $to_add;
    }
}

#   Purpose :   Return all the values of a particular fieldname for
#               a particular template type
#   In      :   $: template name
#               $: field name
#   Out     :   $: list of values in a string or "" if nothing found
#
sub get_values {

    my ( $template_name_to_find, $field_name_to_find ) = @_;

    my ( $field_name, $hash_ref, $list_of_field_values);

    # For each template we extracted ...

    foreach $hash_ref (@TEMPLATES) {

        # If it's the template specified...

        if ( $hash_ref->{$F{TNAME}} eq $template_name_to_find ) {

            # Look for the field specified
               
            foreach $field_name (keys %$hash_ref) {

                # and add it's value to the list if it's there
                    
                &add_to_field(\$list_of_field_values, $hash_ref->{$field_name})
                    if ( $field_name_to_find eq $field_name );

            }
        }
    }

    return $list_of_field_values;
}

#   Purpose :   Say whether a field name is one which is used for internal
#               tracking, i.e. does not come from the form.
#   In      :   $:  The field name to check.
#   Out     :   $:  boolean:    yes/no
#
sub is_internal_field_name {

    return ( scalar( grep { $F{$_} eq $_[0] } @F_INTERNAL ) > 0 );
}

#   Purpose :   Convert a value in slash notation into an integer.
#   In      :   $:  the value in slash notation.
#   Out     :   $:  numeric:    the integer equivalent or undef if conversion
#                               not possible.
#
sub slash2int {

    my $value = shift @_;
    
    # Separate the number off

    $value =~ /(\/)(\d+)/;

    my ( $slash, $length )  = ( $1, $2 );

    # a slash notation value must begin with a slash
    return undef 
        if ( $slash ne "/" );

    # and it should definitely have a length

    return undef 
        if ( ! length ($length) );

    # range check on length

    return undef
        if ( $length < 1 || $length > 32 );

    return ( 2 ** ( 32 - $length) );

}

#   Purpose :   Given a field name, returns the key in %F which is
#               used to access that field name.
#   In      :   $:  The field name.
#   Out     :   $:  The key, or undef if the field name isn't used.
#   Comments:   It's possible that a field name could be referenced
#               by more than one key in %F (due to an error in the
#               configuration file). This routine returns the
#               first occurence only.     
#
sub key_of_field {

    my $field_name = shift @_;
    
    my $field_key;

    foreach $field_key ( keys %F ) {

        return $field_key 

            if ( $F{$field_key} eq $field_name );
    }
    
    return undef;
}

#   Purpose :   Look for and return a ticket number 
#   In      :   VOID
#   Out     :   A found ticket number, or undef if none found.
#
sub find_tickno_in_message {

    my @ticketnumbers = &get_ticketnumbers($TEMPLATES[0]);

    my $tickno = $ticketnumbers[0];

    if ( defined $tickno ) {
        &dprint("FIND TICKNO: found tickno $tickno in message.");
    }
    else {
        &dprint("FIND TICKNO: couldn't find tickno in message.");
    }

    return $tickno;
}

sub find_regid_in_message {

    my $hash_ref = shift @_;
    
    my $regid;

    if ( $REGID_FLAG ) {
        my @regids = &get_regids($hash_ref);
        $regid = $regids[0];

    }
    else {
        &dprint("FIND REGID: REGID_FLAG set to $REGID_FLAG. Using $DEBUG_REGID.");
        $regid = $DEBUG_REGID;
    }

    if ( defined $regid ) {
        &dprint("FIND REGID: found regid $regid in message.");
    }
    else {
        &dprint("FIND REGID: couldn't find regid in message.");
    }

    return $regid;
}


#   Purpose     :   Decide whether a string looks like a NIC handle
#   In          :   $: the handle to check
#   Out         :   $: bool: 1/undef for true/false
#   Comments    :   Taken from DB software. Needs updating in line
#                   with DB software. This is supposed to be a temporary
#                   fix until either the DB software is rewritten or the
#                   original DB source routine can be isolated.
#   Major changes:  accept lowercase
#                   use &iscountry instead of DB conf file country list
#                   changed %NICPOSTFIX to array and defined it, instead
#                   of using DB config file
#
sub isnichandle {

  my $handle = shift;

  $handle =~ tr/a-z/A-Z/;

  my $MAXLENGTHINITIALS = 4;
  my @NICPOSTFIX = (
    'RIPE', 'RADB', 'INTERNIC', 'NIC', 'ANS', 'MCI', 'CANET', 'APNIC'
  );
    
  return 1 if ($handle =~ /[A-Z]{2}\d{3}JP(-JP)?/);

  if ( $handle =~ /^([A-Z]{2,$MAXLENGTHINITIALS})([1-9]\d{0,5})?(-\S+)?$/ ) {

    my ( $initials, $number, $suffix) = ( $1 , $2, $3 );

    # we're done if there is no suffix
    return 1 unless $suffix;

    # strip leading '-'
    $suffix =~ s/^-//;

    # suffix of local sources (better to use the reverse hash of NICPOSTFIX)
    return 1
        if ( scalar ( grep { $suffix eq $_ } @NICPOSTFIX ) > 0 );

    # country codes
    return 1 if ( &iscountry($suffix) );

    #special suffix
    return 1 if (
		 ($suffix eq 'CC-AU') ||
		 ($suffix eq '1-AU') ||
		 ($suffix eq '2-AU') ||
		 ($suffix eq 'ORG') ||
		 ($suffix eq 'ARIN') ||
		 ($suffix eq 'ORG-ARIN') ||
		 ($suffix eq 'AP')
		 );
  }

  return undef;
}

1;
