# Copyright (c) 1993, 1994, 1995, 1996, 1997    The TERENA Association
# Copyright (c) 1998                            RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Filename          :   HtmlReport.pm
# Purpose           :   Provide routines to produce an HTML report about
#                       a set of templates held in @TEMPLATES.  
# Author            :   Lee Wilmot wrote EmailReport.pm. This is a slightly
#                       modified version of that, by MGTM
# Date              :   19990108
# Language Version  :   Perl 5.003_07
# OSs Tested        :   BSD
# Input Files       :   $FROMCUST_MAIL_FILE.'affix'  :  the original request. 'Affix'
#                       is a unique affix to prevent 
#                       interference from other invocations
#                       of the robot.
# Output Files      :   $TOCUST_MAIL_FILE.'affix': temp file for storing mail to
#                       customer before it is sent. 'Affix'
#                       is a unique affix to prevent interference
#                       from other invocations of the robot.
#                       $TOCUST_LOG_DIR/`date`: message to customer logged here
#                       after sending.
# External Programs :   NONE

# Problems          :   
# To Do             :
# Comments          :
# Description       :

############################ PACKAGE INTERFACE ###############################
#
# Please see the package concerned for descriptions of imported symbols.

package HtmlReport;

use strict;

BEGIN {

    use vars qw( @EXPORT_OK @ISA );

    use Exporter ();
    
    @ISA = qw(Exporter);
    
    @EXPORT_OK = qw(
        &html_report
    );
}

use Extract qw(
           @TEMPLATES
);

# MGTM added HTML headers - keep plain text ones as well, as they are matched and removed from the supplied text
use RobotConfig qw(     
            :REGULAR_EXPRESSIONS :DEBUG_FLAGS :RTT_VARS

            $VERSION_NUMBER

            %F $FORM_FS $PLAN_FS $PROB_FS $VISIBLE_FS 
            $MAX_FIELD_KEY_LENGTH

            $ROBOT_TITLE
            $ROBOT_ADDRESS_STANDALONE $ROBOT_ADDRESS_REAL

            $NOT_A_REQUEST_ERROR_TXT

            $FROMCUST_MAIL_FILE $TOCUST_MAIL_FILE $TOCUST_LOG_DIR

            $WAIT_QUEUE_NAME

            %REPORT_TYPE %PROBLEM_TYPE

            $EMAILHEADER_CONTACT_FIELDS_TXT

            $SECTION_ONE_HEADER $SECTION_TWO_HEADER $SECTION_THREE_HEADER
            $SECTION_ONE_HEADER_HTML $SECTION_TWO_HEADER_HTML $SECTION_THREE_HEADER_HTML
              
            $MAX_NO_ERRORS_IN_TEMPLATE_THRESHOLD 
            $MAX_NO_WARNS_IN_TEMPLATE_THRESHOLD 
            $WARNING_REJECT_THRESHOLD

);

use ReportAux qw (
            &get_problem_texts &get_web_texts
            &count_errors_and_warns &substitute_variables_in_text
);


use Misc qw(
            &fatal &centre_pad &ljustify &send
            &add_to_log &make_filename &absent
            &dprint &eprint &exit_gracefully
            &find_tickno_in_message &find_regid_in_message
            &get_values &is_internal_field_name &get_date
);

use ExternalMisc qw(
            &get_email_format_date &get_message_id &remove_robot_addresses 
            &removeRIPEAddresses &isContact &getRegInfo
);

use TemplateAux qw(
            &template_name &get_template_key &template_type
);

use RTT qw(
            &make_new_ticket &add_message_to_ticket &get_list_of_duplicate_tickets
            &change_ticket &number_of_messages_on_ticket
            &add_log_message_to_ticket &find_regid_in_ticket
);

#################### PACKAGE BODY ###############################

# Purpose       :       Main routine for driving report generation.
# In            :       VOID
# Out           :       VOID
# Description   :
#
# Normal Action
#
#   - Read $PROBLEMTEXT_FILE && $EMAILTEXT_FILE
#   - Section 0: Compose and output mail header and intro text.#   - Section 1: Problem summary
#   - Section 2: Template-by-template analysis with error codes attached. Possibly
#     also verbose error codes attached, if requested by customer.
#   - Section 3: MGTM original text so don't bother as we show that
#     in web141.pl.cgi
#
sub html_report {

    &dprint("****** HTML REPORT ********");

    # Make pointer to meta template purely for easier reference.

    my $meta = $TEMPLATES[0];

    # On the report, individual track is kept of the error, warn or
    # inform number. These three variables perform this function.
    # We start at problem number 1 :)

    my ( $warn_counter, $error_counter, $inform_counter ) = ( 1, 1, 1 );

    # Generate a filename unique to this invocation of the robot. This
    # This file is where the original request is found.

    my $fromcust_file = &make_filename($FROMCUST_MAIL_FILE); 

    # Crash if it's dissappeared since we wrote it.

    fatal("File $fromcust_file, containing mail from customer, has disappeared!")
        if ( ! -f $fromcust_file );

    # Check we found some templates. If not, it probably wasn't a ripe-141 form
    # Lee 19980708: In that case, we need to pass this back to Mal

    &exit_gracefully( -1, $NOT_A_REQUEST_ERROR_TXT)
        if ( scalar (@TEMPLATES) < 2 );

    # Read the texts describing the various problems into some easy-access arrays.
    # The first argument indicates that we want the text version.

    my ( %prob_types, %short_prob_texts, %long_prob_texts );
    # MGTM 0 first arg leaves as html with no fixing for plain text 
    &get_problem_texts(
                0,
                \%prob_types, \%short_prob_texts, \%long_prob_texts);

    # Read the texts which link together the report.
    # The first argument indicates that we want the text version.

    my %email_texts;	# MGTM no, I left the name the same
    &get_web_texts( \%email_texts );

    #################### SECTION ZERO: EMAIL HEADER AND INTRODUCTION ###################

    ### Title ###

    my $title  = $meta->{$F{EMAILSUBJECT}} || "";

    # do a count of the problems. This adds several new fields to
    # the templates

    my ( $total_errors, $total_warnings, $total_informs ) = ( 0, 0, 0 );

    &count_errors_and_warns(\@TEMPLATES, \%prob_types, \$total_errors, \$total_warnings, \$total_informs );

    # A flag to say later whether there are any problems

    my $total_problems = $total_errors + $total_warnings + $total_informs;

    # make some count variables based on the total just calculated

    # numbers of the different problem types in the meta template
    my $meta_errors = $meta->{$F{ERRORCOUNT}};
    my $meta_warnings = $meta->{$F{WARNCOUNT}};
    my $meta_informs = $meta->{$F{INFORMCOUNT}};

    &dprint("COUNTED errors: $total_errors warns: $total_warnings informs: $total_informs");

    # Create an initial status message for the email
    # dependent on the ticket number status.

    my ( $intromess, $countmess, $actionmess );

    $intromess = $email_texts{REPORT_INTRO};
    
    $countmess = $email_texts{REPORT_PROBLEM_COUNT};    #substitute counts into message
    $countmess =~ s/\?error/$total_errors/g;
    $countmess =~ s/\?warn/$total_warnings/g;
    $countmess =~ s/\?inform/$total_informs/g;

    # Try to get a tickno and a regid from message

# MGTM don't do regid or ticnum stuff

    
    if ( $total_errors == 0 && $total_warnings <= $WARNING_REJECT_THRESHOLD ) {
        # Pick a 'no problems' message, depending on the mode.
        if ( $MODE_FLAG == 2 ) {
            $actionmess = $email_texts{REPORT_ACTION_NOPROBLEMS_STANDALONE};
            
        }
        else {
            
            $actionmess =  $email_texts{REPORT_ACTION_NOPROBLEMS};
            
        }
    }
    else {

        # Check the history to see if there are too many messages on the ticket. At this
        # point there are errors in this request too: if this request had been OK
        # we'd have just dealt with it as normal.

        $actionmess = $email_texts{REPORT_ACTION_PROBLEMS};
    }

    # Replies should be Re:'d

    #$subject = "Re: $subject";

    # Generate a filename unique to this invocation of the robot for
    # storing the mail we're about to write to the customer.
   
    my $tocust_file = &make_filename($TOCUST_MAIL_FILE);

    &dprint("WRITING $tocust_file");

    open ( TOCUST, ">$tocust_file" ) or
        &fatal("$! while opening $tocust_file to write report to customer.");

    # Use the $tocust_file for as the default output stream
    
    select TOCUST;

    # Print out the header and initial messages

	# MGTM forget dates

    print "\n";

	# We have a special intro message for the standalone
	# version of the robot.
	&print_standalone_disclaimer
		if ( $MODE_FLAG == 2 );

    print $email_texts{REPORT_HELLO};
    print $intromess;
    print $countmess;
    print $actionmess;

    # Added 19980130 lee
    # Only show this blurb if we're not in standalone mode.
    # In standalone mode we already have a large blurb at
    # the top.

    print $email_texts{REPORT_ROBOT_BLURB}
		if ( $MODE_FLAG != 2 ); 

    # Decide whether they wan't a verbose error report or not.

    #my $longack_flag = ( $subject =~ /$LONGACK_REG/) || 0;

# MGTM
my $longack_flag = 1;

    #$longack_flag = 1
    #    if ( $MODE_FLAG == 0 || $MODE_FLAG == 1 );

    # Used for storing a pointer to the template currently being processed.

    my $hash_ref;


    #################### SECTION ONE: PROBLEM SUMMARY ############################

    # We only print anything in this section if there are any problems
    # to report ( since its a PROBLEM summary section).

    if ( $total_problems > 0 ) {

        # Print the section header, centred in the page surrounded by '#' symbols.
    
        print "$SECTION_ONE_HEADER_HTML";
    
        # Remind them that they can use longack to get longer report,
        # if they're not already doing so.
	# MGTM don't
    
        # Print the section intro text.
    
        print $email_texts{REPORT_SECTION_ONE_INTRO};
    
        foreach $hash_ref (@TEMPLATES) {
    
            # Print a message at the start of each template
            # if there are errors/warnings for that template

            if ( $hash_ref->{$F{ERRORCOUNT}} || $hash_ref->{$F{WARNCOUNT}} || $hash_ref->{$F{INFORMCOUNT}} ) {

                # All of this just substitutes '?name' in the text
                # with the current template name.

                my $template_name = $hash_ref->{$F{TNAME}};
                my $header = $email_texts{REPORT_EXTRACTED_PER_TEMPLATE_INTRO};
                $header =~ s/\?name/$template_name/g;
                print "$header";

            }

            # Print any ERRORS
    
            &print_template_problems(
                            $hash_ref, 
                            0,
                            $PROBLEM_TYPE{ERROR},
                            \$error_counter,
                            \%email_texts,
                            \%prob_types,
                            \%short_prob_texts,
                            \%long_prob_texts
            )
                if ( $hash_ref->{$F{ERRORCOUNT}} );
    
            # Print any WARNINGS
    
            &print_template_problems(   
                            $hash_ref,
                            0,
                            $PROBLEM_TYPE{WARNING},
                            \$warn_counter,
                            \%email_texts,
                            \%prob_types,
                            \%short_prob_texts,
                            \%long_prob_texts
            )
                if ( $hash_ref->{$F{WARNCOUNT}} );
    
            # Print any INFORMS
    
            &print_template_problems(   
                            $hash_ref,
                            0,
                            $PROBLEM_TYPE{INFORM},
                            \$inform_counter,
                            \%email_texts,
                            \%prob_types,
                            \%short_prob_texts,
                            \%long_prob_texts
            )
                if ( $hash_ref->{$F{INFORMCOUNT}} );
        }
    }

    #################### SECTION TWO: EXTRACTED TEMPLATES + PROBS  #########################

    # If there were any problems...

    if ( $total_problems > 0 ) {

        # Reset the problem counters: we're going over the problems again below
        
        ( $inform_counter, $warn_counter, $error_counter ) = ( 1, 1, 1 );
        
        # Print the section header, centered in the page and surrounded by '#'s
        
        print "<P>$SECTION_TWO_HEADER_HTML<P>";
        
        # And the intro...
        
        print $email_texts{REPORT_SECTION_TWO_INTRO};
        
        foreach $hash_ref (@TEMPLATES) {
            
            # These four lines just print out the text for above each template,
            # but with '?name' replaced by the name of the current template
            # we're doing
            next 
                if ( ($hash_ref == $meta) && !$meta_errors && !$meta_warnings && !$meta_informs);
            
            my $template_name = $hash_ref->{$F{TNAME}};
            my $message = $email_texts{REPORT_EXTRACTED_PER_TEMPLATE_INTRO};
            $message =~ s/\?name/$template_name/g;
            print "<H3>$message</H3>";
            
            # Display the fields in the template we managed to extract.
            
            &print_template($hash_ref)
                if ( $hash_ref != $meta );
            
            # Then the problems which were found for that template.
            # The second flag indicates that we want verbose output
            # if it was requested by the requester.
            
            # ERRORS
            
            &print_template_problems(   
                        $hash_ref,
                        $longack_flag,
                        $PROBLEM_TYPE{ERROR},
                        \$error_counter,
                        \%email_texts,
                        \%prob_types,
                        \%short_prob_texts,
                        \%long_prob_texts
            )
                if ( $hash_ref->{$F{ERRORCOUNT}} );
            
            # WARNINGS
            
            &print_template_problems(   $hash_ref,
                        $longack_flag,
                        $PROBLEM_TYPE{WARNING},
                        \$warn_counter,
                        \%email_texts,
                        \%prob_types,
                        \%short_prob_texts,
                        \%long_prob_texts
            )
                if ( $hash_ref->{$F{WARNCOUNT}} );
            
            # INFORMS
            
            &print_template_problems(   $hash_ref,
                        $longack_flag,
                        $PROBLEM_TYPE{INFORM},
                        \$inform_counter,
                        \%email_texts,
                        \%prob_types,
                        \%short_prob_texts,
                        \%long_prob_texts
            )
                if ( $hash_ref->{$F{INFORMCOUNT}} );
        }
    }

# MGTM stop here as do section 3 = original message in caller for web report

close TOCUST;

select STDOUT;

# show what we got

system ( "cat $tocust_file" );

return;

    #################### SECTION THREE: ORIGINAL MESSAGE  ############################

    # Print the section header
    
    print "<P>$SECTION_THREE_HEADER_HTML<P>";

    &fatal("$! while opening $fromcust_file, containing original message from customer, for reading.")
        if ( ! open TEMP, $fromcust_file);

    # We don't include the report sections from previous robot
    # reports in our 'original message'. Otherwise, whenever they
    # did a 'reply' and ammended their original submission the
    # robot reports would stack up on top of each other

    my $printing_flag = 1;

    while (<TEMP>) {

        if ( /$SECTION_ONE_HEADER/ ) {
            $printing_flag = 0;
            print "<---- CUT PREVIOUS PROBLEM SUMMARY ON ", &get_date, " ---->\n\n";
        }
    
        if ( /$SECTION_TWO_HEADER/ ) {
            $printing_flag = 0;
            print "<---- CUT PREVIOUS TEMPLATE ANALYSIS ON ", &get_date, " ---->\n\n";
        }
    
        print 
            if $printing_flag;
    
        $printing_flag = 1
            if ( /$SECTION_THREE_HEADER/ );
    }
 
   &eprint("$! while closing $fromcust_file. Continuing.")   
        if ( ! close TEMP );
 
    ############# PRINT END-OF-MESSAGE TEXT ###############

    # Print the 'end of message' line, centered in the page and surrounded
    # by '#'s

    print "\n".&centre_pad($email_texts{REPORT_MESSAGE_FOOTER})."\n";

    select STDOUT;

    &fatal("$! while closing pipe from $tocust_file.")
       if (! close TOCUST);

    ############# SEND MESSAGE. LOG IT, ADD TO TICKET ########

    &send($tocust_file);

    &add_to_log( $TOCUST_LOG_DIR, $tocust_file );

    # We don't add it to the ticket yet, because we don't know yet
    # whether the status should be OPEN-NCC ( request OK ) or
    # OPEN-REG (problems).

    ############ CHECK CONTACT PERSONS OK #####################

    # Lee 19980714
    # Check From: and Reply-to: fields in email against contact fields
    # in registry file.
    
   # &dprint("CONTACT PERSONS checking reg file $regid.");
    
   # my ( $dummy, @regfile_contacts ) = &getRegInfo( $regid );

   # &fatal("Couldn't get registry contact information using '&getRegInfo($regid)'")
   #     if ( ! $dummy );
   # 
    # If neither the from field or the reply field matches one of the
    # contacts from the reg file, add a log message to the ticket.
    
   # &add_log_message_to_ticket( $tickno, "Neither -$F{EMAILFROM}- field nor -$F{EMAILREPLTO}- field seems to match a registered contact for $regid.")
   #     if ( ! &isContact( $from, @regfile_contacts ) && ! &isContact( $replto, @regfile_contacts) );
    
    ############ NOTIFY HUMANS IF REQUEST OK ########################

    # A request is OK if there are NO errors, and an acceptable number of
    # warns.

    if ( $total_errors == 0 && $total_warnings <= $WARNING_REJECT_THRESHOLD ) {

# MGTM ticket stuff removed 

        &dprint("******************* REQUEST SUCCESSFUL *********************");

        my $request_type = &get_values( &template_name('T_OVERREQ'), $F{PIREQ} );

        my $netnames = &get_values( &template_name('T_NETWORK'), $F{NETNAME} );

    }

}

# Purpose       :       Print out all of the problems of a type associated with a template.
# In            :       $%: Pointer to the hash containing the template
#                       $:  boolean: include verbose output ?  
#                       $:  numeric: which type of problem to do
#                       $$: numeric: pointer to the counter for this problem type 
#                       $%: pointer to hash holding the various email texts  
#                       $%: pointer to hash holding the type for each problem code
#                       $%: pointer to hash holding the short text for each problem code
#                       $%: pointer to hash holding the long text for each problem code
# Out           :       VOID
# Comments      :       Problems look like:
#                           CODE1 $PROB_FS ARG1.1 $PROB_FS ARG1.2 $FORM_FS.
#                       The number of args is variable depending on the code.
#                       Individual problems are separated by:
#                           $FORM_FS                        
#                       Problem texts have variables of the form '?1', '?2' etc
#                       We substitute the ARG1.1 etc into the problem text.
#
sub print_template_problems {

    my $hash_ref = shift @_;            # pointer to the template
    my $verbose_flag = shift @_;        # 0 = short description only, 1 = also do verbose
    my $type_to_do = shift @_;          # do errors, warns or informs ?
    my $count_ref = shift @_;           # pointer to a counter for this problem type
                                        # this is a *pointer*because we keep track of
                                        # the numbers of each type of problem ACCROSS
                                        # TEMPLATES.
    my $email_texts_ref = shift @_;     # pointer to the various email texts
    my $problem_types_ref = shift @_;   # holds the type for each problem code
    my $short_prob_texts_ref = shift @_;# holds the short text for each problem code
    my $long_prob_texts_ref = shift @_; # holds the long text for each problem code


    # Stop if there *arent* any problems noted for this template

    return
        if ( ! exists $hash_ref->{$F{PROBLEMS}} );

    # Set the text to print between each problem description

    my $between_problems_message_template;

    if ( $type_to_do == $PROBLEM_TYPE{ERROR} ) { 
        $between_problems_message_template = $email_texts_ref->{REPORT_BETWEEN_ERRORS};  
    }
    elsif ( $type_to_do == $PROBLEM_TYPE{WARNING} ) { 
        $between_problems_message_template = $email_texts_ref->{REPORT_BETWEEN_WARNINGS};
    }
    else { 
        $between_problems_message_template = $email_texts_ref->{REPORT_BETWEEN_INFORMS}; 
    }

    # Separate each problem description into an array.

    my @problems = @{ $hash_ref->{$F{PROBLEMS}} };

    # Then go through each one....

    my ($between_problems_message, $variable_index);
    my ($current_problem, $current_problem_code);
    my (@problem_parts, $string_to_print, $value_to_substitute);
    
    my $this_template_problem_counter = 1;    # Track no of problems found in *this* template

    while ( defined ( $current_problem = shift @problems) ) {

        # Stop if we've reached the threshold for how many errors/warnings to report
        # per template

        last 
            if ( ( $this_template_problem_counter > $MAX_NO_ERRORS_IN_TEMPLATE_THRESHOLD ) &&
                   ($type_to_do == $PROBLEM_TYPE{ERROR} ) );
        last 
            if ( ( $this_template_problem_counter > $MAX_NO_WARNS_IN_TEMPLATE_THRESHOLD) &&
                   ($type_to_do == $PROBLEM_TYPE{WARNING} ) );

        # Split the problem description into its code and arguments    

        @problem_parts = split /$PROB_FS/, $current_problem;

        $current_problem_code = shift @problem_parts;
    
        next 
            if ( $problem_types_ref->{$current_problem_code} =~ /ERROR/i && 
                 $type_to_do != $PROBLEM_TYPE{ERROR} );
        next 
            if ( $problem_types_ref->{$current_problem_code} =~ /WARNING/i && 
                 $type_to_do != $PROBLEM_TYPE{WARNING} );
        next 
            if ( $problem_types_ref->{$current_problem_code} =~ /INFORM/i && 
                 $type_to_do != $PROBLEM_TYPE{INFORM} );

        # Print the between-problem text
        # The substitution is to allow the current problem number
        # to be included between-problems.

        $between_problems_message = $between_problems_message_template;
        $between_problems_message =~ s/\?1/$$count_ref/g;

        print $between_problems_message;
    
        ######  Do the SHORT PROBLEM TEXT #####

        # Check for a problem code being generated by the program but
        # not being defined in the problem texts file.

        &fatal("Program error, undefined problem code $current_problem_code encountered.")
            if ( ! defined $short_prob_texts_ref->{$current_problem_code} );

        # Get the short text for this problem code.

        $string_to_print = $short_prob_texts_ref->{$current_problem_code};

        # Substitute each argument in the problem description into the problem
        # description were going to print.       

        $variable_index = 1;

        while ( defined ( $value_to_substitute = shift @problem_parts ) ) {

            # Some error reports include an argument showing the plan line that was
            # extracted. We want to avoid showing the horrible-looking field
            # separator we use internally. Replace it with the nice one :-)
        
            $value_to_substitute =~ s/$PLAN_FS/$VISIBLE_FS/g; 
        
            # Substitute the argument into the problem description

            $string_to_print =~ s/\?$variable_index/$value_to_substitute/g;

            # Do the next variable number next time.
       
            $variable_index++;
        }

        # Print the short problem text complete with substituted variables

	# MGTM no justification
	# <P> to seperate from long text
        print "$string_to_print<P>\n";
    

        ######  Do the VERBOSE PROBLEM TEXT #####

        # Split the problem description into its code and arguments    

        @problem_parts = split /$PROB_FS/, $current_problem;

        $current_problem_code = shift @problem_parts;

        if ( $verbose_flag ) {

            &fatal("Program error, undefined problem code $current_problem_code encountered.")
                if ( ! defined $long_prob_texts_ref->{$current_problem_code} );

            print "\n";

            # Reset it to 1: we're going through them all again

            $variable_index = 1;

            # Get the long text for this problem code.

            $string_to_print = $long_prob_texts_ref->{$current_problem_code};

            # Substitute each argument in the problem description into the problem
            # description were going to print.       
       
            while ( defined ( $value_to_substitute = shift @problem_parts ) ) {
    
                # Some error reports include an argument showing the plan line that was
                # extracted. We want to avoid showing the horrible-looking field
                # separator we use internally. Replace it with the nice one :-)
        
                $value_to_substitute =~ s/$PLAN_FS/$VISIBLE_FS/g; 

                # Substitute the argument into the problem description

                $string_to_print =~ s/\?$variable_index/$value_to_substitute/g;

                # Do the next variable number next time.

                $variable_index++;

            }

            # Print the short problem text complete with substituted variables
    
	# MGTM no justification
            print "$string_to_print\n";
        }

        $this_template_problem_counter++;

        $$count_ref++;
    }
}

# Purpose       :       Print out all the fields in a template in a nice way :-)
# In            :       $%: Pointer to the hash containing the template
# Out           :       VOID
# Comments      :       
#
sub print_template {

    my $hash_ref = shift @_;

    my ( $field_name, $spaces, $field_value, @field_names );

    # Get the key associated with this type of template

    my $template_key = &get_template_key($hash_ref->{$F{TNAME}});

    # Get the field names in the template
    # Sort them dependent on the type of template it is    

    # Lee 19980204: changed to process bad templates ($template_key == undef) 
    # Previously they were causing a crash.

    if ( ! ( $template_key && &template_type($template_key) eq 'FIELD') ) {

        # None field-templates have an alphanumeric sorted output

        @field_names = sort mixsort keys %$hash_ref;
    }
    else {
    
        # Field templates have their fields printed in the order
        # that they appeared on the form

        @field_names = @{$hash_ref->{$F{FIELD_ORDER}}};

    }
    
    # Output each field in the order we decided above

    foreach $field_name ( @field_names ) {

        ### don't show fields which are used for internal tracking  ###
    
        next
            if ( ! $SHOW_INTERNAL_FIELDS_FLAG && &is_internal_field_name($field_name) );

        # Field name

        print $field_name, '#';

        # Spaces

        my $spaces = $MAX_FIELD_KEY_LENGTH - (length $field_name);
        print " " x $spaces                     
            if ( $spaces >= 0 );

        # Field value or <Undefined>

        $field_value = $hash_ref->{$field_name};
    
        if ( defined $field_value )   {

            # Replace the horrible-looking internal field separators with
            # the nice one :-)

            $field_value =~ s/$FORM_FS/$VISIBLE_FS/g;
            $field_value =~ s/$PLAN_FS/$VISIBLE_FS/g;

            print "\'", $field_value, "\'\n";
        }
        else  {
            print "<Undefined>\n";
        }
    # MGTM seperate lines in html output
    print "<BR>";
    }
}

#   Purpose  :  Provide a comparison function which sorts a list of alphanumeric
#               items, pure numeric items rank higher than non-pure-numeric.
#   In       :  VOID
#   Out      :  $:  boolean: see below for description.
#   Comments :  This would be better placed in in Misc.pm, but Perl stipulates that
#               sort functions must be in the same package as where they are used.
#               Example sort:
#               Before: ( 1, 6, ab, 10, a, 5, g, c )
#               After:  ( 1, 5, 6, 10, a, ab, c, g )
#
sub mixsort {
    
    # Examine both items to check whether they're numeric or non-numeric
        
    my $a_is_numeric_flag = ($a =~ /^\d+$/);
    my $b_is_numeric_flag = ($b =~ /^\d+$/);

    # If both numeric, do <=>

    return $a <=> $b 
        if ( $a_is_numeric_flag && $b_is_numeric_flag );

    # If both non-numeric, do cmp

    return $a cmp $b 
        if ( ! $a_is_numeric_flag && ! $b_is_numeric_flag );

    # In a comparison between numeric and non-numeric, numeric wins

    return -1 
        if ( $a_is_numeric_flag );  

    return 1; 
}

# Purpose       :       Print a piece of text specific to the
#						standalone version of the robot.
# IN            :       VOID
# Out           :       VOID
#
sub print_standalone_disclaimer {

	print '-' x 65, "\n";

	print <<EOF;
This a BETA TEST VERSION ($VERSION_NUMBER) of an IP Request checking
robot for automatic processing of ripe-141 forms.  

While it is intended that at a later stage this interface (and a
forthcoming web interface) will allow you to actually submit requests,
this beta test version does *not* do this yet. You will just get the
diagnostics back. You still have to submit *all* requests via the
standard procedure to

                <hostmaster\@ripe.net>

However you can use the beta test version of the interface to check
your requests for problems and to ensure that they are more easily
readable once submitted, and that you therefore get a faster response
from the humans :-)

Please do not hesitate to send any comments or suggestions you might
have about the operation or the output of the robot to

                <auto-hm-comments\@ripe.net>.

*** If you do not receive a reply from this beta version of  *** 
*** the robot you can safely assume there was a serious      ***
*** problem. The details will be logged and you'll receive a ***
*** manual response.                                         ***
EOF

	print '-' x 65, "\n\n";
}

1;
