# Copyright 2010-2014 Meik Michalke <meik.michalke@hhu.de>
#
# This file is part of the R package rkwarddev.
#
# rkwarddev is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# rkwarddev is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with rkwarddev.  If not, see <http://www.gnu.org/licenses/>.


#' Create XML node "option" for RKWard plugins
#' 
#' @note You will rarely need this function, as options can be defined directly as a list
#' in applicable functions like \code{\link[rkwarddev:rk.XML.radio]{rk.XML.radio}}. The main
#' purpose for having this function is to set an ID for a particular option, e.g. to be able
#' to hide it by logic rules.
#' 
#' To address such an option in your logic section, the \code{id}
#' you need is a combination of \code{<parent id>.<option id>}. That is,
#' you must always prefix it with the parent's \code{id}. If you use the object
#' an object generated by this function inside a parent node, both IDs will
#' automatically be stored internally, so that the correct prefix will be added
#' if needed whenever you apply logic rules to the option object.
#'
#' @param label Character string, a text label for this plugin element.
#' @param val Character string, defines the value to submit if the option is checked.
#' @param chk Logical, should be set \code{TRUE} in the one option which is checked by default.
#' @param id.name Character string, a unique ID for this plugin element.
#'    If \code{"auto"} and a label was provided, an ID will be generated automatically from the label.
#' @param i18n Either a character string or a named list with the optional elements \code{context}
#'    or \code{comment}, to give some \code{i18n_context} information for this node. If set to \code{FALSE},
#'    the attribute \code{label} will be renamed into \code{noi18n_label}.
#' @return An object of class \code{XiMpLe.node}.
#' @export
#' @seealso \href{help:/rkwardplugins/index.html}{Introduction to Writing Plugins for RKWard}
#' @examples
#' test.radio <- rk.XML.radio("Chose one",
#'   options=list(
#'     "First Option"=c(val="val1", chk=TRUE),
#'     test.radio.opt2 <- rk.XML.option("Second Option", val="val2", id.name="auto"),
#'     "third Option"=c(val="val3"))
#' )
#' cat(pasteXML(test.radio))

rk.XML.option <- function(label, val=NULL, chk=FALSE, id.name=NULL, i18n=NULL){
  attr.list <- list(label=label, value=val)
  if(isTRUE(chk)){
    attr.list[["checked"]] <- "true"
  } else {}

  if(identical(id.name, "auto")){
    attr.list[["id"]] <- auto.ids(label, prefix=ID.prefix("optn"))
  } else {
    attr.list[["id"]] <- id.name
  }

  # check for additional i18n info; if FALSE, "label" will be renamed to "noi18n_label"
  attr.list <- check.i18n(i18n=i18n, attrs=attr.list)

  node <- check.i18n(
    i18n=i18n,
    node=XMLNode("option", attrs=attr.list),
    comment=TRUE
  )

  return(node)
}
