# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#
#
#
# transform_config_hw.py -- Generate svn_private_config.h
# from svn_private_config.hw while editing SVN_BUILD_HOST


import os
import re
import sys
import platform


def usage_and_exit(msg):
  if msg:
    sys.stderr.write('%s\n\n' % msg)
  sys.stderr.write(
    'USAGE: %s ARCHITECTURE TEMPLATE_FILE [OUTPUT_FILE]\n'
    '  stdout will be used if OUTPUT_FILE is not provided.\n'
    % os.path.basename(sys.argv[0]))
  sys.stderr.flush()
  sys.exit(1)


_wincpu_map = {
    'x86': 'x86',
    'x64': 'x86_64',
    'amd64': 'x86_64',
    'x86_64': 'x86_64',
    'ia64': 'ia64',
    'powerpc': 'powerpc',
    'alpha': 'alpha',
    }

_arch_map = _wincpu_map.copy()
_arch_map.update({
    'win32': 'x86',
    })

_interesting_rx = re.compile(
    r'^\s*#\s*define\s+SVN_BUILD_HOST\s+(?P<host>"[^"]+")\s*$')

def process_header(input, output, architecture):
    uname = platform.uname()
    winver = uname[3]
    wincpu = _wincpu_map.get(uname[4].lower(), 'unknown')
    arch = _arch_map.get(architecture.lower(), 'unknown')

    if wincpu == arch:
        host = '"%s-microsoft-windows%s"' % (arch, winver)
    else:
        host = '"%s/%s-microsoft-windows%s"' % (arch, wincpu, winver)

    for line in input.split('\n'):
        match = _interesting_rx.match(line)
        if match is not None:
            line = line.replace(match.group('host'), host)
        output.write(line + '\n')


def main(input_filepath, output, architecture):
    filename = os.path.basename(input_filepath)
    input = open(input_filepath, 'r').read()

    output.write(
        '/* This file is automatically generated from %s.\n'
        ' * Do not edit this file -- edit the source and rerun gen-make.py */'
        '\n\n'
        % (filename,))

    process_header(input, output, architecture)


if __name__ == '__main__':
    if os.name != 'nt':
        usage_and_exit('This script should only be run on Windows')

    if len(sys.argv) < 3 or len(sys.argv) > 4:
        usage_and_exit('Incorrect number of arguments')

    architecture = sys.argv[1]
    input_filepath = sys.argv[2]

    if len(sys.argv) > 3:
        output_file = open(sys.argv[3], 'w')
    else:
        output_file = sys.stdout

    main(input_filepath, output_file, architecture)
