=head1 NAME

Sys::Statistics::Linux::DiskUsage - Collect linux disk usage.

=head1 SYNOPSIS

    use Sys::Statistics::Linux::DiskUsage;

    my $lxs  = new Sys::Statistics::Linux::DiskUsage;
    my $stat = $lxs->get;

=head1 DESCRIPTION

Sys::Statistics::Linux::DiskUsage gathers the disk usage with the command C<df>.

For more information read the documentation of the front-end module L<Sys::Statistics::Linux>.

=head1 DISK USAGE INFORMATIONS

Generated by F</bin/df -kP>.

    total       -  The total size of the disk.
    usage       -  The used disk space in kilobytes.
    free        -  The free disk space in kilobytes.
    usageper    -  The used disk space in percent.
    mountpoint  -  The moint point of the disk.

=head2 GLOBAL VARS

If you want to change the path or arguments for C<df> you can use the following
variables...

    $Sys::Statistics::Linux::DiskUsage::DF_PATH = '/bin';
    $Sys::Statistics::Linux::DiskUsage::DF_CMD  = 'df -akP';

Example:

    use Sys::Statistics::Linux;
    use Sys::Statistics::Linux::DiskUsage;
    $Sys::Statistics::Linux::DiskUsage::DF_CMD = 'df -akP';

    my $sys  = Sys::Statistics::Linux->new(diskusage => 1);
    my $disk = $sys->get;

=head1 METHODS

=head2 new()

Call C<new()> to create a new object.

    my $lxs = Sys::Statistics::Linux::DiskUsage->new;

It's possible to set the path to df.

     Sys::Statistics::Linux::DiskUsage->new(
        cmd => {
            # This is the default
            path => '/bin',
            df   => 'df -kP 2>/dev/null',
        }
    );

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

    my $stat = $lxs->get;

=head1 EXPORTS

No exports.

=head1 SEE ALSO

B<df(1)>

=head1 REPORTING BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 COPYRIGHT

Copyright (c) 2006, 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut

package Sys::Statistics::Linux::DiskUsage;

use strict;
use warnings;
use Carp qw(croak);

our $VERSION = '0.14';
our $DF_PATH = undef;
our $DF_CMD  = undef;

sub new {
    my $class = shift;
    my $opts  = ref($_[0]) ? shift : {@_};

    my %self = (
        cmd => {
            path => '/bin',
            df   => 'df -kP 2>/dev/null',
        }
    );

    foreach my $p (keys %{ $opts->{cmd} }) {
        $self{cmd}{$p} = $opts->{cmd}->{$p};
    }

    return bless \%self, $class;
}

sub get {
    my $self   = shift;
    my $class  = ref $self;
    my $cmd    = $self->{cmd};
    my $df_cmd = $DF_CMD || $cmd->{df};
    my (%disk_usage);

    local $ENV{PATH} = $DF_PATH || $cmd->{path};
    open my $fh, "$df_cmd|" or croak "$class: unable to execute '$df_cmd' ($!)";

    # filter the header
    {my $null = <$fh>;}

    while (my $line = <$fh>) {
        next unless $line =~ /^(.+?)\s+(.+)$/;

        @{$disk_usage{$1}}{qw(
            total
            usage
            free
            usageper
            mountpoint
        )} = (split /\s+/, $2)[0..4];

        $disk_usage{$1}{usageper} =~ s/%//;
    }

    close($fh);
    return \%disk_usage;
}

1;
