/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */


#include "playlist_broadcaster.h"
#include <unistd.h>

QTFileBroadcaster::QTFileBroadcaster() 
{
	fRTPFilePtr = NULL;
	fMovieSDPParser = NULL;
	fBasePort = 0;
	fDebug = false;
	fDeepDebug = false;

// transmit time trackers
	fLastTransmitTime = 0.0;
	
	fStreamStartTime  = 0;
	fMovieStartTime  = 0;
	fMovieEndTime = 0;
	fMovieIntervalTimeMicro = 0;
	
	fMovieStart = false;
	fSendTimeOffset = 0.0;
	fMovieDuration = 0.0;
	fMovieTracks = 0;
	fMappedMovieTracks = 0;
	fNumMoviesPlayed = 0;
	fPlay = true;
	fSend = true;
	
}

QTFileBroadcaster::~QTFileBroadcaster()
{ 
	LogFileClose(); 
	if (fRTPFilePtr != NULL)
	{ 	delete fRTPFilePtr; 
	} 
}


int QTFileBroadcaster::SetUp(PLBroadcastDef *broadcastDefPtr)
{
	int result = -1;
	int numStreams = 0;
	

	do 
	{
		if (! broadcastDefPtr) { result = eParam; 			break; };	
		if (!broadcastDefPtr->mSDPFile) { result = eSDPFileInvalidName; break; };
			
		int nameLen = strlen(broadcastDefPtr->mSDPFile);
		if (nameLen > 255) 	{ result = eSDPFileInvalidName; break; };			
		if (0 ==  nameLen) 	{ result = eSDPFileInvalidName; break; };	
		
		
		result = fStreamSDPParser.ReadSDP(broadcastDefPtr->mSDPFile);
		if (result != 0)
		{ 	if (result < 0) { result = eSDPFileNotFound; 	break; };
			if (result > 0) { result = eSDPFileInvalid; 	break; };
		}	
		
		if (!broadcastDefPtr->mBasePort) { result = eSDPFileNoPorts; break; };
		int portLen = strlen(broadcastDefPtr->mBasePort);
		if (0 ==  portLen) { result = eSDPFileNoPorts; 		break; };	
		if (portLen > 5  )  { result = eSDPFileInvalidPort; break; };
					
		int basePort = atoi(broadcastDefPtr->mBasePort);
		if (basePort == 0) { result = eSDPFileNoPorts;  	break; };	
		if 	( (basePort < 1000) || (basePort > 65535) ) 
			{ result = eSDPFileInvalidPort; 				break; };	
		
		numStreams = fStreamSDPParser.GetNumTracks();		
		if (numStreams == 0) { result = eSDPFileNoMedia; break; };
		
		UDPSocketPair *socketArrayPtr = fSocketlist.SetSize(numStreams);
		if (socketArrayPtr == NULL) { result = eMem; break; };
		
		result = fSocketlist.OpenAndBind(basePort,broadcastDefPtr->mDestAddress) ;
		if (result != 0) { result = eNoAvailableSockets; break; }
		
		MediaStream *mediaArrayPtr = fMediaStreamList.SetSize(numStreams);
		if (mediaArrayPtr == NULL) { result = eMem; break; }
				
		for (int i = 0; i < numStreams; i ++)
		{	UDPSocketPair 	*socketPairPtr = fSocketlist.SetPos(i);
			MediaStream 	*mediaStreamPtr = fMediaStreamList.SetPos(i);
			TypeMap 		*streamMediaTypePtr = fStreamSDPParser.fSDPMediaList.SetPos(i);
			
			if (socketPairPtr && mediaStreamPtr && streamMediaTypePtr)
			{
				mediaStreamPtr->fData.fSocketPair = socketPairPtr;
				streamMediaTypePtr->fMediaStreamPtr = mediaStreamPtr;
				mediaStreamPtr->fData.fStreamMediaTypePtr = streamMediaTypePtr;
			}
			else { result = eMem; break; };
		}
		
		
		fMediaStreamList.SetUpStreamSSRCs();
		fStreamStartTime = PlayListUtils::Milliseconds(); 
		fMediaStreamList.StreamStarted(fStreamStartTime);
		result = 0;
		LogFileOpen();
	} while (false);
	return result;
}

PayLoad * QTFileBroadcaster::FindPayLoad(short id, ArrayList<PayLoad> *PayLoadListPtr)
{
	PayLoad *thePayLoadPtr = PayLoadListPtr->Begin();
	while (thePayLoadPtr)
	{
		if (thePayLoadPtr->payloadID == id) 
			break;
			
		thePayLoadPtr = PayLoadListPtr->Next();
	}
	return thePayLoadPtr;
}

bool QTFileBroadcaster::CompareRTPMaps(TypeMap *movieMediaTypePtr, TypeMap *streamMediaTypePtr, short moviePayLoadID)
{
	bool found = false;
		
	do
	{
		PayLoad *moviePayLoadPtr = FindPayLoad(moviePayLoadID, &movieMediaTypePtr->fPayLoadTypes);
		if (!moviePayLoadPtr) break;
		
		PayLoad *streamPayLoadPtr = streamMediaTypePtr->fPayLoadTypes.Begin();
		while (streamPayLoadPtr)
		{
			if (moviePayLoadPtr->timeScale == streamPayLoadPtr->timeScale) 
			{	found =  SimpleParser::Compare(&moviePayLoadPtr->payLoadString,&streamPayLoadPtr->payLoadString );
			}
			
			if (found) 
			{	moviePayLoadPtr->payloadID = streamPayLoadPtr->payloadID; // map movie ID to match stream id
				break;
			}
			streamPayLoadPtr = streamMediaTypePtr->fPayLoadTypes.Next();
		}
			
		
	} while (false);
	return found;
}

bool QTFileBroadcaster::CompareMediaTypes(TypeMap *movieMediaTypePtr, TypeMap *streamMediaTypePtr)
{
	bool found = false;
	
	found = SimpleParser::Compare(&movieMediaTypePtr->fTheTypeStr,&streamMediaTypePtr->fTheTypeStr);
	if (found)
	{	
		found = false; 
		
		short *movieIDPtr = movieMediaTypePtr->fPayLoads.Begin();
		while (movieIDPtr && !found)
		{
			short *streamIDPtr = streamMediaTypePtr->fPayLoads.Begin();
			while (streamIDPtr && !found)
			{
				
				if (*movieIDPtr >= 96)
					found = CompareRTPMaps(movieMediaTypePtr, streamMediaTypePtr, *movieIDPtr);
				else
					found = (*streamIDPtr == *movieIDPtr) ? true : false;
			
				streamIDPtr = streamMediaTypePtr->fPayLoads.Next();		
			}

			movieIDPtr = movieMediaTypePtr->fPayLoads.Next();
		}
	}
	
	
	return found;
}


int QTFileBroadcaster::MapMovieToStream()
{
	int result = -1;
	bool matches = false;
	ArrayList<bool> map;
	bool *isMappedPtr;
	int masterPos = 0;	
	int mappedTracks = 0;
	map.SetSize(fStreamSDPParser.fSDPMediaList.Size());	
	isMappedPtr = map.Begin();
	
	while (isMappedPtr)
	{	*isMappedPtr = false;
		isMappedPtr = map.Next();
	}
	
	TypeMap *movieMediaTypePtr = fMovieSDPParser->fSDPMediaList.Begin();
	while (movieMediaTypePtr)
	{
		TypeMap *streamMediaTypePtr = fStreamSDPParser.fSDPMediaList.Begin();
		while (streamMediaTypePtr)
		{
			matches = CompareMediaTypes(movieMediaTypePtr, streamMediaTypePtr);
			if (matches)
			{
				masterPos = fStreamSDPParser.fSDPMediaList.GetPos();
				isMappedPtr = map.SetPos(masterPos);
				if (isMappedPtr == NULL) break;
				
				if (false == *isMappedPtr) 
				{	movieMediaTypePtr->fMediaStreamPtr = streamMediaTypePtr->fMediaStreamPtr;
					*isMappedPtr = true;
					mappedTracks ++;
					break; 
				}
			}
			streamMediaTypePtr = fStreamSDPParser.fSDPMediaList.Next();
		};
		movieMediaTypePtr = fMovieSDPParser->fSDPMediaList.Next();
	};

	result = mappedTracks;

	return result;

}

UInt32 QTFileBroadcaster::GetSDPTracks(QTRTPFile *newRTPFilePtr)
{
	char *sdpBuffer;
	int  bufferLen = 0;
	UInt32 result = 0;
	
	sdpBuffer = newRTPFilePtr->GetSDPFile(&bufferLen);
	fMovieSDPParser->ParseSDP(sdpBuffer);
	result = fMovieSDPParser->GetNumTracks();

	return result;
}

int QTFileBroadcaster::AddTrackAndStream(QTRTPFile *newRTPFilePtr)
{
	TypeMap *movieMediaTypePtr = fMovieSDPParser->fSDPMediaList.Begin();
	UInt32 trackID;
	char *cookie;
	int err = -1;
	
	while (movieMediaTypePtr)
	{
		if (movieMediaTypePtr->fMediaStreamPtr != NULL)
		{
			trackID = movieMediaTypePtr->fTrackID;
			if (trackID == 0) break;
			
			
			movieMediaTypePtr->fMediaStreamPtr->fData.fMovieMediaTypePtr = movieMediaTypePtr;
			movieMediaTypePtr->fMediaStreamPtr->fData.fRTPFilePtr = newRTPFilePtr;
			movieMediaTypePtr->fMediaStreamPtr->fSend = fSend;

			cookie = (char *) movieMediaTypePtr->fMediaStreamPtr;
    		err = newRTPFilePtr->AddTrack(trackID, false);
			if (err != 0) break;
			
			newRTPFilePtr->SetTrackCookie(trackID, cookie);		
 			newRTPFilePtr->SetTrackSSRC(trackID, (UInt32) 0); // set later
 			
 			err = newRTPFilePtr->Seek(0.0);
 			if (err != QTRTPFile::errNoError) break;
		}	
	
		movieMediaTypePtr = fMovieSDPParser->fSDPMediaList.Next();
	}
	
	return err;
}

int QTFileBroadcaster::SetUpAMovie(char *theMovieFileName)
{
	int err = -1;
	QTRTPFile *newRTPFilePtr = NULL;
	do {
		fMovieTracks = 0;
		fMappedMovieTracks = 0;
		
		if (fMovieSDPParser != NULL) delete fMovieSDPParser;
   		if (fRTPFilePtr != NULL) delete fRTPFilePtr;
   		
  		fMovieSDPParser = NULL;
		fRTPFilePtr = NULL;
		
		if (!theMovieFileName) 			{ err = eMovieFileInvalidName; break; }		
		int nameLen = strlen(theMovieFileName);						
		if (nameLen > 255) 				{ err = eMovieFileInvalidName; break; }
		if (0 == nameLen) 				{ err = eMovieFileInvalidName; break; }
		
		fMovieSDPParser = new SDPFileParser;
		if (NULL == fMovieSDPParser) 	{ err = eMem; break;}
		
		newRTPFilePtr = new QTRTPFile();
		if (NULL == newRTPFilePtr) 		{ err = eMem; break;}
		
		QTRTPFile::ErrorCode result = newRTPFilePtr->Initialize(theMovieFileName);
		err = EvalErrorCode(result);
		if (err) break;
		
		fMovieTracks = GetSDPTracks(newRTPFilePtr);
		if (fMovieTracks < 1) 			{ err = eMovieFileNoHintedTracks; break; }
		
		fMappedMovieTracks = MapMovieToStream();
		if (fMappedMovieTracks < 1)  	{ err = eMovieFileNoSDPMatches; break; }

		err = AddTrackAndStream(newRTPFilePtr);		
		if (err != 0) 					{ err = eMovieFileInvalid; break; }
		
	} while (false);
	
	if (err) 
	{	if (newRTPFilePtr) delete newRTPFilePtr;
		newRTPFilePtr = NULL;
	}
	
	fRTPFilePtr = newRTPFilePtr;
	
	return err;
}

Float64 QTFileBroadcaster::Sleep(Float64 transmitTimeMicro)
{
	Float64 sleepTime;
	Float64 timeToSend;
	Float64 currentTime = PlayListUtils::Microseconds();
	
	if (fMovieStart) 
	{	
		fMovieStart = false;
//		if (transmitTimeMicro < 0)  // make transmitTime time 0 based if negative
//			fSendTimeOffset = -1 * transmitTimeMicro;

//		if (fMovieIntervalTimeMicro > 0) // make the interval time a negative offset 
//		{	fMovieIntervalTimeMicro *= -1; 
//		}
	}
	
//	transmitTimeMicro += fSendTimeOffset;

//	transmitTimeMicro += (Float64) fMovieIntervalTimeMicro;
//	transmitTimeMicro += (Float64) (-eClientBufferSecs * PlayListUtils::eMicro);

	timeToSend = fMovieStartTime + transmitTimeMicro;
	sleepTime =  timeToSend - currentTime;
					
	if (sleepTime >  0.0)
	{	
		unsigned int microseconds = (unsigned int) (sleepTime) ;
		usleep(microseconds);
	}
	
	return sleepTime;
}

int QTFileBroadcaster::Play()
{
	SInt16 	err = 0;
	Float64 transmitTime = 0;
	MediaStream	*theStreamPtr = NULL;	
	RTpPacket 	rtpPacket;
	unsigned int sleptTimeMicro;
	
	fMovieStart = true;
	fNumMoviesPlayed ++;
	
	fMovieStartTime = PlayListUtils::Microseconds();	
	fMediaStreamList.MovieStarted(fMovieStartTime);	
	
	if (fMovieEndTime != 0) 
	{	fMovieIntervalTimeMicro = fMovieStartTime - fMovieEndTime;
	}
	
	fSendTimeOffset = 0.0;
	fMovieDuration = fRTPFilePtr->GetMovieDuration();
	
	while (true) 
	{
		
		transmitTime = fRTPFilePtr->GetNextPacket(&rtpPacket.fThePacket, &rtpPacket.fLength, (void **)&theStreamPtr);
		
		if (NULL == rtpPacket.fThePacket)	{err = 0; break; } // end of movie not an error
		if (NULL == theStreamPtr) 			{err = eMovieFileInvalid; break; }// an error
		
		transmitTime *= (Float64) PlayListUtils::eMicro; // convert to microseconds
		sleptTimeMicro = (unsigned int) Sleep(transmitTime);
			
		err = theStreamPtr->Send(&rtpPacket);
		if (err != 0)  { err = eInternalError; break; } 
		
		SInt64 theTime = PlayListUtils::Milliseconds();
		err = theStreamPtr->UpdateSenderReport(theTime);
		if (err != 0)  { err = eInternalError; break; } 
		
		err = fMediaStreamList.UpdateStreams();
		if (err != 0)  { err = eInternalError; break; } 
		
	};
	
	fMovieEndTime = (SInt64) PlayListUtils::Microseconds();	
	fMediaStreamList.MovieEnded(fMovieEndTime);
	

	Float64 playDuration = ( (Float64) ((SInt64) fMovieEndTime - (SInt64) fMovieStartTime)) / (Float64) PlayListUtils::eMicro;
	Float64 timeDiff =  (Float64) ((Float64) fMovieDuration - (Float64) playDuration);

/*
	if (fMovieDuration > eClientBufferSecs ) 
		timeDiff -= eClientBufferSecs;
//	else
//		timeDiff -= 1.0;
		
	printf("playDuration  = %f movieDuration = %f \n", (float) playDuration,(float) fMovieDuration);
	printf("time diff  = %f \n", (float) timeDiff);

*/
	timeDiff -= .05;
	if (timeDiff > 0.0)
	{	unsigned int sleepTime = (unsigned int) ( (Float64) timeDiff * (Float64) PlayListUtils::eMicro);
		usleep(sleepTime);
//		printf("sleep to catch up = %f \n", timeDiff);
	}

//		usleep(10000000);  // test for delay between movies
	return err;
}

int QTFileBroadcaster::PlayMovie(char *movieFileName)
{
	
	int err = eMovieFileInvalidName;
	if (movieFileName != NULL)
	{	
	
		err = SetUpAMovie(movieFileName);
		
		if (!err && fPlay)
		{	 err = Play();
		}
	}
	return err;
}


int QTFileBroadcaster::EvalErrorCode(QTRTPFile::ErrorCode err)
{	
	int result = eNoErr;
	
	switch( err ) 
	{
		case QTRTPFile::errNoError:
		break;

		case QTRTPFile::errFileNotFound:
			result = eMovieFileNotFound;
		break;
		
		case QTRTPFile::errNoHintTracks:
			result = eMovieFileNoHintedTracks;
		break;

		case QTRTPFile::errInvalidQuickTimeFile:
			result = eMovieFileInvalid;
		break;
		
		case QTRTPFile::errInternalError:
			result = eInternalError;
		break;
		
		default:
			result = eInternalError;
	}
	return result;
}
