/* swapd.c, copyright Neven Lovric <nlovric@linux.hr> */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <asm/page.h>
#include <sys/swap.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <limits.h>
#include "config.h"
#include "copyright.h"
#include "meminfo.h"
#include "procname.h"
#include "fileinfo.h"
#include "swapinfo.h"
#include "swapd.h"
#include "memops.h"
#include "procops.h"

unsigned long memlimit;
unsigned long swapsizevar;
unsigned long maxswaps = 0;
unsigned long timeout;
unsigned long pausevar = 1000000;
char swapdir[256];
char pidfile[256] = PIDFILE;
char conffile[256] = CONFFILE;
char mkswap[256] = MKSWAP;
unsigned long unusedswap = -1;
int die = 0;
unsigned long set = 0;

int main(int argc, char **argv)
{
   unsigned long mem;
   unsigned long swaps = 0;
   time_t *lastused = NULL;
   unsigned long n;
   unsigned long *swapspace = NULL;
   unsigned long size;
   unsigned long oldspace = 0;
   unsigned long sum;
   int meminfofd;

   printf("swapd 0.2 (May 21, 2000), copyright Neven Lovric <nlovric@linux.hr>\n");
   if (!geteuid()) {
      setreuid(0, -1);
   }
   if (getuid()) {
      fprintf(stderr, "swapd must be run as root.\n");
      exit(1);
   }
   if (readopts(argc, argv))
      exit(1);
   if (readconf())
      exit(1);
   if (!(set & MEMLIMIT_SET)) {
      fprintf(stderr, "swapd: memory limit not set.\n");
      die = 1;
   }
   if (!(set & SWAPSIZE_SET)) {
      fprintf(stderr, "swapd: swap size not set.\n");
      die = 1;
   } else
      if (swapsizevar < 65536) {
         fprintf(stderr, "swapd: swapsize too small, must be at least 64k.\n");
         die = 1;
      }
   if (!(set & SWAPDIR_SET)) {
      fprintf(stderr, "swapd: swap directory not set.\n");
      die = 1;
   }
   if (!(set & TIMEOUT_SET)) {
      fprintf(stderr, "swapd: timeout not set.\n");
      die = 1;
   }
   if (die)
      exit(1);
   if (!isfile(mkswap)) {
      fprintf(stderr, "swapd: %s not found.\n", mkswap);
      exit(1);
   }
   if (!isdir(swapdir)) {
      fprintf(stderr, "swapd: %s: no such directory.\n", swapdir);
      exit(1);
   }
   if ((meminfofd = open("/proc/meminfo", O_RDONLY)) == -1) {
      perror("swapd: /proc/meminfo");
      exit(1);
   }
   if (sfork()) {
      signal(SIGINT, exitonsig);
      wait(NULL);
      exit(0);
   }
   usleep(1);
   setpriority(PRIO_PROCESS, 0, -1);
   setprocname(argc, argv, PROCNAME);
   if (chkpidfile())
      exit(1);
   while (1) {
      if (!(size = filesize(swapname(swaps))) || !isfile(swapname(swaps)))
         break;
      swapon(swapname(swaps), 0);
      swaps++;
      lastused = srealloc(lastused, swaps * sizeof(time_t));
      lastused[swaps - 1] = 0;
      swapspace = srealloc(swapspace, swaps * sizeof(time_t));
      swapspace[swaps - 1] = swapsize(swapname(swaps - 1));
      oldspace += size;
   }
   if (swaps)
      printf("swapd: %lu swap files (%lukb total) found and reused.\n", swaps, oldspace >> 10);
   kill(getppid(), SIGINT);
   setsid();
   signal(SIGINT, handlesig);
   signal(SIGHUP, handlesig);
   signal(SIGTERM, handlesig);
   while (1) {
      if (die) {
         close(meminfofd);
         exit(0);
      }
      if ((mem = freemem(meminfofd)) == -1)
         continue;
      if (mem < memlimit && (!maxswaps || swaps < maxswaps)) {
         swaps = newswap(swaps);
         lastused = srealloc(lastused, swaps * sizeof(time_t));
         lastused[swaps - 1] = 0;
         swapspace = srealloc(swapspace, swaps * sizeof(time_t));
         swapspace[swaps - 1] = swapsize(swapname(swaps - 1));
      }
      if (swaps) {
         for (n = 1, sum = 0; n <= swaps; n++) {
            sum += swapspace[swaps - n];
            if (mem > memlimit + sum) {
               if (!lastused[swaps - n])
                  lastused[swaps - n] = time(NULL);
            } else {
               lastused[swaps - n] = 0;
               break;
            }
         }
         if ((lastused[swaps - 1] && time(NULL) - lastused[swaps - 1] > timeout) || (maxswaps && swaps > maxswaps)) {
            swaps = delswap(swaps);
            lastused = srealloc(lastused, swaps * sizeof(time_t));
            swapspace = srealloc(swapspace, swaps * sizeof(time_t));
         }
      }
      if (mem > memlimit)
         usleep(pausevar);
   }
}

int chkpidfile(void)
{
   FILE *file;
   pid_t pid;
   char filename[256];
   char procname[256];
   int ret;
   char dirname[256];
   
   if (isfile(pidfile) && (file = fopen(pidfile, "r"))) {
      ret = fscanf(file, "%i", &pid);
      fclose(file);
      if (ret == 1) {
         sprintf(filename, "/proc/%i/cmdline", pid);
         if ((file = fopen(filename, "r"))) {
            fscanf(file, "%s", procname);
            fclose(file);
            if (!strcmp(PROCNAME, procname)) {
               if (kill(pid, SIGTERM) == -1) {
                  fprintf(stderr, "swapd: error terminating old swapd (PID %u).\n", pid);
                  return(1);
               }
               sprintf(dirname, "/proc/%i", pid);
               while (isdir(dirname))
                  usleep(1);
               printf("swapd: found and terminated existing swapd (PID %i).\n", pid);
            }
         }
      }
      if (remove(pidfile) == -1) {
         fprintf(stderr, "swapd: error removing old pidfile %s\n", pidfile);
         return(1);
      }
   } else
      remove(pidfile);
   if (!(file = fopen(pidfile, "w"))) {
      fprintf(stderr, "swapd: error writing pidfile %s\n", pidfile);
      return(1);
   }
   fprintf(file, "%i\n", getpid());
   if (fclose(file) == EOF) {
      fprintf(stderr, "swapd: error writing pidfile %s\n", pidfile);
      return(1);
   }
   return(0);
}

void phelp(char **argv)
{
   printf("%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
      "\nUsage: ", argv[0], " [options]\n\n",
      "   -h, --help        you are reading it!\n",
      "   --copyright       display the copyright notice\n",
      "   --config file     config file (default: ", CONFFILE, ")\n",
      "   --maxswaps n      maximum number of swap files (0 = unlimited, default: 0)\n",
      "   --pause msec      pause between memory checks in miliseconds\n",
      "   --memlimit kb     memory limit in kilobytes\n",
      "   --mkswap path     full path to mkswap (default: ", MKSWAP, ")\n",
      "   --pidfile file    pid file (default: ", PIDFILE, ")\n",
      "   --swapdir dir     swap directory where all swap files are kept\n",
      "   --swapsize kb     swap file size in kilobytes (minimum 64k)\n",
      "   --timeout sec     time in seconds after which unused swap files are removed\n",
      "   -v, --version     display version information and exit\n\n",
      "Values given on the command line override config file values.\n",
      "The latest version of this program can be found at http://cvs.linux.hr/swapd or\n",
      "at ftp.linux.hr in /pub/swapd.\n");
}

int readopts(int argc, char **argv)
{
   int n;
   
   for (n = 1; n < argc; n++) {
      if (!strcmp(argv[n], "-h") || !strcmp(argv[n], "--help")) {
         phelp(argv);
         return(1);
      }
      if (!strcmp(argv[n], "--copyright")) {
         printf("\n%s\n", copyright);
         return(1);
      }
      if (!strcmp(argv[n], "--config")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --conffile.\n");
            return(1);
         }
         strncpy(conffile, argv[n], 255);
         conffile[255] = 0;
         continue;
      }
      if (!strcmp(argv[n], "--maxswaps")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --maxswaps.\n");
            return(1);
         }
         maxswaps = strtoul(argv[n], NULL, 10);
         set |= MAXSWAPS_SET;
         continue;
      }
      if (!strcmp(argv[n], "--pause")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --pause.\n");
            return(1);
         }
         pausevar = strtoul(argv[n], NULL, 10) * 1000;
         set |= PAUSE_SET;
         continue;
      }
      if (!strcmp(argv[n], "--memlimit")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --memlimit.\n");
            return(1);
         }
         memlimit = strtoul(argv[n], NULL, 10) * 1024;
         set |= MEMLIMIT_SET;
         continue;
      }
      if (!strcmp(argv[n], "--mkswap")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --mkswap.\n");
            return(1);
         }
         strncpy(mkswap, argv[n], 255);
         mkswap[255] = 0;
         set |= MKSWAP_SET;
         continue;
      }
      if (!strcmp(argv[n], "--pidfile")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --pidfile.\n");
            return(1);
         }
         strncpy(pidfile, argv[n], 255);
         pidfile[255] = 0;
         set |= PIDFILE_SET;
         continue;
      }
      if (!strcmp(argv[n], "--swapdir")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --swapdir.\n");
            return(1);
         }
         strncpy(swapdir, argv[n], 255);
         swapdir[255] = 0;
         set |= SWAPDIR_SET;
         continue;
      }
      if (!strcmp(argv[n], "--swapsize")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --swapsize.\n");
            return(1);
         }
         swapsizevar = strtoul(argv[n], NULL, 10) * 1024;
         set |= SWAPSIZE_SET;
         continue;
      }
      if (!strcmp(argv[n], "--timeout")) {
         if ((++n) == argc) {
            fprintf(stderr, "swapd: missing argument for option --timeout.\n");
            return(1);
         }
         timeout = strtoul(argv[n], NULL, 10);
         set |= TIMEOUT_SET;
         continue;
      }
      if (!strcmp(argv[n], "-v") || !strcmp(argv[n], "--version"))
         return(1);
      fprintf(stderr, "swapd: unknown option %s.\n", argv[n]);
      return(1);
   }
   return(0);
}

int readconf(void)
{
   FILE *file;
   unsigned long line = 0;
   int err = 0;
   char buf[1024];
   char *val;
   
   if (!isfile(conffile)) {
      fprintf(stderr, "swapd: %s: no such file.\n", conffile);
      return(1);
   }
   if (!(file = fopen(conffile, "r"))) {
      fprintf(stderr, "swapd: error opening %s.\n", conffile);
      return(1);
   }
   while (!feof(file)) {
      if (fgets(buf, 1024, file) == NULL)
         break;
      if (*buf == '#' || *buf == '\n')
         continue;
      buf[strlen(buf) - 1] = 0;
      if ((val = strchr(buf, ' '))) {
         *val = 0;
         val ++;
      } else
         fprintf(stderr, "swapd: %s, line %lu: no value given for %s.\n", conffile, line, buf);
      line ++;
      if (!strcmp(buf, "memlimit")) {
         if (!(set & MEMLIMIT_SET)) {
            memlimit = strtoul(val, NULL, 10) * 1024;
            set |= MEMLIMIT_SET;
         }
         continue;
      }
      if (!strcmp(buf, "pause")) {
         if (!(set & PAUSE_SET)) {
            pausevar = strtoul(val, NULL, 10) * 1000;
            set |= PAUSE_SET;
         }
         continue;
      }
      if (!strcmp(buf, "swapsize")) {
         if (!(set & SWAPSIZE_SET)) {
            swapsizevar = strtoul(val, NULL, 10) * 1024;
            set |= SWAPSIZE_SET;
         }
         continue;
      }
      if (!strcmp(buf, "maxswaps")) {
         if (!(set & MAXSWAPS_SET))
            maxswaps = strtoul(val, NULL, 10);
         continue;
      }
      if (!strcmp(buf, "timeout")) {
         if (!(set & TIMEOUT_SET)) {
            timeout = strtoul(val, NULL, 10);
            set |= TIMEOUT_SET;
         }
         continue;
      }
      if (!strcmp(buf, "swapdir")) {
         if (!(set & SWAPDIR_SET)) {
            strcpy(swapdir, val);
            set |= SWAPDIR_SET;
         }
         continue;
      }
      if (!strcmp(buf, "pidfile")) {
         if (!(set & PIDFILE_SET))
            strcpy(pidfile, val);
         continue;
      }
      if (!strcmp(buf, "mkswap")) {
         if (!(set & MKSWAP_SET))
            strcpy(mkswap, val);
         continue;
      }
      fprintf(stderr, "swapd: %s, line %lu: unknown option %s.\n", conffile, line, buf);
      err = 1;
   }
   fclose(file);
   if (err)
      return(1);
   return(0);
}

int runmkswap(char *swapfile)
{
   int status;
   
   if (!sfork()) {
      fclose(stdout);
      fclose(stderr);
      execl(mkswap, "mkswap", swapfile, NULL);
      exit(1);
   }
   wait(&status);
   if (status)
      return(-1);
   return(0);
}

char *swapname(unsigned long swapnum)
{
   static char swapfile[256];

   sprintf(swapfile, "%s/linux%lu.swp", swapdir, swapnum);
   return(swapfile);
}

int doswapoff(char *filename)
{
   char *mem;
   unsigned long i;
   
   if (swapoff(filename) == -1)
      return(-1);
   i = filesize(filename) >> 10;
   for (; i > 0; i--) {
      mem = smalloc(1);
      *mem = 1;
      free(mem);
   }
   return(0);
}

unsigned long newswap(unsigned long swaps)
{
   char *swapfile;
   int fd;
   unsigned long n;
   char zero[1024] = { 0 };

   swapfile = swapname(swaps);
   if (!(unusedswap == swaps && isfile(swapfile))) {
      doswapoff(swapfile);
      remove(swapfile);
      unusedswap = -1;
      if ((fd = open(swapfile, O_CREAT | O_WRONLY, 00600)) == -1)
         return(swaps);
      for (n = 0; n < (swapsizevar >> 10); n++)
         if (write(fd, zero, 1024) != 1024) {
            close(fd);
            remove(swapfile);
            return(swaps);
         }
      if (close(fd) == -1) {
         remove(swapfile);
         return(swaps);
      }
      if (runmkswap(swapfile)) {
         remove(swapfile);
         return(swaps);
      }
   }
   if (swapon(swapfile, 0) == -1 && errno != EINVAL) {
      if (errno != ENOENT)
         unusedswap = swaps;
      else
         unusedswap = -1;
      return(swaps);
   } else
      unusedswap = -1;
   return(swaps + 1);
}

unsigned long delswap(unsigned long swaps)
{
   char *swapfile;

   if (unusedswap == swaps) {
      swapfile = swapname(swaps);
      doswapoff(swapfile);
      remove(swapfile);
      unusedswap = -1;
   }
   swapfile = swapname(swaps - 1);
   if (doswapoff(swapfile) == -1 && errno != ENOENT && errno != EINVAL)
      return(swaps);
   remove(swapfile);
   return(swaps - 1);
}

void handlesig(int sig)
{
   signal(sig, handlesig);
   die = 1;
}

void exitonsig(int sig)
{
   exit(0);
}
