/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif
#ifdef ENABLE_LAME

#include "../../player.h"
#include "../../util.h"
#include "../../other.h"

lame_global_flags *
lame_config (long rate_in, long rate_out, int channels_in, int channels_out,
	     int lowpass, int highpass, int bitrate, int quality)
{
  lame_global_flags *lame;

  if (channels_out > 2)
    {
      msg_error (_("Mp3 encoding no support channels>2"));
      return NULL;
    }

  if (!(lame = lame_init ()))
    {
      msg_error (_("Lame init error."));
      return NULL;
    }

  if (lame_set_num_channels (lame,channels_in) < 0)
    {
      msg_error (_("Lame setting channels error."));
      lame_close (lame);
      return NULL;
    }

  if (lame_set_mode (lame,channels_out == 1 ? MONO : JOINT_STEREO) < 0)
    {
      msg_error (_("Lame set mode error."));
      lame_close (lame);
      return NULL;
    }

  if (lame_set_in_samplerate (lame, rate_in) < 0)
    {
      msg_error (_("Lame setting in samplerate"));
      lame_close (lame);
      return NULL;
    }

  if (lame_set_out_samplerate (lame, rate_out) < 0)
    {
      msg_error (_("Lame setting out samplerate"));
      lame_close (lame);
      return NULL;
    }

  lame_set_lowpassfreq (lame, lowpass);
  lame_set_highpassfreq (lame, highpass);

  lame_set_brate (lame, bitrate);

  lame_set_quality (lame, (int) ((10.0 - (double)quality) + 0.499999));

  lame_set_exp_nspsytune (lame, 1);
  lame_set_error_protection (lame, 1);
  lame_init_params (lame);

  return lame;
}

int
lame_write (lame_global_flags * lame, int channels, int bitrate, void *buffer,
	    size_t length, unsigned char **out_buf, int *out_size)
{

  unsigned int sample = (bitrate / 8) * channels;
  unsigned char *b = (unsigned char *) buffer;
  unsigned int processed = length - (length % sample);

  unsigned int n_samples = processed / sample;
  short int left[n_samples];
  short int right[n_samples];

  int ret;

  int mp3_size = (unsigned int) (1.25 * n_samples + 7200);
  unsigned char mp3_buf[mp3_size];

  if (!length)
    {
      *out_buf = NULL;
      *out_size = 0;

      return 0;
    }

  if (!lame)
    return -1;

  if(bitrate == 8)
    conv8 (b, processed, left, right, channels);
  else if(bitrate == 16)
    conv16 (b, processed, left, right, channels);
  else {
    msg_error("Bitrate must be 8 or 16!");
    return -1;
  }

  ret =
    lame_encode_buffer (lame, left, channels == 2 ? right : left,
			n_samples, mp3_buf, mp3_size);

  if (ret < 1)
    *out_buf = NULL;

  else
    {
      if (!
	  (*out_buf =
	   (unsigned char *) malloc (ret * sizeof (unsigned char))))
	fatal (_("Error: memory."));

      memcpy (*out_buf, mp3_buf, ret);
    }

  *out_size = ret;

  return *out_size;
}

void
lame_quit (lame_global_flags * lame)
{

  if (!lame)
    return;

  lame_close (lame);
}

#endif
