/******************************************************************************
*******************************************************************************
**
**  Copyright (C) 2005 Red Hat, Inc.  All rights reserved.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2 of the License, or (at your option) any later version.
**
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**  Lesser General Public License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**
*******************************************************************************
******************************************************************************/

/* Implements saClm API using libcman */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <inttypes.h>
#include <unistd.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <string.h>
#include <errno.h>
#include <stddef.h>
#include "libcman.h"
#include "saClm.h"

struct sa_handle
{
	unsigned int    sa_magic;
	cman_handle_t   sa_cman_handle;
	SaClmCallbacksT sa_callbacks;

	SaClmClusterNotificationBufferT sa_notify_buffer;
	cman_node_t *sa_nodelist; /* For change deltas */
	int sa_numnodes;          /* Size of above */
	int sa_notify_flags;
	int sa_notify_state;
};
#define SA_CMAN_MAGIC 0x34651287

#define VALIDATE_HANDLE(h) do {if (!(h) || (h)->sa_magic != SA_CMAN_MAGIC) {return SA_AIS_ERR_BAD_HANDLE;}} while (0)

static void cman_to_sanode(cman_node_t *cman_node, SaClmClusterNodeT *clusterNode)
{
	memset(clusterNode, 0, sizeof(SaClmClusterNodeT));

	clusterNode->nodeId = cman_node->cn_nodeid;
        strcpy((char *)clusterNode->nodeName.value, cman_node->cn_name);
	clusterNode->nodeName.length = strlen(cman_node->cn_name) + 1;
        clusterNode->member = cman_node->cn_member;
//        clusterNode->bootTimestamp; //Not supported (yet)
//        SaUint64T initialViewNumber;//Not supported (yet)
	if (cman_node->cn_address.cna_address[0] == AF_INET6)
	{
		clusterNode->nodeAddress.family	= SA_CLM_AF_INET6;
		memcpy(clusterNode->nodeAddress.value, cman_node->cn_address.cna_address + offsetof(struct sockaddr_in6, sin6_addr), sizeof(struct in6_addr));
		clusterNode->nodeAddress.length = sizeof(struct in6_addr);
	}
	else
	{
		clusterNode->nodeAddress.family	= SA_CLM_AF_INET;
		memcpy(clusterNode->nodeAddress.value, cman_node->cn_address.cna_address + offsetof(struct sockaddr_in, sin_addr), sizeof(struct in_addr));
		clusterNode->nodeAddress.length = sizeof(struct in_addr);
	}
}


/* cman callback */
static void sa_notify_callback(cman_handle_t h, void *private, int reason, int arg)
{
	struct sa_handle *sah = private;
	cman_node_t *nodes;
	int nodecount;
	int actual_nodes;
	int old_i, new_i;
	int delta_size = 0;
	SaClmClusterNotificationBufferT sa_buffer = sah->sa_notify_buffer;

	if (reason != CMAN_REASON_STATECHANGE)
		return;

	// Bugger, can't do this in a callback...can we?
	nodecount = cman_get_node_count(h);
	if (nodecount <= 0)
		return;

	nodecount += 2; /* contingency */
	nodes = malloc(nodecount * sizeof(cman_node_t));
	if (!nodes)
		return;

	if (cman_get_nodes(h, nodecount, &actual_nodes, nodes))
		return;

	/* cman guarantees that the nodes will be in nodeid order so we know that
	 * both lists can be easily compared. It's also the case that a node will
	 * never be removed from the list so we know that the newest list will
	 * be >= length of the old list
	 */

	/* First pass through the lists is just to get the size of the delta */
	if (sah->sa_notify_flags == SA_TRACK_CHANGES_ONLY)
	{
		delta_size = actual_nodes - sah->sa_numnodes;
		for (old_i = new_i = 0; old_i < sah->sa_numnodes; old_i++, new_i++)
		{
			if (sah->sa_nodelist[old_i].cn_nodeid == nodes[new_i].cn_nodeid)
			{
				if (sah->sa_nodelist[old_i].cn_member != nodes[new_i].cn_member)
					delta_size++;
			}
		}
	}
	else
	{
		delta_size = actual_nodes;
	}

	/* Is there enough space ? */
	if (delta_size > sah->sa_notify_buffer.numberOfItems)
	{
		/* no... */
		sah->sa_callbacks.saClmClusterTrackCallback(&sah->sa_notify_buffer, delta_size, SA_AIS_ERR_NO_SPACE);
		return;
	}

	/* Now fill in the buffer and return it to the caller */
	if (sah->sa_notify_flags == SA_TRACK_CHANGES_ONLY)
	{
		int delta_num = 0;
		for (old_i = new_i = 0; old_i < sah->sa_numnodes; old_i++, new_i++)
		{
			if (sah->sa_nodelist[old_i].cn_nodeid == nodes[new_i].cn_nodeid)
			{
				if (sah->sa_nodelist[new_i].cn_member == nodes[new_i].cn_member)
					continue;
				if (sah->sa_nodelist[new_i].cn_member < nodes[new_i].cn_member)
					sa_buffer.notification[delta_num].clusterChange = SA_CLM_NODE_JOINED;
				if (sah->sa_nodelist[new_i].cn_member > nodes[new_i].cn_member)
					sa_buffer.notification[delta_num].clusterChange = SA_CLM_NODE_LEFT;
				sa_buffer.notification[delta_num].clusterChange = SA_CLM_NODE_NO_CHANGE;
				cman_to_sanode(&nodes[delta_num], &sa_buffer.notification[new_i].clusterNode);
				delta_num++;
			}
			else
			{
				cman_to_sanode(&nodes[delta_num], &sa_buffer.notification[new_i].clusterNode);
				sa_buffer.notification[delta_num].clusterChange = SA_CLM_NODE_JOINED;
				new_i++;
				delta_num++;
			}
		}

	}
	else
	{
		for (old_i = new_i = 0; old_i < sah->sa_numnodes; old_i++, new_i++)
		{
			cman_to_sanode(&nodes[new_i], &sa_buffer.notification[new_i].clusterNode);
			if (sah->sa_nodelist[old_i].cn_nodeid == nodes[new_i].cn_nodeid)
			{
				if (sah->sa_nodelist[new_i].cn_member == nodes[new_i].cn_member)
					sa_buffer.notification[new_i].clusterChange = SA_CLM_NODE_NO_CHANGE;
				if (sah->sa_nodelist[new_i].cn_member < nodes[new_i].cn_member)
					sa_buffer.notification[new_i].clusterChange = SA_CLM_NODE_LEFT;
				if (sah->sa_nodelist[new_i].cn_member > nodes[new_i].cn_member)
					sa_buffer.notification[new_i].clusterChange = SA_CLM_NODE_JOINED;
			}
			else
			{
				sa_buffer.notification[new_i].clusterChange = SA_CLM_NODE_JOINED;
				new_i++;
			}
		}
	}
	/* Replace the node list */
	free(sah->sa_nodelist);
	sah->sa_nodelist = nodes;
	sah->sa_numnodes = actual_nodes;

	/* And tell the user */
	sah->sa_callbacks.saClmClusterTrackCallback(&sah->sa_notify_buffer, delta_size, SA_AIS_OK);
}


/* num_items comes pre-filled with the number of items in the cman_node_t array and shouldn't
   need changing to match those passed back */
static int cman_to_sa_notification(cman_node_t *nodes, SaClmClusterNotificationBufferT *sa_buffer,
				   int num_items, int *num_members)
{
	int i;
	int members = 0;

	for (i=0; i < num_items; i++)
	{
		cman_to_sanode(&nodes[i], &sa_buffer->notification[i].clusterNode);
		sa_buffer->notification->clusterChange = SA_CLM_NODE_NO_CHANGE;
		if (nodes[i].cn_member)
			members++;
	}

	*num_members = members;
	return SA_AIS_OK;
}


SaAisErrorT saClmInitialize(SaClmHandleT *clmHandle, const SaClmCallbacksT *clmCallbacks, SaVersionT *version)
{
	cman_handle_t ch;
	struct sa_handle *sah;

	/* Check version */
	if ((version->releaseCode & 0x5F) != 'B' || version->majorVersion != 1)
		return SA_AIS_ERR_VERSION;

	sah = malloc(sizeof (struct sa_handle));
	if (!sah)
		return SA_AIS_ERR_NO_MEMORY;

	ch = cman_init(sah);
	if (!ch)
	{
		free(sah);
		return SA_AIS_ERR_FAILED_OPERATION;
	}

	sah->sa_cman_handle = ch;
	sah->sa_magic = SA_CMAN_MAGIC;
	sah->sa_callbacks = *clmCallbacks;

	version->releaseCode = 'B';
	version->majorVersion = 1;
	version->minorVersion = 1;

	*clmHandle = (SaClmHandleT)(intptr_t)sah;
	return SA_AIS_OK;
}

SaAisErrorT saClmFinalize(SaClmHandleT clmHandle)
{
	struct sa_handle *sah = (struct sa_handle *)(intptr_t)clmHandle;
	VALIDATE_HANDLE(sah);

	if (cman_finish(sah->sa_cman_handle))
		return SA_AIS_ERR_LIBRARY;

	free(sah);
	return SA_AIS_OK;
}


SaAisErrorT saClmClusterNodeGet(SaClmHandleT clmHandle, SaClmNodeIdT nodeId, SaTimeT timeout, SaClmClusterNodeT *clusterNode)
{
	struct sa_handle *sah = (struct sa_handle *)(intptr_t)clmHandle;
	cman_node_t cman_node;
	int ret;

	VALIDATE_HANDLE(sah);

	if (nodeId == SA_CLM_LOCAL_NODE_ID)
		nodeId = CMAN_NODEID_US;

	ret = cman_get_node(sah->sa_cman_handle, nodeId, &cman_node);
	if (ret)
	{
		if (errno == ENOENT)
			return SA_AIS_ERR_INVALID_PARAM;
		else
			return SA_AIS_ERR_LIBRARY;
	}
	cman_to_sanode(&cman_node, clusterNode);


	return SA_AIS_OK;
}

SaAisErrorT saClmSelectionObjectGet(SaClmHandleT clmHandle, SaSelectionObjectT *selectionObject)
{
	struct sa_handle *sah = (struct sa_handle *)(intptr_t)clmHandle;

	VALIDATE_HANDLE(sah);

	*selectionObject = cman_get_fd(sah->sa_cman_handle);
	if (*selectionObject == -1)
		return SA_AIS_ERR_LIBRARY;

	return SA_AIS_OK;
}

SaAisErrorT saClmDispatch(SaClmHandleT clmHandle, SaDispatchFlagsT dispatchFlags)
{
	struct sa_handle *sah = (struct sa_handle *)(intptr_t)clmHandle;
	int cman_flags = 0;

	VALIDATE_HANDLE(sah);

	if (dispatchFlags & SA_DISPATCH_ONE)
		cman_flags |= CMAN_DISPATCH_ONE;
	if (dispatchFlags & SA_DISPATCH_ALL)
		cman_flags |= CMAN_DISPATCH_ALL;
	if (dispatchFlags & SA_DISPATCH_BLOCKING)
		cman_flags |= CMAN_DISPATCH_BLOCKING;

	cman_dispatch(sah->sa_cman_handle, cman_flags);
	return SA_AIS_OK;
}

SaAisErrorT saClmClusterTrack(SaClmHandleT clmHandle, SaUint8T trackFlags, SaClmClusterNotificationBufferT *notificationBuffer)
{
	struct sa_handle *sah = (struct sa_handle *)(intptr_t)clmHandle;
	int maxnode;
	int ret;
	cman_node_t *nodes;
	int num_items, num_members;

	VALIDATE_HANDLE(sah);

	if (!notificationBuffer || !sah->sa_callbacks.saClmClusterTrackCallback ||
	    !notificationBuffer->notification)
		return SA_AIS_ERR_INVALID_PARAM;

	sah->sa_notify_buffer = *notificationBuffer;
	sah->sa_notify_flags = trackFlags;
	sah->sa_notify_state = 0;

	maxnode = cman_get_node_count(sah->sa_cman_handle);
	if (!maxnode)
		return SA_AIS_ERR_LIBRARY;

	nodes = malloc(sizeof(cman_node_t) * maxnode);

	ret = cman_get_nodes(sah->sa_cman_handle, maxnode, &num_items, nodes);
	if (ret)
		return SA_AIS_ERR_LIBRARY;

	/* if its one-shot then do it now */
	if (trackFlags == SA_TRACK_CURRENT)
	{
		cman_to_sa_notification(nodes, &sah->sa_notify_buffer, num_items, &num_members);
		sah->sa_callbacks.saClmClusterTrackCallback(&sah->sa_notify_buffer, num_members, SA_AIS_OK);
		return SA_AIS_OK;
	}
	/* Save the current state so we can do deltas */
	sah->sa_nodelist = nodes;
	sah->sa_numnodes = num_items;

	cman_start_notification(sah->sa_cman_handle, sa_notify_callback);

	return SA_AIS_OK;
}

SaAisErrorT saClmClusterTrackStop(SaClmHandleT clmHandle)
{
	struct sa_handle *sah = (struct sa_handle *)(intptr_t)clmHandle;

	VALIDATE_HANDLE(sah);
	cman_stop_notification(sah->sa_cman_handle);

	return SA_AIS_OK;
}

/* This is not yet properly supported - as getnode is always synchronous in cman 1.
   in cman 2, we will need to bypass the API but that should be OK.

   For now we'll just do a get_node() and call the callback immediately, hope this
   doesn't cause too many problems ;-)
*/
SaAisErrorT saClmClusterNodeGetAsync(SaClmHandleT clmHandle, SaInvocationT invocation, SaClmNodeIdT nodeId)
{
	struct sa_handle *sah = (struct sa_handle *)(intptr_t)clmHandle;
	cman_node_t cman_node;
	SaClmClusterNodeT clusterNode;
	int ret;

	VALIDATE_HANDLE(sah);

	if (!sah->sa_callbacks.saClmClusterNodeGetCallback)
		return SA_AIS_ERR_INVALID_PARAM;

	if (nodeId == SA_CLM_LOCAL_NODE_ID)
		nodeId = CMAN_NODEID_US;

	ret = cman_get_node(sah->sa_cman_handle, nodeId, &cman_node);
	if (ret)
	{
		if (errno == ENOENT)
			return SA_AIS_ERR_INVALID_PARAM;
		else
			return SA_AIS_ERR_LIBRARY;
	}
	cman_to_sanode(&cman_node, &clusterNode);

	sah->sa_callbacks.saClmClusterNodeGetCallback(invocation, &clusterNode, SA_AIS_OK);

	return SA_AIS_OK;
}
