/*
 * QNixStyle (C) 2001 Derek Greene     <del@chek.com>
 * Based on code from HCStyle & MegaGradient Style :
 *           (C) 2001 Martijn Klingens <mklingens@yahoo.com>
 *  			   (C) 2000 Daniel M. Duley  <mosfet@kde.org>
 *           (C) 2000 Dirk Mueller     <mueller@kde.org>
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*
 * CHANGES:
 *
 * 0.2.1
 *     - Patch by Ueki Masaru <penetre@yahoo.co.jp> fixes large popupmenu problem
 *     - Changed Kicker panel appearance
 *
 * 0.2
 *     - Fix for radio buttons not appearing in Konq/KHTML
 *     - Themed popup menu panels & popup menu items
 *     - Improved visibility of selected task on kicker taskbar
 *     - Changed appearance of tabs to blend in with tab panel
*/

#ifndef INCLUDE_MENUITEM_DEF
#define INCLUDE_MENUITEM_DEF
#endif

#include <qmenudata.h>
#include <kapp.h>
#include <kglobal.h>
#include <kconfig.h>
#include <kdrawutil.h>
#include <kglobalsettings.h>
#include <kpixmapeffect.h>
#include <qpalette.h>
#include <qbitmap.h>
#include <qtabbar.h>
#include <qpopupmenu.h>
#include <qobjectlist.h>
#include <qimage.h>
#include <qtimer.h>
#include <kimageeffect.h>
#include <ktoolbar.h>
#include <qdrawutil.h>
#include <unistd.h>
#include <klocale.h>
#include <kiconloader.h>

#include "qnix_style.h"

#define QCOORDARRLEN(x) sizeof(x)/(sizeof(QCOORD)*2)

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CLASS : QNixStyle
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Constructor
// --------------------------------------------------------------
QNixStyle::QNixStyle()
    : KStyle()
{
	setButtonDefaultIndicatorWidth(0);
	gDict.setAutoDelete(true);
	highcolor = QPixmap::defaultDepth() > 8;

	setScrollBarExtent(15, 15);
}

// ============================================================================================================
// WIDGETS - BUTTONS
// ============================================================================================================

// --------------------------------------------------------------
// Draw plain button
// --------------------------------------------------------------
void QNixStyle::drawButton(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, bool sunken, const QBrush *)
{
	kDrawBeButton(p, x, y, w, h, g, sunken, &g.brush(QColorGroup::Midlight));
}

// --------------------------------------------------------------
// Draw push button
// --------------------------------------------------------------
void QNixStyle::drawPushButton(QPushButton *btn, QPainter *p)
{
	QRect r = btn->rect();
	bool sunken = btn->isOn() || btn->isDown();
	QColorGroup g = btn->colorGroup();
	int x = r.x(), y = r.y(), w = r.width(), h = r.height();

	int x2 = x+w-1;
	int y2 = y+h-1;

	if(highcolor)
	{
		// frame - outermost
		p->setPen(g.mid().light(110));
		p->drawLine(x, y, x2, y);
		p->drawLine(x, y, x, y2-1);
		p->setPen(g.mid().light(140));
		p->drawLine(x, y2, x2, y2);
		p->drawLine(x2, y-1, x2, y2);
		// 1st inner frame
		p->setPen(g.dark().dark(143));
		p->drawRect( x+1, y+1, w-2, h-2 );
		// 2nd inner frame
		p->setPen(sunken ? g.mid().light(115) : g.light().light(105));
		p->drawLine(x+2, y+2, x2-2, y+2);
		p->drawLine(x+2, y+2, x+2, y2-2);
		p->setPen(sunken ? g.light().light(105) : g.mid().light(115) );
		p->drawLine(x+2, y2-2, x2-2, y2-2);
		p->drawLine(x2-2, y+2, x2-2, y2-2);
		// add the gradient - reverse if sunken
		drawVGradient(p, g, QColorGroup::Button, x+3, y+3, w-6, h-6, 0, 0, w-6, h-6, sunken );
	}
	else
	{
		// just plain
		if( sunken)
			kDrawBeButton(p, x, y, w, h, g, true, &g.brush(QColorGroup::Mid));
		else
			drawButton(p, x, y, w, h, g, false);
	}
	// Default indicator
	if(btn->isDefault())
	{
		p->setPen(g.dark().dark(145));
		p->drawRect( r);
	}
}

// --------------------------------------------------------------
// Draw bevel button (e.g. used in listview header)
// --------------------------------------------------------------
void QNixStyle::drawBevelButton(QPainter *p, int x, int y, int w, int h,  const QColorGroup &g, bool sunken,  const QBrush *fill)
{
	int x2 = x+w-1;
	int y2 = y+h-1;

	// frame - outermost
	p->setPen(g.mid().light(140));
	p->drawLine(x, y2, x2, y2);
	// 1st inner frame - one side only
	p->setPen(g.dark().dark(143));
	p->drawLine(x, y2-1, x2, y2-1);
	p->drawLine(x2, y, x2, y2-1);
	// 2nd inner frame
	p->setPen(sunken ? g.mid().light(115) : g.light().light(105));
	p->drawLine(x, y, x2-1, y);
	p->drawLine(x, y, x, y2-2);
	p->setPen(sunken ? g.light().light(105) : g.mid().light(115) );
	p->drawLine(x, y2-2, x2-1, y2-2);
	p->drawLine(x2-1, y, x2-1, y2-2);

	if(highcolor)
			drawVGradient(p, g, QColorGroup::Button, x+1, y+1, w-4, h-5, 0, 0, w-4, h-5, true);	// reverse gradient
	else
			p->fillRect(x+1, y+1, w-4, h-5, fill ? *fill : g.brush(QColorGroup::Button));

	// last colour determines header text colour
	p->setPen(Qt::black);
}

// --------------------------------------------------------------
// Draw label on a pushbutton
// --------------------------------------------------------------
void QNixStyle::drawPushButtonLabel(QPushButton *btn, QPainter *p)
{
	int x1, y1, x2, y2, w, h;
	btn->rect().coords(&x1, &y1, &x2, &y2);
	w = btn->width();
	h = btn->height();

	bool act = btn->isOn() || btn->isDown();
	if(act){
			++x1, ++y1;
	}

	// If this is a button with an associated popup menu, draw an arrow first
	if ( btn->popup() )
	{
		int dx = menuButtonIndicatorWidth( btn->height() );

		QColorGroup g( btn->colorGroup() );
		int xx = x1 + w - dx - 4;
		int yy = y1 - 3;
		int hh = h + 6;

		if ( !act )
		{
			p->setPen( g.light() );
			p->drawLine( xx + 1, yy + 5, xx + 1, yy + hh - 6 );
			p->setPen( g.mid() );
			p->drawLine( xx, yy + 6, xx, yy + hh - 6 );
		}
		else
		{
			p->setPen( g.button() );
			p->drawLine( xx, yy + 4, xx, yy + hh - 4 );
		}
		drawArrow( p, DownArrow, FALSE,	x1 + w - dx - 2, y1 + 2, dx, h - 4,	btn->colorGroup(),	btn->isEnabled() );
		w -= dx;
	}

	// Next, draw iconset, if any
  if ( btn->iconSet() && !btn->iconSet()->isNull() )
  {
		QIconSet::Mode mode = btn->isEnabled()  ? QIconSet::Normal : QIconSet::Disabled;
		if ( mode == QIconSet::Normal && btn->hasFocus() )
	    mode = QIconSet::Active;
		QPixmap pixmap = btn->iconSet()->pixmap( QIconSet::Small, mode );
		int pixw = pixmap.width();
		int pixh = pixmap.height();

		p->drawPixmap( x1+6, y1+h/2-pixh/2, pixmap );
		x1 += pixw + 8;
		w -= pixw + 8;
  }

  if(act || btn->isDefault())
	{
  	QFont font = btn->font();
    font.setBold(true);
    p->setFont(font);
    drawItem( p, x1+1, y1+1, w, h, AlignCenter | ShowPrefix, btn->colorGroup(), btn->isEnabled(), btn->pixmap(), btn->text(), -1, act ? &btn->colorGroup().dark() : &btn->colorGroup().mid());
    drawItem( p, x1, y1, w, h, AlignCenter | ShowPrefix, btn->colorGroup(), btn->isEnabled(),  btn->pixmap(), btn->text(), -1, act ? &btn->colorGroup().light() : &btn->colorGroup().text());
	}
  else
	{
  	drawItem( p, x1 + ( act ? 1 : 0 ), y1 + ( act ? 1 : 0 ), w, h, AlignCenter | ShowPrefix, btn->colorGroup(), btn->isEnabled(),  btn->pixmap(), btn->text(), -1, act ? &btn->colorGroup().light()
		      : &btn->colorGroup().buttonText());
	}
}

void QNixStyle::drawButtonMask(QPainter *p, int x, int y, int w, int h)
{
	int x2 = x+w-1;
	int y2 = y+h-1;
	p->fillRect(x, y, w, h, QBrush(color1, SolidPattern));
	p->setPen(color0);
	p->drawPoint(x, y);
	p->drawPoint(x2, y);
	p->drawPoint(x, y2);
	p->drawPoint(x2, y2);
}

// --------------------------------------------------------------
// Return a QRect representing position/dimensions of button
// --------------------------------------------------------------
QRect QNixStyle::buttonRect(int x, int y, int w, int h)
{
	return(QRect(x+4, y+4, w-6, h-6));
}


// ============================================================================================================
// WIDGETS - COMBO BOXES
// ============================================================================================================

// --------------------------------------------------------------
// Draw combo button
// --------------------------------------------------------------
void QNixStyle::drawComboButton(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, bool sunken, bool edit, bool, const QBrush *)
{
  int x2 = x+w-1;
  int y2 = y+h-1;

	if(highcolor)
	{
		// frame - outermost
		p->setPen(g.mid().light(110));
		p->drawLine(x, y, x2, y);
		p->drawLine(x, y, x, y2-1);
		p->setPen(g.mid().light(140));
		p->drawLine(x, y2, x2, y2);
		p->drawLine(x2, y-1, x2, y2);
		// 1st inner frame
		p->setPen(g.dark().dark(143));
		p->drawRect( x+1, y+1, w-2, h-2 );
		// 2nd inner frame
		p->setPen(sunken ? g.mid().light(115) : g.light().light(105));
		p->drawLine(x+2, y+2, x2-2, y+2);
		p->drawLine(x+2, y+2, x+2, y2-2);
		p->setPen(sunken ? g.light().light(105) : g.mid().light(115) );
		p->drawLine(x+2, y2-2, x2-2, y2-2);
		p->drawLine(x2-2, y+2, x2-2, y2-2);
		// add the gradient - reverse if sunken
		drawVGradient(p, g, QColorGroup::Button, x+3, y+3, w-6, h-6, 0, 0, w-6, h-6, sunken );
	}
	else
	{
		// just plain
		if( sunken)
			kDrawBeButton(p, x, y, w, h, g, true, &g.brush(QColorGroup::Mid));
		else
			drawButton(p, x, y, w, h, g, false);
	}

	// Now draw the arrow
	int arrow_h = h / 3;
	int arrow_w = arrow_h;
	int arrow_x = w - arrow_w - 6;
	int arrow_y = (h - arrow_h) / 2;
	drawArrow(p, DownArrow, false, arrow_x, arrow_y, arrow_w, arrow_h, g, true);

	// draw extra lines
	int lineX = x + arrow_x - 4;
	if( lineX > 5 )
	{
		p->setPen(g.mid().dark(108));
		p->drawLine(lineX, y+2, lineX, y2-3);
		p->setPen(g.mid().light(115));
		p->drawLine(lineX+1, y+2, lineX+1, y2-3);
	}

}

// --------------------------------------------------------------
// Mask for combo button
// --------------------------------------------------------------
void QNixStyle::drawComboButtonMask(QPainter *p, int x, int y, int w, int h)
{
    int x2 = x+w-1;
    int y2 = y+h-1;
    p->fillRect(x, y, w, h, QBrush(color1, SolidPattern));
    p->setPen(color0);
    p->drawPoint(x, y);
    p->drawPoint(x, y+1);
    p->drawPoint(x+1, y);
    p->drawPoint(x2, y);
    p->drawPoint(x, y2);
    p->drawPoint(x2, y2);
}

// --------------------------------------------------------------
// Return a QRect representing position/dimensions of combo button
// --------------------------------------------------------------
QRect QNixStyle::comboButtonRect(int x, int y, int w, int h)
{
    return(QRect(x+3, y+3, w - (h / 3) - 13, h-6));
}

// --------------------------------------------------------------
// Return a QRect representing position/dimensions of focus
// rectangle for combo button
// --------------------------------------------------------------
QRect QNixStyle::comboButtonFocusRect(int x, int y, int w, int h)
{
    return(QRect(x+3, y+3, w-(h/3)-13, h-6));
}

// ============================================================================================================
// WIDGETS - SPLITTERS
// ============================================================================================================

// --------------------------------------------------------------
// Draw splitter control
// --------------------------------------------------------------
void QNixStyle::drawSplitter(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, Orientation orient)
{
	// Adjust so no "double" borders are shown...
	if(orient == Vertical)
	{
		x--; w+=2;
	}
	else
	{
		y--; h+=2;
	}

	int x2 = x+w-1;
  int y2 = y+h-1;
  p->setPen(g.dark());
	p->drawRect(x, y, w, h);

	p->setPen( g.mid().light(140) );
	p->drawLine(x+1, y+1, x+1, y2-1);
	p->drawLine(x+1, y+1, x2-1, y+1);
	p->setPen( g.mid().light(123) );
	p->drawLine(x+2, y+2, x+2, y2-2);
	p->drawLine(x+2, y+2, x2-2, y+2);
	p->setPen(g.mid());
	p->drawLine(x2-1, y+1, x2-1, y2-1);
	p->drawLine(x+1, y2-1, x2-1, y2-1);
	p->fillRect(x+3, y+3, w-5, h-5, g.brush(QColorGroup::Background));

	p->setPen(g.background());
}

// ============================================================================================================
// WIDGETS - MENUS
// ============================================================================================================

// --------------------------------------------------------------
// Draw menubar
// --------------------------------------------------------------
void QNixStyle::drawKMenuBar(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, bool mac, QBrush *)
{
	if(mac)
	{
		p->setPen(g.dark());
		p->drawRect(x, y, w, h);
		++x, ++y, w-=2, h-=2;
		int x2 = x+w-1;
		int y2 = y+h-1;
		p->setPen(g.light());
		p->drawLine(x, y, x2, y);
		p->drawLine(x, y, x, y2);
		p->setPen(g.mid());
		p->drawLine(x2, y, x2, y2);
		p->drawLine(x, y2, x2, y2);
		if(highcolor)
				drawVGradient(p, g, QColorGroup::Background, x+1, y+1, w-2, h-2, 0, 0, w-2, h-2, true);
		else
				p->fillRect(x+1, y+1, w-2, h-2, g.brush(QColorGroup::Midlight));
	}
	else
	{
		if(highcolor)
		{
			qDrawShadePanel(p, x, y, w, h, g, false, 1);
			drawVGradient(p, g, QColorGroup::Background, x+1, y+1, w-2, h-2, 0, 0, w-2, h-2, true);
		}
		else
			p->fillRect(x+1, y+1, w-2, h-2, g.brush(QColorGroup::Midlight));
	}

	// extra line at the bottom
	p->setPen(g.mid());
	p->drawLine(x+1, y+h-2, x+w, y+h-2);
}

// --------------------------------------------------------------
// Draw a menubar menuitem
// --------------------------------------------------------------
void QNixStyle::drawKMenuItem(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, bool active, QMenuItem *mi, QBrush *)
{
	// Same as hcstyle
	if ( p->font() == KGlobalSettings::generalFont() )
		p->setFont( KGlobalSettings::menuFont() );

	// If menubar item is currently selected...
	if(active)
	{
		qDrawShadePanel(p, x+1, y+1, w-1, h-1, g, true, 1,  &g.brush(QColorGroup::Midlight));
		QApplication::style().drawItem(p, x, y, w, h, AlignCenter|ShowPrefix|DontClip|SingleLine,
																	g, mi->isEnabled(), mi->pixmap(), mi->text(),
																	-1, &Qt::black);

	}
	// Otherwise just draw it normally
	else
	{
		QApplication::style().drawItem(p, x, y, w, h,
																	AlignCenter|ShowPrefix|DontClip|SingleLine,
																	g, mi->isEnabled(), mi->pixmap(), mi->text(),
																	-1, &g.text() );
	}
}

// --------------------------------------------------------------
// [0.2] - Draw a popup menu panel
// --------------------------------------------------------------
void QNixStyle::drawPopupPanel(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, int lineWidth, const QBrush *fill)
{
	int x2 = x+w-1;
	int y2 = y+h-2;

	p->fillRect(x, y, w, h, g.brush(QColorGroup::Background) );

	// Adjust for shadow
	h--; 	w--; x2--;

	p->setPen(g.dark().dark(143));
	p->drawRect( x, y, w, h );		// adjust for menu bar?
	p->setPen( g.light().light(105));
	p->drawLine( x+1, y+2, x2-1, y+2);
	p->drawLine( x+1, y+2, x+1, y2-1);
	// shadow
	p->setPen( g.dark().dark(115) );
	p->drawLine( x, y2+1, x2+1, y2+1);	// horiz
	p->drawLine( x2+1, y, x2+1, y2+1);  // vert
}

// --------------------------------------------------------------
// Draw popup/submenu item
// --------------------------------------------------------------
void QNixStyle::drawPopupMenuItem( QPainter* p, bool checkable, int maxpmw, int tab, QMenuItem* mi,  const QPalette& pal, bool active,
	bool enabled, int x, int y, int w, int h)
{
	static const int motifItemFrame         = 2;
	static const int motifItemHMargin       = 3;
	static const int motifItemVMargin       = 2;
	static const int motifArrowHMargin      = 6;
	static const int windowsRightBorder     = 12;

	// Is this menuitem disabled?
	bool   dis = !enabled;
	QColor discol;
	int    checkcol;

	if (checkable)
		maxpmw = QMAX( maxpmw, 20 );
	checkcol = maxpmw;

	if ( p->font() == KGlobalSettings::generalFont() )
		p->setFont( KGlobalSettings::menuFont() );

	//QColorGroup g = dis ? pal.disabled() : pal.active();
	QColorGroup g =  pal.active();

	// [0.2] - If the menu item is current selected, use the "Selected Background" colour as its background
	if( active )
		p->fillRect(x, y, w-1, h, g.brush(QColorGroup::Highlight) );
	// Not selected => use plain background
	else
		p->fillRect(x, y, w-1, h, pal.active().background());

	g = dis ? pal.disabled() : pal.active();

	// [0.2.1] - Patch contributed by UEKI Masaru
	if (!mi) return;


	// [0.2] - Draw a separator
  if(mi->isSeparator())
	{
		p->setPen(g.mid().dark(133));
		p->drawLine(x, y, x+w, y );
		p->setPen( g.mid().light(130) );
    p->drawLine(x, y+1, x+w, y+1 );
    return;	// finished now
  }
	// Has an icon??
	if ( mi->iconSet() )
	{
		QIconSet::Mode mode = dis? QIconSet::Disabled : QIconSet::Normal;
		if (!dis)
				mode = QIconSet::Active;
		QPixmap pixmap = mi->iconSet()->pixmap(QIconSet::Small, mode);
		int pixw = pixmap.width();
		int pixh = pixmap.height();
		QRect cr(x, y, checkcol, h);
		QRect pmr(0, 0, pixw, pixh);
		pmr.moveCenter( cr.center() );
		p->setPen(g.highlightedText());
		p->drawPixmap(pmr.topLeft(), pixmap );
	}
	// Checkbox
	else if(checkable)
	{
		int mw = checkcol + motifItemFrame;
		int mh = h - 2*motifItemFrame;
		if (mi->isChecked())
			drawCheckMark( p, x + motifItemFrame,	y+motifItemFrame, mw, mh, g, active, dis );
	}

	// [0.2] - Text handling
	p->setPen( active ? g.highlightedText() : g.buttonText() );
	if ( dis )
	{
		discol = g.text();
		p->setPen( discol );
	}
	int xm = motifItemFrame + checkcol + motifItemHMargin;
	QString s = mi->text();
  if ( !s.isNull() )
	{
		int t = s.find( '\t' );
		int m = motifItemVMargin;
		const int text_flags = AlignVCenter|ShowPrefix | DontClip | SingleLine;
		if ( t >= 0 )
		{				// draw tab text
	  	if ( dis && !active )
			{
				p->setPen( g.light() );
				p->drawText( x+w-tab-windowsRightBorder-motifItemHMargin-motifItemFrame+1,  y+m+1, tab, h-2*m, text_flags, s.mid( t+1 ));
				p->setPen( discol );
	    }
	    p->drawText( x+w-tab-windowsRightBorder-motifItemHMargin-motifItemFrame, y+m, tab, h-2*m, text_flags, s.mid( t+1 ) );
		}
		if ( dis && !active )
		{
	    p->setPen( g.light() );
	    p->drawText( x+xm+1, y+m+1, w-xm+1, h-2*m, text_flags, s, t );
	    p->setPen( discol );
		}
		p->drawText( x+xm, y+m, w-xm-tab+1, h-2*m, text_flags, s, t );
	}
	// Draw pixmap
	else if (mi->pixmap())
	{
		QPixmap *pixmap = mi->pixmap();
		if (pixmap->depth() == 1)
			p->setBackgroundMode(OpaqueMode);
		p->drawPixmap( x+xm, y+motifItemFrame, *pixmap);
		if (pixmap->depth() == 1)
			p->setBackgroundMode(TransparentMode);
	}
	// draw sub menu arrow
	if ( mi->popup() )
	{
		int dim = (h-2*motifItemFrame) / 2;
		// [0.2] - If highlighted, then set arrow colour to inverse
		if ( active )
		{
	    if ( !dis )
				discol = white;
			QColorGroup g2( discol, g.highlight(), white, white,  dis ? discol : white,  discol, white );
	    drawArrow( p, RightArrow, FALSE,  x+w - motifArrowHMargin - motifItemFrame - dim,  y+h/2-dim/2, dim, dim, g2, TRUE );
		}
		// Inactive => Use standard arrow colour
		else
		{
	    drawArrow( p, RightArrow, FALSE, x+w - motifArrowHMargin - motifItemFrame - dim,  y+h/2-dim/2,  dim, dim, g, mi->isEnabled() );
		}
	}

}

int QNixStyle::popupMenuItemHeight(bool checkable, QMenuItem *mi, const QFontMetrics &fm)
{
	if (mi->isSeparator())
			return 2;

	int h = 0;
	if (mi->pixmap())
			h = mi->pixmap()->height();

	if (mi->iconSet())
			h = QMAX(mi->iconSet()->
							pixmap(QIconSet::Small, QIconSet::Normal).height(), h);

	h = QMAX(fm.height() + 4, h);

	// we want a minimum size of 18
	h = QMAX(h, 18);

	return h;
}

void QNixStyle::polishPopupMenu(QPopupMenu *mnu)
{
  KStyle::polishPopupMenu(mnu);
}

// ============================================================================================================
// WIDGETS - TOOLBARS
// ============================================================================================================

// Draw toolbar
// --------------------------------------------------------------
void QNixStyle::drawKToolBar(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, KToolBarPos, QBrush *)
{
    if(highcolor)
		{
    	qDrawShadePanel(p, x, y, w, h, g, false, 1);
      if(w > h)
      	drawVGradient(p, g, QColorGroup::Background, x+1, y+1, w-2, h-2, 0, 0, w-2, h-2, true);
			else
      	drawHGradient(p, g, QColorGroup::Background, x+1, y+1, w-2, h-2, 0, 0, w-2, h-2, true);
    }
    else
        qDrawShadePanel(p, x, y, w, h, g, false, 1, &g.brush(QColorGroup::Background));

		// extra line at the bottom
		p->setPen(g.mid());
		p->drawLine(x, y+h-2, x+w, y+h-2);
}

// Draw toolbar button
// --------------------------------------------------------------
void QNixStyle::drawKToolBarButton(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, bool sunken,
			bool raised, bool enabled, bool popup, KToolButtonType icontext, const QString& btext, const QPixmap *pixmap, QFont *font, QWidget *btn)
{
    int dx, dy;

    QFontMetrics fm(*font);

    QToolBar* toolbar = 0;
    if(btn->parent() && btn->parent()->isWidgetType() && btn->parent()->inherits("QToolBar"))
        toolbar = static_cast<QToolBar*>(btn->parent());

    if(!highcolor || !toolbar)
        p->fillRect(x, y, w, h, g.brush(QColorGroup::Background));
    else
    {
        // sync with drawKToolBar
        if(toolbar->orientation() == Qt::Horizontal)
            // should be background fill?
            drawVGradient(p, g, QColorGroup::Background, x, y, w, h, x, y, toolbar->width()-2, toolbar->height()-2, true);
        else
            drawHGradient(p, g, QColorGroup::Background, x, y, w, h, x, y, toolbar->width()-2, toolbar->height()-2, true);
    }

    if(raised || sunken) {
        int x2 = x+w;
        int y2 = y+h;
        p->setPen(g.dark());
        p->drawLine(x+1, y+1, x2-2, y+1);
        p->drawLine(x, y+2, x, y2-3);
        p->drawLine(x2-1, y+2, x2-1, y2-3);
        p->drawLine(x+1, y2-2, x2-2, y2-2);

        p->setPen(sunken ? g.mid() : g.light());
        p->drawLine(x+1, y+2, x2-2, y+2);
        p->drawLine(x+1, y+2, x+1, y2-3);
        p->setPen(sunken ? g.light() : g.mid());
        p->drawLine(x2-2, y+3, x2-2, y2-3);
        p->drawLine(x+2, y2-3, x2-2, y2-3);
    }
    p->setPen(g.text());

    if (icontext == Icon){ // icon only
        if (pixmap){
            dx = ( w - pixmap->width() ) / 2;
            dy = ( h - pixmap->height() ) / 2;
            if ( sunken )
            {
                ++dx;
                ++dy;
            }
            p->drawPixmap( x+dx, y+dy, *pixmap );
        }
    }
    else if (icontext == IconTextRight){ // icon and text (if any)
        if (pixmap){
            dx = 4;
            dy = ( h - pixmap->height() ) / 2;
            if ( sunken ){
                ++dx;
                ++dy;
            }
            p->drawPixmap( x+dx, y+dy, *pixmap );
        }
        if (!btext.isNull()){
            int tf = AlignVCenter|AlignLeft;
            if (pixmap)
                dx= 4 + pixmap->width() + 2;
            else
                dx= 4;
            dy = 0;
            if ( sunken ){
                ++dx;
                ++dy;
            }
            if (font)
                p->setFont(*font);
            if(raised)
                p->setPen(KGlobalSettings::toolBarHighlightColor());
            p->drawText(x+dx, y+dy, w-dx, h, tf, btext);
        }
    }
    else if (icontext == Text)
		{ // only text, even if there is a icon
        if (!btext.isNull()){
            int tf = AlignVCenter|AlignLeft;
            if (!enabled)
                p->setPen(g.dark());
            dx= (w - fm.width(btext)) / 2;
            dy= (h - fm.lineSpacing()) / 2;
            if ( sunken ){
                ++dx;
                ++dy;
            }
            if (font)
                p->setFont(*font);
            if(raised)
                p->setPen(KGlobalSettings::toolBarHighlightColor());
            p->drawText(x+dx, y+dy, fm.width(btext), fm.lineSpacing(), tf, btext);
        }
    }
    else if (icontext == IconTextBottom)
		{
        if (pixmap)
				{
            dx = (w - pixmap->width()) / 2;
            dy = (h - fm.lineSpacing() - pixmap->height()) / 2;
            if ( sunken ){
                ++dx;
                ++dy;
            }
            p->drawPixmap( x+dx, y+dy, *pixmap );
        }
        if (!btext.isNull()){
            int tf = AlignBottom|AlignHCenter;
            dy= pixmap->height();
            dx = 2;
            if ( sunken ){
                ++dx;
                ++dy;
            }
            if (font)
                p->setFont(*font);
            if(raised)
                p->setPen(KGlobalSettings::toolBarHighlightColor());
            p->drawText(x, y, w, h-3, tf, btext);
        }
    }
    if (popup){
        if (enabled)
            qDrawArrow (p, Qt::DownArrow, Qt::WindowsStyle, false, w-5, h-5, 0, 0, g, true);
        else
            qDrawArrow (p, Qt::DownArrow, Qt::WindowsStyle, false, w-5, h-5, 0, 0, g, false);
    }
}

// Draw toolbar handle
// --------------------------------------------------------------
void QNixStyle::drawKBarHandle(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, KToolBarPos,  QBrush *)
{
	// vertical   |||
	if(h > w)
	{
		if(highcolor)
				drawVGradient(p, g, QColorGroup::Background, x, y, w, h, 0, 0, w, h, true);
		else
				p->fillRect(x, y, w, h, g.brush(QColorGroup::Background));
		int x2 = x+w-1;
		int y2 = y+h-2;

		p->setPen(g.light());
		p->drawLine(x, y, x, y2-1);
		p->drawPoint( x+1, y );
		p->drawLine(x+3, y, x+3, y2);
		p->drawPoint( x+4, y );
		p->drawLine(x+6, y, x+6, y2);
		p->drawPoint( x+7, y );
		p->drawLine(x+9, y, x+9, y2);

		p->setPen(g.mid());
		p->drawPoint( x+1, y2 );
		p->drawPoint( x+2, y2 );
		p->drawLine(x+2, y, x+2, y2);
		p->drawPoint( x+4, y2 );
		p->drawLine(x+5, y, x+5, y2);
		p->drawPoint( x+7, y2 );
		p->drawLine(x+8, y, x+8, y2);

		p->setPen(g.dark());
		p->drawLine(x, y2+1, x2, y2+1);

	}
	// horizontal  ----
	else
	{
		if(highcolor)
				drawVGradient(p, g, QColorGroup::Background, x, y, w, h, 0, 0, w, h, true);
		else
				p->fillRect(x, y, w, h, g.brush(QColorGroup::Background));
		int x2 = x+w-2;
		int y2 = y+h-1;

		p->setPen(g.light());
		p->drawLine(x, y, x2-1, y);
		p->drawPoint( x, y+1 );
		p->drawLine(x, y+3, x2, y+3);
		p->drawPoint( x, y+4 );
		p->drawLine(x, y+6, x2, y+6);
		p->drawPoint( x, y+7 );
		p->drawLine(x, y+9, x2, y+9);

		p->setPen(g.mid());
		p->drawPoint( x, y+1 );
		p->drawPoint( x, y+2 );
		p->drawLine(x, y+2, x2, y+2);
		p->drawPoint( x, y+4 );
		p->drawLine(x, y+5, x2, y+5);
		p->drawPoint( x, y+7 );
		p->drawLine(x, y+8, x2, y+8);

		p->setPen(g.dark());
		p->drawLine(x2+1, y, x2+1, y2);

	}
}

// ============================================================================================================
// WIDGETS - SCROLLBARS
// ============================================================================================================

// Draw scrollbars
// --------------------------------------------------------------
void QNixStyle::drawScrollBarControls(QPainter *p, const QScrollBar *sb, int sliderStart, uint controls, uint activeControl)
{
	int sliderMin, sliderMax, sliderLength, buttonDim;
	scrollBarMetrics( sb, sliderMin, sliderMax, sliderLength, buttonDim );

	if (sliderStart > sliderMax)
		sliderStart = sliderMax;

	bool	horiz = sb->orientation() == QScrollBar::Horizontal;

	QColorGroup	g = sb->colorGroup();
	QRect	addB, subHC, subB;
	QRect       addPageR, subPageR, sliderR;
	int		addX, addY, subX, subY;
	int		len, extent;
	uint	numButtons = 3;

	if ( horiz )
	{
		len = sb->width();
		extent = sb->height();
		subY = addY = ( extent - buttonDim ) / 2;
		subX = 0;
		addX = len - buttonDim;
	}
	else
	{
		len = sb->height();
		extent = sb->width();
		subX = addX = ( extent - buttonDim ) / 2;
		subY = 0;
		addY = len - buttonDim;
	}

	// Calc number of used buttons:
	if( len < buttonDim * 4 )
		numButtons = 2;

	subB.setRect( subX, subY, buttonDim, buttonDim );
	addB.setRect( addX, addY, buttonDim, buttonDim );
	if( horiz )
		subHC.setRect( addX - buttonDim, addY, buttonDim, buttonDim );
	else
		subHC.setRect( addX, addY - buttonDim, buttonDim, buttonDim );

	int sliderEnd = sliderStart + sliderLength;
	int sliderW = extent;

	// Calculate sizes of the actual scrollbar parts
	if ( horiz )
	{
		subPageR.setRect( subB.right() + 1, 0, sliderStart - subB.right() - 1 , sliderW );
		addPageR.setRect( sliderEnd, 0, addX - sliderEnd -
		( ( numButtons == 3 ) ? buttonDim : 0 ), sliderW );
		sliderR .setRect( sliderStart, 0, sliderLength, sliderW );
	}
	else
	{
		subPageR.setRect( 0, subB.bottom() + 1, sliderW, sliderStart - subB.bottom() - 1 );
		addPageR.setRect( 0, sliderEnd, sliderW, addY - sliderEnd -  ( ( numButtons == 3 ) ? buttonDim : 0 ) );
		sliderR .setRect( 0, sliderStart, sliderW, sliderLength );
	}

	bool maxed = sb->maxValue() == sb->minValue();

	if ( controls & AddLine )
	{
		// Draw scroll down arrow
		drawSBButton(p, addB, g, activeControl == AddLine, horiz);
		drawArrow( p, horiz ? RightArrow : DownArrow, false, addB.x()+4, addB.y()+4,addB.width()-8, addB.height()-8, g, !maxed);
	}

	if ( controls & SubLine )
	{
		// Draw scroll up buttons
		bool isSubLine = activeControl == SubLine;
		Qt::ArrowType arrowType = horiz ? LeftArrow : UpArrow;

		drawSBButton(p, subB, g, isSubLine, horiz);
		drawArrow( p, arrowType, false, subB.x() + 4, subB.y() + 4, subB.width() - 8, subB.height() - 8, g, !maxed );

		if( numButtons == 3 )
		{
			drawSBButton( p, subHC, g, isSubLine, horiz );
			drawArrow( p, arrowType, false, subHC.x() + 4, subHC.y() + 4,
			subHC.width() - 8, subHC.height() - 8, g, !maxed );
		}
	}

	QColor bgCol( g.mid() );

	// Scrollbar background above slider
	if(controls & SubPage)
	{
		if(subPageR.height())
		{
			// Draw vertical scrollbar part below the slider
			if(horiz)
			{
				// Draw vertical scrollbar part above the slider
				p->fillRect(subPageR, bgCol);
				p->setPen(g.dark());
				p->drawLine(subPageR.x(), subPageR.y(), subPageR.right(),	subPageR.y());
				p->drawLine(subPageR.x(), subPageR.bottom(), subPageR.right(),subPageR.bottom());
			}
			else
			{
				p->fillRect(subPageR, bgCol);
				p->setPen(g.dark());
				p->drawLine(subPageR.x(), subPageR.y(), subPageR.x(),	subPageR.bottom());
				p->drawLine(subPageR.right(), subPageR.y(), subPageR.right(),	subPageR.bottom());
			}
		}
	}

	// Scrollbar background below slider
	if(controls & AddPage)
	{
		if(addPageR.width())
		{
			// Draw vertical scrollbar part below the slider
			if(horiz)
			{
				p->fillRect(addPageR, bgCol);
				p->setPen(g.dark());
				p->drawLine(addPageR.x(), addPageR.y(), addPageR.right(),  addPageR.y());
				p->drawLine(addPageR.x(), addPageR.bottom(), addPageR.right(), addPageR.bottom());
			}
			else
			{
				p->fillRect(addPageR, bgCol);
				p->setPen(g.dark());
				p->drawLine(addPageR.x(), addPageR.y(), addPageR.x(),	addPageR.bottom());
				p->drawLine(addPageR.right(), addPageR.y(), addPageR.right(),	addPageR.bottom());
			}
		}
	}

	if ( controls & Slider )
  {
		// Draw slider, but only if the slider is at least 4 pixels wide
		// and 4 pixels heigh!
		if( sliderR.width() > 4 && sliderR.height() > 4 )
		{
			// note that we use reverse gradient for slider
			drawSBButton(p, sliderR, g, activeControl == Slider, !horiz, true);
			g.setColor(QColorGroup::Dark, g.mid());
		}
	}
}

// draw scroll bar button
// --------------------------------------------------------------
void QNixStyle::drawSBButton(QPainter *p, const QRect &r, const QColorGroup &g,  bool down, bool horiz, bool reverse )
{
	p->setPen(g.dark());
	p->drawRect(r);

	if(highcolor)
	{
		//if( (r.width() > r.height() ) || )
		if( horiz )
			drawHGradient(p, g, QColorGroup::Button, r.x()+1, r.y()+1,r.width()-2, r.height()-2, 0, 0, r.width()-2, r.height()-2, reverse);
		else
			drawVGradient(p, g, QColorGroup::Button, r.x()+1, r.y()+1,r.width()-2, r.height()-2, 0, 0, r.width()-2, r.height()-2, reverse);
	}
	else
		p->fillRect(r.x()+1, r.y()+1, r.width()-2, r.height()-2,	g.brush(down ? QColorGroup::Midlight : QColorGroup::Button));

	p->setPen(down ? g.light() : g.mid());
	p->drawLine(r.x()+1, r.bottom()-1, r.right()-1, r.bottom()-1);
	p->drawLine(r.right()-1, r.top()+1, r.right()-1, r.bottom()-1);
	p->setPen(down ? g.mid() : g.light());
	p->drawLine(r.x()+1, r.y()+1, r.right()-1, r.y()+1);
	p->drawLine(r.x()+1, r.y()+1, r.x()+1, r.bottom()-1);
}

// Draw decoration on the scrollbar slider
// In this theme there is no decoration
void QNixStyle::drawSBDeco(QPainter *p, const QRect &r, const QColorGroup &g, bool horiz)
{
	// do nothing
}

void QNixStyle::scrollBarMetrics(const QScrollBar *sb, int &sliderMin,  int &sliderMax, int &sliderLength, int &buttonDim)
{
	int maxLength;
  bool horiz = sb->orientation() == QScrollBar::Horizontal;
  int length = horiz ? sb->width()  : sb->height();
  int extent = horiz ? sb->height() : sb->width();

  if( length > ( extent - 1 ) * 2 )
		buttonDim = extent;
  else
		buttonDim = length / 2 - 1;

	sliderMin = buttonDim;

	uint numButtons = 3;   // Two scroll up/left buttons when possible...
  if( length < buttonDim * 4 )	// ... but otherwise only two!
		numButtons = 2;
	maxLength = length - buttonDim * numButtons;

  if ( sb->maxValue() == sb->minValue() )
	{
		sliderLength = maxLength;
  }
	else
	{
		sliderLength = (sb->pageStep()*maxLength)/(sb->maxValue()-sb->minValue()+sb->pageStep());
		uint range = sb->maxValue()-sb->minValue();
		if ( sliderLength < 9 || range > INT_MAX/2 )
	    sliderLength = 9;
		if ( sliderLength > maxLength )
	    sliderLength = maxLength;
  }
  sliderMax = sliderMin + maxLength - sliderLength;
}

QStyle::ScrollControl QNixStyle::scrollBarPointOver(const QScrollBar *sb, int sliderStart,  const QPoint &p)
{
    if ( !sb->rect().contains( p ) )
        return NoScroll;
    int sliderMin, sliderMax, sliderLength, buttonDim, pos;
    scrollBarMetrics( sb, sliderMin, sliderMax, sliderLength, buttonDim );

    // Calc number of used buttons:
    uint len, numButtons;
    if( sb->orientation() == QScrollBar::Horizontal )
	len    = sb->width();
    else
	len    = sb->height();
    if( len < buttonDim * 4 )
	numButtons = 2;
    else
	numButtons = 3;

    pos = (sb->orientation() == QScrollBar::Horizontal)? p.x() : p.y();
    if ( pos < sliderMin )
	return SubLine;
    if ( pos < sliderStart )
	return SubPage;
    if ( pos < sliderStart + sliderLength )
        return Slider;
    if ( pos < sliderMax + sliderLength)
        return AddPage;
    if( pos > sliderMax + sliderLength + 16 )
        return AddLine;
    if( numButtons == 2 && pos > sliderMax + sliderLength )
        return AddLine;

    return SubLine;
}

// Slider
// --------------------------------------------------------------
void QNixStyle::drawSlider(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, Orientation orient, bool, bool)
{
	int x2 = x+w-1;
  int y2 = y+h-1;

	// frame - outermost
	// 1st inner frame
	p->setPen(g.dark().dark(143));
	p->drawRect( x+1, y+1, w-2, h-2 );
	// 2nd inner frame
	p->setPen(g.light().light(105));
	p->drawLine(x+2, y+2, x2-2, y+2);
	p->drawLine(x+2, y+2, x+2, y2-2);
	p->setPen( g.mid().light(115) );
	p->drawLine(x+2, y2-2, x2-2, y2-2);
	p->drawLine(x2-2, y+2, x2-2, y2-2);
	// FILL
	if(highcolor)
		drawVGradient(p, g, QColorGroup::Button, x+3, y+3, w-6, h-6, 0, 0, w-6, h-6,false);
	else
		p->fillRect(x+3, y+3, w-6, h-6, g.background());

	// SHADOW
	p->setPen(g.dark().light(125));
	// right
	p->drawLine(x2, y+2, x2, y2);
	// bottom
	p->drawLine(x+2, y2, x2, y2);
}

void QNixStyle::drawSliderMask(QPainter *p, int x, int y, int w, int h,
                             Orientation, bool, bool)
{
    int x2 = x+w-1;
    int y2 = y+h-1;
    p->fillRect(x, y, w, h, Qt::color1);
    p->setPen(Qt::color0);
    p->drawPoint(x, y);
    p->drawPoint(x2, y);
    p->drawPoint(x, y2);
    p->drawPoint(x2, y2);
}

int QNixStyle::sliderLength() const
{
    return(18);
}

// Draw arrows
// --------------------------------------------------------------
void QNixStyle::drawArrow(QPainter *p, Qt::ArrowType type, bool on, int x,int y, int w, int h, const QColorGroup &g,
   bool enabled, const QBrush *)
{
    //static const QCOORD u_arrow[]={3,1, 4,1, 2,2, 5,2, 1,3, 6,3, 0,4, 7,4, 0,5, 7,5};
		static const QCOORD u_arrow[]={3,1, 4,1, 2,2, 5,2, 1,3, 6,3, 0,4, 7,4, 3,5, 4,5, 3,6, 4,6 };
		static const QCOORD d_arrow[]={3,1, 4,1, 3,2, 4,2, 0,3, 7,3, 1,4, 6,4, 2,5, 5,5, 3,6, 4,6 };
    //static const QCOORD d_arrow[]={0,2, 7,2, 0,3, 7,3, 1,4, 6,4, 2,5, 5,5, 3,6, 4,6};
    static const QCOORD l_arrow[]={1,3, 1,4, 2,2, 2,5, 3,1, 3,6, 4,0, 4,7, 5,3, 5,4, 6,3, 6,4 };
    static const QCOORD r_arrow[]={1,3, 1,4, 2,3, 2,4, 3,0, 3,7, 4,1, 4,6, 5,2, 5,5, 6,3, 6,4};

    p->setPen(enabled ? on ? g.light() : g.buttonText() : g.mid());
    if(w > 8){
        x = x + (w-8)/2;
        y = y + (h-8)/2;
    }

    QPointArray a;
    switch(type){
    case Qt::UpArrow:
        a.setPoints(QCOORDARRLEN(u_arrow), u_arrow);
        break;
    case Qt::DownArrow:
        a.setPoints(QCOORDARRLEN(d_arrow), d_arrow);
        break;
    case Qt::LeftArrow:
        a.setPoints(QCOORDARRLEN(l_arrow), l_arrow);
        break;
    default:
        a.setPoints(QCOORDARRLEN(r_arrow), r_arrow);
        break;
    }

    a.translate(x, y);
    p->drawLineSegments(a);
}

// ============================================================================================================
// WIDGETS - RADIO BUTTONS
// ============================================================================================================

// ----------------------------------------------------------------
// Draw radio button
// ----------------------------------------------------------------
void QNixStyle::drawExclusiveIndicator(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, bool on,bool down, bool)
{
	// last line is used for shadows
	h--;
	int y2 = y+h-1;
	int x2 = x+w-1;

	// Fill
	if(highcolor)
	{
		drawVGradient(p, g, QColorGroup::Button, x, y, w, h, 0, 0, w, h, false );
		// remove gradient which is "outside" the circle
		p->fillRect(0, 0, 3, 3, g.brush(QColorGroup::Background));		// top left
		p->fillRect(9, 0, 3, 3, g.brush(QColorGroup::Background)); 		// top right
		p->fillRect(0, 9, 3, 3, g.brush(QColorGroup::Background));		// bottom left
		p->fillRect(9, 9, 3, 3, g.brush(QColorGroup::Background)); 		// bottom right
	}
	else
	{
		// just plain fill
		p->fillRect(x, y, w, h, g.brush(QColorGroup::Background));
	}

	// longest light line
	p->setPen( g.background().dark(120) );
	p->drawLine(x,   y+3,  x+3, y);
	p->drawLine(x+8, y, x+11, y+3);
	p->drawLine(x,   y+8,  x+3, y+11);
	p->drawLine(x+8, y+11, x+11, y+8);
	// longer mid line
	p->setPen( g.mid().dark(105) );
	p->drawLine(x+1, 	y+3, x+3,   y+1);
	p->drawLine(x+1, 	y+8, x+3,  y+10);
	p->drawLine(x+8, 	y+1, x+10,  y+3);
	p->drawLine(x+8, y+10, x+10,  y+8);
	// short darker line
	p->setPen( g.dark().light(115) );
	p->drawLine(x+1, 	y+2, x+2,   y+1);
	p->drawLine(x+1, 	y+9, x+2,  y+10);
	p->drawLine(x+9, 	y+1, x+10,  y+2);
	p->drawLine(x+9, y+10, x+10,  y+9);
	// surrounding darkest
	p->setPen(g.dark().light(105));
	p->drawLine(x+4, 		y, x+7,    y);
	p->drawLine(x+4,   y2, x+7,   y2);
	p->drawLine(x,    y+4,   x,  y+7);
	p->drawLine(x+11, y+4, x+11, y+7);
	// darkest
	p->setPen(g.dark().dark(143));
	p->drawLine(x+5, 		y, x+6,    y);
	p->drawLine(x+5,   y2, x+6,   y2);
	p->drawLine(x,    y+5,   x,  y+6);
	p->drawLine(x+11, y+5, x+11, y+6);
	// extra points - darkish
	p->setPen( g.dark().light(123) );
	p->drawPoint(  3,  1 );	// top left
	p->drawPoint(  1,  3 );
	p->drawPoint(  8,  1 ); // top right
	p->drawPoint( 10,  3 );
	p->drawPoint(  3, 10 );	// bottom left
	p->drawPoint(  1,  8 );
	p->drawPoint(  8, 10 ); // bottom right
	p->drawPoint( 10,  8 );
	// extra points - light
	p->setPen( g.mid() );
	p->drawPoint(  4,  1 );	// top
	p->drawPoint(  7,  1 );
	p->drawPoint(  4, 10 );	// bottom
	p->drawPoint(  7, 10 );
	p->drawPoint(  1,  4 );	// left
	p->drawPoint(  1,  7 );
	p->drawPoint( 10,  4 );	// right
	p->drawPoint( 10,  7 );
	p->setPen( Qt::red );

	// extra shadows
	p->setPen( g.background() );
	p->drawLine(x, y+h, x2, y+h);
	p->setPen( g.background().dark(105) );
	p->drawLine(x+3, y+h, x+8, y+h);
	p->setPen( g.background().dark(110) );
	p->drawLine(x+4, y+h, x+7, y+h);

	// Is this radio button selected ?
	if( on || down )
	{
		// if so, draw centre dot...
		p->fillRect(x+4, x+4, 4, 4, g.dark().light(120));
		p->fillRect(x+5, x+5, 2, 2, g.dark().light(103)); 	// innermost
		// next darkest
		p->setPen( g.dark().light(110) );
		p->drawLine(x+4, y+5, x+4, y+6 ); // left
		p->drawLine(x+7, y+5, x+7, y+6 ); // right
		p->drawLine(x+5, y+4, x+6, y+4 ); // top
		p->drawLine(x+5, y+7, x+6, y+7 ); // bottom
		// mid
		p->setPen( g.mid().dark(125) );
		p->drawLine(x+3, y+5, x+3, y+6 ); // left
		p->drawLine(x+8, y+5, x+8, y+6 ); // right
		p->drawLine(x+5, y+3, x+6, y+3 ); // top
		p->drawLine(x+5, y+8, x+6, y+8 ); // bottom
		// mid light
		p->setPen( g.mid() );
		p->drawPoint( x+4, y+3);
		p->drawPoint( x+7, y+3);
		p->drawPoint( x+4, y+8);
		p->drawPoint( x+7, y+8);
		p->drawPoint( x+3, y+4);
		p->drawPoint( x+3, y+7);
		p->drawPoint( x+8, y+4);
		p->drawPoint( x+8, y+7);
		// light
		p->setPen( g.mid().light(115) );
		p->drawPoint( x+3, y+3);
		p->drawPoint( x+3, y+8);
		p->drawPoint( x+8, y+3);
		p->drawPoint( x+8, y+8);
		// lightest
		p->setPen( g.mid().light(130) );
		p->drawLine(x+2, y+5, x+2, y+6 );		// left
		p->drawLine(x+9, y+5, x+9, y+6 );		// right
		p->drawLine(x+5, y+2, x+6, y+2 );		// top
		p->drawLine(x+5, y+9, x+6, y+9 );		// top
	}
}

// ----------------------------------------------------------------
// Draw mask for radio button
// [0.2] Fixed to draw proper mask - radio button now shows up in konq
// ----------------------------------------------------------------
void QNixStyle::drawExclusiveIndicatorMask(QPainter *p, int x, int y, int w, int h, bool down)
{
	int y2 = y+h-1;
	int x2 = x+w-1;
	// last line is used for shadows
	h--;

 	p->fillRect(x, y, w, h, Qt::color0);

	p->setPen(Qt::color1);

	p->drawLine(x+4, y,   x2-3, y );
	p->drawLine(x+3, y+1, x2-2, y+1 );
	p->drawLine(x+2, y+2, x2-1, y+2 );
	p->fillRect(x+1, y+3, w, h-6, Qt::color1);
	p->drawLine(x+2, y2-3, x2-1, y2-3 );
	p->drawLine(x+3, y2-2, x2-2, y2-2 );
	p->drawLine(x+4, y2-1, x2-3, y2-1 );

}

// ----------------------------------------------------------------
// Return the size of a radio button in this style
// [0.2] Changed width from 12 to 13 for mask to work properly with konq
// ----------------------------------------------------------------
QSize QNixStyle::exclusiveIndicatorSize() const
{
	return(QSize(13,13));

}

// ============================================================================================================
// WIDGETS - CHECKBOXES
// ============================================================================================================

// Draw Checkbox widget
// --------------------------------------------------------------
void QNixStyle::drawIndicator(QPainter *p, int x, int y, int w, int h,  const QColorGroup &g, int state, bool down, bool)
{
    int x2 = x+w-1;
    int y2 = y+h-1;

		// common for on/off...
		// frame - outermost
		p->setPen(g.mid().light(109));
		p->drawLine(x, y, x2, y);
		p->drawLine(x, y, x, y2-1);
		p->setPen(g.mid().light(133));
		p->drawLine(x, y2, x2, y2);
		p->drawLine(x2, y-1, x2, y2);
		// 1st inner frame
		p->setPen(g.dark().dark(143));
		p->drawRect( x+1, y+1, w-2, h-2 );

    p->fillRect(x+2, y+2, w-4, h-4, down ? g.brush(QColorGroup::Background) :   g.brush(QColorGroup::Light));

		// not checked
		if(state == QButton::Off)
		{
			p->setPen(g.mid().light(110));
			p->drawRect( x+3, y+3, w-6, h-6 );
		}
		// checked
		else
		{
			// draw central rectangle
			p->setPen(g.mid().light(103));
			p->drawRect( x+5, y+5, 3, 3 );
			// "shadow" diagonal lines
			p->drawLine(x+4, y+3, x2-3, y2-4);
			p->drawLine(x+3, y+4, x2-4, y2-3);
			p->drawLine(x+3, y2-4, x2-4, y+3);
			p->drawLine(x+4, y2-3, x2-3, y+4);
			// draw central dark X
			p->setPen(g.dark().dark(160));
			p->drawLine(x+3, y+3, x2-3, y2-3);
			p->drawLine(x+3, y2-3, x2-3, y+3);
		}
}

QSize QNixStyle::indicatorSize() const
{
    return(QSize(13, 13));
}

void QNixStyle::drawIndicatorMask(QPainter *p, int x, int y, int w, int h, int)
{
    // needed by KHtml
    p->fillRect(x, y, w, h, Qt::color1);
}

// ============================================================================================================
// WIDGETS - PROGRESS BARS
// ============================================================================================================


// Progress bar indicator
// --------------------------------------------------------------
void QNixStyle::drawKProgressBlock(QPainter *p, int x, int y, int w, int h, const QColorGroup &g, QBrush *fill)
{
    p->fillRect(x, y, w, h, fill ? *fill : g.brush(QColorGroup::Text));
}

void QNixStyle::drawFocusRect(QPainter *p, const QRect &r, const QColorGroup &g, const QColor *c, bool atBorder)
{
    KStyle::drawFocusRect(p, r, g, c, atBorder);
}

// ============================================================================================================
// WIDGETS - TABS
// ============================================================================================================

// draw individual tab
// --------------------------------------------------------------
void QNixStyle::drawTab(QPainter *p, const QTabBar *tabBar, QTab *tab, bool selected)
{
	QRect r = tab->rect();
	QColorGroup g = tabBar->colorGroup();
	r.setY(r.y()+1);
	r.setWidth(r.width()+1);

	// only do special draw for top tabs
	if(tabBar->shape() != QTabBar::RoundedAbove)
	{
		KStyle::drawTab(p, tabBar, tab, selected);
		return;
	}

	int yBottom = r.bottom();
	int gap = 3;
	int yTop;
	int xLeft = r.x() + 1 + gap;
	int xRight = r.right()-gap;

	if( selected )
	 	yTop = r.y() + 1;
	else
		yTop = r.y() + 2;

	// [0.1 tab gradient fill]
	/*if( selected )
		drawVGradient(p, g, QColorGroup::Light, xLeft+1, yTop+2, r.width()-(gap*2)-2, r.height()-(yTop+2), 2, 2, r.width()-(gap*2), r.height()-(yTop), true );
	else
		drawVGradient(p, g, QColorGroup::Mid, xLeft+1, yTop+2, r.width()-(gap*2)-2, r.height()-(yTop+2), 2, 2, r.width()-(gap*2), r.height()-(yTop), true );
	*/

	// [0.2] Changed appearance of tabs to blend in with tab panel
	p->fillRect( xLeft+1, yTop+2, r.width()-(gap*2)-2, r.height()-(yTop+2), g.brush(QColorGroup::Background) );

	// INSET
	p->setPen( g.light().light(115));
	// left gap
	p->drawLine(xLeft-gap-1, yBottom+1,  xLeft-1, yBottom+1);
	// left
	p->drawLine( xLeft+1, yTop+1, xLeft+1, yBottom);
	// top
	p->drawLine(xLeft+1, yTop+1, xRight-2, yTop+1);
	// right gap
	p->drawLine(xRight, yBottom+1,  xRight+gap, yBottom+1);
	// right
	p->setPen( g.mid().light(135) );
	p->drawLine(xRight-1, yTop+1, xRight-1, yBottom);

	// NORMAL
	p->setPen(g.dark().dark(143));
	// left gap
	p->drawLine(xLeft-gap-1, yBottom,  xLeft-1, yBottom);
	// left
	p->drawLine(xLeft, yTop, xLeft, yBottom);
	// top
	p->drawLine(xLeft, yTop,  xRight-2, yTop);
	// right
	p->drawLine(xRight-1, yTop+1, xRight-1, yBottom);
	// right gap
	p->drawLine(xRight, yBottom,  xRight+gap, yBottom);

	// RIGHT DESIGN
	// only for larger tabs
	if( r.width() > 35 )
	{
		for( int dx = xRight-10; dx < xRight-1; dx+=3  )
		{
			p->setPen(g.mid() );
			p->drawLine(dx, yTop+1, dx, yBottom);
			p->drawLine(dx+1, yBottom, dx+2, yBottom);
			p->setPen( g.light().light(110));
			p->drawLine(dx+1, yTop+1, dx+1, yBottom-1);
			p->drawPoint( dx+2, yTop+1 );
		}
	}

	if( !selected )
	{
		p->setPen(g.dark().dark(143));
		p->drawLine(xLeft-gap-1, yBottom,  xRight+gap, yBottom);
		p->setPen( g.light().light(115));
		p->drawLine(xLeft-gap-1, yBottom+1,  xRight+gap, yBottom+1);
	}
	else
	{
		// SHADOW
		p->setPen(g.dark().light(125));
		// right
		p->drawLine(xRight, yTop+1, xRight, yBottom-1);
		p->drawPoint( xRight-1, yTop );
	}
}

void QNixStyle::drawTabMask(QPainter *p, const QTabBar*, QTab *tab, bool)
{
	p->fillRect(tab->rect(), Qt::color1);
}

void QNixStyle::tabbarMetrics(const QTabBar *t, int &hFrame, int &vFrame,
                            int &overlap)
{
	if(t->shape() == QTabBar::RoundedAbove)
	{
			overlap = 1;
			hFrame = 24;
			vFrame = 10;
	}
	else
		KStyle::tabbarMetrics(t, hFrame, vFrame, overlap);
}

// ============================================================================================================
// WIDGETS - PANEL
// ============================================================================================================

// Used for textboxes etc..
// --------------------------------------------------------------
void QNixStyle::drawPanel(QPainter *p, int x, int y, int w, int h,
		const QColorGroup &g, bool sunken, int lineWidth, const QBrush *fill)
{
	int x2 = x+w-1;
	int y2 = y+h-1;

	QPen oldPen = p->pen();

  // [0.2.1] - Special appearance for Kicker panel
	if( qstrcmp(kapp->argv()[0], "kicker") == 0 )
	{

		KStyle::drawPanel(p, x+1, y+1, w-2, h-2, g, sunken, lineWidth, fill);
		// extra line at the bottom
		p->setPen(g.mid());
		p->drawLine(x, y+h-2, x+w, y+h-2);
		//p->setPen(g.dark().dark(143));
		p->setPen(g.mid());
		p->drawRect( x, y, w, h );		// adjust for menu bar?
	}
	// Normal panels
	else
	{
		// frame - outermost
		p->setPen(g.mid().light(110));
		p->drawLine(x, y, x2, y);
		p->drawLine(x, y, x, y2-1);
		p->setPen(g.mid().light(140));
		p->drawLine(x, y2, x2, y2);
		p->drawLine(x2, y-1, x2, y2);
		// 1st inner frame
		p->setPen(g.dark().dark(143));
		p->drawRect( x+1, y+1, w-2, h-2 );
		// 2nd inner frame
		p->setPen(sunken ? g.mid().light(115) : g.light());
		p->drawLine(x+2, y+2, x2-2, y+2);
		p->drawLine(x+2, y+2, x+2, y2-2);
		p->setPen(sunken ? g.light() : g.mid().light(115) );
		p->drawLine(x+2, y2-2, x2-2, y2-2);
		p->drawLine(x2-2, y+2, x2-2, y2-2);
	}

  p->setPen(oldPen);
	if(fill)
	{
		// I believe here we are only supposed to fill if there is a
		// specified fill brush...
		p->fillRect(x+2, y+2, w-4, h-4, *fill);
	}
}

// ============================================================================================================
// KICKER
// ============================================================================================================

// --------------------------------------------------------------
// Kicker applet handle
// --------------------------------------------------------------
void QNixStyle::drawKickerAppletHandle(QPainter *p, int x, int y, int w, int h,  const QColorGroup &g, QBrush *)
{
	int x2, y2;

	// vertical   |||
	if(h > w)
	{
		x2 = x+w-1;
		y2 = y+h-2;

		if(highcolor)
			drawVGradient(p, g, QColorGroup::Background, x, y, w, h, 0, 0, w, h, true);
		else
			p->fillRect(x, y, w, h, g.brush(QColorGroup::Background));

		p->setPen(g.light());
		p->drawLine(x, y, x, y2-1);
		p->drawPoint( x+1, y );
		p->drawLine(x+3, y, x+3, y2);
		p->drawPoint( x+4, y );
		p->drawLine(x+6, y, x+6, y2);
		p->drawPoint( x+7, y );
		p->drawLine(x+9, y, x+9, y2);

		p->setPen(g.mid());
		p->drawPoint( x+1, y2 );
		p->drawPoint( x+2, y2 );
		p->drawLine(x+2, y, x+2, y2);
		p->drawPoint( x+4, y2 );
		p->drawLine(x+5, y, x+5, y2);
		p->drawPoint( x+7, y2 );
		p->drawLine(x+8, y, x+8, y2);

		p->setPen(g.dark());
		p->drawLine(x, y2+1, x2, y2+1);

	}
	// horizontal  ----
	else
	{
		x2 = x+w-2;
		y2 = y+h-1;

		if(highcolor)
			drawVGradient(p, g, QColorGroup::Background, x, y, w, h, 0, 0, w, h, true);
		else
			p->fillRect(x, y, w, h, g.brush(QColorGroup::Background));

		p->setPen(g.light());
		p->drawLine(x, y, x2-1, y);
		p->drawPoint( x, y+1 );
		p->drawLine(x, y+3, x2, y+3);
		p->drawPoint( x, y+4 );
		p->drawLine(x, y+6, x2, y+6);
		p->drawPoint( x, y+7 );
		p->drawLine(x, y+9, x2, y+9);

		p->setPen(g.mid());
		p->drawPoint( x, y+1 );
		p->drawPoint( x, y+2 );
		p->drawLine(x, y+2, x2, y+2);
		p->drawPoint( x, y+4 );
		p->drawLine(x, y+5, x2, y+5);
		p->drawPoint( x, y+7 );
		p->drawLine(x, y+8, x2, y+8);

		p->setPen(g.dark());
		p->drawLine(x2+1, y, x2+1, y2);
	}
}

// --------------------------------------------------------------
// Kicker Task button
// [0.2] - have tried to make sunken (active) button appear
//         clearer... still not quite right
// --------------------------------------------------------------
void QNixStyle::drawKickerTaskButton(QPainter *p, int x, int y, int w, int h, const QColorGroup &g,
		const QString &text, bool sunken, QPixmap *pixmap, QBrush *)
{
	// Begin use of Button code
	int x2 = x+w-1;
	int y2 = y+h-1;

	if(highcolor)
	{
		// frame - outermost
		p->setPen(g.mid().light(110));
		p->drawLine(x, y, x2, y);
		p->drawLine(x, y, x, y2-1);
		p->setPen(g.mid().light(140));
		p->drawLine(x, y2, x2, y2);
		p->drawLine(x2, y-1, x2, y2);
		// 1st inner frame
		p->setPen(g.dark().dark(143));
		p->drawRect( x+1, y+1, w-2, h-2 );
		// 2nd inner frame
		p->setPen(sunken ? g.mid().light(115) : g.light().light(105));
		p->drawLine(x+2, y+2, x2-2, y+2);
		p->drawLine(x+2, y+2, x+2, y2-2);
		p->setPen(sunken ? g.light().light(105) : g.mid().light(115) );
		p->drawLine(x+2, y2-2, x2-2, y2-2);
		p->drawLine(x2-2, y+2, x2-2, y2-2);
		// [0.2] special fill for sunken
		if( sunken )
		{
			//drawHGradient(p, g, QColorGroup::Button, x+3, y+3, w-6, h-6, 6, 0, w, h-6, true );
			//kDrawBeButton(p, x, y, w, h, g, true, &g.brush(QColorGroup::Light));
			//p->fillRect(x+3, y+3, w-6, h-6, g.brush(QColorGroup::Highlight) );
			p->fillRect(x+3, y+3, w-6, h-6, g.brush(QColorGroup::Light) );
		}
		else
			drawVGradient(p, g, QColorGroup::Button, x+3, y+3, w-6, h-6, 0, 0, w-6, h-6, false );
	}
	else
	{
		// just plain
		if( sunken)
			kDrawBeButton(p, x, y, w, h, g, true, &g.brush(QColorGroup::Mid));
		else
			drawButton(p, x, y, w, h, g, false);
	}

	// most of the rest of the code here in this function is from hcstyle
	if(text.isNull() && !pixmap)
		return;

	const int pxWidth = 20;
	int textPos = pxWidth;
	QRect br(buttonRect(x, y, w, h));

	if (sunken)
		p->translate(1,1);

	if ( pixmap && !pixmap->isNull() )
	{
		int dx = ( pxWidth - pixmap->width() ) / 2;
		int dy = ( h - pixmap->height() ) / 2;
		p->drawPixmap( br.x()+dx, dy, *pixmap );
	}

	QString s = text;
	static const QString &modStr = KGlobal::staticQString(QString::fromUtf8("[") + i18n("modified") + QString::fromUtf8("]"));

	int modStrPos = s.find(modStr);

	if (-1 != modStrPos)
	{
		// +1 because we include a space after the closing brace.
		s.remove(modStrPos, modStr.length()+1);
		QPixmap modPixmap = SmallIcon("modified");

		int dx = (pxWidth   - modPixmap.width())  / 2;
		int dy = (h  - modPixmap.height()) / 2;
		p->drawPixmap(br.x() + textPos + dx, dy, modPixmap);
		textPos += pxWidth;
	}

	// If we have a title
	if (!s.isEmpty())
	{
		// Truncate where necessary
		if (p->fontMetrics().width(s) > br.width() - textPos)
		{
			int maxLen = br.width() - textPos - p->fontMetrics().width("...");
			while ((!s.isEmpty()) && (p->fontMetrics().width(s) > maxLen))
					s.truncate(s.length() - 1);
			s.append("...");
		}

		// [0.2] make sunken(active) text colour darker
		// so that it is displayed more clearly
		p->setPen(sunken ? g.buttonText().dark(150) : g.buttonText());
		// draw the text
		p->drawText(br.x()+ textPos, -1, w-textPos, h,AlignLeft|AlignVCenter, s);
	}

}

// ============================================================================================================
// UTILITY FUNCTIONS
// ============================================================================================================

// draw vertical gradient pixmaps
// no check here, make sure you're highcolor beforehand
// --------------------------------------------------------------
void QNixStyle::drawVGradient(QPainter *p, const QColorGroup &g,
                            QColorGroup::ColorRole type, int x, int y,
                            int w, int h, int /*sx*/, int sy, int /*tw*/, int th, bool flip=false)
{
    GradientSet *grSet = gDict.find(g.color(type).rgb());
    if(!grSet){
        grSet = new GradientSet(g.color(type));
        gDict.insert(g.color(type).rgb(), grSet);
    }

    int grheight = th - sy;
    if(grheight <= 16)
		{
        p->drawTiledPixmap(x, y, w, h, flip ? *grSet->gradient(VSmallestRev) : *grSet->gradient(VSmallest) );
		}
    else if(grheight <= 24){
        p->drawTiledPixmap(x, y, w, h, flip ? *grSet->gradient(VSmallRev) : *grSet->gradient(VSmall) );
    }
    else if(grheight <= 34){
        p->drawTiledPixmap(x, y, w, h, flip ? *grSet->gradient(VMedRev) : *grSet->gradient(VMed) );
    }
    else if(grheight <= 64){
        p->drawTiledPixmap(x, y, w, h, flip ? *grSet->gradient(VLargeRev) : *grSet->gradient(VLarge) );
    }
    else
		{
        KPixmap *vLarge =  ( flip ? grSet->gradient(VLargeRev) : grSet->gradient(VLarge) );
				p->fillRect(x, y+vLarge->height(), w, h-vLarge->height(), g.color(type).dark(110));
        p->drawTiledPixmap(x, y, w, vLarge->height(), *vLarge);
    }
}

// draw horizontal gradient pixmaps
// no check here, make sure you're highcolor beforehand
// --------------------------------------------------------------
void QNixStyle::drawHGradient(QPainter *p, const QColorGroup &g,
                            QColorGroup::ColorRole type, int x, int y,
                            int w, int h, int sx, int /*sy*/, int tw, int /*th*/, bool flip=false )
{
    GradientSet *grSet = gDict.find(g.color(type).rgb());
    if(!grSet)
		{
        grSet = new GradientSet(g.color(type));
        gDict.insert(g.color(type).rgb(), grSet);
    }
    int grwidth = tw - sx;
    if(grwidth <= 16)
		{
        p->drawTiledPixmap(x, y, w, h, flip ? *grSet->gradient(HSmallestRev) : *grSet->gradient(HSmallest) );
    }
    else if(grwidth <= 24){
        p->drawTiledPixmap(x, y, w, h, flip ? *grSet->gradient(HSmallRev) : *grSet->gradient(HSmallest) );
    }
    else if(grwidth <= 34)
		{
        p->drawTiledPixmap(x, y, w, h,  flip ? *grSet->gradient(HMedRev) : *grSet->gradient(HMed) );
    }
    else if(grwidth <= 52)
		{
        p->drawTiledPixmap(x, y, w, h, flip ? *grSet->gradient(HLarge) : *grSet->gradient(HLarge));
    }
    else
		{
        KPixmap *hLarge =  ( flip ? grSet->gradient(HLargeRev) : grSet->gradient(HLarge) );
        p->fillRect(x+hLarge->width(), y, w-hLarge->width(), h, g.background().dark(110));
        p->drawTiledPixmap(x, y, hLarge->width(), h, *hLarge);
    }
}


// --------------------------------------------------------------
// From hcstyle.cpp
// --------------------------------------------------------------
void QNixStyle::polish(QPalette &appPal )
{
	KConfig *config = KGlobal::config();
	gDict.clear();
}

// --------------------------------------------------------------
// From hcstyle.cpp
// --------------------------------------------------------------
void QNixStyle::polish(QWidget *w)
{
	if(qstrcmp(w->name(), "qt_viewport") == 0 ||	w->testWFlags(WType_Popup) || w->inherits("KDesktop")	|| w->inherits("PanelButtonBase") || w->isTopLevel())
		return;

	if(w->inherits("QPushButton"))
		w->installEventFilter(this);

	if(w->inherits("QMenuBar") || w->inherits("KToolBarButton"))
	{
		w->setBackgroundMode(QWidget::NoBackground);
		return;
	}

	if(w->parent() && w->parent()->inherits("KToolBar"))
	{
		if(w->inherits("QLabel") || w->inherits("KToolBarSeparator"))
			w->setAutoMask(true);
	}
	else if( w->inherits("QButton") || w->inherits("QComboBox"))
		w->setBackgroundOrigin(QWidget::ParentOrigin);

	if(w->inherits("KToolBar"))
	{
		w->installEventFilter(this);
		w->setBackgroundMode(QWidget::NoBackground);
		return;
	}

}

// --------------------------------------------------------------
// From hcstyle.cpp
// --------------------------------------------------------------
void QNixStyle::unPolish(QWidget *w)
{
	if(qstrcmp(w->name(), "qt_viewport") == 0 ||w->testWFlags(WType_Popup) || w->inherits("KDesktop")	|| w->inherits("PanelButtonBase") || w->isTopLevel())
		return;

	w->setBackgroundMode(QWidget::PaletteBackground);

	if(w->inherits("QPushButton"))
	{
		w->removeEventFilter(this);
	}
	if(w->inherits("QMenuBar") || w->inherits("KToolBarButton"))
	{
		w->setBackgroundMode(QWidget::PaletteBackground);
		return;
	}
	if(w->parent() && w->parent()->inherits("KToolBar"))
	{
		if(w->inherits("QLabel") || w->inherits("KToolBarSeparator"))
				w->setAutoMask(false);
	}
	else if( w->inherits("QButton") || w->inherits("QComboBox"))
		w->setBackgroundOrigin(QWidget::WidgetOrigin);

	if(w->inherits("KToolBar"))
	{
		w->removeEventFilter(this);
		w->setBackgroundMode(QWidget::PaletteBackground);
		return;
	}
}

bool QNixStyle::eventFilter(QObject *obj, QEvent *ev)
{
    if(obj->inherits("KToolBar")){
        if(ev->type() == QEvent::Resize){
            QObjectList *tbChildList = obj->queryList("KToolBarButton", NULL,
                                                      false, false);
            QObjectListIt it(*tbChildList);
            QObject *child;
            while((child = it.current()) != NULL){
                ++it;
                if(child->isWidgetType())
                    ((QWidget *)child)->repaint(false);
            }
	    delete tbChildList;
        }
    }
    else if(obj->inherits("QPushButton")){
        if(ev->type() == QEvent::Enter){
            QWidget *btn = (QWidget *)obj;
            if (btn->isEnabled()){
                //highlightWidget = btn;
                btn->repaint(false);
            }
        }
        else if(ev->type() == QEvent::Leave){
            QWidget *btn = (QWidget *)obj;
            //if (btn == highlightWidget)
              //  highlightWidget = 0L;
            btn->repaint(false);
        }
    }
    return(false);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CLASS : GradientSet
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

// --------------------------------------------------------------
// Constructor
// --------------------------------------------------------------
GradientSet::GradientSet(const QColor &baseColor)
{
  int i;
	c = baseColor;

  for( i = 0; i < 16; i++ )
		gradients[i] = NULL;
}

// Actually create the gradients
// --------------------------------------------------------------
KPixmap * GradientSet::gradient(GradientType type)
{
	int lightValue = 109;
	int darkValue = 108;
	int lightValueRev = 112;
	int darkValueRev = 105;

  if(gradients[type])
 		return(gradients[type]);

	switch(type)
 	{
		// VERTICAL
    case VSmallest:
        gradients[VSmallest] = new KPixmap;
        gradients[VSmallest]->resize(18, 16);
        KPixmapEffect::gradient(*gradients[VSmallest], c.light(lightValue-3), c.dark(darkValue), KPixmapEffect::VerticalGradient);
        break;
    case VSmallestRev:
        gradients[VSmallestRev] = new KPixmap;
        gradients[VSmallestRev]->resize(18, 16);
        KPixmapEffect::gradient(*gradients[VSmallestRev], c.dark(darkValueRev), c.light(lightValueRev-3), KPixmapEffect::VerticalGradient);
        break;
    case VSmall:
        gradients[VSmall] = new KPixmap;
        gradients[VSmall]->resize(18, 24);
        KPixmapEffect::gradient(*gradients[VSmall], c.light(lightValue), c.dark(darkValue), KPixmapEffect::VerticalGradient);
        break;
    case VSmallRev:
        gradients[VSmallRev] = new KPixmap;
        gradients[VSmallRev]->resize(18, 24);
        KPixmapEffect::gradient(*gradients[VSmallRev], c.dark(darkValueRev), c.light(lightValueRev), KPixmapEffect::VerticalGradient);
        break;
    case VMed:
        gradients[VMed] = new KPixmap;
        gradients[VMed]->resize(18, 34);
        KPixmapEffect::gradient(*gradients[VMed], c.light(lightValue), c.dark(darkValue), KPixmapEffect::VerticalGradient);
        break;
    case VMedRev:
        gradients[VMedRev] = new KPixmap;
        gradients[VMedRev]->resize(18, 34);
        KPixmapEffect::gradient(*gradients[VMedRev], c.dark(darkValueRev),c.light(lightValueRev),  KPixmapEffect::VerticalGradient);
        break;
    case VLarge:
        gradients[VLarge] = new KPixmap;
        gradients[VLarge]->resize(18, 64);
        KPixmapEffect::gradient(*gradients[VLarge], c.light(lightValue), c.dark(darkValue),KPixmapEffect::VerticalGradient);
        break;
    case VLargeRev:
        gradients[VLargeRev] = new KPixmap;
        gradients[VLargeRev]->resize(18, 64);
        KPixmapEffect::gradient(*gradients[VLargeRev],  c.dark(darkValueRev),c.light(lightValueRev),KPixmapEffect::VerticalGradient);
        break;
		// HORIZONTAL
    case HSmallest:
        gradients[HSmallest] = new KPixmap;
        gradients[HSmallest]->resize(18, 16);
        KPixmapEffect::gradient(*gradients[HSmallest], c.light(lightValue-3), c.dark(darkValue), KPixmapEffect::HorizontalGradient);
        break;
    case HSmallestRev:
        gradients[HSmallestRev] = new KPixmap;
        gradients[HSmallestRev]->resize(18, 16);
        KPixmapEffect::gradient(*gradients[HSmallestRev], c.dark(darkValueRev), c.light(lightValueRev-3), KPixmapEffect::HorizontalGradient);
        break;
    case HSmall:
        gradients[HSmall] = new KPixmap;
        gradients[HSmall]->resize(18, 24);
        KPixmapEffect::gradient(*gradients[HSmall], c.light(lightValue-3), c.dark(darkValue), KPixmapEffect::HorizontalGradient);
        break;
    case HSmallRev:
        gradients[HSmallRev] = new KPixmap;
        gradients[HSmallRev]->resize(18, 24);
        KPixmapEffect::gradient(*gradients[HSmallRev], c.dark(darkValueRev), c.light(lightValueRev-3), KPixmapEffect::HorizontalGradient);
        break;
    case HMed:
        gradients[HMed] = new KPixmap;
        gradients[HMed]->resize(34, 18);
        KPixmapEffect::gradient(*gradients[HMed], c.light(lightValue), c.dark(darkValue), KPixmapEffect::HorizontalGradient);
        break;
    case HMedRev:
        gradients[HMedRev] = new KPixmap;
        gradients[HMedRev]->resize(34, 18);
        KPixmapEffect::gradient(*gradients[HMedRev],  c.dark(darkValueRev), c.light(lightValueRev),KPixmapEffect::HorizontalGradient);
        break;
    case HLarge:
        gradients[HLarge] = new KPixmap;
        gradients[HLarge]->resize(52, 18);
        KPixmapEffect::gradient(*gradients[HLarge], c.light(lightValue), c.dark(darkValue), KPixmapEffect::HorizontalGradient);
        break;
    case HLargeRev:
        gradients[HLargeRev] = new KPixmap;
        gradients[HLargeRev]->resize(52, 18);
        KPixmapEffect::gradient(*gradients[HLargeRev], c.light(darkValueRev), c.dark(lightValueRev), KPixmapEffect::HorizontalGradient);
        break;
    default:
        break;
  }
  return(gradients[type]);
}

// --------------------------------------------------------------
// Destructor - free pixmaps
// --------------------------------------------------------------
GradientSet::~GradientSet()
{
	int i;
	for( i = 0; i < 16; ++i )
	{
		if(gradients[i])
			delete gradients[i];
	}
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////

