/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file camera.cpp
 * \brief Implementation of the bear::camera class.
 * \author Julien Jorge
 */
#include "generic_items/camera.hpp"

#include "engine/level.hpp"
#include "engine/export.hpp"

#include <climits>
#include <limits>

BASE_ITEM_EXPORT( camera, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::camera::camera()
  : m_valid_area
  (0, 0,
   std::numeric_limits<bear::universe::coordinate_type>::infinity(),
   std::numeric_limits<bear::universe::coordinate_type>::infinity()),
    m_max_move_length
  ( std::numeric_limits<bear::universe::coordinate_type>::infinity() ),
    m_max_zoom_length
  ( std::numeric_limits<bear::universe::coordinate_type>::infinity() ),
    m_shaker_force(0)
{
  set_global(true);
  set_phantom(true);
  set_can_move_items(false);
} // camera::camera()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the item.
 */
void bear::camera::build()
{
  super::build();

  universe::coordinate_type left = m_valid_area.left();
  universe::coordinate_type right = m_valid_area.right();
  universe::coordinate_type top = m_valid_area.top();
  universe::coordinate_type bottom = m_valid_area.bottom();

  if (left < 0)
    left = 0;
  else if (left > get_level().get_size().x)
    left = get_level().get_size().x;

  if (bottom < 0)
    bottom = 0;
  else if (bottom > get_level().get_size().y)
    bottom = get_level().get_size().y;

  if (right > get_level().get_size().x)
    right = get_level().get_size().x;

  if (top > get_level().get_size().y)
    top = get_level().get_size().y;

  m_valid_area.set(left, bottom, right, top);

  m_default_size = m_wanted_size = get_size();
} // camera::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Do one step in the progression of the item.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::camera::progress( bear::universe::time_type elapsed_time )
{
  progress_zoom(elapsed_time);
} // camera::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type <real>.
 * \param value The value of the field.
 */
bool bear::camera::set_real_field( const std::string& name, double value )
{
  bool result = true;

  if ( name == "camera.valid_min.x" )
    m_valid_area.first_point.x = value;
  else if ( name == "camera.valid_min.y" )
    m_valid_area.first_point.y = value;
  else if ( name == "camera.valid_max.x" )
    m_valid_area.second_point.x = value;
  else if ( name == "camera.valid_max.y" )
    m_valid_area.second_point.y = value;
  else if ( name == "camera.max_move_length" )
    m_max_move_length = value;
  else if ( name == "camera.max_zoom_length" )
    m_max_zoom_length = value;
  else
    result = super::set_real_field(name, value);

  return result;
} // camera::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progressively change the size of the camera.
 * \param s The size to attain.
 */
void bear::camera::set_wanted_size( const universe::size_box_type& s )
{
  m_wanted_size = s;
} // camera::set_wanted_size()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the force of the shaker of the camera.
 * \param shaker_force The new shaker force.
 */
void bear::camera::set_shaker_force( double shaker_force )
{
  m_shaker_force = shaker_force;
} // camera::set_shaker_force()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress toward the wanted size.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::camera::progress_zoom( universe::time_type elapsed_time )
{
  if ( get_size() != m_wanted_size )
    {
      const universe::position_type c = get_center_of_mass();

      progress_zoom_with_ratio(elapsed_time);

      set_center_of_mass(c);
    }

  m_wanted_size = m_default_size;
} // camera::progress_zoom()
 
/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the position of the center of the camera.
 * \param center_position New position for the center of the camera.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::camera::adjust_position
( const bear::universe::position_type& center_position,
  bear::universe::time_type elapsed_time )
{
  double a_x = (m_shaker_force * rand() / RAND_MAX);
  double a_y = (m_shaker_force * rand() / RAND_MAX);

  adjust_position_x(a_x + center_position.x, elapsed_time * m_max_move_length);
  adjust_position_y(a_y + center_position.y, elapsed_time * m_max_move_length);

  m_shaker_force = 0;
} // camera::adjust_position()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a new position for the center of the camera, but keep it in the
 *        valid area.
 * \param center_position New position for the center of the camera.
 */
void bear::camera::teleport
( const bear::universe::position_type& center_position )
{
  set_center_of_mass(center_position);
  stay_valid();
} // camera::teleport()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the X position of the center of the camera.
 * \param center_position X-coordinate to set to the center of the camera.
 * \param max_move The longest feasible movement.
 */
void bear::camera::adjust_position_x
( bear::universe::coordinate_type center_position,
  bear::universe::coordinate_type max_move )
{
  const bear::universe::coordinate_type current_x = get_center_of_mass().x;
  bear::universe::coordinate_type distance;

  if ( center_position < current_x )
    {
      distance = std::min(current_x - center_position, max_move);

      if ( get_left() - m_valid_area.left() >= distance )
        set_left( get_left() - distance );
      else
        set_left( m_valid_area.left() );
    }
  else if ( center_position > current_x )
    {
      distance = std::min(center_position - current_x, max_move);

      if ( get_right() + distance <= m_valid_area.right() )
        set_left( get_left() + distance );
      else
        set_right( m_valid_area.right() );
    }
} // camera::adjust_position_x()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the Y position of the center of the camera.
 * \param center_position Y-coordinate to set to the center of the camera.
 * \param max_move The longest feasible movement.
 */
void bear::camera::adjust_position_y
( bear::universe::coordinate_type center_position,
  bear::universe::coordinate_type max_move )
{
  const bear::universe::coordinate_type current_y = get_center_of_mass().y;
  bear::universe::coordinate_type distance;

  if ( center_position < current_y )
    {
      distance = std::min(current_y - center_position, max_move);

      if ( get_bottom() - distance >= m_valid_area.bottom() )
        set_bottom( get_bottom() - distance );
      else
        set_bottom( m_valid_area.bottom() );
    }
  else if ( center_position > current_y )
    {
      distance = std::min(center_position - current_y, max_move);

      if ( get_top() + distance <= m_valid_area.top() )
        set_bottom( get_bottom() + distance );
      else
        set_top( m_valid_area.top() );
    }
} // camera::adjust_position_y()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the camera is in its valid area and adjust its position if
 *        needed.
 */
void bear::camera::stay_valid()
{
  if ( get_left() < m_valid_area.left() )
    set_left( m_valid_area.left() );

  if ( get_bottom() < m_valid_area.bottom() )
    set_bottom( m_valid_area.bottom() );

  if ( get_right() > m_valid_area.right() )
    {
      if ( m_valid_area.right() > get_width() )
        set_right( m_valid_area.right() );
      else
        set_left(0);
    }

  if ( get_top() > m_valid_area.top() )
    {
      if ( m_valid_area.top() > get_height() )
        set_top( m_valid_area.top() );
      else
        set_bottom(0);
    }
} // camera::stay_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Progress toward the wanted size and keep the ratio width/height.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::camera::progress_zoom_with_ratio( universe::time_type elapsed_time )
{
  const universe::coordinate_type r = m_wanted_size.x / m_wanted_size.y;

  const universe::coordinate_type d_x =
    std::min( std::abs(m_wanted_size.x - get_width()),
              m_max_zoom_length * elapsed_time );
  const universe::coordinate_type d_y =
    std::min( std::abs(m_wanted_size.y - get_height()),
              m_max_zoom_length * elapsed_time );

  if ( d_x > d_y )
    {
      if ( m_wanted_size.x > get_width() )
        set_width( get_width() + d_x );
      else
        set_width( get_width() - d_x );

      set_height( get_width() / r );
    }
  else
    {
      if ( m_wanted_size.y > get_height() )
        set_height( get_height() + d_y );
      else
        set_height( get_height() - d_y );

      set_width( get_height() * r );
    }
} // camera::progress_zoom_with_ratio()
