/**
  @file devlist.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/  
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <ctype.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btsearch-bt.h"
#include "dbus.h"
#include "bt-dbus.h"
#include "dbus-helper.h"
#include "inquiry.h"
#include "devlist.h"

extern GMainLoop *event_loop;

static char *list_receiver = NULL;

static void shutdown_and_exit()
{
    inquiry_stop();
    if (!send_dbus_search_complete(get_dbus_connection()))
        error("Unable to send search_complete signal!");
    g_main_loop_quit(event_loop);
}

void stop_search(void)
{
    inquiry_stop();
    shutdown_and_exit();
}

static void name_exit_callback(const char *name, gpointer unused_data)
{
    debug("List receiver exited");
    stop_search();
}

/** Start the search process
 * @param receiver D-BUS address of the list receiver, NULL to enable test mode
 * @returns TRUE on success, FALSE on failure
 */
gboolean init_search(const char *receiver)
{
    if (receiver) {
        list_receiver = g_strdup(receiver);
        add_name_listener(get_dbus_connection(), receiver,
                          (name_cb)name_exit_callback, NULL);
    }
    else
        list_receiver = NULL;

    return inquiry_start();
}

/* Send "New device found" signal to D-BUS */
gboolean send_dbus_dev_found(DBusConnection *connection, DevInfo *dev)
{
    DBusMessage *signal;
    uint8_t *cls = dev->cls;
    uint32_t clk = (uint32_t)dev->clock;

    debug("Found: BDA=%s, name=%s, class=0x%02x%02x%02x, rssi=0x%02x, clock=0x%04X",
            dev->bda, dev->name,
            dev->cls[2], dev->cls[1], dev->cls[0],
            dev->rssi, dev->clock);

    /* This happens in test mode */
    if (list_receiver == NULL)
        return TRUE;

    signal = new_dbus_signal(BTSEARCH_SIG_PATH,
                             BTSEARCH_SIG_INTERFACE,
                             BTSEARCH_DEV_FOUND_SIG,
                             list_receiver);

    append_dbus_args(signal,
                     DBUS_TYPE_STRING, &dev->bda,
                     DBUS_TYPE_STRING, &dev->name,
                     DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &cls, 3,
                     DBUS_TYPE_BYTE, &dev->rssi,
                     DBUS_TYPE_UINT32, &clk,
                     DBUS_TYPE_INVALID);

    return send_and_unref(connection, signal);
}

/* Send search_complete signal to D-BUS */
gboolean send_dbus_search_complete(DBusConnection *connection)
{
    DBusMessage *signal;

    /* This happens in test mode */
    if (list_receiver == NULL)
        return TRUE;

    signal = new_dbus_signal(BTSEARCH_SIG_PATH,
                             BTSEARCH_SIG_INTERFACE,
                             BTSEARCH_SEARCH_COMPLETE_SIG,
                             list_receiver);

    return send_and_unref(connection, signal);
}


gboolean search_in_progress(void)
{
    if (list_receiver == NULL)
        return FALSE;
    else
        return TRUE;
}

