(* 
   Copyright 2003 Savonet team

   This file is part of Ocaml-vorbis.
   
   Ocaml-vorbis is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   Ocaml-vorbis is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with Ocaml-vorbis; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)
(**
  Decode from or encode to the Ogg Vorbis compressed audio format; or get informations about an Ogg Vorbis file.

  @author Samuel Mimram, Julien Cristau
*)

(* $Id: vorbis.ml,v 1.27 2004/06/29 22:56:47 smimram Exp $ *)

type encoder

exception Invalid_parameters
exception Invalid_quality
exception Invalid_bitrate
exception Invalid_channels
exception Invalid_sample_freq
exception Could_not_open_file
exception Not_a_vorbis_file
exception Hole_in_data
exception Utf8_failure of string

let _ =
  Callback.register_exception "exn_end_of_file" End_of_file;
  Callback.register_exception "vorbis_exn_invalid_parameters" Invalid_parameters;
  Callback.register_exception "vorbis_exn_invalid_quality" Invalid_quality;
  Callback.register_exception "vorbis_exn_invalid_bitrate" Invalid_bitrate;
  Callback.register_exception "vorbis_exn_invalid_channels" Invalid_channels;
  Callback.register_exception "vorbis_exn_invalid_sample_freq" Invalid_sample_freq;
  Callback.register_exception "vorbis_exn_could_not_open_file" Could_not_open_file;
  Callback.register_exception "vorbis_exn_not_a_vorbis_file" Not_a_vorbis_file;
  Callback.register_exception "vorbis_exn_hole_in_data" Hole_in_data;
  Callback.register_exception "vorbis_exn_utf8_failure" (Utf8_failure "")

type enc_params =
    {
      enc_bitrate : int option;
      enc_min_bitrate : int option;
      enc_max_bitrate : int option;
      enc_quality : float;
      enc_channels : int;
      enc_sample_freq : int option;
      enc_managed : bool;
      enc_in_channels : int;
      enc_in_sample_freq : int;
      enc_in_sample_size : int;
      enc_in_big_endian : bool;
    }

external create_encoder_pre : enc_params -> string option -> string option -> string option -> string option -> string option -> string option -> encoder*string = "ocaml_vorbis_create_encoder_byte" "ocaml_vorbis_create_encoder"

let create_encoder ?title ?artist ?genre ?date ?album ?tracknum params =
  create_encoder_pre params title artist genre date album tracknum

let create_encoder_opt title artist genre date album tracknum params =
  create_encoder_pre params title artist genre date album tracknum

external encode_buffer : encoder -> string -> string = "ocaml_vorbis_encode_buffer"

type dec_params =
    {
      channels : int;
      sample_freq : int;
      sample_size : int;
      big_endian : bool;
      signed : bool;
    }

type dec_file

external open_dec_file : string -> dec_params -> dec_file = "ocaml_vorbis_open_dec_file"

external decode : dec_file -> string -> int -> int -> int = "ocaml_vorbis_decode"

external close_dec_file : dec_file -> unit = "ocaml_vorbis_close_dec_file"


external encoder_reset_pre : encoder -> string option -> string option -> string option -> string option -> string option -> string option -> string = "ocaml_vorbis_encoder_reset_byte" "ocaml_vorbis_encoder_reset"
let encoder_reset ?title ?artist ?genre ?date ?album ?tracknum ep = encoder_reset_pre ep title artist genre date album tracknum
let encoder_reset_opt title artist genre date album tracknum ep = encoder_reset_pre ep title artist genre date album tracknum


type info = { 
  vorbis_version : int;
  audio_channels : int;
  audio_sample_rate : int;
  bitrate_maximum : int option;
  bitrate_nominal : int option;
  bitrate_minimum : int option;
  blocksize_0 : int;
  blocksize_1 : int;
  duration : int;
}

external utf8_decode : string -> string = "ocaml_vorbis_utf8_decode"

let read32 s = 
  let a = int_of_char(s.[0]) 
  and b = int_of_char(s.[1])
  and c = int_of_char(s.[2]) 
  and d = int_of_char(s.[3]) in
    ((d lsl 24) lor (c lsl 16) 
     lor (b lsl 8) lor a)

let get_lacing_values file = 
  ignore(Unix.lseek file 26 Unix.SEEK_CUR);
  let buf = String.create 1 in
    ignore(Unix.read file buf 0 1);
    let nb_segments = int_of_char buf.[0] in
    let segments = String.create nb_segments in
      ignore(Unix.read file segments 0 nb_segments);
      segments
	
let last_pos file =
  let pos = Unix.lseek file 0 Unix.SEEK_CUR in
  let buf = String.create 27 in
    ignore(Unix.read file buf 0 27);
    let eos = ((int_of_char buf.[5]) lsr 2) land 1 in
      if eos=1 then 
	Some (String.sub buf 6 8)
      else begin
	ignore(Unix.lseek file pos Unix.SEEK_SET);
	None
      end
	
let samples filename =
  let file = Unix.openfile filename [Unix.O_RDONLY] 0400 in
  let granule_pos = ref "" in
    while !granule_pos = "" do
      match last_pos file with 
	| None -> let segments = get_lacing_values file in
	  let size = ref 0 in
	    for i = 0 to (String.length segments) - 1 do
	      size := !size + (int_of_char segments.[i])
	    done;
	    ignore(Unix.lseek file !size Unix.SEEK_CUR)
	| Some n -> granule_pos := n
    done;
    Unix.close file;
    let nb_samples = ref 0 in
      for i = 7 downto 0 do 
	nb_samples := (!nb_samples lsl 8) + int_of_char !granule_pos.[i] 
      done;
      !nb_samples
	
let decode_comments filename = 
  let comments = ref "" in
  let file = Unix.openfile filename [Unix.O_RDONLY ; Unix.O_NONBLOCK] 0400 in
    ignore (Unix.lseek file 58 Unix.SEEK_SET);
    let found_end = ref false in
      while not !found_end do
	let segments = get_lacing_values file in
	let i = ref 0 in
	  while !i < String.length segments 
	    && (int_of_char segments.[!i]) = 255 do
	      let buf = String.create 255 in
		ignore(Unix.read file buf 0 255);
		comments := !comments ^ buf;
	  done;
	  if !i < String.length segments then begin
	    found_end := true;
	    let buf = String.create (int_of_char segments.[!i]) in
	      ignore(Unix.read file buf 0 (int_of_char segments.[!i]));
	      comments := !comments ^ buf
	  end
      done;
      Unix.close file;
      !comments
	
let get_comments filename = 
  let s = decode_comments filename in
    if String.sub s 1 6 <> "vorbis" then
      invalid_arg "This file is not a valid vorbis audio file.";
    let vendorlen = read32 (String.sub s 7 4) in
    let vendor = utf8_decode(String.sub s 11 vendorlen) in
    let nb_comments = read32 (String.sub s (11+vendorlen) 4) in
    let comments = Array.make (nb_comments) ("","") in
    let offset = ref (11+vendorlen+4) in
      for i = 0 to nb_comments-1 do
	let length = read32 (String.sub s !offset 4) in
	let equal_pos = String.index_from s (!offset+4) '=' in
	  comments.(i) <- String.uppercase (String.sub s (!offset + 4) (equal_pos- !offset - 4)),
	  utf8_decode(String.sub s 
			(equal_pos + 1) (length-equal_pos + !offset + 3));
	  offset := !offset + length + 4;
      done;
      vendor,comments
	
let get_info filename = 
  let file = Unix.openfile filename [Unix.O_RDONLY;Unix.O_NONBLOCK] 0400 in
    ignore(Unix.lseek file 35 Unix.SEEK_SET);
    let s = String.create 22 in
      ignore(Unix.read file s 0 22);
      Unix.close file;
      let br_max = read32 (String.sub s 9 4) 
      and br_nom = read32 (String.sub s 13 4)
      and br_min = read32 (String.sub s 17 4) 
      and blocksize_1 = ((int_of_char s.[21]) asr 4) land 15
      and blocksize_0 = (int_of_char s.[21]) land 15 
      and sample_rate = read32 (String.sub s 5 4)
      in
	{  
	  vorbis_version = read32 (String.sub s 0 4);
	  audio_channels = int_of_char s.[4];
	  audio_sample_rate = sample_rate;
	  bitrate_maximum = if br_max>0 then Some br_max else None;
	  bitrate_nominal = if br_nom>0 then Some br_nom else None;
	  bitrate_minimum = if br_min>0 then Some br_min else None;
	  blocksize_0 = blocksize_0;
	  blocksize_1 = blocksize_1;
	  duration = (samples filename)/sample_rate;
	}
	  
let file_size filename = 
  (Unix.stat filename).Unix.st_size

module type Iofile =
sig
  type file_descr
  type open_flag = O_RDONLY | O_WRONLY | O_RDWR | O_CREAT | O_TRUNC
  type file_perm = int
  type seek_command = SEEK_SET | SEEK_CUR | SEEK_END
  val openfile : string -> open_flag list -> file_perm -> file_descr
  val close : file_descr -> unit
  val read : file_descr -> string -> int -> int -> int
  val lseek : file_descr -> int -> seek_command -> int
end
  
module Info (Io: Iofile) =
struct
  (* Internal type for info without duration *)
  type header = {
    version : int;
    channels : int;
    sample_rate : int;
    bitmax : int option;
    bitnom : int option;
    bitmin : int option;
    block0 : int;
    block1 : int;
  }

  (* Find the lacing values of the segments in the current ogg page *)
  let get_lacing_values file = 
    ignore(Io.lseek file 26 Io.SEEK_CUR);
    let buf = String.create 1 in
      ignore(Io.read file buf 0 1);
      let nb_segments = int_of_char buf.[0] in
      let segments = String.create (256*nb_segments) in
	ignore(Io.read file segments 0 (256*nb_segments));
	segments
	  
  (* If the current page is the last one, return the absolute granule position *)
  let last_pos file =
    let pos = Io.lseek file 0 Io.SEEK_CUR in
    let buf = String.create 27 in
      ignore(Io.read file buf 0 27);
      let eos = ((int_of_char buf.[5]) lsr 2) land 1 in
	if eos=1 then 
	  Some (String.sub buf 6 8)
	else begin
	  ignore(Io.lseek file pos Io.SEEK_SET);
	  None
	end
	  
  (* Return the number of samples in the vorbis file *)
  let samples filename =
    let file = Io.openfile filename [Io.O_RDONLY] 0400 in
    let granule_pos = ref "" in
      while !granule_pos = "" do
	match last_pos file with 
	  | None -> let segments = get_lacing_values file in
	    let size = ref 0 in
	      for i = 0 to (String.length segments) - 1 do
		size := !size + (int_of_char segments.[i])
	      done;
	      ignore(Io.lseek file !size Io.SEEK_CUR)
	  | Some n -> granule_pos := n
      done;
      Io.close file;
      let nb_samples = ref 0 in
	for i = 7 downto 0 do 
	  nb_samples := (!nb_samples lsl 8) + int_of_char !granule_pos.[i] 
	done;
	!nb_samples
	  
  let decode_comments filename = 
    let comments = ref "" in
    let file = Io.openfile filename [Io.O_RDONLY] 0400 in
      ignore (Io.lseek file 58 Io.SEEK_SET);
      let found_end = ref false in
	while not !found_end do
	  let segments = get_lacing_values file in
	  let i = ref 0
	  and size_comments = ref 0 in
	    while !i < (String.length segments)/256
	      && (int_of_char segments.[!i]) = 255 do
		size_comments := !size_comments + 255;
		incr i
	    done;
	    if !i < (String.length segments)/256 then begin
	      found_end := true;
	      size_comments := !size_comments + (int_of_char segments.[!i])
	    end;
	    let s = String.sub segments ((String.length segments)/256) 
		      !size_comments in
	      comments := !comments ^ s;
	done;
	Io.close file;
	flush stdout;
	!comments
	  
  let get_comments filename = 
    let s = decode_comments filename in
      if String.sub s 1 6 <> "vorbis" then 
	invalid_arg "This file is not a valid vorbis audio file.";
      let vendorlen = read32 (String.sub s 7 4) in
      let vendor = utf8_decode(String.sub s 11 vendorlen) in
      let nb_comments = read32 (String.sub s (11+vendorlen) 4) in
      let comments = Array.make (nb_comments) ("","") in
      let offset = ref (11+vendorlen+4) in
	for i = 0 to nb_comments - 1 do
	  let length = read32 (String.sub s !offset 4) in
	  let equal_pos = String.index_from s (!offset+4) '=' in
	    comments.(i) <- 
	    String.uppercase (String.sub s (!offset+4) (equal_pos - !offset - 4)),
	    utf8_decode(String.sub s (equal_pos+1) 
			  (length - equal_pos + !offset + 3));
	    offset := !offset + length + 4;
	done;
	vendor,comments
	  
  let read_info filename = 
    let file = Io.openfile filename [Io.O_RDONLY] 0400 in
      ignore(Io.lseek file 35 Io.SEEK_SET);
      let s = String.create 22 in
	ignore(Io.read file s 0 22);
	Io.close file;
	let br_max = read32 (String.sub s 9 4) 
	and br_nom = read32 (String.sub s 13 4)
	and br_min = read32 (String.sub s 17 4) 
	and blocksize_1 = ((int_of_char s.[21]) asr 4) land 15
	and blocksize_0 = (int_of_char s.[21]) land 15 
	and sample_rate = read32 (String.sub s 5 4)
	in
	  {  
	    version = read32 (String.sub s 0 4);
	    channels = int_of_char s.[4];
	    sample_rate = sample_rate;
	    bitmax = if br_max>0 then Some br_max else None;
	    bitnom = if br_nom>0 then Some br_nom else None;
	    bitmin = if br_min>0 then Some br_min else None;
	    block0 = blocksize_0;
	    block1 = blocksize_1;
	  }

  let file_size filename =
    let file = Io.openfile filename [Io.O_RDONLY] 0400 in
    let size = Io.lseek file 0 Io.SEEK_END in
      Io.close file; size

  let heuristical_duration filename = 
    let size = 5000 in
    let file = Io.openfile filename [Io.O_RDONLY] 0400 in
    let r = Str.regexp "OggS" in
    let buf = String.create size in
    let offset = Io.lseek file (-size) Io.SEEK_END in
      Io.close file;
      try
	let pos = Str.search_backward r buf (size-5) in
	let eos = ((int_of_char buf.[pos + 5]) lsr 2) land 1 in
	  if eos = 1 then begin
	    let dur = ref 0 in
	      for i = pos + 13 downto pos + 6 do
		dur := (!dur lsl 8) + (int_of_char buf.[i]);
	      done;
	      !dur
	  end
	  else begin
	    (* Printf.eprintf "Heuristic failed to find duration of Vorbis file\n%!"; *)
	    raise Not_found
	  end
      with Not_found ->
	(* Printf.eprintf "Heuristic failed to find duration of Vorbis file\n%!"; *)
	raise Not_found
	  
  let fill_header filename samples =
    let header = read_info filename in
      {
	vorbis_version = header.version;
	audio_channels = header.channels;
	audio_sample_rate = header.sample_rate;
	bitrate_nominal = header.bitnom;
	bitrate_maximum = header.bitmax;
	bitrate_minimum = header.bitmin;
	blocksize_0 = header.block0;
	blocksize_1 = header.block1;
	duration = samples/header.sample_rate;
      }

  let get_info filename = 
    fill_header filename (samples filename)

  let get_heuristical_info filename = 
    fill_header filename (heuristical_duration filename)
end
