/*
 * Implementation of the proxy around XMMS2's client interface.
 *
 * Music Applet
 * Copyright (C) 2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include <config.h>

#include "ma-xmms2-proxy.h"

#include <stdlib.h>

#include <xmmsclient/xmmsclient.h>
#include <xmmsclient/xmmsclient-glib.h>


#define GET_PRIVATE(o) 		(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_XMMS2_PROXY, MaXMMS2ProxyPrivate))


typedef struct _MaXMMS2ProxyPrivate	MaXMMS2ProxyPrivate;


struct _MaXMMS2ProxyPrivate
{
	xmmsc_connection_t *conn;
	void *source;

	guint current_id;

	xmmsc_result_t *bcast_id;
	xmmsc_result_t *bcast_entry;
	xmmsc_result_t *bcast_status;

	xmmsc_result_t *sig_playtime;

	xmmsc_result_t *call_id;
	xmmsc_result_t *call_playtime;
	xmmsc_result_t *call_status;
	xmmsc_result_t *call_minfo;
};

static MaProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_xmms2_proxy_class_init (MaXMMS2ProxyClass *klass);
static void ma_xmms2_proxy_init (MaXMMS2Proxy *xproxy);

static void ma_xmms2_proxy_dispose (GObject *object);
static void ma_xmms2_proxy_finalize (GObject *object);

static void ma_xmms2_proxy_toggle_playback (MaProxy *proxy);
static void ma_xmms2_proxy_previous (MaProxy *proxy);
static void ma_xmms2_proxy_next (MaProxy *proxy);

static void ma_xmms2_proxy_rate_song (MaProxy *proxy, gdouble rating);

static void ma_xmms2_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml);

static void ma_xmms2_proxy_launch (MaProxy *proxy);

static void ma_xmms2_proxy_enable (MaProxy *proxy);
static void ma_xmms2_proxy_disable (MaProxy *proxy);

static void ma_xmms2_proxy_poll_disconnected (MaProxy *proxy);

static void cancel_result (xmmsc_result_t **res);

static void disconnect_cb (void *proxy);

static void minfo_cb (xmmsc_result_t *res, void *proxy);
static void current_id_cb (xmmsc_result_t *res, void *proxy);
static void playtime_cb (xmmsc_result_t *res, void *proxy);
static void entry_changed_cb (xmmsc_result_t *res, void *proxy);
static void status_cb (xmmsc_result_t *res, void *proxy);

/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_xmms2_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaXMMS2ProxyClass),			/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_xmms2_proxy_class_init,	/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaXMMS2Proxy),				/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_xmms2_proxy_init,	/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_PROXY, "MaXMMS2Proxy", &info, 0);
	}

	return type;
}

static void
ma_xmms2_proxy_class_init (MaXMMS2ProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_xmms2_proxy_dispose;
	object_class->finalize = ma_xmms2_proxy_finalize;

	proxy_class->toggle_playback = ma_xmms2_proxy_toggle_playback;
	proxy_class->previous = ma_xmms2_proxy_previous;
	proxy_class->next = ma_xmms2_proxy_next;

	proxy_class->rate_song = ma_xmms2_proxy_rate_song;

	proxy_class->prepare_prefs = ma_xmms2_proxy_prepare_prefs;

	proxy_class->launch = ma_xmms2_proxy_launch;

	proxy_class->enable = ma_xmms2_proxy_enable;
	proxy_class->disable = ma_xmms2_proxy_disable;

	proxy_class->poll_disconnected = ma_xmms2_proxy_poll_disconnected;

	g_type_class_add_private (klass, sizeof (MaXMMS2ProxyPrivate));
}

static void
ma_xmms2_proxy_init (MaXMMS2Proxy *xproxy)
{
	MaXMMS2ProxyPrivate *priv = GET_PRIVATE (xproxy);

	priv->conn = NULL;
	priv->source = NULL;

	priv->current_id = 0;

	priv->bcast_id = NULL;
	priv->bcast_entry = NULL;
	priv->bcast_status = NULL;

	priv->sig_playtime = NULL;

	priv->call_id = NULL;
	priv->call_playtime = NULL;
	priv->call_status = NULL;
	priv->call_minfo = NULL;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

MaProxy *
ma_xmms2_proxy_new (MaConf *conf)
{
	MaProxy *proxy;

	g_return_val_if_fail (conf != NULL, NULL);

	proxy = g_object_new (MA_TYPE_XMMS2_PROXY,
			      "name", "XMMS2",
			      "conf", conf,
			      NULL);

	return proxy;
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_xmms2_proxy_dispose (GObject *object)
{
	ma_xmms2_proxy_disable (MA_PROXY (object));

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
ma_xmms2_proxy_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_xmms2_proxy_toggle_playback (MaProxy *proxy)
{
	MaXMMS2ProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	if (ma_proxy_get_playing (proxy))
		xmmsc_result_unref (xmmsc_playback_pause (priv->conn));
	else
		xmmsc_result_unref (xmmsc_playback_start (priv->conn));
}

static void
ma_xmms2_proxy_previous (MaProxy *proxy)
{
	MaXMMS2ProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	xmmsc_result_unref (xmmsc_playlist_set_next_rel (priv->conn, -1));
	xmmsc_result_unref (xmmsc_playback_tickle (priv->conn));
}

static void
ma_xmms2_proxy_next (MaProxy *proxy)
{
	MaXMMS2ProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	xmmsc_result_unref (xmmsc_playlist_set_next_rel (priv->conn, 1));
	xmmsc_result_unref (xmmsc_playback_tickle (priv->conn));
}

static void
ma_xmms2_proxy_rate_song (MaProxy *proxy, gdouble rating)
{
	MaXMMS2ProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	gchar *str = g_strdup_printf ("%d", (int) rating);
	xmmsc_result_unref (xmmsc_medialib_entry_property_set_with_source (priv->conn,
									   priv->current_id,
									   "clients/generic",
									   "rating", str));
	g_free (str);
}

static void
ma_xmms2_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	MaConf *conf;

	GtkWidget *page;
	GtkWidget *connPath;
	GtkWidget *command;
	GtkWidget *browse;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	conf = _ma_proxy_get_conf (proxy);

	page = glade_xml_get_widget (xml, "xmms2");
	connPath = glade_xml_get_widget (xml, "xmms2-conn-path");
	command = glade_xml_get_widget (xml, "xmms2-command");
	browse = glade_xml_get_widget (xml, "xmms2-browse");

	ma_conf_bind_string_entry (conf, "xmms2_conn_path", GTK_ENTRY (connPath));
	ma_conf_bind_string_entry (conf, "xmms2_command", GTK_ENTRY (command));
	ma_conf_bind_string_browse (conf, "xmms2_command", GTK_BUTTON (browse));

	gtk_widget_show (page);
}

static void
ma_xmms2_proxy_launch (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	_ma_proxy_launch_command (proxy, "xmms2_command");
}

static void
ma_xmms2_proxy_enable (MaProxy *proxy)
{
	MaXMMS2ProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	priv->conn = xmmsc_init ("MusicApplet");
	priv->source = NULL;
}

static void
ma_xmms2_proxy_disable (MaProxy *proxy)
{
	MaXMMS2ProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	disconnect_cb (MA_XMMS2_PROXY (proxy));

	if (priv->conn != NULL)
	{
		xmmsc_unref (priv->conn);
		priv->conn = NULL;
	}
}

static void
ma_xmms2_proxy_poll_disconnected (MaProxy *proxy)
{
	MaXMMS2ProxyPrivate *priv;

	MaConf *conf;
	gchar *conn_path;
	int connected;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_XMMS2_PROXY (proxy));

	priv = GET_PRIVATE (proxy);
	conf = _ma_proxy_get_conf (proxy);

	/* Try to connect to the XMMS2 daemon. */

	conn_path = ma_conf_get_string (conf, "xmms2_conn_path");
	if (conn_path != NULL && conn_path[0] != '\0')
		connected = xmmsc_connect (priv->conn, conn_path);
	else
		connected = xmmsc_connect (priv->conn, g_getenv ("XMMS_PATH"));
	g_free (conn_path);

	if (!connected)
		return;

	/* Connect broadcast and signal handlers, if necessary. */

	priv->source = xmmsc_mainloop_gmain_init (priv->conn);

	xmmsc_disconnect_callback_set (priv->conn, disconnect_cb, proxy);

	priv->bcast_id = xmmsc_broadcast_playback_current_id (priv->conn);
	xmmsc_result_notifier_set (priv->bcast_id, current_id_cb, proxy);

	priv->bcast_entry = xmmsc_broadcast_medialib_entry_changed (priv->conn);
	xmmsc_result_notifier_set (priv->bcast_entry, entry_changed_cb, proxy);

	priv->sig_playtime = xmmsc_signal_playback_playtime (priv->conn);
	xmmsc_result_notifier_set (priv->sig_playtime, playtime_cb, proxy);

	priv->bcast_status = xmmsc_broadcast_playback_status (priv->conn);
	xmmsc_result_notifier_set (priv->bcast_status, status_cb, proxy);

	/* Issue requests for current status. */

	priv->call_id = xmmsc_playback_current_id (priv->conn);
	xmmsc_result_notifier_set (priv->call_id, current_id_cb, proxy);

	priv->call_playtime = xmmsc_playback_playtime (priv->conn);
	xmmsc_result_notifier_set (priv->call_playtime, playtime_cb, proxy);

	priv->call_status = xmmsc_playback_status (priv->conn);
	xmmsc_result_notifier_set (priv->call_status, status_cb, proxy);

	_ma_proxy_set_connected (proxy, TRUE);
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static void
cancel_result (xmmsc_result_t **res)
{
	if (*res != NULL)
	{
		xmmsc_result_disconnect (*res);
		xmmsc_result_unref (*res);
		*res = NULL;
	}
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
disconnect_cb (void *proxy)
{
	MaXMMS2ProxyPrivate *priv = GET_PRIVATE (proxy);

	cancel_result (&priv->bcast_id);
	cancel_result (&priv->bcast_entry);
	cancel_result (&priv->bcast_status);

	cancel_result (&priv->sig_playtime);

	cancel_result (&priv->call_id);
	cancel_result (&priv->call_playtime);
	cancel_result (&priv->call_status);
	cancel_result (&priv->call_minfo);

	if (priv->source != NULL)
	{
		xmmsc_mainloop_gmain_shutdown (priv->conn, priv->source);
		priv->source = NULL;
	}

	_ma_proxy_set_connected (MA_PROXY (proxy), FALSE);
}

static void
minfo_cb (xmmsc_result_t *res, void *proxy)
{
	static const gchar *const pref[] = { "server", "clients/generic", NULL };

	MaXMMS2ProxyPrivate *priv = GET_PRIVATE (proxy);

	gchar *title;
	gchar *artist;
	gchar *album;
	gchar *rating;
	gint duration = 0;

	xmmsc_result_get_dict_entry_str (res, "title", &title);

	if (title != NULL)
	{
		xmmsc_result_get_dict_entry_str (res, "artist", &artist);
		xmmsc_result_get_dict_entry_str (res, "album", &album);
		xmmsc_result_get_dict_entry_int32 (res, "duration", &duration);

		_ma_proxy_set_title (MA_PROXY (proxy), title);
		_ma_proxy_set_artist (MA_PROXY (proxy), artist);
		_ma_proxy_set_album (MA_PROXY (proxy), album);
		_ma_proxy_set_duration (MA_PROXY (proxy), duration / 1000);

		xmmsc_result_source_preference_set (res, (gchar **) pref);

		if (xmmsc_result_get_dict_entry_str (res, "rating", &rating))
			_ma_proxy_set_rating (MA_PROXY (proxy), strtol (rating, NULL, 10));
		else
			_ma_proxy_set_rating (MA_PROXY (proxy), 0.0);
	}
	else
		_ma_proxy_set_no_song (MA_PROXY (proxy));

	/* Always called directly. */

	xmmsc_result_unref (priv->call_minfo);
	priv->call_minfo = NULL;
	xmmsc_result_unref (res);
}

static void
current_id_cb (xmmsc_result_t *res, void *proxy)
{
	MaXMMS2ProxyPrivate *priv = GET_PRIVATE (proxy);

	xmmsc_result_get_uint (res, &priv->current_id);

	if (priv->call_minfo == NULL)
	{
		priv->call_minfo = xmmsc_medialib_get_info (priv->conn, priv->current_id);
		xmmsc_result_notifier_set (priv->call_minfo, minfo_cb, proxy);
	}

	if (res == priv->call_id)
	{
		/* Called directly. */

		xmmsc_result_unref (priv->call_id);
		priv->call_id = NULL;
		xmmsc_result_unref (res);
	}

	/* No cleanup if broadcast. */
}

static void
playtime_cb (xmmsc_result_t *res, void *proxy)
{
	MaXMMS2ProxyPrivate *priv = GET_PRIVATE (proxy);

	guint elapsed;

	xmmsc_result_get_uint (res, &elapsed);

	_ma_proxy_set_elapsed (MA_PROXY (proxy), elapsed / 1000);

	if (xmmsc_result_get_class (res) == XMMSC_RESULT_CLASS_SIGNAL)
	{
		/* A signal callback. */

		xmmsc_result_unref (priv->sig_playtime);
		priv->sig_playtime = xmmsc_result_restart (res);
		xmmsc_result_unref (res);
	}
	else if (res == priv->call_playtime)
	{
		/* Called directly. */

		xmmsc_result_unref (priv->call_playtime);
		priv->call_playtime = NULL;
		xmmsc_result_unref (res);
	}
}

static void
entry_changed_cb (xmmsc_result_t *res, void *proxy)
{
	MaXMMS2ProxyPrivate *priv = GET_PRIVATE (proxy);

	guint id;

	xmmsc_result_get_uint (res, &id);

	if (priv->current_id == id)
	{
		cancel_result (&priv->call_minfo);
		priv->call_minfo = xmmsc_medialib_get_info (priv->conn, priv->current_id);
		xmmsc_result_notifier_set (priv->call_minfo, minfo_cb, proxy);
	}

	/* Always a broadcast callback -- no cleanup. */
}

static void
status_cb (xmmsc_result_t *res, void *proxy)
{
	MaXMMS2ProxyPrivate *priv = GET_PRIVATE (proxy);

	guint status;

	xmmsc_result_get_uint (res, &status);

	if (status == XMMS_PLAYBACK_STATUS_STOP)
		_ma_proxy_set_no_song (MA_PROXY (proxy));

	_ma_proxy_set_playing (MA_PROXY (proxy), (status == XMMS_PLAYBACK_STATUS_PLAY));

	if (res == priv->call_status)
	{
		/* Called directly. */

		xmmsc_result_unref (priv->call_status);
		priv->call_status = NULL;
		xmmsc_result_unref (res);
	}

	/* No cleanup if broadcast. */
}
