/*
 * Implementation of the proxy around Rhythmbox's D-Bus interface.
 *
 * Music Applet
 * Copyright (C) 2004-2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include <config.h>

#include "ma-rhythmbox-dbus-proxy.h"

#include <string.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_RHYTHMBOX_DBUS_PROXY, MaRhythmboxDBusProxyPrivate))


typedef struct _MaRhythmboxDBusProxyPrivate	MaRhythmboxDBusProxyPrivate;


struct _MaRhythmboxDBusProxyPrivate
{
	DBusGProxy *player_proxy;
	DBusGProxy *shell_proxy;

	gchar *uri;

	DBusGProxyCall *call_playing;
	DBusGProxyCall *call_uri;
	DBusGProxyCall *call_elapsed;
	DBusGProxyCall *call_song;
};

static MaDBusProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_rhythmbox_dbus_proxy_class_init (MaRhythmboxDBusProxyClass *klass);
static void ma_rhythmbox_dbus_proxy_init (MaRhythmboxDBusProxy *rproxy);

static void ma_rhythmbox_dbus_proxy_dispose (GObject *object);
static void ma_rhythmbox_dbus_proxy_finalize (GObject *object);

static void ma_rhythmbox_dbus_proxy_toggle_playback (MaProxy *proxy);
static void ma_rhythmbox_dbus_proxy_previous (MaProxy *proxy);
static void ma_rhythmbox_dbus_proxy_next (MaProxy *proxy);

static void ma_rhythmbox_dbus_proxy_rate_song (MaProxy *proxy, gdouble rating);

static void ma_rhythmbox_dbus_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml);

static void ma_rhythmbox_dbus_proxy_launch (MaProxy *proxy);

static void ma_rhythmbox_dbus_proxy_connect (MaDBusProxy *dproxy, DBusGConnection *connection);
static void ma_rhythmbox_dbus_proxy_disconnect (MaDBusProxy *dproxy);

static void cache_song_metadata (MaRhythmboxDBusProxy *rproxy, const gchar *uri);

static void playing_changed_cb (DBusGProxy *player_proxy,
				gboolean playing,
				MaRhythmboxDBusProxy *rproxy);

static void get_playing_cb (DBusGProxy *player_proxy,
			    DBusGProxyCall *call,
			    gpointer proxy);

static void playing_uri_changed_cb (DBusGProxy *player_proxy,
				    const gchar *uri,
				    MaRhythmboxDBusProxy *rproxy);

static void get_playing_uri_cb (DBusGProxy *player_proxy,
				DBusGProxyCall *call,
				gpointer proxy);

static void elapsed_changed_cb (DBusGProxy *player_proxy,
				guint elapsed,
				MaRhythmboxDBusProxy *rproxy);

static void get_elapsed_cb (DBusGProxy *player_proxy,
			    DBusGProxyCall *call,
			    gpointer proxy);

static void get_song_properties_cb (DBusGProxy *shell_proxy,
				    DBusGProxyCall *call,
				    gpointer proxy);

/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_rhythmbox_dbus_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaRhythmboxDBusProxyClass),			/* class_size */
			NULL,							/* base_init */
			NULL,							/* base_finalize */
			(GClassInitFunc) ma_rhythmbox_dbus_proxy_class_init,	/* class_init */
			NULL,							/* class_finalize */
			NULL,							/* class_data */
			sizeof (MaRhythmboxDBusProxy),				/* instance_size */
			0,							/* n_preallocs */
			(GInstanceInitFunc) ma_rhythmbox_dbus_proxy_init,	/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_DBUS_PROXY, "MaRhythmboxDBusProxy", &info, 0);
	}

	return type;
}

static void
ma_rhythmbox_dbus_proxy_class_init (MaRhythmboxDBusProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	MaDBusProxyClass *dproxy_class = (MaDBusProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_rhythmbox_dbus_proxy_dispose;
	object_class->finalize = ma_rhythmbox_dbus_proxy_finalize;

	proxy_class->toggle_playback = ma_rhythmbox_dbus_proxy_toggle_playback;
	proxy_class->previous = ma_rhythmbox_dbus_proxy_previous;
	proxy_class->next = ma_rhythmbox_dbus_proxy_next;

	proxy_class->rate_song = ma_rhythmbox_dbus_proxy_rate_song;

	proxy_class->prepare_prefs = ma_rhythmbox_dbus_proxy_prepare_prefs;

	proxy_class->launch = ma_rhythmbox_dbus_proxy_launch;

	dproxy_class->connect = ma_rhythmbox_dbus_proxy_connect;
	dproxy_class->disconnect = ma_rhythmbox_dbus_proxy_disconnect;

	g_type_class_add_private (klass, sizeof (MaRhythmboxDBusProxyPrivate));
}

static void
ma_rhythmbox_dbus_proxy_init (MaRhythmboxDBusProxy *rproxy)
{
	MaRhythmboxDBusProxyPrivate *priv = GET_PRIVATE (rproxy);

	priv->player_proxy = NULL;
	priv->shell_proxy = NULL;

	priv->uri = NULL;

	priv->call_playing = NULL;
	priv->call_uri = NULL;
	priv->call_elapsed = NULL;
	priv->call_song = NULL;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

MaProxy *
ma_rhythmbox_dbus_proxy_new (MaConf *conf)
{
	MaProxy *proxy;

	g_return_val_if_fail (conf != NULL, NULL);

	proxy = g_object_new (MA_TYPE_RHYTHMBOX_DBUS_PROXY,
			      "name", "Rhythmbox",
			      "icon-name", "rhythmbox",
			      "conf", conf,
			      "dbus-name", "org.gnome.Rhythmbox",
			      NULL);

	return proxy;
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_rhythmbox_dbus_proxy_dispose (GObject *object)
{
	ma_rhythmbox_dbus_proxy_disconnect (MA_DBUS_PROXY (object));

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
ma_rhythmbox_dbus_proxy_finalize (GObject *object)
{
	/* priv->uri freed in _disconnect, called at dispose time */

	G_OBJECT_CLASS (parent_class)->finalize (object);
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_rhythmbox_dbus_proxy_toggle_playback (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_DBUS_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "playPause",
				    G_TYPE_BOOLEAN, FALSE,
				    G_TYPE_INVALID);
}

static void
ma_rhythmbox_dbus_proxy_previous (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_DBUS_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "previous",
				    G_TYPE_INVALID);
}

static void
ma_rhythmbox_dbus_proxy_next (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_DBUS_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "next",
				    G_TYPE_INVALID);
}

static void
ma_rhythmbox_dbus_proxy_rate_song (MaProxy *proxy, gdouble rating)
{
	MaRhythmboxDBusProxyPrivate *priv;
	GValue rating_value;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_DBUS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	memset (&rating_value, 0, sizeof (rating_value));
	g_value_init (&rating_value, G_TYPE_DOUBLE);
	g_value_set_double (&rating_value, rating);

	dbus_g_proxy_call_no_reply (priv->shell_proxy, "setSongProperty",
				    G_TYPE_STRING, priv->uri,
				    G_TYPE_STRING, "rating",
				    G_TYPE_VALUE, &rating_value,
				    G_TYPE_INVALID);
}

static void
ma_rhythmbox_dbus_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	/* Nothing to do? */
}

static void
ma_rhythmbox_dbus_proxy_launch (MaProxy *proxy)
{
	/* MaRhythmboxProxy handles the non-D-Bus case. */

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_DBUS_PROXY (proxy));

	_ma_dbus_proxy_launch (MA_DBUS_PROXY (proxy));
}


/*********************************************************************
 *
 * MaDBusProxy overrides
 *
 *********************************************************************/

static void
ma_rhythmbox_dbus_proxy_connect (MaDBusProxy *dproxy, DBusGConnection *connection)
{
	MaRhythmboxDBusProxyPrivate *priv;

	g_return_if_fail (dproxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_DBUS_PROXY (dproxy));

	priv = GET_PRIVATE (dproxy);

	priv->player_proxy = dbus_g_proxy_new_for_name (connection,
							"org.gnome.Rhythmbox",
							"/org/gnome/Rhythmbox/Player",
							"org.gnome.Rhythmbox.Player");

	dbus_g_proxy_add_signal (priv->player_proxy, "playingChanged",
				 G_TYPE_BOOLEAN,
				 G_TYPE_INVALID);

	dbus_g_proxy_connect_signal (priv->player_proxy, "playingChanged",
				     G_CALLBACK (playing_changed_cb), dproxy, NULL);

	dbus_g_proxy_add_signal (priv->player_proxy, "playingUriChanged",
				 G_TYPE_STRING,
				 G_TYPE_INVALID);

	dbus_g_proxy_connect_signal (priv->player_proxy, "playingUriChanged",
				     G_CALLBACK (playing_uri_changed_cb), dproxy, NULL);

	dbus_g_proxy_add_signal (priv->player_proxy, "elapsedChanged",
				 G_TYPE_UINT,
				 G_TYPE_INVALID);

	dbus_g_proxy_connect_signal (priv->player_proxy, "elapsedChanged",
				     G_CALLBACK (elapsed_changed_cb), dproxy, NULL);

	priv->shell_proxy = dbus_g_proxy_new_for_name (connection,
						       "org.gnome.Rhythmbox",
						       "/org/gnome/Rhythmbox/Shell",
						       "org.gnome.Rhythmbox.Shell");

	priv->call_playing = dbus_g_proxy_begin_call (priv->player_proxy, "getPlaying",
						      get_playing_cb, dproxy, NULL,
						      G_TYPE_INVALID);

	priv->call_uri = dbus_g_proxy_begin_call (priv->player_proxy, "getPlayingUri",
						  get_playing_uri_cb, dproxy, NULL,
						  G_TYPE_INVALID);

	priv->call_elapsed = dbus_g_proxy_begin_call (priv->player_proxy, "getElapsed",
						      get_elapsed_cb, dproxy, NULL,
						      G_TYPE_INVALID);
}

static void
ma_rhythmbox_dbus_proxy_disconnect (MaDBusProxy *dproxy)
{
	MaRhythmboxDBusProxyPrivate *priv;

	g_return_if_fail (dproxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_DBUS_PROXY (dproxy));

	priv = GET_PRIVATE (dproxy);

	g_free (priv->uri);
	priv->uri = NULL;

	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_playing);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_uri);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_elapsed);
	_ma_dbus_proxy_cancel_call (priv->shell_proxy, &priv->call_song);

	if (priv->shell_proxy != NULL)
	{
		g_object_unref (priv->shell_proxy);
		priv->shell_proxy = NULL;
	}

	if (priv->player_proxy != NULL)
	{
		dbus_g_proxy_disconnect_signal (priv->player_proxy, "playingChanged",
						G_CALLBACK (playing_changed_cb), dproxy);

		dbus_g_proxy_disconnect_signal (priv->player_proxy, "playingUriChanged",
						G_CALLBACK (playing_uri_changed_cb), dproxy);

		dbus_g_proxy_disconnect_signal (priv->player_proxy, "elapsedChanged",
						G_CALLBACK (elapsed_changed_cb), dproxy);

		g_object_unref (priv->player_proxy);
		priv->player_proxy = NULL;
	}
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static void
cache_song_metadata (MaRhythmboxDBusProxy *rproxy, const gchar *uri)
{
	MaRhythmboxDBusProxyPrivate *priv = GET_PRIVATE (rproxy);

	g_free (priv->uri);

	if (uri == NULL || uri[0] == '\0')
	{
		priv->uri = NULL;
		_ma_proxy_set_no_song (MA_PROXY (rproxy));
	}
	else
	{
		priv->uri = g_strdup (uri);
		priv->call_song = dbus_g_proxy_begin_call (priv->shell_proxy, "getSongProperties",
							   get_song_properties_cb, rproxy, NULL,
							   G_TYPE_STRING, uri,
							   G_TYPE_INVALID);
	}
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
playing_changed_cb (DBusGProxy *player_proxy,
		    gboolean playing,
		    MaRhythmboxDBusProxy *rproxy)
{
	_ma_proxy_set_playing (MA_PROXY (rproxy), playing);

	/**
	 * Since uri_changed_cb doesn't get called when playback finishes,
	 * we have to check for not-playing AND null-elapsed-time.
	 */

	if (!playing && ma_proxy_get_elapsed (MA_PROXY (rproxy)) == -1)
		_ma_proxy_set_no_song (MA_PROXY (rproxy));
}

static void
get_playing_cb (DBusGProxy *player_proxy,
		DBusGProxyCall *call,
		gpointer proxy)
{
	MaRhythmboxDBusProxyPrivate *priv = GET_PRIVATE (proxy);

	gboolean playing = FALSE;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_BOOLEAN, &playing,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_playing (MA_PROXY (proxy), playing);
	}
	else
		_ma_dbus_proxy_report_error ("getPlaying", &error);

	priv->call_playing = NULL;
}

static void
playing_uri_changed_cb (DBusGProxy *player_proxy,
			const gchar *uri,
			MaRhythmboxDBusProxy *rproxy)
{
	cache_song_metadata (rproxy, uri);
}

static void
get_playing_uri_cb (DBusGProxy *player_proxy,
		    DBusGProxyCall *call,
		    gpointer proxy)
{
	MaRhythmboxDBusProxyPrivate *priv = GET_PRIVATE (proxy);

	gchar *uri = NULL;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_STRING, &uri,
				   G_TYPE_INVALID))
	{
		cache_song_metadata (MA_RHYTHMBOX_DBUS_PROXY (proxy), uri);
		g_free (uri);
	}
	else
		_ma_dbus_proxy_report_error ("getPlayingUri", &error);

	priv->call_uri = NULL;
}

static void
elapsed_changed_cb (DBusGProxy *player_proxy,
		    guint elapsed,
		    MaRhythmboxDBusProxy *rproxy)
{
	_ma_proxy_set_elapsed (MA_PROXY (rproxy), elapsed);

	/**
	 * Since uri_changed_cb doesn't get called when playback finishes,
	 * we have to check for not-playing AND null-elapsed-time.
	 */

	if (elapsed == (guint) -1 && !ma_proxy_get_playing (MA_PROXY (rproxy)))
		_ma_proxy_set_no_song (MA_PROXY (rproxy));
}

static void
get_elapsed_cb (DBusGProxy *player_proxy,
		DBusGProxyCall *call,
		gpointer proxy)
{
	MaRhythmboxDBusProxyPrivate *priv = GET_PRIVATE (proxy);

	glong elapsed = -1;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_UINT, &elapsed,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_elapsed (MA_PROXY (proxy), elapsed);
	}
	else
		_ma_dbus_proxy_report_error ("getElapsed", &error);

	priv->call_elapsed = NULL;
}

static void
get_song_properties_cb (DBusGProxy *shell_proxy,
			DBusGProxyCall *call,
			gpointer proxy)
{
	MaRhythmboxDBusProxyPrivate *priv = GET_PRIVATE (proxy);

	GHashTable *table;
	GValue *value;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (shell_proxy, call, &error,
				   dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE), &table,
				   G_TYPE_INVALID))
	{
		value = (GValue *) g_hash_table_lookup (table, "title");
		if (value != NULL && G_VALUE_HOLDS_STRING (value))
			_ma_proxy_set_title (MA_PROXY (proxy), g_value_get_string (value));

		value = (GValue *) g_hash_table_lookup (table, "artist");
		if (value != NULL && G_VALUE_HOLDS_STRING (value))
		{
			const gchar *artist = g_value_get_string (value);
			_ma_proxy_set_artist (MA_PROXY (proxy), (artist[0] != '\0') ? artist : NULL);
		}

		value = (GValue *) g_hash_table_lookup (table, "album");
		if (value != NULL && G_VALUE_HOLDS_STRING (value))
		{
			const gchar *album = g_value_get_string (value);
			_ma_proxy_set_album (MA_PROXY (proxy), (album[0] != '\0') ? album : NULL);
		}

		value = (GValue *) g_hash_table_lookup (table, "duration");
		if (value != NULL && G_VALUE_HOLDS_UINT (value))
		{
			gint duration = g_value_get_uint (value);
			_ma_proxy_set_duration (MA_PROXY (proxy), duration);
		}

		value = (GValue *) g_hash_table_lookup (table, "rating");
		if (value != NULL && G_VALUE_HOLDS_DOUBLE (value))
			_ma_proxy_set_rating (MA_PROXY (proxy), g_value_get_double (value));
	}
	else
		_ma_dbus_proxy_report_error ("getSongProperties", &error);

	priv->call_song = NULL;
}
