/*
 * Implementation of the proxy around Banshee's D-Bus interface.
 *
 * Music Applet
 * Copyright (C) 2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include <config.h>

#include "ma-banshee-proxy.h"

#include <string.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_BANSHEE_PROXY, MaBansheeProxyPrivate))


typedef struct _MaBansheeProxyPrivate	MaBansheeProxyPrivate;


struct _MaBansheeProxyPrivate
{
	DBusGProxy *player_proxy;

	gchar *uri;
	gboolean requery_duration;

	DBusGProxyCall *call_uri;
	DBusGProxyCall *call_status;
	DBusGProxyCall *call_position;

	DBusGProxyCall *call_title;
	DBusGProxyCall *call_artist;
	DBusGProxyCall *call_album;
	DBusGProxyCall *call_duration;
};

static MaDBusProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_banshee_proxy_class_init (MaBansheeProxyClass *klass);
static void ma_banshee_proxy_init (MaBansheeProxy *bproxy);

static void ma_banshee_proxy_dispose (GObject *object);
static void ma_banshee_proxy_finalize (GObject *object);

static void ma_banshee_proxy_toggle_playback (MaProxy *proxy);
static void ma_banshee_proxy_previous (MaProxy *proxy);
static void ma_banshee_proxy_next (MaProxy *proxy);

static void ma_banshee_proxy_rate_song (MaProxy *proxy, gdouble rating);

static void ma_banshee_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml);

static void ma_banshee_proxy_launch (MaProxy *proxy);

static void ma_banshee_proxy_poll_connected (MaProxy *proxy);

static void ma_banshee_proxy_connect (MaDBusProxy *dproxy, DBusGConnection *connection);
static void ma_banshee_proxy_disconnect (MaDBusProxy *dproxy);

static void get_playing_uri_cb (DBusGProxy *player_proxy,
				DBusGProxyCall *call,
				gpointer proxy);

static void get_playing_status_cb (DBusGProxy *player_proxy,
				   DBusGProxyCall *call,
				   gpointer proxy);

static void get_playing_position_cb (DBusGProxy *player_proxy,
				     DBusGProxyCall *call,
				     gpointer proxy);

static void get_playing_title_cb (DBusGProxy *player_proxy,
				  DBusGProxyCall *call,
				  gpointer proxy);

static void get_playing_artist_cb (DBusGProxy *player_proxy,
				   DBusGProxyCall *call,
				   gpointer proxy);

static void get_playing_album_cb (DBusGProxy *player_proxy,
				  DBusGProxyCall *call,
				  gpointer proxy);

static void get_playing_duration_cb (DBusGProxy *player_proxy,
				     DBusGProxyCall *call,
				     gpointer proxy);

static void command_toggled_cb (GtkToggleButton *button, GtkWidget *widget);

/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_banshee_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaBansheeProxyClass),			/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_banshee_proxy_class_init,	/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaBansheeProxy),			/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_banshee_proxy_init,	/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_DBUS_PROXY, "MaBansheeProxy", &info, 0);
	}

	return type;
}

static void
ma_banshee_proxy_class_init (MaBansheeProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	MaDBusProxyClass *dproxy_class = (MaDBusProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_banshee_proxy_dispose;
	object_class->finalize = ma_banshee_proxy_finalize;

	proxy_class->toggle_playback = ma_banshee_proxy_toggle_playback;
	proxy_class->previous = ma_banshee_proxy_previous;
	proxy_class->next = ma_banshee_proxy_next;

	proxy_class->rate_song = ma_banshee_proxy_rate_song;

	proxy_class->prepare_prefs = ma_banshee_proxy_prepare_prefs;

	proxy_class->launch = ma_banshee_proxy_launch;

	proxy_class->poll_connected = ma_banshee_proxy_poll_connected;

	dproxy_class->connect = ma_banshee_proxy_connect;
	dproxy_class->disconnect = ma_banshee_proxy_disconnect;

	g_type_class_add_private (klass, sizeof (MaBansheeProxyPrivate));
}

static void
ma_banshee_proxy_init (MaBansheeProxy *bproxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (bproxy);

	priv->player_proxy = NULL;

	priv->uri = NULL;
	priv->requery_duration = FALSE;

	priv->call_uri = NULL;
	priv->call_status = NULL;
	priv->call_position = NULL;

	priv->call_title = NULL;
	priv->call_artist = NULL;
	priv->call_album = NULL;
	priv->call_duration = NULL;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

MaProxy *
ma_banshee_proxy_new (MaConf *conf)
{
	MaProxy *proxy;

	g_return_val_if_fail (conf != NULL, NULL);

	proxy = g_object_new (MA_TYPE_BANSHEE_PROXY,
			      "name", "Banshee",
			      "icon-name", "music-player-banshee",
			      "conf", conf,
			      "dbus-name", "org.gnome.Banshee",
			      NULL);

	return proxy;
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_banshee_proxy_dispose (GObject *object)
{
	ma_banshee_proxy_disconnect (MA_DBUS_PROXY (object));

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
ma_banshee_proxy_finalize (GObject *object)
{
	/* priv->uri freed in _disconnect, called at dispose time */

	G_OBJECT_CLASS (parent_class)->finalize (object);
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_banshee_proxy_toggle_playback (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "TogglePlaying",
				    G_TYPE_INVALID);
}

static void
ma_banshee_proxy_previous (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "Previous",
				    G_TYPE_INVALID);
}

static void
ma_banshee_proxy_next (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (proxy));

	dbus_g_proxy_call_no_reply (GET_PRIVATE (proxy)->player_proxy, "Next",
				    G_TYPE_INVALID);
}

static void
ma_banshee_proxy_rate_song (MaProxy *proxy, gdouble rating)
{
	g_warning ("Banshee does not support changing the current song's rating");
}

static void
ma_banshee_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	MaConf *conf;

	GtkWidget *page;
	GtkWidget *launch_dbus;
	GtkWidget *launch_command;
	GtkWidget *command;
	GtkWidget *browse;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (proxy));

	conf = _ma_proxy_get_conf (proxy);

	page = glade_xml_get_widget (xml, "banshee");
	launch_dbus = glade_xml_get_widget (xml, "banshee-launch-dbus");
	launch_command = glade_xml_get_widget (xml, "banshee-launch-command");
	command = glade_xml_get_widget (xml, "banshee-command");
	browse = glade_xml_get_widget (xml, "banshee-browse");

	ma_conf_bind_string_radio (conf, "banshee_launch", "D-Bus", GTK_RADIO_BUTTON (launch_dbus));
	ma_conf_bind_string_radio (conf, "banshee_launch", "Command", GTK_RADIO_BUTTON (launch_command));
	ma_conf_bind_string_entry (conf, "banshee_command", GTK_ENTRY (command));
	ma_conf_bind_string_browse (conf, "banshee_command", GTK_BUTTON (browse));

	gtk_widget_set_sensitive (command, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (launch_command)));
	gtk_widget_set_sensitive (browse, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (launch_command)));

	g_signal_connect (launch_command, "toggled", G_CALLBACK (command_toggled_cb), command);
	g_signal_connect (launch_command, "toggled", G_CALLBACK (command_toggled_cb), browse);

	gtk_widget_show (page);
}

static void
ma_banshee_proxy_launch (MaProxy *proxy)
{
	MaConf *conf;
	gchar *launch;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (proxy));

	conf = _ma_proxy_get_conf (proxy);
	launch = ma_conf_get_string (conf, "banshee_launch");

	if (strcmp (launch, "Command") == 0)
		_ma_proxy_launch_command (proxy, "banshee_command");
	else
		_ma_dbus_proxy_launch (MA_DBUS_PROXY (proxy));

	g_free (launch);
}

static void
ma_banshee_proxy_poll_connected (MaProxy *proxy)
{
	MaBansheeProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	if (priv->call_uri == NULL)
	{
		priv->call_uri = dbus_g_proxy_begin_call (priv->player_proxy, "GetPlayingUri",
							  get_playing_uri_cb, proxy, NULL,
							  G_TYPE_INVALID);
	}

	if (priv->call_status == NULL)
	{
		priv->call_status = dbus_g_proxy_begin_call (priv->player_proxy, "GetPlayingStatus",
							     get_playing_status_cb, proxy, NULL,
							     G_TYPE_INVALID);
	}

	if (priv->call_position == NULL)
	{
		priv->call_position = dbus_g_proxy_begin_call (priv->player_proxy, "GetPlayingPosition",
							       get_playing_position_cb, proxy, NULL,
							       G_TYPE_INVALID);
	}

	if (priv->requery_duration && priv->call_duration == NULL)
	{
		priv->call_duration = dbus_g_proxy_begin_call (priv->player_proxy, "GetPlayingDuration",
							       get_playing_duration_cb, proxy, NULL,
							       G_TYPE_INVALID);
		priv->requery_duration = FALSE;
	}

	/*
	 * When playing an Internet radio stream, periodically refresh the
	 * title information, and hope we don't run afowl of a Banshee 0.10.10
	 * and earlier crasher if it's not available yet.
	 * (See http://bugzilla.gnome.org/show_bug.cgi?id=344774)
	 */

	if (priv->call_title == NULL &&
	    priv->uri != NULL && g_str_has_prefix (priv->uri, "http://") &&
	    _ma_proxy_should_refresh_stream_metadata (proxy))
	{
		priv->call_title = dbus_g_proxy_begin_call (priv->player_proxy, "GetPlayingTitle",
							    get_playing_title_cb, proxy, NULL,
							    G_TYPE_INVALID);
	}
}


/*********************************************************************
 *
 * MaDBusProxy overrides
 *
 *********************************************************************/

static void
ma_banshee_proxy_connect (MaDBusProxy *dproxy, DBusGConnection *connection)
{
	g_return_if_fail (dproxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (dproxy));

	GET_PRIVATE (dproxy)->player_proxy = dbus_g_proxy_new_for_name (connection,
									"org.gnome.Banshee",
									"/org/gnome/Banshee/Player",
									"org.gnome.Banshee.Core");
}

static void
ma_banshee_proxy_disconnect (MaDBusProxy *dproxy)
{
	MaBansheeProxyPrivate *priv;

	g_return_if_fail (dproxy != NULL);
	g_return_if_fail (MA_IS_BANSHEE_PROXY (dproxy));

	priv = GET_PRIVATE (dproxy);

	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_uri);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_status);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_position);

	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_title);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_artist);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_album);
	_ma_dbus_proxy_cancel_call (priv->player_proxy, &priv->call_duration);

	if (priv->player_proxy != NULL)
	{
		g_object_unref (priv->player_proxy);
		priv->player_proxy = NULL;
	}

	if (priv->uri != NULL)
	{
		g_free (priv->uri);
		priv->uri = NULL;
	}
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
get_playing_uri_cb (DBusGProxy *player_proxy,
		    DBusGProxyCall *call,
		    gpointer proxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (proxy);

	gchar *uri;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_STRING, &uri,
				   G_TYPE_INVALID))
	{
		if (uri != NULL && uri[0] != '\0' && (priv->uri == NULL || strcmp(priv->uri, uri) != 0))
		{
			g_free (priv->uri);
			priv->uri = uri;

			/*
			 * Banshee 0.10.10 and earlier can crash if a string-returning
			 * function is called when playing a Internet radio stream.
			 * (See http://bugzilla.gnome.org/show_bug.cgi?id=344774)
			 * So, don't make those calls if the URI looks like a stream.
			 * And since streams have no duration, there's no point making
			 * that call either, while we're at it.
			 */

			if (!g_str_has_prefix (priv->uri, "http://"))
			{
				_ma_dbus_proxy_cancel_call (player_proxy, &priv->call_title);
				priv->call_title = dbus_g_proxy_begin_call (player_proxy, "GetPlayingTitle",
									    get_playing_title_cb, proxy, NULL,
									    G_TYPE_INVALID);

				_ma_dbus_proxy_cancel_call (player_proxy, &priv->call_artist);
				priv->call_artist = dbus_g_proxy_begin_call (player_proxy, "GetPlayingArtist",
									     get_playing_artist_cb, proxy, NULL,
									     G_TYPE_INVALID);

				_ma_dbus_proxy_cancel_call (player_proxy, &priv->call_album);
				priv->call_album = dbus_g_proxy_begin_call (player_proxy, "GetPlayingAlbum",
									    get_playing_album_cb, proxy, NULL,
									    G_TYPE_INVALID);

				_ma_dbus_proxy_cancel_call (player_proxy, &priv->call_duration);
				priv->call_duration = dbus_g_proxy_begin_call (player_proxy, "GetPlayingDuration",
									       get_playing_duration_cb, proxy, NULL,
									       G_TYPE_INVALID);
			}
			else
			{
				_ma_proxy_set_title (MA_PROXY (proxy), _("Unknown"));
				_ma_proxy_set_artist (MA_PROXY (proxy), NULL);
				_ma_proxy_set_album (MA_PROXY (proxy), NULL);
				_ma_proxy_set_duration (MA_PROXY (proxy), 0);
			}
		}
		else if ((uri == NULL || uri[0] == '\0') && priv->uri != NULL)
		{
			g_free (priv->uri);
			priv->uri = NULL;
			_ma_proxy_set_no_song (MA_PROXY (proxy));
		}
	}
	else
		_ma_dbus_proxy_report_error ("GetPlayingUri", &error);

	priv->call_uri = NULL;
}

static void
get_playing_status_cb (DBusGProxy *player_proxy,
		       DBusGProxyCall *call,
		       gpointer proxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (proxy);

	gint playing;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_INT, &playing,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_playing (MA_PROXY (proxy), (playing == 1));
	}
	else
		_ma_dbus_proxy_report_error ("GetPlayingStatus", &error);

	priv->call_status = NULL;
}

static void
get_playing_position_cb (DBusGProxy *player_proxy,
			 DBusGProxyCall *call,
			 gpointer proxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (proxy);

	gint elapsed;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_INT, &elapsed,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_elapsed (MA_PROXY (proxy), elapsed);
	}
	else
		_ma_dbus_proxy_report_error ("GetPlayingElapsed", &error);

	priv->call_position = NULL;
}

static void
get_playing_title_cb (DBusGProxy *player_proxy,
		      DBusGProxyCall *call,
		      gpointer proxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (proxy);

	gchar *title;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_STRING, &title,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_title (MA_PROXY (proxy), title);
		g_free (title);
	}
	else
		_ma_dbus_proxy_report_error ("GetPlayingTitle", &error);

	priv->call_title = NULL;
}

static void
get_playing_artist_cb (DBusGProxy *player_proxy,
		       DBusGProxyCall *call,
		       gpointer proxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (proxy);

	gchar *artist;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_STRING, &artist,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_artist (MA_PROXY (proxy), artist);
		g_free (artist);
	}
	else
		_ma_dbus_proxy_report_error ("GetPlayingArtist", &error);

	priv->call_artist = NULL;
}

static void
get_playing_album_cb (DBusGProxy *player_proxy,
		      DBusGProxyCall *call,
		      gpointer proxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (proxy);

	gchar *album;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_STRING, &album,
				   G_TYPE_INVALID))
	{
		_ma_proxy_set_album (MA_PROXY (proxy), album);
		g_free (album);
	}
	else
		_ma_dbus_proxy_report_error ("GetPlayingAlbum", &error);

	priv->call_album = NULL;
}

static void
get_playing_duration_cb (DBusGProxy *player_proxy,
			 DBusGProxyCall *call,
			 gpointer proxy)
{
	MaBansheeProxyPrivate *priv = GET_PRIVATE (proxy);

	gint duration = -1;
	GError *error = NULL;

	if (dbus_g_proxy_end_call (player_proxy, call, &error,
				   G_TYPE_INT, &duration,
				   G_TYPE_INVALID))
	{
		/* Sometimes Banshee returns a bogus duration, for no apparent
		 * reason.  It will eventually return the correct value, but
		 * recalling immediately just wastes CPU time; Banshee spends
		 * its time sending replies instead of figuring out the
		 * duration.
		 */

		if (duration != 0)
			_ma_proxy_set_duration (MA_PROXY (proxy), duration);
		else
			priv->requery_duration = TRUE;
	}
	else
		_ma_dbus_proxy_report_error ("GetPlayingDuration", &error);

	priv->call_duration = NULL;
}

static void
command_toggled_cb (GtkToggleButton *button, GtkWidget *widget)
{
	gtk_widget_set_sensitive (widget, gtk_toggle_button_get_active (button));
}
