//   -*- mode: c++; -*-
/*
  
Grammar of "string"'s:
  cat    = value | (value cat)
  value  = const | func | var
  const  = "\"" .* "\""
  var    = "$"name
  func   = name "(" args ")"
  args   = cat | (args "," cat)
  name   = [a-z,A-Z,_][a-z,A-Z,_,0-9]*

Structure of container classes
  str         general container
    const     contains constant string
    cat       vector of str's
    var       contains a named variable (like section, title, needs, ...).
    func      a function (print, ifelse, iffile, ..). 
      func1   with 1 argument
      func2   with 2 arguments
      func3   with 3 arguments

Example "str":
  "title=" print($title) ", mess:" \
   ifelse($longtitle, print(title), $longtitle)
*/

#ifndef MENU_METHOD_H
#define MENU_METHOD_H

#include <iostream.h>
#include <vector.h>
#include <map.h>
#include <set.h>
#include "common.h"
#include "adstring.h"

//Exception classes

class narg_mismatch:public except_pi_String{ //number of args mismatch to func.
public:
  narg_mismatch(parsestream *p, String s):except_pi_String(p,s){};
  String message(){
    return Sprintf(_("Number of arguments to function %s doesn't match"),
		   msg);
  };
};  
class unknown_function:public except_pi_String{
public:
  unknown_function(parsestream *p, String s):except_pi_String(p,s){};
  String message(){
    return Sprintf(_("Unknown function: \"%s\""),
		   msg);
  };
};  
class unknown_indirect_function:public except_pi_String{
public:
  unknown_indirect_function(parsestream *p, String s):except_pi_String(p,s){};
  String message(){
    return Sprintf(_("Indirectly used, but not defined function: \"%s\""),
		   msg);
  };
};  
class unknown_ident: public except_pi_String{
public:
  unknown_ident(parsestream *p, String s):except_pi_String(p,s){};
  String message(){
    return Sprintf(_("Unknown identifier: \"%s\""),
		   msg);
  };
};  

class dir_createerror: public except_String{ //Cannot create dir
public:
  dir_createerror(String s):except_String(s){};
  String message(){
    return Sprintf(_("Could not open dir \"%s\""),msg);
  };
};  

class missing_tag:public except_pi_String{
public:
  missing_tag(parsestream *p, String s):except_pi_String(p,s){};
  String message(){
    return Sprintf(_("Missing (or empty) tag: %s\n"
		     "This tag needs to defined for the menuentry to make sense.\n"
		     "Note, BTW, that update-menus re-arranges the order of the\n"
                     "tags found in the menu entry files, so that the part above\n"
                     "isn't literal"),
		   msg);
  };
};  

class informed_fatal:public genexcept{};

//container classes:

class func;

class str{
public:
  virtual ostream &output(ostream &o, 
			  map<String, String, less<String> > &menuentry)=NULL;
  virtual ostream &debuginfo(ostream &)=NULL;
  virtual ~str();
};

class const_str:public str{
private:
  String data;
public:
  const_str(parsestream &);
  virtual ostream &output(ostream &o, 
				     map<String, String, less<String> > &menuentry);
  ostream &debuginfo(ostream &o);
};

class cat_str:public str{
private:
  vector<str *> v;
public:
  cat_str(parsestream &);
  ostream &output(ostream &o, 
		  map<String, String, less<String> > &menuentry);
  void output(map<String, String, less<String> > &menuentry);
  String  soutput(map<String, String, less<String> > &menuentry);
  ostream &debuginfo(ostream &o);
};

class var_str:public str{
private:
  String var_name;
public:
  ostream &output(ostream &o, 
		  map<String, String, less<String> > &menuentry);
  var_str(parsestream &i);
  ostream &debuginfo(ostream &o);
};

class func_str:public str{
private:
  vector<cat_str *> args;
  func *f;
public:
  func_str(parsestream &);
  ostream &output(ostream &o, 
		  map<String, String, less<String> > &menuentry);
  ostream &debuginfo(ostream &o);
};


/////////////////////////////////////////////////////
//  prototypes of install-menu functions:
//

class func{
public: 
  virtual int nargs()=NULL;
  virtual ostream &output(ostream &, vector<cat_str *> &,
			  map<String, String, less<String> > &)=NULL;
  virtual String name()=NULL;
};
class func0:public func{
public:
  int nargs(){return 0;};
};
class func1:public func{
public:
  int nargs(){return 1;};
};
class func2:public func{
public:
  int nargs(){return 2;};
};
class func3:public func{
public:
  int nargs(){return 3;};
};
class func4:public func{
public:
  int nargs(){return 4;};
};
class func_def:public func{
  cat_str *f;
  String n;
  StrVec args_name;
public:
  func_def(parsestream &i);
  int nargs(){return args_name.size();};
  ostream &output(ostream &, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return n;};
  virtual ~func_def();
};

/////////////////////////////////////////////////////
//  Function that can be used in install-menu (declarations)
//

class prefix_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "prefix";}
};
class ifroot_func: public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifroot";}
};

class print_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "print";}
};
class ifempty_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifempty";}
};
class ifnempty_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifnempty";}
};
class iffile_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "iffile";}
};
class ifelsefile_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifelsefile";}
};
class catfile_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "catfile";}
};
class forall_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "forall";}
};

class esc_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "esc";}
};
class add_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "add";}
};
class sub_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "sub";}
};
class mult_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "mult";}
};
class div_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "div";}
};


class ifelse_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifelse";}
};

class ifeq_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifeq";}
};
class ifneq_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifneq";}
};
class ifeqelse_func: public func4{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifeqelse";}
};

class cond_surr_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "cond_surr";}
};
class escwith_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "escwith";}
};
class escfirst_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "escfirst";}
};
class tolower_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "tolower";}
};
class toupper_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "toupper";}
};
class replacewith_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "replacewith";}
};
class nstring_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "nstring";}
};

class cppesc_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "cppesc";}
};
class parent_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "parent";}
};
class basename_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "basename";}
};
class stripdir_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "stripdir";}
};
class entrycount_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "entrycount";}
};
class entryindex_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "entryindex";}
};
class firstentry_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "firstentry";}
};
class lastentry_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "lastentry";}
};
class level_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "level";}
};


class rcfile_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "rcfile";}
};
class examplercfile_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "examplercfile";}
};
class mainmenutitle_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "mainmenutitle";}
};
class rootsection_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "rootsection";}
};
class rootprefix_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "rootprefix";}
};
class userprefix_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "userprefix";}
};
class treewalk_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "treewalk";}
};
class postoutput_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "postoutput";}
};
class preoutput_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "preoutput";}
};

class cwd_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "cwd";}
};

class translate_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "translate";}
};


/////////////////////////////////////////////////////
//  install-menu Function declarations
//


class supportedinfo {
private:
  class supinf{
  public:
    cat_str *c;
    int prec;
  };
  map<String, supinf, less<String> > sup;
public:
  supportedinfo(parsestream &);
  int prec(String &);
  void subst(map<String, String, less<String> > );
  ostream &debuginfo(ostream &);
};

class configinfo{
private:
  String compt,rcf, exrcf, mainmt, roots, treew;
  String preout, postout;

  void check_config();
public:
  bool keep_sections;
  configinfo(parsestream &);
  cat_str *startmenu, *endmenu, *submenutitle, *hkexclude,
    *genmenu, *postrun, *prerun, *sort, *rootpref, *userpref,
    *repeat_lang, *also_run;

  bool  onlyrunasroot, onlyrunasuser;
  bool onlyuniquetitles;
  cat_str *preruntest;

  // hint stuff:
  bool   hint_optimize;
  double hint_nentry;
  double hint_topnentry;
  double hint_mixedpenalty;
  double hint_minhintfreq;
  double hint_mlpenalty;
  int    hint_max_ntry;
  int    hint_max_iter_hint;
  bool   hint_debug;

  int hotkeycase; //0=insensitive, 1=sensitive
  ostream &debuginfo(ostream &);
  
  String &compat(){return compt;};

  //String &get_examplercfile(){return exrcf;};
  //String &get_mainmenutitle(){return mainmt;};
  //String &rootsection(){return roots;};
  //String &userprefix(){return userpref;};
  //String &treewalk(){return treew;};
  //String &postoutput(){return postout;};
  //String &preoutput(){return prerout;};
  String &rcfile(){return rcf;};
  String &examplercfile(){return exrcf;};
  String &mainmenutitle(){return mainmt;};
  String &rootsection(){return roots;};
  cat_str *rootprefix(){
    if(rootpref == NULL)
      throw unknown_indirect_function(NULL, "rootprefix");
    return rootpref;
  };
  cat_str *userprefix(){
    if(userpref == NULL)
      throw unknown_indirect_function(NULL, "userprefix");
    return userpref;
  };
  String &treewalk(){return treew;};
  String &preoutput(){return preout;};
  String &postoutput(){return postout;};

  String prefix();
};

extern configinfo *config;
extern supportedinfo *supported;
extern bool testuniqueness(map<String, String, less<String> > &menuentry);

#endif /* MENU_METHOD_H */
