/*
 * $Id: printf.c,v 1.2 2001/07/06 20:18:48 doviende Exp $
 *
 * libarr - a screen management toolkit
 *
 * Copyright (C) 2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Author: Chris Bond
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <sys/types.h>

#include <stdarg.h>
#include <stdlib.h>

#include "printf.h"
#include "types.h"

/* int_buffer[] is the integer conversion buffer that will be used in the string
 * formatting routines.
 */
static char int_buffer[12];

/* malloc_calc_len() calculates how long a formatted string will be.
 */
size_t
malloc_calc_len(str, args)
	const char *str;
	va_list args;
{
	register char *ptr;
	size_t length;

	for (length = 0; *str != NUL; ++str) {
		if ((*str != '%') || (*++str == '%')) {
			++length;
			continue;
		}

		if (*str == 'c') {
			/* We need to pop the character off the stack even
			 * though we don't do anything with it.
			 */
			(void)va_arg(args, int);
			continue;
		}

		if ((*str == 's') && ((ptr = va_arg(args, char *)) != NULL)) {
			for (; *ptr++ != NUL; ++length);
			continue;
		}

		if ((*str == 'l') && (*++str == 'u')) {		/* u_long */
			u_long long_conv;

			++length;
			if ((long_conv = (u_long)va_arg(args, u_long)) != 0)
				for (; (long_conv /= 10) != 0; ++length);

			continue;
		}

		if (*str == 'p') {				/* pointer */
			ptr_size_t pointer;

			length += 3;
			if ((pointer = (ptr_size_t)va_arg(args, ptr_size_t)))
				for (; (pointer /= 16) != 0; ++length);

			continue;
		}

		if (*str == 'd') {				/* integer */
			int number;

			++length;
			if ((number = (int)va_arg(args, int)) != 0) {
				if (number < 0) {
					++length;		/* '-' */
					number -= number * 2;
				}
				for (; (number /= 10) != 0; ++length);
			}

			continue;
		}
	}

	return length;
}

/* format_copy() prints a formatted string (from format and args) to `pointer'.
 */
size_t
format_copy(pointer, str, args)
	char *pointer, *str;
	va_list args;
{
	register char *copy;
	char *save, chr;

	/* `save' is stored so we can calculate how big our string is when we're
	 * done without having to increment another variable every time we copy.
	 */
	save = pointer;

	for (; *str != NUL; ++str) {
		if ((*str != '%') || (*++str == '%')) {
			*pointer++ = *str;
			continue;
		}

		if ((*str == 'c') && ((chr = va_arg(args, int)) != NUL)) {
			*pointer++ = chr;
			continue;
		}

		if ((*str == 's') && ((copy = va_arg(args, char *)) != NULL)) {
			for (; *copy != 0; *pointer++ = *copy++);
			continue;
		}

		if ((*str == 'l') && (*++str == 'u')) {
			u_long long_conv;

			if ((long_conv = (u_long)va_arg(args, u_long)) != 0) {
				copy = (char *)int_buffer;

				/* We need to copy this into a temporary storage
				 * buffer and then copy it *backwards* into the
				 * buffer that `pointer' points to.
				 */
				for (; long_conv != 0; long_conv /= 10)
					*copy++ = (long_conv % 10) + '0';
				for (--copy; copy >= (char *)int_buffer;)
					*pointer++ = *copy--;
			}
			else
				*pointer++ = '0';
		}

		if (*str == 'p') {
			ptr_size_t ptr;

			*pointer++ = '0';
			*pointer++ = 'x';

			if ((ptr = (ptr_size_t)va_arg(args, ptr_size_t)) == 0) {
				*pointer++ = '0';
				continue;
			}

			copy = (char *)int_buffer;
			for (; ptr != 0; ptr /= 16)
				if ((ptr % 16) > 9)
					*copy++ = (ptr % 8) + ('0' * 2) - 1;
				else
					*copy++ = (ptr % 16) + '0';
			for (--copy; copy >= (char *)int_buffer; --copy)
				*pointer++ = *copy;
		}

		if (*str == 'd') {
			int num_conv;

			if ((num_conv = (int)va_arg(args, int)) == 0) {
				*pointer++ = '0';
				continue;
			}
			if (num_conv < 0) {
				*pointer++ = '-';
				num_conv = -num_conv;
			}

			copy = (char *)int_buffer;
			for (; num_conv != 0; num_conv /= 10)
				*copy++ = (num_conv % 10) + '0';
			for (--copy; copy >= (char *)int_buffer; --copy)
				*pointer++ = *copy;
		}
	}

	*pointer = 0;

	return (size_t)(pointer - save);
}

/* malloc_printf() allocates enough space for the formatted string and copies
 * to a buffer that is supplied by the caller.
 */
size_t
malloc_printf(char **pointer, char *str, ...)
{
	va_list args;
	size_t length;

	va_start(args, str);
	length = malloc_calc_len((const char *)str, args);

	if ((*pointer = (char *)malloc(length + 1)) != NULL)
		format_copy(*pointer, str, args);
	else
		length = 0;

	va_end(args);

	return length;
}
