/*
 * $Id: arr.h,v 1.28 2000/11/29 01:53:36 doviende Exp $
 */

#if !defined(__include_arr_h__)
#define __include_arr_h__

#include <sys/types.h>

#include <stdio.h>

/* The macro _MAJOR is the major version number of the package, and _MINOR the
 * minor.  Please heed these values, as things are subject to change.
 */
#define ARR_MAJOR 0
#define ARR_MINOR 1

/* The following typedefs are pretty much i386-only.  Essentially, these types
 * need to be this large:
 * 	- byte_t and arr_color_t need to be a byte;
 * 	- tsize_t should be 32-bits or larger.
 *
 * Perhaps these should be made portable some day.
 */
typedef u_int8_t byte_t;		/* XXX */
typedef u_int8_t arr_color_t;	/* XXX */
typedef u_int32_t tsize_t;		/* XXX */

typedef	struct __acs_map_t {
	byte_t chr;
	byte_t ext;
} acs_map_t;

typedef struct __arr_scr_t {
	struct arr_term {
		short      co;		/* screen columns */
		short      li;		/* screen lines */
		acs_map_t *acs_map;	/* ACS character map */
	} term;
	struct {			/* position on screen */
		tsize_t x;		/* from left of screen */
		tsize_t y;		/* from top of screen */
		tsize_t max;		/* `end' of screen (buffer-wise) */
	} seek;
	struct {
		tsize_t x;		/* cursor position from left of scr */
		tsize_t y;		/* cursor position from top of scr */
		byte_t visible;		/* cursor is visible or not */
	} cursor;
	struct {			/* write buffer */
		char *buf;		/* actual write buffer */
		char *ptr;		/* current position */
		char *end;		/* end of buffer */
	} buf;

	arr_color_t *cmain;		/* main screen color map */
	arr_color_t *cback;		/* back buffer color map */
	arr_color_t pen_color;		/* current pen color */

	struct bchr {
		/* Each coordinate on the screen is represented by these 4
		 * bytes.
		 */
		u_char chr;		/* character */
		arr_color_t color;	/* color of this character */
		unsigned ext:1;		/* extended character */
		unsigned reserved:15;	/* unused (for now...) */
	} *main, *back;			/* main and back buffers */
} arr_scr_t;

/* TERM_SIZE needs to be the size (in bytes) of a &term structure's screen in an
 * arr_scr_t.
 */
#define TERM_SIZE(t)		((t)->co * (t)->li)

/* WRITE_BUF_LEN is the size of the write buffer (per screen). */
#define WRITE_BUF_LEN		8192

extern int arr_scr_init(void);
extern void arr_scr_free(void);
extern void arr_buffer_init(char _byte, arr_color_t _color);

extern int arr_initial_dump(void);
extern int arr_dump_screen(void);
extern int arr_redraw(void);

/* TERM_SEEK() is the position we're at (buffer-wise) on the screen. */
#define TERM_SEEK(s)		(((s)->term.co * (s)->seek.y) + (s)->seek.x)

/*
 * Keyboard input:
 */
/* arr_kqueue is what is returned to an application when it requests a read on
 * standard input.  It's a list of all the processed keys that were read.
 */
typedef struct __arr_kqueue {
	struct __arr_kqueue *next;
	int key;
} arr_kqueue;

#define MODIFIER_ALT	0x200
#define MODIFIER_CTRL	0x400
#define MODIFIER_SHIFT	0x800
#define MODIFIER_ARROW	0x1000
#define MODIFIER_MKEY	0x2000

/* arr_key() is the function that needs to be called when reading from the
 * keyboard.
 */
extern arr_kqueue *arr_key(int (*_func)(FILE *));

/*
 * Keys:
 */
/* The default read function is fgetc(), which reads a byte from standard input.
 * If you like, however, you can give another reading function to arr_key() if
 * you need to read from something else (like GPM).
 */
#define arr_read_func	fgetc

/* The AK_ macros are used to see, well, what key is what.  AK_CTRL() should be
 * used to convert a single character into a `Control'-modified character (one
 * that will be placed in a kqueue structure).
 */
#define AK_ENTER	'\n'

#define AK_BACKSPACE	8
#define AK_BDEL		4

#define AK_CTRL(c)	(MODIFIER_CTRL | c)

#define AK_ARROW_UP	(MODIFIER_ARROW | 'A')
#define AK_ARROW_DOWN	(MODIFIER_ARROW | 'B')
#define AK_ARROW_LEFT	(MODIFIER_ARROW | 'D')
#define AK_ARROW_RIGHT	(MODIFIER_ARROW | 'C')

#define AK_BREAK	(MODIFIER_MKEY | 'E')
#define AK_PAGEUP	(MODIFIER_MKEY | 'F')
#define AK_PAGEDOWN	(MODIFIER_MKEY | 'G')
#define AK_HOME		(MODIFIER_MKEY | 'H')
#define AK_END		(MODIFIER_MKEY | 'I')
#define AK_DELETE	(MODIFIER_MKEY | 'J')
#define AK_INSERT	(MODIFIER_MKEY | 'K')

#define AK_NPAD_CENTER	(MODIFIER_ARROW | 'L')

/*
 * Colors:
 */
/*
 * A color mask:
 * 	1 bit: foreground attribute
 * 	3 bits: foreground color
 * 	3 bits: background color
 * 	1 bit wasted (color_t is a byte, 8 bits)
 */

#define FG_BLACK	0x0
#define FG_RED		0x10
#define FG_GREEN	0x20
#define FG_BROWN	0x30
#define FG_BLUE		0x40
#define FG_MAGENTA	0x50
#define FG_CYAN		0x60
#define FG_WHITE	0x70

#define FG_MASK		0xF0

#define FGA_BOLD	0x80

#define FG_BBLACK	(FG_BLACK | FGA_BOLD)
#define FG_BRED		(FG_RED | FGA_BOLD)
#define FG_BGREEN	(FG_GREEN | FGA_BOLD)
#define FG_YELLOW	(FG_BROWN | FGA_BOLD)
#define FG_BBLUE	(FG_BLUE | FGA_BOLD)
#define FG_BMAGENTA	(FG_MAGENTA | FGA_BOLD)
#define FG_BCYAN	(FG_CYAN | FGA_BOLD)
#define FG_BWHITE	(FG_WHITE | FGA_BOLD)

#define BG_BLACK	0x0
#define BG_RED		0x1
#define BG_GREEN	0x2
#define BG_BROWN	0x3
#define BG_BLUE		0x4
#define BG_MAGENTA	0x5
#define BG_CYAN		0x6
#define BG_WHITE	0x7

#define BG_MASK		0x07

#define CL_INVALID	0x08

/* ansi_color_t is (and should only be) used internally.  Don't use mine,
 * roll your own!
 */
typedef struct __ansi_color_t {
	byte_t attrib;			/* text attribute */
	byte_t fg;			/* foreground color */
	byte_t bg;			/* background color */
} ansi_color_t;

extern ansi_color_t *build_color(arr_color_t _mask);
extern void arr_bold_col_list(arr_color_t _fc, ...);
extern void arr_inverse_col_list(arr_color_t _fc, ...);

/*
 * Printing characters:
 */
/* These redundant functions are used to modify arr's screen buffer, so when you
 * do another screen update, it can calculate the changes and write a minimal
 * amount of output.
 */
extern void arr_set_col(arr_color_t _color);
extern arr_color_t arr_get_col(void);
extern arr_color_t arr_get_col_xy(int _x, int _y);
extern char arr_get_chr_xy(int _x, int _y);
extern void arr_set_col_xy(int _x, int _y, arr_color_t _color);
extern void arr_goto_xy(int _x, int _y);
extern arr_color_t arr_get_col(void);
extern short arr_get_x(void);
extern short arr_get_y(void);
extern short arr_scr_x(void);
extern short arr_scr_y(void);
extern char arr_get_chr(void);
extern int arr_acs_char(byte_t _chr);

extern void arr_nprint_col(u_char _chr, arr_color_t _color, int _n);
extern void arr_nprint_xy_col(u_char _chr, tsize_t _x, tsize_t _y,
		arr_color_t _color, int _n);

#define arr_nprint(c, n)		arr_nprint_col(c, arr_get_col(), n)
#define arr_nprint_xy(c, x, y, n)	arr_nprint_xy_col(c, x, y, \
							arr_get_col(), n)
#define arr_print(c)			arr_nprint(c, 1)
#define arr_print_xy(c, x, y)		arr_nprint_xy(c, x, y, 1)
#define arr_print_col(ch, c)		arr_nprint_col(ch, c, 1)
#define arr_print_xy_col(ch, x, y, c)	arr_nprint_xy_col(ch, x, y, c, 1)

extern void arr_nprint_acs_col(int _chr, arr_color_t _color, int _n);
extern void arr_nprint_acs_xy_col(int _chr, tsize_t _x, tsize_t _y,
		arr_color_t _color, int _n);

#define arr_nprint_acs(c, n)		arr_nprint_acs_col(c, arr_get_col(), n)
#define arr_nprint_acs_xy(c, x, y, n)	arr_nprint_acs_xy_col(c, x, y, \
							arr_get_col(), n)

#define arr_print_acs(c)			arr_nprint_acs(c, 1)
#define arr_print_acs_xy(c, x, y)		arr_nprint_acs_xy(c, x, y, 1)
#define arr_print_acs_col(ch, c)		arr_nprint_acs_col(ch, c, 1)
#define arr_print_acs_xy_col(ch, x, y, c)	arr_nprint_acs_xy_col(ch, x, y, c, 1)

extern void arr_strprint_col(char *_str, arr_color_t _color);
extern void arr_strprint_xy_col(char *_str, tsize_t _x, tsize_t _y,
		arr_color_t _color);
extern void arr_nstrprint(char *_str, int _n);

#define arr_strprint(s)			arr_strprint_col(s, arr_get_col())
#define arr_strprint_xy(s, x, y)	arr_strprint_xy_col(s, x, y, \
						arr_get_col())

extern void arr_hide_cursor(void);
extern void arr_show_cursor(void);
extern void arr_set_cursor_pos(tsize_t _x, tsize_t _y);

extern void arr_beep(void);

extern void arr_clear(void);

extern const acs_map_t acs_default[];
extern const int       sizeof_acs_default;

#define TC_MAP(x)	(x + 256 - '+')

#define TC_RARROW	TC_MAP('+')	/* right arrow */
#define TC_LARROW	TC_MAP(',')	/* left arrow */
#define TC_UARROW	TC_MAP('-')	/* up arrow */
#define TC_DARROW	TC_MAP('.')	/* down arrow */
#define TC_BLOCK	TC_MAP('0')	/* square block */
#define TC_DIAMOND	TC_MAP('`')	/* diamond */
#define TC_CHKBOARD	TC_MAP('a')	/* checker board */
#define TC_DEGREE	TC_MAP('f')	/* degree sign */
#define TC_PLUSMINUS	TC_MAP('g')	/* plus or minus */
#define TC_SQBOARD	TC_MAP('h')	/* board of squares */
#define TC_LANTERN	TC_MAP('i')	/* lantern sign */
#define TC_LRCORNER	TC_MAP('j')	/* lower right corner */
#define TC_URCORNER	TC_MAP('k')	/* upper right corner */
#define TC_ULCORNER	TC_MAP('l')	/* upper left corner */
#define TC_LLCORNER	TC_MAP('m')	/* lower left corner */
#define TC_PLUS		TC_MAP('n')	/* large plus or crossover */
#define TC_SCANL1	TC_MAP('o')	/* scan line 1 */
#define TC_SCANL3	TC_MAP('p')	/* scan line 3 */
#define TC_HORZLINE	TC_MAP('q')	/* horizontal line */
#define TC_SCANL7	TC_MAP('r')	/* scan line 7 */
#define TC_SCANL9	TC_MAP('s')	/* scan line 9 */
#define TC_RTEE		TC_MAP('t')	/* tee pointing right */
#define TC_LTEE		TC_MAP('u')	/* tee pointing left */
#define TC_UTEE		TC_MAP('v')	/* tee pointing up */
#define TC_DTEE		TC_MAP('w')	/* tee pointing down */
#define TC_VERTLINE	TC_MAP('x')	/* veritcal line */
#define TC_LESSEQUAL	TC_MAP('y')	/* less or equal */
#define TC_MOREEQUAL	TC_MAP('z')	/* greater or equal */
#define TC_PI		TC_MAP('{')	/* Pi */
#define TC_NOTEQUAL	TC_MAP('|')	/* not equal */
#define TC_STERLING	TC_MAP('}')	/* UK pound sign */
#define TC_BULLET	TC_MAP('~')	/* bullet */

#endif
