//*****************************************************************************
//                                 DlgSrcCfg.cpp                              *
//                                ---------------                             *
//  Started     : 10/05/2005                                                  *
//  Last Update : 14/09/2007                                                  *
//  Copyright   : (C) 2005 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/dialogs/DlgSigSrcCfg.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( DlgSigSrcCfg, wxDialog )

  EVT_BUTTON( ID_BTN_OK,     DlgSigSrcCfg::OnBtnOk     )
  EVT_BUTTON( ID_BTN_CLEAR,  DlgSigSrcCfg::OnBtnClear  )
  EVT_BUTTON( ID_BTN_CANCEL, DlgSigSrcCfg::OnBtnCancel )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List:
//   poWin - A pointer to the dialog parent window

DlgSigSrcCfg::DlgSigSrcCfg( wxWindow * poWin ) : wxDialog( poWin, -1, wxT(""),
    wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE, wxDialogNameStr )
{
  SetTitle( wxT("Signal Source Setup") );
  Initialize( );
  bClear( );
  bSetDefaults( );
}

//*****************************************************************************
// Destructor.

DlgSigSrcCfg::~DlgSigSrcCfg( )
{
}

//*****************************************************************************
// Initialize object attributes.

void  DlgSigSrcCfg::Initialize( void )
{
  // Call all the initialization functions
  Create( );
  ToolTips( );

  // Layout the display objects
  DoLayout( );

  // Initialize temporary storage with the spin control values
  GetValues( );
}

//*****************************************************************************
// Create the display objects.

void  DlgSigSrcCfg::Create( void )
{
  wxPanel * poPnl;

  // Create the sinusoid panel and controls
  poPnl = new wxPanel( this );

  m_oPnlSinOffset.bCreate( poPnl, ID_PNL_SIN_OFFSET, 105 );
  m_oPnlSinAmp   .bCreate( poPnl, ID_PNL_SIN_AMP,    105 );
  m_oPnlSinFreq  .bCreate( poPnl, ID_PNL_SIN_FREQ,   105 );
  m_oPnlSinDelay .bCreate( poPnl, ID_PNL_SIN_DELAY,  105 );
  m_oPnlSinDamp  .bCreate( poPnl, ID_PNL_SIN_DAMP,   105 );

  m_oPnlSinOffset.bSetName( wxT("DC Offset")      );
  m_oPnlSinAmp   .bSetName( wxT("Amplitude")      );
  m_oPnlSinFreq  .bSetName( wxT("Frequency")      );
  m_oPnlSinDelay .bSetName( wxT("Initial Delay")  );
  m_oPnlSinDamp  .bSetName( wxT("Damping Factor") );

  m_oPnlSinOffset.bSetUnitsType( ChoUnits::eUNITS_VOLT );
  m_oPnlSinAmp   .bSetUnitsType( ChoUnits::eUNITS_VOLT );
  m_oPnlSinFreq  .bSetUnitsType( ChoUnits::eUNITS_FREQ );
  m_oPnlSinDelay .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlSinDamp  .bSetUnitsType( ChoUnits::eUNITS_SCLR );

  // Create the pulse panel and controls
  poPnl = new wxPanel( this );

  m_oPnlPulInitV .bCreate( poPnl, ID_PNL_PUL_INITV,  105 );
  m_oPnlPulMaxV  .bCreate( poPnl, ID_PNL_PUL_MAXV,   105 );
  m_oPnlPulDelay .bCreate( poPnl, ID_PNL_PUL_DELAY,  105 );
  m_oPnlPulRise  .bCreate( poPnl, ID_PNL_PUL_RISE,   105 );
  m_oPnlPulWidth .bCreate( poPnl, ID_PNL_PUL_WIDTH,  105 );
  m_oPnlPulFall  .bCreate( poPnl, ID_PNL_PUL_FALL,   105 );
  m_oPnlPulPeriod.bCreate( poPnl, ID_PNL_PUL_PERIOD, 105 );

  m_oPnlPulInitV .bSetName( wxT("Initial Value") );
  m_oPnlPulMaxV  .bSetName( wxT("Pulsed Value")  );
  m_oPnlPulDelay .bSetName( wxT("Initial Delay") );
  m_oPnlPulRise  .bSetName( wxT("Rise Time")     );
  m_oPnlPulWidth .bSetName( wxT("Pulse Width")   );
  m_oPnlPulFall  .bSetName( wxT("Fall Time")     );
  m_oPnlPulPeriod.bSetName( wxT("Period")        );

  m_oPnlPulInitV .bSetUnitsType( ChoUnits::eUNITS_VOLT );
  m_oPnlPulMaxV  .bSetUnitsType( ChoUnits::eUNITS_VOLT );
  m_oPnlPulDelay .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulRise  .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulWidth .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulFall  .bSetUnitsType( ChoUnits::eUNITS_TIME );
  m_oPnlPulPeriod.bSetUnitsType( ChoUnits::eUNITS_TIME );

  // Create buttons panel and the button controls
  poPnl = new wxPanel( this );

  m_oBtnOk    .Create( poPnl, ID_BTN_OK,     wxT("OK")     );
  m_oBtnClear .Create( poPnl, ID_BTN_CLEAR,  wxT("Clear")  );
  m_oBtnCancel.Create( poPnl, ID_BTN_CANCEL, wxT("Cancel") );
}

//*****************************************************************************
// Initialize the tool tips.

void  DlgSigSrcCfg::ToolTips( void )
{
  m_oPnlSinOffset.SetToolTip( wxT("The DC offset of the sinusoidal signal") );
  m_oPnlSinAmp   .SetToolTip( wxT("The amplitude of the sinusoidal signal") );
  m_oPnlSinFreq  .SetToolTip( wxT("The frequency of the sinusoidal signal") );
  m_oPnlSinDelay .SetToolTip( wxT("The initial delay of the sinusoidal signal") );
  m_oPnlSinDamp  .SetToolTip( wxT("The damping factor of the sinusoidal signal") );

  m_oPnlPulInitV .SetToolTip( wxT("The initial value of the pulse source") );
  m_oPnlPulMaxV  .SetToolTip( wxT("The maximum value of each pulse") );
  m_oPnlPulDelay .SetToolTip( wxT("The time until the first pulse") );
  m_oPnlPulRise  .SetToolTip( wxT("The time taken to go from 0 to the pulsed value") );
  m_oPnlPulWidth .SetToolTip( wxT("The width of each pulse") );
  m_oPnlPulFall  .SetToolTip( wxT("The time taken to go from the pulsed value to 0") );
  m_oPnlPulPeriod.SetToolTip( wxT("The period of the pulse train") );
}

//*****************************************************************************
// Layout the dialog display objects.

void  DlgSigSrcCfg::DoLayout( void )
{
  wxPanel     * poPnl;
  wxBoxSizer  * poSzr;
  wxSizerFlags  oFlags;

  // Create and set the underlying dialog's sizer
  poSzr = new wxBoxSizer( wxVERTICAL );
  SetSizer( poSzr );

  // Create and set the sizer for the sinusoid parameter panel
  poPnl = (wxPanel *) m_oPnlSinFreq.GetParent( );
  poSzr = new wxStaticBoxSizer( wxVERTICAL, poPnl, wxT(" Sinusoid ") );
  poPnl->SetSizer( poSzr );

  // Layout the sine wave parameter panel
  oFlags.Align( wxALIGN_CENTER );
  oFlags.Border( wxTOP, 5 );
  poSzr->Add( &m_oPnlSinOffset, oFlags );
  oFlags.Border( wxLEFT | wxRIGHT, 10 );
  poSzr->Add( &m_oPnlSinAmp,    oFlags );
  poSzr->Add( &m_oPnlSinFreq,   oFlags );
  poSzr->Add( &m_oPnlSinDelay,  oFlags );
  oFlags.Border( wxBOTTOM, 5 );
  poSzr->Add( &m_oPnlSinDamp,   oFlags );
  poSzr->SetSizeHints( poPnl );

  oFlags.Border( wxTOP | wxLEFT | wxRIGHT, 10 );
  GetSizer( )->Add( poPnl, oFlags );

  // Create and set the sizer for the pulse parameter panel
  poPnl = (wxPanel *) m_oPnlPulInitV.GetParent( );
  poSzr = new wxStaticBoxSizer( wxVERTICAL, poPnl, wxT(" Pulse ") );
  poPnl->SetSizer( poSzr );

  // Layout the pulse parameter panel
  oFlags.Align( wxALIGN_CENTER );
  oFlags.Border( wxTOP, 5 );
  poSzr->Add( &m_oPnlPulInitV,  oFlags );
  oFlags.Border( wxLEFT | wxRIGHT, 10 );
  poSzr->Add( &m_oPnlPulMaxV,   oFlags );
  poSzr->Add( &m_oPnlPulDelay,  oFlags );
  poSzr->Add( &m_oPnlPulRise,   oFlags );
  poSzr->Add( &m_oPnlPulWidth,  oFlags );
  poSzr->Add( &m_oPnlPulFall,   oFlags );
  oFlags.Border( wxBOTTOM, 5 );
  poSzr->Add( &m_oPnlPulPeriod, oFlags );
  poSzr->SetSizeHints( poPnl );

  oFlags.Border( wxALL, 10 );
  GetSizer( )->Add( poPnl, oFlags );

  // Create and set the sizer the button panel
  poPnl = (wxPanel *) m_oBtnOk.GetParent( );
  poSzr = new wxBoxSizer( wxHORIZONTAL );
  poPnl->SetSizer( poSzr );

  // Layout the button panel
  oFlags.Border( wxTOP | wxBOTTOM, 10 );
  oFlags.Align( wxALIGN_RIGHT );
  poSzr->Add( &m_oBtnOk,     oFlags );
  poSzr->AddSpacer( 10 );
  oFlags.Align( wxALIGN_CENTER );
  poSzr->Add( &m_oBtnClear,  oFlags );
  poSzr->AddSpacer( 10 );
  oFlags.Align( wxALIGN_LEFT );
  poSzr->Add( &m_oBtnCancel, oFlags );
  poSzr->SetSizeHints( poPnl );

  oFlags.Align( wxALIGN_CENTER );
  oFlags.Border( wxBOTTOM, 10 );
  GetSizer( )->Add( poPnl, oFlags );

  // Set dialogues minimum size and initial size as calculated by the sizer
  GetSizer( )->SetSizeHints( this );
}

//*****************************************************************************
// Get a reference to a value panel associated with a control ID number.
//
// Argument List:
//   iPnlID - The control ID number
//
// Return Values:
//   A reference to a value panel

PnlValue & DlgSigSrcCfg::roGetPanel( int iPnlID )
{
  // No argument tests here so use this function responsibly or suffer the pain
  switch( iPnlID )
  {
    case ID_PNL_SIN_OFFSET : return( m_oPnlSinOffset );
    case ID_PNL_SIN_AMP    : return( m_oPnlSinAmp    );
    case ID_PNL_SIN_FREQ   : return( m_oPnlSinFreq   );
    case ID_PNL_SIN_DELAY  : return( m_oPnlSinDelay  );
    case ID_PNL_SIN_DAMP   : return( m_oPnlSinDamp   );

    case ID_PNL_PUL_INITV  : return( m_oPnlPulInitV  );
    case ID_PNL_PUL_MAXV   : return( m_oPnlPulMaxV   );
    case ID_PNL_PUL_DELAY  : return( m_oPnlPulDelay  );
    case ID_PNL_PUL_RISE   : return( m_oPnlPulRise   );
    case ID_PNL_PUL_WIDTH  : return( m_oPnlPulWidth  );
    case ID_PNL_PUL_FALL   : return( m_oPnlPulFall   );
    case ID_PNL_PUL_PERIOD : return( m_oPnlPulPeriod );

    default                : return( m_oPnlPulInitV  );
  }
}

//*****************************************************************************
// Set the values in the spin controls from temporary storage.

void  DlgSigSrcCfg::SetValues( void )
{
  SetEvtHandlerEnabled( FALSE );

  for( int i1=ID_PNL_FST; i1<=ID_PNL_LST; i1++ )
    roGetPanel( i1 ).bSetValue( (float) m_dfPnlValue[ i1 ] );

  SetEvtHandlerEnabled( TRUE );
}

//*****************************************************************************
// Get the values from the spin controls and put them in temporary storage.

void  DlgSigSrcCfg::GetValues( void )
{
  SetEvtHandlerEnabled( FALSE );

  for( int i1=ID_PNL_FST; i1<=ID_PNL_LST; i1++ )
    m_dfPnlValue[ i1 ] = roGetPanel( i1 ).dfGetValue( );

  SetEvtHandlerEnabled( TRUE );
}

//*****************************************************************************
// Test the values against each other.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigSrcCfg::bValidate( void )
{
  wxString  osErrMsg;
  double    df1, df2, df3, df4, df5;

  // Test the sinusoid parameter values
  df1 = m_oPnlSinAmp .dfGetValue( );
  df2 = m_oPnlSinFreq.dfGetValue( );
  if( df1!=0.0 && df2!=0.0 )
  {
    if(      df1 == 0.0 )
      osErrMsg << wxT("The amplitude must be greater than zero.\n");
    else if( df2 == 0.0 )
      osErrMsg << wxT("The frequency must be greater than zero.\n");
  }

  // Test the pulse parameter values
  df1 = m_oPnlPulRise  .dfGetValue( );
  df2 = m_oPnlPulWidth .dfGetValue( );
  df3 = m_oPnlPulFall  .dfGetValue( );
  df4 = m_oPnlPulPeriod.dfGetValue( );
  df5 = m_oPnlPulMaxV  .dfGetValue( );
  if( df2!=0.0 || df4!=0.0 || df5!=0.0 )
  {
    if(      df4 < df1 )
      osErrMsg << wxT("The rise time is greater than the pulse period.\n");
    else if( df4 < df2 )
      osErrMsg << wxT("The pulse width is greater than the pulse period.\n");
    else if( df4 < df3 )
      osErrMsg << wxT("The fall time is greater than the pulse period.\n");
    else if( df4 < df1+df2+df3 )
      osErrMsg << wxT("The pulse period is less then the total pulse width.\n");

    // Test overall logic
    df1 = m_oPnlPulMaxV.dfGetValue( );
    if( df4>0.0  && df1==0.0 )
      osErrMsg << wxT("The pulse timing has been defined but no level has been specified.\n");
    if( df4==0.0 && df1!=0.0 )
      osErrMsg << wxT("The pulse level has been defined but no timing has been specified.\n");
  }

  // Display a message if an error was detected
  if( ! osErrMsg.IsEmpty( ) )
  {
    wxMessageDialog  oMsgDlg( this, osErrMsg, wxT("Error"), wxOK | wxICON_ERROR );
    oMsgDlg.ShowModal( );
  }

  return( osErrMsg.IsEmpty( ) );
}

//*****************************************************************************
// Parse a independent source definition string.
//
// Argument List:
//   rosCmd - The independent source definition to be parsed
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigSrcCfg::bSetSrcCmd( const wxString & rosCmd )
{
  wxStringTokenizer  ostk1;
  wxString  osCmd;
  size_t    szSin, szPulse;
  size_t    sz1, sz2;
  bool      bRtn=TRUE;

  // Change all characters to upper case
  osCmd   = rosCmd.Upper( );
  szSin   = osCmd.find( wxT("SIN(") );
  szPulse = osCmd.find( wxT("PULSE(") );
  if( szSin == wxString::npos && szPulse == wxString::npos ) return( FALSE );

  // Set all panel values to zero
  bClear( );
  for( int i1=ID_PNL_FST; i1<=ID_PNL_LST; i1++ )
    roGetPanel( i1 ).bSetValue( (float) 0.0, TRUE );

  // Extract the sinusoid function parameter list
  if( szSin != wxString::npos )
  {
    sz1 = osCmd.find( wxT('('), szSin );
    sz2 = osCmd.find( wxT(')'), szSin );
    if( sz1!=wxString::npos && sz2!=wxString::npos )
    {
      ostk1.SetString( osCmd.substr( sz1+1, sz2-sz1-1 ) );
      if( ostk1.CountTokens( ) == 5 )
      { // Extract each sinusoid function parameter values
        m_oPnlSinOffset.bSetValue( ostk1.GetNextToken( ) );
        m_oPnlSinAmp   .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlSinFreq  .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlSinDelay .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlSinDamp  .bSetValue( ostk1.GetNextToken( ) );
      }
      else bRtn = FALSE;
    }
    else bRtn = FALSE;
  }

  // Extract the pulse function parameter list
  if( szPulse != wxString::npos )
  {
    sz1 = osCmd.find( wxT('('), szPulse );
    sz2 = osCmd.find( wxT(')'), szPulse );
    if( sz1!=wxString::npos && sz2!=wxString::npos )
    {
      ostk1.SetString( osCmd.substr( sz1+1, sz2-sz1-1 ) );
      if( ostk1.CountTokens( ) == 7 )
      { // Extract each pulse function parameter values
        m_oPnlPulInitV .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlPulMaxV  .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlPulDelay .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlPulRise  .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlPulFall  .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlPulWidth .bSetValue( ostk1.GetNextToken( ) );
        m_oPnlPulPeriod.bSetValue( ostk1.GetNextToken( ) );
      }
      else bRtn = FALSE;
    }
    else bRtn = FALSE;
  }

  // Update temporary storage
  GetValues( );

  return( bRtn );
}

//*****************************************************************************
// Create the independent voltage source definition string.
//
// Return Values:
//   The generator command

const wxString & DlgSigSrcCfg::rosGetSrcCmd( void )
{
  m_osSrcCmd = wxT("DC 0.0");

  if( roGetPanel( ID_PNL_SIN_AMP ).dfGetValue( ) != 0.0 )
  { // Create the sinusoid function section
    m_osSrcCmd << wxT(" SIN( ");
    m_osSrcCmd << roGetPanel( ID_PNL_SIN_OFFSET ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_SIN_AMP    ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_SIN_FREQ   ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_SIN_DELAY  ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_SIN_DAMP   ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << wxT(')');
  }

  if( roGetPanel( ID_PNL_PUL_MAXV ).dfGetValue( ) != 0.0 )
  { // Create the pulse function section
    m_osSrcCmd << wxT(" PULSE( ");
    m_osSrcCmd << roGetPanel( ID_PNL_PUL_INITV  ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_PUL_MAXV   ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_PUL_DELAY  ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_PUL_RISE   ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_PUL_FALL   ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_PUL_WIDTH  ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << roGetPanel( ID_PNL_PUL_PERIOD ).rosGetValue( ) << wxT(' ');
    m_osSrcCmd << wxT(')');
  }

  return( m_osSrcCmd );
}

//*****************************************************************************
// Reset all dialog settings to defaults.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigSrcCfg::bClear( void )
{
  bool  bRtn=TRUE;

  if( ! m_oPnlSinOffset.bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlSinAmp   .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlSinFreq  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlSinDelay .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlSinDamp  .bClear( ) ) bRtn = FALSE;

  if( ! m_oPnlPulInitV .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulMaxV  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulDelay .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulRise  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulWidth .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulFall  .bClear( ) ) bRtn = FALSE;
  if( ! m_oPnlPulPeriod.bClear( ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Set the value on a panel.
//
// Argument List:
//   iPnlID   - Spin control identifier
//   fValue   - The value to set the spin control
//   rosUnits - The units to display
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigSrcCfg::bSetValue( int iPnlID, float fValue,
                            const wxString & rosUnits )
{
  // Check for a valid panel identifier
  if( iPnlID<ID_PNL_FST || iPnlID>ID_PNL_LST )                 return( FALSE );

  // Set the panel value
  if( ! roGetPanel( iPnlID ).bSetValue( (float)fValue,TRUE ) ) return( FALSE );
  m_dfPnlValue[ iPnlID ] = fValue;

  if( ! rosUnits.IsEmpty( ) )
  { // Set the units
    if( ! roGetPanel( iPnlID ).bSetUnits( rosUnits ) )         return( FALSE );
    m_dfPnlValue[ iPnlID ] = roGetPanel( iPnlID ).dfGetValue( );
  }

  return( TRUE );
}
//*****************************************************************************
// Enable or disable a value panel.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigSrcCfg::bEnablePanel( int iPnlID, bool bState )
{
  if( iPnlID<ID_PNL_FST || iPnlID>ID_PNL_LST ) return( FALSE );

  if( roGetPanel( iPnlID ).Enable( bState ) )  return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Set default values in all of the dialog panels.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  DlgSigSrcCfg::bSetDefaults( void )
{
  bool  bRtn=TRUE;

  if( ! bSetValue( ID_PNL_SIN_OFFSET, 0.0, wxT("V")    ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_SIN_AMP,    0.0, wxT("V")    ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_SIN_FREQ,   0.0, wxT("KHz")  ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_SIN_DELAY,  0.0, wxT("msec") ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_SIN_DAMP,   0.0, wxT("none") ) ) bRtn = FALSE;

  if( ! bSetValue( ID_PNL_PUL_INITV,  0.0, wxT("V")    ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_MAXV,   0.0, wxT("V")    ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_DELAY,  0.0, wxT("msec") ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_RISE,   0.0, wxT("msec") ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_WIDTH,  0.0, wxT("msec") ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_FALL,   0.0, wxT("msec") ) ) bRtn = FALSE;
  if( ! bSetValue( ID_PNL_PUL_PERIOD, 0.0, wxT("msec") ) ) bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
//
//                             Event Handlers
//
//*****************************************************************************
// OK button event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event (not used)

void  DlgSigSrcCfg::OnBtnOk( wxCommandEvent & roEvtCmd )
{
  if( bValidate( ) )
  {
    GetValues( );
    EndModal( wxID_OK );
    rosGetSrcCmd( );
  }
}

//*****************************************************************************
// Clear button event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event (not used)

void  DlgSigSrcCfg::OnBtnClear( wxCommandEvent & roEvtCmd )
{
  bSetDefaults( );
}

//*****************************************************************************
// Cancel button event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event (not used)

void  DlgSigSrcCfg::OnBtnCancel( wxCommandEvent & roEvtCmd )
{
  SetValues( );
  EndModal( wxID_CANCEL );
}

//*****************************************************************************
