//*****************************************************************************
//                              CmdNgSpiceDC.cpp                              *
//                             ------------------                             *
//  Started     : 21/08/2007                                                  *
//  Last Update : 22/02/2008                                                  *
//  Copyright   : (C) 2007 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/commands/CmdNgSpiceDC.hpp"

//*****************************************************************************
// Constructor.

CmdNgSpiceDC::CmdNgSpiceDC( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdNgSpiceDC::~CmdNgSpiceDC( )
{
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .DC Vin 0.00 200.00m 25.00m
//        .DC TEMP 0.00 200.00 25.00
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceDC::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  double             df1;

  // Clear the object attributes
  os1 = *this;
  bClear( );
  *this = os1;

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) != 5 ) return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 3 ).Upper( );
  if( os1 != wxT(".DC") )         return( FALSE );
  m_osName = wxT("DC");

  m_bIsOk = TRUE;

  // Extract the sweep type: a signal source or temperature
  m_osSource = ostk1.GetNextToken( );
  if( m_osSource.IsEmpty( ) )     m_bIsOk = FALSE;

  // Extract the start value
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStart = (float) df1;
  else                            m_bIsOk = FALSE;

  // Extract the stop value
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStop  = (float) df1;
  else                            m_bIsOk = FALSE;

  // Extract the step increment
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStep  = (float) df1;
  else                            m_bIsOk = FALSE;

  return( m_bIsOk );
}

//*****************************************************************************
// Format the command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceDC::bFormat( void )
{
  wxString  osCmd, os1;

  m_bIsOk = FALSE;

  // Set the command name
  osCmd = wxT(".DC");
  if( m_osName.IsEmpty( ) ) m_osName = wxT("DC");

  // Set signal source
  osCmd << wxT(' ') << m_osSource;

  // Set sweep parameters
  if( ! ConvertType::bFltToStrEng( (double) m_fStart, os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;
  if( ! ConvertType::bFltToStrEng( (double) m_fStop,  os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;
  if( ! ConvertType::bFltToStrEng( (double) m_fStep,  os1 ) ) return( FALSE );
  osCmd << wxT(' ') << os1;

  *this = osCmd;

  m_bIsOk = TRUE;

  return( m_bIsOk );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceDC::bClear( void )
{
  CmdBase::bClear( );

  m_fStart = -FLT_MAX;
  m_fStop  = -FLT_MAX;
  m_fStep  = -FLT_MAX;

  m_osSource.Empty( );

  return( TRUE );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   psPrefix - A prefix to every line displayed (usually just spaces)

void  CmdNgSpiceDC::Print( const char * psPrefix )
{
  CmdBase::Print( psPrefix );

  cout << psPrefix << "m_fStart    : " << m_fStart << '\n';
  cout << psPrefix << "m_fStop     : " << m_fStop << '\n';
  cout << psPrefix << "m_fStep     : " << m_fStep << '\n';

  cout << psPrefix << "m_osSource  : " << m_osSource.mb_str( ) << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdNgSpiceDC Test Utility"
       << "\n     Version 1.00 (30/01/2008)\n";

  // Create a NG-SPICE DC command object
  CmdNgSpiceDC  oCmd_DC;

  // Use the following command example to check the formatter and the parser :
  osCmd = wxT(".DC Vin 0.00 100.00m 10.00m");

  // Set things up for a formatter test
  oCmd_DC.bClear( );
  oCmd_DC.m_fStart   = 0.0;
  oCmd_DC.m_fStop    = 100.0E-3;
  oCmd_DC.m_fStep    = 10.0E-3;
  oCmd_DC.m_osSource = wxT("Vin");
  cout << "\nRun Formatter    : " << ( oCmd_DC.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format  : " << ( oCmd_DC == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\noCmd_DC Contents : " << oCmd_DC.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_DC.bClear( );
  oCmd_DC = osCmd;
  cout << "\nRun Parser       : " << ( oCmd_DC.bParse( ) ? "OK" : "FAULT" );
  oCmd_DC.bFormat( );
  cout << "\nTest Cmd Format  : " << ( oCmd_DC == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\noCmd_DC Contents : " << oCmd_DC.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
