/* This file is part of MetaDOM
 * a generic bind package for the Document Object Model API.
 * Copyright (C) 2001  Luca Padovani <luca.padovani@cs.unibo.it>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the author's home page
 * http://www.cs.unibo.it/~lpadovan
 * or send an email to <luca.padovani@cs.unibo.it>
 */

#include <config.h>

#include <gdome.h>
#include <gdome-util.h>

#include "GdomeSmartDOM.hh"

namespace GdomeSmartDOM {

DOMImplementation::DOMImplementation()
{
  gdome_obj = gdome_di_mkref();
}

DOMImplementation::DOMImplementation(const DOMImplementation& obj)
{
  GdomeException exc_;
  
  if (gdome_obj != 0) {
    gdome_di_unref(gdome_obj, &exc_);
    gdome_obj = 0;
  }
  
  if (exc_ == 0) {
    gdome_obj = obj.gdome_obj;
    gdome_di_ref(gdome_obj, &exc_);
  }
  
  if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::DOMImplementation");
}

DOMImplementation::DOMImplementation(GdomeDOMImplementation* obj)
{
  gdome_obj = obj;
  if (gdome_obj != 0) {
    GdomeException exc_;
    gdome_di_ref(gdome_obj, &exc_);
    if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::DOMImplementation");
  }
}

DOMImplementation::~DOMImplementation()
{
  if (gdome_obj != 0) {
    GdomeException exc_;
    gdome_di_unref(gdome_obj, &exc_);
    if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::~DOMImplementation");
    gdome_obj = 0;
  }
}

bool DOMImplementation::hasFeature(const DOMString& feature, const DOMString& version)
{
  GdomeException exc_;
  GdomeDOMString* feature_ = GdomeDOMString_of_DOMString(feature);
  GdomeDOMString* version_ = GdomeDOMString_of_DOMString(version);
  GdomeBoolean res_ = gdome_di_hasFeature(gdome_obj, feature_, version_, &exc_);
  if (feature_ != 0) gdome_str_unref(feature_);
  if (version_ != 0) gdome_str_unref(version_);
  if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::hasFeature");
  return res_;
}

DocumentType DOMImplementation::createDocumentType(const DOMString& qualifiedName, const DOMString& publicId, const DOMString& systemId)
{
  GdomeException exc_;
  GdomeDOMString* qualifiedName_ = GdomeDOMString_of_DOMString(qualifiedName);
  GdomeDOMString* publicId_ = GdomeDOMString_of_DOMString(publicId);
  GdomeDOMString* systemId_ = GdomeDOMString_of_DOMString(systemId);
  GdomeDocumentType* res_ = gdome_di_createDocumentType(gdome_obj, qualifiedName_, publicId_, systemId_, &exc_);
  if (qualifiedName_ != 0) gdome_str_unref(qualifiedName_);
  if (publicId_ != 0) gdome_str_unref(publicId_);
  if (systemId_ != 0) gdome_str_unref(systemId_);
  if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::createDocumentType");
  return DocumentType(res_);
}

Document DOMImplementation::createDocument(const DOMString& namespaceURI, const DOMString& qualifiedName, const DocumentType& doctype)
{
  GdomeException exc_;
  GdomeDOMString* namespaceURI_ = GdomeDOMString_of_DOMString(namespaceURI);
  GdomeDOMString* qualifiedName_ = GdomeDOMString_of_DOMString(qualifiedName);
  GdomeDocument* res_ = gdome_di_createDocument(gdome_obj, namespaceURI_, qualifiedName_, (GdomeDocumentType*) doctype.gdome_object(), &exc_);
  if (namespaceURI_ != 0) gdome_str_unref(namespaceURI_);
  if (qualifiedName_ != 0) gdome_str_unref(qualifiedName_);
  if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::createDocument");
  return Document(res_);
}

Document
DOMImplementation::createDocumentFromURI(const char* uri, unsigned long mode)
{
  GdomeException exc_;
  GdomeDocument* res_ = gdome_di_createDocFromURI(gdome_obj, uri, mode, &exc_);
  if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::createDocumentFromURI");
  return Document(res_);
}

Document
DOMImplementation::createDocumentFromMemory(const char* buffer, unsigned long mode)
{
  GdomeException exc_;
  GdomeDocument* res_ = gdome_di_createDocFromMemory(gdome_obj, const_cast<char*>(buffer), mode, &exc_);
  if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::createDocumentFromMemory");
  return Document(res_);
}

bool
DOMImplementation::saveDocumentToFile(const Document& doc, const char* fileName, unsigned long mode)
{
  GdomeException exc_;
  GdomeBoolean res_ = gdome_di_saveDocToFile(gdome_obj, (GdomeDocument*) doc.gdome_object(), fileName, (GdomeSavingCode) mode, &exc_);
  if (exc_ != 0) throw DOMException(exc_, "DOMImplementation::saveDocToFile");
  return res_;
}

}

