/*
   Copyright (c) 2004, 2005 by AOSASA Shigeru and Red Hat, Inc.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above
   copyright notice, this list of conditions and the following
   disclaimer in the documentation and/or other materials provided
   with the distribution.  

   - Neither the name of the AOSASA Shigeru, Red Hat, Inc
   nor the names of its contributors may be used to endorse or
   promote products derived from this software without specific
   prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
   COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
   SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
   STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
   OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
   This software development was supported by
   Information-technorogy Promotion Agency, Japan (IPA).
*/

#include "ginspector-gtk.h"

#include "ginspector-ui-parameditor.h"

#include "ginspector-ui-stock.h"
#include "changednotifypair.h"

#include "gtksimpletogglebutton.h"
#include "gtkvalueeditor.h"

#include "glrr-gobject.h"



static void g_inspector_tag_to_value_editor (GtkValueEditor *value_editor);


#define PARAM_EDITOR_DATA "param-editor-data"

typedef struct _ParamEditorInfo ParamEditorInfo;
typedef struct _ParamEditorData ParamEditorData;

struct _ParamEditorInfo
{
  void (* get_value) (ParamEditorData *pedata,
		      GValue          *value);
  void (* set_value) (ParamEditorData *pedata,
		      GValue          *value);
};

struct _ParamEditorData
{
  const ParamEditorInfo *peinfo;

  GObject    *object;
  GParamSpec *param_spec;
  
  GtkWidget *editor;

  gboolean set_value_blocked;
  gboolean editor_value_changed;

  ChangedNotifyPair *changed_notify_pair;
};


static ParamEditorData *
param_editor_data_new (const ParamEditorInfo *peinfo,
		       GObject               *object,
		       GParamSpec            *spec)
{
  ParamEditorData *pedata;

  pedata = g_new (ParamEditorData, 1);

  pedata->peinfo = peinfo;

  pedata->object     = object;
  pedata->param_spec = spec;

  pedata->editor = NULL;

  pedata->set_value_blocked    = FALSE;
  pedata->editor_value_changed = FALSE;

  pedata->changed_notify_pair = NULL;


  g_param_spec_ref (pedata->param_spec);

  return pedata;
}

static void 
param_editor_data_destroy (ParamEditorData *pedata)
{
  g_param_spec_unref (pedata->param_spec);

  g_free (pedata);
}

/***/

static GtkWidget* create_toggle_button (ParamEditorData *pedata);

static void update        (ParamEditorData *pedata);
static void value_changed (GtkValueEditor  *value_editor,
			   gpointer         user_data);
static void notify        (GObject         *object,
			   GParamSpec      *pspec,
			   gpointer         user_data);


static GtkWidget *
create_param_editor (const ParamEditorInfo *peinfo,
		     GObject               *object,
		     GParamSpec            *spec)
{
  GtkWidget *editor;


  if (spec->flags & G_PARAM_WRITABLE)
    {
      GType param_spec_type;

      param_spec_type = G_PARAM_SPEC_TYPE (spec);

      if ((param_spec_type == G_TYPE_PARAM_CHAR)    ||
	  (param_spec_type == G_TYPE_PARAM_UCHAR)   ||
	  (param_spec_type == G_TYPE_PARAM_INT)     ||
	  (param_spec_type == G_TYPE_PARAM_UINT)    ||
	  (param_spec_type == G_TYPE_PARAM_LONG)    ||
	  (param_spec_type == G_TYPE_PARAM_ULONG)   ||
	  (param_spec_type == G_TYPE_PARAM_INT64)   ||
	  (param_spec_type == G_TYPE_PARAM_UINT64)  ||
	  (param_spec_type == G_TYPE_PARAM_FLOAT)   ||
	  (param_spec_type == G_TYPE_PARAM_DOUBLE)  ||
	  (param_spec_type == G_TYPE_PARAM_BOOLEAN) ||
	  (param_spec_type == G_TYPE_PARAM_ENUM)    ||
	  (param_spec_type == G_TYPE_PARAM_STRING))
	{
	    ParamEditorData *pedata;
	    ChangedNotifyPair *changed_notify_pair;

	    pedata = param_editor_data_new (peinfo, object, spec);

	    editor = gtk_value_editor_new_with_param_spec (spec);
	    g_inspector_tag_to_value_editor (GTK_VALUE_EDITOR (editor));

	    changed_notify_pair = changed_notify_pair_new (G_OBJECT (editor),
							   object);
	    changed_notify_pair_changed_connect (changed_notify_pair,
						 "value_changed",
						 G_CALLBACK (value_changed),
						 pedata);
	    changed_notify_pair_notify_connect (changed_notify_pair,
						"notify",
						G_CALLBACK (notify),
						pedata);

	    pedata->editor = editor;
	    pedata->changed_notify_pair = changed_notify_pair;

	    g_object_set_data_full (G_OBJECT (editor),
				    PARAM_EDITOR_DATA, pedata,
				    (GDestroyNotify)param_editor_data_destroy);

	    update (pedata);


	    {
	      GtkWidget *toggle_button;

	      toggle_button = create_toggle_button (pedata);

	      gtk_box_pack_end (GTK_BOX (editor), toggle_button,
				FALSE, FALSE, 0);
	    }
	}
      else
	{
	  editor = NULL;
	}
    }
  else
    {
      editor = NULL;
    }


  return editor;
}



static void
update (ParamEditorData *pedata)
{
  GValue value;


  g_value_fill0 (&value);
  g_value_init (&value, pedata->param_spec->value_type);

  if (pedata->param_spec->flags & G_PARAM_READABLE)
    {
      (* pedata->peinfo->get_value) (pedata, &value);
    }

  changed_notify_pair_block_changed_handler (pedata->changed_notify_pair);

  gtk_value_editor_set_value (GTK_VALUE_EDITOR (pedata->editor),
			      &value);

  changed_notify_pair_unblock_changed_handler (pedata->changed_notify_pair);


  g_value_unset (&value);
}

static void
value_changed (GtkValueEditor *value_editor,
	       gpointer        user_data)
{
  ParamEditorData *pedata;


  pedata = (ParamEditorData *)user_data;


  if (pedata->set_value_blocked == FALSE)
    {
      GValue *value;

      value = gtk_value_editor_get_value (GTK_VALUE_EDITOR (pedata->editor));

      (* pedata->peinfo->set_value) (pedata,
				     value);
    }
  else
    {
      pedata->editor_value_changed = TRUE;
    }
}

static void
notify (GObject    *object,
	GParamSpec *pspec,
	gpointer    user_data)
{
  ParamEditorData *pedata;

  pedata = (ParamEditorData *)user_data;


  if (pspec == pedata->param_spec)
    {
      update (pedata);
    }
}

static void
toggled (GtkToggleButton *toggle_button,
	 gpointer         user_data)
{
  ParamEditorData *pedata;


  pedata = (ParamEditorData *)user_data;


  {
    if (gtk_toggle_button_get_active (toggle_button) == TRUE)
      {
	if (pedata->editor_value_changed == TRUE)
	  {
	    GValue *value;

	    value = gtk_value_editor_get_value (GTK_VALUE_EDITOR (pedata->editor));


	    changed_notify_pair_block_notify_handler
	      (pedata->changed_notify_pair);

	    (* pedata->peinfo->set_value) (pedata,
					   value);

	    changed_notify_pair_unblock_notify_handler
	      (pedata->changed_notify_pair);
	  }

	pedata->set_value_blocked    = FALSE;
	pedata->editor_value_changed = FALSE;
      }
    else
      {
	pedata->set_value_blocked    = TRUE;
	pedata->editor_value_changed = FALSE;
      }
  }


  {
    GtkWidget *prev_child;
    GtkWidget *image;

    {
      prev_child = GTK_BIN (toggle_button)->child;

      if (prev_child != NULL)
	{
	  gtk_widget_destroy (prev_child);
	}
    }

    {
      GtkIconSize icon_size;

#if 1
      icon_size = GTK_ICON_SIZE_MENU;
#else
      icon_size = GTK_ICON_SIZE_BUTTON;
#endif

      if (gtk_toggle_button_get_active (toggle_button) == TRUE)
	{
	  image = gtk_image_new_from_stock (G_INSPECTOR_STOCK_CONNECTED,
					    icon_size);
	}
      else
	{
	  image = gtk_image_new_from_stock (G_INSPECTOR_STOCK_DISCONNECTED,
					    icon_size);
	}

      gtk_container_add (GTK_CONTAINER (toggle_button),
			 image);

      gtk_widget_show (image);
    }
  }
}

static GtkWidget *
create_toggle_button (ParamEditorData *pedata)
{
  GtkWidget *toggle_button;

  toggle_button = gtk_simple_toggle_button_new ();
  GTK_WIDGET_UNSET_FLAGS (GTK_WIDGET (toggle_button), GTK_CAN_FOCUS);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggle_button),
				TRUE);

  g_signal_connect (toggle_button,
                    "toggled",
                    G_CALLBACK (toggled),
                    pedata);

  gtk_toggle_button_toggled (GTK_TOGGLE_BUTTON (toggle_button));

  return toggle_button;
}


/*****/

static void property_get_value (ParamEditorData *pedata,
				GValue          *value);
static void property_set_value (ParamEditorData *pedata,
				GValue          *value);

GtkWidget *
g_inspector_create_property_editor (GObject    *object,
				    GParamSpec *spec)
{
  static const ParamEditorInfo peinfo = {
    property_get_value,
    property_set_value
  };

  GtkWidget *w;
  

  w = create_param_editor (&peinfo, object, spec);

  return w;
}

static void
property_get_value (ParamEditorData *pedata,
		    GValue          *value)
{
  g_object_get_property (pedata->object,
			 pedata->param_spec->name,
			 value);
}

static void
property_set_value (ParamEditorData *pedata,
		    GValue          *value)
{
  g_object_set_property (pedata->object,
			 pedata->param_spec->name,
			 value);
}

/***/
static void child_property_get_value (ParamEditorData *pedata,
				      GValue          *value);
static void child_property_set_value (ParamEditorData *pedata,
				      GValue          *value);

GtkWidget *
g_inspector_create_child_property_editor (GObject    *object,
					  GParamSpec *spec)
{
  static const ParamEditorInfo peinfo = {
    child_property_get_value,
    child_property_set_value
  };

  GtkWidget *w;
  

  w = create_param_editor (&peinfo, object, spec);

  return w;
}

static void
child_property_get_value (ParamEditorData *pedata,
			  GValue          *value)
{
  GtkWidget    *child;
  GtkContainer *container;
  gchar        *property_name;


  child         = GTK_WIDGET (pedata->object);
  container     = GTK_CONTAINER (gtk_widget_get_parent (child));
  property_name = pedata->param_spec->name;

  gtk_container_child_get_property (container,
				    child,
				    property_name,
				    value);
}

static void
child_property_set_value (ParamEditorData *pedata,
			  GValue          *value)
{
  GtkWidget    *child;
  GtkContainer *container;
  gchar        *property_name;


  child         = GTK_WIDGET (pedata->object);
  container     = GTK_CONTAINER (gtk_widget_get_parent (child));
  property_name = pedata->param_spec->name;

  gtk_container_child_set_property (container,
				    child,
				    property_name,
				    value);
}






/***/

#undef GTK_DISABLE_DEPRECATED /* XXX */
#include <gtk/gtkoptionmenu.h>

static void
g_inspector_tag_to_value_editor (GtkValueEditor *value_editor)
{
  g_return_if_fail (value_editor != NULL);
  g_return_if_fail (GTK_IS_VALUE_EDITOR (value_editor));


  g_inspector_tag_to_widget (GTK_WIDGET (value_editor));

  {
    GtkWidget *widget;
    GList *tmp;

    tmp = value_editor->widget_list;
    while (tmp != NULL)
      {
	widget = GTK_WIDGET (tmp->data);
	tmp = g_list_next (tmp);

	if (GTK_IS_OPTION_MENU (widget))
	  {
	    GtkOptionMenu *option_menu;
	    GtkMenu *menu;

	    option_menu = GTK_OPTION_MENU (widget);
	    
	    menu = GTK_MENU (gtk_option_menu_get_menu (option_menu));

	    g_inspector_tag_to_menu (menu);
	  }
      }
  }
}
