
/*
  
   1998 Roberto Alameda
  You may modify and distribute this file under the terms of the GNU
  General Public License, version 2, or any later version, at your
  convenience. See the file COPYING for details. 

*/ 


#include <config.h>

#if HAVE_UNISTD_H
# include <sys/types.h>
# include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <errno.h>
#include <sys/stat.h>
#include <time.h>


#include "gfont.h"


// *** Imported variables
extern TT_Engine ttengine;

extern GdkVisual *visual;
extern GdkGC *bwGC, *defGC, *blueGC;
extern GtkStyle *stylebold;
extern GdkPixmap *emptypixmap;

extern GtkWidget *mainwindow;
extern GtkWidget *samplet, *stbar, *prbar;

extern bool kerning, antialiasing;
extern int xresolution, yresolution;

extern GdkColor graypalette[];



// *** Module variables
static const char *weight_table[] = {
    "Thin", "Extra Light", "Light", "Regular", "Medium",
    "Semibold", "Bold", "Extra Bold", "Black"
};

static const char *width_table[] = {
  "Ultra Condensed", "Extra Condensed", "Condensed", 
  "Semi Condensed", "Medium", "Semi Expanded", 
  "Expanded", "Extra Expanded", "Ultra Expanded"
};


/* Structure associated to a font table window */
struct WInfo {
  FontData *fd;
  TT_Instance instance;
  double fontsize;
  int cellwidth, cellheight;
  int page;
  GdkPixmap *pixmap;
  GtkWidget *drawing_area, *info_label, *combo;
  GtkItemFactory *item_factory;
  guint antialiasing:1;  /* Whether the fonts in this window are smoothed */
  guint hinting:1;  /* Whether TT hinting is activated */
  guint baseline:1; /* Whether to show the baseline as reference */
  guint origin:1;   /* Whether to show the origin (in x) as reference */
};


/* TT_Table contains data used when downloading as Type42 font */
struct TT_Table {
  TT_ULong tag;
  TT_ULong checksum;
  TT_ULong offset;
  TT_ULong length;
  const char *name;
  guchar *data;  /* The table itself */
};


/* TTInfo contains data used in the properties window */
struct TT_Info {
  GtkWidget *property, *value;
};


// ************** Function prototypes (internally used in this module)
static Callback destroy_fonttable;
static GdkImage* gdk_image_subimage(GdkImage* origimage, gint x, gint y, 
				    gint w, gint h);
static gint drawarea_event(GtkWidget *widget, GdkEvent *event, gpointer);
static TT_FWord getkerning(FontData* fd, TT_UShort ch1, TT_UShort ch2);
static void fill_pixmap(WInfo *winfo);
static bool drawfont_inframe(GtkWidget *frame, WInfo *winfo);

static bool get_table_list(FontData *fd);
static void free_table_list(void);
static void write_string(FILE *ofp, bool pad, TT_Table *tb, 
			 TT_ULong offset, TT_ULong length);
static TT_ULong get_glyph_offset(TT_Table *loca, TT_UShort ch, int Index_To_Loc_Format);
static void make_offset_table(FontData *fd, char *ot, TT_UShort numTables);
static TT_Table *make_new_table(TT_Table *orig, TT_ULong len);


static void make_entries(GtkWidget *frame, struct TT_Info *props, int num);
static GtkWidget* general_info(FontData *fd);
static GtkWidget* specific_info(FontData *fd);





// ********************* Begin of program functions

static gint expose_event(GtkWidget *widget, GdkEvent *gevent, gpointer data)
{
  GdkEventExpose *event = (GdkEventExpose*)gevent;
  WInfo *winfo = (WInfo*)data;
  //printf("In the expose event, pixmap:%p\n", pixmap);
  gdk_draw_pixmap(widget->window, bwGC, winfo->pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);
  return FALSE;
}


static GdkImage* gdk_image_subimage(GdkImage* origimage, gint x, gint y, 
				    gint w, gint h)
{
  GdkImage *image;
  GdkImagePrivate *priv, *privorig;
  
  g_return_val_if_fail(origimage != NULL, 0);
  
  priv = g_new(GdkImagePrivate, 1);
  privorig = (GdkImagePrivate*)origimage;
  
  image = (GdkImage *)priv;
  priv->xdisplay = privorig->xdisplay;
  priv->image_put = privorig->image_put;
  image->type = origimage->type;
  image->visual = origimage->visual;
  image->width = w;
  image->height = h;
  image->depth = origimage->depth;
  priv->ximage = XSubImage(privorig->ximage, x, y, w, h);
  priv->ximage->bitmap_bit_order = privorig->ximage->bitmap_bit_order;
  priv->ximage->byte_order = privorig->ximage->byte_order;
  image->byte_order = origimage->byte_order;
  image->mem = priv->ximage->data;
  image->bpl = priv->ximage->bytes_per_line;
  image->bpp = origimage->bpp;
  return image;
}


/*
  Return a string with the requested (idx) name of the font
  It tries to find a Unicode encoded english string of the given type
  idx can be TT_NAME_ID_COPYRIGHT, TT_NAME_ID_FONT_FAMILY, TT_NAME_ID_FONT_SUBFAMILY,
  TT_NAME_ID_UNIQUE_ID, TT_NAME_ID_FULL_NAME, TT_NAME_ID_VERSION_STRING, 
  TT_NAME_ID_PS_NAME, TT_NAME_ID_TRADEMARK
  OpenType names: 
  TT_NAME_ID_MANUFACTURER, TT_NAME_ID_DESIGNER, TT_NAME_ID_DESCRIPTION,
  TT_NAME_ID_VENDOR_URL, TT_NAME_ID_DESIGNER_URL, TT_NAME_ID_LICENSE, 
  TT_NAME_ID_LICENSE_URL
  More: TT_NAME_ID_PREFERRED_FAMILY, TT_NAME_ID_PREFERRED_SUBFAMILY, 
  TT_NAME_ID_MAC_FULL_NAME
*/
const char* getfontname(FontData* fd, int idx)
{
  TT_UShort len;
  char* string;
  TT_UShort platform, encoding, language, id;
  static char *fontName=NULL;
  static int fontNameLen = 256;

  if (fontName == NULL) fontName = (char*)g_malloc(fontNameLen);
  if (fontName == NULL) out_of_memory(fontNameLen, __FILE__, __LINE__);

  TT_Face face = fd->ttdata.face;
  int found = 0;
  int i;
  for (i=0; i<fd->ttdata.properties.num_Names; i++) {
    int error = TT_Get_Name_ID(face, i, &platform, &encoding, &language, &id);
    if (error || id!=idx) continue;
    error = TT_Get_Name_String(face, i, &string, &len);
    if (error) continue;
    if ((platform == TT_PLATFORM_MICROSOFT) && 
	(encoding==TT_MS_ID_SYMBOL_CS||encoding==TT_MS_ID_UNICODE_CS) && 
	((language&0xFF)==0x9)) {
      found = 1;
      break;
    }
    if (platform==TT_PLATFORM_APPLE_UNICODE && 
	language==TT_MAC_LANGID_ENGLISH) {
      found = 1;
      break;
    }
  }
  if (!found) return "";
  // Found a Unicode english name
  while ((len/2+1)>fontNameLen) {
    fontNameLen *= 2;
    fontName = (char*)g_realloc(fontName, fontNameLen);
    if (fontName == NULL) out_of_memory(fontNameLen, __FILE__, __LINE__);
  }
  for (i=0; i<len; i+=2) fontName[i/2] = string[i+1];
  fontName[len/2] = '\0';
  return fontName;
}



// Tries to find a character map and returns it in charmap
// It looks for a Unicode or Symbol MS character map
// This is called when trying to show a character from its code
TT_Error tt_findcharmap(FontData* fd, TT_CharMap* charmap)
{
  TT_UShort platformID, encodingID;
  TT_UShort i;

  TT_Face face = fd->ttdata.face;
  int num = fd->ttdata.properties.num_CharMaps;
  for (i=0; i<num; i++) {
    int error = TT_Get_CharMap_ID(face, i, &platformID, &encodingID);
    if (error) continue;
    if (platformID==TT_PLATFORM_MICROSOFT && 
	(encodingID==TT_MS_ID_UNICODE_CS||encodingID==TT_MS_ID_SYMBOL_CS))
      return TT_Get_CharMap(face, i, charmap);
  }
  return TT_Err_Invalid_CharMap_Handle;
}



// Tries to find a normal (text) character map and returns it in charmap
// It looks for a Unicode character map
// This is called when trying to show a string
TT_Error tt_findtextcharmap(FontData* fd, TT_CharMap* charmap)
{
  TT_UShort platformID, encodingID;
  TT_UShort i;
  TT_Face face = fd->ttdata.face;
  int num = fd->ttdata.properties.num_CharMaps;

  // Try first to find a Unicode charmap
  for (i=0; i<num; i++) {
    int error = TT_Get_CharMap_ID(face, i, &platformID, &encodingID);
    if (error) continue;
    if ((platformID==TT_PLATFORM_MICROSOFT && 
	 encodingID==TT_MS_ID_UNICODE_CS) || 
	(platformID==TT_PLATFORM_APPLE_UNICODE))
      return TT_Get_CharMap(face, i, charmap);
  }

  // Try now a MacRoman charmap
  for (i=0; i<num; i++) {
    int error = TT_Get_CharMap_ID(face, i, &platformID, &encodingID);
    if (error) continue;
    if (platformID==TT_PLATFORM_MACINTOSH && encodingID==TT_MAC_ID_ROMAN) 
      return TT_Get_CharMap(face, i, charmap);
  }

  // No luck
  return TT_Err_Invalid_CharMap_Handle;
}



/*
  Look up the glyph with name 'name' in the given font and return
  its index in 'index'. Return value: Success or not
 */
bool tt_lookup(FontData *fd, const char *name, TT_UShort *index)
{
  TT_Error error;
  TT_Post  post;
  char *psname;

  if (fd->ttdata.PSloaded == FALSE) {
    error = TT_Load_PS_Names(fd->ttdata.face, &post);
    if (error) {
      add_error(fd, _("Cannot load PostScript names: %s"), 
	       TT_ErrToString18(error));
      return FALSE;
    }
    fd->ttdata.PSloaded = TRUE;
  }
  for (TT_UShort i=0; i<fd->num_glyphs; i++) {
    error = TT_Get_PS_Name(fd->ttdata.face, i, &psname);
    if (error) continue;
    if (strcmp(psname, name)==0) {
      *index = i;
      return TRUE;
    }
  }
  return FALSE;
}



/* Fills the FontData with data; fd->ttdata.face must be a valid value */
void fill_ttfontdata(FontData* fd)
{
  int kerntable;
  int error;

  // Get font properties
  TT_Get_Face_Properties(fd->ttdata.face, &fd->ttdata.properties);
  fd->num_glyphs = fd->ttdata.properties.num_Glyphs;
  fd->fontName = g_strdup(getfontname(fd, TT_NAME_ID_FULL_NAME));
  fd->fontFamily = g_strdup(getfontname(fd, TT_NAME_ID_FONT_FAMILY));
  fd->fontType = TTFONT;
  fd->ttdata.PSloaded = FALSE;

  // Get kerning directory and kerning pairs
  fd->ttdata.kernpairs = 0;
  error = TT_Get_Kerning_Directory(fd->ttdata.face, &fd->ttdata.kerndir);
  if (error) fd->ttdata.kerndir.nTables = 0;
  if (fd->ttdata.kerndir.nTables > 0) {
    for (kerntable=0; kerntable<fd->ttdata.kerndir.nTables; kerntable++) {
      error = TT_Load_Kerning_Table(fd->ttdata.face, kerntable);
      if (error) {
	add_error(fd, TT_ErrToString18(error));
	continue;
      }
      // By now, we only understand format 0, not format 2
      if (fd->ttdata.kerndir.tables[kerntable].format == 0) break;
    }
    if (kerntable == fd->ttdata.kerndir.nTables) 
      add_error(fd, _("Cannot find a kern table in format 0"));
    else {
      fd->ttdata.kernpairs = fd->ttdata.kerndir.tables[kerntable].t.kern0.nPairs;
      fd->ttdata.kerntable = kerntable;
    }
  }
}



bool is_ttfontcollection (gchar *pathname, FontData *fd)
{
  int error;
  /* Look if it has a .ttc extension */
  size_t len = strlen(pathname);
  if (len<4) return FALSE;
  if (!(pathname[len-4] == '.' &&
	(pathname[len-3] == 't' || pathname[len-3] == 'T') &&
	(pathname[len-2] == 't' || pathname[len-2] == 'T') &&
	(pathname[len-1] == 'c' || pathname[len-1] == 'C'))) return FALSE;
  
  error = TT_Open_Face(ttengine, pathname, &fd->ttdata.face);
  if (error) {
    add_error(fd, TT_ErrToString18(error));
    return FALSE;
  }
  TT_Get_Face_Properties(fd->ttdata.face, &fd->ttdata.properties);
  if (fd->ttdata.properties.num_Faces < 1) {
    TT_Close_Face(fd->ttdata.face);
    return FALSE;
  }
  return TRUE;
}


bool is_ttfont (gchar *pathname, FontData *fd)
{
  /* Look if it has a .ttf extension */
  size_t len = strlen(pathname);
  if (len<4) return FALSE;
  if (!(pathname[len-4] == '.' &&
	(pathname[len-3] == 't' || pathname[len-3] == 'T') &&
	(pathname[len-2] == 't' || pathname[len-2] == 'T') &&
	(pathname[len-1] == 'f' || pathname[len-1] == 'F'))) return FALSE;
  
  int error = TT_Open_Face(ttengine, pathname, &fd->ttdata.face);
  if (error) {
    add_error(fd, TT_ErrToString18(error));
    return FALSE;
  }
  fill_ttfontdata(fd);
  return TRUE;
}



GdkImage* tt_makechar(FontData* fd, TT_UShort index, double size, 
		      bool antialiasing)
{
  TT_Instance        instance;
  TT_Glyph           glyph;
  TT_Glyph_Metrics   metrics;
  TT_Raster_Map      bitmap;
  GdkImage* gdkimage;
  int       error;

  error = TT_New_Instance(fd->ttdata.face, &instance);
  if (error) {
    add_error(fd, _("Could not create instance"));
    return NULL;
  }
  error = TT_Set_Instance_Resolutions(instance, xresolution, yresolution);
  if (error) {
    add_error(fd, _("Could not set resolution"));
    return NULL;
  }
  error = TT_Set_Instance_CharSize(instance, (TT_F26Dot6)(size*64));
  if (error) {
    add_error(fd, _("Could not set size to %f"), size);
    return NULL;
  }
  error = TT_New_Glyph(fd->ttdata.face, &glyph);
  if (error) {
    add_error(fd, _("Could not create glyph object"));
    return NULL;
  }
  error = TT_Load_Glyph(instance, glyph, index, TTLOAD_DEFAULT);
  if (error) {
    add_error(fd, _("Could not load glyph %x"), index);
    return NULL;
  }

  TT_Get_Glyph_Metrics(glyph, &metrics); // In 26.6 format and grid-fitted
  TT_BBox bbox = metrics.bbox;  
  int width  = (bbox.xMax-bbox.xMin)/64; // In pixels
  int height = (bbox.yMax-bbox.yMin)/64;
  if (width==0 || height==0) return NULL;

  bitmap.width = width;   // number of pixels per line
  bitmap.rows  = height;  // number of rows (pixels)
  bitmap.flow  = TT_Flow_Down;
  if (antialiasing) // number of columns (bytes) per row
    bitmap.cols = (bitmap.width+3) & -4; // 4 byte padding
  else
    bitmap.cols = (bitmap.width+7)/8;    // 1 byte padding
  bitmap.size = bitmap.rows * bitmap.cols; // bit/pixmap size in bytes
  bitmap.bitmap = g_malloc0(bitmap.size);
  if (bitmap.bitmap == NULL) out_of_memory(bitmap.size, __FILE__, __LINE__);

  if (antialiasing)
    error = TT_Get_Glyph_Pixmap(glyph, &bitmap, -bbox.xMin, -bbox.yMin);
  else
    error = TT_Get_Glyph_Bitmap(glyph, &bitmap, -bbox.xMin, -bbox.yMin);
  if (error) {
    add_error(fd, _("Could not create glyph: %s"), TT_ErrToString18(error));
    return NULL;
  }
  TT_Done_Glyph(glyph);
  TT_Done_Instance(instance);

  if (antialiasing) {
    int image_w = width;  // or bitmap.cols
    gdkimage = gdk_image_new(GDK_IMAGE_NORMAL, visual, image_w, height);
    // Translate the freetype gray values (0-4) to the pixel values contained in graypalette[]
    for (int y=0, bytesdone=0; y<height; y++, bytesdone+=bitmap.cols) {
      for (int x=0; x<image_w; x++) {
	guint8 pixel = ((char*)bitmap.bitmap)[bytesdone+x];  // 0 - 4
	gdk_image_put_pixel(gdkimage, x, y, graypalette[pixel].pixel);
      }
    }
    g_free(bitmap.bitmap);
  }
  else // gdk_image_destroy will free the bitmap 
    gdkimage = gdk_image_new_bitmap(visual, bitmap.bitmap, width, height);
  return gdkimage;
}





static gint drawarea_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  WInfo *winfo;
  FontData *fd;
  GdkModifierType mask;
  gint state;
  gint x, y;
  gint width_w, height_w;  // Of the drawing_area window
  gint width, height;      // Of a cell
  gint col, row;  // In drawing_area
  gint chrsize;

  static GString *info_string=NULL;  // Character names shown below drawing_area
  static GtkWidget *popup = NULL;    // Enlarged character window
  static guint buttonclicked = 0;    // Button causing popup to activate

  if (info_string == NULL) info_string = g_string_new(NULL);

  winfo = (WInfo*)data;
  switch(event->type) {

  case GDK_LEAVE_NOTIFY:
    // Delete info_label if no mouse key pressed
    state = event->crossing.state;
    if (state&GDK_BUTTON1_MASK || state&GDK_BUTTON2_MASK || 
	state&GDK_BUTTON3_MASK)
      return FALSE;  
    g_string_truncate(info_string, 0);
    gdk_window_clear(winfo->info_label->window);  // Delete old info string
    return FALSE;

  case GDK_MOTION_NOTIFY:
    {
      if (event->motion.is_hint)
	gdk_window_get_pointer(event->motion.window, &x, &y, &mask);
      // Do not go further if mouse moved with a mouse key pressed
      if (mask&GDK_BUTTON1_MASK || mask&GDK_BUTTON2_MASK || 
	  mask&GDK_BUTTON3_MASK)
	return FALSE;  
      width = winfo->cellwidth;  height = winfo->cellheight;
      col = x/width;  row = y/height;
      TT_UShort index = 256*winfo->page + 16*row + col;  // Glyph number in the file  
      if (col<0 || col>=16 || row<0 || row>=16 || 
	  index>=winfo->fd->num_glyphs) {
	g_string_truncate(info_string, 0);
	gdk_window_clear(winfo->info_label->window);  // Delete old info string
	return FALSE; // Out of the grid
      }
      
      char *ps_name;
      if (winfo->fd->ttdata.PSloaded == FALSE) return FALSE;
      int error = TT_Get_PS_Name(winfo->fd->ttdata.face, index, &ps_name);
      if (error) return FALSE;
      if (strcmp(ps_name, info_string->str) == 0) return FALSE; // info_string not changed
      
      g_string_assign(info_string, ps_name);
      gdk_window_clear(winfo->info_label->window);  // Delete old info string
      gdk_draw_string(winfo->info_label->window, 
		      winfo->info_label->style->font, 
		      winfo->info_label->style->fg_gc[GTK_STATE_NORMAL], 
		      2, 2 + winfo->info_label->style->font->ascent, 
		      info_string->str);
      return FALSE; 
    }

  case GDK_BUTTON_PRESS:
    {
      fd = winfo->fd;
      if (buttonclicked != 0) return FALSE;  // Do not allow a second parallel click
      buttonclicked = event->button.button;  // Record which button was clicked
      switch (event->button.button) {
      case 1: chrsize = 100; break;
      case 2: chrsize = 200; break;
      case 3: chrsize = 300; break;
      default: // sorry, scrolling wheels don't do anything here 
	return FALSE;
      }
      width = winfo->cellwidth;
      height = winfo->cellheight;
      col = ((int)event->button.x)/width;
      row = ((int)event->button.y)/height;
      if (col<0 || col>=16 || row<0 || row>=16) return FALSE; // Out of the grid
      
      TT_UShort index = 256*winfo->page + 16*row + col;  // Glyph number in the file  
      if (index >= fd->num_glyphs) return FALSE;  // No such glyph
      GdkImage* gdkimage = tt_makechar(fd, index, chrsize, FALSE);  // No antialiasing
      if (gdkimage == NULL) return FALSE;
      GtkWidget* gtkimage = gtk_image_new(gdkimage, NULL);
      
      // Make popup window with the image
      popup = gtk_window_new(GTK_WINDOW_POPUP);
      gtk_container_border_width(GTK_CONTAINER(popup), 10);
      gtk_container_add(GTK_CONTAINER(popup), gtkimage);
      // Calculate position of the popup in the screen
      int pos_x = (int)event->button.x_root;
      int pos_y = (int)event->button.y_root;
      if (pos_x+gdkimage->width > gdk_screen_width()) 
	pos_x -= gdkimage->width;
      if (pos_y+gdkimage->height > gdk_screen_height()) 
	pos_y -= gdkimage->height;
      gtk_widget_set_uposition(popup, pos_x, pos_y);
      
      gtk_signal_connect(GTK_OBJECT(popup), "delete_event", 
			 GTK_SIGNAL_FUNC(gtk_false), NULL);
      gtk_signal_connect(GTK_OBJECT(popup), "destroy", 
			 (GtkSignalFunc)delete_image, (gpointer)gdkimage);
      gtk_widget_show_all(popup);
      gtk_grab_add(widget);
      return FALSE;
    }

  case GDK_BUTTON_RELEASE:
    if (event->button.button != buttonclicked) return FALSE;
    buttonclicked = 0;
    if (popup) {
      gtk_widget_destroy(popup);
      gtk_widget_destroyed(popup, &popup);  // Sets popup to NULL (2nd argument)
      gtk_grab_remove(widget);
      g_string_truncate(info_string, 0);
      gdk_window_clear(winfo->info_label->window);  // Delete old info string
    }
    return FALSE;

  default: break;
  }
  return FALSE;
}




static void destroy_fonttable(GtkWidget *widget, gpointer data) 
{
  WInfo *winfo = (WInfo*)data;
  FontData *fd = winfo->fd;
  fd->refcount--;  // FontData* no longer used by this window
  // If the font is no longer being displayed in the font list, try to delete it
  if (!fd->visible) destroyfont(fd);
  TT_Done_Instance(winfo->instance);

  gdk_pixmap_unref(winfo->pixmap);
  g_free(winfo);
}




static void saveasGIF(gpointer callback_data,
		      guint    callback_action,
		      GtkWidget *widget) 
{  
  int width, height;
  GtkWidget *window = (GtkWidget*)callback_data;  // The font map window
  WInfo *winfo = (WInfo*)gtk_object_get_data(GTK_OBJECT(window), "winfo");

  /* Get image from pixmap and put the reference to it 
     in the 'window' object */
  gdk_window_get_size(winfo->pixmap, &width, &height);
  GdkImage *image = gdk_image_get(winfo->pixmap, 0, 0, width, height);
  GtkWidget *savewindow = save_asgif(image);
  gtk_window_set_position(GTK_WINDOW(savewindow), GTK_WIN_POS_MOUSE);
  gtk_window_set_transient_for(GTK_WINDOW(savewindow), 
			       GTK_WINDOW(window));
  gtk_widget_show(savewindow);
  /* Do not return until the file save dialog window is closed,
   because the image will be destroyed */
  while (GTK_WIDGET_VISIBLE(savewindow)) gtk_main_iteration(); 
  gdk_image_destroy(image); 
}



/* 
   Actually draw the contents of the fonttable in the pixmap
   256 characters will be drawn, the page indicated by winfo->page
 */
static void fill_pixmap(WInfo *winfo)
{
  FontData *fd = winfo->fd;
  int sizex = 0, sizey = 0; /* Max size (in x or y) of the 256 glyphs */
  int width_w, height_w;    /* Width and height of the grid */
  int xMin = INT_MAX, yMin = INT_MAX;
  int maxnum = MIN(256, fd->num_glyphs - winfo->page*256);  // Number of glyphs on this page
  GdkImage *gdkimage;
  int error;

  TT_Glyph      glyph;
  TT_Raster_Map bitmap;
  TT_Header     *header = fd->ttdata.properties.header;
  GtkWidget *toplevel = gtk_widget_get_toplevel(winfo->drawing_area);


  error = TT_New_Glyph(fd->ttdata.face, &glyph);
  if (error) {
    errormsg(toplevel, _("Could not create glyph object"));
    return;
  }

  /* Loop through the 256 glyphs to get the bbox (sizex, sizey)
     and the left and lower most coordinate (xMin, yMin) */
  for (TT_UShort i=0; i<256; i++) {
    TT_UShort flags = TTLOAD_SCALE_GLYPH;
    TT_Glyph_Metrics metrics;

    error = TT_Load_Glyph(winfo->instance, glyph, 256*winfo->page + i, flags);
    TT_Get_Glyph_Metrics(glyph, &metrics);
    
    xMin = MIN(xMin, metrics.bbox.xMin);
    yMin = MIN(yMin, metrics.bbox.yMin);
    sizex = MAX(sizex, (metrics.bbox.xMax-metrics.bbox.xMin)/64);
    sizey = MAX(sizey, (metrics.bbox.yMax-metrics.bbox.yMin)/64);
  }

  /* Convert xMin and yMin from fractional pixels (26.6) to integers */
  int maxleft = xMin/64;
  int maxdescent = yMin/64; 

  int cellwidth = winfo->cellwidth = sizex - maxleft + 2*sizex/8;
  int cellheight = winfo->cellheight = sizey - maxdescent + 2*sizey/8;
  width_w = 16*cellwidth + 1;
  height_w = 16*cellheight + 1;

  /* The glyphs are not drawn on each square at (0,0) but displaced by 
     a certain offset, in order for them to have enough room and fit within
     the square */
  int offset_x = sizex/8 - maxleft;
  int offset_y = sizey/8 - maxdescent;

  /* Make pixmap whre drawing will take place */
  if (winfo->pixmap) gdk_pixmap_unref(winfo->pixmap);
  winfo->pixmap = gdk_pixmap_new(mainwindow->window, width_w,
				 height_w, -1);

  bitmap.width = width_w;  // number of pixels per line
  bitmap.rows = height_w;  // number of rows (pixels)
  bitmap.flow = TT_Flow_Down;
  if (winfo->antialiasing) // number of columns (bytes) per row
    bitmap.cols = (bitmap.width+3) & -4; // 4 byte padding
  else
    bitmap.cols = (bitmap.width+7)/8;    // 1 byte padding
  bitmap.size = bitmap.rows * bitmap.cols; // bitmap size in bytes
  bitmap.bitmap = g_malloc0(bitmap.size);
  if (bitmap.bitmap == NULL) out_of_memory(bitmap.size, __FILE__, __LINE__);

  GtkWidget *top = gtk_widget_get_toplevel(winfo->drawing_area);
  if (GTK_WIDGET_VISIBLE(top)) set_window_busy(top, TRUE);

  // Make pixmap white
  gdk_draw_rectangle(winfo->pixmap, mainwindow->style->white_gc, 
		     TRUE, 0, 0, -1, -1); 

  // Put glyphs; i=column, j=row
  for (int j=0; j<=(maxnum-1)/16; j++)  
    for (int i=0; i<MIN(16, maxnum-j*16); i++) {
      
      gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 
			      double(16*j+i)/maxnum);
      while (gtk_events_pending()) gtk_main_iteration();
      
      TT_UShort idx = 256*winfo->page + 16*j + i; 
      TT_UShort flags = TTLOAD_SCALE_GLYPH;
      if (winfo->hinting) flags |= TTLOAD_HINT_GLYPH;
      error = TT_Load_Glyph(winfo->instance, glyph, idx, flags);
      if (error) {
	add_error(fd, _("Could not load glyph %hu"), idx);
	continue;
      }
      if (winfo->antialiasing)
	error = TT_Get_Glyph_Pixmap(glyph, &bitmap, 
				    (cellwidth*i+offset_x)*64, 
				    (cellheight*(15-j)+offset_y)*64); 
      else
	error = TT_Get_Glyph_Bitmap(glyph, &bitmap, 
				    (cellwidth*i+offset_x)*64, 
				    (cellheight*(15-j)+offset_y)*64); 
      if (error) {
	add_error(fd, _("Could not create glyph %hu: %s"), idx, 
		  TT_ErrToString18(error));
	continue;
      }
    }

  if (winfo->antialiasing) {
    gdkimage = gdk_image_new(GDK_IMAGE_NORMAL, visual, width_w, height_w);
    /* Translate the freetype gray values (0-4) to the pixel values 
       contained in graypalette[] */
    for (int y=0, bytesdone=0; y<height_w; y++, bytesdone+=bitmap.cols) {
      for (int x=0; x<width_w; x++) {
	guint8 pixel = ((char*)bitmap.bitmap)[bytesdone+x];  // 0 - 4
	gdk_image_put_pixel(gdkimage, x, y, graypalette[pixel].pixel);
      }
    }
  }
  else {
    gdkimage = gdk_image_new_bitmap(visual, bitmap.bitmap, 
				    width_w, height_w);
  }

  /* Draw the image with the glyphs on the pixmap */
  gdk_draw_image(winfo->pixmap, bwGC, gdkimage, 0, 0, 0, 0, 
		 width_w, height_w);
  /* Draw the grid and lines */
  for (int i=0; i<=16; i++) {  
    /* Grid */
    gdk_draw_line(winfo->pixmap, defGC, 0, cellheight*i, 
		  16*cellwidth, cellheight*i);
    gdk_draw_line(winfo->pixmap, defGC, cellwidth*i, 0, 
		  cellwidth*i, 16*cellheight);
    /* Baselines */
    if (winfo->baseline && i!=0)
      gdk_draw_line(winfo->pixmap, blueGC, 
		    0, cellheight*i-offset_y,
		    16*cellwidth, cellheight*i-offset_y);
    /* Origins */
    if (winfo->origin && i!=16)
      gdk_draw_line(winfo->pixmap, blueGC, 
		    cellwidth*i+offset_x, 0,
		    cellwidth*i+offset_x, 16*cellheight);
  }
  
  gdk_image_destroy(gdkimage);
  /* If no antialiasing, gdk_image_destroy already freed bitmap.bitmap */
  if (winfo->antialiasing) g_free(bitmap.bitmap);
  TT_Done_Glyph(glyph);

  /* Set the size of the area so the scwindow knows */
  gtk_drawing_area_size(GTK_DRAWING_AREA(winfo->drawing_area), 
			width_w, height_w);
  /* Notify the window that the pixmap changed */
  gdk_draw_pixmap(winfo->drawing_area->window, bwGC, winfo->pixmap, 
		  0, 0, 0, 0, -1, -1); 

  gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 0.0);
  if (GTK_WIDGET_VISIBLE(top)) set_window_busy(top, FALSE);
}




/* 
   Called to toggle the representation of the map in the 
   font table window
*/
static void switch_options(gpointer callback_data,
			   guint    callback_action,
			   GtkWidget *widget) 
{
  GtkWidget *item;
  GtkWidget *window = (GtkWidget*)callback_data;

  WInfo *winfo = (WInfo*)gtk_object_get_data(GTK_OBJECT(window), "winfo");

  item = gtk_item_factory_get_item(winfo->item_factory, 
				   _("/View/Antialiasing"));
  winfo->antialiasing = GTK_CHECK_MENU_ITEM(item)->active;
  item = gtk_item_factory_get_item(winfo->item_factory, 
				   _("/View/Hinting"));
  winfo->hinting = GTK_CHECK_MENU_ITEM(item)->active;
  item = gtk_item_factory_get_item(winfo->item_factory, 
				   _("/View/Baseline"));
  winfo->baseline = GTK_CHECK_MENU_ITEM(item)->active;
  item = gtk_item_factory_get_item(winfo->item_factory, 
				   _("/View/Origin"));
  winfo->origin = GTK_CHECK_MENU_ITEM(item)->active;

  fill_pixmap(winfo); 
}



static GtkItemFactoryEntry ftmenu_items[] = { 
  // path accelerator callback callback_action item_type
{ N_("/_File"),      NULL,           
  NULL,  0, "<Branch>" },
{ N_("/File/_Save as GIF"),  N_("<control>G"),  
    (GtkItemFactoryCallback)saveasGIF,  0, NULL },
{ N_("/File/_Close"),  N_("<control>C"),  
    (GtkItemFactoryCallback)gtk_widget_destroy,  0, NULL },
{ N_("/_View"),      NULL,           
    NULL, 0, "<Branch>" },
{ N_("/View/_Antialiasing"),  N_("<control>A"),  
    (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"},
{ N_("/View/_Hinting"),  N_("<control>H"),  
    (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"},
{ N_("/View/_Baseline"),  N_("<control>B"),  
    (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"},
{ N_("/View/_Origin"),  N_("<control>O"),  
    (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"},
};



/* Return success status */
static bool drawfont_inframe(GtkWidget *frame, WInfo *winfo)
{
  int error;
  TT_Instance instance;
  TT_Header *header = winfo->fd->ttdata.properties.header;
  GtkWidget *toplevel = gtk_widget_get_toplevel(frame);

  error = TT_New_Instance(winfo->fd->ttdata.face, &instance);
  if (error) {
    errormsg(toplevel, _("Could not create instance"));
    return FALSE;
  }
  error = TT_Set_Instance_Resolutions(instance, 
				      xresolution, yresolution);
  if (error) {
    errormsg(toplevel, _("Could not set resolution"));
    return FALSE;
  }
  error = TT_Set_Instance_CharSize(instance, (TT_F26Dot6)(winfo->fontsize*64));
  if (error) {
    errormsg(toplevel, _("Could not set size to %f"), winfo->fontsize);
    return FALSE;
  }

  // Scrolling window with drawing area
  GtkWidget* scwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwindow), 
				 GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(frame), scwindow);

  GtkWidget* drawing_area = gtk_drawing_area_new();
  gtk_widget_set_events(drawing_area, GDK_EXPOSURE_MASK |
			GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK |
			GDK_POINTER_MOTION_MASK | 
			GDK_POINTER_MOTION_HINT_MASK |
			GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwindow), 
					drawing_area);

  // Set infos for the mouse_event function (called when mouse clicked on drawing area)
  winfo->drawing_area = drawing_area;
  winfo->instance = instance;
  fill_pixmap(winfo);

  /* And the size of the scwindow to request a view of the whole pixmap;
     if the dimensions of the screen are exceeded, it will be limited in
     tt_fonttable()
   */
  gtk_widget_set_usize(GTK_BIN(scwindow)->child, 
		       16*winfo->cellwidth+1, 16*winfo->cellheight+1);

  // Connections
  gtk_signal_connect(GTK_OBJECT(drawing_area), "destroy", 
		     (GtkSignalFunc)destroy_fonttable, (gpointer)winfo);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "expose_event", 
		     (GtkSignalFunc)expose_event, (gpointer)winfo);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "motion_notify_event", 
		     (GtkSignalFunc)drawarea_event, (gpointer)winfo);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "leave_notify_event", 
		     (GtkSignalFunc)drawarea_event, (gpointer)winfo);

  if (winfo->fontsize<100) {  // Expansion window only for small sizes
    gtk_signal_connect(GTK_OBJECT(drawing_area), "button_press_event", 
		       (GtkSignalFunc)drawarea_event, (gpointer)winfo);
    gtk_signal_connect(GTK_OBJECT(drawing_area), "button_release_event", 
		       (GtkSignalFunc)drawarea_event, (gpointer)winfo);
  }
  return TRUE;
}


/* Called when user pressed ENTER on combo box */
static void do_fill_pixmap(GtkWidget *, WInfo *winfo)
{
  /* The page number is the number of the selected item in the combo list */
  GtkList *list = GTK_LIST(GTK_COMBO(winfo->combo)->list);
  int item_index = g_list_index(list->children, list->selection->data);
  winfo->page = item_index;
  fill_pixmap(winfo);
}



/*
  Called when mouse key is released and the combo box was displayed
 */
static int combo_release(GtkWidget *widget, GdkEvent *event, WInfo *winfo) 
{
  /* If the popwin still has the grab, the click was not in the 
     list area, so ignore */
  if (!GTK_WIDGET_HAS_GRAB(GTK_COMBO(winfo->combo)->popwin)) {
    while (gtk_events_pending()) gtk_main_iteration(); 
    XSync(GDK_DISPLAY(), False);  /* Get the combox box out of screen */
    do_fill_pixmap(widget, winfo);
  }
  return FALSE;
}



void tt_fonttable(FontData* fd, double size)
{
  int error, i;

  if (fd->num_glyphs < 1) return;  // Ill case

  if (fd->ttdata.PSloaded == FALSE) {
    TT_Post post;
    error = TT_Load_PS_Names(fd->ttdata.face, &post);
    if (error) 
      add_error(fd, _("Cannot load PostScript name table: %s"), 
		TT_ErrToString18(error));
    else 
      fd->ttdata.PSloaded = TRUE;
  }

  // Create window
  GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name(window, "fonttable window");
  gtk_window_set_title(GTK_WINDOW(window), 
		       getfontname(fd, TT_NAME_ID_FULL_NAME));
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
  gtk_signal_connect_object(GTK_OBJECT(window), "delete_event", 
			    GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_widget_realize(window);


  // Global vertical container
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(window), vbox);

  // Menubar
  gint nmenu_items = sizeof(ftmenu_items) / sizeof (ftmenu_items[0]);
  for (i=0; i<nmenu_items; i++) // Translate menu labels
    ftmenu_items[i].path = _(ftmenu_items[i].path);
  for (i=0; i<nmenu_items; i++) // Translate menu shortcuts
    if (ftmenu_items[i].accelerator)  // Might be NULL...
      ftmenu_items[i].accelerator = _(ftmenu_items[i].accelerator);

  GtkAccelGroup *accel_group = gtk_accel_group_new();
  GtkItemFactory *item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, 
						      "<main>", accel_group);
  // Pass window as callback_data
  gtk_item_factory_create_items(item_factory, nmenu_items, 
				ftmenu_items, window);
  GtkWidget *menubar = gtk_item_factory_get_widget(item_factory, "<main>");
  gtk_accel_group_attach(accel_group, GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, FALSE, 0);

  // Set antialias menu item to current state of check button
  GtkWidget *aaliasmenu = gtk_item_factory_get_item(item_factory, 
						    _("/View/Antialiasing"));
  GTK_CHECK_MENU_ITEM(aaliasmenu)->active = antialiasing;

  // Set hinting menu item to TRUE
  GtkWidget *hintmenu = gtk_item_factory_get_item(item_factory, 
						  _("/View/Hinting"));
  GTK_CHECK_MENU_ITEM(hintmenu)->active = TRUE;

  // Set showing the baseline to FALSE
  GtkWidget *baselinemenu = gtk_item_factory_get_item(item_factory, 
						  _("/View/Baseline"));
  GTK_CHECK_MENU_ITEM(baselinemenu)->active = FALSE;

  // Set showing the origin to FALSE
  GtkWidget *originmenu = gtk_item_factory_get_item(item_factory, 
						  _("/View/Origin"));
  GTK_CHECK_MENU_ITEM(originmenu)->active = FALSE;

  // Upper horizontal box containing table selector
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 10);

  // Actually draw the font map
  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_NONE);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

  WInfo *winfo = g_new(WInfo, 1);
  winfo->fd = fd;
  winfo->page = 0;
  winfo->pixmap = NULL;
  winfo->item_factory = item_factory;
  winfo->fontsize = size;
  winfo->hinting = GTK_CHECK_MENU_ITEM(hintmenu)->active;
  winfo->antialiasing = GTK_CHECK_MENU_ITEM(aaliasmenu)->active;
  winfo->baseline = GTK_CHECK_MENU_ITEM(baselinemenu)->active;
  winfo->origin = GTK_CHECK_MENU_ITEM(originmenu)->active;

  bool success = drawfont_inframe(frame, winfo);
  if (success == FALSE) {
    g_free(winfo);
    gtk_widget_destroy(window);
    return;
  }
  winfo->fd->refcount++;  // FontData* is now used by this window
  gtk_object_set_data(GTK_OBJECT(window), "winfo", winfo);

  // Table combo box
  GtkWidget *clabel = gtk_label_new(_("Table:"));
  gtk_widget_set_style(clabel, stylebold);
  gtk_box_pack_start(GTK_BOX(hbox), clabel, FALSE, FALSE, 10);
  GtkWidget *combo = gtk_combo_new();
  gtk_combo_disable_activate(GTK_COMBO(combo)); // Do not popup list when ENTER
  gtk_combo_set_use_arrows_always(GTK_COMBO(combo), TRUE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
  gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);

  gtk_list_clear_items(GTK_LIST(GTK_COMBO(combo)->list), 0, -1); 
  for (int page=0; page<=(fd->num_glyphs-1)/256; page++) {
    char labelstr[128];
    sprintf(labelstr, _("Page %d"), page+1);           
    GtkWidget *item = gtk_list_item_new_with_label(labelstr);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(combo)->list), item); 
  }

  
  // Refresh font map when user presses ENTER
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "activate", 
		     (GtkSignalFunc)do_fill_pixmap, winfo);
  // Also when user releases mouse button in combo box list
  gtk_signal_connect_after(GTK_OBJECT(GTK_COMBO(combo)->list), 
			   "button_release_event", 
			   (GtkSignalFunc)combo_release, winfo);

  // Info label
  GtkWidget *evb = gtk_event_box_new();
  gtk_widget_set_name(evb, "glyphname label");
  gtk_box_pack_end(GTK_BOX(vbox), evb, FALSE, FALSE, 0);
  GtkWidget *label = gtk_label_new(NULL);
  gtk_widget_set_name(label, "glyphname label");
  gtk_misc_set_padding(GTK_MISC(label), 0, 2);
  gtk_container_add(GTK_CONTAINER(evb), label);

  winfo->info_label = evb;
  winfo->combo = combo;

  /* Compute the size requested by the widgets in the window */
  GtkRequisition req;
  gtk_widget_show_all(vbox);
  gtk_widget_size_request(window, &req);

  /* Set the size so that the complete fontmap is visible on screen, 
     but only up to the screen size */
  gtk_widget_set_usize(window, 
		       MIN(req.width,  gdk_screen_width() -30),
  		       MIN(req.height, gdk_screen_height()-40));
  gtk_widget_show(window);
}




static inline guint32 val(const char* ptr)
{
  guint16 l = ((guint16*)ptr)[0];
  guint16 r = ((guint16*)ptr)[1];
  return (((guint32)l)<<16) + (guint32)r;
}


// Returns kerning value in FUnits
static TT_FWord getkerning(FontData* fd, TT_UShort ch1, TT_UShort ch2)
{
  int kerntable = fd->ttdata.kerntable;
  guint32 target = (((guint32)ch1)<<16) + (guint32)ch2;

  if (ch2 == 0) return 0;   // Save time: end of string
  TT_UShort searchRange = fd->ttdata.kerndir.tables[kerntable].t.kern0.searchRange;
  TT_UShort rangeShift  = fd->ttdata.kerndir.tables[kerntable].t.kern0.rangeShift;
  char* base = (char*)(fd->ttdata.kerndir.tables[kerntable].t.kern0.pairs);
  char* top  = base + searchRange;

  if (target < val(base)) return 0;  // value < lower limit
  if (searchRange>6 && target>val(top-6)) {  // if value > upper limit
    // Should also be done with top = base if searchRange/6 is not a power of 2
    // Linear search in this zone
    for (char* ptr = top; ptr < top+rangeShift; ptr += 6) 
      if (target == val(ptr)) return ((TT_FWord*)ptr)[2];
    return 0;  // Not found
  }  
  if (target==val(base)) return ((TT_FWord*)base)[2];

  // Binary search between base and top (open interval)
  while (searchRange>6) {  // Each element in the table has 6 bytes
    searchRange /= 2;
    char *next = base + searchRange;
    if (target>val(next)) base = next;
    else { 
      if (target<val(next)) 
	top = next;
      else 
	return ((TT_FWord*)next)[2];   // Found!!
    }
  }
  return 0;  // No such kerning pair
}




GdkImage* tt_makestring(FontData* fd, const char* string, double size, 
			bool kerning, bool antialiasing)
{
  TT_Instance        instance;
  TT_Glyph           glyph;

  TT_Glyph_Metrics  metrics;
  TT_Raster_Map     bitmap;
  TT_BBox           bbox;
  TT_CharMap        charmap;
  GdkImage *gdkimage, *gdkimagebig;
  int error;

  error = tt_findtextcharmap(fd, &charmap);
  if (error) {
    add_error(fd, _("Could not find proper text charmap"));
    return NULL;
  }
  error = TT_New_Instance(fd->ttdata.face, &instance);
  if (error) {
    add_error(fd, _("Could not create instance"));
    return NULL;
  }
  error = TT_Set_Instance_Resolutions(instance, xresolution, yresolution);
  if (error) {
    add_error(fd, _("Could not set resolution"));
    return NULL;
  }
  error = TT_Set_Instance_CharSize(instance, (TT_F26Dot6)(size*64));
  if (error) {
    add_error(fd, _("Could not set size to %f"), size);
    return NULL;
  }
  error = TT_New_Glyph(fd->ttdata.face, &glyph);
  if (error) {
    add_error(fd, _("Could not create glyph object"));
    return NULL;
  }

  TT_Header *header = fd->ttdata.properties.header;
  bbox.xMin = header->xMin;  // BBox of the whole font, in Font Units
  bbox.yMin = header->yMin;
  bbox.xMax = header->xMax;
  bbox.yMax = header->yMax;

  int sizex = bbox.xMax - bbox.xMin;  // In font units
  int sizey = bbox.yMax - bbox.yMin;
  if (sizex<=0 || sizey<=0) {
    add_error(fd, _("Invalid bounding box in font"));
    return NULL;
  }
  double xscale = xresolution/72.0;
  double yscale = yresolution/72.0;

  TT_F26Dot6 maxdescent = (TT_F26Dot6)(bbox.yMin*size*yscale/header->Units_Per_EM)*64;  // In 26.6 format 
  // Get the maximum size of the cell surrounding the glyph
  int width = (int)((sizex*size*xscale)/header->Units_Per_EM);  // Pixels
  int height = (int)((sizey*size*yscale)/header->Units_Per_EM);
  width *= strlen(string);  // Maximum size of the whole string

  bitmap.width = width;   // number of pixels per line
  bitmap.rows  = height;  // number of rows (pixels)
  bitmap.flow = TT_Flow_Down;
  if (antialiasing) // number of columns (bytes) per row
    bitmap.cols = (bitmap.width+3) & -4; // 4 byte padding
  else
    bitmap.cols = (bitmap.width+7)/8;    // 1 byte padding
  bitmap.size = bitmap.rows * bitmap.cols; // bit/pixmap size in bytes
  bitmap.bitmap = g_malloc0(bitmap.size);  // Initialised to zeroes
  if (bitmap.bitmap == NULL) out_of_memory(bitmap.size, __FILE__, __LINE__);

  TT_F26Dot6 pen_x = 0;    // Pen positions in the bitmap in 26.6 format
  TT_F26Dot6 pen_y = -maxdescent;
  TT_F26Dot6 maxy = 0;  // Maximal value of the yMax of the bbox of all characters
  TT_F26Dot6 miny = 0;
  TT_F26Dot6 mx = 0; // Maximal value of x where a pixel in the bitmap is written
  for (const guchar *ptr=(guchar*)string; *ptr; ptr++) { // ptr must be unsigned!
    TT_UShort index = TT_Char_Index(charmap, *ptr);
    error = TT_Load_Glyph(instance, glyph, index, TTLOAD_DEFAULT);
    if (error) {
      add_error(fd, _("Could not load glyph %hx"), index);
      continue;
    }
    TT_Get_Glyph_Metrics(glyph, &metrics); // In 26.6 format and grid-fitted
    if (ptr==(guchar*)string) pen_x = -metrics.bbox.xMin; // For the first character
    maxy = MAX(maxy, metrics.bbox.yMax);
    miny = MIN(miny, metrics.bbox.yMin);
    mx   = MAX(mx, pen_x + metrics.bbox.xMax);
    if (antialiasing)
      error = TT_Get_Glyph_Pixmap(glyph, &bitmap, pen_x, pen_y);  
    else
      error = TT_Get_Glyph_Bitmap(glyph, &bitmap, pen_x, pen_y);  
    if (error) {
      add_error(fd, _("Could not create glyph %c: %s"), *ptr, 
		TT_ErrToString18(error));
      continue;
    }
    pen_x += metrics.advance;
    if (kerning && fd->ttdata.kerndir.nTables>0) {
      double kern = getkerning(fd, index, TT_Char_Index(charmap, ptr[1]))*size*xscale/header->Units_Per_EM;  // In pixels
      pen_x += ((TT_F26Dot6)(64*kern)+32)&-64;  // Round it correctly
    }
  }
  TT_Done_Glyph(glyph);
  TT_Done_Instance(instance);

  if (antialiasing) {
    gdkimagebig = gdk_image_new(GDK_IMAGE_NORMAL, visual, width, height);
    // Translate the freetype gray values (0-4) to the pixel values contained in graypalette[]
    for (int y=0, bytesdone=0; y<height; y++, bytesdone+=bitmap.cols) {
      for (int x=0; x<width; x++) {
	guint8 pixel = ((char*)bitmap.bitmap)[bytesdone+x];  // 0 - 4
	gdk_image_put_pixel(gdkimagebig, x, y, graypalette[pixel].pixel);
      }
    }
    g_free(bitmap.bitmap);
  }
  else // no antialiasing: gdk_image_destroy will free the bitmap 
    gdkimagebig = gdk_image_new_bitmap(visual, bitmap.bitmap, width, height);

  // Now get the string without margins
  gdkimage = gdk_image_subimage(gdkimagebig, 0, height-(pen_y+maxy)/64, 
				mx/64, (maxy-miny)/64); 
  gdk_image_destroy(gdkimagebig);
  return gdkimage;
}






// ********************* Section handling the download of T42 fonts

static GSList *table_list;   // Global; makes things easier, so please allow it
static const int LINE = 36;  // Bytes per line
static const guint MAXSTRING = 65534;  // Maximal length of a PostScript string


inline int scale(FontData* fd, TT_Pos v)
{
  return (v*1000)/fd->ttdata.properties.header->Units_Per_EM;
}



// Makes list of tables to download available in table_list
// Return TRUE for success, FALSE for failure
static bool get_table_list(FontData *fd)
{
  static const char *t42_table_name[] = {"cvt ", "fpgm", "glyf", "head", "hhea", "hmtx", "loca", "maxp", "prep"};  // Sorted list!!
  TT_Long tlen;

  // Try to load all tables in t42_table_name
  for (guint i=0; i<sizeof(t42_table_name)/sizeof(t42_table_name[0]); i++) {
    TT_ULong tag = MAKE_TT_TAG(t42_table_name[i][0],t42_table_name[i][1],t42_table_name[i][2],t42_table_name[i][3]);
    tlen = 0;
    // Get length of table
    int error = TT_Get_Font_Data(fd->ttdata.face, tag, 0, NULL, &tlen);
    if (error) 
      if (strcmp(t42_table_name[i], "cvt ")==0 || 
	  strcmp(t42_table_name[i], "fpgm")==0 || 
	  strcmp(t42_table_name[i], "prep")==0) {  // Optional tables
	/*add_error(fd, _("Warning: table %s not present"), 
	  t42_table_name[i]);*/
	continue;
      }
      else {
	add_error(fd, _("Invalid font: Table %s is missing"), 
		  t42_table_name[i]);
	return FALSE;
      }
    // Check length of table; only glyf can be larger than MAXSTRING
    if (tlen>(TT_Long)MAXSTRING && strcmp(t42_table_name[i], "glyf")!=0) {  
      add_error(fd, _("Invalid font: Table %s is longer than %d bytes"), 
		t42_table_name[i], MAXSTRING);
      return FALSE;
    }

    /* Reserve memory for the table; the amount shall be a multiple of 4,
       so that we can read later in groups of 4 bytes */
    TT_Table *table = make_new_table(NULL, tlen);
    // Load table into memory
    error = TT_Get_Font_Data(fd->ttdata.face, tag, 0, table->data, &tlen);
    if (error) {
      add_error(fd, _("Invalid font: Cannot load table %s"), table_list[i]);
      return FALSE;
    }
    table->name = t42_table_name[i];
    table->tag = tag;
    table->length = tlen;
    table_list = g_slist_append(table_list, table);
  }
  return TRUE;
}


/*
  Calculate checksum and offset of all tables
*/
static void complete_tables(FontData *fd)
{
  guint num_tables = g_slist_length(table_list);  // Number of tables in downloaded font
  TT_ULong offset = 12 + 16*num_tables;     // Begin + table directory

  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    TT_ULong padded_length = ((tb->length+3)&~3); 
    int n_longs = padded_length/sizeof(TT_ULong);
    tb->checksum = 0;
    guchar *p = tb->data;
    while (n_longs--) {
      TT_ULong value = MAKE_TT_TAG(p[0], p[1], p[2], p[3]);
      tb->checksum += value;
      p += sizeof(TT_ULong);
    }
    tb->offset = offset;
    offset += padded_length;  // a table must start at a 4 byte boundary
  }  
}


// Free the table_list list made by get_table_list
static void free_table_list(void)
{
  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    g_free(tb->data);
    g_free(tb);
  }
  g_slist_free(table_list);
  table_list = NULL;
}



// Write a table piece (a string for Adobe)
static void write_string(FILE *ofp, bool pad, TT_Table *tb, 
			 TT_ULong offset, TT_ULong length)
{
  static const char *hexchar = "0123456789ABCDEF";
  int pos = 0;  // Counts the number of bytes so far in the line

  fputs("<\n", ofp);
  for (guchar *p=tb->data+offset; p<tb->data+offset+length; p++) {
    if (pos++ == LINE) {
      fputc('\n', ofp);
      pos = 1;
    }
    fputc(hexchar[(*p>>4)&0xf], ofp); // Translate the byte at p to hex
    fputc(hexchar[*p&0xf], ofp);
  }
  if (pad) {
    TT_ULong nbytes = ((tb->length+3)&~3) - tb->length;  // 0-3
    while (nbytes--) fputs("00", ofp);  // Pad table with zeros to a 4 byte boundary
  }
  fputs("\n00>", ofp);  // Padding byte for compatibility with PS versions prior to 2013
}



static void set_glyph_offset(TT_Table *loca, TT_UShort index, 
			     TT_Short fmt, TT_ULong offset)
{
  TT_ULong pos;

  switch (fmt) {
  case 0:  // Short format
    pos = sizeof(TT_UShort) * index;
    offset /= 2;
    loca->data[pos] = (offset&0xFFFF)>>8;
    loca->data[pos+1] = offset&0xFF;
    break;
  case 1:  // Long format
    pos = sizeof(TT_ULong) * index;
    loca->data[pos] = (offset>>24)&0xFF;
    loca->data[pos+1] = (offset>>16)&0xFF;
    loca->data[pos+2] = (offset>>8)&0xFF;
    loca->data[pos+3] = offset&0xFF;
    break;
  }
}


static TT_ULong get_glyph_offset(TT_Table *loca, TT_UShort ch, 
				 TT_Short Index_To_Loc_Format)
{
  TT_ULong pos, offset;

  switch (Index_To_Loc_Format) {
  case 0:  // Short format
    pos = sizeof(TT_UShort) * ch;
    offset = 2*((loca->data[pos] << 8) + loca->data[pos+1]);
    break;
  case 1:  // Long format
    pos = sizeof(TT_ULong) * ch;
    offset = MAKE_TT_TAG(loca->data[pos], loca->data[pos+1], loca->data[pos+2], loca->data[pos+3]);
    break;
  }
  return offset;
}


/* Make new TT_Table copying original table, of length len.
   The new table will be padded to a 4 byte boundary length */
static TT_Table* make_new_table(TT_Table *orig, TT_ULong len)
{
  TT_ULong padlen = (len+3) & ~3;
  TT_Table *tb = g_new(TT_Table, 1);
  if (tb && orig) *tb = *orig;
  if (tb) tb->data = g_new(guchar, padlen);
  if (tb==NULL || tb->data==NULL) 
    out_of_memory(padlen*sizeof(guchar), __FILE__, __LINE__);

  while (padlen > len) 
    tb->data[--padlen] = 0;  // Pad extra bytes
  tb->length = len;
  return tb;
}


void array_insert(GArray *array, TT_UShort v)
{
  for (TT_UShort i=0; i<array->len; i++) {
    TT_UShort index = g_array_index(array, TT_UShort, i);
    if (index == v) return;
  }
  g_array_append_vals(array, &v, 1);
}


static int compare_ushorts(const void *p1, const void *p2)
{
  TT_UShort v1 = *(TT_UShort*)p1;
  TT_UShort v2 = *(TT_UShort*)p2;
  return v1-v2;
}


/*
  Go thru all glyphs (indexes) contained in 'codes', and see if
  they are compound (i.e., depend on other glyphs). If so,
  include these glyphs in the array.
  After this function, 'codes' contains the complete list of 
  glyphs to download
 */
static void add_compound_glyphs(FontData *fd, GArray *codes)
{
  guint n_glyphs = codes->len;
  TT_Header *header = fd->ttdata.properties.header;
  TT_Short fmt = header->Index_To_Loc_Format;

  // Find the 'glyf' table 
  TT_Table *glyf = NULL;
  for (GSList *elem = table_list; elem; elem = elem->next) {
    glyf = (TT_Table*)elem->data;
    if (strcmp(glyf->name, "glyf")==0) break;
  }
  // Find the 'loca' table 
  TT_Table *loca = NULL;
  for (GSList *elem = table_list; elem; elem = elem->next) {
    loca = (TT_Table*)elem->data;
    if (strcmp(loca->name, "loca")==0) break;
  }

  /* 
     Go thru all glyphs present originally in the array;
     if any is a compound glyph (number of contours < 0) look
     for the glyphs it needs and add them to the array
  */
  for (TT_UShort i=0; i<n_glyphs; i++) {
    TT_UShort index = g_array_index(codes, TT_UShort, i);
    TT_ULong o1 = get_glyph_offset(loca, index, fmt);   
    TT_Short nc = (glyf->data[o1]<<8) + glyf->data[o1+1];
    if (nc >= 0) continue;  

    TT_UShort flags, gi;
    TT_ULong off = o1+10;
    do {
      flags = (glyf->data[off]<<8) + glyf->data[off+1];
      gi = (glyf->data[off+2]<<8) + glyf->data[off+3];
      array_insert(codes, gi);
      off += 4;

      if (flags & 0x01) off += 4; else off += 2;
      if (flags & 0x08) off += 2;
      else if (flags & 0x40) off += 4;
      else if (flags & 0x80) off += 8;
    } while(flags & 0x20);
  }
}


TT_UShort array_lookup(GArray *array, TT_UShort value)
{
   for (TT_UShort i=0; i<array->len; i++) {
    TT_UShort index = g_array_index(array, TT_UShort, i);
    if (index == value) return i;
   }
   return 0;
}



/*
  Performs font subsetting in 'fd'. The indexes contained in the
  array 'codes' are scanned, and only the needed glyphs are
  stored in the tables.
  It changes the tables pointed to by 'table_list' with the modified data
 */
static void change_tables(FontData *fd, GArray *codes)
{
  TT_Header *header = fd->ttdata.properties.header;
  TT_Short fmt = header->Index_To_Loc_Format;
  TT_ULong offset;
  TT_ULong glyph_table_size;
  TT_UShort nhmetrics, new_nhmetrics;
  TT_Table *loca, *newloca;
  TT_Table *glyf, *newglyf;
  TT_Table *hhea, *hmtx, *newhmtx;
  TT_Post   post;

  /* The order in which to traverse the tables is critical */
  static const char *table_name[] = {"hhea", "hmtx", "loca", "cvt ", "fpgm", "head", "prep", "glyf", "maxp"};  /* Order to traverse the table */

  for (guint table=0; table<sizeof(table_name)/sizeof(table_name[0]); table++) {
    GSList *elem;
    TT_Table *tb;
    TT_UShort i;
    TT_ULong tag = MAKE_TT_TAG(table_name[table][0],table_name[table][1],table_name[table][2],table_name[table][3]);
    for (elem = table_list; elem; elem = elem->next) {
      tb = (TT_Table*)elem->data;
      if (tb->tag == tag) break;
    }

    switch(tb->tag) {
    case 0x63767420: // cvt_
      // Nth to change
      break;
    case 0x6670676D: // fpgm
      // Nth to change
      break;
    case 0x676C7966: // glyf
      glyf = tb;
      newglyf = make_new_table(glyf, glyph_table_size);
      /* Copy the glyph data from the old to the new table, glyph by glyph */
      offset = 0;  // Offset within the glyf table for a glyph
      for (i=0; i<codes->len; i++) {
	TT_UShort index = g_array_index(codes, TT_UShort, i);
	TT_ULong o1 = get_glyph_offset(loca, index, fmt);
	TT_ULong o2 = get_glyph_offset(loca, index+1, fmt);
	memcpy(newglyf->data+offset, tb->data+o1, o2-o1);
	set_glyph_offset(newloca, i, fmt, offset);

	TT_Short nc = (newglyf->data[offset]<<8) + newglyf->data[offset+1];
	TT_ULong off = offset+10; // Offset within glyph data, after BBox
	offset += (o2-o1);
	if (nc >= 0) continue; // Composite glyph: number of contours < 0

	/* 
	   If glyph is composite, change the index of the referred glyphs
	   in the new table to reflect the new index (position in the font)
	*/
	TT_UShort flags, gi;
	do {
	  flags = (newglyf->data[off]<<8) + newglyf->data[off+1];
	  gi = (newglyf->data[off+2]<<8) + newglyf->data[off+3];
	  TT_UShort ngi = array_lookup(codes, gi);  // New glyph index
	  newglyf->data[off+2] = (ngi>>8)&0xFF;
	  newglyf->data[off+3] = ngi&0xFF;

	  /* Skip the rest of the data for this referred glyph.
	     Refer to the TT spec, definition of the glyf table */
	  off += 4;
	  if (flags & 0x01) off += 4; else off += 2;
	  if (flags & 0x08) off += 2;
	  else if (flags & 0x40) off += 4;
	  else if (flags & 0x80) off += 8;
	} while(flags & 0x20);
      }
      set_glyph_offset(newloca, codes->len, fmt, offset);

      g_free(loca->data);
      g_free(loca);
      
      g_free(glyf->data);
      g_free(glyf);
      elem->data = newglyf;
      break;
    case 0x68656164: // head
      // Nth to change
      break;
    case 0x68686561: // hhea
      hhea = tb;
      nhmetrics = (hhea->data[34]<<8) + hhea->data[35];
      break;
    case 0x686D7478: // hmtx
      hmtx = tb;
      /* Add glyph with index nhmetrics-1 if not present
	 Important when numberOfHMetrics < number of glyphs in font */
      if (nhmetrics < codes->len) array_insert(codes, nhmetrics-1);

      /* Add glyph indexes referenced by compound glyphs */
      add_compound_glyphs(fd, codes);

      /* Sort the new indexes in ascending order */
      qsort(codes->data, codes->len, sizeof(TT_UShort), compare_ushorts);
     
      /* Calculate new numberOfHMetrics */
      new_nhmetrics = 0;
      for (i=0; i<codes->len; i++) {
	TT_UShort index = g_array_index(codes, TT_UShort, i);
	if (index < nhmetrics) new_nhmetrics++;
      }
      newhmtx = make_new_table(hmtx, 
			       new_nhmetrics*4+(codes->len-new_nhmetrics)*2);
      /* Copy the structs longHorMetrics */
      offset = 0;
      for (i=0; i<new_nhmetrics; i++) {
	TT_UShort index = g_array_index(codes, TT_UShort, i);
	memcpy(newhmtx->data+offset, hmtx->data+4*index, 4);
	offset += 4;
      }
      /* And the rest of the table is an array of leftSideBearings */
      for (i=new_nhmetrics; i<codes->len; i++) {
	TT_UShort index = g_array_index(codes, TT_UShort, i);
	guint pos = nhmetrics*4+(index-nhmetrics)*2;
	newhmtx->data[offset++]  = hmtx->data[pos++];
	newhmtx->data[offset++]  = hmtx->data[pos];
      }
      /* New value in hhea table */
      hhea->data[34] = (new_nhmetrics>>8)&0xFF;
      hhea->data[35] = new_nhmetrics&0xFF;

      g_free(hmtx->data);
      g_free(hmtx);
      elem->data = newhmtx;
      break;
    case 0x6C6F6361: // loca
      loca = tb;
      /* Make new table, maintaining the original Index_To_Loc_Format */
      newloca = make_new_table(loca, (codes->len+1) *
			       ((fmt==1)?sizeof(TT_ULong):sizeof(TT_UShort)));

      /* Calculate the size of the new glyf table by adding the size
	 of all glyphs in it */
      glyph_table_size = 0;
      for (i=0; i<codes->len; i++) {
	TT_UShort index = g_array_index(codes, TT_UShort, i);
	TT_ULong o1 = get_glyph_offset(loca, index, fmt);
	TT_ULong o2 = get_glyph_offset(loca, index+1, fmt);
	glyph_table_size += (o2-o1);
      }
      elem->data = newloca;
      break;
    case 0x6D617870: // maxp
      /* Store new number of glyphs */
      tb->data[4] = (codes->len>>8)&0xFF;
      tb->data[5] = codes->len&0xFF;
      break;
    case 0x70726570: // prep
      // Nth to change
      break;
    }
  }
}



static void make_offset_table(FontData *fd, char *ot, TT_UShort numTables)
{
  TT_UShort searchRange=1, entrySelector=0, rangeShift=0;
  
  while(searchRange <= numTables) {
    searchRange *= 2;
    entrySelector++;
  } 
  searchRange = 16*(searchRange/2);
  entrySelector--;
  rangeShift = 16*numTables - searchRange;

  sprintf(ot, "00010000%02hX%02hX%02hX%02hX%02hX%02hX%02hX%02hX", numTables>>8, numTables&0xff, searchRange>>8, searchRange&0xff, entrySelector>>8, entrySelector&0xff, rangeShift>>8, rangeShift&0xff);
}


bool tt_downloadfont(FILE *ofp, FontData* fd, GArray *codes)
{
  const char *fname = getfontname(fd, TT_NAME_ID_PS_NAME);
  fprintf(ofp, "%%%%BeginResource: font %s\n", fname);
  bool rc = tt_download_as_t42(ofp, fd, codes);
  fputs("%%EndResource\n", ofp);
  return rc;
}


/*
  Download the Type 42 font onto the given file (for printing)
  'codes' is an array describing which glyphs to embed (by their index); if
  NULL, all are embedded
  Return success as boolean
  Refer to 'The Type 42 font format specification', Adobe technical note #5012,
  if you want to understand the code
  The code in 'ttftot42' by Dieter Baron (dillo@giga.or.at) was used as example
*/
bool tt_download_as_t42(FILE *ofp, FontData* fd, GArray *codes)
{
  int error;
  TT_Post   post;
  TT_Header *header = fd->ttdata.properties.header;
  TT_Postscript *ps = fd->ttdata.properties.postscript;
  TT_OS2       *os2 = fd->ttdata.properties.os2;
  
  if (fd->ttdata.PSloaded == FALSE) {
    error = TT_Load_PS_Names(fd->ttdata.face, &post);
    if (error) {
      add_error(fd, _("Cannot load PostScript names: %s"), 
	       TT_ErrToString18(error));
      return FALSE;
    }
    fd->ttdata.PSloaded = TRUE;
  }
  // Compute the tables to write in the T42 font
  bool rc = get_table_list(fd);  // Sets table_list
  if (rc == FALSE) {
    free_table_list();
    return FALSE;
  }
  // Do font subsetting if list of codes (glyph indexes) is present
  if (codes) change_tables(fd, codes);
  complete_tables(fd);
  guint num_glyphs = codes?codes->len:fd->num_glyphs;

  // Start font
  fprintf(ofp, "%%!PS-TrueTypeFont-%ld.%ld-%ld.%ld\n", 
	  header->Table_Version >> 16, 
	  ((header->Table_Version & 0xFFFF)*1000)/0x10000, 
	  header->Font_Revision >> 16, 
	  ((header->Font_Revision & 0xFFFF)*1000)/0x10000);
  fprintf(ofp, "%%%%Creator: GFontView V%s (from %s)\n", 
	  VERSION, g_basename(fd->fontFile)); 
  if (ps->minMemType42!=0 && ps->maxMemType42!=0) 
    fprintf(ofp, "%%%%VMusage: %lu %lu\n", ps->minMemType42, ps->maxMemType42);
  
  // Dictionary
  // Entries in all types of font dictionaries
  fprintf(ofp, "11 dict begin\n");
  fprintf(ofp, "  /FontName (%s) def\n", getfontname(fd, TT_NAME_ID_PS_NAME));
  fprintf(ofp, "  /FontType 42 def\n");
  fprintf(ofp, "  /FontMatrix [1 0 0 1 0 0] def\n");
  fprintf(ofp, "  /FontInfo 9 dict dup begin\n");
  fprintf(ofp, "    /version (%03ld.%03ld) readonly def \n", 
	  header->Font_Revision >> 16, 
	  ((header->Font_Revision & 0xFFFF)*1000)/0x10000);
  fprintf(ofp, "    /FullName (%s) readonly def\n", 
	  getfontname(fd, TT_NAME_ID_FULL_NAME));
  fprintf(ofp, "    /FamilyName (%s) readonly def\n", 
	  getfontname(fd, TT_NAME_ID_FONT_FAMILY));
  fprintf(ofp, "    /Notice (%s) readonly def\n", 
	  getfontname(fd, TT_NAME_ID_COPYRIGHT));
  guint weight = os2->usWeightClass/100-1;
  if (os2->usWeightClass<100 || os2->usWeightClass>900) {
    add_error(fd, _("Warning: Invalid weight value in OS/2 table: %hu"), 
	      os2->usWeightClass);
    weight = 3; // Regular
  }
  fprintf(ofp, "    /Weight (%s) readonly def\n", weight_table[weight]);

  /* Inquiry actual locale for decimal dot, modify it to use an english
   decimal dot, do it and set back to original locale */
  char *loc = setlocale(LC_NUMERIC, NULL);
  setlocale(LC_NUMERIC, "C");
  fprintf(ofp, "    /ItalicAngle %.1f def\n",
	  (ps->italicAngle>>16) + (ps->italicAngle&0xFFFF)/65536.0);
  setlocale(LC_NUMERIC, loc);

  fprintf(ofp, "    /isFixedPitch %s def\n", ps->isFixedPitch?"true":"false");
  fprintf(ofp, "    /UnderlinePosition %hd def\n", ps->underlinePosition);
  fprintf(ofp, "    /UnderlineThickness %hd def\n", ps->underlineThickness);
  fprintf(ofp, "  end readonly def\n");

  // Additional entries in all base fonts (FontType not 0)
  fprintf(ofp, "  /Encoding ISOLatin1Encoding def\n");  // Enough for our purpose
  fprintf(ofp, "  /FontBBox [%d %d %d %d] def\n", 
	  scale(fd, header->xMin), scale(fd, header->yMin), 
	  scale(fd, header->xMax), scale(fd, header->yMax));

  // Additional and modified entries in Type 42 fonts
  fprintf(ofp, "  /PaintType 0 def\n");  // Filled glyphs (2 for stroked)
  fprintf(ofp, "  /CharStrings %d dict dup begin\n", num_glyphs);

  for (TT_UShort i=0; i<num_glyphs; i++) {
    char *psname;
    TT_UShort index = codes?g_array_index(codes, TT_UShort, i):i;
    error = TT_Get_PS_Name(fd->ttdata.face, index, &psname);
    if (error) {
      add_error(fd, _("Cannot get PostScript name for glyph %hu: %s"), 
		index, TT_ErrToString18(error));
      continue;
    }
    fprintf(ofp, "    /%s %d def\n", psname, i);
  }
  fprintf(ofp, "  end readonly def\n");

  /* The TrueType tables follow
   They consist of a series of PS strings (enclosed in <>).
   Each string must begin at TrueType table boundaries or at individual 
   glyph boundaries within the glyf table.
   Each string must have one additional padding byte appended: '00'
   The strings should consist of lines of constant n characters, separated by m
   characters of white space and/or control characters.
   Each line should not be greater than 255 characters long.
   Each string can only be 65534 bytes long (plus 1 byte padding)

   The possibly used TT tables are head, hhea, hmtx, loca, prep, fpgm, glyf,
   cvt_, maxp; from PS version 3011 on, also vhea, vmtx.
   Of these, cvt_, fpgm and prep might not be present in the ttf font
  */

  // Find the 'head' table (to store the font checksum)
  TT_Table *head = NULL;
  for (GSList *elem = table_list; elem; elem = elem->next) {
    head = (TT_Table*)elem->data;
    if (strcmp(head->name, "head")==0) break;
  }
  
  // Find the 'loca' table (so that we can split the glyf table)
  TT_Table *loca = NULL;
  for (GSList *elem = table_list; elem; elem = elem->next) {
    loca = (TT_Table*)elem->data;
    if (strcmp(loca->name, "loca")==0) break;
  }


  char ot[32];  // Offset table (Why is it called like that?)
  fprintf(ofp, "%% Font download\n");
  make_offset_table(fd, ot, g_slist_length(table_list));
  fprintf(ofp, "/sfnts [ <\n%s\n", ot);  // font header
  TT_ULong checksum = MAKE_TT_TAG(ot[0], ot[1], ot[2], ot[3]) + 
    MAKE_TT_TAG(ot[4], ot[5], ot[6], ot[7]) + 
    MAKE_TT_TAG(ot[8], ot[9], ot[10], ot[11]);

  // Write table directory
  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    checksum += tb->tag + 2*tb->checksum + tb->offset + tb->length;
    fprintf(ofp, "%8.8lX%8.8lX%8.8lX%8.8lX\n", tb->tag, tb->checksum, 
	    tb->offset, tb->length);
  }
  fprintf(ofp, ">");

  // Write the computed checksum in the 'head' table
  checksum = 0xb1b0afba - checksum;
  head->data[8]  = (checksum>>24)&0xff;
  head->data[9]  = (checksum>>16)&0xff;
  head->data[10] = (checksum>>8)&0xff;
  head->data[11] = checksum&0xff;

  // Go table by table and download it
  for (GSList *elem = table_list; elem; elem = elem->next) {
    TT_Table *tb = (TT_Table*)elem->data;
    fprintf(ofp, "\n%% %s table\n", tb->name);
    if (strcmp(tb->name, "glyf")!=0) {
      // All but the glyf table (already checked length of table)
      write_string(ofp, TRUE, tb, 0, tb->length);
      continue;
    }
    // The glyf table (has to be splitted probably)
    for (TT_UShort ch=0; ch<=num_glyphs; ch++) {
      TT_ULong offset_now, offset_next, offset_lastwritten;
      if (ch==0) offset_now = get_glyph_offset(loca, ch, 
					       header->Index_To_Loc_Format);
      else offset_now = offset_next;
      if (ch == num_glyphs) {
	// Reached last character, so flush all not written glyphs
	write_string(ofp, TRUE, tb, offset_lastwritten, 
		     offset_now - offset_lastwritten);
	break;
      }
      offset_next = get_glyph_offset(loca, ch + 1, 
				     header->Index_To_Loc_Format);
      TT_ULong chlength = offset_next - offset_now;
      if (ch==0) offset_lastwritten = offset_now;
      if (offset_now - offset_lastwritten + chlength > MAXSTRING) {
	write_string(ofp, FALSE, tb, offset_lastwritten, 
		     offset_now - offset_lastwritten);
	offset_lastwritten = offset_now;
      }
    }  
  }
  fprintf(ofp, " ] def\n");
  
  // Buf, done
  fprintf(ofp, "FontName currentdict end definefont pop\n");
  free_table_list();
  return TRUE;
}




// *******************  Functions to show properties of the font

static void make_entries(GtkWidget *frame, struct TT_Info *props, int num)
{
  static GtkStyle *hlink = NULL;  // Hyperlink fg color, gray bg
  if (!hlink) {
    GdkColor color = {0, 0x5000, 0x2000, 0xA000};
    hlink = gtk_style_copy(frame->style);
    hlink->fg[GTK_STATE_NORMAL] = color;
    hlink->base[GTK_STATE_NORMAL] = hlink->bg[GTK_STATE_NORMAL];
  }

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_container_border_width(GTK_CONTAINER(vbox), 5);

  GtkWidget *table = gtk_table_new(num, 2, FALSE); // num rows, 2 columns
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

  for (int i=0; i<num; i++) {
    GtkWidget *label = props[i].property;
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.0);
    gtk_widget_set_style(label, stylebold);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, i, i+1, 
		     (GtkAttachOptions)(GTK_FILL),
		     (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), 0, 0);
    label = props[i].value;
    if (GTK_IS_MISC(label))
      gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
    if (GTK_IS_LABEL(label))
      gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    if (GTK_IS_ENTRY(label)) {
      gtk_widget_set_style(label, hlink);
      gtk_entry_set_editable(GTK_ENTRY(label), FALSE);
      gtk_entry_set_position(GTK_ENTRY(label), 0);
    }
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, i, i+1, 
		     (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), 
		     (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), 0, 0);
  }
}



static GtkWidget* opentype_info(FontData *fd)
{
  GtkWidget *scrolled_win, *desc_l, *desc_f;
  struct TT_Info props[20];
  char buf[50], *str;
  int nprops = 0;

  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 3);

  props[nprops].property = gtk_label_new(_("Manufacturer:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_MANUFACTURER));

  props[nprops].property = gtk_label_new(_("Designer:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_DESIGNER));

  props[nprops].property = gtk_label_new(_("License:"));
  scrolled_win = gtk_scrolled_window_new(NULL, NULL); 
  gtk_widget_set_usize(scrolled_win, -1, 100);  
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win), 
				 GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  desc_l = gtk_label_new(getfontname(fd, TT_NAME_ID_LICENSE));
  gtk_label_set_line_wrap(GTK_LABEL(desc_l), TRUE);
  desc_f = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(desc_f), GTK_SHADOW_NONE); 
  gtk_container_add(GTK_CONTAINER(desc_f), desc_l);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_win), desc_f);
  props[nprops++].value = scrolled_win;

  props[nprops].property = gtk_label_new(_("Description:"));
  scrolled_win = gtk_scrolled_window_new(NULL, NULL); 
  gtk_widget_set_usize(scrolled_win, -1, 100);  
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win), 
				 GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  desc_l = gtk_label_new(getfontname(fd, TT_NAME_ID_DESCRIPTION));
  gtk_label_set_line_wrap(GTK_LABEL(desc_l), TRUE);
  desc_f = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(desc_f), GTK_SHADOW_NONE); 
  gtk_container_add(GTK_CONTAINER(desc_f), desc_l);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_win), desc_f);
  props[nprops++].value = scrolled_win;

  props[nprops].property = gtk_label_new(_("Vendor URL:"));
  props[nprops].value = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(props[nprops++].value), 
		     getfontname(fd, TT_NAME_ID_VENDOR_URL));

  props[nprops].property = gtk_label_new(_("Designer URL:"));
  props[nprops].value = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(props[nprops++].value), 
		     getfontname(fd, TT_NAME_ID_DESIGNER_URL));

  props[nprops].property = gtk_label_new(_("License URL:"));
  props[nprops].value = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(props[nprops++].value), 
		     getfontname(fd, TT_NAME_ID_LICENSE_URL));


  make_entries(frame, props, nprops);
  gtk_widget_show_all(frame);
  return frame;
}



static char *apple_enc[] = {"Default", "Unicode 1.1", "ISO 10646", "Unicode 2.0"};
static char *mac_enc[] = {"Roman", "Japanese", "Traditional chinese", "Korean", "Arabic", "Hebrew", "Greek", "Russian", "RSymbol", "Devanagari", "Gurmukhi", "Gujarati", "Oriya", "Bengali", "Tamil", "Telugu", "Kannada", "Malayalam", "Sinhalese", "Burmese", "Khmer", "Thai", "Laotian", "Georgian", "Armenian", "Maldivian", "Simplified chinese", "Tibetan", "Mongolian", "Geez", "Slavic", "Vietnamese", "Sindhi", "Uninterp"};
static char *iso_enc[] = {"7-bit ASCII", "10646", "8859-1"};
static char *ms_enc[] = {"Symbol", "Unicode", "Shift JIS", "Big 5", "PRC", "Wansung", "Johab"};

struct cmap {      /* Used to store the character maps of a platform */
  char *p_name;    /* Platform name */
  char **enc_name; /* Array of encoding names */
  int n_encs;      /* Number of elements in above array */
};
/* Array with the list of known character maps */
static struct cmap cmap_list[] = {
  {"Apple Unicode", apple_enc, sizeof(apple_enc)/sizeof(char*)},
  {"Macintosh", mac_enc, sizeof(mac_enc)/sizeof(char*)},
  {"ISO", iso_enc, sizeof(iso_enc)/sizeof(char*)},
  {"Microsoft", ms_enc, sizeof(ms_enc)/sizeof(char*)}
};

static GtkWidget* charmap_info(FontData *fd)
{
  char* titles[] = {N_("Platform"), N_("Encoding")};

  /* Translate titles */
  int n_titles = sizeof(titles)/sizeof(char*);
  for (int i=0; i<n_titles; i++)  
    titles[i] = _(titles[i]);

  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 3);
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

  /* Make window with clist */
  GtkWidget* scwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwindow), 
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  GtkWidget *clist = gtk_clist_new_with_titles(n_titles, titles);
  gtk_clist_column_titles_passive(GTK_CLIST(clist));
  gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE); 
  gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE); 

  gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_SINGLE);
  gtk_clist_set_shadow_type(GTK_CLIST(clist), GTK_SHADOW_ETCHED_OUT);
  gtk_container_border_width(GTK_CONTAINER(clist), 0);

  /* Make base (background) color not white, but the same as the rest */
  clist->style->base[GTK_STATE_NORMAL] = clist->style->bg[GTK_STATE_NORMAL];

  for (TT_UShort cm=0; cm<fd->ttdata.properties.num_CharMaps; cm++) {
    TT_UShort platformID, encodingID;
    TT_Face face = fd->ttdata.face;
    char rowentries[2][50];
    char *entries[2];

    for (int i=0; i<2; i++) entries[i] = rowentries[i];
    int error = TT_Get_CharMap_ID(face, cm, &platformID, &encodingID);
    if (error) continue;

    if (platformID < sizeof(cmap_list)/sizeof(struct cmap)) {
      sprintf(rowentries[0], "%s", cmap_list[platformID].p_name);
      if (encodingID < cmap_list[platformID].n_encs)
	sprintf(rowentries[1], "%s", 
		cmap_list[platformID].enc_name[encodingID]);
      else
	sprintf(rowentries[1], _("Unknown (%hu)"), encodingID);
    }
    else {
      sprintf(rowentries[0], _("Unknown (%hu)"), platformID);
      rowentries[1][0] = '\0';
    }
    gtk_clist_append(GTK_CLIST(clist), entries);
    gtk_clist_set_background(GTK_CLIST(clist), GTK_CLIST(clist)->rows - 1, 
			     &clist->style->bg[GTK_STATE_NORMAL]);
  }

  gtk_container_add(GTK_CONTAINER(scwindow), clist);
  gtk_box_pack_start(GTK_BOX(vbox), scwindow, TRUE, TRUE, 0); 

  gtk_widget_show_all(frame);
  return frame;
}



static GtkWidget* metrics_info(FontData *fd)
{
  struct TT_Info props[20];
  char buf[50], *str;
  int nprops = 0;

  TT_Header *header = fd->ttdata.properties.header;
  TT_OS2 *os2 = fd->ttdata.properties.os2;
  TT_Postscript *ps = fd->ttdata.properties.postscript;

  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 3);

  props[nprops].property = gtk_label_new(_("Units per EM:"));
  sprintf(buf, "%hu", header->Units_Per_EM);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new(_("Average width:"));
  sprintf(buf, "%hd", os2->xAvgCharWidth);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new(_("Glyphs:"));
  sprintf(buf, "%u", fd->num_glyphs);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new(_("Kern pairs:"));
  sprintf(buf, "%u", fd->ttdata.kernpairs);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new(_("Weight:"));
  if (os2->usWeightClass>=100 && os2->usWeightClass<=900 && 
      os2->usWeightClass==(os2->usWeightClass/100)*100)
    sprintf(buf, "%s", weight_table[os2->usWeightClass/100-1]);
  else
    sprintf(buf, _("Invalid (%hu)"), os2->usWeightClass);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new(_("Width:"));
  if (os2->usWidthClass>=1 && os2->usWidthClass<=9)
    sprintf(buf, "%s", width_table[os2->usWidthClass-1]);
  else
    sprintf(buf, _("Invalid (%hu)"), os2->usWidthClass);
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new(_("Italic angle:"));
  sprintf(buf, "%.1f", (ps->italicAngle>>16) + (ps->italicAngle&0xFFFF)/65536.0);
  props[nprops++].value = gtk_label_new(buf);
  
  props[nprops].property = gtk_label_new(_("Monospaced:"));
  sprintf(buf, "%s", ps->isFixedPitch?_("yes"):_("no"));
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new(_("Character maps:"));
  sprintf(buf, "%d", fd->ttdata.properties.num_CharMaps);
  props[nprops++].value = gtk_label_new(buf);


  make_entries(frame, props, nprops);
  gtk_widget_show_all(frame);
  return frame;
}


static GtkWidget* general_info(FontData *fd)
{
  struct TT_Info props[20];
  char buf[128], *str;
  int nprops = 0;
  static time_t sec1904 = 0; /* seconds since the epoch (1970) to 1904 */
  static GString *size = NULL;
  struct stat statbuf;

  if (size == NULL) size = g_string_new(NULL);
  if (sec1904==0) {
    struct tm t1904; /* 1/1/1904 0:0:0 */
    t1904.tm_year = 4;
    t1904.tm_mon = 0;
    t1904.tm_mday = 1;
    t1904.tm_hour = 0;
    t1904.tm_min = 0;
    t1904.tm_sec = 0;
    sec1904 = mktime(&t1904);
  }

  TT_Header *header = fd->ttdata.properties.header;
  TT_OS2 *os2 = fd->ttdata.properties.os2;

  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 3);

  props[nprops].property = gtk_label_new(_("File name:"));
  props[nprops++].value = gtk_label_new(g_basename(fd->fontFile));

  props[nprops].property = gtk_label_new(_("File size:"));
  if (!stat(fd->fontFile, &statbuf))
    g_string_sprintf(size, "%ldK", statbuf.st_size/1024);
  else
    g_string_truncate(size, 0);
  props[nprops++].value = gtk_label_new(size->str);

  props[nprops].property = gtk_label_new(_("Font family:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_FONT_FAMILY));

  props[nprops].property = gtk_label_new(_("Font subfamily:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_FONT_SUBFAMILY));

  props[nprops].property = gtk_label_new(_("Font full name:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_FULL_NAME));

  props[nprops].property = gtk_label_new(_("Font PS-name:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_PS_NAME));

  props[nprops].property = gtk_label_new(_("Font version:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_VERSION_STRING));

#ifdef G_HAVE_GINT64
  size_t res;
  gint64 created_mac = (((gint64)header->Created[0])<<32) + (gint64)(header->Created[1]);
  time_t created_t = created_mac + sec1904;
  struct tm *created = localtime(&created_t);
  res = strftime(buf, sizeof(buf), "%c", created);
  props[nprops].property = gtk_label_new(_("Created:"));
  props[nprops++].value = gtk_label_new(res?buf:"");

  gint64 modified_mac = (((gint64)header->Modified[0])<<32) + (gint64)(header->Modified[1]);
  time_t modified_t = modified_mac + sec1904;
  struct tm *modified = localtime(&modified_t);
  res = strftime(buf, sizeof(buf), "%c", modified);
  props[nprops].property = gtk_label_new(_("Modified:"));
  props[nprops++].value = gtk_label_new(res?buf:"");
#endif

  props[nprops].property = gtk_label_new(_("Copyright:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_COPYRIGHT));

  props[nprops].property = gtk_label_new(_("Trademark:"));
  props[nprops++].value = gtk_label_new(getfontname(fd, TT_NAME_ID_TRADEMARK));

  props[nprops].property = gtk_label_new(_("Embedding\nlicense:"));
  if (os2->fsType==0x0000) 
    str = _("No restrictions");
  else if (os2->fsType&0x0008)
    str = _("Editable embedding allowed");
  else if (os2->fsType&0x0004)
    str = _("Read-only embedding allowed");
  else if (os2->fsType&0x0002)
    str = _("No embedding allowed");
  else
    str = _("Unknown");
  props[nprops++].value = gtk_label_new(str);


  make_entries(frame, props, nprops);
  gtk_widget_show_all(frame);
  return frame;
}




void tt_showproperties(FontData *fd, GtkWidget *window)
{
  GtkWidget *frame, *label;

  GtkWidget *notebook = gtk_notebook_new();
  gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);
  /*
  gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
		     GTK_SIGNAL_FUNC(page_switch), NULL);
  */
  gtk_container_add(GTK_CONTAINER(window), notebook);

  // General info
  frame = general_info(fd);
  label = gtk_label_new(_("General"));
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  // More specific
  frame = metrics_info(fd);
  label = gtk_label_new(_("Metrics"));
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  // Character Maps info
  frame = charmap_info(fd);
  label = gtk_label_new(_("Character maps"));
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  // OpenType fields
  frame = opentype_info(fd);
  label = gtk_label_new(_("OpenType"));
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  gtk_widget_show(notebook);
}




