/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#ifdef GTK_GL
#include <gtkgl/gtkglarea.h>
#endif

#include "gdis.h"

#include "logo_left.xpm"
#include "logo_right.xpm"
#include "folder.xpm"
#include "disk.xpm"
#include "split_none.xpm"
#include "split_vert.xpm"
#include "split_horz.xpm"
#include "split_both.xpm"
#include "axes.xpm"
#include "rotate.xpm"
#include "cross.xpm"
#include "bulb.xpm"
#include "element.xpm"
#include "geom.xpm"
#include "ball.xpm"
#ifdef GTK_GL
#include "yellow_ball.xpm"
#endif
#include "methane.xpm"
#include "box.xpm"
#include "diamond2.xpm"
#include "left_arrow1.xpm"
#include "right_arrow1.xpm"

/* top level data structure */
extern struct sysenv_pak sysenv;

/* bit messy - put some of these in sysenv? */
/* main window */
GtkWidget *window;
/* main drawing area */
GtkWidget *drawing_area;  /* TODO - change to gdk_area/canvas */
/* backing pixmap */
GdkPixmap *pixmap = NULL;
/* model tree (put in sysenv?) */
GtkWidget *tree;
/* model pane stuff */
GtkWidget *scrolled_window, *clist;
/* gdisrc? */
gint pane_width=65;
/* view panel data */
GtkWidget *xangle, *yangle, *zangle;
/* text tray - TODO - make vertically scrollable? */
static GtkWidget *tbox;

#ifdef GTK_GL
int attrlist[] = {GDK_GL_RGBA, GDK_GL_DOUBLEBUFFER, 
                  GDK_GL_DEPTH_SIZE, 1, GDK_GL_NONE};

/********************************************/
/* draw with gdk/render using opengl toggle */
/********************************************/
void gl_toggle()
{
static gint flag=1;

flag ^= 1;

if (flag)
  {
  gtk_widget_hide(sysenv.glarea);
  gtk_widget_show(drawing_area);
  }
else
  {
  gtk_widget_hide(drawing_area);
  gtk_widget_show(sysenv.glarea);
  }
}
#endif

/***************************************************/
/* decide which drawing routine needs to be called */
/***************************************************/
void redraw_canvas(gint action)
{
#if GTK_GL
gint gdk, gl;

gdk = GTK_WIDGET_VISIBLE(drawing_area);
gl = GTK_WIDGET_VISIBLE(sysenv.glarea);

if (gl & !gdk)
  opengl_draw(sysenv.glarea);
if (gdk & !gl)
#endif
  gdk_draw(action);
}

/*********************************/
/* close type associated dialogs */
/*********************************/
#define DEBUG_DIALOGS 0
void close_dialog_type(gint type)
{
gint i;

for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (sysenv.dialog[i].type == type && sysenv.dialog[i].active)
    {
    sysenv.dialog[i].active = FALSE;
    gtk_widget_destroy(sysenv.dialog[i].win);
    }
  }
}

/***************************/
/* close a specific dialog */
/***************************/
void close_dialog(gint id)
{
struct model_pak *data;

/* prevent double closure from gtk_widget_destroy() call */
if (sysenv.dialog[id].active)
  {
#if DEBUG_DIALOGS
printf("destroying dialog: %d, associated model: %d\n",id,sysenv.dialog[id].model);
#endif
  sysenv.dialog[id].active = FALSE;

/* NEW - cleanup to stop widgets being updated that have had their */
/* assoc. dialogs destroyed */
/* FIXME - a better way? */
/* eg calls to retrieve dialog data (if it exists!) for a particular */
/* model  - that way we know if we can/need to update a dialog */
  data= model_ptr(sysenv.dialog[id].model, RECALL);
  switch(sysenv.dialog[id].type)
    {
    case GULP:
      data->gulp.energy_entry = NULL;
      data->gulp.esurf_entry = NULL;
      data->gulp.eatt_entry = NULL;
      break;
    }
  gtk_widget_destroy(sysenv.dialog[id].win);
  }
return;
}
void event_close_dialog(GtkWidget *w, gint id)
{
close_dialog(id);
}

/******************************************/
/* ask for a model/type associated dialog */
/******************************************/
gint request_dialog(gint model, gint type)
{
gint i;

#if DEBUG_DIALOGS
printf("requested dialog type: %d, associated model: %d\n",type,model);
#endif

/* is another of the same type active? */
for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (!sysenv.dialog[i].active)
    continue;
/* type dependent testing */
  switch(type)
    {
    case ANIM:
    case COLOUR:
    case DISPLAY:
    case FILE_LOAD:
    case GENSURF:
    case GEOMETRY:
    case GPERIODIC:
    case MDI:
    case POVRAY:
    case SURF:
    case TASKMAN:
      if (sysenv.dialog[i].type == type)
        {
/*
        show_text("Only one dialog of this type allowed.");
*/
        pick_model(model);
/* NEW - raise an existing dialog to the front */
        gdk_window_show((sysenv.dialog[i].win)->window);
        return(-1);
        }
      break;
    case GULP:
    case MORPH:
    case SGINFO:
    case SYMMETRY:
    case OPENGL:
    case OPENGL_OPTIONS:
      if (sysenv.dialog[i].model == model && sysenv.dialog[i].type == type)
        {
/*
        show_text("Dialog of this type is already open for this model.");
*/
        pick_model(model);
/* NEW - raise an existing dialog to the front */
        gdk_window_show((sysenv.dialog[i].win)->window);
        return(-1);
        }
    break;
    default:
      printf("Unknown dialog type requested.\n");
      return(-1);
    }
  }

/* find the first available dialog */ 
for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (!sysenv.dialog[i].active)
    {
/* init and return dialog id */
    sysenv.dialog[i].active = TRUE;
    sysenv.dialog[i].model = model;
    sysenv.dialog[i].type = type;
    return(i);
    }
  }
show_text("Maximum allowed dialogs reached.");
return(-1);
}

/***********************/
/* window change event */
/***********************/
/* should all these re-draw type events go in draw.c ? */
#define DEBUG_CONFIG_EVENT 0
static gint configure_event (GtkWidget *w, GdkEventConfigure *event)
{
gint i;
struct model_pak *data;

/* Create a new backing pixmap of the appropriate size */
if (pixmap)
  {
  gdk_pixmap_unref(pixmap);
/* pixmap exists -> window resize -> save new size */
  sysenv.write_gdisrc = TRUE;
  }
pixmap = gdk_pixmap_new(w->window,w->allocation.width,w->allocation.height, 
                        sysenv.depth);
if (pixmap == NULL)
  {
  printf("Error: could not create pixmap.\n");
  exit(1);
  }

/* background to black */
gdk_draw_rectangle (pixmap, w->style->black_gc, TRUE,
                    w->allocation.x,
                    w->allocation.y, 
                    w->allocation.width,
                    w->allocation.height);

/* update the drawing area dimension */
sysenv.x = w->allocation.x;
sysenv.y = w->allocation.y;
sysenv.width = w->allocation.width;
sysenv.height = w->allocation.height;

#if DEBUG_CONFIG_EVENT
printf("Relative canvas origin: (%d,%d)\n",sysenv.x,sysenv.y);
printf("     Canvas dimensions:  %dx%d\n",sysenv.width,sysenv.height);
printf("          Canvas depth:  %d bit\n",sysenv.depth);
#endif

/* re-init subwindow dimensions */
mod_screen(sysenv.subx, sysenv.suby);

/* update coords */
for (i=0 ; i<sysenv.num_displayed ; i++)
  {
  data = model_ptr(sysenv.displayed[i], RECALL);
  if (data)
    init_objs(REDO_COORDS, data);
  }

/* redraw screen frames & models */
#ifdef GTK_GL
if (GTK_IS_GL_AREA(w))
  opengl_draw(w);
else
  redraw_canvas(ALL);
#else
redraw_canvas(ALL);
#endif

return TRUE;
}

/*************************/
/* EVENT - expose window */
/*************************/
static gint expose_event(GtkWidget *w, GdkEventExpose *event)
{
/* Redraw the screen from the backing pixmap */
gdk_draw_pixmap(w->window,
                w->style->fg_gc[GTK_WIDGET_STATE (w)],
                pixmap,
                event->area.x, event->area.y,
                event->area.x, event->area.y,
                event->area.width, event->area.height);
return TRUE;
}

/*****************************/
/* locate clicked sub window */
/*****************************/
gint get_subwin(gint x, gint y)
{
gint i;

/* find subwin in which the click happened */
for (i=0 ; i<sysenv.num_displayed ; i++)
  if (abs(sysenv.subcenx[i] - x) < sysenv.subwidth/2)
    if (abs(sysenv.subceny[i] - y) < sysenv.subheight/2)
      return(i);
/* failed */
return(-1);
}

/************************/
/* EVENT - button press */
/************************/
gint button_press_event (GtkWidget *w, GdkEventButton *event)
{
gint model, subwin, refresh=0, i, x, y;
struct model_pak *data;
GdkModifierType state;

x = event->x;
y = event->y;

/* test which subwin was clicked */
subwin = get_subwin(x,y);
if (subwin < 0)
  return(FALSE);

/* nothing there? */
if (sysenv.displayed[subwin] < 0)
  return(FALSE);

/* retrieve/check model */
model = sysenv.active;
if (model < 0)
  return(FALSE);
data = model_ptr(model, RECALL);
g_return_val_if_fail(data != NULL, FALSE);

#ifdef GTK_GL
/* if not-active - make active, then exit */
if (GTK_IS_GL_AREA(w))
  {
  for (i=0 ; i<MAX_DIALOGS ; i++)
    if (sysenv.dialog[i].type == OPENGL)
      if (w == sysenv.dialog[i].data)
        if (sysenv.active != sysenv.dialog[i].model)
          {
          tree_select(sysenv.dialog[i].model);
          return(FALSE);
          }
  }
else
#endif
/* if not-active - make active, then exit */
if (sysenv.active != sysenv.displayed[subwin])
  {
/* select appropriate model in the tree */
  tree_select(sysenv.displayed[subwin]);
  return(FALSE);
  }

/* only want button 1 (for now) */
if (event->button != 1)
  return(FALSE);

/* NEW - allow shift+click to add/remove single atoms in the selection */
/* TODO - depending on mode add/remove objects in selection eg atom/mols */
state = (GdkModifierType) event->state;
if ((state & GDK_SHIFT_MASK) == GDK_SHIFT_MASK)
  {
  switch(data->mode)
    {
    default:
      i = seek_atom(x,y,data);
      if (i > -1)
        {
        if (sysenv.select_mode == ATOM)
          {
/* if atom is already in the selection - remove it */
          if (add_select(data, i) == 2)
            {
            del_select(data, i);
            (data->atoms+i)->status &= ~SELECT_HL;
            }
          else
            (data->atoms+i)->status |= SELECT_HL;
          }
        else
          {
/* molecules */
          pick_mol(x, y, data);
          }
/* print info */
        print_atom(data, i);
        data->atom_info = i;
        refresh++;
        }
      else
        {
/* otherwise start a box selection */
        update_box(x,y,data,START);
        }
      break;
    }
  }
else
  {
/* determine the type of action required */
  switch(data->mode)
    {
    case BOND_INFO:
      info_bond(subwin,x,y,data);
      break;
    case BOND_SINGLE:
      user_bond(subwin,x,y,data,SINGLE);  
      break;
    case BOND_DOUBLE:
      user_bond(subwin,x,y,data,DOUBLE);  
      break;
    case BOND_TRIPLE:
      user_bond(subwin,x,y,data,TRIPLE);  
      break;
    case ATOM_ADD:
      add_atom(x,y);
      refresh++; 
      break;
    case ATOM_MOVE:
      move_atom(x,y,START);
      refresh++;
      break;
    case ATOM_DELETE:
      delete_atom_at(x,y,data);
      calc_bonds(data);
      calc_mols(data);
      refresh++;
      break;
    case ATOM_CHANGE_TYPE:
      change_atom_at(x,y,data);
      calc_bonds(data);
      calc_mols(data);
      refresh++;
      break;
    case MOL_MOVE:
      move_mol(x,y,START);
      refresh++;
      break;
    case MOL_DELETE:
      delete_mol(x,y,data);
      calc_bonds(data);
      calc_mols(data);
      refresh++;
      break;
    case REGION_LITE:
      i = seek_atom(x,y,data);
      if (i < 0)
        break;
      data->region[(data->atoms+i)->region] ^= 1;
      init_objs(REFRESH_LITE, data);
      refresh++;
      break;
    default:
      clear_select();
/* if an atom was under the mouse - select */
      i = seek_atom(x,y,data);
      if (i > -1)
        {
/* select */
        if (sysenv.select_mode == ATOM)
          {
          add_select(data, i);
          (data->atoms+i)->status |= SELECT_HL;
          }
        else
          pick_mol(x,y,data);
/* print info */
        print_atom(data, i);
        data->atom_info = i;
        }
      else
        {
/* otherwise, begin the selection box */
        update_box(x,y,data,START);
        }
      refresh++;
      break; 
    }
  }

/* do we need a redraw? */
if (refresh)
  {
#if GTK_GL
/* NEW - don't redraw OpenGL window as (currently) nothing will have changed */
  if (!GTK_IS_GL_AREA(w))
    redraw_canvas(SINGLE);
#endif
  }

return(FALSE);
}

/***************************/
/* EVENT - button released */
/***************************/
gint button_release_event (GtkWidget *w, GdkEventButton *event)
{
gint x, y, subwin;
struct model_pak *data;

/* get mouse position */
x = event->x;
y = event->y;
/* test which subwin was clicked */
subwin = get_subwin(x,y);
if (subwin < 0)
  return(FALSE);
/* nothing there? */
if (sysenv.displayed[subwin] < 0)
  return(FALSE);
/* get model */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return(FALSE);

/* first mouse button */
if (event->state & GDK_BUTTON1_MASK)
  {
/* clean up after move operations */
  switch(data->mode)
    {
    case DIST_INFO:
      info_dist(x,y,data);
      break;
    case ANGLE_INFO:
      info_angle(x,y,data);
      break;
    case DIHEDRAL_INFO:
      info_torsion(x,y,data);
      break;
    case ATOM_MOVE:
      move_atom(0,0,STOP);
      break;
    case MOL_MOVE:
      move_mol(0,0,STOP);
      break;
    default:
/* commit the box contents to the selection */
/* if shift pressed - append selection, otherwise create new */
      if (data->box_on)
        select_box();
/* turn the box off */
      data->box_on = FALSE;
      break;
    }
  }
redraw_canvas(SINGLE);
return(FALSE);
}

/************************/
/* EVENT - mouse motion */
/************************/
#define DEBUG_MOTION 0
gint motion_notify_event (GtkWidget *w, GdkEventMotion *event)
{
gint i, m, x, y, px, py, dx, dy, fx, fy, ds, da, atom, shell, model, refresh;
gint shift;
gfloat movx, movy, movz, movrx, movry, movrz;
static gint ox=0, oy=0;
struct model_pak *data;
GdkModifierType state;

/* nothing to do if no models are loaded */
if (!sysenv.num_models)
  return(FALSE);

/* get mouse data */
if (event->is_hint)
  gdk_window_get_pointer (event->window, &x, &y, &state);
else
  {
  x = event->x;
  y = event->y;
  state = (GdkModifierType) event->state;
  }
if ((state & GDK_SHIFT_MASK) == GDK_SHIFT_MASK)
  shift = TRUE;
else
  shift = FALSE;

/* perform action for active model */
/* FIXME - recall only if button pressed (speed) */
model = sysenv.active;
data = model_ptr(model, RECALL);
if (!data)
  return(FALSE);

/* single update */
refresh=0;

/* first mouse button */
if (state & GDK_BUTTON1_MASK)
  {
/* NEW - if shift pressed, override with a scale change */
    switch(data->mode)
      {
/* no rotation allowed if one of these is selected */
      case ATOM_MOVE:
        move_atom(x, y, UPDATE);
        refresh++;
        break;
      case MOL_MOVE:
        move_mol(x, y, UPDATE);
        refresh++;
        break;
      case ATOM_ADD:
        break;
      case FREE:
        update_box(x, y, data, UPDATE);
        refresh++;
        break;
      default:
/*
        clear_select();
*/
        break;
      }
  }

/* second mouse button */
if (state & GDK_BUTTON2_MASK)
  {
/* shift - translate */
  if (shift)
    {
/* selection only translation */
    if (data->mode == ALTER_SELECT)
      {
      for (i=0 ; i<data->select_size ; i++)
        {
        atom = *(data->select_list+i);
/* pixel position where we want the atom to be */
        px = (data->atoms+atom)->px + x - ox;
        py = (data->atoms+atom)->py + y - oy;
/* map real coords (save, so can use difference to update the shell) */
        movx = -(data->atoms+atom)->x;
        movy = -(data->atoms+atom)->y;
        movz = -(data->atoms+atom)->z;
        movrx = -(data->atoms+atom)->rx;
        movry = -(data->atoms+atom)->ry;
        movrz = -(data->atoms+atom)->rz;
        pixel_coord_map(px,py,data,atom);
        movx += (data->atoms+atom)->x;
        movy += (data->atoms+atom)->y;
        movz += (data->atoms+atom)->z;
        movrx += (data->atoms+atom)->rx;
        movry += (data->atoms+atom)->ry;
        movrz += (data->atoms+atom)->rz;
/* TODO - how do we do the shell? (if it has one) */
/* use a before/after difference from the atom calc on the shell? */
        if ((data->atoms+atom)->has_shell)
          {
          shell = (data->atoms+atom)->idx_shell;
          (data->shells+shell)->px += x - ox;
          (data->shells+shell)->py += y - oy;
          (data->shells+shell)->x += movx;
          (data->shells+shell)->y += movy;
          (data->shells+shell)->z += movz;
          (data->shells+shell)->rx += movrx;
          (data->shells+shell)->ry += movry;
          (data->shells+shell)->rz += movrz;
          }

/* TODO */
        if (data->periodic)
          {
/* update cartesian coords? */
          }
        }
/* recalc connectivity */
      calc_bonds(data);
      calc_mols(data);
      }
    else
      {
/* zoom */
/* convert relative mouse motion to a scale increment */
      dx = x-ox;
      dy = oy-y;        /* inverted y */
      data->scale += PIX2SCALE * (gfloat) (dx+dy);
/* lower limit check (upper? or leave to user discretion?) */
      if (data->scale < 0.1)
        data->scale = 0.1;
/* update */
      init_objs(REDO_COORDS, data);
      }
    refresh++;
    }
  else
    {
/* translate everything */
    data->offset[0] += x-ox;
    data->offset[1] += y-oy;
    for (i=data->num_ghosts ; i-- ; )
      {
      data = model_ptr(*(data->ghosts+i), RECALL);
      data->offset[0] += x-ox;
      data->offset[1] += y-oy;
      }
    refresh++;
    }
  }

/* third mouse button clicked? */
if (state & GDK_BUTTON3_MASK)
  {
/* shift clicked? */
  if (shift)
    {
/* yaw */
    dx = x-ox;
    dy = oy-y;
    if (dx || dy)
      {
      m = get_subwin(x,y);
/* rotation amount */
      da = abs(dx) + abs(dy);
/* vector from center to mouse pointer (different for OpenGL window) */
#ifdef GTK_GL
      if (GTK_IS_GL_AREA(w))
        {
        fx = x - data->offset[0] - (w->allocation.x + w->allocation.width/2);
        fy = data->offset[1] + (w->allocation.y + w->allocation.height/2 - y);
        }
      else
#endif
        {
        fx = (x - data->offset[0] - sysenv.subcenx[m]);
        fy = (data->offset[1] + sysenv.subceny[m] - y);
        }
#if DEBUG_MOTION
printf("(%d,%d) x (%d,%d)\n",dx,dy,fx,fy);
#endif

/* rotation direction via z component of cross product (+=clock, -=anti) */
      if ((fx*dy - dx*fy) > 0)
        ds = -1;
      else
        ds = 1;
/* assign and calculate */
      data->da = (gfloat) (ds*da);
      data->da *= ROTATE_SCALE * PI / 180.0;
      calc_coords(ROLL, data);
      data->da = 0.0;
      }
    refresh++;
    }
  else
    {
/* pitch and roll */
    dy = y-oy;
    dx = x-ox;
    if (dy)
      {
      data->da = (gfloat) ROTATE_SCALE * dy * PI / 180.0;
      calc_coords(PITCH, data);
      data->da = 0.0;
      refresh++;
      }
    if (dx)
      {
      data->da = (gfloat) ROTATE_SCALE * dx * PI / 180.0;
      calc_coords(ROTATE, data);
      data->da = 0.0;
      refresh++;
      }
    }
  }

/* save old values */
ox=x;
oy=y;

/* redraw? */
if (refresh)
  {
#ifdef GTK_GL
  if (GTK_IS_GL_AREA(w))
    opengl_draw(w);
  else
    redraw_canvas(SINGLE);
#else
  redraw_canvas(SINGLE);
#endif
  }
 
return(TRUE);
}

/**************************/
/* switch selection modes */
/**************************/
void selection_mode_atoms()
{
sysenv.select_mode = ATOM;
}

void selection_mode_mols()
{
sysenv.select_mode = MOL;
}

/*******************/
/* clear selection */
/*******************/
void clear_select()
{
gint i, atom;
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

/* clean up all selection crap */
for (i=0 ; i<data->select_size ; i++)
  {
  atom = *(data->select_list+i);
  (data->atoms+atom)->status &= ~SELECT_HL;
  }
data->select_size = 0;
/*
data->box_on = FALSE;
*/

/* redraw */
redraw_canvas(SINGLE);
}

/******************/
/* copy selection */
/******************/
void copy_select(void)
{
gint model;
struct model_pak *data;

/* setup & check */
model = sysenv.active;
data = model_ptr(model, RECALL);
if (data == NULL)
  return;
if (!data->select_size)
  {
  show_text("Nothing selected.");
  return;
  }

/* save ptr as this won't change if models (other than itself) are deleted */
sysenv.select_source = data;
}

/*********************/
/* paste a selection */
/*********************/
void paste_select(void)
{
gint model, start, i, j;
struct model_pak *data, *src;

/* checks */
model = sysenv.active;
data = model_ptr(model, RECALL);
src = sysenv.select_source;
if (data == NULL || src == NULL)
  return;
if (data->id == NODATA || !src->select_size)
  return;

/* clean up any old selection highlighting in destination model */
for (i=0 ; i<data->select_size ; i++)
  {
  j = *(data->select_list+i);
  (data->atoms+j)->status &= ~SELECT_HL;
  }

/* add the atoms */
start = data->num_atoms;
for (i=0 ; i<src->select_size ; i++)
  {
  j = *(src->select_list+i);
  copy_core(data, src, j);
  }

/* make pasted selection a selection in the new model */
g_free(data->select_list);
data->select_size = src->select_size;
data->select_limit = src->select_size;
data->select_list = (gint *) g_malloc(data->select_size*sizeof(gint));

/* construct & highlight the pasted selection */
for (i=0 ; i<data->select_size ; i++)
  {
/* highlight */
  (data->atoms+start)->status |= SELECT_HL;
  *(data->select_list+i) = start;
  start++;
  }

/* update */
init_objs(REDO_COORDS, data);
calc_bonds(data);
calc_mols(data);

data->mode = FREE;
switch_mode(ALTER_SELECT);
}

/********************/
/* delete selection */
/********************/
void delete_select(void)
{
gint i, atom;
struct model_pak *data;

/* exit if nothing is loaded */
if (!sysenv.num_models)
  return;
/* deletion for the active model only */
data = model_ptr(sysenv.active, RECALL);
/* exit if nothing selected */
if (!data->select_size)
  return;
/* delete */
for (i=0 ; i<data->select_size ; i++)
  {
  atom = *(data->select_list+i);
  (data->atoms+atom)->status |= DELETED;
  }
sysenv.select_source = NULL;

/* NEW - irrevocable */
data->select_size = 0;
mem_shrink(data);

/* update */
calc_bonds(data);
calc_mols(data);
/* draw */
redraw_canvas(SINGLE);
}

/******************/
/* hide selection */
/******************/
void hide_select(void)
{
gint i, atom;
struct model_pak *data;

/* exit if nothing is loaded */
if (!sysenv.num_models)
  return;
/* deletion for the active model only */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;
/* exit if nothing selected */
if (!data->select_size)
  return;
/* hide */
for (i=0 ; i<data->select_size ; i++)
  {
  atom = *(data->select_list+i);
/* FIXME - need checks (here & elsewhere) to ensure index is within range */
  (data->atoms+atom)->status |= HIDDEN;
  }
sysenv.select_source = NULL;

/* update */
calc_bonds(data);
calc_mols(data);
/* draw */
redraw_canvas(SINGLE);
}

/***************************/
/* expose all hidden atoms */
/***************************/
void unhide_atoms(void)
{
gint i;
struct model_pak *data;

/* exit if nothing is loaded */
if (!sysenv.num_models)
  return;
/* deletion for the active model only */
data = model_ptr(sysenv.active, RECALL);

/* unhide */
for (i=0 ; i<data->num_atoms ; i++)
  (data->atoms+i)->status &= ~HIDDEN;

/* update */
calc_bonds(data);
calc_mols(data);
redraw_canvas(SINGLE);
}

/****************************************/
/* view - filenames in the drawing area */
/****************************************/
void view_names()
{
sysenv.show_names ^= 1;
redraw_canvas(ALL);
}
/*************************************/
/* view - energy in the drawing area */
/*************************************/
void view_energy()
{
sysenv.show_energy ^= 1;
redraw_canvas(ALL);
}

/****************/
/* change mode */
/***************/
void switch_mode(gint new_mode)
{
gint i,model;
struct model_pak *data;

if (!sysenv.num_models)
  return;

/* go through all selected models */
model = sysenv.active;
data = model_ptr(model, RECALL);

/* clean up (if necessary) from previous mode */
switch (data->mode)
  {
  case ALTER_SELECT:
    clear_select();
    break;
  case ATOM_MOVE:
    move_atom(0,0,STOP);
    break;
  case MOL_MOVE:
    move_mol(0,0,STOP);
    break;
  case DIST_INFO:
    info_dist(-1,-1,data);           /* reset static vars */
  case BOND_INFO:
    for (i=data->num_atoms ; i-- ; )
      (data->atoms+i)->status &= ~SQUARE_HL;
    break;
  }
/* setup (if necesary) for new mode */
switch(new_mode)
  {
  case REGION_LITE:
    init_objs(REFRESH_LITE, data);
    break;
  case ALTER_SELECT:
    if (data->fractional)
      {
      if (data->sginfo.lattice != XS_ORTHOROMBIC && 
          data->sginfo.lattice != XS_TETRAGONAL && 
          data->sginfo.lattice != XS_CUBIC)
        {
show_text("Warning: can't do this properly with non cartesian axes yet.");
        }
      }
/* NEW - inverse matrix stuff no longer used */
    break;
  }

/* set the new mode */
data->mode = new_mode;
redraw_canvas(SINGLE);
}

/* Need this for the menu item factory widget */
void gtk_switch_mode(GtkWidget *w, gint new_mode)
{
switch_mode(new_mode);
}

/***************/
/* change view */
/***************/
void switch_view(gint new_mode)
{
gint i,j;
gint mod=0, inc=0, count;

/* - change the number of displayed models */
i=sysenv.subx;
j=sysenv.suby;
switch (new_mode)
  {
  case SPLIT_NONE:
    i=j=1;
    mod++;
    break;
  case SPLIT_VERT:
    i++;
    mod++;
    break;
  case SPLIT_HORZ:
    j++;
    mod++;
    break;
  case SPLIT_BOTH:
    i++;
    j++;
    mod++;
    break;
  case PREV_MODEL:
    inc=-2;
  case NEXT_MODEL:
    inc++;
/* don't do if equal or more display slots than models */
    if ((sysenv.num_displayed - sysenv.num_models) >= 0)
      break;
/* find active model number and apply change */
    for (i=0 ; i<sysenv.num_displayed ; i++)
      if (sysenv.displayed[i] == sysenv.active)
        {
        sysenv.displayed[i] += inc;
        break;
        }
/* wrap around */
    if (sysenv.displayed[i] < 0)
      sysenv.displayed[i] += sysenv.num_models;
    if (sysenv.displayed[i] >= sysenv.num_models)
      sysenv.displayed[i] -= sysenv.num_models;
    sysenv.active = sysenv.displayed[i];
/* avoid repeat displays of the active model */
    do
      {
      count=0;
      for (j=0 ; j<sysenv.num_displayed ; j++)
        if (sysenv.displayed[j] == sysenv.active)
          count++;
/* found a duplicate - try the next */
      if (count == 2)
        sysenv.displayed[i] += inc;
/* wrap around (need to redo count if wrap occurs) */
      if (sysenv.displayed[i] < 0)
        {
        sysenv.displayed[i] += sysenv.num_models;
        count=2;
        }
      if (sysenv.displayed[i] >= sysenv.num_models)
        {
        sysenv.displayed[i] -= sysenv.num_models;
        count=2;
        }
      sysenv.active = sysenv.displayed[i];
      }
    while (count == 2);
    break;
  default:
    printf("Mode %d not implemented yet!\n",new_mode);
  }
/* this call updates subwin dimensions & num_displayed */
if (mod)
  mod_screen(i,j);

/* update the screen */
redraw_canvas(ALL);
}

/* GTK hook */
void gtk_switch_view(GtkWidget *w, gint new_mode)
{
switch_view(new_mode);
}

/************************************/
/* pick a (loaded) model to display */
/************************************/
#define DEBUG_PICK_MODEL 0
void pick_model(gint model)
{
gint i;

/* check */
if (model < 0 || model > sysenv.num_models)
  {
#if DEBUG_PICK_MODEL
  printf("pick_model() error: bad model number.\n");
#endif
  return;
  }
#if TIMER
start_timer("pick_model");
#endif

/* make the model active */
sysenv.active = model;
/* updates */
refresh_view();
update_creator();

/* cycle through list of dialogs & update accordingly */
for (i=0 ; i<MAX_DIALOGS ; i++)
  {
  if (!sysenv.dialog[i].active)
    continue;
/* dialog dependent refresh */
  switch(sysenv.dialog[i].type)
    {
    case GEOMETRY:
      update_geom_info();
      break;
    case OPENGL_OPTIONS:
/* TODO - update dialog with new model's opengl options */
printf("TODO - update opengl_options dialog\n");
      break;
    default:;
/* dialogs that don't need refreshing */
    }
  }
redraw_canvas(ALL);

#if TIMER
stop_timer("pick_model");
#endif
}

/****************************/
/* event hook for the above */
/****************************/
void event_pick_model(GtkWidget *w, struct model_pak *data)
{
g_return_if_fail(data != NULL);
g_return_if_fail(data->id != NODATA);
pick_model(data->number);
}

/*****************************************/
/* force the selection of main tree item */
/*****************************************/
void tree_select(gint model)
{
if (GTK_IS_TREE(tree))
  gtk_tree_select_item(GTK_TREE(tree), model);
}

/*****************************/
/* view panel button handler */
/*****************************/
void reset_view(GtkWidget *w, gint signal)
{
struct model_pak *data;

/* do we have a loaded model? */
data = model_ptr(sysenv.active, RECALL);
if (data == NULL)
  return;

/* make the changes */
switch(signal)
  {
  case ROTATION:
    init_rotmat(data->rotmat);
    VEC3SET(data->angle, 0.0, 0.0, 0.0);
/* FIXME - here or elsewhere??? */
    data->scale = 1.0;
    if (sysenv.rotframe_on)
      {
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(xangle),0);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(yangle),0);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(zangle),0);
      }
    break;
  default:
    printf("(1) Unknown signal.\n");
  }

/* update */
init_objs(INIT_COORDS, data);
redraw_canvas(SINGLE);
}

/*******************************/
/* refresh spin buttons values */
/*******************************/
void refresh_view()
{
struct model_pak *data;

if (!sysenv.rotframe_on)
  return;

data = model_ptr(sysenv.active, RECALL);
g_return_if_fail(data != NULL);

gtk_spin_button_set_value(GTK_SPIN_BUTTON(xangle), data->angle[0]);
gtk_signal_emit_by_name (GTK_OBJECT (xangle), "changed");
gtk_spin_button_set_value(GTK_SPIN_BUTTON(yangle), data->angle[1]);
gtk_signal_emit_by_name (GTK_OBJECT (yangle), "changed");
gtk_spin_button_set_value(GTK_SPIN_BUTTON(zangle), data->angle[2]);
gtk_signal_emit_by_name (GTK_OBJECT (zangle), "changed");
}

/******************************/
/* The omnipresent view panel */
/******************************/
void change_view(GtkWidget *w, gint signal)
{
gfloat new;
struct model_pak *data;

if (!sysenv.rotframe_on)
  return;

/* do we have a loaded model? */
data = model_ptr(sysenv.active, RECALL);
if (data == NULL)
  {
/* user is clicking & there is no model - force to starting values */
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(xangle),0);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(yangle),0);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(zangle),0);
  return;
  }

/* ok, go ahead and make the changes */
switch(signal)
  {
  case UPDATE_X:
    new = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(xangle));
    data->da = PI*(new - data->angle[0])/180.0;
    calc_coords(ROTATE, data);
    data->da = 0.0;
    data->angle[0] = new;
    break;
  case UPDATE_Y:
    new = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(yangle));
    data->da = PI*(new - data->angle[1])/180.0;
    calc_coords(PITCH, data);
    data->da = 0.0;
    data->angle[1] = new;
    break;
  case UPDATE_Z:
    new = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(zangle));
    data->da = PI*(new - data->angle[2])/180.0;
    calc_coords(ROLL, data);
    data->da = 0.0;
    data->angle[2] = new;
    break;
  default:
    printf("(2) Unknown signal.\n");
  }

redraw_canvas(SINGLE);
}

/***************************/
/* create/update text tray */
/***************************/
void show_text(gchar *message)
{
static GtkWidget *tray=NULL;

if (!tray)
  {
  tray = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX (tbox), tray, TRUE, TRUE, 0);
  gtk_widget_show(tray);
  gtk_entry_set_editable(GTK_ENTRY(tray), FALSE);
  }
/* strdup so message can be subsequently freed by caller */
if (message)
  gtk_entry_set_text(GTK_ENTRY(tray), g_strdup(message));
}

/*************************************/
/* rotation spinner display toggling */
/*************************************/
void rotframe_toggle()
{
sysenv.rotframe_on ^= 1;

if (sysenv.rotframe_on)
  gtk_widget_show(sysenv.rotframe);
else
  gtk_widget_hide(sysenv.rotframe);
}

/********************/
/* Property toggles */
/********************/
void at_toggle(struct model_pak *data)
{
/* select/highlight the assoc. model */
data->axes_type ^= 1;
pick_model(data->number);
}
void hp_toggle(struct model_pak *data)
{
/* select/highlight the assoc. model */
data->hpr ^= 1;
pick_model(data->number);
}
void lp_toggle(struct model_pak *data)
{
/* select/highlight the assoc. model */
data->morph_label ^= 1;
pick_model(data->number);
}
void al_toggle(struct model_pak *data)
{
/* select/highlight the assoc. model */
data->atom_labels ^= 1;
pick_model(data->number);
}
void ss_toggle(struct model_pak *data)
{
/* select/highlight the assoc. model */
data->show_shells ^= 1;
pick_model(data->number);
}
void sc_toggle(struct model_pak *data)
{
/* select/highlight the assoc. model */
data->cell_on ^= 1;
pick_model(data->number);
}

/* colourize by scaling with SOF */
void cs_toggle(struct model_pak *data)
{
data->sof_colourize ^= 1;
if (data->sof_colourize)
  init_objs(SOF_COLOUR, data);
else
  init_objs(REFRESH_COLOUR, data);
pick_model(data->number);
}

void rl_toggle(struct model_pak *data)
{
static gint on=0;

on ^= 1;
if (!on)
  init_objs(REFRESH_COLOUR, data);
else
  switch_mode(REGION_LITE);

pick_model(data->number);
}

/*********************************/
/* add new model to tree display */
/*********************************/
void new_tree_item(gint model, gint mode)
{
GtkWidget *subtree, *subitem, *subprop, *button;
GtkWidget *hbox, *wid;
GdkPixmap *pix=NULL;
GdkBitmap *mask;
GtkStyle *style;
struct model_pak *data;

/* checks */
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);
g_return_if_fail(tree != NULL);

switch(mode)
  {
  case REPLACE:
    gtk_label_set_text(GTK_LABEL(data->tree_name), data->basename);
    return;
  case APPEND:
    data->tree_name = gtk_label_new(data->basename);
    break;
  default:
    printf("new_tree_item() warning: unknown calling mode.\n");
    return;
  }

/* the main model tree item */
data->tree_label = gtk_tree_item_new();
hbox = gtk_hbox_new(FALSE,3);
gtk_container_add(GTK_CONTAINER(data->tree_label), hbox);

/* setup and add pixmap */
style = gtk_widget_get_style(window);
switch(data->id)
  {
  case MORPH:
      pix = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                                         diamond2_xpm);
    break;
  default:
    if (!data->periodic)
      pix = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                                         methane_xpm);
    else
      pix = gdk_pixmap_create_from_xpm_d(window->window, &mask, &style->white,
                                                         box_xpm);
  }
/* add icon to main tree item */
wid = gtk_pixmap_new(pix, mask);
gtk_box_pack_start(GTK_BOX(hbox), wid, FALSE, FALSE, 0);

/* add name to main tree item */
gtk_box_pack_start(GTK_BOX(hbox), data->tree_name, FALSE, FALSE, 0);

/* main tree */
gtk_tree_append (GTK_TREE(tree), data->tree_label);
gtk_signal_connect (GTK_OBJECT(data->tree_label), "select",
                    GTK_SIGNAL_FUNC(event_pick_model), (gpointer) data);

/* add sub tree */
subtree = gtk_tree_new();
gtk_tree_item_set_subtree(GTK_TREE_ITEM(data->tree_label), subtree);

/* model dependent symmetry items */
if (data->periodic > 1)
  {
  subitem = gtk_tree_item_new_with_label ("periodicity");
  gtk_tree_append (GTK_TREE(subtree), subitem);
  gtk_signal_connect (GTK_OBJECT(subitem), "select",
                      GTK_SIGNAL_FUNC(space_info), (gpointer) data);
  gtk_widget_show_all(subitem);
  }
else
  {
/* symmetry analyzer */
  if (data->id != MORPH)
    {
    subitem = gtk_tree_item_new_with_label ("symmetry");
    gtk_tree_append (GTK_TREE(subtree), subitem);
    gtk_signal_connect (GTK_OBJECT(subitem), "select",
                        GTK_SIGNAL_FUNC(sym_info), (gpointer) data);
    gtk_widget_show_all(subitem);
    }
  }

/* model dependent item - Animate */
if (data->animation)
  {
  subitem = gtk_tree_item_new_with_label ("animation");
  gtk_tree_append (GTK_TREE(subtree), subitem);
  gtk_signal_connect (GTK_OBJECT(subitem), "select",
                      GTK_SIGNAL_FUNC(animate_setup_widget), NULL);
  gtk_widget_show_all (subitem);
  }

/* model dependent item - rdf */
if (data->id == BIOSYM)
{
  subitem = gtk_tree_item_new_with_label("rdf");
  gtk_tree_append (GTK_TREE(subtree), subitem);
  gtk_signal_connect (GTK_OBJECT(subitem), "select",
                      GTK_SIGNAL_FUNC(create_rdfwindow), NULL);
  gtk_widget_show_all (subitem);
}

/* all but morphology - gulp template data */
if (data->id != MORPH)
  {
  subitem = gtk_tree_item_new_with_label("energetics");
  gtk_tree_append(GTK_TREE(subtree), subitem);
  gtk_signal_connect(GTK_OBJECT(subitem), "select",
                     GTK_SIGNAL_FUNC(gulp_widget), (gpointer) data);
  gtk_widget_show_all(subitem);
  }
else
  {
/* morphology only - facets table */
  subitem = gtk_tree_item_new_with_label("facet list");
  gtk_tree_append(GTK_TREE(subtree), subitem);
  gtk_signal_connect(GTK_OBJECT(subitem), "select",
                     GTK_SIGNAL_FUNC(morph_widget), (gpointer) data);
  gtk_widget_show_all(subitem);
  }

/* all - model dependent properties */

/* base tree item */
subitem = gtk_tree_item_new_with_label ("display properties");
gtk_tree_append (GTK_TREE(subtree), subitem);
gtk_widget_show(subitem);

/* list of properties */
subprop = gtk_tree_new();
gtk_tree_item_set_subtree (GTK_TREE_ITEM(subitem), subprop);

/* create sub tree's */
switch(data->id)
  {
  case MORPH:
/* property 1 */
    subitem = gtk_tree_item_new();
    button = gtk_check_button_new_with_label ("all planes");
    gtk_container_add (GTK_CONTAINER (subitem), button);
    gtk_tree_append (GTK_TREE(subprop), subitem);
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                               GTK_SIGNAL_FUNC (hp_toggle), (gpointer) data);
    gtk_widget_show_all(subitem);
/* property 2 */
    subitem = gtk_tree_item_new();
    button = gtk_check_button_new_with_label ("no labels");
    gtk_container_add (GTK_CONTAINER (subitem), button);
    gtk_tree_append (GTK_TREE(subprop), subitem);
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                               GTK_SIGNAL_FUNC (lp_toggle), (gpointer) data);
    gtk_widget_show_all(subitem);

  break;
  default:
/* property 1 */
    subitem = gtk_tree_item_new();
    button = gtk_check_button_new_with_label ("atom labels");
    gtk_container_add (GTK_CONTAINER (subitem), button);
    gtk_tree_append (GTK_TREE(subprop), subitem);
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                               GTK_SIGNAL_FUNC (al_toggle), (gpointer) data);
    gtk_widget_show_all(subitem);
/* property 2 */
/*
    if (data->num_shells)
*/
/* if shells are pasted in we have no on the fly way of making this yet, */
/* so always include it */
    if (1)
      {
      subitem = gtk_tree_item_new();
      button = gtk_check_button_new_with_label ("shells");
      gtk_container_add (GTK_CONTAINER (subitem), button);
      gtk_tree_append (GTK_TREE(subprop), subitem);
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                                 GTK_SIGNAL_FUNC (ss_toggle), (gpointer) data);
      gtk_widget_show_all(subitem);
      }

    if (data->periodic > 1)
      {
/* property 4 */
      subitem = gtk_tree_item_new();
      button = gtk_check_button_new_with_label ("axes type");
      gtk_container_add (GTK_CONTAINER (subitem), button);
      gtk_tree_append (GTK_TREE(subprop), subitem);
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                                 GTK_SIGNAL_FUNC (at_toggle), (gpointer) data);
      gtk_widget_show_all(subitem);
/* property 5 */
      subitem = gtk_tree_item_new();
      button = gtk_check_button_new_with_label ("hide frame");
      gtk_container_add (GTK_CONTAINER (subitem), button);
      gtk_tree_append (GTK_TREE(subprop), subitem);
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                                 GTK_SIGNAL_FUNC (sc_toggle), (gpointer) data);
      gtk_widget_show_all(subitem);
      }

/* gulp site occupancy factor */
    if (data->has_sof)
      {
      subitem = gtk_tree_item_new();
      button = gtk_check_button_new_with_label ("colour by SOF");
      gtk_container_add (GTK_CONTAINER (subitem), button);
      gtk_tree_append (GTK_TREE(subprop), subitem);
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                                 GTK_SIGNAL_FUNC (cs_toggle), (gpointer) data);
      gtk_widget_show_all(subitem);
      }

/* property 3 - better test? */
/*
    if (data->id == MARVIN || data->id == MVNOUT || data->id == GULP)
*/
    if (!data->region_empty[REGION2A])
      {
      subitem = gtk_tree_item_new();
      button = gtk_check_button_new_with_label ("regions");
      gtk_container_add (GTK_CONTAINER (subitem), button);
      gtk_tree_append (GTK_TREE(subprop), subitem);
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                                 GTK_SIGNAL_FUNC (rl_toggle), (gpointer) data);
      gtk_widget_show_all(subitem);
      }

  break;
  }

/* show new model name */
gtk_widget_show_all(data->tree_label);
}

/********************************/
/* Remove a model from the tree */
/********************************/
#define DEBUG_DEL_TREE_ITEM 0
void del_tree_item(gint model)
{
struct model_pak *data;

data = model_ptr(model, RECALL);

g_return_if_fail(tree != NULL);
g_return_if_fail(data != NULL);

/* some GTK-CRIT's used to occur here, related to selected children */
/* of the tree item (that are deleted) getting a signal after deletion */
/* problem fixed now ??? */
/* get occ. core dumps */
#if DEBUG_DEL_TREE_ITEM
printf("Attempting to destroy model tree item...\n");
#endif
/*
if (GTK_IS_TREE(tree) && GTK_IS_TREE_ITEM(data->tree_label))
  gtk_container_remove(GTK_CONTAINER(tree), data->tree_label);
*/
/* non trivial to reproduce core dumps occur here */
/* one way is to use view-atom-info mode - click on lots of atoms */
/* and then delete the model - not 100% reproducible, but usually works */
if (GTK_IS_TREE(tree) && GTK_IS_TREE_ITEM(data->tree_label))
  gtk_tree_remove_item(GTK_TREE(tree), data->tree_label);

#if DEBUG_DEL_TREE_ITEM
printf("done.\n");
#endif
}

/*******************************/
/* write gdisrc before exiting */
/*******************************/
void gdis_exit()
{
#ifdef TIMER
print_times();
#endif
/* if user has changed element colours, but doesn't want them saved */
/* this will do it anyway - but they can just click reset next time */
if (sysenv.write_gdisrc)
  write_gdisrc();
gtk_exit(0);
}

/******************/
/* MENU structure */
/******************/
static GtkItemFactoryEntry menu_items[] = 
  {
    { "/File",          NULL, NULL, 0, "<Branch>" },
    { "/File/Load",     NULL, load_widget, 0, NULL },
    { "/File/Save",     NULL, file_save, 0, NULL },
    { "/File/Save as",  NULL, save_as_widget, 0, NULL },
    { "/File/sep1",     NULL, NULL, 0, "<Separator>" },
    { "/File/Quit",     NULL, gdis_exit, 0, NULL },

    { "/Edit",                  NULL, NULL, 0, "<Branch>" },
    { "/Edit/Coordinates",      NULL, modify_model, 0, NULL },
    { "/Edit/sep1",             NULL, NULL, 0, "<Separator>" },
    { "/Edit/Copy selection",   NULL, copy_select, 0, NULL },
    { "/Edit/Paste selection",  NULL, paste_select, 0, NULL },
    { "/Edit/Move selection",   NULL, gtk_switch_mode, ALTER_SELECT, NULL },
    { "/Edit/sep1",             NULL, NULL, 0, "<Separator>" },
    { "/Edit/Delete selection", NULL, delete_select, 0, NULL },
    { "/Edit/Hide selection",   NULL, hide_select, 0, NULL },

    { "/View",                  NULL, NULL, 0, "<Branch>" },
    { "/View/Normal viewing",   NULL, gtk_switch_mode, FREE, NULL },
    { "/View/Unhide atoms",     NULL, unhide_atoms, 0, NULL },
    { "/View/Overlay models",   NULL, overlay_models_dialog, 0, NULL },
    { "/View/sep1",             NULL, NULL, 0, "<Separator>" },
    { "/View/Model names",      NULL, view_names, 0, "<ToggleItem>" },
    { "/View/Model energy",     NULL, view_energy, 0, "<ToggleItem>" },
    { "/Tools",                 NULL, NULL, 0, "<Branch>" },
    { "/Tools/MD initializer",  NULL, mdi_setup_widget, 0, NULL },
    { "/Tools/2D VDW surface",  NULL, surf_dialog, 0, NULL },
/* NB: the next is not fully implemented yet */
    { "/Tools/3D VDW surface",  NULL, surf2_dialog, 0, NULL },
    { "/Tools/Surfaces",        NULL, surface_dialog, 0, NULL },
    { "/Info",                  NULL, NULL, 0, "<LastBranch>" },
    { "/Info/About",            NULL, about, 0, NULL },
  };

/*********/
/* SETUP */
/*********/
void connect_events()
{
gint i, pointsize, pixelsize;
gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
GString *value;
gchar **token;
GtkWidget *hpaned, *main_vbox, *vbox, *vbox2, *hbox, *menu_bar, *toolbar;
GtkWidget *frame, *label, *event_box, *button, *menu, *list, *elem;
GdkBitmap *mask;
GtkStyle *style, *logo_style;
GtkWidget *gdis_wid;
GdkPixmap *gdis_pix=NULL;
GtkAdjustment *adj;
GtkItemFactory *item;
/* tree display */
GtkWidget *scrolled_win;

/* NEW - cope with funny visual types (eg SG displays) */
/* use maximum available depth */
sysenv.depth = gdk_visual_get_best_depth();
sysenv.visual = gdk_visual_get_best_with_depth(sysenv.depth);
if (sysenv.visual == NULL)
  {
  printf("Error: could not get requested visual.\n");
  exit(1);
  }
sysenv.colourmap = gdk_colormap_new(sysenv.visual, TRUE);
if (sysenv.colourmap == NULL)
  {
  printf("Error: could not allocate colourmap.\n");
  exit(1);
  }

/* main window */
window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
gtk_window_set_title(GTK_WINDOW(window),"GTK DISMOL");
gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
/* label creation g_string */
value = g_string_new(NULL);

/* vbox for the menubar */
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_add (GTK_CONTAINER(window), vbox);

/* item factory menu creation */
item = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", NULL);
gtk_item_factory_create_items (item, nmenu_items, menu_items, NULL);
menu_bar = gtk_item_factory_get_widget (item, "<main>");
/* FALSE,FALSE => don't expand to fill (eg on resize) */
gtk_box_pack_start (GTK_BOX(vbox), menu_bar, FALSE, FALSE, 0);

/* toolbar */
toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
gtk_container_set_border_width(GTK_CONTAINER(toolbar), 2);
gtk_toolbar_set_space_size(GTK_TOOLBAR(toolbar), 6);
gtk_box_pack_start(GTK_BOX(vbox), toolbar, FALSE, FALSE, 0);

/* extract some important info for pixmap creation */
gtk_widget_realize(window);
style = gtk_widget_get_style(window);

/* load button */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, folder_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,                  /* label */
                        "Load new model",      /* tooltip */
                        "Private",             /* private info */
                        gdis_wid,              /* icon widget */
                        GTK_SIGNAL_FUNC(load_widget), /* signal */
                        NULL);                       /* signal data */
/* save button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, disk_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,                  /* label */
                        "Save active model",   /* tooltip */
                        "Private",             /* private info */
                        gdis_wid,              /* icon widget */
                        GTK_SIGNAL_FUNC(save_as_widget), /* signal */
                        NULL);                          /* signal data */
/* creator button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, bulb_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Create new model",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(create_new_model),
                        NULL);
/* delete button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, cross_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,                   /* label */
                        "Close active model",   /* tooltip */
                        "Private",              /* private info */
                        gdis_wid,               /* icon widget */
                        GTK_SIGNAL_FUNC(delete_active), /* signal */
                        NULL);                  /* signal data */
/* GAP */
for (i=0 ; i<4 ; i++)
  gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* gperiodic button */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, element_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Periodic table",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gperiodic_win),
                        NULL);

/* geometry button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, geom_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Measurements",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(geom_info),
                        NULL);

/* render button */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, ball_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "Rendering setup",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(render_setup_widget),
                        NULL);

/* OpenGL button */
#ifdef GTK_GL
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, yellow_ball_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "OpenGL toggle",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gl_toggle),
                        NULL);
#endif

/* GAP */
for (i=0 ; i<4 ; i++)
  gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* orientation etc. reset button */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, axes_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "reset orientation and scale",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(reset_view),
                        (gpointer) ROTATION);

/* rotation spinner toggle */
gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, rotate_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "toggle rotation spinners",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(rotframe_toggle),
                        NULL);

/* GAP */
for (i=0 ; i<4 ; i++)
  gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* one pane */ 
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_none_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "normal view",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) SPLIT_NONE);

gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* vpane */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_vert_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split vertically",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) SPLIT_VERT);

gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* hpane */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_horz_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split horizontally",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) SPLIT_HORZ);

gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* vhpane */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, split_both_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "split both",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) SPLIT_BOTH);

/* GAP */
for (i=0 ; i<4 ; i++)
  gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* NEW - make prev model active */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, left_arrow1_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "display previous model",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) PREV_MODEL);

gtk_toolbar_append_space(GTK_TOOLBAR(toolbar));

/* NEW - make next model active */
gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                                    &style->white, right_arrow1_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);                  /* icon widget */
gtk_toolbar_append_item(GTK_TOOLBAR (toolbar),
                        NULL,
                        "display next model",
                        "Private",
                        gdis_wid,
                        GTK_SIGNAL_FUNC(gtk_switch_view),
                        (gpointer) NEXT_MODEL);

/* MAIN LEFT/RIGHT HBOX PANE */
/* NEW - paned window */
hpaned = gtk_hpaned_new();
gtk_container_add (GTK_CONTAINER(GTK_BOX(vbox)), hpaned);
/*
gtk_paned_set_handle_size (GTK_PANED(hpaned), 10);
gtk_paned_set_gutter_size (GTK_PANED(hpaned), 10); 
*/

/* LEFT PANE */
main_vbox = gtk_vbox_new(FALSE, 0);
gtk_paned_pack1(GTK_PANED(hpaned), main_vbox, TRUE, FALSE);

/* MODEL TREE */
vbox = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(main_vbox),vbox,TRUE,TRUE,0);
gtk_container_set_border_width (GTK_CONTAINER(vbox), 4);

/* A generic scrolled window */
scrolled_win = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
                                GTK_POLICY_AUTOMATIC,
                                GTK_POLICY_AUTOMATIC);
/* TODO - set default size instead of usize */
gtk_widget_set_usize (scrolled_win, 140, 200);
gtk_container_add (GTK_CONTAINER(vbox), scrolled_win);
gtk_widget_show (scrolled_win);

/* Create the root tree */
tree = gtk_tree_new();
gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW(scrolled_win), tree);
gtk_tree_set_selection_mode (GTK_TREE(tree), GTK_SELECTION_SINGLE);
/*
gtk_tree_set_view_lines(GTK_TREE(tree), FALSE);
*/
gtk_widget_show (tree);


/* ROTATION FRAME */
sysenv.rotframe_on = 1;
sysenv.rotframe = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(main_vbox),sysenv.rotframe,FALSE,FALSE,0);
gtk_container_set_border_width (GTK_CONTAINER(sysenv.rotframe), PANEL_SPACING);

vbox2 = gtk_vbox_new(FALSE, 5);
gtk_container_add (GTK_CONTAINER(sysenv.rotframe), vbox2);

hbox = gtk_hbox_new(TRUE, 5);
gtk_box_pack_start(GTK_BOX(vbox2),hbox,FALSE,FALSE,0);
gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(hbox)), 5);

/* rotation around x */
vbox = gtk_vbox_new(FALSE, 5);
gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
label = gtk_label_new ("Rotate");
gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,TRUE,0);
gtk_misc_set_alignment(GTK_MISC(label), 0.2f, 0.5f);
adj = (GtkAdjustment *) gtk_adjustment_new (0, 0, 360, 5, 10, 0);
xangle = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (xangle), TRUE);
gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (xangle), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(vbox),xangle,FALSE,FALSE,0);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (change_view), (gpointer) UPDATE_X);

/* rotation around y */
vbox = gtk_vbox_new(FALSE, 5);
gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
label = gtk_label_new ("Pitch");
gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,TRUE,0);
gtk_misc_set_alignment(GTK_MISC(label), 0.2f, 0.5f);
adj = (GtkAdjustment *) gtk_adjustment_new (0, 0, 360, 5, 10, 0);
yangle = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (yangle), TRUE);
gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (yangle), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(vbox),yangle,FALSE,FALSE,0);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (change_view), (gpointer) UPDATE_Y);

/* spinner for rotation around z */
vbox = gtk_vbox_new(FALSE, 5);
gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
label = gtk_label_new ("Yaw");
gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,TRUE,0);
gtk_misc_set_alignment(GTK_MISC(label), 0.2f, 0.5f);
adj = (GtkAdjustment *) gtk_adjustment_new (0, 0, 360, 5, 10, 0);
zangle = gtk_spin_button_new (adj, 0, 0);
gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (zangle), TRUE);
gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (zangle), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(vbox),zangle,FALSE,FALSE,0);
gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
                    GTK_SIGNAL_FUNC (change_view), (gpointer) UPDATE_Z);

/* GDIS LOGO */
frame = gtk_frame_new(NULL);
gtk_box_pack_end(GTK_BOX(main_vbox), frame, FALSE, TRUE, 0);
gtk_container_set_border_width (GTK_CONTAINER(frame), PANEL_SPACING);

/* NEW - event box (get an x window for setting black background) */
event_box = gtk_event_box_new();
gtk_container_add(GTK_CONTAINER(frame), event_box);

/* force black background & white foreground */
logo_style = gtk_style_copy(style);
logo_style->bg[GTK_STATE_NORMAL].red   = 0;
logo_style->bg[GTK_STATE_NORMAL].green = 0;
logo_style->bg[GTK_STATE_NORMAL].blue  = 0;
gtk_widget_set_style(GTK_WIDGET(event_box), logo_style);

hbox = gtk_hbox_new (FALSE, 10);
gtk_container_add (GTK_CONTAINER(event_box), hbox);
gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);

/*
vbox = gtk_vbox_new (FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
label = gtk_label_new ("GDIS");
gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,FALSE,0);
g_string_sprintf(value, "version %4.2f", VERSION);
label = gtk_label_new (value->str);
gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,FALSE,0);
*/

gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                          &style->bg[GTK_STATE_NORMAL], logo_left_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);
gtk_box_pack_start(GTK_BOX(hbox), gdis_wid, FALSE, FALSE, 0);

gdis_pix = gdk_pixmap_create_from_xpm_d(window->window, &mask,
                          &style->bg[GTK_STATE_NORMAL], logo_right_xpm);
gdis_wid = gtk_pixmap_new(gdis_pix, mask);
gtk_box_pack_end(GTK_BOX(hbox), gdis_wid, FALSE, FALSE, 0);


/* RIGHT PANE */
vbox = gtk_vbox_new (FALSE, 0);
gtk_paned_pack2(GTK_PANED(hpaned), vbox, TRUE, TRUE);


/* NEW - cope with funny visual types (eg SG displays) */
/* use maximum available depth */
/*
sysenv.depth = gdk_visual_get_best_depth();
sysenv.visual = gdk_visual_get_best_with_depth(sysenv.depth);
if (sysenv.visual == NULL)
  {
  printf("Error: could not get requested visual.\n");
  exit(1);
  }
sysenv.colourmap = gdk_colormap_new(sysenv.visual, TRUE);
if (sysenv.colourmap == NULL)
  {
  printf("Error: could not allocate colourmap.\n");
  exit(1);
  }
*/
/* SGI - make the new visual active (and push the old) */
gtk_widget_push_colormap(sysenv.colourmap);
gtk_widget_push_visual(sysenv.visual);

/*
#if GOK
*/
/* create the main drawing area */
drawing_area = gtk_drawing_area_new();
gtk_drawing_area_size(GTK_DRAWING_AREA(drawing_area),
                      sysenv.width, sysenv.height);
/* NB: expand to fill on resize */
gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);
gtk_widget_show (drawing_area);

/* signals */
gtk_signal_connect (GTK_OBJECT (drawing_area), "expose_event",
                   (GtkSignalFunc) expose_event, NULL);
gtk_signal_connect (GTK_OBJECT(drawing_area),"configure_event",
                   (GtkSignalFunc) configure_event, NULL);
gtk_signal_connect (GTK_OBJECT (drawing_area), "motion_notify_event",
                   (GtkSignalFunc) motion_notify_event, NULL);
gtk_signal_connect (GTK_OBJECT (drawing_area), "button_press_event",
                   (GtkSignalFunc) button_press_event, NULL);
gtk_signal_connect (GTK_OBJECT (drawing_area), "button_release_event",
                   (GtkSignalFunc) button_release_event, NULL);

gtk_widget_set_events (GTK_WIDGET(drawing_area), GDK_EXPOSURE_MASK
                                               | GDK_LEAVE_NOTIFY_MASK
                                               | GDK_BUTTON_PRESS_MASK
                                               | GDK_BUTTON_RELEASE_MASK
                                               | GDK_POINTER_MOTION_MASK
                                               | GDK_POINTER_MOTION_HINT_MASK);
#if GTK_GL
/* check for opengl support, then create a canvas */
if (gdk_gl_query() == FALSE)
  {
  show_text("OpenGL initialization failed!");
  return;
  }

sysenv.glarea = GTK_WIDGET(gtk_gl_area_new(attrlist));
if (!sysenv.glarea)
  {
  printf("Error creating OpenGL drawing area!");
  return;
  }

/* always want x=y, since OpenGL will warp if non sq */
gtk_box_pack_start (GTK_BOX (vbox), sysenv.glarea, TRUE, TRUE, 0);

/* drawing signals */
gtk_signal_connect (GTK_OBJECT (sysenv.glarea), "expose_event",
                   (GtkSignalFunc) gl_expose_event, NULL);
gtk_signal_connect (GTK_OBJECT(sysenv.glarea),"configure_event",
                   (GtkSignalFunc) gl_configure_event, NULL);
/* pointer signals */
gtk_signal_connect (GTK_OBJECT (sysenv.glarea), "motion_notify_event",
                   (GtkSignalFunc) motion_notify_event, NULL);
gtk_signal_connect (GTK_OBJECT (sysenv.glarea), "button_press_event",
                   (GtkSignalFunc) button_press_event, NULL);
gtk_signal_connect (GTK_OBJECT (sysenv.glarea), "button_release_event",
                   (GtkSignalFunc) button_release_event, NULL);

gtk_widget_set_events (GTK_WIDGET(sysenv.glarea), GDK_EXPOSURE_MASK
                                            | GDK_LEAVE_NOTIFY_MASK
                                            | GDK_BUTTON_PRESS_MASK
                                            | GDK_BUTTON_RELEASE_MASK
                                            | GDK_POINTER_MOTION_MASK
                                            | GDK_POINTER_MOTION_HINT_MASK);

gtk_widget_show_all(sysenv.glarea);
#endif

/* SGI - restore the old visual */
gtk_widget_pop_visual();
gtk_widget_pop_colormap();

/* TEXT TRAY */
tbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX (vbox), tbox, FALSE, FALSE, 0);

gtk_widget_show(tbox);
show_text("Welcome to GDIS.");

/* Selection mode pulldown */
menu = gtk_option_menu_new();
gtk_widget_ref(menu);
gtk_box_pack_start(GTK_BOX (tbox), menu, FALSE, FALSE, 0);
list = gtk_menu_new();

/* add pulldowns */
elem = gtk_menu_item_new_with_label("Atoms");
gtk_menu_append(GTK_MENU(list), elem);
gtk_signal_connect(GTK_OBJECT(elem), "activate",
                  (GtkSignalFunc) selection_mode_atoms, elem);
gtk_widget_show(elem);

elem = gtk_menu_item_new_with_label("Mols");
gtk_menu_append(GTK_MENU(list), elem);
gtk_signal_connect(GTK_OBJECT(elem), "activate",
                  (GtkSignalFunc) selection_mode_mols, elem);
gtk_object_set_data(GTK_OBJECT(elem), "id", (gpointer) i);
gtk_widget_show(elem);

gtk_option_menu_set_menu(GTK_OPTION_MENU(menu), list);


/* Task button */
button = gtk_button_new_with_label(" Tasks ");
gtk_box_pack_start(GTK_BOX(tbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC(task_dialog),
                          NULL);
sysenv.task_label = gtk_label_new(" 0 ");
gtk_box_pack_end(GTK_BOX(tbox), sysenv.task_label, FALSE, FALSE, 0);

/* window signals */
gtk_signal_connect (GTK_OBJECT (window), "destroy",
                    GTK_SIGNAL_FUNC (gdis_exit), NULL);

/* final display configuration */
sysenv.show_names = FALSE;
gtk_widget_show_all(window);

#if GTK_GL
gtk_widget_hide(sysenv.glarea);
#endif

sysenv.dfont = gdk_font_load(sysenv.font);	
rotframe_toggle();

/* get font size (for auto text placement calcs) */
/*
value = g_string_new(sysenv.font);
*/
g_string_assign(value, sysenv.font);
token = g_strsplit(value->str, "-", 10);

pixelsize=pointsize=0;
if (strlen(token[7]))
  sscanf(token[7],"%d",&pixelsize);
if (strlen(token[8]))
  sscanf(token[8],"%d",&pointsize);

/* FIXME - how do we PROPERLY convert pointsize into pixel size */
if (pixelsize)
  sysenv.dfontsize = pixelsize;
else
  if (pointsize)
    sysenv.dfontsize = (gint) pointsize/10;

if (!sysenv.dfontsize)
  {
  printf("WARNING: failed to properly determine font size.\n");
  sysenv.dfontsize = 10;
  }

g_string_free(value, TRUE);
}

