/*
 * Electric(tm) VLSI Design System
 *
 * File: usrcomek.c
 * User interface tool: command handler for E through K
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "usr.h"
#include "usrtrack.h"
#include "usrdiacom.h"
#include "efunction.h"
#include "tecart.h"
#include "tecgen.h"
#include "tecschem.h"
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#define MAXLINE 500		/* maximum length of news/help file input lines */

/* working memory for "us_iterate()" */
static INTBIG *us_iterateaddr, *us_iteratetype;
static INTBIG  us_iteratelimit=0;

/*
 * Routine to free all memory associated with this module.
 */
void us_freecomekmemory(void)
{
	if (us_iteratelimit > 0)
	{
		efree((char *)us_iterateaddr);
		efree((char *)us_iteratetype);
	}
}

void us_echo(INTBIG count, char *par[])
{
	REGISTER INTBIG lastquiet, j;
	REGISTER BOOLEAN err;

	lastquiet = ttyquiet(0);
	err = initinfstr();
	for(j=0; j<count; j++)
	{
		err |= addstringtoinfstr(par[j]);
		err |= addtoinfstr(' ');
	}
	if (err)
	{
		ttyputnomemory();
		return;
	}
	ttyputmsg("%s", returninfstr());
	(void)ttyquiet(lastquiet);
}

void us_editfacet(INTBIG count, char *par[])
{
	REGISTER INTBIG implicit, i, len, newwindow, nonredundant, push,
		intoicon, lambda, inplace;
	BOOLEAN newframe;
	INTBIG lx, hx, ly, hy;
	REGISTER NODEPROTO *np, *onp, *np1, *curfacet;
	REGISTER NODEINST *ni, *stacknodeinst;
	REGISTER LIBRARY *lib, *olib;
	REGISTER WINDOWPART *win;
	REGISTER char *pt;
	NODEINST *hini;
	XARRAY rotarray, transarray, xfarray, prevarray;
	PORTPROTO *hipp;

	/* get proper highlighting in subfacet if port is selected */
	us_findlowerport(&hini, &hipp);

	/* find the nodeinst in this window to go "down into" (if any) */
	newwindow = 0;
	nonredundant = 0;
	intoicon = 0;
	inplace = 0;
	if (count == 1)
	{
		len = strlen(par[0]);
		if (len == 0) count--; else
		{
			if (namesamen(par[0], "in-place", len) == 0)
			{
				inplace = 1;
				count--;
			}
		}
	}
	if (count == 0)
	{
		implicit = 1;
		ni = (NODEINST *)us_getobject(VNODEINST, FALSE);
		if (ni == NONODEINST) return;
		stacknodeinst = ni;
		np = ni->proto;

		/* translate this reference if this is an icon facet */
		if (np->cellview == el_iconview)
		{
			implicit = 0;
			if (np->cell != ni->parent->cell)
			{
				onp = contentsview(np);
				if (onp != NONODEPROTO) np = onp;
			} else
			{
				intoicon = 1;
			}
		}
	} else
	{
		/* check for options */
		for(i=1; i<count; i++)
		{
			len = strlen(par[i]);
			if (namesamen(par[i], "new-window", len) == 0 && len > 1) newwindow++; else
			if (namesamen(par[i], "non-redundant", len) == 0 && len > 1) nonredundant++; else
			{
				ttyputbadusage("editfacet");
				return;
			}
		}

		/* see if specified facet exists */
		implicit = 0;
		np = getnodeproto(par[0]);

		/* if it is a new facet, create it */
		if (np == NONODEPROTO)
		{
			lib = el_curlib;
			for(pt = par[0]; *pt != 0; pt++) if (*pt == ':') break;
			if (*pt != ':') pt = par[0]; else
			{
				i = *pt;
				*pt = 0;
				lib = getlibrary(par[0]);
				*pt = (char)i;
				if (lib == NOLIBRARY)
				{
					us_abortcommand(_("Cannot find library for new facet %s"), par[0]);
					return;
				}
				pt++;
			}
			np = us_newnodeproto(pt, lib);
			if (np == NONODEPROTO)
			{
				us_abortcommand(_("Cannot create facet %s"), par[0]);
				return;
			}
			ttyputverbose(M_("Editing new facet: %s"), par[0]);
		}

		/* look through window for instances of this facet */
		stacknodeinst = NONODEINST;
		curfacet = getcurfacet();
		if (curfacet != NONODEPROTO)
		{
			for(ni = curfacet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
				if (ni->proto == np)
			{
				stacknodeinst = ni;
				break;
			}
		}
	}

	/* make sure nodeinst is not primitive and in proper technology */
	if (np->primindex != 0)
	{
		us_abortcommand(_("Cannot edit primitive nodes"));
		return;
	}

	push = 0;
	if (stacknodeinst != NONODEINST)
	{
		push = 1;
		onp = contentsview(np);
		if (onp == NONODEPROTO || intoicon != 0) onp = np;
		(void)setvalkey((INTBIG)onp, VNODEPROTO, us_descent_path_key, (INTBIG)stacknodeinst,
			VNODEINST|VDONTSAVE);

		/* save the current view */
		(void)initinfstr();
		(void)formatinfstr("%ld %ld %ld %ld", el_curwindowpart->screenlx,
			el_curwindowpart->screenhx, el_curwindowpart->screenly,
			el_curwindowpart->screenhy);
		transcpy(el_curwindowpart->intofacet, xfarray);
		(void)formatinfstr(" %ld %ld %ld %ld %ld %ld %ld %ld %ld",
			xfarray[0][0], xfarray[0][1], xfarray[0][2],
			xfarray[1][0], xfarray[1][1], xfarray[1][2],
			xfarray[2][0], xfarray[2][1], xfarray[2][2]);
		transcpy(el_curwindowpart->outoffacet, xfarray);
		(void)formatinfstr(" %ld %ld %ld %ld %ld %ld %ld %ld %ld",
			xfarray[0][0], xfarray[0][1], xfarray[0][2],
			xfarray[1][0], xfarray[1][1], xfarray[1][2],
			xfarray[2][0], xfarray[2][1], xfarray[2][2]);
		(void)setvalkey((INTBIG)onp, VNODEPROTO, us_descent_view_key, (INTBIG)returninfstr(),
			VSTRING|VDONTSAVE);
	}

	/* check dates of subfacets */
	if ((us_useroptions&CHECKDATE) != 0)
	{
		for(olib = el_curlib; olib != NOLIBRARY; olib = olib->nextlibrary)
			for(np1 = olib->firstnodeproto; np1 != NONODEPROTO; np1 = np1->nextnodeproto)
				np1->temp1 = 0;
		us_check_facet_date(np, np->revisiondate);
	}

	/* if a nonredundant display is needed, see if it already exists */
	if (nonredundant != 0)
	{
		for(win = el_topwindowpart; win != NOWINDOWPART; win = win->nextwindowpart)
			if (win->curnodeproto == np) break;
		if (win != NOWINDOWPART)
		{
			/* switch to window "win" */
			bringwindowtofront(win->frame);
			us_highlightwindow(win, FALSE);
			return;
		}
	}

	/* determine window area */
	if (el_curwindowpart == NOWINDOWPART)
	{
		lx = np->lowx;   hx = np->lowx;
		ly = np->lowy;   hy = np->lowy;
	} else
	{
		lx = el_curwindowpart->screenlx;   hx = el_curwindowpart->screenhx;
		ly = el_curwindowpart->screenly;   hy = el_curwindowpart->screenhy;
		if (el_curwindowpart->curnodeproto == NONODEPROTO)
		{
			lambda = el_curlib->lambda[el_curtech->techindex];
			lx = -lambda * 25;
			hx =  lambda * 25;
			ly = -lambda * 25;
			hy =  lambda * 25;
		}
	}
	if (implicit == 0)
	{
		/* make the new facet fill the window */
		us_fullview(np, &lx, &hx, &ly, &hy);
	} else
	{
		/* make the current facet be in the same place in the window */
		if (inplace != 0)
		{
			lx = el_curwindowpart->screenlx;   hx = el_curwindowpart->screenhx;
			ly = el_curwindowpart->screenly;   hy = el_curwindowpart->screenhy;
		} else
		{
			lx += np->lowx - stacknodeinst->lowx;
			hx += np->lowx - stacknodeinst->lowx;
			ly += np->lowy - stacknodeinst->lowy;
			hy += np->lowy - stacknodeinst->lowy;
		}
	}

	/* edit the facet (creating a new frame/partition if requested) */
	if (newwindow != 0 || el_curwindowpart == NOWINDOWPART)
	{
		newframe = TRUE;
		push = 0;
	} else
	{
		newframe = FALSE;
	}
	if (implicit != 0 && inplace != 0)
	{
		makerotI(stacknodeinst, rotarray);
		maketransI(stacknodeinst, transarray);
		transmult(rotarray, transarray, xfarray);
		transcpy(el_curwindowpart->intofacet, prevarray);
		transmult(prevarray, xfarray, el_curwindowpart->intofacet);

		makerot(stacknodeinst, rotarray);
		maketrans(stacknodeinst, transarray);
		transmult(transarray, rotarray, xfarray);
		transcpy(el_curwindowpart->outoffacet, prevarray);
		transmult(xfarray, prevarray, el_curwindowpart->outoffacet);
		el_curwindowpart->state |= INPLACEEDIT;
	} else
	{
		if (el_curwindowpart != NOWINDOWPART)
		{
			transid(el_curwindowpart->intofacet);
			transid(el_curwindowpart->outoffacet);
			el_curwindowpart->state &= ~INPLACEEDIT;
		}
	}
	us_switchtofacet(np, lx, hx, ly, hy, hini, hipp, newframe, push);
}

void us_erase(INTBIG count, char *par[])
{
	REGISTER INTBIG i, j, l;
	INTBIG textcount;
	char **textinfo;
	BOOLEAN allvisible, foundnode;
	REGISTER NODEINST *ni;
	ARCINST *ai;
	REGISTER NODEPROTO *np;
	HIGHLIGHT high;
	REGISTER char *pt;
	REGISTER GEOM **list;
	REGISTER LIBRARY *lib;
	REGISTER WINDOWPART *w;
	HIGHLIGHT newhigh;

	if (count > 0)
	{
		l = strlen(pt = par[0]);
		if (namesamen(pt, "clean-up-all", l) == 0 && l > 8)
		{
			us_clearhighlightcount();
			for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				us_cleanupfacet(np, FALSE);
			return;
		}
		if (namesamen(pt, "clean-up", l) == 0)
		{
			np = us_needfacet();
			if (np == NONODEPROTO) return;
			us_clearhighlightcount();
			us_cleanupfacet(np, TRUE);
			return;
		}
		if (namesamen(pt, "geometry", l) == 0)
		{
			np = us_needfacet();
			if (np == NONODEPROTO) return;
			us_erasegeometry(np);
			return;
		}
	}

#if 0
	np = us_needfacet();
	if (np == NONODEPROTO) return;
	highvar = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
	if (highvar == NOVARIABLE)
	{
		us_abortcommand(_("First highlight something to erase"));
		return;
	}
	(void)us_makehighlight(((char **)highvar->addr)[0], &high);

	/* special case if a text object is highlighted */
	if (getlength(highvar) == 1 && (high.status&HIGHTYPE) == HIGHTEXT)
	{
		/* disallow erasing if lock is on */
		if (us_cantedit(np, NONODEPROTO, TRUE)) return;

		/* deleting variable on object */
		if (high.fromvar != NOVARIABLE)
		{
			us_clearhighlightcount();
			if (high.fromgeom == NOGEOM)
			{
				us_undrawfacetvariable(high.fromvar, np);
				(void)delval((INTBIG)np, VNODEPROTO, makename(high.fromvar->key));
			} else if (high.fromgeom->entryisnode)
			{
				ni = high.fromgeom->entryaddr.ni;
				startobjectchange((INTBIG)ni, VNODEINST);

				/* if deleting port variables, do that */
				if (high.fromport != NOPORTPROTO)
				{
					(void)delval((INTBIG)high.fromport, VPORTPROTO, makename(high.fromvar->key));
				} else
				{
					/* if deleting text on invisible pin, delete pin too */
					if (ni->proto == gen_invispinprim) us_erasenodeinst(ni); else
						(void)delval((INTBIG)ni, VNODEINST, makename(high.fromvar->key));
				}
				endobjectchange((INTBIG)ni, VNODEINST);
			} else
			{
				ai = high.fromgeom->entryaddr.ai;
				startobjectchange((INTBIG)ai, VARCINST);
				(void)delval((INTBIG)ai, VARCINST, makename(high.fromvar->key));
				endobjectchange((INTBIG)ai, VARCINST);
			}
		} else if (high.fromport != NOPORTPROTO)
		{
			us_clearhighlightcount();
			ni = high.fromgeom->entryaddr.ni;
			startobjectchange((INTBIG)ni, VNODEINST);
			us_undoportproto((NODEINST *)ni, high.fromport);
			endobjectchange((INTBIG)ni, VNODEINST);
			high.status = HIGHFROM;
			(void)us_addhighlight(&high);
		} else if (high.fromgeom->entryisnode)
			us_abortcommand(_("Cannot delete facet name"));
		return;
	}
#endif

	/* get list of highlighted objects to be erased */
	np = us_needfacet();
	if (np == NONODEPROTO)
	{
		us_abortcommand(_("No current facet"));
		return;
	}
	list = us_gethighlighted(WANTARCINST|WANTNODEINST, &textcount, &textinfo);
	if (list[0] == NOGEOM && textcount == 0)
	{
		us_abortcommand(_("Find an object to erase"));
		return;
	}

	/* make sure that all requested objects are displayed */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			np->temp1 = 0;
	for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		if (w->curnodeproto != NONODEPROTO)
			w->curnodeproto->temp1 = 1;
	allvisible = TRUE;
	for(i=0; list[i] != NOGEOM; i++)
	{
		np = geomparent(list[i]);
		if (np->temp1 == 0) allvisible = FALSE;
	}
	for(i=0; i<textcount; i++)
	{
		(void)us_makehighlight(textinfo[i], &high);
		if (high.fromgeom == NOGEOM) continue;
		np = geomparent(high.fromgeom);
		if (np->temp1 == 0) allvisible = FALSE;
	}
	if (!allvisible)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(_("Some of the objects to be deleted are not displayed.  Delete anyway?"));
		(void)addstringtoinfstr(_(" (click 'no' to delete only visible objects, 'yes' to delete all selected)"));
		j = us_noyescanceldlog(returninfstr(), par);
		if (j == 0) return;
		if (namesame(par[0], "cancel") == 0) return;
		if (namesame(par[0], "no") == 0)
		{
			j = 0;
			for(i=0; list[i] != NOGEOM; i++)
			{
				np = geomparent(list[i]);
				if (np->temp1 == 0) continue;
				list[j++] = list[i];
			}
			if (j == 0)
			{
				us_abortcommand(_("No displayed objects to delete"));
				return;
			}
			list[j] = NOGEOM;
		}
	}

	/* remove from list if a node is locked */
	j = 0;
	foundnode = FALSE;
	for(i=0; list[i] != NOGEOM; i++)
	{
		if (list[i]->entryisnode)
		{
			ni = list[i]->entryaddr.ni;
			if (us_cantedit(np, ni->proto, TRUE)) continue;
			foundnode = TRUE;
		}
		list[j++] = list[i];
	}
	list[j] = NOGEOM;
	if (list[0] == NOGEOM && textcount == 0)
	{
		us_abortcommand(_("All selected objects are locked"));
		return;
	}

	if (!foundnode)
	{
		/* disallow erasing if lock is on */
		if (us_cantedit(np, NONODEPROTO, TRUE)) return;
	}

	/* unhighlight */
	us_clearhighlightcount();

	/* if one node is selected, see if it can be handled with reconnection */
	if (list[0] != NOGEOM && list[0]->entryisnode && list[1] == NOGEOM && textcount == 0)
	{
		j = us_erasepassthru(ni, FALSE, &ai);
		if (j == 2)
		{
			/* worked: highlight the arc */
			newhigh.status = HIGHFROM;
			newhigh.fromgeom = ai->geom;
			newhigh.facet = np;
			us_setfind(&newhigh, 0, 1, 0, 0);
			return;
		}
	}

	/* delete the text */
	for(i=0; i<textcount; i++)
	{
		(void)us_makehighlight(textinfo[i], &high);

		/* disallow erasing if lock is on */
		np = high.facet;
		if (np != NONODEPROTO)
		{
			if (us_cantedit(np, NONODEPROTO, TRUE)) continue;
		}

		/* do not deal with text on an object if the object is already in the list */
		if (high.fromgeom != NOGEOM)
		{
			for(j=0; list[j] != NOGEOM; j++)
				if (list[j] == high.fromgeom) break;
			if (list[j] != NOGEOM) continue;
		}

		/* deleting variable on object */
		if (high.fromvar != NOVARIABLE)
		{
			if (high.fromgeom == NOGEOM)
			{
				us_undrawfacetvariable(high.fromvar, np);
				(void)delval((INTBIG)np, VNODEPROTO, makename(high.fromvar->key));
			} else if (high.fromgeom->entryisnode)
			{
				ni = high.fromgeom->entryaddr.ni;
				startobjectchange((INTBIG)ni, VNODEINST);

				/* if deleting port variables, do that */
				if (high.fromport != NOPORTPROTO)
				{
					(void)delval((INTBIG)high.fromport, VPORTPROTO, makename(high.fromvar->key));
				} else
				{
					/* if deleting text on invisible pin, delete pin too */
					if (ni->proto == gen_invispinprim) us_erasenodeinst(ni); else
						(void)delval((INTBIG)ni, VNODEINST, makename(high.fromvar->key));
				}
				endobjectchange((INTBIG)ni, VNODEINST);
			} else
			{
				ai = high.fromgeom->entryaddr.ai;
				startobjectchange((INTBIG)ai, VARCINST);
				(void)delval((INTBIG)ai, VARCINST, makename(high.fromvar->key));
				endobjectchange((INTBIG)ai, VARCINST);
			}
		} else if (high.fromport != NOPORTPROTO)
		{
			ni = high.fromgeom->entryaddr.ni;
			startobjectchange((INTBIG)ni, VNODEINST);
			us_undoportproto((NODEINST *)ni, high.fromport);
			endobjectchange((INTBIG)ni, VNODEINST);
			high.status = HIGHFROM;
			(void)us_addhighlight(&high);
		} else if (high.fromgeom->entryisnode)
			us_abortcommand(_("Cannot delete facet name"));
	}

	/* look for option to re-connect arcs into an erased node */
	if (count > 0 && namesamen(par[0], "pass-through", strlen(par[0])) == 0)
	{
		for(i=0; list[i] != NOGEOM; i++)
		{
			if (!list[i]->entryisnode) continue;
			ni = list[i]->entryaddr.ni;
			j = us_erasepassthru(ni, TRUE, &ai);
			switch (j)
			{
				case 2:
					break;
				case -1:
					us_abortcommand(_("Arcs to node %s are of different type"),
						describenodeinst(ni));
					break;
				case -5:
					us_abortcommand(_("Cannot create connecting arc"));
					break;
				default:
					us_abortcommand(_("Must be 2 arcs on node %s (it has %ld)"),
						describenodeinst(ni), j);
					break;
			}
		}
		return;
	}

	/* handle simple erasing */
	us_eraseobjectsinlist(np, list);
}

void us_find(INTBIG count, char *par[])
{
	REGISTER INTBIG i, j, l, findport, findpoint, findexclusively, findangle,
		findwithin, findstill, findspecial, size,
		findeasy, findhard, total, type, addr, len, first;
	REGISTER INTBIG areasizex, areasizey, x, y, *newlist, extrainfo, findmore, findnobox;
	INTBIG xcur, ycur;
	XARRAY trans;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *np, *onp;
	REGISTER LIBRARY *lib;
	REGISTER GEOM *geom, **glist;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp;
	REGISTER NETWORK *net;
	REGISTER VARIABLE *var, *highvar;
	REGISTER char *pt;
	char **list;
	REGISTER HIGHLIGHT *high;
	HIGHLIGHT newhigh, newhightext;

	if (count >= 1)
	{
		l = strlen(pt = par[0]);
		if (namesamen(pt, "constraint-angle", l) == 0 && l >= 12)
		{
			if (count >= 2)
			{
				(void)setvalkey((INTBIG)us_tool, VTOOL, us_interactiveanglekey,
					atofr(par[1])*10/WHOLE, VINTEGER|VDONTSAVE);
			}
			var = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER, us_interactiveanglekey);
			if (var == NOVARIABLE) findangle = 0; else
				findangle = var->addr;
			if (findangle == 0)
			{
				ttyputverbose(M_("Interactive dragging done with no constraints"));
			} else
			{
				ttyputverbose(M_("Interactive dragging constrained to %ld degree angles"),
					findangle);
			}
			return;
		}
	}

	/* make sure there is a facet being edited */
	np = us_needfacet();
	if (np == NONODEPROTO) return;
	if ((np->cellview->viewstate&TEXTVIEW) != 0)
	{
		l = strlen(pt = par[0]);
		if (namesamen(pt, "all", l) == 0 && l >= 2)
		{
			/* special case: "find all" selects all text */
			i = us_totallines(el_curwindowpart);
			us_highlightline(el_curwindowpart, 0, i-1);
			return;
		}
		us_abortcommand(M_("There are no components to select in a text-only facet"));
		return;
	}
	if ((el_curwindowpart->state&WINDOWTYPE) == DISP3DWINDOW)
	{
		us_abortcommand(M_("Cannot select objects in a 3D window"));
		return;
	}

	/* establish the default highlight environment */
	highvar = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
	if (highvar != NOVARIABLE)
		(void)us_makehighlight(((char **)highvar->addr)[0], &newhigh); else
	{
		newhigh.status = 0;
		newhigh.fromgeom = NOGEOM;
		newhigh.fromport = NOPORTPROTO;
		newhigh.fromvar = NOVARIABLE;
		newhigh.fromvarnoeval = NOVARIABLE;
		newhigh.frompoint = 0;
	}

	/* look for qualifiers */
	findport = findpoint = findexclusively = findwithin = findmore = 0;
	findstill = findspecial = findnobox = 0;
	extrainfo = 0;
	while (count > 0)
	{
		l = strlen(pt = par[0]);
		if (namesamen(pt, "extra-info", l) == 0 && l >= 3)
		{
			extrainfo = HIGHEXTRA;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "more", l) == 0 && l >= 1)
		{
			findmore++;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "no-box", l) == 0 && l >= 3)
		{
			findnobox++;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "special", l) == 0 && l >= 2)
		{
			findspecial++;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "still", l) == 0 && l >= 2)
		{
			findstill++;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "exclusively", l) == 0 && l >= 3)
		{
			findexclusively++;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "within", l) == 0 && l >= 1)
		{
			if (newhigh.status == 0)
			{
				us_abortcommand(M_("Find an object before working 'within' it"));
				return;
			}
			if ((newhigh.status&HIGHTYPE) != HIGHFROM || !newhigh.fromgeom->entryisnode)
			{
				us_abortcommand(M_("Must find a node before working 'within'"));
				return;
			}

			findwithin++;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "port", l) == 0 && l >= 1)
		{
			findport = 1;
			count--;   par++;   continue;
		}
		if (namesamen(pt, "vertex", l) == 0 && l >= 2)
		{
			findpoint = 1;
			count--;   par++;   continue;
		}
		break;
	}

	if (count >= 1)
	{
		l = strlen(pt = par[0]);

		if (namesamen(pt, "all", l) == 0 && l >= 2)
		{
			if (findpoint != 0 || findwithin != 0 || findmore != 0 ||
				findexclusively != 0 || findstill != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find all'"));
				return;
			}
			findeasy = findhard = 1;
			if (count >= 2)
			{
				l = strlen(pt = par[1]);
				if (namesamen(pt, "easy", l) == 0) findhard = 0; else
				if (namesamen(pt, "hard", l) == 0) findeasy = 0; else
				{
					ttyputusage("find all [easy|hard]");
					return;
				}
			}
			total = 0;
			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst) total++;
			for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst) total++;
			for(i=0; i<np->numvar; i++)
			{
				var = &np->firstvar[i];
				if ((var->type&VDISPLAY) != 0) total++;
			}
			if (total == 0) return;
			list = (char **)emalloc((total * (sizeof (char *))), el_tempcluster);
			if (list == 0) return;

			newhigh.status = HIGHFROM | extrainfo;
			if (findnobox != 0) newhigh.status |= HIGHNOBOX;
			newhigh.facet = np;
			newhigh.frompoint = 0;
			newhigh.fromvar = NOVARIABLE;
			newhigh.fromvarnoeval = NOVARIABLE;

			newhightext.status = HIGHTEXT;
			newhightext.facet = np;
			newhightext.fromport = NOPORTPROTO;
			newhightext.frompoint = 0;
			newhightext.fromvarnoeval = NOVARIABLE;

			total = 0;
			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			{
				if (ni->proto->primindex == 0 && (us_useroptions&NOINSTANCESELECT) != 0)
				{
					/* facet instance that is hard to select */
					if (findhard == 0) continue;
				} else
				{
					/* regular node: see if it should be selected */
					if (findeasy == 0 && (ni->userbits&HARDSELECTN) == 0) continue;
					if (findhard == 0 && (ni->userbits&HARDSELECTN) != 0) continue;
				}
				if (ni->proto->primindex != 0 && (ni->proto->userbits&NINVISIBLE) != 0)
					continue;

				/* if this is an invisible primitive with text, select the text */
				if (ni->proto == gen_invispinprim)
				{
					j = 0;
					for(i=0; i<ni->numvar; i++)
					{
						var = &ni->firstvar[i];
						if ((var->type&VDISPLAY) == 0) continue;
						newhightext.fromgeom = ni->geom;
						newhightext.fromvar = var;
						pt = us_makehighlightstring(&newhightext);
						(void)allocstring(&list[total], pt, el_tempcluster);
						total++;
						j = 1;
					}
					if (j != 0) continue;
				}

				newhigh.fromgeom = ni->geom;
				newhigh.fromport = NOPORTPROTO;
				if (findport != 0)
				{
					pp = ni->proto->firstportproto;
					if (pp != NOPORTPROTO && pp->nextportproto == NOPORTPROTO)
						newhigh.fromport = pp;
				}
				pt = us_makehighlightstring(&newhigh);
				(void)allocstring(&list[total], pt, el_tempcluster);
				total++;
			}
			for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			{
				if (findeasy == 0 && (ai->userbits&HARDSELECTA) == 0) continue;
				if (findhard == 0 && (ai->userbits&HARDSELECTA) != 0) continue;
				if ((ai->proto->userbits&AINVISIBLE) != 0) continue;
				newhigh.fromgeom = ai->geom;
				newhigh.fromport = NOPORTPROTO;
				pt = us_makehighlightstring(&newhigh);
				(void)allocstring(&list[total], pt, el_tempcluster);
				total++;
			}
			if (findeasy != 0)
			{
				for(i=0; i<np->numvar; i++)
				{
					var = &np->firstvar[i];
					if ((var->type&VDISPLAY) == 0) continue;
					newhightext.fromgeom = NOGEOM;
					newhightext.fromvar = var;
					pt = us_makehighlightstring(&newhightext);
					(void)allocstring(&list[total], pt, el_tempcluster);
					total++;
				}
			}
			if (total == 0) us_clearhighlightcount(); else
			{
				(void)setvalkey((INTBIG)us_tool, VTOOL, us_highlightedkey, (INTBIG)list,
					VSTRING|VISARRAY|(total<<VLENGTHSH)|VDONTSAVE);
			}
			for(i=0; i<total; i++) efree(list[i]);
			efree((char *)list);
			us_showallhighlight();
			if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO);
			return;
		}

		if (namesamen(pt, "arc", l) == 0 && l >= 3)
		{
			if (count <= 1)
			{
				ttyputusage("find arc ARCNAME");
				return;
			}
			if (findport != 0 || findpoint != 0 || findwithin != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find arc'"));
				return;
			}
			if (findexclusively != 0 && us_curarcproto == NOARCPROTO)
			{
				us_abortcommand(M_("Must select an arc for exclusive finding"));
				return;
			}
			for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			{
				if (findexclusively != 0 && ai->proto != us_curarcproto) continue;
				var = getvalkey((INTBIG)ai, VARCINST, VSTRING, el_arc_name_key);
				if (var == NOVARIABLE) continue;
				if (namesame((char *)var->addr, par[1]) == 0) break;
			}
			if (ai == NOARCINST)
			{
				us_abortcommand(_("Sorry, no %s arc named '%s' in this facet"),
					(findexclusively==0 ? "" : describearcproto(us_curarcproto)), par[1]);
				return;
			}
			newhigh.status = HIGHFROM;
			newhigh.fromgeom = ai->geom;
			newhigh.facet = np;
			us_setfind(&newhigh, 0, extrainfo, findmore, findnobox);
			return;
		}

		if (namesamen(pt, "area-define", l) == 0 && l >= 6)
		{
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findnobox != 0 ||
				findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find area-define'"));
				return;
			}

			us_clearhighlightcount();
			trackcursor(FALSE, us_ignoreup, us_finddbegin, us_stretchdown, us_stoponchar,
				us_invertdragup, TRACKDRAGGING);
			if (el_pleasestop != 0) return;
			us_finddterm(&newhigh.stalx, &newhigh.staly);
			if (us_demandxy(&xcur, &ycur)) return;
			if (xcur >= newhigh.stalx) newhigh.stahx = xcur; else
			{
				newhigh.stahx = newhigh.stalx;   newhigh.stalx = xcur;
			}
			if (ycur >= newhigh.staly) newhigh.stahy = ycur; else
			{
				newhigh.stahy = newhigh.staly;   newhigh.staly = ycur;
			}
			newhigh.status = HIGHBBOX;
			newhigh.facet = np;
			us_addhighlight(&newhigh);
			return;
		}

		if (namesamen(pt, "area-move", l) == 0 && l >= 6)
		{
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findnobox != 0 ||
				findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find area-move'"));
				return;
			}
			if (us_demandxy(&xcur, &ycur)) return;
			gridalign(&xcur, &ycur, 1);

			/* set the highlight */
			if ((newhigh.status&HIGHTYPE) == HIGHBBOX)
			{
				areasizex = newhigh.stahx - newhigh.stalx;
				areasizey = newhigh.stahy - newhigh.staly;
			} else
			{
				areasizex = (el_curwindowpart->screenhx-el_curwindowpart->screenlx) / 5;
				areasizey = (el_curwindowpart->screenhy-el_curwindowpart->screenly) / 5;
			}

			us_clearhighlightcount();

			/* adjust the cursor position if selecting interactively */
			if ((us_tool->toolstate&INTERACTIVE) != 0)
			{
				us_findinit(areasizex, areasizey);
				trackcursor(FALSE, us_ignoreup, us_findmbegin, us_dragdown, us_stoponchar,
					us_dragup, TRACKDRAGGING);
				if (el_pleasestop != 0) return;
				if (us_demandxy(&xcur, &ycur)) return;
				gridalign(&xcur, &ycur, 1);
			}
			newhigh.status = HIGHBBOX;
			newhigh.facet = np;
			newhigh.stalx = xcur;   newhigh.stahx = xcur + areasizex;
			newhigh.staly = ycur;   newhigh.stahy = ycur + areasizey;
			us_addhighlight(&newhigh);
			return;
		}

		if (namesamen(pt, "area-size", l) == 0 && l >= 6)
		{
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findnobox != 0 ||
				findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find area-size'"));
				return;
			}
			if (us_demandxy(&xcur, &ycur)) return;
			gridalign(&xcur, &ycur, 1);

			if (newhigh.status != HIGHBBOX)
			{
				us_abortcommand(M_("Use 'find area-move' first, then this"));
				return;
			}
			if (np != newhigh.facet)
			{
				us_abortcommand(M_("Not in same facet as highlight area"));
				return;
			}

			us_clearhighlightcount();

			/* adjust the cursor position if selecting interactively */
			if ((us_tool->toolstate&INTERACTIVE) != 0)
			{
				us_findinit(newhigh.stalx, newhigh.staly);
				trackcursor(FALSE, us_ignoreup, us_findsbegin, us_stretchdown,
					us_stoponchar, us_invertdragup, TRACKDRAGGING);
				if (el_pleasestop != 0) return;
				if (us_demandxy(&xcur, &ycur)) return;
				gridalign(&xcur, &ycur, 1);
			}
			if (xcur >= newhigh.stalx) newhigh.stahx = xcur; else
			{
				newhigh.stahx = newhigh.stalx;   newhigh.stalx = xcur;
			}
			if (ycur >= newhigh.staly) newhigh.stahy = ycur; else
			{
				newhigh.stahy = newhigh.staly;   newhigh.staly = ycur;
			}
			us_addhighlight(&newhigh);
			return;
		}

		if (namesamen(pt, "clear", l) == 0 && l >= 2)
		{
			us_clearhighlightcount();
			return;
		}

		if (namesamen(pt, "comp-interactive", l) == 0 && l >= 3)
		{
			if (findpoint != 0 || findexclusively != 0 || extrainfo != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find comp-interactive'"));
				return;
			}
			var = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER, us_interactiveanglekey);
			if (var == NOVARIABLE) findangle = 0; else
				findangle = var->addr;
			us_findiinit(findport, extrainfo, findangle, 1-findmore, findstill,
				findnobox, findspecial);
			trackcursor(FALSE, us_ignoreup, us_findcibegin, us_findidown, us_stoponchar,
				us_findiup, TRACKDRAGGING);
			return;
		}

		if (namesamen(pt, "deselect-arcs", l) == 0 && l >= 2)
		{
			var = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
			if (var == NOVARIABLE) return;
			len = getlength(var);
			list = (char **)emalloc(len * (sizeof (char *)), el_tempcluster);
			if (list == 0) return;
			j = 0;
			for(i=0; i<len; i++)
			{
				if (us_makehighlight(((char **)var->addr)[i], &newhigh)) break;
				if ((newhigh.status&HIGHTYPE) == HIGHFROM &&
					!newhigh.fromgeom->entryisnode) continue;
				(void)allocstring(&list[j], ((char **)var->addr)[i], el_tempcluster);
				j++;
			}
			if (j == 0)
			{
				(void)delvalkey((INTBIG)us_tool, VTOOL, us_highlightedkey);
			} else
			{
				(void)setvalkey((INTBIG)us_tool, VTOOL, us_highlightedkey, (INTBIG)list,
					VSTRING|VISARRAY|(j<<VLENGTHSH)|VDONTSAVE);
			}
			for(i=0; i<j; i++)
				efree((char *)list[i]);
			efree((char *)list);
			return;
		}

		if (namesamen(pt, "down-stack", l) == 0 && l >= 2)
		{
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find down-stack'"));
				return;
			}
			us_pushhighlight();
			ttyputverbose(M_("Pushed"));
			return;
		}

		if (namesamen(pt, "dragging-selects", l) == 0 && l >= 2)
		{
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find dragging-selects'"));
				return;
			}
			if (count <= 1)
			{
				ttyputusage("find dragging-selects WHAT");
				return;
			}
			len = strlen(pt = par[1]);
			if (namesamen(pt, "any-touching", len) == 0)
			{
				(void)setvalkey((INTBIG)us_tool, VTOOL, us_optionflagskey,
					us_useroptions & ~MUSTENCLOSEALL, VINTEGER);
				ttyputverbose(M_("Dragging selects anything touching the area"));
				return;
			}
			if (namesamen(pt, "only-enclosed", len) == 0)
			{
				(void)setvalkey((INTBIG)us_tool, VTOOL, us_optionflagskey,
					us_useroptions | MUSTENCLOSEALL, VINTEGER);
				ttyputverbose(M_("Dragging selects only objects inside the area"));
				return;
			}
			ttyputbadusage("find dragging-selects");
			return;
		}

		if (namesamen(pt, "export", l) == 0 && l >= 3)
		{
			if (count <= 1)
			{
				ttyputusage("find export PORTNAME");
				return;
			}
			if (findport != 0 || findpoint != 0 || findexclusively != 0 || findwithin != 0 ||
				findspecial != 0)
			{
				us_abortcommand(M_("'find export' cannot accept other control"));
				return;
			}
			for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
				if (namesame(par[1], pp->protoname) == 0) break;
			if (pp == NOPORTPROTO)
			{
				us_abortcommand(_("Sorry, no export named '%s' in this facet"), par[1]);
				return;
			}
			newhigh.status = HIGHFROM;
			newhigh.fromgeom = pp->subnodeinst->geom;
			newhigh.fromport = pp->subportproto;
			newhigh.facet = np;
			us_setfind(&newhigh, 0, extrainfo, findmore, findnobox);
			return;
		}

		if (namesamen(pt, "interactive", l) == 0)
		{
			if (findexclusively != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find interactively'"));
				return;
			}

			/* special case: "find within vertex interactive" for polygon-editing */
			if (findpoint != 0 && findwithin != 0)
			{
				ni = newhigh.fromgeom->entryaddr.ni;
				us_pointinit(ni, 0);
				trackcursor(FALSE, us_ignoreup, us_findpointbegin, us_movepdown,
					us_stoponchar, us_dragup, TRACKDRAGGING);
				if (el_pleasestop != 0) return;

				highvar = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
				if (highvar == NOVARIABLE) return;
				(void)us_makehighlight(((char **)highvar->addr)[0], &newhigh);
				if (us_demandxy(&xcur, &ycur)) return;
				gridalign(&xcur, &ycur, 1);
				var = gettrace(ni);
				if (var == NOVARIABLE) return;
				size = getlength(var) / 2;
				newlist = (INTBIG *)emalloc((size*2*SIZEOFINTBIG), el_tempcluster);
				if (newlist == 0) return;
				makerot(ni, trans);
				x = (ni->highx + ni->lowx) / 2;
				y = (ni->highy + ni->lowy) / 2;
				for(i=0; i<size; i++)
				{
					if (i+1 == newhigh.frompoint)
					{
						newlist[i*2] = xcur;
						newlist[i*2+1] = ycur;
					} else xform(((INTBIG *)var->addr)[i*2]+x, ((INTBIG *)var->addr)[i*2+1]+y,
						&newlist[i*2], &newlist[i*2+1], trans);
				}

				/* now re-draw this trace */
				us_pushhighlight();
				us_clearhighlightcount();
				us_settrace(ni, newlist, size);
				us_pophighlight(FALSE);
				efree((char *)newlist);
				return;
			}

			/* traditional interactive selection */
			var = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER, us_interactiveanglekey);
			if (var == NOVARIABLE) findangle = 0; else
				findangle = var->addr;
			us_findiinit(findport, extrainfo, findangle, 1-findmore, findstill,
				findnobox, findspecial);
			trackcursor(FALSE, us_ignoreup, us_findibegin, us_findidown, us_stoponchar,
				us_findiup, TRACKDRAGGING);
			return;
		}

		if (namesamen(pt, "just-objects", l) == 0)
		{
			/* must be a single "area select" */
			var = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
			if (var == NOVARIABLE) return;
			len = getlength(var);
			if (len != 1) return;
			(void)us_makehighlight(((char **)var->addr)[0], &newhigh);
			if ((newhigh.status&HIGHTYPE) != HIGHBBOX) return;

			/* remove it and select everything in that area */
			np = newhigh.facet;
			us_clearhighlightcount();
			total = us_selectarea(np, newhigh.stalx, newhigh.stahx, newhigh.staly,
				newhigh.stahy, 0, 0, 0, &list);
			if (total > 0)
				(void)setvalkey((INTBIG)us_tool, VTOOL, us_highlightedkey, (INTBIG)list,
					VSTRING|VISARRAY|(total<<VLENGTHSH)|VDONTSAVE);
			for(i=0; i<total; i++) efree(list[i]);
			if (total > 0) efree((char *)list);
			return;
		}

		if (namesamen(pt, "name", l) == 0 && l >= 2)
		{
			if (count <= 1)
			{
				ttyputusage("find name HIGHLIGHTNAME");
				return;
			}
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				findwithin != 0 || findmore != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find name'"));
				return;
			}
			(void)initinfstr();
			(void)addstringtoinfstr("USER_highlight_");
			(void)addstringtoinfstr(par[1]);
			var = getval((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, returninfstr());
			if (var == NOVARIABLE)
			{
				us_abortcommand(_("Cannot find saved highlight '%s'"), par[1]);
				return;
			}

			(void)setvalkey((INTBIG)us_tool, VTOOL, us_highlightedkey, var->addr, var->type|VDONTSAVE);
			if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO);
			return;
		}

		if (namesamen(pt, "node", l) == 0 && l >= 3)
		{
			if (count <= 1)
			{
				ttyputusage("find node NODENAME");
				return;
			}
			if (findport != 0 || findwithin != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find node'"));
				return;
			}
			if (findexclusively != 0 && us_curnodeproto == NONODEPROTO)
			{
				us_abortcommand(M_("Must select a node for exclusive finding"));
				return;
			}
			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			{
				if (findexclusively != 0 && ni->proto != us_curnodeproto) continue;
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name_key);
				if (var == NOVARIABLE) continue;
				if (namesame((char *)var->addr, par[1]) == 0) break;
			}
			if (ni == NONODEINST)
			{
				us_abortcommand(_("Sorry, no %s node named '%s' in this facet"),
					(findexclusively==0 ? "" : describenodeproto(us_curnodeproto)), par[1]);
				return;
			}
			newhigh.status = HIGHFROM;
			newhigh.fromgeom = ni->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.facet = np;
			us_setfind(&newhigh, findpoint, extrainfo, findmore, findnobox);
			return;
		}

		if (namesamen(pt, "nonmanhattan", l) == 0 && l >= 3)
		{
			i = 0;
			for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
				for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
					onp->temp1 = 0;
			for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			{
				for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
				{
					for(ai = onp->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
					{
						if (ai->proto->tech == gen_tech || ai->proto->tech == art_tech ||
							ai->proto->tech == sch_tech)
								continue;
						var = getvalkey((INTBIG)ai, VARCINST, VINTEGER, el_arc_radius_key);
						if (var != NOVARIABLE || (ai->end[0].xpos != ai->end[1].xpos &&
							ai->end[0].ypos != ai->end[1].ypos)) onp->temp1++;
					}
					for(ni = onp->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
					{
						if ((ni->rotation % 900) != 0) onp->temp1++;
					}
				}
			}
			for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			{
				if (ai->proto->tech == gen_tech || ai->proto->tech == art_tech)
					continue;
				var = getvalkey((INTBIG)ai, VARCINST, VINTEGER, el_arc_radius_key);
				if (var == NOVARIABLE && (ai->end[0].xpos == ai->end[1].xpos ||
					ai->end[0].ypos == ai->end[1].ypos)) continue;
				if (i == 0) us_clearhighlightcount();
				newhigh.status = HIGHFROM;
				newhigh.facet = np;
				newhigh.fromgeom = ai->geom;
				newhigh.fromport = NOPORTPROTO;
				newhigh.frompoint = 0;
				us_addhighlight(&newhigh);
				i++;
			}
			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			{
				if ((ni->rotation % 900) == 0) continue;
				if (i == 0) us_clearhighlightcount();
				newhigh.status = HIGHFROM;
				newhigh.facet = np;
				newhigh.fromgeom = ni->geom;
				newhigh.fromport = NOPORTPROTO;
				newhigh.frompoint = 0;
				us_addhighlight(&newhigh);
				i++;
			}
			if (i == 0) ttyputmsg(_("No nonmanhattan objects in this facet")); else
				ttyputmsg(_("%ld objects are not manhattan in this facet"), i);
			for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			{
				if ((lib->userbits&HIDDENLIBRARY) != 0) continue;
				i = 0;
				for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
					if (onp != np) i += onp->temp1;
				if (i == 0) continue;
				if (lib == el_curlib)
				{
					l = 0;
					(void)initinfstr();
					for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
					{
						if (onp == np || onp->temp1 == 0) continue;
						if (l != 0) (void)addtoinfstr(' ');
						(void)addstringtoinfstr(describenodeproto(onp));
						l++;
					}
					if (l == 1)
					{
						ttyputmsg(_("Found nonmanhattan geometry in facet %s"), returninfstr());
					} else
					{
						ttyputmsg(_("Found nonmanhattan geometry in these facets: %s"),
							returninfstr());
					}
				} else
				{
					ttyputmsg(_("Found nonmanhattan geometry in library %s"), lib->libname);
				}
			}
			return;
		}

		if (namesamen(pt, "object", l) == 0)
		{
			if (count < 2)
			{
				ttyputusage("find object (TYPE ADDRESS | TYPEADDRESS)");
				return;
			}
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				findwithin != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find object'"));
				return;
			}

			/* determine type and address to highlight */
			if (count == 3)
			{
				type = us_variabletypevalue(par[1]);
				addr = myatoi(par[2]);
			} else
			{
				type = VUNKNOWN;
				if (namesamen(par[1], "node", 4) == 0)
				{
					type = VNODEINST;
					addr = myatoi(&par[1][4]);
				}
				if (namesamen(par[1], "arc", 3) == 0)
				{
					type = VARCINST;
					addr = myatoi(&par[1][3]);
				}
				if (namesamen(par[1], "port", 4) == 0)
				{
					type = VPORTPROTO;
					addr = myatoi(&par[1][4]);
				}
				if (namesamen(par[1], "network", 7) == 0)
				{
					type = VNETWORK;
					addr = myatoi(&par[1][7]);
				}
			}
			if (type == VUNKNOWN)
			{
				us_abortcommand(_("Unknown object type in 'find object' command"));
				return;
			}
			switch (type)
			{
				case VNODEINST:
					ni = (NODEINST *)addr;
					if (ni == 0 || ni == NONODEINST) return;
					if (ni->parent != np)
					{
						us_abortcommand(_("Cannot find node %ld in this facet"), addr);
						return;
					}
					newhigh.status = HIGHFROM;
					newhigh.fromgeom = ni->geom;
					newhigh.fromport = NOPORTPROTO;
					newhigh.facet = np;
					us_setfind(&newhigh, findpoint, extrainfo, findmore, findnobox);
					break;
				case VARCINST:
					ai = (ARCINST *)addr;
					if (ai == 0 || ai == NOARCINST) return;
					if (ai->parent != np)
					{
						us_abortcommand(_("Cannot find arc %ld in this facet"), addr);
						return;
					}
					newhigh.status = HIGHFROM;
					newhigh.fromgeom = ai->geom;
					newhigh.fromport = NOPORTPROTO;
					newhigh.facet = np;
					us_setfind(&newhigh, findpoint, extrainfo, findmore, findnobox);
					break;
				case VNETWORK:
					net = (NETWORK *)addr;
					if (net == 0 || net == NONETWORK) return;
					if (net->parent != np)
					{
						us_abortcommand(_("Cannot find network %ld in this facet"), addr);
						return;
					}
					for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
					{
						if (ai->network != net) continue;
						newhigh.status = HIGHFROM;
						newhigh.fromgeom = ai->geom;
						newhigh.fromport = NOPORTPROTO;
						newhigh.facet = np;
						us_setfind(&newhigh, findpoint, extrainfo, findmore, findnobox);
					}
					for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					{
						if (pp->network != net) continue;
						newhigh.status = HIGHFROM;
						newhigh.fromgeom = pp->subnodeinst->geom;
						newhigh.fromport = pp->subportproto;
						newhigh.facet = np;
						us_setfind(&newhigh, 0, extrainfo, findmore, findnobox);
					}
					break;
				case VPORTPROTO:
					pp = (PORTPROTO *)addr;
					if (pp == 0 || pp == NOPORTPROTO) return;
					if (pp->parent != np)
					{
						us_abortcommand(_("Cannot find port %ld in this facet"), addr);
						return;
					}
					newhigh.status = HIGHFROM;
					newhigh.fromgeom = pp->subnodeinst->geom;
					newhigh.fromport = pp->subportproto;
					newhigh.facet = np;
					us_setfind(&newhigh, 0, extrainfo, findmore, findnobox);
					break;
				default:
					us_abortcommand(_("Cannot highlight objects of type %s"),
						us_variabletypename(type));
					return;
			}
			return;
		}

		if (namesamen(pt, "save", l) == 0 && l >= 2)
		{
			if (count <= 1)
			{
				ttyputusage("find save HIGHLIGHTNAME");
				return;
			}
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find save'"));
				return;
			}
			var = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
			if (var == NOVARIABLE)
			{
				us_abortcommand(_("Highlight something before saving highlight"));
				return;
			}
			(void)initinfstr();
			(void)addstringtoinfstr("USER_highlight_");
			(void)addstringtoinfstr(par[1]);
			(void)setval((INTBIG)us_tool, VTOOL, returninfstr(), var->addr, var->type|VDONTSAVE);
			ttyputverbose(M_("%s saved"), par[1]);
			return;
		}

		if (namesamen(pt, "set-easy-selection", l) == 0 && l >= 5)
		{
			glist = us_gethighlighted(WANTARCINST|WANTNODEINST, 0, 0);
			if (glist[0] == NOGEOM)
			{
				us_abortcommand(_("Select something before making it easy-to-select"));
				return;
			}
			for(i=0; glist[i] != NOGEOM; i++)
			{
				geom = glist[i];
				if (!geom->entryisnode)
				{
					ai = geom->entryaddr.ai;
					ai->userbits &= ~HARDSELECTA;
				} else
				{
					ni = geom->entryaddr.ni;
					ni->userbits &= ~HARDSELECTN;
				}
			}
			return;
		}

		if (namesamen(pt, "set-hard-selection", l) == 0 && l >= 5)
		{
			glist = us_gethighlighted(WANTARCINST|WANTNODEINST, 0, 0);
			if (glist[0] == NOGEOM)
			{
				us_abortcommand(_("Select something before making it easy-to-select"));
				return;
			}
			for(i=0; glist[i] != NOGEOM; i++)
			{
				geom = glist[i];
				if (!geom->entryisnode)
				{
					ai = geom->entryaddr.ai;
					ai->userbits |= HARDSELECTA;
				} else
				{
					ni = geom->entryaddr.ni;
					ni->userbits |= HARDSELECTN;
				}
			}
			return;
		}

		if (namesamen(pt, "similar", l) == 0 && l >= 2)
		{
			if (findpoint != 0 || findwithin != 0 || findmore != 0 ||
				findexclusively != 0 || findstill != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find similar'"));
				return;
			}

			high = us_getonehighlight();
			if (high == NOHIGHLIGHT)
			{
				us_abortcommand(_("Must select a single object before selecting all like it"));
				return;
			}
			if ((high->status&HIGHTYPE) != HIGHFROM)
			{
				us_abortcommand(_("Must select a node or arc before selecting all like it"));
				return;
			}
			(void)initinfstr();
			first = 0;
			if (high->fromgeom->entryisnode)
			{
				for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
				{
					if (ni->proto != high->fromgeom->entryaddr.ni->proto) continue;
					if (first != 0) (void)addtoinfstr('\n');
					first++;
					(void)formatinfstr("FACET=%s FROM=0%lo;-1;0",
						describenodeproto(np), (INTBIG)ni->geom);
				}
			} else
			{
				for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
				{
					if (ai->proto != high->fromgeom->entryaddr.ai->proto) continue;
					if (first != 0) (void)addtoinfstr('\n');
					first++;
					(void)formatinfstr("FACET=%s FROM=0%lo;-1;0",
						describenodeproto(np), (INTBIG)ai->geom);
				}
			}
			us_setmultiplehighlight(returninfstr(), FALSE);
			return;
		}

		if (namesamen(pt, "snap-mode", l) == 0 && l >= 2)
		{
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find snap-mode'"));
				return;
			}

			if (count < 2)
			{
				switch (us_state&SNAPMODE)
				{
					case SNAPMODENONE:     ttyputmsg(M_("Snapping mode: none"));              break;
					case SNAPMODECENTER:   ttyputmsg(M_("Snapping mode: center"));            break;
					case SNAPMODEMIDPOINT: ttyputmsg(M_("Snapping mode: midpoint"));          break;
					case SNAPMODEENDPOINT: ttyputmsg(M_("Snapping mode: end point"));         break;
					case SNAPMODETANGENT:  ttyputmsg(M_("Snapping mode: tangent"));           break;
					case SNAPMODEPERP:     ttyputmsg(M_("Snapping mode: perpendicular"));     break;
					case SNAPMODEQUAD:     ttyputmsg(M_("Snapping mode: quadrant"));          break;
					case SNAPMODEINTER:    ttyputmsg(M_("Snapping mode: any intersection"));  break;
				}
				return;
			}
			l = strlen(pt = par[1]);
			if (namesamen(pt, "none", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODENONE;
				ttyputverbose(M_("Snapping mode: none"));
				return;
			}
			if (namesamen(pt, "center", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODECENTER;
				ttyputverbose(M_("Snapping mode: center"));
				return;
			}
			if (namesamen(pt, "midpoint", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODEMIDPOINT;
				ttyputverbose(M_("Snapping mode: midpoint"));
				return;
			}
			if (namesamen(pt, "endpoint", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODEENDPOINT;
				ttyputverbose(M_("Snapping mode: endpoint"));
				return;
			}
			if (namesamen(pt, "tangent", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODETANGENT;
				ttyputverbose(M_("Snapping mode: tangent"));
				return;
			}
			if (namesamen(pt, "perpendicular", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODEPERP;
				ttyputverbose(M_("Snapping mode: perpendicular"));
				return;
			}
			if (namesamen(pt, "quadrant", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODEQUAD;
				ttyputverbose(M_("Snapping mode: quadrant"));
				return;
			}
			if (namesamen(pt, "intersection", l) == 0)
			{
				us_state = (us_state & ~SNAPMODE) | SNAPMODEINTER;
				ttyputverbose(M_("Snapping mode: any intersection"));
				return;
			}
			us_abortcommand(M_("Unknown snapping mode: %s"), pt);
			return;
		}

		if (namesamen(pt, "up-stack", l) == 0 && l >= 1)
		{
			if (findport != 0 || findpoint != 0 || findexclusively != 0 ||
				extrainfo != 0 || findwithin != 0 || findmore != 0 || findspecial != 0)
			{
				us_abortcommand(M_("Illegal options given to 'find up-stack'"));
				return;
			}
			us_pophighlight(FALSE);
			return;
		}

		if (namesamen(pt, "variable", l) == 0 && l >= 2)
		{
			if (count <= 1)
			{
				ttyputusage("find variable VARNAME");
				return;
			}
			if (findport != 0 || findpoint != 0 || findexclusively != 0 || findwithin != 0 ||
				findspecial != 0)
			{
				us_abortcommand(M_("'find variable' cannot accept other control"));
				return;
			}
			ni = (NODEINST *)us_getobject(VNODEINST, TRUE);
			if (ni == NONODEINST) return;
			for(i=0; i<ni->numvar; i++)
			{
				var = &ni->firstvar[i];
				if ((var->type&VDISPLAY) == 0) continue;
				if (namesame(par[1], makename(var->key)) == 0) break;
			}
			if (i >= ni->numvar)
			{
				us_abortcommand(_("Sorry, no variable named '%s' on the current node"),
					par[1]);
				return;
			}
			newhigh.status = HIGHTEXT;
			newhigh.fromgeom = ni->geom;
			newhigh.fromport = NOPORTPROTO;
			newhigh.fromvar = var;
			newhigh.facet = np;
			us_setfind(&newhigh, 0, extrainfo, findmore, findnobox);
			return;
		}
		ttyputbadusage("find");
		return;
	}

	/* get the cursor co-ordinates */
	if (us_demandxy(&xcur, &ycur)) return;

	/* find the closest object to the cursor */
	if (findwithin == 0)
		us_findobject(xcur, ycur, el_curwindowpart, &newhigh, findexclusively, 0, findport, 0, findspecial);
	if (newhigh.status == 0) return;
	if (findport == 0) newhigh.fromport = NOPORTPROTO;
	newhigh.facet = np;
	us_setfind(&newhigh, findpoint, extrainfo, findmore, findnobox);
}

void us_getproto(INTBIG count, char *par[])
{
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap, *lat;
	REGISTER INTBIG cellcount, facetcount, doarc, i, l, firstinpass;
	BOOLEAN butstate;
	REGISTER char *pp;
	REGISTER CELL *cell;
	HIGHLIGHT high;
	extern COMCOMP us_getproto1p;
	REGISTER VARIABLE *highvar;
	GEOM *fromgeom, *togeom;
	PORTPROTO *fromport, *toport;
	REGISTER USERCOM *uc, *ucsub;
	POPUPMENU *pm, *cpm, *pmsub;
	REGISTER POPUPMENUITEM *mi, *miret, *misub;
	extern COMCOMP us_userp;

	if (count == 0)
	{
		count = ttygetparam(M_("Getproto option: "), &us_getproto1p, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}
	l = strlen(pp = par[0]);

	if (namesamen(pp, "instance", l) == 0)
	{
		/* show a popup menu with instances */
		cellcount = 0;
		for(cell = el_curlib->firstcell; cell != NOCELL; cell = cell->nextcell)
			cellcount++;
		pm = (POPUPMENU *)emalloc(sizeof(POPUPMENU), us_tool->cluster);
		if (pm == 0) return;
		mi = (POPUPMENUITEM *)emalloc(cellcount * sizeof(POPUPMENUITEM), us_tool->cluster);
		if (mi == 0) return;
		pm->name = "x";
		(void)initinfstr();
		if (cellcount != 0) (void)formatinfstr(_("Facets in %s"), el_curlib->libname); else
			(void)formatinfstr(_("No facets in %s"), el_curlib->libname);
		(void)allocstring(&pm->header, returninfstr(), us_tool->cluster);
		pm->list = mi;
		pm->total = cellcount;

		/* fill the menu */
		cellcount = 0;
		for(cell = el_curlib->firstcell; cell != NOCELL; cell = cell->nextcell)
		{
			uc = us_allocusercom();
			mi[cellcount].response = uc;
			mi[cellcount].response->active = -1;
			mi[cellcount].value = 0;

			facetcount = 0;
			for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
				if (np->cell == cell) facetcount++;
			if (facetcount == 1)
			{
				/* only one facet of this cell: create it directly */
				uc->active = parse("getproto", &us_userp, TRUE);
				(void)allocstring(&uc->comname, "getproto", us_tool->cluster);
				uc->count = 2;
				(void)allocstring(&uc->word[0], "node", us_tool->cluster);
				(void)allocstring(&uc->word[1], describenodeproto(cell->firstincell),
					us_tool->cluster);
				mi[cellcount].attribute = uc->word[1];
			} else
			{
				/* multiple facets of this cell: make a submenu */
				mi[cellcount].attribute = cell->cellname;
				mi[cellcount].value = ">>";
				mi[cellcount].maxlen = -1;
				pmsub = (POPUPMENU *)emalloc(sizeof(POPUPMENU), us_tool->cluster);
				if (pmsub == 0) return;
				misub = (POPUPMENUITEM *)emalloc(facetcount * sizeof(POPUPMENUITEM), us_tool->cluster);
				if (misub == 0) return;
				pmsub->name = "x";
				pmsub->header = 0;
				pmsub->list = misub;
				pmsub->total = facetcount;
				facetcount = 0;
				for(i=0; i<2; i++)
				{
					firstinpass = facetcount;
					for(np = el_curlib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
					{
						if (np->cell != cell) continue;

						/* see if this facet should be included on this pass */
						if (i == 0)
						{
							/* first pass: only include facets that match the current view */
							if (!us_facetfromtech(np, el_curtech)) continue;
						} else
						{
							/* first pass: only include facets that don't match the current view */
							if (us_facetfromtech(np, el_curtech)) continue;
						}

						ucsub = us_allocusercom();
						misub[facetcount].response = ucsub;
						misub[facetcount].response->active = -1;
						(void)allocstring(&misub[facetcount].attribute, describenodeproto(np),
							us_tool->cluster);
						misub[facetcount].value = 0;

						/* only one facet of this cell: create it directly */
						ucsub->active = parse("getproto", &us_userp, TRUE);
						(void)allocstring(&ucsub->comname, "getproto", us_tool->cluster);
						ucsub->count = 2;
						(void)allocstring(&ucsub->word[0], "node", us_tool->cluster);
						(void)allocstring(&ucsub->word[1], describenodeproto(np),
							us_tool->cluster);
						facetcount++;
					}
					esort(&misub[firstinpass], facetcount-firstinpass,
						sizeof (POPUPMENUITEM), us_sortpopupmenuascending);
				}
				uc->menu = pmsub;
			}
			cellcount++;
		}

		/* invoke the popup menu */
		esort(mi, cellcount, sizeof (POPUPMENUITEM), us_sortpopupmenuascending);
		butstate = TRUE;
		cpm = pm;
		miret = us_popupmenu(&cpm, &butstate, TRUE, -1, -1, 4);
		if (miret == 0)
		{
			us_abortcommand(_("Sorry, popup menus are not available"));
		} else
		{
			if (miret != NOPOPUPMENUITEM)
				us_execute(miret->response, FALSE, FALSE, FALSE);
		}
		for(i=0; i<cellcount; i++)
		{
			uc = mi[i].response;
			if (uc->menu != NOPOPUPMENU)
			{
				for(l=0; l<uc->menu->total; l++)
				{
					efree((char *)uc->menu->list[l].attribute);
					us_freeusercom(uc->menu->list[l].response);
				}
				efree((char *)uc->menu->list);
				efree((char *)uc->menu);
			}
			us_freeusercom(uc);
		}
		efree((char *)mi);
		efree((char *)pm->header);
		efree((char *)pm);
		return;
	}

	if (namesamen(pp, "node", l) == 0 && l >= 2 && count > 1)
	{
		np = getnodeproto(par[1]);
		if (np == NONODEPROTO)
		{
			us_abortcommand(_("Cannot find node '%s'"), par[1]);
			return;
		}
		us_setnodeproto(np);
		return;
	}
	if (namesamen(pp, "arc", l) == 0 && l >= 1 && count > 1)
	{
		ap = getarcproto(par[1]);
		if (ap == NOARCPROTO) us_abortcommand(_("Cannot find arc '%s'"), par[1]); else
			us_setarcproto(ap, TRUE);
		return;
	}

	if (namesamen(pp, "this-proto", l) == 0 && l >= 1)
	{
		np = us_needfacet();
		if (np == NONODEPROTO) return;
		us_setnodeproto(np);
		return;
	}

	/* decide whether arcs are the default */
	doarc = 0;
	if (!us_gettwoobjects(&fromgeom, &fromport, &togeom, &toport)) doarc = 1; else
	{
		highvar = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_highlightedkey);
		if (highvar != NOVARIABLE && getlength(highvar) == 1)
		{
			(void)us_makehighlight(((char **)highvar->addr)[0], &high);
			if ((high.status&HIGHFROM) != 0 && !high.fromgeom->entryisnode) doarc++;
		}
	}

	if (namesamen(pp, "next-proto", l) == 0 && l >= 2)
	{
		if (doarc)
		{
			/* advance to the next arcproto */
			ap = us_curarcproto->nextarcproto;
			if (ap == NOARCPROTO) ap = el_curtech->firstarcproto;
			us_setarcproto(ap, TRUE);
		} else
		{
			/* advance to the next nodeproto */
			np = us_curnodeproto;
			if (np->primindex == 0) np = el_curtech->firstnodeproto; else
			{
				/* advance to next after "np" */
				np = np->nextnodeproto;
				if (np == NONODEPROTO) np = el_curtech->firstnodeproto;
			}
			us_setnodeproto(np);
		}
		return;
	}

	if (namesamen(pp, "prev-proto", l) == 0 && l >= 1)
	{
		if (doarc)
		{
			/* backup to the previous arcproto */
			for(ap = el_curtech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
				if (ap->nextarcproto == us_curarcproto) break;
			if (ap == NOARCPROTO)
				for(lat = el_curtech->firstarcproto; lat != NOARCPROTO; lat = lat->nextarcproto)
					ap = lat;
			us_setarcproto(ap, TRUE);
		} else
		{
			/* backup to the previous nodeproto */
			np = us_curnodeproto;
			if (np->primindex == 0) np = el_curtech->firstnodeproto; else
			{
				/* back up to previous of "np" */
				np = np->lastnodeproto;
				if (np == NONODEPROTO)
					for(np = el_curtech->firstnodeproto; np->nextnodeproto != NONODEPROTO;
						np = np->nextnodeproto) ;
			}
			us_setnodeproto(np);
		}
		return;
	}

	/* must be a prototype name */
	if (doarc != 0)
	{
		ap = getarcproto(pp);
		if (ap != NOARCPROTO)
		{
			us_setarcproto(ap, TRUE);
			return;
		}
	}

	np = getnodeproto(pp);
	if (np == NONODEPROTO)
	{
		if (doarc != 0) us_abortcommand(_("Cannot find node or arc '%s'"), pp); else
			us_abortcommand(_("Cannot find node '%s'"), pp);
		return;
	}
	us_setnodeproto(np);
}

void us_grid(INTBIG count, char *par[])
{
	REGISTER INTBIG i, j;
	REGISTER INTBIG l;
	REGISTER char *pp;

	/* no arguments: toggle the grid state */
	if (count == 0)
	{
		if (us_needwindow()) return;
		if ((el_curwindowpart->state&WINDOWTYPE) != DISPWINDOW)
		{
			us_abortcommand(_("Cannot show grid in this type of window"));
			return;
		}

		/* save highlight */
		us_pushhighlight();
		us_clearhighlightcount();

		startobjectchange((INTBIG)el_curwindowpart, VWINDOWPART);
		us_gridset(el_curwindowpart, ~el_curwindowpart->state);
		endobjectchange((INTBIG)el_curwindowpart, VWINDOWPART);

		/* restore highlighting */
		us_pophighlight(FALSE);
		return;
	}

	l = strlen(pp = par[0]);
	if (namesamen(pp, "alignment", l) == 0 && l >= 1)
	{
		if (count >= 2)
		{
			i = atofr(par[1]);
			if (i < 0)
			{
				us_abortcommand(_("Alignment must be positive"));
				return;
			}
			(void)setvalkey((INTBIG)us_tool, VTOOL, us_alignment_ratio_key, i, VINTEGER);
		}
		ttyputverbose(M_("Cursor alignment is %s lambda"), frtoa(us_alignment_ratio));
		return;
	}

	if (namesamen(pp, "edges", l) == 0 && l >= 1)
	{
		if (count >= 2)
		{
			i = atofr(par[1]);
			if (i < 0)
			{
				us_abortcommand(_("Alignment must be positive"));
				return;
			}
			(void)setvalkey((INTBIG)us_tool, VTOOL, us_alignment_edge_ratio_key, i, VINTEGER);
		}
		if (us_edgealignment_ratio == 0) ttyputverbose(M_("No edge alignment done")); else
			ttyputverbose(M_("Edge alignment is %s lambda"), frtoa(us_edgealignment_ratio));
		return;
	}

	if (namesamen(pp, "size", l) == 0 && l >= 1)
	{
		if (count < 2)
		{
			ttyputusage("grid size X [Y]");
			return;
		}
		i = atola(par[1]);
		if (i&1) i++;
		if (count >= 3)
		{
			j = atola(par[2]);
			if (j&1) j++;
		} else j = i;
		if (i <= 0 || j <= 0)
		{
			us_abortcommand(_("Invalid grid spacing"));
			return;
		}

		if (us_needwindow()) return;

		/* save highlight */
		us_pushhighlight();
		us_clearhighlightcount();
		startobjectchange((INTBIG)el_curwindowpart, VWINDOWPART);

		/* turn grid off if on */
		if ((el_curwindowpart->state&GRIDON) != 0)
			(void)setval((INTBIG)el_curwindowpart, VWINDOWPART, "state",
				el_curwindowpart->state & ~GRIDON, VINTEGER);

		/* adjust grid */
		(void)setval((INTBIG)el_curwindowpart, VWINDOWPART, "gridx", i, VINTEGER);
		(void)setval((INTBIG)el_curwindowpart, VWINDOWPART, "gridy", j, VINTEGER);

		/* show new grid */
		us_gridset(el_curwindowpart, GRIDON);

		/* restore highlighting */
		endobjectchange((INTBIG)el_curwindowpart, VWINDOWPART);
		us_pophighlight(FALSE);
		return;
	}
	ttyputbadusage("grid");
}

#define NEWSFILE   "newsfile"			/* file with news */

#define NEWSDATE   ".electricnews"		/* file with date of last news */

void us_help(INTBIG count, char *par[])
{
	REGISTER char *pp;
	REGISTER INTBIG len, lastquiet, nday, nmonth, nyear, on;
	INTBIG day, month, year, hour, minute, second;
	FILE *in;
	char line[256], *filename, *hd;
	UINTBIG clock;

	if (count > 0) len = strlen(pp = par[0]);

	/* show the user manual */
	if (count >= 1 && namesamen(pp, "manual", len) == 0)
	{
#ifdef DOCDIR
		strcpy(line, DOCDIR);
		strcat(line, "index.html");
		if (fileexistence(line) == 1)
		{
			if (browsefile(line))
			{
				us_abortcommand(_("Cannot bring up the user's manual on this system"));
			}
			return;
		}
#endif
		strcpy(line, el_libdir);
		len = strlen(line);
		if (line[len-1] == DIRSEP) line[len-1] = 0;
		len = strlen(line);
		if (namesame(&line[len-3], "lib") == 0) line[len-3] = 0;
		strcat(line, "html");
		strcat(line, DIRSEPSTR);
		strcat(line, "manual");
		strcat(line, DIRSEPSTR);
		strcat(line, "index.html");
		if (fileexistence(line) != 1)
		{
			us_abortcommand(_("Sorry, cannot locate the user's manual"));
			return;
		}

		if (browsefile(line))
		{
			us_abortcommand(_("Cannot bring up the user's manual on this system"));
		}
		return;
	}

	/* print news */
	if (count >= 1 && namesamen(pp, "news", len) == 0)
	{
		/* determine last date of news reading */
		(void)initinfstr();
		hd = hashomedir();
		if (hd == 0) (void)addstringtoinfstr(el_libdir); else
			(void)addstringtoinfstr(hd);
		(void)addstringtoinfstr(NEWSDATE);
		pp = truepath(returninfstr());
		in = fopen(pp, "r");
		if (in == 0) year = month = day = 0; else
		{
			fclose(in);
			clock = filedate(pp);
			parsetime(clock, &year, &month, &day, &hour, &minute, &second);
			month++;
		}

		/* get the news file */
		(void)initinfstr();
		(void)addstringtoinfstr(el_libdir);
		(void)addstringtoinfstr(NEWSFILE);
		pp = truepath(returninfstr());
		in = xopen(pp, us_filetypenews, "", &filename);
		if (in == NULL)
		{
			ttyputerr(_("Sorry, cannot find the news file: %s"), pp);
			return;
		}

		/* read the file */
		on = 0;

		/* enable messages (if they were off) */
		lastquiet = ttyquiet(0);
		for(;;)
		{
			if (xfgets(line, MAXLINE, in)) break;
			if (on != 0)
			{
				ttyputmsg("%s", line);
				continue;
			}
			if (line[0] == ' ' || line[0] == 0) continue;

			/* line with date, see if it is current */
			pp = line;
			nmonth = atoi(pp);
			while (*pp != '/' && *pp != 0) pp++;
			if (*pp == '/') pp++;
			nday = atoi(pp);
			while (*pp != '/' && *pp != 0) pp++;
			if (*pp == '/') pp++;
			nyear = atoi(pp);
			if (nyear < year) continue; else if (nyear > year) on = 1;
			if (nmonth < month) continue; else if (nmonth > month) on = 1;
			if (nday >= day) on = 1;
			if (on != 0) ttyputmsg("%s", line);
		}
		xclose(in);

		/* restore message output state */
		(void)ttyquiet(lastquiet);

		if (on == 0) ttyputmsg(_("No news"));

		/* now mark the current date */
		(void)initinfstr();
		hd = hashomedir();
		if (hd == 0) (void)addstringtoinfstr(el_libdir); else
			(void)addstringtoinfstr(hd);
		(void)addstringtoinfstr(NEWSDATE);
		xclose(xcreate(truepath(returninfstr()), us_filetypenews, 0, 0));
		return;
	}

	/* illustrate commands */
	if (count >= 1 && namesamen(pp, "illustrate", len) == 0)
	{
		us_illustratecommandset();
		return;
	}

	/* dump pulldown menus */
	if (count >= 1 && namesamen(pp, "pulldowns", len) == 0)
	{
		us_dumppulldownmenus();
		return;
	}

	/* general dialog-based help on command-line */
	(void)us_helpdlog("CL");
}

void us_if(INTBIG count, char *par[])
{
	REGISTER INTBIG term1, term1type, term2, term2type;
	REGISTER INTBIG relation;
	REGISTER USERCOM *com;

	/* make sure the proper number of parameters is given */
	if (count < 4)
	{
		ttyputusage("if TERM1 RELATION TERM2 COMMAND");
		return;
	}

	/* get term 1 */
	if (isanumber(par[0]))
	{
		term1 = myatoi(par[0]);
		term1type = VINTEGER;
	} else
	{
		term1 = (INTBIG)par[0];
		term1type = VSTRING;
	}

	/* get term 2 */
	if (isanumber(par[2]))
	{
		term2 = myatoi(par[2]);
		term2type = VINTEGER;
	} else
	{
		term2 = (INTBIG)par[2];
		term2type = VSTRING;
	}

	/* make sure the two terms are comparable */
	if (term1type != term2type)
	{
		if (term1 == VINTEGER)
		{
			term1 = (INTBIG)par[0];
			term1type = VSTRING;
		} else
		{
			term2 = (INTBIG)par[1];
			term2type = VSTRING;
		}
	}

	/* determine the relation being tested */
	relation = -1;
	if (strcmp(par[1], "==") == 0) relation = 0;
	if (strcmp(par[1], "!=") == 0) relation = 1;
	if (strcmp(par[1], "<")  == 0) relation = 2;
	if (strcmp(par[1], "<=") == 0) relation = 3;
	if (strcmp(par[1], ">")  == 0) relation = 4;
	if (strcmp(par[1], ">=") == 0) relation = 5;
	if (relation < 0)
	{
		us_abortcommand(_("Unknown relation: %s"), par[1]);
		return;
	}

	/* make sure that qualitative comparison is done on numbers */
	if (relation > 1 && term1type != VINTEGER)
	{
		us_abortcommand(_("Inequality comparisons must be done on numbers"));
		return;
	}

	/* see if the command should be executed */
	switch (relation)
	{
		case 0:		/* == */
			if (term1type == VINTEGER)
			{
				if (term1 != term2) return;
			} else
			{
				if (namesame((char *)term1, (char *)term2) != 0) return;
			}
			break;
		case 1:		/* != */
			if (term1type == VINTEGER)
			{
				if (term1 == term2) return;
			} else
			{
				if (namesame((char *)term1, (char *)term2) == 0) return;
			}
			break;
		case 2:		/* < */
			if (term1 >= term2) return;
			break;
		case 3:		/* <= */
			if (term1 > term2) return;
			break;
		case 4:		/* > */
			if (term1 <= term2) return;
			break;
		case 5:		/* >= */
			if (term1 < term2) return;
			break;
	}

	/* condition is true: create the command to execute */
	com = us_buildcommand(count-3, &par[3]);
	if (com == NOUSERCOM)
	{
		us_abortcommand(_("Condition true but there is no command to execute"));
		return;
	}
	us_execute(com, FALSE, FALSE, FALSE);
	us_freeusercom(com);
}

void us_interpret(INTBIG count, char *par[])
{
	char *retstr;
	REGISTER char *pp;
	REGISTER INTBIG language, len;
	UINTBIG type;
	REGISTER BOOLEAN fromfile;

	language = VLISP;
	fromfile = FALSE;
	if (namesamen(par[0], "file", strlen(par[0])) == 0)
	{
		count--;
		par++;
		fromfile = TRUE;
	}

	if (count > 0)
	{
		len = strlen(pp = par[0]);
		if (namesamen(pp, "lisp", len) == 0)
		{
			language = VLISP;
			count--;
			par++;
		} else if (namesamen(pp, "tcl", len) == 0)
		{
			language = VTCL;
			count--;
			par++;
		} else if (namesamen(pp, "java", len) == 0)
		{
			language = VJAVA;
			count--;
			par++;
		}
	}

	/* handle "file" option */
	if (fromfile)
	{
		if (count <= 0)
		{
			ttyputusage("interpret file LANGUAGE FILENAME");
			return;
		}
		if (loadcode(par[0], language))
			ttyputerr(_("Error loading code"));
		return;
	}

	/* with no parameters, simply drop into the interpreter loop */
	if (count == 0)
	{
		/* "languageconverse" returns false if it wants to continue conversation */
		if (!languageconverse(language)) us_state |= LANGLOOP; else
			ttyputmsg(_("Back to Electric"));
		return;
	}

	if (count > 1)
	{
		us_abortcommand(_("Please provide only one parameter to be interpreted"));
		return;
	}

	type = VSTRING;
	if (doquerry(par[0], language, &type, (INTBIG *)&retstr)) ttyputmsg("%s", par[0]); else
		ttyputmsg("%s => %s", par[0], retstr);
}

void us_iterate(INTBIG count, char *par[])
{
	REGISTER INTBIG i, j, times;
	REGISTER BOOLEAN repeatcommand;
	REGISTER USERCOM *uc;
	REGISTER INTBIG len, total;
	INTBIG cindex, objaddr, objtype;
	VARIABLE *var, fvar;
	char *qual, parnames[30];
	BOOLEAN comvar;
	REGISTER USERCOM *com;

	/* see if this is the "repeat last command" form */
	repeatcommand = FALSE;
	if (count == 0)
	{
		repeatcommand = TRUE;
		times = 1;
	} else
	{
		if (isanumber(par[0]))
		{
			repeatcommand = TRUE;
			times = atoi(par[0]);
		}
	}

	if (repeatcommand)
	{
		/* make sure there was a valid previous command */
		if (us_lastcom == NOUSERCOM || us_lastcom->active < 0)
		{
			us_abortcommand(_("No last command to repeat"));
			return;
		}

		/* copy last command into new one */
		uc = us_allocusercom();
		if (uc == NOUSERCOM)
		{
			ttyputnomemory();
			return;
		}
		for(i=0; i<us_lastcom->count; i++)
			if (allocstring(&uc->word[i], us_lastcom->word[i], us_tool->cluster))
		{
			ttyputnomemory();
			return;
		}
		uc->active = us_lastcom->active;
		uc->count = us_lastcom->count;
		(void)allocstring(&uc->comname, us_lastcom->comname, us_tool->cluster);
		uc->menu = us_lastcom->menu;

		/* execute this command */
		for(j=0; j<times; j++) us_execute(uc, TRUE, FALSE, TRUE);
		us_freeusercom(uc);
	} else
	{
		/* implement the iterate over array-variable form */
		if (count < 2)
		{
			ttyputusage("iterate ARRAY-VARIABLE MACRO");
			return;
		}
		if (us_getvar(par[0], &objaddr, &objtype, &qual, &comvar, &cindex))
		{
			us_abortcommand(_("Incorrect iterator variable name: %s"), par[0]);
			return;
		}
		if (*qual != 0)
		{
			var = getval(objaddr, objtype, -1, qual);
			if (var == NOVARIABLE)
			{
				us_abortcommand(_("Cannot find iterator variable: %s"), par[0]);
				return;
			}
		} else
		{
			fvar.addr = objaddr;
			fvar.type = objtype;
			var = &fvar;
		}
		len = getlength(var);
		if (len < 0) len = 1;
		if (us_expandaddrtypearray(&us_iteratelimit, &us_iterateaddr,
			&us_iteratetype, len)) return;
		if ((var->type&VISARRAY) == 0)
		{
			us_iterateaddr[0] = var->addr;
			us_iteratetype[0] = var->type;
			total = 1;
		} else
		{
			if ((var->type&VTYPE) == VGENERAL)
			{
				for(i=0; i<len; i += 2)
				{
					us_iterateaddr[i/2] = ((INTBIG *)var->addr)[i];
					us_iteratetype[i/2] = ((INTBIG *)var->addr)[i+1];
				}
				total = len / 2;
			} else
			{
				for(i=0; i<len; i++)
				{
					us_iterateaddr[i] = ((INTBIG *)var->addr)[i];
					us_iteratetype[i] = var->type;
				}
				total = len;
			}
		}

		/* now iterate with this value */
		for(i=0; i<total; i++)
		{
			(void)initinfstr();
			(void)addstringtoinfstr(par[1]);
			(void)sprintf(parnames, " %s 0%lo", us_variabletypename(us_iteratetype[i]),
				us_iterateaddr[i]);
			(void)addstringtoinfstr(parnames);
			com = us_makecommand(returninfstr());
			if (com->active < 0) { us_freeusercom(com);   break; }
			us_execute(com, FALSE, FALSE, FALSE);
			us_freeusercom(com);
			if (stopping(STOPREASONITERATE)) break;
		}
	}
}

void us_killfacet(INTBIG count, char *par[])
{
	REGISTER NODEPROTO *np, *onp;
	REGISTER NODEINST *ni;
	REGISTER LIBRARY *lib;
	extern COMCOMP us_showdp;

	if (count == 0)
	{
		count = ttygetparam(M_("Facet name: "), &us_showdp, MAXPARS, par);
		if (count == 0)
		{
			us_abortedmsg();
			return;
		}
	}

	np = getnodeproto(par[0]);
	if (np == NONODEPROTO)
	{
		us_abortcommand(_("No facet called %s"), par[0]);
		return;
	}
	if (np->primindex != 0)
	{
		us_abortcommand(_("Can only kill facets"));
		return;
	}

	/* disallow killing if lock is on */
	if (us_cantedit(np, NONODEPROTO, TRUE)) return;

	/* if there are still instances of the facet, mention them */
	if (np->firstinst != NONODEINST)
	{
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
				onp->temp1 = 0;
		for(ni = np->firstinst; ni != NONODEINST; ni = ni->nextinst)
			ni->parent->temp1++;
		ttyputerr(_("Erase all of the following instances of this facet first:"));
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			for(onp = lib->firstnodeproto; onp != NONODEPROTO; onp = onp->nextnodeproto)
				if (onp->temp1 != 0)
					ttyputmsg(_("  %ld instance(s) in facet %s"), onp->temp1, describenodeproto(onp));
		return;
	}

	/* kill the facet */
	us_dokillfacet(np);
	ttyputmsg(_("Facet %s:%s deleted"), np->cell->lib->libname, nldescribenodeproto(np));
}
