/* Copyright (C) 2001 to 2004 Chris Vine

This program is distributed under the General Public Licence, version 2.
For particulars of this and relevant disclaimers see the file
COPYING distributed with the source files.

*/

#ifndef FAX_LIST_MANAGER_H
#define FAX_LIST_MANAGER_H

#include "prog_defs.h"

#include <string>
#include <list>
#include <map>
#include <set>
#include <fstream>
#include <utility>

#include <gtkmm/treeview.h>
#include <gtkmm/treeviewcolumn.h>
#include <gtkmm/treemodelcolumn.h>
#include <gtkmm/liststore.h>
#include <gtkmm/treemodel.h>
#include <gtkmm/treepath.h>
#include <gtkmm/treestore.h>
#include <gtkmm/targetentry.h>
#include <gtkmm/treeiter.h>
#include <gtkmm/container.h>

#include <gdkmm/pixbuf.h>

#include <glibmm/ustring.h>
#include <glibmm/refptr.h>

#include <sigc++/sigc++.h>

#include "tree_model_row_reference.h"
#include "selected_rows_handle.h"


typedef std::map<std::string, TreeModelRowReference> FolderToRowMap;
typedef std::multimap<std::string, std::string> FolderToFaxMap;
typedef std::map<std::string, std::string> FaxToFolderMap;

/* The following typedefs are in selected_rows_handle.h
typedef std::list<Gtk::TreeModel::Path> RowPathList;
typedef std::list<TreeModelRowReference> RowRefList;
*/

namespace FaxListEnum {
  enum Mode {received, sent};
}

class FaxListManager;

class FolderListModelColumns: public Gtk::TreeModel::ColumnRecord {
  Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > icon;
  Gtk::TreeModelColumn<Glib::ustring> name;
  Gtk::TreeModelColumn<bool> root_only;
  // private constructor - only FaxListManager can create objects of this class
  FolderListModelColumns(void) {add(icon); add(name); add(root_only);}
public:
  friend class FaxListManager;
};

class FaxListModelColumns: public Gtk::TreeModel::ColumnRecord {
  Gtk::TreeModelColumn<Glib::ustring> name;
  Gtk::TreeModelColumn<Glib::ustring> fax_description;
  // private constructor - only FaxListManager can create objects of this class
  FaxListModelColumns(void) {add(name); add(fax_description);}
public:
  friend class FaxListManager;
};

class FolderNameValidator {
  std::set<std::string> folder_set;
public:
  void clear(void) {folder_set.clear();}
  void insert_folder_name(const std::string& name) {folder_set.insert(name);}
  void erase_folder_name(const std::string& name) {folder_set.erase(name);}
  std::pair<bool, Glib::ustring> validate(const Glib::ustring&);
};

class FaxListManager: public sigc::trackable {

  FaxListEnum::Mode mode;

  bool folder_drag_source_enabled;
  bool fax_drag_source_enabled;
  Gtk::TreeModel::iterator folder_drag_source_row_iter;
  RowRefList fax_drag_source_row_refs;
  bool drag_is_fax;
  
  FolderListModelColumns folder_model_columns;
  FaxListModelColumns fax_model_columns;
  Gtk::TreeView folder_tree_view;
  Gtk::TreeView fax_tree_view;
  Gtk::TreeView::Column folder_tree_view_column;
  Glib::RefPtr<Gtk::TreeStore> folder_tree_store_r;
  Glib::RefPtr<Gtk::ListStore> fax_list_store_r;
  std::list<Gtk::TargetEntry> target_list;

  Glib::RefPtr<Gdk::Pixbuf> folder_icon_r;
  Glib::RefPtr<Gdk::Pixbuf> trash_icon_r;

  FolderToFaxMap folder_to_fax_map;
  FaxToFolderMap fax_to_folder_map;

  TreeModelRowReference trash_row_ref;

  FolderNameValidator folder_name_validator;

  std::string get_pathname_for_folder(Gtk::TreeModel::iterator);
  void populate_fax_list(void);
  bool get_folders(FolderToRowMap&, const std::string&);
  void move_fax(const Gtk::TreeModel::iterator&);
  void move_folder(Gtk::TreeModel::iterator, Gtk::TreeViewDropPosition);
  void move_child_folders_for_level(const Gtk::TreeModel::Children&, const Gtk::TreeModel::Children&,
				    const std::string&, const std::string&, bool);
  bool is_valid_drop_path(const std::string&, const std::string&);
  void remap_faxes_in_folder(const std::string&, const std::string&);
  void display_faxes_slot(void);
  void write_path(void);
  void write_folderpath_for_level(const Gtk::TreeModel::Children&, std::ofstream&);
  void fax_selected_slot(void) {selection_notify();}
  void folder_selected_slot(void) {selection_notify();}
  void folder_tree_view_drag_begin_slot(const Glib::RefPtr<Gdk::DragContext>&);
  void fax_tree_view_drag_begin_slot(const Glib::RefPtr<Gdk::DragContext>&);
  bool drag_motion_slot(const Glib::RefPtr<Gdk::DragContext>&, int, int, guint);
  void folder_tree_view_motion_notify_slot(GdkEventMotion*);
  void fax_tree_view_motion_notify_slot(GdkEventMotion*);

#if GTKMM_VERSION >= 24
  void drag_data_get_slot(const Glib::RefPtr<Gdk::DragContext>&,
			  Gtk::SelectionData&, guint, guint);
  void drag_data_received_slot(const Glib::RefPtr<Gdk::DragContext>&,
			       int, int, const Gtk::SelectionData&, guint, guint);
 #else
  void drag_data_get_slot(const Glib::RefPtr<Gdk::DragContext>&,
			  GtkSelectionData*, guint, guint);
  void drag_data_received_slot(const Glib::RefPtr<Gdk::DragContext>&,
			       int, int, GtkSelectionData*, guint, guint);
#endif
public:
  sigc::signal0<void> selection_notify;

  void insert_folder_tree_view(Gtk::Container& container) {
    container.add(folder_tree_view);
  }
  void insert_fax_tree_view(Gtk::Container& container) {
    container.add(fax_tree_view);
  }

  void delete_fax(void);
  void delete_folder(void);
  std::pair<bool, Glib::ustring> is_folder_name_valid(const Glib::ustring& folder_name) {
    return folder_name_validator.validate(folder_name);
  }
  void make_folder(const Glib::ustring&, bool test_valid = true);
  void refresh(void);
  void describe_fax(const Glib::ustring&);
  void empty_trash_folder(void);

  RowPathList::size_type is_fax_selected(void);
  bool is_folder_selected(void) {return (bool)folder_tree_view.get_selection()->get_selected();}
  bool is_selected_folder_empty(void);
  bool is_selected_folder_permanent(void);
  bool show_trash_folder_icon(void);

  bool are_selected_faxes_in_trash_folder(void);
  void move_selected_faxes_to_trash_folder(void);

  Glib::ustring get_fax_number(void);
  Glib::ustring get_fax_description(void);
  Glib::ustring get_folder_name(void);

  FaxListManager(FaxListEnum::Mode);
};

#endif
