/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#include "gCycle.h"
#include "nConfig.h"
#include "rModel.h"
//#include "eTess.h"
#include "eGrid.h"
#include "rTexture.h"
#include "eTimer.h"
#include "tInitExit.h"
#include "rScreen.h"
#include "rFont.h"
#include "gSensor.h"
#include "ePlayer.h"
#include "eSound.h"
#include "eFloor.h"
#include "gSparks.h"
#include "gExplosion.h"
#include "gWall.h"
#include "nKrawall.h"
#include "gAIBase.h"

#include "tMath.h"
#include <stdlib.h>

#ifndef DEDICATED
#include "rRender.h"
#include "rSDL.h"
#endif

// static nVersionFeature sg_DoubleSpeed( 1 );

tCONTROLLED_PTR(ePlayerNetID)   lastEnemyInfluence;  	// the last enemy wall we encountered
REAL							lastTime;				// the time it was drawn at

	
// the last enemy possibly responsible for our death
const ePlayerNetID* gEnemyInfluence::GetEnemy()
{
	return lastEnemyInfluence.GetPointer();
}

gEnemyInfluence::gEnemyInfluence()
{
	lastTime = -1000.0f;
}

// add the result of the sensor scan to our data
void gEnemyInfluence::AddSensor( const gSensor& sensor, REAL timePenalty )
{
	// check if the sensor hit an enemy wall
	if ( sensor.type != gSENSOR_ENEMY )
		return;

	// get the wall
	if ( !sensor.ehit )
		return;

	eWall* wall = sensor.ehit->GetWall();
	if ( !wall )
		return;

	// see if it is a player wall
	gPlayerWall *playerWall = dynamic_cast<gPlayerWall*>( wall );
	if ( !playerWall )
		return;

	// get the approximate time the wall was drawn
	REAL time = playerWall->Time( 0.5f ) - timePenalty;

	// get the cycle
	gCycle *cycle = playerWall->Cycle();
	if ( !cycle )
		return;

	// make sure he is alive
	if ( !cycle->Alive() )
		return;

	// get the player
	ePlayerNetID* player = cycle->Player();
	if ( !player )
		return;

	const ePlayerNetID* pInfluence = this->lastEnemyInfluence.GetPointer();

	if ( !pInfluence  || 
		 !pInfluence->Object() ||
		 !pInfluence->Object()->Alive() )
		lastTime -= 1000;

	if ( time > lastTime )
	{
		lastEnemyInfluence = player;
		lastTime		   = time;
	}
}

static float sg_speedMultiplier = .5f;
static nSettingItem<float> conf_mult ("REAL_CYCLE_SPEED_FACTOR", sg_speedMultiplier);

float gCycle::SpeedMultiplier()    
{ 
//	if ( sg_DoubleSpeed.Supported() )
//		return sg_speedMultiplier * 2.0; 
//	else
		return sg_speedMultiplier; 
}

void gCycle::SetSpeedMultiplier(float mult)
{
	conf_mult.Set( mult );
}


void clamp(REAL &c, REAL min, REAL max){
	tASSERT(min < max);

	if (!finite(c))
		c = 0;

	if (c<min)
		c = min;

	if (c>max)
		c = max;
}


REAL		gCycle::wallsStayUpDelay=10.0f;	// the time the cycle walls stay up ( negative values: they stay up forever )

REAL		gCycle::wallsLength=100.0f;		// the maximum total length of the walls
REAL		gCycle::explosionRadius=10.0f;	// the radius of the holes blewn in by an explosion

static REAL sg_cycleBrakeRefill  = 0.1;
static REAL sg_cycleBrakeDeplete = 1.0;

static nSettingItem<REAL> sg_cycleBrakeRefillConf("CYCLE_BRAKE_REFILL",sg_cycleBrakeRefill);
static nSettingItem<REAL> sg_cycleBrakeDepleteConf("CYCLE_BRAKE_DEPLETE",sg_cycleBrakeDeplete);

static		nSettingItem<REAL> *c_pwsud = NULL, *c_pwl = NULL, *c_per = NULL;

void gCycle::PrivateSettings()
{
	static nSettingItem<REAL> c_wsud("CYCLE_WALLS_STAY_UP_DELAY",wallsStayUpDelay);
	static nSettingItem<REAL> c_wl("CYCLE_WALLS_LENGTH",wallsLength);
	static nSettingItem<REAL> c_er("CYCLE_EXPLOSION_RADIUS",explosionRadius);

	c_pwsud=&c_wsud;
	c_pwl  =&c_wl;
	c_per  =&c_er;
}

// base speed of cycle im m/s
static REAL sg_speedCycle=10;
static nSettingItem<REAL> c_s("CYCLE_SPEED",sg_speedCycle);

// start speed of cycle im m/s
static REAL sg_speedCycleStart=20;
static tSettingItem<REAL> c_st("CYCLE_START_SPEED",
							   sg_speedCycleStart);

// min time between turns
REAL sg_delayCycle=.1;
static nSettingItem<REAL> c_d("CYCLE_DELAY",
							  sg_delayCycle);

// acceleration multiplicator
static REAL sg_accelerationCycle=10;
static nSettingItem<REAL> c_a("CYCLE_ACCEL",
							  sg_accelerationCycle);

// acceleration multiplicator
static REAL sg_accelerationCycleOffs=2;
static nSettingItem<REAL> c_ao("CYCLE_ACCEL_OFFSET",
							   sg_accelerationCycleOffs);

// when is a eWall near?
static REAL sg_nearCycle=6;
static nSettingItem<REAL> c_n("CYCLE_WALL_NEAR",
							  sg_nearCycle);

// sound speed divisor
static REAL sg_speedCycleSound=15;
static nSettingItem<REAL> c_ss("CYCLE_SOUND_SPEED",
							   sg_speedCycleSound);

// strength of brake
static REAL sg_brakeCycle=30;
static nSettingItem<REAL> c_ab("CYCLE_BRAKE",
							   sg_brakeCycle);

#ifndef DEDICATED
#define MAXRUBBER 1
#else
#define MAXRUBBER 3
#endif

// niceness when crashing a eWall
static REAL sg_rubberCycle=MAXRUBBER;
static nSettingItem<REAL> c_r("CYCLE_RUBBER",
							  sg_rubberCycle);
REAL* globalcyclerubber = &sg_rubberCycle; /// added by subby


// niceness when crashing a eWall, influence of your ping
static REAL sg_rubberCyclePing=3;
static nSettingItem<REAL> c_rp("CYCLE_PING_RUBBER",
							   sg_rubberCyclePing);



// moviepack hack
//static bool moviepack_hack=false;       // do we use it?
//static tSettingItem<bool> ump("MOVIEPACK_HACK",moviepack_hack);



static int score_die=-2;
static tSettingItem<int> s_d("SCORE_DIE",score_die);

static int score_kill=3;
static tSettingItem<int> s_k("SCORE_KILL",score_kill);

static int score_suicide=-4;
static tSettingItem<int> s_s("SCORE_SUICIDE",score_suicide);

// input control

uActionPlayer gCycle::s_brake("CYCLE_BRAKE", -5);

static eWavData cycle_run("moviesounds/engine.wav","sound/cyclrun.wav");
static eWavData turn_wav("moviesounds/cycturn.wav","sound/expl.wav");
static eWavData scrap("sound/expl.wav");

// a class of textures where the transparent part of the
// image is replaced by the player color
class gTextureCycle: public rTexture{
	gRealColor color_; // player color
	bool wheel; // wheel or body
public:
	gTextureCycle(const char *fileName, const gRealColor& color,bool repx=0,bool repy=0,bool wheel=false);

	virtual void ProcessImage(SDL_Surface *im);

	void Select();
};


gTextureCycle::gTextureCycle(const char *fileName, const gRealColor& color,bool repx,bool repy,bool w)
	:rTexture(rTEX_OBJ,fileName,repx,repy),
	 color_(color),wheel(w)
{
	Select();
}

void gTextureCycle::ProcessImage(SDL_Surface *im){
#ifndef DEDICATED
	GLubyte R=int(color_.r*255);
	GLubyte G=int(color_.g*255);
	GLubyte B=int(color_.b*255);
  
	GLubyte *pixels =reinterpret_cast<GLubyte *>(im->pixels);

	for(int i=im->w*im->h-1;i>=0;i--){
		GLubyte alpha=pixels[4*i+3];
		pixels[4*i  ] = (alpha * pixels[4*i  ] + (255-alpha)*R) >> 8;
		pixels[4*i+1] = (alpha * pixels[4*i+1] + (255-alpha)*G) >> 8;
		pixels[4*i+2] = (alpha * pixels[4*i+2] + (255-alpha)*B) >> 8;
		pixels[4*i+3] = 255;
	}
#endif
}

void gTextureCycle::Select(){
#ifndef DEDICATED
	rTexture::Select();

	if(TextureMode[rTEX_OBJ]<0){
		REAL R=color_.r,G=color_.g,B=color_.b;
		if(wheel){
			R*=.7;
			G*=.7;
			B*=.7;
		}
		glColor3f(R,G,B);
		GLfloat color[4]={R,G,B,1};
    
		glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,color);
		glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,color);
	}
#endif
}


//  *****************************************************************



//  *****************************************************************



//  *****************************************************************


// take pos,dir and time from a cycle
gDestination::gDestination(const gCycle &c)
	:position(c.Position()),
	 direction(c.Direction()),
	 gameTime(c.lastTime),
	 distance(c.distance),
	 speed(c.speed),
	 braking(c.braking),
	 chatting(false),
	 hasBeenUsed(false),
	 next(NULL),
	 list(NULL){
#ifdef DEBUG
	if (!finite(gameTime) || !finite(speed) || !finite(distance))
		st_Breakpoint();
#endif
	if ( c.Owner() && c.Player() )
	  chatting = c.Player()->IsChatting();
}

// or from a message
gDestination::gDestination(nMessage &m)
	:gameTime(0),distance(0),speed(0),
	 hasBeenUsed(false),
	 next(NULL),list(NULL){
	m >> position;
	m >> direction;
	m >> distance;

	unsigned short flags;
	m >> flags;
	braking  = flags & 0x01;
	chatting = flags & 0x02;
}
  
// write all the data into a nMessage
void gDestination::WriteCreate(nMessage &m){
	m << position;
	m << direction;
	m << distance;

	unsigned short flags = 0;
	if ( braking )
	  flags |= 0x01;
	if ( chatting )
	  flags |= 0x02;
	m << flags;
}

gDestination *gDestination::RightBefore(gDestination *list, REAL dist){
	if (!list || list->distance > dist)
		return NULL;

	gDestination *ret=list;
	while (ret && ret->next && ret->next->distance < dist)
		ret=ret->next;

	return ret;
}


// insert yourself into a list ordered by gameTime
void gDestination::InsertIntoList(gDestination **l){
	if (!l)
		return;
  
	RemoveFromList();
  
	list=l;

	while (l && *l && (*l)->distance < distance)
		l = &((*l)->next);

	tASSERT(l);
  
	next=*l;
	*l=this;
}



void gDestination::RemoveFromList(){
	if (!list)
		return;
    
	while (list && *list && *list != this)
		list = &((*list)->next);
  
	tASSERT(list);
	tASSERT(*list);
  
	(*list) = next;
	next=NULL;
	list=NULL;
}

//  *****************************************************************

static nNOInitialisator<gCycle> cycle_init(320,"cycle");

static void new_destination_handler(nMessage &m){
	// read the destination 
	gDestination *dest=new gDestination(m);

  // and the ID of the cycle the destination is added to
	unsigned short cycle_id;
	m.Read(cycle_id);
	nNetObject *o=nNetObject::ObjectDangerous(cycle_id);
	if (o && &o->CreatorDescriptor() == &cycle_init){
		if ((sn_GetNetState() == nSERVER) && (m.SenderID() != o->Owner()))
			return;
//			Cheater(m.SenderID());

		if(o->Owner() != ::sn_myNetID)
		{
			gCycle* c = dynamic_cast<gCycle *>(o);
			if (c)
			{
				c->AddDestination(dest);
				if ( c->Player() && !dest->Chatting() )
					c->Player()->Activity();
			}
		}
	}
}

static nDescriptor destination_descriptor(321,&new_destination_handler,"destinaton");

static void BroadCastNewDestination(gCycle *c, gDestination *dest){
	nMessage *m=new nMessage(destination_descriptor);
	dest->WriteCreate(*m);
	m->Write(c->ID());
	m->BroadCast();
}

//  *****************************************************************


const eTempEdge* gCycle::Edge(){
	if (currentWall)
		return currentWall->Edge();
	else
		return NULL;
}

const gPlayerWall* gCycle::CurrentWall(){
	if (currentWall)
		return currentWall->Wall();
	else
		return NULL;
}

const gPlayerWall* gCycle::LastWall(){
	if (lastWall)
		return lastWall->Wall();
	else
		return NULL;
}


void gCycle::MyInitAfterCreation(){
#ifdef DEBUG
	// con << "creating cycle.\n";
#endif
	engine  = tNEW(eSoundPlayer)(cycle_run,true);
	turning = tNEW(eSoundPlayer)(turn_wav);
	spark   = tNEW(eSoundPlayer)(scrap);

	correctDistSmooth=correctTimeSmooth=correctSpeedSmooth=0;

	brakingReservoir = 1.0f;

	currentDestination = NULL;
	destinationList = NULL;

	braking = false;
	mp=sg_MoviePack();

	lastTimeAnim = 0;
	timeCameIntoView = 0;

	if (mp)
		customModel=new rModel("moviepack/cycle.ASE","moviepack/cycle.ase");
	else{
		body=new rModel("models/cycle_body.mod");
		front=new rModel("models/cycle_front.mod");
		rear=new rModel("models/cycle_rear.mod");
	}

	correctPosSmooth=eCoord(0,0);

	rotationFrontWheel=rotationRearWheel=eCoord(1,0);

	acceleration=skew=skewDot=0;

	dir=dirDrive;
	alive=1;


	if (sn_GetNetState()!=nCLIENT){
		if(!player)
		{ // distribute AI colors
			// con << current_ai << ':' << take_ai << ':' << maxmindist <<  "\n\n\n";
			color_.r=1;
			color_.g=1;
			color_.b=1;

			trailColor_.r=1;
			trailColor_.g=1;
			trailColor_.b=1;
		}
		else
		{
			player->Color(color_.r,color_.g,color_.b);
			player->TrailColor(trailColor_.r,trailColor_.g,trailColor_.b);
		}

		se_MakeColorValid( color_.r, color_.g, color_.b, 1.0f );
		se_MakeColorValid( trailColor_.r, trailColor_.g, trailColor_.b, .5f );
	}

	//
 
	if (mp)
		customTexture=new gTextureCycle("moviepack/bike.png",color_,0,0);
	else{
		wheelTex=new gTextureCycle("textures/cycle_wheel.png",color_,0,0,true);
		bodyTex=new gTextureCycle("textures/cycle_body.png",color_,0,0);
	}

	distance=0;
	//last_turn=new ePoint(pos);
	//last_turnTime=lastTime;
	//currentPos=new ePoint(pos);
	//last_turn->InsertIntoGrid();
	// current_eEdge=new eEdge(last_turn,currentPos,
	//		       new gPlayerWall(this,lastTime,lastTime,distance,distance));
	z=.75;
	pendingTurns=0;
	nextTurn=lastTime-10;
#ifdef DEBUG
	//con << "Created cycle.\n";
#endif
	nextSync=lastTime+1;
 
	if (!currentWall)
	{
		currentWall=new gNetPlayerWall(this,pos,dirDrive,se_GameTime(),0);

/*
		if (sn_GetNetState()!=nCLIENT)
			currentWallID=currentWall->ID();
*/
	}
	lastWall=currentWall;

	if (sn_GetNetState()!=nCLIENT)
		RequestSync();

	grid->AddGameObjectInteresting(this);

	lastTimeAnim=lastTime;

	if ( engine )
		engine->Reset(10000);

	if ( turning )
		turning->End();

	nextChatAI=lastTime;

	turns=0;

	if (!finite(speed)){
		st_Breakpoint();
		speed = 1;
	}

	if (speed < .1)
		speed=.1;
}

void gCycle::InitAfterCreation(){
#ifdef DEBUG
	if (!finite(speed))
		st_Breakpoint();
#endif
	eNetGameObject::InitAfterCreation();
#ifdef DEBUG
	if (!finite(speed))
		st_Breakpoint();
#endif
	MyInitAfterCreation();
}

gCycle::gCycle(eGrid *grid, const eCoord &pos,const eCoord &d,ePlayerNetID *p,bool autodelete)
	:eNetGameObject(grid, pos,d,p,autodelete),
	 engine(NULL),
	 turning(NULL),
	 destinationList(NULL),currentDestination(NULL),
	 dirDrive(d),skew(0),skewDot(0),speed(sg_speedCycleStart * SpeedMultiplier()),
	 acceleration(0),
	 rotationFrontWheel(1,0),rotationRearWheel(1,0),heightFrontWheel(0),heightRearWheel(0),
	 alive(1),
	 currentWall(NULL),
	 lastWall(NULL)
//	 currentWallID(0)
{  
	windingNumber = Grid()->DirectionWinding(d);

	rubber=0;
	pendingTurns=0;
	turns=0;
	MyInitAfterCreation();
}

gCycle::~gCycle(){
#ifdef DEBUG
	//  con << "deleting cylce...\n";
#endif
	// clear the destination list

	currentDestination = NULL;

	tDESTROY(engine);
	tDESTROY(turning);
	tDESTROY(spark);

	while(destinationList)
		delete destinationList;

	this->RemoveFromGame();

	if (mp){
		delete customModel;
		delete customTexture;
	}
	else{
		delete body;
		delete front;
		delete rear;
		delete wheelTex;
		delete bodyTex;
	}
#ifdef DEBUG
	//con << "Deleted cycle.\n";
#endif
	/*
	  delete currentPos;
	  delete last_turn;
	*/
	speed=distance=0;
}

void gCycle::RemoveFromGame()
{
	if ( this->GetRefcount() > 0 )
	{
		this->Turn(1);
		this->Kill();
	}

	if (currentWall)
		currentWall->CopyIntoGrid();
	currentWall=NULL;
	lastWall=NULL;
}

inline void rotate(eCoord &r,REAL angle){
	REAL x=r.x;
	r.x+=r.y*angle;
	r.y-=x*angle;
	r=r*(1/sqrt(r.NormSquared()));
}


bool crash_sparks=true;
static bool forceTime=false;

// from nNetwork.C
extern REAL planned_rate_control[MAXCLIENTS+2];

bool gCycle::Timestep(REAL currentTime){
	// remove old destinations
	while(destinationList && 
		  destinationList != currentDestination && 
		  destinationList->distance < distance - 100)
		delete destinationList;

  // nothing special if simulating backwards
	if (currentTime < lastTime)
		return TimestepCore(currentTime);

  // no targets are given
  
	if (!currentDestination || pendingTurns){
		if (currentTime>nextChatAI && bool(player) && 
			((player->IsChatting()    && player->Owner() == sn_myNetID) ||
			 (!player->IsActive() && sn_GetNetState() == nSERVER) )
			){
			nextChatAI=currentTime+1;

			gSensor front(this,pos,dir);
			gSensor left(this,pos,dir.Turn(eCoord(0,1)));
			gSensor right(this,pos,dir.Turn(eCoord(0,-1)));
 
			REAL range=speed*4;
      
			front.detect(range);
			left.detect(range);
			right.detect(range);
      
			enemyInfluence.AddSensor( front, 30 );
			enemyInfluence.AddSensor( right, 30 );
			enemyInfluence.AddSensor( left, 30 );

			if (front.hit<left.hit*.9 || front.hit<right.hit*.9){
				REAL lr=front.lr;
				if (front.type==gSENSOR_SELF) // NEVER close yourself in.
					lr*=-100; 
				lr-=left.hit-right.hit;
				if (lr>0)
					Act(&se_turnRight,1);
				else
					Act(&se_turnLeft,1);
			}
		}


		bool simulate=(alive!=0);
    
		// if we predict the object, stop simulation if there is a eWall ahead
		// as long as possible.
#ifndef DEDICATED
		if (sr_predictObjects)
#endif
			if (!forceTime) {
				REAL ts=currentTime-lastTime;
				if (ts>0){
					gSensor fr(this,pos,dirDrive);
					fr.detect(speed*1.5*ts);
					enemyInfluence.AddSensor( fr, -10 );
					REAL wishTime=currentTime;
					if (fr.hit<speed*1.3*ts){ // problem: we are hitting a eWall
						wishTime=lastTime;
						if (wishTime<currentTime-Lag()*3)
							wishTime=currentTime-Lag()*3;
						else
							simulate=false; // no not simulate right now
					}
					currentTime=wishTime;
				}
			}

		if (pendingTurns)
			simulate=true;
    
		// do no simulation if currentTime is smaller than the last
		// client action
    
		if (lastClientsideAction>currentTime)
			simulate=false;
    

		if (simulate)
			return TimestepCore(currentTime);
		else
			return !alive;
	}
  
	int timeout=10;
	while (!pendingTurns && currentDestination && timeout > 0){
		timeout --;

		// the distance to the next destination
		REAL dist_to_dest = eCoord::F(currentDestination->position - pos, dirDrive);

		REAL ts=currentTime - lastTime;

		// our speed
		REAL avgspeed=speed;
		if (acceleration > 0)
			avgspeed += acceleration * SpeedMultiplier() * ts * .5;
    
		if (dist_to_dest > ts * avgspeed)
			break; // no need to worry; we won't reach the next destination

		if (dist_to_dest < 0.01){
			// the destination is very close. Now is the time to turn towards
			// it or turn to the direction it gives us

			REAL t = currentDestination->direction * dirDrive;
      
			bool missed=(fabs(t)<.5);
			if (int(braking) != int(currentDestination->braking))
				missed=!missed;
      
			if (!missed){ // then we did not miss a destination
				currentDestination->hasBeenUsed = true;

				if (fabs(t)>.5)
					Turn(-t);
				else{
					braking = currentDestination->braking;
					RequestSync();
				}

				/*
				  con << "turning alon " << currentDestination->position << "," 
				  << currentDestination->direction << "," 
				  << currentDestination->distance << "\n";
				*/
			}
			else{ // OK, we missed a turn. Don't panic. Just turn
				// towards the destination:
				REAL side = (currentDestination->position - pos) * dirDrive;
				if (fabs(side)>1)
				{
					Turn(-side);
				}
				else
					currentDestination->hasBeenUsed = true;

				/*
				  con << "turning to   " << currentDestination->position << "," 
				  << currentDestination->direction << "," 
				  << currentDestination->distance << "\n";
				*/
			}
			do
				currentDestination = currentDestination->next;
			while (currentDestination && currentDestination->hasBeenUsed);
		}
		// ok, the dest is right ahead, but not close enough.
		// how long does it take at least
		// (therefore the strange average speed) to get there?
		REAL tsTodo = dist_to_dest/avgspeed;
		if ( tsTodo < 0.001 )
		{
			tsTodo = 0.001;
		}
		if ( tsTodo > ts )
		{
			tsTodo = ts;
		}
		TimestepCore( lastTime + tsTodo );
	}
  
	// do the rest of the timestep
	return TimestepCore(currentTime);
}



static void blocks(const gSensor &s, const gCycle *c, int lr)
{
	if ( nCLIENT == sn_GetNetState() )
		return;

	if (s.type == gSENSOR_RIM)
		gAIPlayer::CycleBlocksRim(c, lr);
	else if (s.type == gSENSOR_TEAMMATE || s.type == gSENSOR_ENEMY && s.ehit)
    {
		gPlayerWall *w = dynamic_cast<gPlayerWall*>(s.ehit->GetWall());
		if (w)
		{
			// int turn     = c->Grid()->WindingNumber();
			//	  int halfTurn = turn >> 1;

			// calculate the winding number.
			int windingBefore = c->WindingNumber();  // we start driving in c's direction
			// we need to make a sharp turn in the lr-direction
			//	  windingBefore   += lr * halfTurn;

			// after the transfer, we need to drive in the direction of the other
			// wall:
			int windingAfter = w->WindingNumber();
	  
			// if the other wall drives in the opposite direction, we
			// need to turn around again:
			//	  if (s.lr == lr)
			// windingAfter -= lr * halfTurn;

			// make the winding difference a multiple of the winding number
			/*
			  int compensation = ((windingAfter - windingBefore - halfTurn) % turn)
			  + halfTurn;
			  while (compensation < -halfTurn)
			  compensation += turn;
			*/

			// only if the two walls are parallel/antiparallel, there is true blocking.
			if (((windingBefore - windingAfter) & 1) == 0)
				gAIPlayer::CycleBlocksWay(c, w->Cycle(),
										  lr, s.lr,
										  w->Pos(s.ehit->Ratio(s.before_hit)),
										  - windingAfter + windingBefore);
		}
    }
}



bool gCycle::TimestepCore(REAL currentTime){
	bool emergency=false; // set to TRUE if the cycle is about to hit a wall

	if (!finite(skew))
		skew=0;
	if (!finite(skewDot))
		skewDot=0;
  
	eCoord oldpos=pos;
  
  
	REAL ts=(currentTime-lastTime);

#ifdef DEBUG
	if ( ts > 2.0f )
	{
		int x;	
		x = 0;
	}

	if ( speed > 30.0f )
	{
		int x;	
		x = 0;
	}

	if ( acceleration > 100.0f )
	{
		int x;	
		x = 0;
	}
#endif

	clamp(ts, -10, 10);
	clamp(correctTimeSmooth, -100, 100);
	clamp(correctDistSmooth, -100, 100);
	clamp(correctSpeedSmooth, -100, 100);
	clamp(speed, -1000, 1000);
  
	if (ts > 0){
		REAL scts = correctTimeSmooth * ts;
		if (scts > ts)
			scts = ts;
		correctTimeSmooth -= scts;
		ts -= scts;

		REAL scs = correctSpeedSmooth * ts;
		speed += scs;
		correctSpeedSmooth -= scs;
    
		REAL scd = correctDistSmooth * ts;
		distance += scd;
		correctDistSmooth -= scd;
	}


	REAL step=speed*ts; // +.5*acceleration*ts*ts;
  
	int numTries = 0;

	// be a little nice and don't drive into the eWall
	if (player){
		gSensor fr(this,pos,dirDrive);
		fr.detect(step*3 + rubber);
    
		if (fr.hit < step * 2.1)
			emergency = true;

		if (fr.hit<step*1.1){
			REAL rubberneeded = step - fr.hit * .5;
			if (rubberneeded < 0)
				rubberneeded = 0;
      
			REAL rubber_granted=sg_rubberCycle+Lag()*sg_rubberCyclePing;
      
			if (rubberneeded+rubber>rubber_granted)
				rubberneeded=rubber_granted-rubber;
      
			rubber+=rubberneeded;

			numTries = int((rubber_granted - rubber - .1)/(step*1.5 + .1));

			step-=rubberneeded;
      
			if (step<0){
				rubber+=step;
				step=0;
			}



		}

		if (ts > 0)
			rubber/=(1+ts*.1);

		player->SetRubber(rubber);
	}
	eCoord nextpos;
	if (speed >0)
		nextpos=pos+dirDrive*step;
	else
		nextpos=pos;
  
	if (alive>=1){
		distance+=step;
		Move(nextpos,lastTime,currentTime);
	}
  
	if (alive==-1){
		Move(oldpos,currentTime,currentTime);
    
		if (currentWall){
			currentWall->Update(lastTime,pos);
			currentWall->CopyIntoGrid();
			currentWall=NULL;	
		}
    
		alive=0;
		//    eEdge::SeethroughHasChanged();
	}
  
  
	REAL animts=currentTime-lastTimeAnim;
	if (animts<0 || !finite(animts))
		animts=0;
	else
		lastTimeAnim=currentTime;
  
	if (animts>.2)
		animts=.2;
  
	rotate(rotationFrontWheel,2*speed*animts/.43);
	rotate(rotationRearWheel,2*speed*animts/.73);
  
	//const REAL extension=.5;
	//const REAL extension=1;
	const REAL extension=.25;
  
	// sr_laggometer
	if (eNetGameObject::Timestep(currentTime))
		return true;
  
  
	eCoord correct=correctPosSmooth*animts*3;
	correctPosSmooth=correctPosSmooth-correct;
	pos=pos+correct;
  
	dir=dir+dirDrive*animts*speed*3;
	dir=dir*(1/sqrt(dir.NormSquared()));
  
  
	if (alive){
		if (currentWall)
			currentWall->Update(currentTime,pos);
    
		gSensor fl(this,pos,dirDrive.Turn(1,1));
		gSensor fr(this,pos,dirDrive.Turn(1,-1));
    
		fl.detect(extension*4);
		fr.detect(extension*4);

		enemyInfluence.AddSensor( fr, 0 );
		enemyInfluence.AddSensor( fl, 0 );

		if (fl.ehit)
			blocks(fl, this, -1);

		if (fr.ehit)
			blocks(fr, this,  1);

		if (fl.hit > extension)
			fl.hit = extension;

		if (fr.hit > extension)
			fr.hit = extension;

		REAL lr=(fl.hit-fr.hit)/extension;
    
		// ODE for skew
		const REAL skewrelax=24;
		skewDot-=128*(skew+lr/2)*animts;
		skewDot/=1+skewrelax*animts;
		if (skew*skewDot>4) skewDot=4/skew;
		skew+=skewDot*animts;
		
		REAL fac = 0.5f;
		if ( skew > fr.hit * fac )
		{
			skew = fr.hit * fac;
		}
		if ( skew < -fl.hit * fac )
		{
			skew = -fl.hit * fac;
		}

		eCoord sparkpos,sparkdir;


		if (fl.ehit && fl.hit<extension){
			sparkpos=pos+dirDrive.Turn(1,1)*fl.hit;
			sparkdir=dirDrive.Turn(0,-1);
		}
		if (fr.ehit && fr.hit<extension){
			//      blocks(fr, this, -1);
			sparkpos=pos+dirDrive.Turn(1,-1)*fr.hit;
			sparkdir=dirDrive.Turn(0,1);
		}
    
		/*
		  if (crash_sparks && animts>0)
		  new gSpark(pos,dirDrive,currentTime);
		*/
    
		if (fabs(skew)<fabs(lr*.8) ){
			skewDot-=lr*1000*animts;
			if (crash_sparks && animts>0)
			{
    			    gPlayerWall *tmpplayerWall;
			
			    if(fl.ehit) tmpplayerWall=dynamic_cast<gPlayerWall*>(fl.ehit->GetWall());
			    if(fr.ehit) tmpplayerWall=dynamic_cast<gPlayerWall*>(fr.ehit->GetWall());
			    			   
			    if(tmpplayerWall) {
				gCycle *tmpcycle = tmpplayerWall->Cycle();
				    
				if( tmpcycle )
				    new gSpark(grid, sparkpos-dirDrive*.1,sparkdir,currentTime,color_.r,color_.g,color_.b,tmpcycle->color_.r,tmpcycle->color_.g,tmpcycle->color_.b);
			    }
			    else
			        new gSpark(grid, sparkpos-dirDrive*.1,sparkdir,currentTime,color_.r,color_.g,color_.b,1,1,1);
			    
			    if ( spark )
				spark->Reset();
			}
		}
    
		if (fabs(skew)<fabs(lr*.9) ){
			skewDot-=lr*100*animts;
			if (crash_sparks && animts>0)
			{
    			    gPlayerWall *tmpplayerWall;
			
			    if(fl.ehit) tmpplayerWall=dynamic_cast<gPlayerWall*>(fl.ehit->GetWall());
			    if(fr.ehit) tmpplayerWall=dynamic_cast<gPlayerWall*>(fr.ehit->GetWall());
			    			   
			    if(tmpplayerWall) {
				gCycle *tmpcycle = tmpplayerWall->Cycle();
				    
				if( tmpcycle )
				    new gSpark(grid, sparkpos-dirDrive*.1,sparkdir,currentTime,color_.r,color_.g,color_.b,tmpcycle->color_.r,tmpcycle->color_.g,tmpcycle->color_.b);
			    }
			    else
			        new gSpark(grid, sparkpos-dirDrive*.1,sparkdir,currentTime,color_.r,color_.g,color_.b,1,1,1);
			}
		}
		/*
		  if (fl.hit+fr.hit<extension*.4)
		  Kill();
		*/
	}

	if (skew>.5){
		skew=.5;
		skewDot=0;
	}

	if (skew<-.5){
		skew=-.5;
		skewDot=0;
	}

	// sense near eWalls and calculate acceleration
  
	acceleration=0;
  
	static nVersionFeature brakeDepletion(2);

	if ( brakeDepletion.Supported() )
	{
		if(braking)
		{
			if ( brakingReservoir > 0.0 )
			{
				brakingReservoir -= ts * sg_cycleBrakeDeplete;
				acceleration-=sg_brakeCycle * SpeedMultiplier();
			}
			else
				brakingReservoir = 0.0f;
		}
		else
		{
			if ( brakingReservoir < 1.0 )
			{
				brakingReservoir += ts * sg_cycleBrakeRefill;
			}
			else
				brakingReservoir = 1.0f;
		}
	}
	else
	{
		if(braking)
		{
			acceleration-=sg_brakeCycle * SpeedMultiplier();
		}
	}
    
	if (speed<sg_speedCycle) 
		acceleration+=(sg_speedCycle * SpeedMultiplier() - speed) * 5;
	else
		acceleration+=(sg_speedCycle  * SpeedMultiplier() - speed)/10;
  
	for(int d=1;d>=-1;d-=2){
		gSensor rear(this,pos,dirDrive.Turn(-1,d));
		rear.detect(sg_nearCycle);
		enemyInfluence.AddSensor( rear, 1 );
		if (rear.type!=gSENSOR_RIM)
			acceleration+=SpeedMultiplier() * sg_accelerationCycle * ((1/(rear.hit+sg_accelerationCycleOffs))
																	   -(1/(sg_nearCycle+sg_accelerationCycleOffs)));
	}    
  
	if (speed >0){
		speed+=acceleration*ts;
		if (speed<sg_speedCycle*SpeedMultiplier()*.25){
			speed=sg_speedCycle*SpeedMultiplier()*.25;
			acceleration=0;
		}
	}

	lastTime=currentTime;
	static bool recursion=0;
	if (pendingTurns && lastTime>nextTurn){
		if (!recursion){
			recursion=1;
			Timestep(nextTurn);
			recursion=0;
		}
		//con << "Executing delayed turn at time " << lastTime << "\n";
		if (pendingTurns>0){
			Turn(1);
			pendingTurns--;
		}
		if (pendingTurns<0){
			Turn(-1);
			pendingTurns++;
		}
	}
	// if (last_turnTime+4 < currentTime) Turn(0);
  
	if (nextSync< currentTime && sn_GetNetState()==nSERVER){
		RequestSync(false);
		nextSync=currentTime+1;
	}  
  
	/*
	  if (sn_GetNetState()==nSERVER && owner!=0 && 
	  planned_rate_control[owner]>200)
	  RequestSync(owner,false);
	*/

	// give the AI a chance to evade just in time
	if (emergency && bool(player))
    {
		player->RightBeforeDeath(numTries);
    }

	return (!alive);
}


void gCycle::InteractWith(eGameObject *target,REAL,int){
	/*
	  if (alive && target->type()==ArmageTron_CYCLE){
      gCycle *c=(gCycle *)target;
      if (c->alive){
	  const eEdge *current_eEdge=Edge();
	  const eEdge *other_eEdge=c->Edge();
	  if(current_eEdge && other_eEdge){
	  ePoint *meet=current_eEdge->IntersectWith(other_eEdge);
	  if (meet){ // somebody is going to die!
	  REAL ratio=current_eEdge->Ratio(*meet);
	  REAL time=CurrentWall()->Time(ratio);
	  PassEdge(other_eEdge,time,other_eEdge->Ratio(*meet),0);
	  }
	  }
      }
	  }
	*/
}
   

void gCycle::Hunt(gCycle *hunter, gCycle *prey, const eCoord& pos){
	if (prey->alive<1 || sn_GetNetState()==nCLIENT)
		return;

	prey->pos = pos;

	if ( hunter->Team() == prey->Team() )
	{
		const ePlayerNetID* pEnemy = prey->enemyInfluence.GetEnemy();
		if ( pEnemy )
			hunter = dynamic_cast<gCycle*> ( pEnemy->Object() );
		
		if ( !hunter )
			hunter = prey;
	}

#ifdef KRAWALL_SERVER
	if (hunter->Player()           && prey->Player()          &&
		!dynamic_cast<gAIPlayer*>(hunter->Player())           && 
		!dynamic_cast<gAIPlayer*>(prey->Player())             &&
		hunter->Player()->IsAuth() && prey->Player()->IsAuth())
		nKrawall::ServerFrag(hunter->Player()->name, prey->Player()->name);
#endif

	if (hunter==prey){
		if (hunter->player)
			hunter->player->AddScore(score_suicide, tOutput(), "$player_loose_suicide");
    
	}
	else{
		if (hunter->player)
		{
			tOutput loose;
			tOutput win;
			if (prey->player)
			{
				if (prey->player->CurrentTeam() != hunter->player->CurrentTeam()) {
					tString preyName;
					preyName << *prey->player;
					preyName << ColorString(1,1,1);
					win.SetTemplateParameter(3, preyName);
					win << "$player_win_frag";
					hunter->player->AddScore(score_kill, win, loose);
				}
				else {
					tString hunterName;
					hunterName << *hunter->player;
					sn_ConsoleOut(hunterName << ColorString(1,1,1) << " core dumped a teammate! No points for that!\n");
				}
			}
			else
			{
				win << "$player_win_frag_ai";
				hunter->player->AddScore(score_kill, win, loose);
			}
		}
	//	if (prey->player && (prey->player->CurrentTeam() != hunter->player->CurrentTeam()))
		if (prey->player)
			prey->player->AddScore(score_die,tOutput(),"$player_loose_frag");
	}
	prey->Kill();
}



bool gCycle::EdgeIsDangerous(const eWall* ww, REAL time, REAL a) const{
	if (!ww)
		return false;

	const gPlayerWall *w = dynamic_cast<const gPlayerWall*>(ww);
	if (w)
    {
		if((currentWall && w == currentWall->Wall() ) || 
		   (lastWall    && w == lastWall->Wall()    )      )
			return false;  	     // this is our current eWall. Ignore it.

		// have we entered a hole?
		if ( !w->IsDangerous( a ) )
			return false;

		gCycle *otherPlayer=w->Cycle();
		if (otherPlayer==this && time < w->Time(a)+2.5*sg_delayCycle/SpeedMultiplier()) 
			return false;        // impossible to make such a small circle

		if ( WallsStayUpDelay() >= 0.0f )
		{
			// walls disappeear after death
			if (otherPlayer->alive < 1 && otherPlayer->deathTime+ WallsStayUpDelay()+0.2f<=time)
				return false;        // the other guy is already dead and the stay up delay has passed
		}

		if ( WallsLength() > 0 )
		{
			// walls have finite length
			REAL walldistance = w->Pos( a );
			if ( walldistance + WallsLength() < otherPlayer->distance )
				return false;	// hit was after the wall length
		}
    }
  
	return true; // it is a real eWall.
}

void gCycle::PassEdge(const eWall *ww,REAL time,REAL a,int){
	if (!EdgeIsDangerous(ww,time,a))
		return;

	eCoord collPos = ww->Point( a );

	const gPlayerWall *w = dynamic_cast<const gPlayerWall*>(ww);

	if (w)
    {
		gCycle *otherPlayer=w->Cycle();
      
		if(time < w->Time(a)-EPS) // we were first!
			Hunt(this,otherPlayer, collPos );
		else if (time > w->Time(a)+EPS) // sad but true
		{
			Hunt(otherPlayer,this, collPos );

			//			REAL dist = w->Pos( a );
			//			const_cast<gPlayerWall*>(w)->BlowHole( dist - explosionRadius, dist + explosionRadius );
		}
	}
	else
    {
		if (bool(player) && sn_GetNetState()!=nCLIENT && alive>=1)
		{
			if ( enemyInfluence.GetEnemy() && enemyInfluence.GetEnemy()->Object() )
				Hunt( dynamic_cast<gCycle*>(enemyInfluence.GetEnemy()->Object()), this, collPos );
			else
				player->AddScore(score_suicide, tOutput(),
								 "$player_loose_rim");

			
		}
		Kill();
    }
}

REAL gCycle::PathfindingModifier( const eWall *w ) const
{
	if (!w)
		return 1;
	if (dynamic_cast<const gPlayerWall*>(w))
		return .9f;
	else
		return 1;
}


bool gCycle::Act(uActionPlayer *Act, REAL x){
	if (!alive && sn_GetNetState()==nSERVER)
		RequestSync(false);

	if(se_turnLeft==*Act && x>.5){
		//SendControl(lastTime,&se_turnLeft,1);
		Turn(-1);
		return true;
	}
	else if(se_turnRight==*Act && x>.5){
		//SendControl(lastTime,&se_turnRight,1);
		Turn(1);
		return true;
	}
	else if(s_brake==*Act){
		//SendControl(lastTime,&brake,x);
		braking=(x>0);
		AddDestination();
		return true;
	}
	return false;
}

void gCycle::Turn(REAL d){
	if (d>0)
		Turn(1);
	else if (d<0)
		Turn(-1);
}

void gCycle::Turn(int d){
	if (d >  1) d =  1;
	if (d < -1) d = -1;

	if (alive){
		if ( turning )
			turning->Reset();

		clientside_action();

		//if (sn_GetNetState()==nCLIENT)
		//turns++;
		/*
		  std::cerr << "Turning " << d << " at time " << lastTime 
		  << ", pos " << pos << ", speed " << speed << ".\n";
		*/


		if (lastTime>=nextTurn-.001){
			//if (sn_GetNetState()!=nCLIENT)
			/*
			  if (sn_GetNetState()==nCLIENT && owner==::sn_myNetID){
			  if(d<0)
			  SendControl(lastTime,&se_turnLeft,1);
			  else if(d>0)
			  SendControl(lastTime,&se_turnRight,1);
			  }
			  turns++;
			*/
			/*
			  else{
			  if (d<0)
			  SendControl(lastTime,&se_turnLeft,1);
			  if (d>0)
			  SendControl(lastTime,&se_turnRight,1);
			  }
			*/
			windingNumber += d;

			speed*=.95;
			skewDot+=4*d;
			if (d){
				REAL swap=dirDrive.x;
				dirDrive.x=d*dirDrive.y;
				dirDrive.y=-d*swap;
			}

			if (sn_GetNetState() == nCLIENT && Owner() == ::sn_myNetID)
				AddDestination();

			/*
			  eSensor kil(this,pos,dirDrive);
			  kil.detect(speed/10.0);
			  if (kil.hit<speed/15.0)
			  Kill();
			*/

			//con << "self-assigning new eWall..\n";

			if (sn_GetNetState()!=nCLIENT){
				RequestSync();
			}
      
			if(currentWall){
				lastWall=currentWall;
				currentWall->Update(lastTime,pos);
				currentWall->CopyIntoGrid();
				currentWall=NULL;
			}
			currentWall=new gNetPlayerWall
				(this,pos,dirDrive,lastTime,distance);
/*
			if (sn_GetNetState()!=nCLIENT)
				currentWallID=currentWall->ID();
*/
#ifndef DEDICATED
			nextTurn=lastTime+sg_delayCycle/SpeedMultiplier();
#else
			nextTurn=lastTime+sg_delayCycle*.7/SpeedMultiplier();
#endif
		}
		else
			pendingTurns+=d;
	}
}

void gCycle::Kill(){
	if (sn_GetNetState()!=nCLIENT){
		RequestSync(true);
		if (alive>=1){
			deathTime=lastTime;
			alive=-1;
			new gExplosion(grid, pos,lastTime, color_);
			//	 eEdge::SeethroughHasChanged();

			if ( currentWall )
			{
				currentWall->CopyIntoGrid();
				currentWall = NULL;
			}
		}
	}
	else if (Owner() == ::sn_myNetID)
		AddDestination();
	/*
	  else if (owner!=::sn_myNetID)
	  speed=-.01;
	*/
}
   
   
void gCycle::Turbo(bool turbo){
	if (turbo && speed<30){
		speed=40;
		Turn(0);
	}

	if (!turbo && speed>=30){
		speed=20;
		Turn(0);
	}
}

static rTexture cycle_shad(rTEX_FLOOR,"textures/shadow.png",0,0,true);

#ifndef DEDICATED
void gCycle::Render(const eCamera *cam){
	if (!finite(z) || !finite(pos.x) ||!finite(pos.y)||!finite(dir.x)||!finite(dir.y)
		|| !finite(skew))
		st_Breakpoint();
	if (Alive()){
		//con << "Drawing cycle at " << pos << '\n';

#ifdef DEBUG
		/*     {
			   gDestination *l = destinationList;
			   glDisable(GL_LIGHTING);
			   glColor3f(1,1,1);
			   while(l){
			   if (l == currentDestination)
			   glColor3f(0,1,0);

			   glBegin(GL_LINES);
			   glVertex3f(l->position.x, l->position.y, 0);
			   glVertex3f(l->position.x, l->position.y, 100);
			   glEnd();
	 
			   if (l == currentDestination)
			   glColor3f(0,1,1);
	 
			   l=l->next;
			   }
			   } */
#endif

		GLfloat color[4]={1,1,1,1};
		static GLfloat lposa[4] = { 320, 240, 200,0};
		static GLfloat lposb[4] = { -240, -100, 200,0};
		static GLfloat lighta[4] = { 1, .7, .7, 1 };   
		static GLfloat lightb[4] = { .7, .7, 1, 1 };   
     
		glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,color);
		glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,color);
     
		glLightfv(GL_LIGHT0, GL_DIFFUSE, lighta);
		glLightfv(GL_LIGHT0, GL_SPECULAR, lighta);
		glLightfv(GL_LIGHT0, GL_POSITION, lposa);
		glLightfv(GL_LIGHT1, GL_DIFFUSE, lightb);
		glLightfv(GL_LIGHT1, GL_SPECULAR, lightb);
		glLightfv(GL_LIGHT1, GL_POSITION, lposb);
     
     
		ModelMatrix();
		glPushMatrix();
		eCoord p = PredictPosition();
		glTranslatef(p.x,p.y,0);
		glScalef(.5f,.5f,.5f);


		eCoord ske(1,skew);
		ske=ske*(1/sqrt(ske.NormSquared()));

		GLfloat m[4][4]={{dir.x,dir.y,0,0},
						 {-dir.y,dir.x,0,0},
						 {0,0,1,0},
						 {0,0,0,1}};
		glMultMatrixf(&m[0][0]);

		glPushMatrix();
		//glTranslatef(-1.84,0,0);     
		if (!mp)
			glTranslatef(-1.5,0,0);

		glPushMatrix();
     
		GLfloat sk[4][4]={{1,0,0,0},
						  {0,ske.x,ske.y,0},
						  {0,-ske.y,ske.x,0},
						  {0,0,0,1}};
     
		glMultMatrixf(&sk[0][0]);

   
		glEnable(GL_LIGHT0);
		glEnable(GL_LIGHT1);
		glEnable(GL_LIGHTING);



		TexMatrix();
		IdentityMatrix();

		if (mp){

			ModelMatrix();
			glPushMatrix();
			/*
			  GLfloat sk[4][4]={{0,.1,0,0},
			  {-.1,0,0,0},
			  {0,0,.1,0},
			  {1,.2,-1.05,1}};

			  if (moviepack_hack)
			  glMultMatrixf(&sk[0][0]);
			*/

			customTexture->Select();
			//glDisable(GL_TEXTURE_2D);
			//glDisable(GL_TEXTURE);
			glColor3f(1,1,1);

			//glPolygonMode(GL_FRONT, GL_FILL);
			//glDepthFunc(GL_LESS);
			//glCullFace(GL_BACK);
			customModel->Render();
			//glLineWidth(2);
			//glPolygonMode(GL_BACK,GL_LINE);
			//glDepthFunc(GL_LEQUAL);
			//glCullFace(GL_FRONT);
			//customModel->Render();
			//sr_ResetRenderState();

			glPopMatrix();
			glPopMatrix();
			glTranslatef(-1.5,0,0);
		}
		else{
			/*
			  glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
			  glEnable(GL_TEXTURE_GEN_S);
       
			  glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
			  glEnable(GL_TEXTURE_GEN_T);
       
			  glTexGeni(GL_R,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
			  glEnable(GL_TEXTURE_GEN_R);
       
			  glTexGeni(GL_Q,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
			  glEnable(GL_TEXTURE_GEN_Q);
			*/
       
			glEnable(GL_TEXTURE_2D);

			/*       
					 static    GLfloat tswap[4][4]={{1,0,0,0},
					 {0,0,1,0},
					 {0,-1,0,0},
					 {.5,.5,0,1}};
       
					 static    GLfloat tswapb[4][4]={{1,0,0,0},
					 {0,0,1,0},
					 {0,-1,0,0},
					 {.2,1.2,0,1}};
			*/
       
			//       TexMatrix();
			//       glLoadMatrixf(&tswapb[0][0]);
			//       glScalef(.4,.4,.8);
			ModelMatrix();
       
       
			bodyTex->Select();
			body->Render();
       
			wheelTex->Select();
       
			glPushMatrix();
			glTranslatef(0,0,.73);
       
			GLfloat mr[4][4]={{rotationRearWheel.x,0,rotationRearWheel.y,0},
							  {0,1,0,0},
							  {-rotationRearWheel.y,0,rotationRearWheel.x,0},
							  {0,0,0,1}};
       
       
			glMultMatrixf(&mr[0][0]);
       
       
			//       TexMatrix();
			//       glLoadMatrixf(&tswap[0][0]);
			//       glScalef(.65,.65,.65);
			//       ModelMatrix();
       
			rear->Render();
			glPopMatrix();
       
			glPushMatrix();
			glTranslatef(1.84,0,.43);
       
			GLfloat mf[4][4]={{rotationFrontWheel.x,0,rotationFrontWheel.y,0},
							  {0,1,0,0},
							  {-rotationFrontWheel.y,0,rotationFrontWheel.x,0},
							  {0,0,0,1}};
       
			glMultMatrixf(&mf[0][0]);
       
       
			//       TexMatrix();
			//       glLoadMatrixf(&tswap[0][0]);
			//       glScalef(1.2,1.2,1.2);
			//       ModelMatrix();
       
			front->Render();
			glPopMatrix();
			glPopMatrix();
     

		}
       

		//     TexMatrix();
		//     IdentityMatrix();
		ModelMatrix();

		/*
		  glDisable(GL_TEXTURE_GEN_S);
		  glDisable(GL_TEXTURE_GEN_T);
		  glDisable(GL_TEXTURE_GEN_Q);
		  glDisable(GL_TEXTURE_GEN_R);
		*/

		glDisable(GL_LIGHT0);
		glDisable(GL_LIGHT1);
		glDisable(GL_LIGHTING);
       
		//glDisable(GL_TEXTURE);
		glDisable(GL_TEXTURE_2D);
		glColor3f(1,1,1);

		{
			bool renderPyramid = false;
			gRealColor colorPyramid; 
			REAL alpha = 1;
			const REAL timeout = .5f;

			if ( bool(player) )
			{
				if ( player->IsChatting() )
				{
					renderPyramid = true;
					colorPyramid.b = 0.0f;
				} 
				else if ( !player->IsActive() )
				{
					renderPyramid = true;
					colorPyramid.b = 0.0f;
					colorPyramid.g = 0.0f;
				}
				else if ( cam && cam->Center() == this && se_GameTime() < timeout && player->CurrentTeam() && player->CurrentTeam()->NumPlayers() > 1 )
				{
					renderPyramid = true;
					alpha = timeout - se_GameTime();
				}
			}

			if ( renderPyramid )
			{
				GLfloat s=sin(lastTime);
				GLfloat c=cos(lastTime);

				GLfloat m[4][4]={{c,s,0,0},
								 {-s,c,0,0},
								 {0,0,1,0},
								 {0,0,1,1}};

				glPushMatrix();

				glMultMatrixf(&m[0][0]);
				glScalef(.5,.5,.5);


				BeginTriangles();

				glColor4f( colorPyramid.r,colorPyramid.g,colorPyramid.b, alpha );
				glVertex3f(0,0,3);
				glVertex3f(0,1,4.5);
				glVertex3f(0,-1,4.5);

				glColor4f( colorPyramid.r * .7f,colorPyramid.g * .7f,colorPyramid.b * .7f, alpha );
				glVertex3f(0,0,3);
				glVertex3f(1,0,4.5);
				glVertex3f(-1,0,4.5);
       
				RenderEnd();

				glPopMatrix();
			}
		}

		
		// Name
		RenderName( cam );
		

		// shadow:

		sr_DepthOffset(true);
       

		REAL h=0;//se_cameraZ*.005+.03;

		glEnable(GL_CULL_FACE);

		if(sr_floorDetail>rFLOOR_GRID && TextureMode[rTEX_FLOOR]>0 && sr_alphaBlend){
			glColor3f(0,0,0);
			cycle_shad.Select();
			BeginQuads();
			glTexCoord2f(0,1);
			glVertex3f(-.6,.4,h);
       
			glTexCoord2f(1,1);
			glVertex3f(-.6,-.4,h);

			glTexCoord2f(1,0);
			glVertex3f(2.1,-.4,h);
       
			glTexCoord2f(0,0);
			glVertex3f(2.1,.4,h);
       
			RenderEnd();
		}

		glDisable(GL_CULL_FACE);

		// sr_laggometer;


		REAL f=speed;

		REAL l=Lag();

		glPopMatrix();

		h=cam->CameraZ()*.005+.03;

		if (sn_GetNetState() != nSTANDALONE && sr_laggometer && f*l>.5){
			//&& owner!=::sn_myNetID){
			glPushMatrix();
       
			glColor3f(1,1,1);
			//glDisable(GL_TEXTURE);
			glDisable(GL_TEXTURE_2D);
       
			glTranslatef(0,0,h);
			//glScalef(.5*f,.5*f,.5*f);
			glScalef(f,f,f);

			// move the sr_laggometer ahead a bit
			if (!sr_predictObjects || sn_GetNetState()==nSERVER)
				glTranslatef(l,0,0);

       
			BeginLineLoop();
       
       
			glVertex2f(-l,-l);
			glVertex2f(0,0);
			glVertex2f(-l,l);
			if(l>.2){
				glVertex2f(-2*l+.1,.1);
				glVertex2f(-2*l+.2,0);
				glVertex2f(-2*l+.1,-.1);
			}
			else if (l>.1){
				glVertex2f(-2*l+.1,.1);
				glVertex2f(-l,.2-l);
				glVertex2f(-l,-(.2-l));
				glVertex2f(-2*l+.1,-.1);
			}
       
			RenderEnd();
			glPopMatrix();
		}
		sr_DepthOffset(false);

		glPopMatrix();

	}
}

static REAL fadeOutNameAfter = 5.0f;	/* 0: never show, < 0 always show */
//static int fadeOutNameMode = 1;			// 0: never show, 1: show for fadeOutNameAfter, 2: always show
static bool showOwnName = 0;			// show name on own cycle?

static tSettingItem< bool > sg_showOwnName( "SHOW_OWN_NAME", showOwnName );
//static tSettingItem< int > sg_fadeOutNameMode( "FADEOUT_NAME_MODE", showOwnName )
static tSettingItem< REAL > sg_fadeOutNameAfter( "FADEOUT_NAME_DELAY", fadeOutNameAfter );

void gCycle::RenderName( const eCamera* cam ) {
	if ( !this->player )
		return;
	
	float modelviewMatrix[16], projectionMatrix[16];
	float x, y, z, w;
	float xp, yp, wp;
	float alpha = 0.75;

	if (fadeOutNameAfter == 0) return; /* XXX put that in ::Render() */
	if ( !cam->RenderingMain() ) return; // no name in mirrored image
	if ( !showOwnName && cam->Player() == this->player ) return; // don't show own name

	glPushMatrix();
	/* position sign above center of cycle */
	glTranslatef(0.8, 0, 2.0);
	glGetFloatv(GL_MODELVIEW_MATRIX, modelviewMatrix);
	glGetFloatv(GL_PROJECTION_MATRIX, projectionMatrix);
	glPopMatrix();
	
	/* get coordinates of sign */
	x = modelviewMatrix[12];
	y = modelviewMatrix[13];
	z = modelviewMatrix[14];
	w = modelviewMatrix[15];

	/* multiply by projection matrix */
	xp = projectionMatrix[0] * x + projectionMatrix[4] * y +
		projectionMatrix[8] * z + projectionMatrix[12] * w;
	yp = projectionMatrix[1] * x + projectionMatrix[5] * y +
		projectionMatrix[9] * z + projectionMatrix[13] * w;
	wp = projectionMatrix[3] * x + projectionMatrix[7] * y +
		projectionMatrix[11] * z + projectionMatrix[15] * w;

	if (wp <= 0) {
		/* behind camera */
		timeCameIntoView = 0;
		return;
	}

	xp /= wp;
	yp /= wp;
	yp += rCHEIGHT_NORMAL;// * 2.0;

	if (xp <= -1 || xp >= 1 || yp <= -1 || yp >= 1) {
		/* out of screen */
		timeCameIntoView = 0;
		return;
	}

	/* it is visible */
	
	if (fadeOutNameAfter > 0) {
		REAL now = tSysTimeFloat();
		if (timeCameIntoView == 0)
			timeCameIntoView = now;

		if (now - timeCameIntoView > fadeOutNameAfter) {
			return;
		} else if (now - timeCameIntoView > fadeOutNameAfter - 1) {
			/* start to fade out */
			alpha = 0.75 - (now - timeCameIntoView -
				(fadeOutNameAfter - 1)) * 0.75;
		}
	}

	ModelMatrix();
	glPushMatrix();
	glLoadIdentity();

	ProjMatrix();
	glPushMatrix();
	glLoadIdentity();

	glColor4f(1, 1, 1, alpha);
	DisplayText(xp, yp, rCWIDTH_NORMAL, rCHEIGHT_NORMAL,
			this->player->name, 0, 0);

	ProjMatrix();
	glPopMatrix();

	ModelMatrix();
	glPopMatrix();
}


bool gCycle::RenderCockpitFixedBefore(bool){
	/*
	  if (alive)
	  return true;
	  else{
	  REAL rd=se_GameTime()-deathTime;
	  if (rd<1)
      return true;
	  else{
      REAL d=1.25-rd;
      d*=8;
      if (d<0) d=0;
      glColor3f(d,d/2,d/4);
      glDisable(GL_TEXTURE_2D);
      glDisable(GL_TEXTURE);
      glDisable(GL_DEPTH_TEST);
      glRectf(-1,-1,1,1);
      glColor4f(1,1,1,rd*(2-rd/2));
      DisplayText(0,0,.05,.15,"You have been deleted.");
      return false;
	  }
	  }
	*/
	return true;
}

void gCycle::SoundMix(Uint8 *dest,unsigned int len,
					  int viewer,REAL rvol,REAL lvol){
	if (Alive()){
		/*
		  if (!cycle_run.alt){
		  rvol*=4;
		  lvol*=4;
		  }
		*/

		if (engine)
			engine->Mix(dest,len,viewer,rvol,lvol,speed/(sg_speedCycleSound * SpeedMultiplier()));
    
		if (turning)
			if (turn_wav.alt)
				turning->Mix(dest,len,viewer,rvol,lvol,5);
			else
				turning->Mix(dest,len,viewer,rvol,lvol,1);

		if (spark)
			spark->Mix(dest,len,viewer,rvol*.5,lvol*.5,4);
	}
}
#endif

eCoord gCycle::PredictPosition(){
	eCoord p = pos + dir * (speed * se_PredictTime());
	gSensor s(this,pos, p-pos);
	s.detect(1);
	return s.before_hit;
} 

eCoord gCycle::CamPos()
{
//	return PredictPosition() + dir.Turn(0 ,-skew*z);
	return pos + dir.Turn(0 ,-skew*z);
}

eCoord  gCycle::CamTop()
{
	return dir.Turn(0,-skew);
}


#ifdef POWERPAK_DEB
void gCycle::PPDisplay(){
	int R=int(r*255);
	int G=int(g*255);
	int B=int(b*255);
	PD_PutPixel(DoubleBuffer,
				se_X_ToScreen(pos.x),
				se_Y_ToScreen(pos.y),
				PD_CreateColor(DoubleBuffer,R,G,B));
	/*
	  PD_PutPixel(DoubleBuffer,
	  se_X_ToScreen(pos.x+1),
	  se_Y_ToScreen(pos.y),
	  PD_CreateColor(DoubleBuffer,R,G,B));
	  PD_PutPixel(DoubleBuffer,
	  se_X_ToScreen(pos.x-1),
	  se_Y_ToScreen(pos.y),
	  PD_CreateColor(DoubleBuffer,R,G,B));
	  PD_PutPixel(DoubleBuffer,
	  se_X_ToScreen(pos.x),
	  se_Y_ToScreen(pos.y+1),
	  PD_CreateColor(DoubleBuffer,R,G,B));
	  PD_PutPixel(DoubleBuffer,
	  se_X_ToScreen(pos.x),
	  se_Y_ToScreen(pos.y-1),
	  PD_CreateColor(DoubleBuffer,R,G,B));
	*/
}
#endif





// cycle network routines:
gCycle::gCycle(nMessage &m)
	:eNetGameObject(m),
	 engine(NULL),
	 turning(NULL),
	 spark(NULL),
	 destinationList(NULL),currentDestination(NULL),
	 dirDrive(1,0),
	 skew(0),skewDot(0),speed(5),acceleration(0),
	 rotationFrontWheel(1,0),rotationRearWheel(1,0),heightFrontWheel(0),heightRearWheel(0),
	 alive(1),
	 currentWall(NULL),
	 lastWall(NULL)
//	 currentWallID(0)
{
	windingNumber = 2;

	rubber=0;
	pendingTurns=0;
	turns=0;
	speed=0;
	correctTimeSmooth =0;
	correctDistSmooth =0;
	correctSpeedSmooth =0;
	distance = 0;
	wallContDistance = 5;
	deathTime = 0;

	m >> color_.r;
	m >> color_.g;
	m >> color_.b;

	trailColor_ = color_;

	se_MakeColorValid( color_.r, color_.g, color_.b, 1.0f );
	se_MakeColorValid( trailColor_.r, trailColor_.g, trailColor_.b, .5f );
}
	

void gCycle::WriteCreate(nMessage &m){
	eNetGameObject::WriteCreate(m);
	m << color_.r;
	m << color_.g;
	m << color_.b;
}

void gCycle::WriteSync(nMessage &m){
//	eNetGameObject::WriteSync(m);

	if ( Alive() )
	{
		m << lastTime;
	}
	else
	{
		m << deathTime;
	}
	m << Direction();
	m << Position();

	m << speed;
	m << alive;
	m << distance;
	if (!currentWall || currentWall->preliminary)
		m.Write(0);
	else
		m.Write(currentWall->ID());

	m.Write(turns);
	m.Write(braking);

	// write beginning of wall
	eCoord begWall = pos;
	if ( currentWall )
	{
		begWall = currentWall->beg;
	}
	m << begWall;
}

bool gCycle::SyncIsNew(nMessage &m){
	bool ret=eNetGameObject::SyncIsNew(m);

	REAL dummy2;
	short al;
	unsigned short Turns;
	unsigned short b;

	m >> dummy2;
	m >> al;
	m >> dummy2;
	m.Read(Turns);

	if (!m.End())
		m.Read(Turns);
	else
		Turns=turns;

	if (!m.End())
		m.Read(b);

	eCoord begWall = pos;
	if ( !m.End() )
	{
		m >> begWall;
	}
	else if ( currentWall )
	{
		begWall = currentWall->beg;
	}

	// con << turns << ":" << Turns << ":" << al << "\n";

	//Update(turns,Turns);

	if (turns>Turns){

		if (al==1 && 
			lastAttemptedSyncTime<lastClientsideAction+1.2*sn_Connections[0].ping+.1){
#ifdef DEBUG
			//con << "rejecting sync message.\n";
#endif
			return false;
		}
#ifdef DEBUG
		else   if (ret){
			con << "accepting sync message in spite of obvious latency:"
				<< lastAttemptedSyncTime << ":" << lastClientsideAction << "\n";
		}
#endif
	}  
  
	if (ret || al!=1){
		// con << "accepting..\n";
		return true;
	}
	else
		return false;

	return ret || al!=1;
}

void gCycle::ReadSync(nMessage &m){
	REAL sync_time,sync_dist,sync_speed;
	short sync_alive;
	unsigned short sync_turns=turns,sync_braking=braking,sync_eWall;
	eCoord sync_dir,sync_pos;

	// warning: depends on the implementation of eNetGameObject::WriteSync.
	m >> sync_time;
	m >> sync_dir;
	m >> sync_pos;

	m >> sync_speed;
	m >> sync_alive;
	m >> sync_dist;
	m.Read(sync_eWall);
	if (!m.End())
		m.Read(sync_turns);
	if (!m.End())
		m.Read(sync_braking);

	eCoord begWall = pos;
	if ( !m.End() )
	{
		m >> begWall;
	}
	else if ( currentWall )
	{
		begWall = currentWall->beg;
	}

	gDestination emergency_aft(*this);

#ifdef DEBUG
	if ( sync_alive != alive )
	{
		int x;
		x = 0;
	}

    if (!finite(sync_speed) || !finite(sync_dist))
		st_Breakpoint();
#endif
  
	// all the data was read. check where it fits in our destination list:
	gDestination *bef=gDestination::RightBefore(destinationList, sync_dist);
	gDestination *aft=NULL;
	if (bef){
		aft=bef->next;
		if (!aft)
			aft=&emergency_aft;
	}

	REAL better_distance = sync_dist;

	bool distanceBased = aft && aft != &emergency_aft && Owner()==::sn_myNetID;
  
	if ( aft ){
		better_distance = aft->distance - sqrt((sync_pos-pos).NormSquared());

		/*
		  if (fabs(better_distance - sync_dist) > 100){
		  // con << "Wanring: bad sync.\n";
		  distanceBased = false;
		  }
		*/
	}
  
	if (distanceBased){
		// new way: correct our time and speed

#ifdef DEBUG
		// destination *save = bef;
#endif

		REAL ratio = (better_distance - bef->distance)/
			(aft->distance - bef->distance);

		if (!finite(ratio))
			ratio = 0;

		// TODO: better interpolation
		REAL interpolatedTime = bef->gameTime * (1-ratio) + 
			aft->gameTime * ratio; 
		REAL interpolated_speed = bef->speed * (1-ratio) + aft->speed * ratio; 


		REAL correctTime  = (sync_time  - interpolatedTime);
		REAL correct_speed = (sync_speed - interpolated_speed);
		REAL correct_dist  = (sync_dist - better_distance);

		clamp(correctTime , -10, 10);
		clamp(correct_speed, -10, 10);
		clamp(correct_dist , -10, 10);

		if (correctTime > 0){
			correctTime*=.5;
			correct_speed*=.5;
			correct_dist*=.5;
		}

		correctTimeSmooth  += correctTime;
		correctSpeedSmooth += correct_speed;
		correctDistSmooth  += correct_dist;

#ifdef DEBUG
		if (!finite(correctTime) || !finite(correct_speed))
			st_Breakpoint();
#endif
    
		while (bef){
			bef->gameTime += correctTime;
			bef->speed    += correct_speed;
			//bef->distance += correct_dist;
			bef = bef->next;
		}

		if (fabs(correctTimeSmooth > 5))
			st_Breakpoint();


#ifdef DEBUG
		if (!finite(speed) || !finite(correctTimeSmooth) || fabs(correctTimeSmooth)> 100)
			st_Breakpoint();
#endif

		clamp(correctTimeSmooth,-100,100);
	}
  
	else
    {
		// the sync is newer than our last turn. correct our position.
		correctTimeSmooth = 0;
     
		REAL t=(dirDrive*sync_dir);
		if (fabs(t)>.2){
			if ( Owner()==sn_myNetID && ( currentWall && sync_time < currentWall->tBeg + 2.0f ) )
			{
#ifdef DEBUG
				con << "Warning! Turned cycle!\n";
#endif
				// server cannot simply turn us!
//				AddDestination();
				sync_dir = dirDrive;
			}
			else
			{
				if (turning)
					turning->Reset();

				eCoord crossPos = sync_pos;
				REAL crossDist = sync_dist;

				if(currentWall){
					// make sure the position lies on the last path
					if (Owner() != ::sn_myNetID)
					{
						crossPos = crossPos - dirDrive * eCoord::F(dirDrive,crossPos-sync_pos);

						REAL distplace = eCoord::F(sync_dir,crossPos-pos);
						crossPos = crossPos - sync_dir * distplace;
						crossDist -= distplace;
					}


					lastWall=currentWall;
					currentWall->real_Update(sync_time,crossPos,true);
					if (Owner() == ::sn_myNetID)
						currentWall->CopyIntoGrid();
					else
					{
					  tControlledPTR< gNetPlayerWall > bounce( currentWall );
					}
					currentWall=NULL;
				}
				distance = sync_dist;
				currentWall=new gNetPlayerWall
					(this,crossPos,sync_dir,sync_time,crossDist);
	/*
				if (sn_GetNetState()!=nCLIENT)
					currentWallID=currentWall->ID();
	*/	 
			}
		}
      
		// side bending
		skewDot+=4*t;
      
		if ( currentWall )
		{
			currentWall->beg = begWall;
		}

		REAL ts = sync_time - lastTime;

		eCoord intPos = pos + dirDrive * (ts * speed + .5 * ts*ts*acceleration);
		REAL  int_speed = speed + acceleration*ts;

		dirDrive = sync_dir;
      
		if (Owner()!=::sn_myNetID){
			pos = sync_pos;
			speed = sync_speed;
			REAL oldTime = lastTime;
			lastTime = sync_time;
			clientside_action();
			forceTime=true;
			if ( sync_alive > 0)
			{
				TimestepThis(oldTime, this);
			}
			forceTime=false;
		}
		else if (fabs(t)>.1 || (intPos-sync_pos).NormSquared() > 40.0f){
			pos = sync_pos + pos - intPos;
#ifdef DEBUG
			if (!finite(int_speed))
				st_Breakpoint();
#endif
			speed = sync_speed + speed - int_speed;
		}
		else{
			correctPosSmooth = sync_pos - intPos;
			correctSpeedSmooth = sync_speed - int_speed;
		}

		braking = sync_braking;
		distance = sync_dist - speed * ts - acceleration * ts*ts*.5;

		correctTimeSmooth = 0;
    }
  
	if (alive && sync_alive!=1){
		if ( currentWall )
		{
			currentWall->real_Update( sync_time, sync_pos, false );
			currentWall->CopyIntoGrid();
			currentWall = NULL;
		}
		
		deathTime=sync_time;
		//    eEdge::SeethroughHasChanged();
		pos = sync_pos;

		new gExplosion(grid, sync_pos,sync_time,color_);
	}
	alive=sync_alive;
  
	sn_Update(turns,sync_turns);

	if (fabs(correctTimeSmooth > 5))
		st_Breakpoint();

	// calculate new winding number. Try to change it as little as possible.
	int halfTurn   = Grid()->WindingNumber() / 2;
	int newWinding = Grid()->DirectionWinding(dirDrive);
	int difference = ((newWinding - windingNumber + halfTurn) % Grid()->WindingNumber()) - halfTurn;
	while (difference < -halfTurn)
		difference += Grid()->WindingNumber();

	windingNumber += difference;
}

/*
void gCycle::old_ReadSync(nMessage &m){
  REAL oldTime=lastTime;
  eCoord oldpos=pos;
  //+correctPosSmooth;
  //correctPosSmooth=0;
  eCoord olddir=dir;
  eNetGameObject::ReadSync(m);

  REAL t=(dirDrive*dir);
  if (fabs(t)>.2){
#ifdef DEBUG
    if (owner==sn_myNetID)
      con << "Warning! Turned cycle!\n";
#endif
    turning.Reset();
  }

  // side bending
  skewDot+=4*t;


  dirDrive=dir;
  dir=olddir;

  m >> speed;
  short new_alive;
  m >> new_alive;
  if (alive && new_alive!=1){
    new gExplosion(pos,oldTime,r,g,b);
    deathTime=oldTime;
    eEdge::SeethroughHasChanged();
  }
  alive=new_alive;

  m >> distance;

  // go to old time frame

  eCoord realpos=pos;
  REAL realtime=lastTime;
  REAL realdist=distance;

  if (currentWall)
    lastWall=currentWall;

  m.Read(currentWallID);

  unsigned short Turns;
  if (!m.End())
    m.Read(Turns);
  else
    Turns=turns;

  if (!m.End())
    m.Read(braking);
  else
    braking=false;

  TimestepThis(oldTime,this);

  if (!currentWall || fabs(t)>.3 || currentWall->inGrid){
    //REAL d=eCoord::F(pos-realpos,olddir);
    //eCoord crosspos=realpos+olddir*d;
    //d=eCoord::F(oldpos-crosspos,dir);
    //crosspos=crosspos+dir*d;

    eCoord crosspos=realpos;

    if (currentWall){
      currentWall->Update(realtime,crosspos);
      //currentWall->Update(realtime,realpos);
      currentWall->CopyIntoGrid();
    }
    //con << "NEW\n";
    currentWall=new gNetPlayerWall
      (this,crosspos,dirDrive,realtime,realdist);
  }


  // smooth correction
  if ((oldpos-pos).NormSquared()<4 && fabs(t)<.5){
    correctPosSmooth=pos-oldpos;
    pos=oldpos;
  }
  


#ifdef DEBUG
  //int old_t=turns;
  //if(sn_Update(turns,Turns))
  //con << "Updated turns form " << old_t << " to " << turns << "\n";
#endif
  sn_Update(turns,Turns);
}
*/

void gCycle::ReceiveControl(REAL time,uActionPlayer *act,REAL x){
	forceTime=true;
	TimestepThis(time,this);
	Act(act,x);
	forceTime=false;
}

void gCycle::PrintName(tString &s) const
{
	s << "gCycle nr. " << ID();
	if ( this->player )
	  {
		s << " owned by ";
		this->player->PrintName( s );
	  }
}

bool gCycle::ActionOnQuit()
{
//  currentWall = NULL;
//  lastWall = NULL;
	if ( sn_GetNetState() == nSERVER )
	{
		TakeOwnership();
		Kill();
		return false;
	}
	else
	{
	  return true;
	}
}


nDescriptor &gCycle::CreatorDescriptor() const{
	return cycle_init;
}

void gCycle::AddDestination(gDestination *dest){
	//  con << "got new dest " << dest->position << "," << dest->direction 
	// << "," << dest->distance << "\n";

	dest->InsertIntoList(&destinationList);

	// if the new destination was inserted at the end of the list
	//if (!currentDestination && !dest->next && 
	//if ((dest->distance >= distance || (!currentDestination && !dest->next)) && 

	if (dest->next && dest->next->hasBeenUsed){
		delete dest;
		return;
	}
    

	if ((!currentDestination || currentDestination->distance > dest->distance) && 
		sn_GetNetState()!=nSTANDALONE && Owner()!=::sn_myNetID){
		currentDestination=dest;
		// con << "setting new cd\n";
	}

	//  if (sn_GetNetState()==nSERVER || ::sn_myNetID == owner)
	if (sn_GetNetState()==nCLIENT && ::sn_myNetID == Owner())
		BroadCastNewDestination(this,dest);
}

void gCycle::AddDestination()
{
	gDestination* dest = tNEW(gDestination)(*this);
//	dest->position = dest->position + dest->direction.Turn( 0, 10.0f );
	AddDestination( dest );
}


// ************************************************************




// ************************************************************


// the time the cycle walls stay up ( negative values: they stay up forever )
void 	gCycle::SetWallsStayUpDelay	( REAL delay )
{
	c_pwsud->Set( delay );
}

// the maximum total length of the walls
void 	gCycle::SetWallsLength			( REAL length)
{
	c_pwl->Set( length );
}

// the radius of the holes blewn in by an explosion
void 	gCycle::SetExplosionRadius		( REAL radius)
{
	c_per->Set( radius );
}

/*
void gCycle::AddRef()
{
	eNetGameObject::AddRef();
}

void gCycle::Release()
{
	eNetGameObject::Release();
}
*/
