# library-revisions.sh: list revisions in the revision library
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "list revisions in the revision library\\n"
                printf "usage: library-revisions [options] [[archive]/version]"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf " -r --reverse                  reverse the listing\\n"
		printf " -s --summary                  print a summary of each patch\\n"
		printf " -c --creator                  print the creator id of each patch\\n"
		printf " -D --date                     print the date of each patch\\n"
		printf " --full                        print full patch level names\\n"
                printf "\\n"
		printf "List all revisions within a particular archive/version with\\n"
		printf "records in the revision library.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

reverse=
summary=
creator=
date=
full=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "library-find: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "library-find: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -r|--reverse)	shift
    			reverse=-r
			;;

    -s|--summary)	shift
    			summary=--summary
			;;

    -c|--creator)	shift
    			creator=--creator
			;;

    -D|--date)		shift
    			date=--date
			;;

    --full)		shift
    			full=--full
			;;

    -*)                 printf "library-revisions: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: library-revisions [options] [[archive]/version]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  versionspec="$1"
  shift
fi

################################################################
# Sanity Check and Process Defaults
# 

library="`larch my-revision-library -e library-revisions`"

if larch valid-package-name "$versionspec" ; then

  v="`larch library-versions --reverse \"$versionspec\" | head -n 1`"

  if test -z "$v" ; then
    printf "\\n" 1>&2
    printf "library-revisions: no versions on branch\\n" 1>&2
    printf "  branch: %s\\n" "$versionspec" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi

  versionspec="$v"
fi


if test ! -z "$versionspec" ; then 
  larch valid-package-name -e library-revisions --vsn --tolerant "$versionspec"
  archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$versionspec\"`"
  category="`larch parse-package-name --basename \"$versionspec\"`"
  branch="`larch parse-package-name \"$versionspec\"`"
  version="`larch parse-package-name --package-version \"$versionspec\"`"
  if larch valid-package-name --lvl "$versionspec" ; then
    lvl="`larch parse-package-name --lvl \"$versionspec\"`"
  else
    lvl=
  fi
else
  tree_default="`larch tree-version`"
  archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$tree_default\"`"
  category="`larch parse-package-name --basename \"$tree_default\"`"
  branch="`larch parse-package-name \"$tree_default\"`"
  version="`larch parse-package-name --package-version \"$tree_default\"`"
fi

################################################################
# Do It
# 

cd "$library"

if test ! -d "$archive/$category/$branch/$version" ; then
  printf "\\n" 1>&2
  printf "library-revisions: version not found in library\\n" 1>&2
  printf "  archive: %s\\n" "$archive" 1>&2
  printf "  version: %s\\n" "$version" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

cd "$archive/$category/$branch/$version"

basere="([a-zA-Z]([a-zA-Z0-9]*(-[a-zA-Z0-9]+)?)*)"
vsnre="([0-9]+\\.[0-9]+)"
patchre="(base-0|patch-[0-9]+|version-0|versionfix-[0-9]+)"

if test ! -z "$lvl" ; then

  lvls="$version--$lvl"

  if test ! -d "$version--$lvl" ; then
    printf "\\n" 1>&2
    printf "library-revisions: revision not in library\\n" 1>&2
    printf "  revision %s\\n" "$version--$lvl" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi
  
else

  lvls="`ls \
          | grep -E \"^($basere)(--$basere)?(--$vsnre)(--$patchre)\$\" \
          | sed -e \"s/.*--\\([^-]*-[0-9][0-9]*\\)/\\1-&/\" \
          | sort  -t- -k 1,1${reverse#-} -k 2,2${reverse#-}n \
          | sed -e \"s/[^-]*-[^-]*-//\"`"

fi


for f in $lvls ; do


  if test ! -z "$full" ; then
    printf "%s\\n" "$f"
  else
    printf "%s\\n" "$f" | sed -e 's/.*--//'
  fi

  if test ! -z "$summary$creator$date" ; then
    cat $f/,,patch-set/=log.txt \
    | awk -v summary="$summary" \
  	  -v creator="$creator" \
	  -v date="$date" \
	  -f "$ARCH_SUBCMD_ROOT/patch-logs/patch-list-description.awk"
  fi

done


# tag: Tom Lord Sun Jan 20 01:11:00 2002 (library/library-revisions.sh)
#
