# 
# file-syntax-filter.sh: filter input files
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "filter input files\\n"
		printf "usage: file-syntax-filter [options]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -e --errname name             use ERRNAME for error messages\\n"
		printf "\\n"
		printf " --sh-comments                 remove line suffixes beginning with #\\n"
		printf " --blank-lines                 remove empty lines\\n"
		printf " --trailing-spaces             remove whitespace suffixes\\n"
		printf "\\n"
		printf "Copy standard input to standard output, processing syntax in a generic\\n"
		printf "way\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

errname=
sh_comments=
blank_lines=
trailing_spaces=

while test $# -ne 0 ; do

  case "$1" in 

    --sh-comments)	shift
    				sh_comments=--sh-comments
				;;

    --blank-lines)	shift
    				blank_lines=--blank-lines
				;;

    --trailing-spaces)	shift
    				trailing_spaces=--trailing-spaces
				;;

    --)			shift
    			break
			;;
			
    -e|--errname)	shift
    			if test $# -eq 0 ; then
			  printf "file-syntax-filter: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -*)			printf "file-syntax-filter: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 0 ; then
  printf "usage: file-syntax-filter [options]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

################################################################
# Sanity Check and Process Defaults
# 


################################################################
# Filter Input
# 

prog=

if test ! -z "$sh_comments" ; then
  prog="`printf \"%s\\ns/#.*//\\n\" \"$prog\"`"
fi

if test ! -z "$blank_lines" ; then 
  prog="`printf \"%s\\n/^[[:space:]]*$/d\\n\" \"$prog\"`"
fi

if test ! -z "$trailing_spaces" ; then 
  prog="`printf \"%s\\ns/[[:space:]]*$//\\n\" \"$prog\"`"
fi

prog="`printf \"{\\n%s\\n}\\n\" \"$prog\"`"

sed -e "$prog"

# tag: Tom Lord Mon Dec 31 17:12:10 2001 (input/file-syntax-filter.sh)
#
