
#define LOCAL_DEBUG
#include "debug.h"

#include "pkgimport.h"
#include "header.h"
#include "dirwalk.h"
#include "meta.h"
#include "acfg.h"
#include "filereader.h"
#include "dlcon.h"
#include "csmapping.h"

#include <string>
//#include <iostream>
#include <cstdio>
#include <errno.h>

using namespace MYSTD;

#define SCACHEFILE (acfg::cachedir+sPathSep+"_impkeycache")
extern bool bSigTaskAbort;
extern pthread_mutex_t abortMx;

/*
 * Algorithm:
 *
 * Scan the _import directory context and make a dictionary with
 * fingerprint(size, cs) -> (path, bUsed, mtime)
 * in the ProcessRegular callback function
 *
 * Parse the vol. files, do work, and mark used file with bUsed
 *
*/

pkgimport::pkgimport(int fd) : expiration(fd), m_bPickIfiles(false)
{
}

bool pkgimport::ProcessOther(const string &sPath, const struct stat &stinfo) 
{
	// NOOP;
	return true;
}

// Evil stuff. Tries to guess the appropriate checksum type 
// and returns a pointer to static parametrized fingerprint object
inline tFingerprint *CsTypeGuessing(const string & sPath)
{
	static tFingerprint fprBuf;

	static string sDiffHint(".diff" SZPATHSEP);
	tStrPos pos;
	if (endsWithSzAr(sPath,".gz") && ( sPath.find(sDiffHint)!=stmiss ||
			( stmiss!=(pos=sPath.rfind(SZPATHSEP "20")) && sPath.find_first_not_of("0123456789.-", pos+3)==sPath.size()-2)))
	{
		fprBuf.csType=CSTYPE_SHA1;
		fprBuf.bUnpack=true;
	}
	else
	{
		fprBuf.csType=CSTYPE_MD5;
		fprBuf.bUnpack=false;
	}
	return &fprBuf;
}


bool pkgimport::ProcessRegular(const string &sPath, const struct stat &stinfo)
{

	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return false;
	}
	if(endsWithSzAr(sPath, ".head"))
		return true;
	
	//string sRelName=fileNode->sPath.substr(acfg::cachedir.length()+1);
	rechecks::eFileKind kind = rechecks::GetFiletype(sPath);
	if(m_bPickIfiles)
	{
		if(0==sPath.compare(m_nDropPrefLen, 1, "_"))
			return true; // not for us, also includes _import
		
		if( rechecks::FILE_INDEX == kind )
		{
			//cout << "Is ifile\n";
			m_volatileFiles.insert(sPath);
		}
	}
	else /* also try to import ifiles when matched, why not?! if (rechecks::FILE_PKG == kind) */
	{
		// get a fingerprint by checksumming if not already there from the fpr cache

		if(m_precachedList.find(sPath)!=m_precachedList.end())
			return true; // have that already, somewhere...
		
		tFingerprint *fpr=CsTypeGuessing(sPath);
		if (fpr->ReadCsFromFile(sPath))
		{
			SendChunk(string("<font color=blue>Checked ")+sPath
					+" (fingerprint created)</font><br>\n");
		}
		else
		{
			SendChunk(string("<font color=red>Error checking ")+sPath
					+"</font><br>\n");
			return true;
		}
		
		// add this entry immediately if needed and get its reference
		tImpFileInfo & node = m_importMap[*fpr];
		if (node.sPath.empty())
		{ // fresh, just added to the map
			node.sPath=sPath;
			node.mtime=stinfo.st_mtime;
		}
		else if(node.sPath!=sPath)
		{
			SendChunk(string("<font color=orange>Duplicate found, ")+sPath
								+" vs. "+node.sPath+", ignoring new entry.</font><br>\n");
		}
	}
	return true;
}

// linking not possible? different filesystems?
bool FileCopy(const string &from, const string to)
{
	acbuf buf;
	buf.init(50000);
	int in(-1), out(-1);

	in=open(from.c_str(), O_RDONLY);
	if (in<0) // error, here?!
		return false;

	while (true)
	{
		int err;
		err=buf.sysread(in);
		if (err<0)
		{
			if (err==-EAGAIN || err==-EINTR)
				continue;
			else
				goto error_copying;
		}
		else if (err==0)
			break;
		// don't open unless the input is readable, for sure
		if (out<0)
		{
			out=open(to.c_str(), O_WRONLY | O_CREAT | O_TRUNC, 00644);
			if (out<0)
				goto error_copying;
		}
		err=buf.syswrite(out);
		if (err<=0)
		{
			if (err==-EAGAIN || err==-EINTR)
				continue;
			else
				goto error_copying;
		}

	}

	forceclose(in);
	forceclose(out);
	return true;

	error_copying: 

	checkforceclose(in);
	checkforceclose(out);
	return false;
}
bool pkgimport::LinkOrCopy(const MYSTD::string &from, const MYSTD::string &to)
{
	mkbasedir(to);
	
	//ldbg("Moving " <<from<< " to " << to);
	
	// hardlinks don't work with symlinks properly, create a similar symlink then
	struct stat stinfo, toinfo;
	
	if(0==lstat(from.c_str(), &stinfo) && S_ISLNK(stinfo.st_mode))
	{
		char buf[PATH_MAX+1];
		buf[PATH_MAX]=0x0;
		
		if (!realpath(from.c_str(), buf))
			return false;
		return (0==symlink(buf, to.c_str()));		
	}
	else
	{
		if(0!=stat(from.c_str(), &stinfo))
				return false;
		
		if(0==stat(to.c_str(), &toinfo))
		{
			// same file, don't care
			if(stinfo.st_dev==toinfo.st_dev && stinfo.st_ino == toinfo.st_ino)
				return true;
			
			// unlink, this file must go. If that fails, we have weird permissions anyway.
			if(0!=unlink(to.c_str()))
				return false;
		}
		if (0==link(from.c_str(), to.c_str()))
			return true;

		SendChunk("(couldn't link, copying contents)");
		if(!FileCopy(from,to))
			return false;
		
		// make sure the copy is ok
		return (0 == stat(from.c_str(), &stinfo) 
				&& 0 == stat(to.c_str(), &toinfo)
				&& toinfo.st_size==stinfo.st_size);
		
	}

	return false;
}

			
void pkgimport::Action(const string &cmd) 
{
// TODO: import path from cmd?
	m_sSrcPath=acfg::cachedir+sPathSep+"_import";
	
	SendChunk(string("Importing from ")+m_sSrcPath+" directory, scanning...<br>\n");
	
	_LoadKeyCache(SCACHEFILE);
	if(!m_precachedList.empty())
		SendChunk(string("Loaded ")+ltos(m_importMap.size())
				+(m_precachedList.size()==1 ? " entry" : " entries")
				+" from the fingerprint cache<br>\n");
	
	m_bPickIfiles=true;
	DirectoryWalk(acfg::cachedir, this);

	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return;
	}
	
	if(m_volatileFiles.empty())
	{
		SendChunk("<font size=0 color=red>No index files detected. Unable to continue, cannot map files to internal locations.</font>");
		return;
	}

	_UpdateVolatileFiles();

	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return;
	}
	
	if(m_bErrAbort && m_nErrorCount>0)
	{
		SendChunk("Found errors during processing, aborting as requested.");
		return;
	}
	
	m_bPickIfiles=false;
	DirectoryWalk(m_sSrcPath, this);

	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return;
	}
	
	if(m_importMap.empty())
	{
		SendChunk("No appropriate files found in the _import directory.<br>\nDone.<br>\n");
		return;
	}
	
	_ParseVolatileFilesAndHandleEntries();

	{
		lockguard g(&abortMx);
		if(bSigTaskAbort)
			return;
	}
	
	FILE *cfh = fopen(SCACHEFILE.c_str(), "wb");
	if (!cfh)
	{
		unlink(SCACHEFILE.c_str());
		cfh=fopen(SCACHEFILE.c_str(), "wb");
	}
	if(cfh)
		fprintf(cfh, "FMT3\n");
	
	for(tImportMap::const_iterator it=m_importMap.begin();
	it!=m_importMap.end();it++)
	{
		// delete imported stuff, and cache the fingerprint only when file was deleted
		
		if(it->second.bFileUsed && 0==unlink(it->second.sPath.c_str()))
				continue;
		
		// deal with unpacked checksums from pdiffs? don't care for now
		if(cfh)
		{
			fprintf(cfh, 
					"%"PRIu64"\n%d\n%d\n%s\n"
					"%s\n%"PRIu64"\n\n", 
					(uint64_t) it->first.size,
					(int) it->first.csType,
					(int) it->first.bUnpack,
					it->first.GetCsAsString().c_str(),
					it->second.sPath.c_str(),
					(uint64_t) it->second.mtime	);
		}
	}
	checkForceFclose(cfh);
}

#if 0
void HttpEscape(string &s)
{
	tStrPos p;
	while(stmiss!=(p=s.find_first_of("~ ")))
	{
		switch(s[p])
		{
		case('~'): s.replace(p, 1, "%7e"); break;
		case(' '): s.replace(p, 1, "&nbsp"); break;
		}
	}
}

extern uint_fast16_t hexmap[];

inline void UrlDecode(string &s)
{
	for(string::size_type i=0; i<s.length(); i++)
	{
		if(s[i]!='%')
			continue;
		if(i>s.length()-3) // cannot start another sequence here (too short), keep as-is
			continue;
		char decoded=16*hexmap[(unsigned char) s[i+1]] + hexmap[(unsigned char) s[i+2]];
		s.replace(i, 3, 1, decoded);
	}
}

#endif

void pkgimport::_HandlePkgEntry(const tRemoteFileInfo &entry)
{
	
	//if(entry.sFileName.find("unp_1.0.15.tar.gz")!=stmiss)
	//	int wtf=1;
	//typedef MYSTD::map<tFingerprint, tImpFileInfo, ltfingerprint> tImportMap;
	tImportMap::iterator hit = m_importMap.find(entry.fpr);
	if (hit==m_importMap.end())
		return;
	
	string sDest=entry.sDirectory+entry.sFileName;
	string sDestHead=sDest+".head";
	const string &sFrom=hit->second.sPath;

	SendChunk(string("<font color=green>HIT: ")+sFrom
			+ "<br>\nDESTINATION: "+sDest+"</font><br>\n");

	// linking and moving would shred them when the link leads to the same target
	struct stat tmp1, tmp2;

	if (0==stat(sDest.c_str(), &tmp1) && 0==stat(sFrom.c_str(), &tmp2)
			&& tmp1.st_ino==tmp2.st_ino&& tmp1.st_dev==tmp2.st_dev)
	{
		//cerr << "Same target file, ignoring."<<endl;
		hit->second.bFileUsed=true;
		SendChunk("<font color=orange>Same file exists</font><br>\n");
		if(0!=access(sDestHead.c_str(), F_OK))
		{
			SendChunk("<font color=orange>Header is missing, will recreate...</font><br>\n");
			goto gen_header;
		}
		return;
	}

	unlink(sDest.c_str());
	
	if (!LinkOrCopy(sFrom, sDest))
	{
		SendChunk("<font color=red>ERROR: couldn't link or copy file.</font><br>\n");
		return;
	}

	gen_header:
	unlink(sDestHead.c_str());

	header h;
	h.frontLine="HTTP/1.1 200 OK";
	if(entry.fpr.bUnpack)
	{
		static struct stat stbuf;
		if(0!=stat(sFrom.c_str(), &stbuf))
			return; // weird...
		h.set(header::CONTENT_LENGTH, stbuf.st_size);
	}
	else
		h.set(header::CONTENT_LENGTH, entry.fpr.size);
	
	h.type=header::ANSWER;
	if (h.StoreToFile(sDest+".head")<=0)
	{
		aclog::err("Unable to store generated header");
		return; // junk may remain but that's a job for cleanup later
	}
	hit->second.bFileUsed=true;
}


void pkgimport::_LoadKeyCache(const string & sFileName)
{
	filereader reader;
	string sLine;
	if(!reader.OpenFile(sFileName)||!reader.GetOneLine(sLine)||sLine!="FMT3")
		return;

	/*typedef MYSTD::map<tFingerprint, tImpFileInfo, ltfingerprint> tImportMap;

	fprintf(cfh, 
			"%"PRIu64"\n%d\n%d\n%s\n"
			"%s\n%"PRIu64"\n", 
			(uint64_t) it->first.size,
			(int) it->first.csType,
			(int) it->first.bUnpack,
			it->first.GetCsAsString().c_str(),
			it->second.sPath.c_str(),
			(uint64_t) it->second.mtime	);
	*/

#define _getline { if(!reader.GetOneLine(sLine)) break; }

	tImpFileInfo info;
	tFingerprint fpr;
	
	while(true)
	{
		info.bFileUsed=false;
		_getline
		fpr.size=atoofft(sLine.c_str());
		_getline
		fpr.csType = (CSTYPES) atoi(sLine.c_str());
		_getline
		fpr.bUnpack=(0!=atoi(sLine.c_str()));
		_getline
		fpr.ReadCsFromString(sLine);
		_getline
		info.sPath=sLine;
		_getline
		info.mtime=atol(sLine.c_str());
		_getline
		if(!sLine.empty())
			break;
			
		struct stat stbuf;
		if(0==stat(info.sPath.c_str(), &stbuf) 
				&& info.mtime==stbuf.st_mtime
				&& (fpr.size==stbuf.st_size || fpr.bUnpack))
		{
			m_importMap[fpr]=info;
			m_precachedList.insert(info.sPath);
		}
		
	}
}


/*
void tFInfo::Dump(FILE *fh, const string & second)
{
	fprintf(fh, "%s;%lu;%lu\n%s\n", sPath.c_str(), nSize, nTime, second.c_str());
}

void pkgimport::_GetCachedKey(const string & sPath, const struct stat &stinfo, string &out)
{
	char buf[PATH_MAX+5+3*sizeof(unsigned long)];
	sprintf(buf, "%s\n%lu\n%lu\n", sPath.c_str(), (unsigned long) stinfo.st_size,
			(unsigned long) stinfo.st_mtime);
	tStringMap::const_iterator it=m_cacheMap.find(buf);
	if(it==m_cacheMap.end())
		out.clear();
	else
		out=it->second;
}

 */
