
#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"
#include "fileitem.h"
#include "header.h"
#include "acfg.h"
#include "acbuf.h"
#include "fileio.h"

#include <errno.h>

using namespace MYSTD;

header const * fileitem::GetHeaderUnlocked()
{
	return &m_head;
}

fileitem::fileitem(MYSTD::string sPath) :
	condition(),
	m_bCheckFreshness(true),
	m_nIncommingCount(0),
	m_nSizeSeen(0),
	m_sPath(acfg::cachedir+sPathSep+sPath),
	m_sKey(sPath),
	m_nSizeChecked(0),
	m_filefd(-1),
	m_nDlRefsCount(0),
	status(FIST_FRESH),
	m_bAllowStoreData(true)
{

	// not using hexmap, just converting only few allowed chars

	for (int i = 0; i < int(m_sPath.length()) - 3; i++)
	{
		if (m_sPath[i] == '%')
		{
			switch (m_sPath[i + 1])
			{
			case '7':
				switch (m_sPath[i + 2])
				{
				case 'E':
				case 'e':
					m_sPath.replace(i, 3, "~", 1);
					break;
				}
				break;
			case '5':
				switch (m_sPath[i + 2])
				{
				case 'B':
				case 'b':
					m_sPath.replace(i, 3, "[", 1);
					break;
				case 'D':
				case 'd':
					m_sPath.replace(i, 3, "]", 1);
					break;
				}
				break;
			default:
				continue;
			}
		}
	}
	//ldbg("item created, m_sPath: "<< m_sPath);
}

void fileitem::AddDownloaderRef()
{
	setLockGuard;
	m_nDlRefsCount++;
}

void fileitem::DelDownloaderRef(const string &sReason)
{
	setLockGuard;
	
	m_nDlRefsCount--;
	if(m_nDlRefsCount>0)
		return; // someone will care...
	
	notifyAll();
	if (status<FIST_COMPLETE)
	{
		status=FIST_ERROR;
		m_head.clear();
		m_head.frontLine=string("HTTP/1.1 ")+sReason;
		m_head.type=header::ANSWER;

		if (acfg::debug>1)
			aclog::misc(string("Download of ")+m_sKey+" aborted");
	}
	checkforceclose(m_filefd);
}

uint64_t fileitem::GetTransferCount()
{
	setLockGuard;
	uint64_t ret=m_nIncommingCount;
	m_nIncommingCount=0;
	return ret;
}

int fileitem::GetFileFd() {
	LOGSTART("fileitem::GetFileFd");
	setLockGuard;
	if(status<FIST_DLGOTHEAD)
		return -1;
	ldbg("Opening " << m_sPath);
	int fd=open(m_sPath.c_str(), O_RDONLY);
	
#ifdef HAVE_FADVISE
	// optional, experimental
	if(status==FIST_COMPLETE)
		posix_fadvise(fd, 0, m_nSizeChecked, POSIX_FADV_SEQUENTIAL);
#endif

	return fd;
}


FiStatus fileitem::Setup(bool bCheckFreshness, bool bIgnoreCached) {
	
	setLockGuard;
	if(status>FIST_FRESH)
		return status;
	
	status=FIST_INITED;
	struct stat stbuf;
	
	m_bCheckFreshness = bCheckFreshness;
	
	if(bIgnoreCached)
		return status;
	
	if(m_head.LoadFromFile(m_sPath+".head") >0 && m_head.type==header::ANSWER )
	{
		if(200 != m_head.getStatus())
			goto error_clean;
		
		if (0==::stat(m_sPath.c_str(), &stbuf))
			m_nSizeSeen=stbuf.st_size;
		

		// some plausibility checks
		if(m_bCheckFreshness)
		{
			const char *p=m_head.h[header::LAST_MODIFIED];
			if(!p)
				goto error_clean; // suspicious, cannot use it
		}
		else
		{
			// non-volatile files, so could accept the length, do some checks first
			const char *pContLen=m_head.h[header::CONTENT_LENGTH];
			if(pContLen)
			{
				off_t nContLen=atoofft(pContLen); // if it's 0 then we assume it's 0
				
				// file larger than it could ever be?
				if(nContLen < m_nSizeSeen)
					goto error_clean;
				
				// accept length for non-volatile files...
				m_nSizeChecked=m_nSizeSeen;
				// ... are they even complete? a 0 value also looks weird, try to continue
				if(m_nSizeSeen == nContLen && nContLen>0)
					status=FIST_COMPLETE;	
			}
			else
			{
				// no content length known, assume it's ok
				m_nSizeChecked=m_nSizeSeen;
			}				
		}
	}
	else // -> no .head file
	{
		if(bCheckFreshness)
		{
			// maybe there is some left-over without head file? Don't thrust volatile data, but otherwise try to reuse?
			// No, evil, deletes too fast and there is no way back if redownload fails unlink(m_sPath.c_str());
		}
		else if (0==::stat(m_sPath.c_str(), &stbuf))
			m_nSizeSeen=stbuf.st_size;
	}
	return status;

	error_clean:
			unlink((m_sPath+".head").c_str());
			m_head.clear();
			m_nSizeSeen=0;
			status=FIST_INITED;
			return status; // unuseable, to be redownloaded
}

inline void _LogWithErrno(const char *msg, const string & sFile)
{
	aclog::errnoFmter f;
	aclog::err( (sFile+" storage error ["+msg+"]: "+f.msg).c_str());
}

bool fileitem::DownloadStartedStoreHeader(const header & h)
{
	LOGSTART("fileitem::DownloadStartedStoreHeader");

	setLockGuard;

	USRDBG(5, "Download started, storeHeader for " << m_sKey << ", current status: " << status);
	
	if(status > FIST_DLPENDING) // already started? error? whatever
		return false;
	
	if(status<FIST_DLGOTHEAD)
		status=FIST_DLGOTHEAD; // assume that for now, may become error state...
	
	m_nIncommingCount+=h.m_nEstimLength;

	// optional optimization: hints for the filesystem resp. kernel
	off_t hint_start(0), hint_length(0);
	
	// status will change, most likely... ie. BOUNCE action
	notifyAll();

#define SETERROR(x) { m_head.frontLine="HTTP/1.1 "; m_head.frontLine+=x; status=FIST_ERROR; _LogWithErrno(x, m_sPath); }
#define SETERRORKILLFILE(x) { SETERROR(x); goto kill_file; }
#define BOUNCE(x) { SETERROR(x); _LogWithErrno(x, m_sPath); return false; }
	
	int code=h.getStatus();
	ldbg("http code: "<< code);
	bool bStoreData(false);
	string sHeadPath=m_sPath+".head";
	
	if(200 == code)
	{
		bStoreData=m_bAllowStoreData;
		m_nSizeChecked=0;
		m_head=h;
		const char *p=h.h[header::CONTENT_LENGTH];
		hint_length=p ? atoofft(p) : 0;
	}
	else if(206 == code)
	{
		if(m_nSizeSeen<=0)
		{
			// wtf? Cannot have requested partial content
			BOUNCE("500 Unexpected Partial Response");
		}
		/*
		 * Range: bytes=453291-
		 * ...
		 * Content-Length: 7271829
		 * Content-Range: bytes 453291-7725119/7725120
		 */
		const char *p=h.h[header::CONTENT_RANGE];
		if(!p)
			BOUNCE("500 Missing Content-Range in Partial Response");
		off_t myfrom, myto, mylen;
		int n=sscanf(p, "bytes " OFF_T_FMT "-" OFF_T_FMT "/" OFF_T_FMT, &myfrom, &myto, &mylen);
		if(n<=0)
			n=sscanf(p, "bytes=" OFF_T_FMT "-" OFF_T_FMT "/" OFF_T_FMT, &myfrom, &myto, &mylen);
		
		ldbg("resuming? n: "<< n << " und myfrom: " <<myfrom << 
				" und myto: " << myto << " und mylen: " << mylen);
		if(n!=3  // check for nonsense
				|| myfrom != m_nSizeSeen-1
				|| myfrom<0 || mylen<0)
			BOUNCE("500 Server reports illegal range");
	
		m_nSizeChecked=myfrom;
		bStoreData=m_bAllowStoreData;
		
		hint_start=myfrom;
		hint_length=mylen;

		m_head=h;
		m_head.frontLine="HTTP/1.1 200 OK";
		m_head.del(header::CONTENT_RANGE);
		m_head.set(header::CONTENT_LENGTH, mylen);
		m_head.set(header::XORIG, h.h[header::XORIG]);
	}
	else if(416 == code) // that's always bad; it cannot be complete (-1 trick) -> kill cached file ASAP
	{
		SETERRORKILLFILE("503 Server disagrees on file size, cleaning up");
	}
	else
	{
		bStoreData=false;
		// have a clear one with just the error message
		m_head.frontLine=h.frontLine;
		m_head.type=header::ANSWER;
		m_head.set(header::XORIG, h.h[header::XORIG]);

		// pass-through the new location if needed
		if(code == 301 || code == 302)
			m_head.set(header::LOCATION, h.h[header::LOCATION]);
	}
			
	if(bStoreData)
	{
		// using adaptive Delete-Or-Replace-Or-CopyOnWrite strategy
		
		if(acfg::debug>1)
			aclog::misc(string("Download of ")+m_sKey+" started");
		
		// First opening the file first to be sure that it can be written. Header storage is the critical point,
		// every error after that leads to full cleanup to not risk inconsistent file contents 
		
		int flags = O_WRONLY | O_CREAT | O_BINARY;
		struct stat stbuf;
				
		mkbasedir(m_sPath);
		m_filefd=open(m_sPath.c_str(), flags, acfg::fileperms);
		ldbg("file opened?! returned: " << m_filefd);
		
		// self-recovery from cache poisoned with files with wrong permissions
		if (m_filefd<0)
		{
			if(m_nSizeChecked>0) // OOOH CRAP! CANNOT APPEND HERE! Do what's still possible.
			{
				string temp=m_sPath+".tmp";
				if(FileCopy(m_sPath, temp) && 0==unlink(m_sPath.c_str()) )
				{
					if(0!=rename(temp.c_str(), m_sPath.c_str()))
						BOUNCE("503 Cannot rename files");
					
					// be sure about that
					if(0!=stat(m_sPath.c_str(), &stbuf) || stbuf.st_size!=m_nSizeSeen)
						BOUNCE("503 Cannot copy file parts, filesystem full?");
					
					m_filefd=open(m_sPath.c_str(), flags, acfg::fileperms);
					ldbg("file opened after copying around: ");
				}
				else
					BOUNCE("503 Cannot store or remove files");
			}
			else
			{
				unlink(m_sPath.c_str());
				m_filefd=open(m_sPath.c_str(), flags, acfg::fileperms);
				ldbg("file force-opened?! returned: " << m_filefd);
			}
		}
		
		if (m_filefd<0)
			BOUNCE(errno==ENOSPC ? "503 OUT OF DISK SPACE" : 
			"503 Cache storage error while opening data file");
		
		if(0!=fstat(m_filefd, &stbuf) || !S_ISREG(stbuf.st_mode))
			SETERRORKILLFILE("503 Not a regular file");
		
		// crop, but only if the new size is smaller. MUST NEVER become larger (would fill with zeros)
		if(m_nSizeChecked <= m_nSizeSeen)
		{
			if(0!=ftruncate(m_filefd, m_nSizeChecked))
				SETERRORKILLFILE("503 Cannot change file size");
		}
		else if(m_nSizeChecked>m_nSizeSeen) // should never happen and caught by the checks above
			SETERRORKILLFILE("503 Internal error on size checking");
		// else... nothing to fix since the expectation==reality

		falloc_helper(m_filefd, hint_start, hint_length);
		
		ldbg("Storing header as "+sHeadPath);
		int count=m_head.StoreToFile(sHeadPath);
		
		// unlink and retry
		if(count<0)
			unlink(sHeadPath.c_str());
		count=m_head.StoreToFile(sHeadPath);

		if(count<0)
			SETERRORKILLFILE( (-count!=ENOSPC) ? "503 Cache storage error" : "503 OUT OF DISK SPACE");
			
		// double-check the sane state
		if(0!=fstat(m_filefd, &stbuf) || stbuf.st_size!=m_nSizeChecked)
			SETERRORKILLFILE("503 Inconsistent file state");
			
		if(m_nSizeChecked!=lseek(m_filefd, m_nSizeChecked, SEEK_SET))
			SETERRORKILLFILE("503 IO error, positioning");
	}
	
	status=FIST_DLGOTHEAD;
	return true;

	kill_file:
	if(m_filefd>=0)
		forceclose(m_filefd);
	unlink(m_sPath.c_str());
	unlink(sHeadPath.c_str());
	
	status=FIST_ERROR;
	return false;
}

bool fileitem::StoreFileData(const char *data, unsigned int size) {
	
	setLockGuard;

	// something might care, most likely... also about BOUNCE action
	notifyAll();
	
	m_nIncommingCount+=size;
	
	if(status >= FIST_ERROR || status < FIST_DLGOTHEAD)
		return false;
	
	if (size==0)
	{
		status=FIST_COMPLETE;

		if(acfg::debug>1)
			aclog::misc(string("Download of ")+m_sKey+" finished");
		
		// we are done! Fix header from chunked transfers?
		if (m_filefd>=0 && ! m_head.h[header::CONTENT_LENGTH])
		{
			m_head.set(header::CONTENT_LENGTH, m_nSizeChecked);
			m_head.StoreToFile(m_sPath+".head");
		}
	}
	else
	{
		status = FIST_DLRECEIVING;

		if (m_filefd>=0)
		{
			while(size>0)
			{
				int r=write(m_filefd, data, size);
				if(r<0)
				{
					if(EINTR==errno || EAGAIN==errno)
						continue;
					if(ENOSPC==errno)
						BOUNCE("503 OUT OF DISK SPACE");
					BOUNCE("503 General storage error");
				}
				m_nSizeChecked+=r;
				size-=r;
				data+=r;
			}
			
		}
	}
	return true;
}

fileitem::~fileitem()
{
	setLockGuard;
	m_head.clear();
	checkforceclose(m_filefd);
}


struct tFileRefEntry
{
	int nRefCount;
	tFileItemPtr ref;
	tFileRefEntry() : nRefCount(0) { };
};
static map<string, tFileRefEntry> mapItems;
lockable mapLck;

void fileitem::Unreg()
{
	LOGSTART("fileitem::Unreg");

   lockguard managementLock(mapLck); 
	
   std::map<string,tFileRefEntry>::iterator entry = mapItems.find(m_sKey);
   if(mapItems.end() == entry)
   {
	   aclog::err("INTERNAL ERROR, attempt to unregister non-existing download item");
	   return;
   }
   if(entry->second.ref.get() != this)
   {
	   aclog::err("INTERNAL ERROR, doppelganger running amok");
	   return;
   }
   entry->second.nRefCount--;
   if(entry->second.nRefCount<=0)
   {
	   LOG("*this is last entry, deleting dl/fi mapping");
	  setLockGuard;
	  status=FIST_ERRNOUSER;
	  notifyAll();
	  mapItems.erase(entry);
   }
}

tFileItemPtr fileitem::GetFileItem(MYSTD::string sPathKey)
{
	LOGSTART("fileitem::GetFileItem");

	tFileItemPtr p;
	MYTRY
	{
		lockguard lockGlobalMap(mapLck);

		map<string, tFileRefEntry>::iterator it=mapItems.find(sPathKey);
		if(it!=mapItems.end())
		{
			it->second.nRefCount++;
			LOG("Sharing existing file item");
			return it->second.ref;
		}
		p.reset(new fileitem(sPathKey));
		LOG("Created new file item");
		tFileRefEntry & entry = mapItems[sPathKey];
		LOG("Registering new file item...");
		if(!entry.ref)
		{
			entry.ref=p;
			entry.nRefCount++;
			LOG("New reference count: " << entry.nRefCount);
		}
		return entry.ref;
	}
	MYCATCH(std::bad_alloc)
	{
		if(p)
			p.reset();
		return p;
	}

}

#ifdef DEBUG

void DumpItems()
{
/*	lockguard lockGlobalMap(mapLck);

	map<string, tFileRefEntry>::iterator it=mapItems.begin();
	cout << "Remaining file items:\n";
	for(;it!=mapItems.end(); it++)
	{
		cout << it->second.ref->status << ": " << it->first <<endl;
	}
	*/
}
#endif
