# Code structure extractor for C preprocessor language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/code"
require "adlint/report"

module AdLint #:nodoc:
module Cpp #:nodoc:

  class ObjectLikeMacroExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_object_like_macro_defined += method(:extract)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract(define_line, macro)
      MACRODEF(define_line.location, macro.name.value, "O")
    end
  end

  class FuncLikeMacroExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_function_like_macro_defined += method(:extract)
      interp.on_va_function_like_macro_defined += method(:extract)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract(define_line, macro)
      MACRODEF(define_line.location, macro.name.value, "F")
    end
  end

  class IncludeDirectiveExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_system_header_included += method(:extract_system_include)
      interp.on_user_header_included += method(:extract_user_include)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract_system_include(system_include_line, system_header)
      INCLUDE(system_include_line.location, "<#{system_header.fpath}>")
    end

    def extract_user_include(user_include_line, user_header)
      INCLUDE(user_include_line.location, "\"#{user_header.fpath}\"")
    end
  end

  class DirectiveExtraction < CodeExtraction
    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_if_statement += method(:extract_if)
      visitor.enter_ifdef_statement += method(:extract_ifdef)
      visitor.enter_ifndef_statement += method(:extract_ifndef)
      visitor.enter_elif_statement += method(:extract_elif)
      visitor.enter_else_statement += method(:extract_else)
      visitor.enter_endif_line += method(:extract_endif)
      visitor.enter_user_include_line += method(:extract_user_include)
      visitor.enter_system_include_line += method(:extract_system_include)
      visitor.enter_object_like_define_line += method(:extract_define)
      visitor.enter_function_like_define_line += method(:extract_define)
      visitor.enter_va_function_like_define_line += method(:extract_define)
      visitor.enter_undef_line += method(:extract_undef)
      visitor.enter_line_line += method(:extract_line)
      visitor.enter_error_line += method(:extract_error)
      visitor.enter_pragma_line += method(:extract_pragma)
      visitor.enter_null_directive += method(:extract_null)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract_if(if_statement)
      PP_DIRECTIVE(if_statement.location, if_statement.keyword.value,
                   if_statement.expression.to_s)
    end

    def extract_ifdef(ifdef_statement)
      PP_DIRECTIVE(ifdef_statement.location, ifdef_statement.keyword.value,
                   ifdef_statement.identifier.value)
    end

    def extract_ifndef(ifndef_statement)
      PP_DIRECTIVE(ifndef_statement.location, ifndef_statement.keyword.value,
                   ifndef_statement.identifier.value)
    end

    def extract_elif(elif_statement)
      PP_DIRECTIVE(elif_statement.location, elif_statement.keyword.value,
                   elif_statement.expression.to_s)
    end

    def extract_else(else_statement)
      PP_DIRECTIVE(else_statement.location, else_statement.keyword.value, nil)
    end

    def extract_endif(endif_line)
      PP_DIRECTIVE(endif_line.location, endif_line.keyword.value, nil)
    end

    def extract_user_include(user_include_line)
      PP_DIRECTIVE(user_include_line.location, user_include_line.keyword.value,
                   user_include_line.header_name.value)
    end

    def extract_system_include(system_include_line)
      PP_DIRECTIVE(system_include_line.location,
                   system_include_line.keyword.value,
                   system_include_line.header_name.value)
    end

    def extract_define(define_line)
      PP_DIRECTIVE(define_line.location, define_line.keyword.value,
                   define_line.identifier.value)
    end

    def extract_undef(undef_line)
      PP_DIRECTIVE(undef_line.location, undef_line.keyword.value,
                   undef_line.identifier.value)
    end

    def extract_line(line_line)
      PP_DIRECTIVE(line_line.location, line_line.keyword.value,
                   line_line.pp_tokens ? line_line.pp_tokens.to_s : "")
    end

    def extract_error(error_line)
      PP_DIRECTIVE(error_line.location, error_line.keyword.value,
                   error_line.pp_tokens.tokens.map { |t| t.value }.join(" "))
    end

    def extract_pragma(pragma_line)
      PP_DIRECTIVE(pragma_line.location, pragma_line.keyword.value,
                   pragma_line.pp_tokens ? pragma_line.pp_tokens.to_s : "")
    end

    def extract_null(null_directive)
      PP_DIRECTIVE(null_directive.location, null_directive.token.value, nil)
    end
  end


end
end
