#include <gtk/gtk.h>

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "gprop.h"
#include "gui_tools.h"
#include "nls.h"
#include "indexdb.h"
#include "re.h"
#include "nscache.h"

#ifdef HAVE_FNMATCH
#include <fnmatch.h>
#else
#include "fnmatch.h"
#endif

#include "pixmaps/stock_close.xpm"
#include "pixmaps/stock_stop.xpm"
#include "pixmaps/stock_search.xpm"

typedef struct {
	char *filename;
	char *url_str;
	char *mime_type;
} search_entry_info_t;

typedef struct {
	GtkWidget	*top_w;
	GtkWidget	*pattern_w;
	GtkWidget	*mime_type_w;
	GtkWidget	*case_w;
	GtkWidget	*full_match_w;
#ifdef HAVE_REGEX
	GtkWidget	*re_w;
#endif
	GtkWidget	*list_w;
	GtkWidget	*progress_w;
	GtkWidget	*search_w;
	GtkWidget	*stop_w;
	GtkWidget	*close_w;

	GSList		*docs;
	guint		num_docs;
	guint		total_num_docs;
	guint		idle_id;

	gboolean	case_sensitive;
	gboolean	full_match;
	char		*pattern;
	char		*mime_type;
#ifdef HAVE_REGEX
	gboolean	with_reg_exp;
	re_entry	*re;
#endif
	int		nfound;
} search_info_t;

extern GSList *elist;
extern char *db_name;
extern gint popup_tmenu();
extern GtkWidget *flow_menu, *tlist;

GList *search_get_available_types()
{
	GList *rv = NULL;
	GSList *ptr;

	for (ptr = elist; ptr; ptr = ptr->next)
	{
		ns_cache_record *nr = (ns_cache_record *)ptr->data;

		if (!g_list_find_custom(rv, nr->content_type, (GCompareFunc)strcmp))
		{
			rv = g_list_append(rv, nr->content_type);
		}
	}

	rv = g_list_sort(rv, (GCompareFunc) strcmp);

	rv = g_list_prepend(rv, "image/*");
	rv = g_list_prepend(rv, "text/html*");
	rv = g_list_prepend(rv, "text/*");
	rv = g_list_prepend(rv, "*");
	rv = g_list_prepend(rv, "");

	return rv;
}

static search_entry_info_t *search_entry_new(urls, file, mimet)
char *urls;
char *file;
char *mimet;
{
	search_entry_info_t *rv;
	char *p;

	rv = g_malloc(sizeof(search_entry_info_t *));

	rv->url_str = g_strdup(urls);
	rv->mime_type = g_strdup(mimet);

	p = strrchr(db_name, '/');
	p = p ? g_strndup(db_name, p - db_name) : g_strdup(".");
	rv->filename = g_strconcat(p, "/", file, NULL);
	g_free(p);

	return rv;
}

static void search_entry_free(info)
search_entry_info_t *info;
{
	g_free(info->filename);
	g_free(info->url_str);
	g_free(info->mime_type);
	g_free(info);
}

static GSList *search_get_cache_docs(info)
search_info_t *info;
{
	GSList *ptr, *rv = NULL;

	for (ptr = elist; ptr; ptr = ptr->next)
	{
		ns_cache_record *nr = (ns_cache_record *)ptr->data;

		if (!info->mime_type || !nr->content_type ||
		    !fnmatch(info->mime_type, nr->content_type, 0))
		{
			rv = g_slist_append(rv, search_entry_new(nr->urlstr,
				nr->filename, nr->content_type));
		}
	}

	return rv;
}

static void search_dlg_free_runtime_data(info)
search_info_t *info;
{
	if (info->idle_id)
	{
		gtk_idle_remove(info->idle_id);
		info->idle_id = 0;
	}

	g_free(info->pattern);
	info->pattern = NULL;

	g_free(info->mime_type);
	info->mime_type = NULL;

#ifdef HAVE_REGEX
	if (info->re)
	{
		re_free(info->re);
		info->re = NULL;
	}
#endif

	while (info->docs)
	{
		search_entry_free((search_entry_info_t *) info->docs->data);
		info->docs = g_slist_remove_link(info->docs, info->docs);
	}
}

static void search_dlg_destroy(w, info)
GtkWidget *w;
search_info_t *info;
{
	if (info->idle_id)
		gtk_idle_remove(info->idle_id);
	search_dlg_free_runtime_data(info);
	g_free(info);
}

static void search_dlg_search_stop(w, info)
GtkWidget *w;
search_info_t *info;
{
	search_dlg_free_runtime_data(info);

	gtk_widget_set_sensitive(info->stop_w, FALSE);
	gtk_widget_set_sensitive(info->search_w, TRUE);
	gtk_widget_hide(info->progress_w);


	if (info->nfound)
	{
		char pom[256];
		sprintf(pom, gettext("NScache: Find (found %d)"), info->nfound);
		gtk_window_set_title(GTK_WINDOW(info->top_w), pom);
	}
	else
		gtk_window_set_title(GTK_WINDOW(info->top_w),
			gettext("NScache: Find (not found)"));

	gdk_beep();
}

static char *lowerstr(s) 
char *s;
{
        char *p ; 

        for(p = s ; *p != '\0' ; p ++) *p = tolower(*p); 

        return s;
}

#define IS_DELIM(c) ((c) == '\0' || strchr(" \t\r\n,.;:(){}[]\'\"<>?!/\\", (c)))

static char *search_dlg_search_file(info, filename)
search_info_t *info;
char *filename;
{
	FILE *f;
	char buf[4096];
	char *rv = NULL;

	f = fopen(filename, "r");

	if (!f)
	{
		perror(filename);
		return NULL;
	}

	while (fgets(buf, sizeof(buf) - 1, f))
	{
		if (!info->case_sensitive)
			lowerstr(buf);

#ifdef HAVE_REGEX
		if (info->with_reg_exp)
		{
			if (re_pmatch(info->re, buf))
				rv = g_strdup(buf);
		}
		else
#endif
		{
			char *p;

			if ((p = strstr(buf, info->pattern)))
			{
				if (info->full_match)
				{
					if (p == buf || IS_DELIM(*(p-1)))
					{
						p += strlen(info->pattern);

						if (IS_DELIM(*p))
							rv = g_strdup(buf);
					}
				}
				else
					rv = g_strdup(buf);
			}
		}

		if (rv)
			break;
	}
	fclose(f);

	return rv;
}

static int search_dlg_search_doc(info)
search_info_t *info;
{
	search_entry_info_t *doc;
	char *matchln;

	if (!info->docs)
	{
		info->idle_id = 0;
		search_dlg_search_stop(NULL, info);
		return FALSE;
	}

	doc = (search_entry_info_t *)info->docs->data;
	info->docs = g_slist_remove_link(info->docs, info->docs);
	info->num_docs--;
	gtk_progress_bar_update(GTK_PROGRESS_BAR(info->progress_w),
		(gfloat)(1.0 - ((double)info->num_docs)/((double)info->total_num_docs)));

	matchln = search_dlg_search_file(info, doc->filename);

	if (matchln)
	{
		char *pp[3];
		int row;
		struct nscache_icon_t *icon_info;

		icon_info = nscache_icon_get(doc->mime_type);

		pp[0] = doc->url_str;
		pp[1] = matchln;
		pp[2] = doc->filename;

		row = gtk_clist_append(GTK_CLIST(info->list_w), pp);
                gtk_clist_set_pixtext(GTK_CLIST(info->list_w), row, 0, pp[0],
                        3, icon_info->pixmap, icon_info->mask);
		gtk_clist_set_row_data_full(GTK_CLIST(info->list_w), row,
			doc, (GtkDestroyNotify)search_entry_free);

		if (!row)
			gtk_clist_select_row(GTK_CLIST(info->list_w), row, 0);

		g_free(matchln);

		info->nfound++;
	}
	else
	{
		search_entry_free(doc);
	}

	return TRUE;
}

static void search_dlg_start(w, info)
GtkWidget *w;
search_info_t *info;
{
	char *p;

	gtk_window_set_title(GTK_WINDOW(info->top_w), gettext("NScache: Find"));
	gtk_clist_clear(GTK_CLIST(info->list_w));
	search_dlg_free_runtime_data(info);

	info->case_sensitive = GTK_TOGGLE_BUTTON(info->case_w)->active;
	info->full_match = GTK_TOGGLE_BUTTON(info->full_match_w)->active;

	p = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(info->mime_type_w)->entry));
	if (p && *p)
		info->mime_type = g_strdup(p);

	p = gtk_entry_get_text(GTK_ENTRY(info->pattern_w));
	if (!p || !*p)
	{
		gdk_beep();
		return;
	}
	info->pattern = g_strdup(p);
	if (!info->case_sensitive)
		lowerstr(info->pattern);

#ifdef HAVE_REGEX
	info->with_reg_exp = GTK_TOGGLE_BUTTON(info->re_w)->active;

	if (info->with_reg_exp)
	{
		info->re = re_make(info->pattern);

		if (!info->re)
		{
			search_dlg_free_runtime_data(info);
			gdk_beep();
			return;
		}
	}
	gprop_set_bool("find/with_reg_exp", info->with_reg_exp);
#endif

	gprop_set_str("find/pattern", info->pattern);
	gprop_set_str("find/mime_type", info->mime_type);
	gprop_set_bool("find/case_sensitive", info->case_sensitive);
	gprop_set_bool("find/full_match", info->full_match);

	info->docs = search_get_cache_docs(info);
	info->total_num_docs = info->num_docs = g_slist_length(info->docs);

	info->nfound = 0;

	info->idle_id = gtk_idle_add((GtkFunction)search_dlg_search_doc, info);

	gtk_widget_set_sensitive(info->stop_w, TRUE);
	gtk_widget_set_sensitive(info->search_w, FALSE);
	gtk_progress_bar_update(GTK_PROGRESS_BAR(info->progress_w) , 0.0);
	gtk_widget_show(info->progress_w);
}

static int find_by_urlstr(nr, url_str)
ns_cache_record *nr;
char *url_str;
{
	return strcmp(url_str, nr->urlstr);
}

static void search_dlg_result_list_select(object, row, col, event, info)
GtkObject *object;
int row;
int col;
GdkEvent *event;
search_info_t *info;
{
	search_entry_info_t *doc;
	GSList *ptr;

	doc = gtk_clist_get_row_data(GTK_CLIST(info->list_w), row);

	if (!doc)
		return;

	ptr = g_slist_find_custom(elist, doc->url_str, (GCompareFunc)find_by_urlstr);

	if (ptr)
	{
		ns_cache_record *nr;

		nr = (ns_cache_record *)ptr->data;
		gtk_ctree_select(GTK_CTREE(tlist), nr->tree_node);
	}
}

static void search_dlg_result_list_resize_column(w, col, width, fdata)
GtkWidget *w;
int col;
int width;
gpointer fdata;
{
	char pom[64];

	sprintf(pom, "result_list_col[%d].width", col);
	gprop_set_int(pom , width);
}

static void search_dlg_tl_resize(w, sallocation, fdata)
GtkWidget *w;
GtkAllocation *sallocation;
{
	gprop_set_int("find/width", sallocation->width);
	gprop_set_int("find/height", sallocation->height);
}

void search_dlg_open()
{
	search_info_t *info;
	GtkWidget *tbox, *box, *frame, *tab, *swin;
	GList *types;
	int rv,w,h;
	char *p;

	info = g_malloc(sizeof(search_info_t));
	memset(info, '\0', sizeof(search_info_t));

	info->top_w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(info->top_w), gettext("NScache: Find"));
	gtk_container_set_border_width(GTK_CONTAINER(info->top_w), 4);
	w = -1; h = -1;
	gprop_get_int("find/width", &w);
	gprop_get_int("find/height",&h);
	gtk_window_set_default_size(GTK_WINDOW(info->top_w), w, h);

	gtk_signal_connect(GTK_OBJECT(info->top_w), "destroy",
		GTK_SIGNAL_FUNC(search_dlg_destroy), info);
	gtk_signal_connect(GTK_OBJECT(info->top_w) , "size_allocate" ,
		GTK_SIGNAL_FUNC(search_dlg_tl_resize) , NULL);


	tbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(info->top_w), tbox);
	gtk_widget_show(tbox);
	
	box = gtk_table_new(1, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(tbox), box, FALSE, FALSE, 0);
	gtk_widget_show(box);

	frame = gtk_frame_new(NULL);
	gtk_table_attach(GTK_TABLE(box), frame, 0, 1, 0, 1, GTK_FILL|GTK_EXPAND, GTK_FILL, 2, 2);
	gtk_widget_show(frame);

	tab = gtk_table_new(2, 5, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), tab);
	gtk_widget_show(tab);

	info->pattern_w = guitl_tab_add_entry(tab, gettext("Pattern: "),
		0, 0, FALSE);
	if (gprop_get_str("find/pattern", &p))
		gtk_entry_set_text(GTK_ENTRY(info->pattern_w), p);

	types = search_get_available_types();
	info->mime_type_w = guitl_tab_add_combo(tab, gettext("MIME type: "),
		types, 0, 1, TRUE);
	if (gprop_get_str("find/mime_type", &p))
		gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(info->mime_type_w)->entry), p);

	info->case_w = gtk_check_button_new_with_label(gettext("Case sensitive"));
	gtk_table_attach(GTK_TABLE(tab), info->case_w, 0, 2, 2, 3,
		GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(info->case_w);
	if (gprop_get_bool("find/case_sensitive", &rv))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->case_w), rv);

	info->full_match_w = gtk_check_button_new_with_label(gettext("Match whole word"));
	gtk_table_attach(GTK_TABLE(tab), info->full_match_w, 0, 2, 3, 4,
		GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(info->full_match_w);
	if (gprop_get_bool("find/full_match", &rv))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->full_match_w), rv);


#ifdef HAVE_REGEX
	info->re_w = gtk_check_button_new_with_label(gettext("Regular pattern"));
	gtk_table_attach(GTK_TABLE(tab), info->re_w, 0, 2, 4, 5,
		GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(info->re_w);
	if (gprop_get_bool("find/with_reg_exp", &rv))
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(info->re_w), rv);
#endif

	tab = gtk_vbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(tab), GTK_BUTTONBOX_EDGE);
	gtk_table_attach(GTK_TABLE(box), tab, 1, 2, 0, 1,
		GTK_FILL, GTK_FILL, 2, 2);
	gtk_widget_show(tab);

	info->search_w = guitl_pixmap_button(stock_search_xpm, gettext("Search"));
	gtk_box_pack_start(GTK_BOX(tab), info->search_w, FALSE, FALSE, 2);
	gtk_signal_connect(GTK_OBJECT(info->search_w), "clicked",
		GTK_SIGNAL_FUNC(search_dlg_start), info);
	gtk_widget_show(info->search_w);

	info->stop_w = guitl_pixmap_button(stock_stop_xpm, gettext("Stop"));
	gtk_box_pack_start(GTK_BOX(tab), info->stop_w, FALSE, FALSE, 2);
	gtk_signal_connect(GTK_OBJECT(info->stop_w), "clicked",
		GTK_SIGNAL_FUNC(search_dlg_search_stop), info);
	gtk_widget_set_sensitive(info->stop_w, FALSE);
	gtk_widget_show(info->stop_w);

	info->close_w = guitl_pixmap_button(stock_close_xpm, gettext("Close"));
	gtk_box_pack_start(GTK_BOX(tab), info->close_w, FALSE, FALSE, 2);
	gtk_signal_connect(GTK_OBJECT(info->close_w), "clicked",
		GTK_SIGNAL_FUNC(guitl_CloseWin), info->top_w);
	gtk_widget_show(info->close_w);

	frame = gtk_frame_new(gettext("Result"));
	gtk_box_pack_start(GTK_BOX(tbox), frame, TRUE, TRUE, 1);
	gtk_widget_show(frame);

	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_set_usize(swin, 270, 170);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
		GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	gtk_container_add(GTK_CONTAINER(frame), swin);
	gtk_widget_show (swin);

	info->list_w = gtk_clist_new(3);
	gtk_clist_set_row_height(GTK_CLIST(info->list_w), MAX(
		info->list_w->style->font->ascent + info->list_w->style->font->descent + 1, 14));
	gtk_clist_column_titles_show(GTK_CLIST(info->list_w));
	gtk_clist_set_selection_mode(GTK_CLIST(info->list_w), GTK_SELECTION_BROWSE);
	gtk_clist_set_column_title(GTK_CLIST(info->list_w), 0, gettext("URL"));
	rv = 100;
	gprop_get_int("result_list_col[0].width", &rv);
	gtk_clist_set_column_width(GTK_CLIST(info->list_w), 0, rv);
	
	gtk_clist_set_column_title(GTK_CLIST(info->list_w), 1, gettext("Found"));
	rv = 100;
	gprop_get_int("result_list_col[1].width", &rv);
	gtk_clist_set_column_width(GTK_CLIST(info->list_w), 1, rv);

	gtk_clist_set_column_title(GTK_CLIST(info->list_w), 2, gettext("File"));
	rv = 100;
	gprop_get_int("result_list_col[2].width", &rv);
	gtk_clist_set_column_width(GTK_CLIST(info->list_w), 2, rv);

	gtk_container_add(GTK_CONTAINER(swin), info->list_w);
	gtk_widget_show(info->list_w);

	gtk_signal_connect(GTK_OBJECT(info->list_w), "button_press_event" ,
		GTK_SIGNAL_FUNC(popup_tmenu), flow_menu);

	gtk_signal_connect(GTK_OBJECT(info->list_w) , "select_row" ,
		GTK_SIGNAL_FUNC(search_dlg_result_list_select), info);

	gtk_signal_connect(GTK_OBJECT(info->list_w), "resize_column", 
		GTK_SIGNAL_FUNC(search_dlg_result_list_resize_column), NULL);

	info->progress_w = gtk_progress_bar_new();
	gtk_box_pack_start(GTK_BOX(tbox), info->progress_w, FALSE, FALSE, 1);
	gtk_widget_show(info->progress_w);

	gtk_widget_show(info->top_w);
	gtk_widget_hide(info->progress_w);
}

