/* GSnes9x -- util.c: Util functions.
 * Copyright (C) 1999 Canek Pelez Valds <canek@abulafia.fciencias.unam.mx>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsnes9x.h"

void
rom_get_defaults (ROMInfo *rom, GlobalPrefs *prefs)
{
	gint i, j;

	F_NAME();
	
	for (i = 0; i < MIX_BOOL_PREFS; i++)
		rom->bool_prefs[i] = prefs->bool_prefs[i];

	for (i = 0; i < MIX_NUM_PREFS; i++)
		rom->number_prefs[i] = prefs->number_prefs[i];
	
	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			rom->buttons[i][j]  = prefs->buttons[i][j];
}

GSnes9xData*
gsnes9x_data_new ()
{
	GSnes9xData *data = g_new (GSnes9xData, 1);

	F_NAME();
	
	data->app         = NULL;
	data->paned       = NULL;
	data->image_w     = NULL;
	data->roms_w      = NULL;
	data->pixmap      = NULL;
	data->clist       = NULL;
	data->status      = NULL;
	data->prefs       = NULL;
	data->roms_list   = NULL;
	data->current_rom = -1;
	data->dialog_open = FALSE;
	data->about_open  = FALSE;
	data->help_open   = FALSE;
	data->width       = IMAGE_WIDTH;
	data->height      = IMAGE_HEIGHT;
	
	return data;
}

GList*
get_roms_list ()
{
	ROMInfo *rom_info;
	FILE    *names;
	GList   *list      = NULL;
	gchar   *roms_file = gnome_util_home_file (ROMS_FILE);
	gchar   *str       = g_new (gchar, CONF_STR_LEN);
	gint     size      = CONF_STR_LEN;
	
	F_NAME();
	
	names = fopen (roms_file, "r");
	
	if (!names)
		return list;

	getline (&str, &size, names);
	str[strlen (str)-1] = '\0'; /* Rip out the '\n'. */
	
	while (!feof (names)) {
		
		rom_info = read_rom_info_file (str);

		if (rom_info) {
			if (!rom_info->string_prefs[NAME]     ||
			    !rom_info->string_prefs[COMPANY]  ||
			    !rom_info->string_prefs[ROM_FILE] ||
			    !rom_info->string_prefs[IMAGE_FILE]) {
				g_warning (_("The ROM info file %s is corrupt "
					     "or incomplete. Skipping it"),
					   gnome_util_home_file (str));
				g_free (rom_info);
				rom_info = NULL;
			}
		}

		if (rom_info)
			list = g_list_prepend (list, (gpointer)rom_info);

		getline (&str, &size, names);
		str[strlen(str)-1] = '\0';
	}

	fclose (names);
	g_free (str);

	return list;
}

gchar**
get_command_line (ROMInfo *rom_info, GlobalPrefs *prefs)
{
	gint i, j, cont = 0;
	gchar **process = g_new (gchar*, 300), *str = g_new (gchar, 100);

	process[cont++] =
		g_strdup (prefs->string_prefs[SNES9X_COMMAND-ROM_STR_PREFS]);
	
	if (prefs->bool_prefs[NETPLAY-ROM_BOOL_PREFS]) {
		process[cont++] = g_strdup ("-server");
		process[cont++] = g_strdup
			(prefs->string_prefs[SERVER-ROM_STR_PREFS]);
		if (prefs->string_prefs[NETPORT-ROM_STR_PREFS]) {
			process[cont++] = g_strdup ("-port");
			process[cont++] = g_strdup
				(prefs->string_prefs[NETPORT-ROM_STR_PREFS]);
		}
	}
	
	if (rom_info->bool_prefs[TRANSPARENCY])
		process[cont++] = g_strdup ("-transparency");
	if (rom_info->bool_prefs[SIXTEEN])
		process[cont++] = g_strdup ("-sixteen");
	if (rom_info->bool_prefs[HIRES])
		process[cont++] = g_strdup ("-hires");
	if (rom_info->bool_prefs[NOMODESWITCH])
		process[cont++] = g_strdup ("-nomodeswitch");
	if (rom_info->bool_prefs[FULLSCREEN])
		process[cont++] = g_strdup ("-fullscreen");
	if (rom_info->bool_prefs[SCALE])
		process[cont++] = g_strdup ("-scale");
	if (rom_info->bool_prefs[DISPLAYFRAMERATE])
		process[cont++] = g_strdup ("-displayframerate");
	if (rom_info->number_prefs[FILTER_NUMBER] > 0) {
			g_snprintf (str, 100, "%d", (gint)
			 rom_info->number_prefs[FILTER_NUMBER]);
		process[cont++] = g_strjoin (NULL, "-y", str, NULL);
	}
	if (rom_info->bool_prefs[NOSOUND])
		process[cont++] = g_strdup ("-nosound");
	if (rom_info->bool_prefs[SOUNDSKIP]) {
		process[cont++] = g_strdup ("-soundskip");
		g_snprintf (str, 100, "%d", (gint)
			 rom_info->number_prefs[SOUNDSKIP_NUMBER]);
		process[cont++] = g_strdup (str);
	}
	if (rom_info->bool_prefs[STEREO])
		process[cont++] = g_strdup ("-stereo");
	else
		process[cont++] = g_strdup ("-mono");
	if (rom_info->bool_prefs[SOUNDQUALITY]) {
		process[cont++] = g_strdup ("-soundquality");
		g_snprintf (str, 100, "%d", (gint)
			 rom_info->number_prefs[SOUNDQUALITY_NUMBER]);
		process[cont++] = g_strdup (str);
	}
	if (rom_info->bool_prefs[ENVX])
		process[cont++] = g_strdup ("-envx");
	if (rom_info->bool_prefs[SOUNDSYNC])
		process[cont++] = g_strdup ("-soundsync");
	if (rom_info->bool_prefs[ALTSAMPLEDECODE])
		process[cont++] = g_strdup ("-altsampledecode");
	if (rom_info->bool_prefs[INTERPOLATEDSOUND])
		process[cont++] = g_strdup ("-interpolatedsound");
	if (rom_info->bool_prefs[BUFFERSIZE]) {
		process[cont++] = g_strdup ("-buffersize");
		g_snprintf (str, 100, "%d", (gint)
			 rom_info->number_prefs[BUFFERSIZE_NUMBER]);
		process[cont++] = g_strdup (str);
	}
	if (rom_info->bool_prefs[NOSAMPLECACHING])
		process[cont++] = g_strdup ("-nosamplecaching");
	if (rom_info->bool_prefs[NOECHO])
		process[cont++] = g_strdup ("-noecho");
	if (rom_info->bool_prefs[NOMASTERVOLUME])
		process[cont++] = g_strdup ("-nomastervolume");

	if (rom_info->bool_prefs[GAMEGENIE])
		for (i = 0; i < MAX_CODES; i++)
			if (rom_info->gamegenie_code[i]) {
				process[cont++] = g_strdup ("-gg");
				process[cont++] = g_strdup
					(rom_info->gamegenie_code[i]);
			}

	if (rom_info->bool_prefs[ACTIONREPLAY])
		for (i = 0; i < MAX_CODES; i++)
			if (rom_info->actionreplay_code[i]) {
				process[cont++] = g_strdup ("-ar");
				process[cont++] = g_strdup
					(rom_info->actionreplay_code[i]);
			}
	if (rom_info->bool_prefs[GOLDFINGER])
		for (i = 0; i < MAX_CODES; i++)
			if (rom_info->goldfinger_code[i]) {
				process[cont++] = g_strdup ("-gf");
				process[cont++] = g_strdup
					(rom_info->goldfinger_code[i]);
			}
	
	if (rom_info->bool_prefs[FRAMESKIP]) {
		process[cont++] = g_strdup ("-frameskip");
		g_snprintf (str, 100, "%d", (gint)
			 rom_info->number_prefs[FRAMESKIP_NUMBER]);
		process[cont++] = g_strdup (str);
	}
	if (rom_info->bool_prefs[FRAMETIME]) {
		process[cont++] = g_strdup ("-frametime");
		g_snprintf (str, 100, "%f", (gfloat)
			 rom_info->number_prefs[FRAMETIME_NUMBER]);
		process[cont++] = g_strdup (str);
	}
	if (rom_info->bool_prefs[CYCLES]) {
		process[cont++] = g_strdup ("-cycles");
		g_snprintf (str, 100, "%d", (gint)
			 rom_info->number_prefs[CYCLES_NUMBER]);
		process[cont++] = g_strdup (str);
	}
	if (rom_info->bool_prefs[INTERLEAVED])
		process[cont++] = g_strdup ("-i");
	if (rom_info->bool_prefs[INTERLEAVED2])
		process[cont++] = g_strdup ("-i2");
	if (rom_info->bool_prefs[HIROM])
		process[cont++] = g_strdup ("-hirom");
	if (rom_info->bool_prefs[LOROM])
		process[cont++] = g_strdup ("-lorom");
	if (rom_info->bool_prefs[HEADER])
		process[cont++] = g_strdup ("-header");
	if (rom_info->bool_prefs[NOHEADER])
		process[cont++] = g_strdup ("-noheader");
	if (rom_info->bool_prefs[PAL])
		process[cont++] = g_strdup ("-pal");
	if (rom_info->bool_prefs[NTSC])
		process[cont++] = g_strdup ("-ntsc");
	if (rom_info->bool_prefs[LAYERING])
		process[cont++] = g_strdup ("-layering");
	if (rom_info->bool_prefs[LOADSNAPSHOT] &&
	    rom_info->string_prefs[SNAPSHOT_FILE]) {
		process[cont++] = g_strdup ("-loadsnapshot");
		process[cont++] = g_strdup
			(rom_info->string_prefs[SNAPSHOT_FILE]);
	}
	if (rom_info->bool_prefs[NOHDMA])
		process[cont++] = g_strdup ("-nohdma");
	if (rom_info->bool_prefs[NOSPEEDHACKS])
		process[cont++] = g_strdup ("-nospeedhacks");
	if (rom_info->bool_prefs[NOWINDOWS])
		process[cont++] = g_strdup ("-nowindows");
	for (i = 0; i < NUM_OF_JOYS; i++) {
		if (rom_info->bool_prefs[JOYMAP0+i]) {
			g_snprintf (str, 100, "%s%d", "-joymap", i+1);
			process[cont++] = g_strdup (str);
			for (j = 0; j < NUM_OF_BUTTONS; j++) {
				g_snprintf (str, 100, "%d",
					 rom_info->buttons[i][j]);
				process[cont++] = g_strdup (str);
			}
		}
	}
	if (rom_info->bool_prefs[SWAP])
		process[cont++] = g_strdup ("-swap");
	if (rom_info->bool_prefs[NOJOY])
		process[cont++] = g_strdup ("-nojoy");
	process[cont++] = g_strdup (rom_info->string_prefs[ROM_FILE]);
	
	g_free (str);
	
	for (i = cont; i < 300; i++)
		process[i] = NULL;
	
	return process;
}

GList*
get_codes_for_rom (const gchar* name, GSnes9xCheatCodes cc)
{
	CheatCode *cheat_code;
	GList     *list = NULL;
	gchar     *filename, *type = NULL;
	FILE      *file;
	gchar     *str = g_new (gchar, CONF_STR_LEN);
	gint       size = CONF_STR_LEN;

	F_NAME();
	
	switch (cc) {
	case GG_CODE:
		type = "gg";
		break;
	case AR_CODE:
		type = "ar";
		break;
	case GF_CODE:
		type = "gf";
		break;
	}

	filename = g_strdup_printf ("%s/%s.%s", CONF_DIR, name, type);
	
	file = fopen (gnome_util_home_file (filename), "r");

	g_free (filename);
	
	if (!file)
		return list;

	getline (&str, &size, file);
	str[strlen (str)-1] = '\0'; /* Rip out '\n' */
	if (!strcmp (str, ""))
		return list;
	
	do {
		cheat_code = g_new (CheatCode, 1);
		cheat_code->code = g_strdup (str);
		getline (&str, &size, file);
		str[strlen (str)-1] = '\0'; /* Rip out '\n' */
		cheat_code->name = g_strdup (str);
		getline (&str, &size, file);
		str[strlen (str)-1] = '\0'; /* Rip out '\n' */
		cheat_code->descr = g_strdup (str);
		getline (&str, &size, file);
		
		if (strcmp (str, cheat_code->descr))
			str[strlen (str)-1] = '\0'; /* Rip out '\n' */
		
		list = g_list_prepend (list, (gpointer) cheat_code);
				
	} while (!feof (file));

	g_free (str);

	fclose (file);
	
	return list;
}

gchar*
get_default_roms_dir ()
{
	gchar *path;

	/* Try every posible path for ROMS. */
	path = g_getenv ("GSNES9X_ROM_DIR");
	if (!path)
		path = g_getenv ("SNES96_ROM_DIR");
	if (!path)
		path = g_getenv ("GSNES9X_IMAGE_DIR");
	if (!path)
		path = g_get_home_dir ();
	if (path)
		path = g_strconcat (path, "/", NULL);
	return path;
}

gchar*
get_default_snap_dir ()
{
	gchar *path;

	/* Try every posible path for snaps. */
	path = g_getenv ("GSNES9X_SNAPSHOT_DIR");
	if (!path)
		path = g_getenv ("SNES96_SNAPSHOT_DIR");
	if (!path)
		path = g_strconcat (g_get_home_dir (),
			       	"/.snes96_snapshots", NULL);
	if (path)
		path = g_strconcat (path, "/", NULL);
	return path;
}

gboolean
check_rom_info (GtkWidget **entry)	
{
	gchar      **sides;
	gchar      *temp_name;
		
	F_NAME();
	
	if (!strlen (gtk_entry_get_text (GTK_ENTRY (entry[ROM_FILE]))))
		return FALSE;

	/* If name is empty, get name from rom file name. */	
	if (!strlen (gtk_entry_get_text (GTK_ENTRY (entry[NAME]))))
	{
		sides = g_strsplit (gtk_entry_get_text(
				GTK_ENTRY (entry[ROM_FILE])), ".", 2);
		temp_name = strrchr(sides[0], '/');
		if (temp_name == NULL)
			gtk_entry_set_text (GTK_ENTRY (entry[NAME]),
			    	g_strdup (sides[0]));
		else
			gtk_entry_set_text (GTK_ENTRY (entry[NAME]),
				g_strdup (++temp_name));
		g_strfreev (sides);
	}

	/* If company or image file is empty, put something in there. */
	if (!strlen (gtk_entry_get_text (GTK_ENTRY (entry[COMPANY]))))
		gtk_entry_set_text (GTK_ENTRY (entry[COMPANY]), "?");

	if (!strlen (gtk_entry_get_text (GTK_ENTRY (entry[IMAGE_FILE]))))
		gtk_entry_set_text (GTK_ENTRY (entry[IMAGE_FILE]), "?");
	
	return TRUE;	
}


/* util.c ends here. */
