/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  May 1994                                 */
/*-----------------------------------------------------------------------*/
/*                    Pitchmarking program                               */
/*************************************************************************/
#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include "EST_unix.h"
#include "EST_cmd_line_options.h"
#include "EST_cmd_line.h"
#include "EST_speech_class.h"

EST_Track pitchmark(EST_Wave &lx, EST_Option &op);
void pm_to_label(EST_Track &pm, EST_Stream &lab);
void find_pm(EST_Wave &sig, EST_Track &pm);
static EST_write_status save_msec(EST_Track &pm, EST_String filename);
static EST_write_status save_ogi_bin(EST_Track &pm, EST_String filename, int sr);

void pm_min_check(EST_Track &pm, float min);
void pm_sanity_check(EST_Track &pm, float new_end,
		     float max, float min, float def);

void pm_fill(EST_Track &pm, float new_end, float max, 
	     float min, float def);

int main (int argc, char *argv[])
{
    EST_Track pm;
    EST_Wave lx;
    EST_Option al, op;
    EST_String out_file("-");
    EST_StrList files;

    parse_command_line(argc, argv, 
       EST_String("Usage:   ")+
       "pitchmark <input file> -o <output file> <options>\n"+
       "Summary: pitchmark laryngograph (lx) files\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       options_wave_input()+ "\n"+
       options_track_output()+ "\n" +
       "-lx_lf <int>     lx low frequency cutoff\n"+
       "-lx_lo <int>     lx low order\n"+
       "-lx_hf <int>     lx high frequency cutoff\n"+
       "-lx_ho <int>     lx high order\n"
       "-df_lf <int>     df low frequeny cutoff\n"+
       "-df_lo <int>     df low order\n"+
       "-med_o <int>     median smoothing order\n"+
       "-mean_o <int>    mean smoothing order\n"+
       "-inv             Invert polarity of lx signal\n"+
       "-min <float>     Minimum pitch period in seconds\n"+
       "-max <float>     Minimum pitch period in seconds\n"+
       "-def <float>     Default pitch period in seconds\n"+
       "-pm <ifile>      Input is raw pitchmark file\n"+
       "-fill            Insert and remove pitchmarks according to min, max\n"+
       "            	 and def period values\n"+
       "-end <float>     End time of the last pitchmark, for use with\n"+
       "                 the -fill option\n"+
       "-wave_end        Use the end of a waveform to specify when the\n"+
       "                 last pitchmark position should be. The waveform\n"+
       "                 file is only read to determine its end, no\n"+
       "                 processing is performed\n"+
       "-debug           Output intermediate waveforms\n"+
       "-style <string>  \"track\" or \"lab\"\n", files, al);

    override_lib_ops(op, al);

    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    if (!al.present("-pm") || (al.present("-pm") && al.present("-wave_end")))
	if (read_wave(lx, files.first(), al) != read_ok)
	    exit(-1);

    if (al.present("-pm"))
	pm.load(al.val("-pm"));
    else
    {
	if (al.present("-inv"))
	    invert(lx);
	pm = pitchmark(lx, op);
    }

    // this allows the end to be aligned with the end of a waveform
    op.override_fval("pm_end", lx.end());

    // various options for filling he gaps between distant pitchmarks
    // and removing pitchmarks that are too close together

    if (al.present("-fill"))
    {
	pm_fill(pm, op.fval("pm_end"), op.fval("max_period"), 
			op.fval("min_period"), op.fval("def_period"));
	pm_fill(pm, op.fval("pm_end"), op.fval("max_period"), 
			op.fval("min_period"), op.fval("def_period"));
    }
    else if (al.present("-min"))
	pm_min_check(pm, al.fval("-min"));

    if (al.present("-style"))
    {
	// label format
	if (al.val("-style").contains("lab"))
	{
	    EST_Stream lab;
	    pm_to_label(pm, lab);
	    if (lab.save(out_file + ".pm_lab") != write_ok)
		exit(-1);
	}
	// save file in "traditional" milli-second format
	if (al.val("-style").contains("msec"))
	    save_msec(pm, out_file + ".pm");

	// ogi binary integer sample point format
	if (al.val("-style").contains("ogi_bin"))
	    save_ogi_bin(pm, out_file + ".pmv", lx.sample_rate());
    }
    else if (pm.save(out_file, al.val("-otype", 0)) != write_ok)
    {
	cerr << "pitchmark: failed to write output to \"" 
	    << out_file << "\"" << endl;
	exit(-1);
    }
    return 0;
}

static EST_write_status save_msec(EST_Track &pm, EST_String filename)
{
    ostream *outf;
    
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))
	return write_fail;
    
    outf->precision(5);
    outf->setf(ios::fixed, ios::floatfield);
    outf->width(8);
    
    for (int i = 0; i < pm.num_frames(); ++i)
	*outf << pm.t(i)  * 1000.0 << endl;
    
    return write_ok;
}

static EST_write_status save_ogi_bin(EST_Track &pm, EST_String filename, int sr)
{
    int *d;
    FILE *fp;
    int i;
    
    d = new int[pm.num_frames()];
    
    for (i = 0; i < pm.num_frames(); ++i)
	d[i] = int(pm.t(i) * (float) sr);
    
    if ((fp = fopen(filename, "wb")) == NULL)
	return misc_write_error;
    
    if (fwrite(d, pm.num_frames(), sizeof(int), fp) != 1)
    {
	fclose(fp);
	return misc_write_error;
    }
    delete d;
    
    return write_ok;
}

void override_lib_ops(EST_Option &op, EST_Option &al)
{
    op.override_ival("lx_low_frequency", 400);
    op.override_ival("lx_low_order", 19);
    op.override_ival("lx_high_frequency", 40);
    op.override_ival("lx_high_order", 19);
    op.override_ival("df_low_frequency", 1000);
    op.override_ival("df_low_order", 19);
    op.override_fval("min_period", 0.003);
    op.override_fval("max_period", 0.02);
    op.override_fval("def_period", 0.01);
    op.override_fval("pm_end", -1.0);
    
    if (al.present("-lx_lf"))
	op.override_ival("lx_low_frequency", al.ival("-lx_lf", 0));
    if (al.present("-lx_lo"))
	op.override_ival("lx_low_order", al.ival("-lx_lo", 0));
    if (al.present("-lx_hf"))
	op.override_ival("lx_high_frequency", al.ival("-lx_hf", 0));
    if (al.present("-lx_ho"))
	op.override_ival("lx_high_order", al.ival("-lx_ho", 0));
    if (al.present("-med_o"))
	op.override_ival("median_order", al.ival("-med_o", 0));
    if (al.present("-mean_o"))
	op.override_ival("mean_order", al.ival("-mean_o", 0));
    if (al.present("-df_lf"))
	op.override_ival("df_low_frequency", al.ival("-df_lf", 0));
    if (al.present("-df_lo"))
	op.override_ival("df_low_order", al.ival("-df_lo", 0));
    if (al.present("-min"))
	op.override_fval("min_period", al.fval("-min", 0));
    if (al.present("-max"))
	op.override_fval("max_period", al.fval("-max", 0));
    if (al.present("-def"))
	op.override_fval("def_period", al.fval("-def", 0));
    if (al.present("-end"))
	op.override_fval("pm_end", al.fval("-end", 0));
    if (al.present("-debug"))
	op.override_ival("pm_debug", 1);
}

