/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1998                            */
/*                        All Rights Reserved.                           */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author :  Alan W Black                          */
/*                       Date   :  February 1998                         */
/* --------------------------------------------------------------------- */
/*                                                                       */
/*  General class for representing linguistic information within a       */
/*  EST_Relation.  This consists of two parts, the relation specific     */
/*  part and the information content part.  The information content      */
/*  part may be shared between multiple EST_Items.                       */
/*                                                                       */
/*  This is typically used to represent things like words, phones but    */
/*  more abstract entities like NPs and nodes in metrical trees.         */
/*                                                                       */
/*************************************************************************/
#ifndef __EST_ITEM_H__
#define __EST_ITEM_H__

#include "EST_String.h"
#include "EST_Features.h"
#include "ling_class/EST_Item_Content.h"

#define FEATURE_ABSENCE_ERROR false

class EST_Relation;

/** A class for containing individual linguistic objects such as
words or phones.

These contain two types of infomation. This first is specific to the
\Ref{EST_Relation} we are viewing this ling item from, the second
part constists of a set fo rfeatures.  These features may be
shared by instances of this ling item in different \Ref{EST_Relation}s
within the same \Ref{EST_Utterance}s.

The shared part of an \Ref{EST_Item} is represented by 
the class \Ref{EST_Item_Content}.  It should not normally be accessed
by the general users as reverse links from the contents to
each of the \Ref{EST_Item}s it is part of are held ensure the
integrity of the structures.  Changing these without maintain
the appropriate links is unlikely to be stable.  

We believe this structure is the most efficient for the most natural
use we envisage.  Traversal of the items with the \Ref{EST_Relation} a
particular \Ref{EST_Item} is in is trivial with simple next,
prev, daughter1 etc functions.  Accessing the features themselves are
done through a set of functions which properly deal with features
whose values are functions (by calling them on the current item).

At present only simple lists and trees are directly supported within
this version of \Ref{EST_Item}s.  However we envisage the requirement
for more complex structures later.  This will be done by extending 
the basic fixed relationship pointers (next, prev, up and down) to
a named form.  

Trees are implemented on top of the basic next, prev, up, down,
pointers and should normally be accessed and created through the
provided functions, parent, daughter1 etc.  The actual representation
is that daughters arepresented as a list *thoguh next and prev) whose
first daughter alone is related to its parent through the up,down
pointers.  At present this is the simplest method, without building
in named links.

*/
class EST_Item 
{
  private:
    EST_Item_Content *p_contents;
    EST_Relation *p_relation;
    // In general (when we need it)
    // KVL <EST_String, EST_Item *> arcs;
    // but specifically
    EST_Item *n;
    EST_Item *p;
    EST_Item *u;
    EST_Item *d;

    void unref_contents();
    void ref_contents();
    void copy(const EST_Item &s);

    /**@name Internal manipulation functions */
    //@{
    /// used by tree manipulation functions
    void set_contents(EST_Item_Content *li);
    /// Get the daughters of node, removing reference to them
    EST_Item *grab_daughters(void);
    /** Get the contents, removing reference to them, this doesn't
        delete the contents if this item is the only reference */
    EST_Item_Content *grab_contents(void);
    //@}

  public:
    /**@name Constructor Functions */
    //@{
    /// Default constructor
    EST_Item();
    /// Copy constructure only makes reference to contents 
    EST_Item(const EST_Item &item);
    /// Includes reference to relation 
    EST_Item(EST_Relation *rel);
    /// Most common form of construction
    EST_Item(EST_Relation *rel, EST_Item *si);
    /// Deletes it and references to it in its contents
    ~EST_Item();
    //@}

    /// get contents from item
    EST_Item_Content *contents() { return (this == 0) ? 0 : p_contents; }

    /**@name Content accessing functions */
    /// Reference to the features in the contents
    /// The item's name
    const EST_String name() const
    { return (this == 0) ? EST_String::Empty : f("name").string(); }
    /// Set item's name
    void set_name(const EST_String &name) const
    { if (this != 0) p_contents->set_name(name); }

    /// Shouldn't normally be needed, except for iteration
    EST_Features &features() const { return p_contents->f; }

    /// Because so often we have char * access we provide direct functions
    const EST_Val f(const char *name, bool m=FEATURE_ABSENCE_ERROR) const
    { 
	if (this == 0) return EST_Features::feature_default_value;
        else
        {
	    EST_Val v; 
	    for (v=p_contents->f.val(name, m);
		 v.type() == val_func;
		 v=(v.function())((EST_Item *)(void *)this));
	    return v;
	}
    }

    const float fF(const EST_String &name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).Float();}
    const EST_String fS(const EST_String &name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).string();}
    const int fI(const EST_String &name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).Int();}
    const void *fP(const EST_String &name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).ptr();}

    const float fF(const char *name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).Float();}
    const EST_String fS(const char *name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).string();}
    const int fI(const char *name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).Int();}
    const void *fP(const char *name, bool m=FEATURE_ABSENCE_ERROR) const
       {return f(name, m).ptr();}

    /// TRUE if feature is present, FALSE otherwise
    int f_present(const EST_String &name) const
       {return features().present(name); }

    /// Setting functions for features
    void fset(const EST_String &name, int ival)
         { features().set(name,ival); }
    void fset(const EST_String &name, float fval)
         { features().set(name,fval); }
    void fset(const EST_String &name, double fval)
         { features().set(name,(float)fval); }
    void fset(const EST_String &name, const EST_String &sval)
         { features().set(name,sval); }
    void fset(const EST_String &name, const char *cval)
         { features().set(name,cval); }
    void fset(const EST_String &name,void *c,void (*f)(void *))
         { features().set(name,c,f); }
    void fset(const EST_String &name,EST_feature_function f)
         { features().set(name,f); }
    void fset_val(const EST_String &name, EST_Val sval)
         { features().set_val(name,sval); }
    
    void f_remove(const EST_String &name)
    { features().remove(name); }

    //@}

    // RJC: for now. Should be replaced by a clever feature mechanism

    float start() const;
    float mid() const;
    float end() const;

    /**@name Cross relational access */
    /// View item from another relation
    EST_Item *as_relation(const EST_String &relname)
    { return (this == 0) ? 0 : p_contents->Relation(relname); }
    /// View item from another relation (const char *) method
    EST_Item *as_relation(const char *relname)
    { return (this == 0) ? 0 : p_contents->Relation(relname); }

    /// TRUE if this item is in named relation
    int in_relation(const EST_String &relname)
    { return (this == 0) ? 0 : p_contents->in_relation(relname); }

    /// Access to the KVL of relation links
    KVL <EST_String, EST_Item *> &relations() 
    { return p_contents->relations; }

    /// The relation name of this particular item
    const EST_String &relation_name() const;
    /// The relation of this particular item
    EST_Relation *relation(void) 
       { return (this == 0) ? 0 : p_relation; }
    /// True if li is the same item ignoring its relation viewpoint
    int same_item(EST_Item *li)
    { return contents() && li->contents() && (contents() == li->contents()); }
    //@}

    /**@name Traversal within this relation */
    ///
    EST_Item *next() const { return this == 0 ? 0 : n; }
    ///
    EST_Item *prev() const { return this == 0 ? 0 : p; }
    ///
    EST_Item *down() const { return this == 0 ? 0 : d; }
    ///
    EST_Item *up() const   { return this == 0 ? 0 : u; }
    /// Last item (most next) at this level
    EST_Item *last() const;
    /// First item (most prev) at this level
    EST_Item *first() const;
    /// Highest (most up)
    EST_Item *top() const;
    /// Lowest (most down)
    EST_Item *bottom() const;
    /// First item which has no down, within the descendants of this item
    EST_Item *first_leaf() const;
    /// Next item which has no down, following above this item if necessary
    EST_Item *next_leaf() const;
    /// Last item which has no down, following above this item if necessary
    EST_Item *last_leaf() const;
    /// Next item in pre order (root, daughters, siblings)
    EST_Item *next_item() const;
    /// The parent in a tree relation
    EST_Item *parent() const { return first()->up(); }
    /// The first daughter in a tree relation
    EST_Item *daughter1() const { return down(); }
    /// The second daughter in a tree relation
    EST_Item *daughter2() const { return down()->next(); }
    /// The last daughter in a tree relation
    EST_Item *daughtern() const { return down()->last(); }
    /// The first daughter in a tree relation
    EST_Item *daughters() const { return down(); }
    /// Number of items (including this) until no next item.
    int length() const;
    //@}

    /**@name Base construction methods */
    /// Insert a new item after this, with li's contents
    EST_Item *insert_after(EST_Item *li);
    /// Insert a new item after this
    EST_Item *insert_after();
    /// Insert a new item before this, with li's contents
    EST_Item *insert_before(EST_Item *li);
    /// Insert a new item before this
    EST_Item *insert_before();
    /// Insert a new item below this, with li's contents (see tree methods)
    EST_Item *insert_below(EST_Item *li);
    /// Insert a new item below this (see tree methods)
    EST_Item *insert_below();
    /// Insert a new item above this, with li's contents (see tree methods)
    EST_Item *insert_above(EST_Item *li);
    /// Insert a new item above this (see tree methods)
    EST_Item *insert_above();
    //@}

    /**@name Tree  construction methods */
    /// Append a new daughter to this, with li's contents
    EST_Item *append_daughter(EST_Item *li);
    /// Append a new daughter to this
    EST_Item *append_daughter();
    /// Insert a new parent above this, with li's contents
    EST_Item *insert_parent(EST_Item *li);
    /// Insert a new parent above this, with li's contents
    EST_Item *insert_parent();

    // Delete this item and all its occurrences in other relations
    void unref_all();

    // Verification, double links are consistent (used after reading in)
    int verify() const;
	
    friend int same_item(EST_Item *l1,EST_Item *l2)
    { return l1->contents() && l2->contents() && 
	  (l1->contents() == l2->contents()); }
    friend int move_item(EST_Item *from, EST_Item *to);
    friend int merge_item(EST_Item *from, EST_Item *to);
    friend int move_sub_tree(EST_Item *from, EST_Item *to);
    friend int exchange_sub_trees(EST_Item *from,EST_Item *to);

    EST_Item &operator=(const EST_Item &s);
    friend ostream& operator << (ostream &s, const EST_Item &a);
    friend bool operator !=(const EST_Item &a, const EST_Item &b)
    { return &a != &b; }
    friend bool operator ==(const EST_Item &a, const EST_Item &b)
    { return &a == &b; }

    friend class EST_Relation;
};

inline EST_Item *next(EST_Item *n) { return n->next(); }
inline EST_Item *prev(EST_Item *n) { return n->prev(); }
inline EST_Item *up(EST_Item *n) { return n->up(); }
inline EST_Item *parent(EST_Item *n) { return n->first()->up(); }
inline EST_Item *down(EST_Item *n) { return n->down(); }
inline EST_Item *daughters(EST_Item *n) { return n->down(); }
inline EST_Item *last(EST_Item *n) { return n->last(); }
inline EST_Item *first(EST_Item *n) { return n->first(); }
inline EST_Item *daughter1(EST_Item *n) { return n->daughter1(); }
inline EST_Item *daughter2(EST_Item *n) { return n->daughter2(); }
inline EST_Item *daughtern(EST_Item *n) { return n->daughtern(); }

inline EST_Item *next(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->next(); }
inline EST_Item *prev(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->prev(); }
inline EST_Item *up(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->up(); }
inline EST_Item *parent(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->first()->up(); }
inline EST_Item *down(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->down(); }
inline EST_Item *daughters(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->down(); }
inline EST_Item *last(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->last(); }
inline EST_Item *first(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->first(); }
inline EST_Item *daughter1(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->daughter1(); }
inline EST_Item *daughter2(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->daughter2(); }
inline EST_Item *daughtern(EST_Item *n,const EST_String &relname)
    { return n->as_relation(relname)->daughtern(); }

inline EST_Item *next_leaf(EST_Item *node) 
    { return node->next_leaf(); }
inline EST_Item *first_leaf(EST_Item *node) 
    { return node->first_leaf(); }
inline EST_Item *last_leaf(EST_Item *node) 
    { return node->last_leaf(); }
inline EST_Item *next_item(EST_Item *node) 
    { return node->next_item(); }

inline EST_Item *append_daughter(EST_Item *n) 
      { return n->append_daughter(); }
inline EST_Item *append_daughter(EST_Item *n, EST_Item *p)
      { return n->append_daughter(p); }
inline EST_Item *append_daughter(EST_Item *n, 
				      const EST_String &relname)
      { return n->as_relation(relname)->append_daughter(); }
inline EST_Item *append_daughter(EST_Item *n, 
				      const EST_String &relname,
				      EST_Item *p)
      { return n->as_relation(relname)->append_daughter(p); }

EST_Item *first_leaf_in_tree(EST_Item *root);
EST_Item *last_leaf_in_tree(EST_Item *root);

int in_list(EST_Item *c, EST_Item *l);
int in_tree(EST_Item *c, EST_Item *t);

void remove_item(EST_Item *l, const EST_String &relname);

void copy_node_tree(EST_Item *from, EST_Item *to);
void copy_node_tree_contents(EST_Item *from, EST_Item *to);

// Feature function support
typedef EST_Val (*EST_Item_featfunc)(EST_Item *s);
void register_featfunc(const EST_String &name, EST_Item_featfunc func);
EST_Item_featfunc get_featfunc(const EST_String &name);

#endif
