/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1997,1998                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                Author :  Alan W Black                                 */
/*                Date   :  July 1997                                    */
/*-----------------------------------------------------------------------*/
/*  Optional support for /dev/dsp under FreeBSD and Linux                */
/*  These use the same underlying sound drivers (voxware).  This uses    */
/*  16bit linear if the device supports it otherwise it uses 8bit.  The  */
/*  8bit dirver is still better than falling back to the "sunaudio" ulaw */
/*  8K as this driver can cope with various sample rates (and saves on   */
/*  resampling).                                                         */
/*                                                                       */
/*  Combined FreeBSD and Voxware code Feb 98                             */
/*                                                                       */
/*  This may work on NetBSD and OpenBSD but I haven't tried it           */
/*                                                                       */
/*=======================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include "EST_cutils.h"
#include "EST_walloc.h"
#include "EST_Wave.h"
#include "EST_Option.h"
#include "audioP.h"
#include "EST_io_aux.h"

#ifdef SUPPORT_FREEBSD16
#include <machine/soundcard.h>
#include <fcntl.h>
int freebsd16_supported = TRUE;
int linux16_supported = FALSE;
static char *aud_sys_name = "FreeBSD";
#endif

#ifdef SUPPORT_LINUX16
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
int linux16_supported = TRUE;
int freebsd16_supported = FALSE;
static char *aud_sys_name = "Linux";
#endif

#if defined(SUPPORT_LINUX16) || defined(SUPPORT_FREEBSD16)

static int sb_set_sample_rate(int sbdevice, int samp_rate)
{
    int fmt;
    int sfmts;

    ioctl(sbdevice,SNDCTL_DSP_RESET,0);
    ioctl(sbdevice,SNDCTL_DSP_SPEED,&samp_rate);
    ioctl(sbdevice,SNDCTL_DSP_GETFMTS,&sfmts);

    if (sfmts == AFMT_U8)
	fmt = AFMT_U8;         // its really an 8 bit only device
    else if (EST_LITTLE_ENDIAN)
	fmt = AFMT_S16_LE;  
    else
	fmt = AFMT_S16_BE;  
    
    ioctl(sbdevice,SNDCTL_DSP_SETFMT,&fmt);
    
    return fmt;
}

#define AUDIOBUFFSIZE 256

int play_voxware_wave(EST_Wave &inwave, EST_Option &al)
{
    int sample_rate;
    short *waveform;
    int num_samples;
    int audio,actual_fmt;
    int i,r,n;
    (void)al;
    
    if ((audio = open("/dev/dsp",O_WRONLY)) == -1)
    {
	cerr << aud_sys_name << ": can't open /dev/dsp" << endl;
	return -1;
    }
 
    //int tmp=open("/tmp/vox_play_wave",O_WRONLY|O_CREAT);
    
    waveform = inwave.values().memory();
    num_samples = inwave.num_samples();
    sample_rate = inwave.sample_rate();

    actual_fmt = sb_set_sample_rate(audio,sample_rate);
    
    if (sb_set_sample_rate(audio,sample_rate) == AFMT_U8)
    {
	// Its actually 8bit unsigned so convert the buffer;
	unsigned char *uchars = walloc(unsigned char,num_samples);
	for (i=0; i < num_samples; i++)
	    uchars[i] = waveform[i]/256+128;
	for (i=0; i < num_samples; i += r)
	{
	    if (num_samples > i+AUDIOBUFFSIZE)
		n = AUDIOBUFFSIZE;
	    else
		n = num_samples-i;
	    r = write(audio,&uchars[i], n);
	    if (r == 0)
	    {
		cerr << aud_sys_name << ": failed to write to buffer" <<
		    sample_rate << endl;
		close(audio);
		return -1;
	    }
	}
	wfree(uchars);
    }
    else if ((actual_fmt == AFMT_S16_LE) || 
	     (actual_fmt == AFMT_S16_BE))
    {
	for (i=0; i < num_samples; i += r/2)
	{
	    if (num_samples > i+AUDIOBUFFSIZE)
		n = AUDIOBUFFSIZE;
	    else
		n = num_samples-i;
	    // r = write(tmp,&waveform[i], n*2);
	    r = write(audio,&waveform[i], n*2);
	    if (r == 0)
	    {
		cerr << aud_sys_name << ": failed to write to buffer" <<
		    sample_rate << endl;
		close(audio);
		return -1;
	    }
	}
    }
    else
    {
	cerr << aud_sys_name << ": unable to set sample rate " <<
	    sample_rate << endl;
	close(audio);
	return -1;
    }
    
    // close(tmp);
    close(audio);
    return 1;
}


#else

int linux16_supported = FALSE;
int freebsd16_supported = FALSE;

int play_voxware_wave(EST_Wave &inwave, EST_Option &al)
{
    (void)inwave;
    (void)al;
    cerr << "Audio: voxware not compiled in this version" << endl;
    return -1;
}

#endif
