#include "cthugha.h"
#include "sound.h"
#include "options.h"
#include "display.h"
#include "cd_player.h"
#include "information.h"
#include "interface.h"
#include "server.h"
#include "imath.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <signal.h>
#include <memory.h>
#include <sys/wait.h>

/*
 * public Variables 
 */
char2 * sound_data;			/* Sound-Data in 8 bit signed, stereo */
char2 * sound_data_unproc;		/* Sound-Data in 8 bit signed, stereo, unprocessed */
int sound_bsize = 0;			/* size of sound-buffer (samples) 
					   always a power of 2 */
int sound_fork = 0;			/* read in an extra process */
int sound_use[snddev_max] = {
#if USE_DSP == 1	/* dsp reading */
    1,
#else
    0,
#endif

    0,			/* dsp writing */

#if USE_MIXER == 1	/* mixer */
    1,
#else
    0,
#endif

    0,			/* CD */
    0,			/* debug */
    0,			/* sock */
    0,			/* play */
    0,			/* exec */
};

sound_opt sound_options_init = {
    2,					/* number of channels */
    44000,				/* speed / sample rate */
    8,					/* bits per sample */
    0					/* how many fragments to use */
};
sound_opt * sound_options = & sound_options_init;
sound_action_t * sound_action;
int * sound_stop;


/* 
 * local variables
 */
static int sound_child;
static int sound_shm_key;
static int sound_alloc = 0;		/* sound currently allocated */


static void * sound_shared;		/* pointer to the shared memory */


int sound_use_clear() {
    memset(sound_use, 0, sizeof(int)*snddev_max);
    return 0;
}

/* 
 *  Initialize the Sound-Interface
 */
int init_sound() {

    /* make sue the sound_volumes is initialized */
#if USE_MIXER == 1
    mixer_init_names();
#endif

    if ( sound_use[snddev_cd] ) {
#if USE_CDROM == 1
	init_cd();
#else
	printfe("CD support was disabled.\n");
	return 1;
#endif
    }

    sound_bsize =  1 << ilog2(max(BUFF_WIDTH, BUFF_HEIGHT));

    if( alloc_sound() )
	return 1;

    return 0;
}


/*
 * Clean up the sound-interface
 */
int exit_sound() {
#if USE_CDROM == 1
    if ( sound_use[snddev_cd] )
	exit_cd();
#endif

    return free_sound();
}

/* 
 * get the sound devices
 */
int alloc_sound() {

    /* check, if already allocated */
    if(sound_alloc)
	return 0;


    sound_data_unproc = cth_memory(NULL, sound_bsize*2, "Can not allocate sound_data.");
    sound_data = cth_memory(NULL, sound_bsize*2, "Can not allocate sound_data.");

    if(sound_fork) {
	if( sound_fork_process() == 0) {
	    sound_alloc = 1;		/* only allocate once */
	    return 0;
	}
    }


    /* 
     * initialize only the sound devices needed
     */
	
    if( sound_use[snddev_sock]) {
	printfv(2, "  initializing network sound.\n");
	if( sound_net_init() )
	    return 1;
    }
	
    if( sound_use[snddev_play]) {
	printfv(2, "  initializing play sound.\n");
	if( sound_play_init() )
	    return 1;
    }
	
    if( sound_use[snddev_exec]) {
	printfv(2, "  initializing exec sound.\n");
	if ( sound_exec_init() )
	    return 1;
    }
	
    if( sound_use[snddev_mixer]) {
#if USE_MIXER == 1
	printfv(2, "  initializing mixer.\n");
	if( sound_set_mixer() )
	    return 1;
#else
	printfv(0, "Mixer was disabled.\n");
	sound_use[snddev_mixer] = 0;
#endif
    }
	
    if( sound_use[snddev_dsp_r]) {
#if USE_DSP == 1
	printfv(2, "  initiliazing dsp for reading.\n");
	if( sound_dsp_init(O_RDONLY) )
	    return 1;
#else
	printfe("DSP was disabled.\n");
	return 1;
#endif
    }
	
    if( sound_use[snddev_dsp_w]) {
#if USE_DSP == 1
	printfv(2, "  initiliazing dsp for writing.\n");
	if( sound_dsp_init(O_WRONLY) )
	    return 1;
#else
	printfe("DSP was disabled.\n");
	return 1;
#endif
    }

    sound_alloc = 1;		/* only allocate once */
    return 0;
}



void sig_tty_parent(int a) {
    printfv(0, "Stopping in child...\n");

    kill(getppid(), SIGTSTP);

    *sound_stop = 1;
}

/*
 * fork the sound reading process
 */
int sound_fork_process() {
    printfv(3,"    starting sound reading process\n");
    if( (sound_shm_key = shmget(IPC_PRIVATE, sound_bsize*2 + 
				sizeof(sound_opt) + sizeof(sound_action_t) + sizeof(int), 
				IPC_CREAT | 0777) ) == -1) {
	printfee("Can not create shared memory segment.");
	sound_fork = 0;
	return 1;
    }
    if( (sound_shared=shmat(sound_shm_key, 0, 0)) == (void*)-1) {
	printfee("Can not attach shared memory segment.");
	sound_fork = 0;
	return 1;
    }

    /* point the sound_options to the shared memory segment */
    sound_options = (sound_opt*)sound_shared;
    *sound_options = sound_options_init;
    sound_shared += sizeof(sound_opt);

    sound_action = (sound_action_t*)sound_shared;
    *sound_action = sa_continue;
    sound_shared += sizeof(sound_opt);

    sound_stop = (int*)sound_shared;
    *sound_stop = 0;
    sound_shared += sizeof(int);


    switch( sound_child = fork() ) {
    case -1:		/* error */
	printfee("Can not fork sound child.");
	shmdt((char*)sound_data);
	sound_fork = 0;
	return 1;

    case 0:		/* child */
	sound_fork = 0;			/* don't fork again */
	if(alloc_sound()) {		/* open sound device */
	    abort();
	}

	/* set signal handler */
	signal(SIGTSTP, sig_tty_parent);	/* react to ^Z */

	sound_data = sound_shared;

	do {
	    sound_read();
	    switch(*sound_action) {
	    case sa_update:		/* update DSP sound */
		sound_dsp_update();
		*sound_action = sa_continue;
		break;

	    case sa_continue:
		;
	    }
	} while(*sound_stop == 0);
	printfv(3, "    closing sound reading child.\n");
	abort();
    }

    return 0;
}

/*
 * kill the sound reading process
 */
int sound_kill_process() {
    int dummy;

    printfv(3,"    stopping sound reading process.\n");
    *sound_stop = 1;

    wait(&dummy);

    fflush(stdout);

    shmdt((char*)sound_shared);
    shmctl(sound_shm_key, IPC_RMID, 0);

    return 0;
}

/*
 * free the sound device
 */
int free_sound() {

    if(!sound_alloc)
	return 0;

    if(sound_fork) {
	sound_kill_process();
    } else {
	free(sound_data);
    }

    if(sound_use[snddev_sock])
	sound_net_exit();
    if(sound_use[snddev_play])
	sound_play_exit();
    if(sound_use[snddev_exec])
	sound_exec_exit();
#if USE_DSP == 1
    if(sound_use[snddev_dsp_r])
	sound_dsp_exit();
#endif

    sound_alloc = 0;
    return 0;
}    

/*****************************************************************************/

int sound_read() {

    if(sound_use[snddev_debug])
	sound_random_read();
    if(sound_use[snddev_sock])
	sound_net_read();
    if(sound_use[snddev_play])
	sound_play_read();
    if(sound_use[snddev_exec])
	sound_exec_read();
#if USE_DSP == 1
    if(sound_use[snddev_dsp_r])
	sound_dsp_read();
#endif

    return 0;
}

/*
 * Get a "line" of sound from the soundcard.
 */
int get_sound() {
    if(sound_fork) {
	memcpy(sound_data, sound_shared, sound_bsize*sound_options->channels);
    } else {
	sound_read();
    }
    
    memcpy(sound_data_unproc, sound_data, sound_bsize*sound_options->channels);

    return 0;
}

/*****************************************************************************/


int sound_convert(unsigned char * data, int stereo, int bps) {
    unsigned int  * data_i = (unsigned int*)data;
    int i;

    if( (bps == 8) && (stereo == 2) ) {
	for(i=0; i < sound_bsize; i++) {
	    sound_data[i][1] = (int)(*data ++) - 128;
	    sound_data[i][0] = (int)(*data ++) - 128;
	}
    } else if( (bps == 8) && (stereo == 1) ) {
	for(i=0; i < sound_bsize; i++) {
	    sound_data[i][0] = (int)(*data) - 128;
	    sound_data[i][1] = (int)(*data ++) - 128;
	}	
    } else if( (bps == 16) && (stereo == 2) ) {
	for(i=0; i < sound_bsize; i++) {
	    sound_data[i][1] = (int)((*data_i ++) >> 8) - 128;
	    sound_data[i][0] = (int)((*data_i ++) >> 8) - 128;
	}
    } else if( (bps == 16) && (stereo == 1) ) {
	for(i=0; i < sound_bsize; i++) {
	    sound_data[i][1] = (int)((*data_i) >> 8) - 128;
	    sound_data[i][0] = (int)((*data_i ++) >> 8) - 128;
	}
    } 
    return 0;
}





