//
// STIL class
//
// AUTHOR: LaLa
// Email : lala@interaccess.com
// (C) 1998 by LaLa
//

//
// Given the location of HVSC this class can extract STIL information for a
// given tune of a given SID file. (Sounds simple, huh?)
//
// PLEASE, READ THE ACCOMPANYING README.TXT FILE BEFORE PROCEEDING!!!!
//

#ifndef _STIL_H
#define _STIL_H

#include <fstream.h>
#include "stilcomm.h"

class STIL {

    public:

        // Enum to use for asking for specific fields.
        enum STILField {
            all,
            title,
            artist,
            comment
        };

        // Enum that describes the possible errors this class may encounter.
        enum STILerror {
            NO_STIL_ERROR = 0,
            BUG_OPEN,           // INFO ONLY: failed to open BUGlist.txt.
            WRONG_DIR,          // INFO ONLY: absolute path was not within HVSC base dir.
            NOT_IN_STIL,        // INFO ONLY: requested entry was not found in STIL.txt.
            NOT_IN_BUG,         // INFO ONLY: requested entry was not found in BUGlist.txt.
            CRITICAL_STIL_ERROR = 5,
            BASE_DIR_LENGTH,    // The length of the HVSC base dir was wrong.
            STIL_OPEN,          // Failed to open STIL.txt.
            NO_EOL,             // Failed to determine EOL char(s).
            NO_STIL_DIRS,       // Failed to get sections (subdirs) when parsing STIL.txt.
            NO_BUG_DIRS         // Failed to get sections (subdirs) when parsing BUGlist.txt.
        };

        // To turn debug output on
        bool STIL_DEBUG;

        //----//

        // CONSTRUCTOR
        //      Allocates necessary memory
        STIL();

        // DECONSTRUCTOR
        //      Deallocates the necessary memory
        ~STIL();

        //
        // getVersion()
        //
        // FUNCTION: Returns a formatted string telling what the version
        //           number is for the STIL class and other info.
        //           If it is called after setBaseDir(), the string also
        //           has the STIL.txt file's version number in it.
        // ARGUMENTS:
        //      NONE
        // RETURNS:
        //      char * - printable formatted string with version and copyright
        //               info
        char *getVersion();

        //
        // getVersionNo()
        //
        // FUNCTION: Returns a floating number telling what the version
        //           number is of this STIL class.
        // ARGUMENTS:
        //      NONE
        // RETURNS:
        //      float - version number
        float getVersionNo();

        //
        // setBaseDir()
        //
        // FUNCTION: Tell the object where the HVSC base directory is - it
        //           figures that the STIL should be in /DOCUMENTS/STIL.txt
        //           and that the BUGlist should be in /DOCUMENTS/BUGlist.txt.
        //           It should not matter whether the path is given in UNIX,
        //           WinDOS, or Mac format (ie. '\' vs. '/' vs. ':')
        // ARGUMENTS:
        //      pathToHVSC = HVSC base directory in your machine's format
        // RETURNS:
        //      false - Problem opening or parsing STIL/BUGlist
        //      true  - All okay
        bool setBaseDir(const char *pathToHVSC);

        //
        // getSTILVersionNo()
        //
        // FUNCTION: Returns a floating number telling what the version
        //           number is of the STIL.txt file.
        //           To be called only after setBaseDir()!
        // ARGUMENTS:
        //      NONE
        // RETURNS:
        //      float - version number (0.0 if setBaseDir() was not called, yet)
        float getSTILVersionNo();

        //
        // getEntry()
        //
        // FUNCTION: Given an HVSC pathname, a tune number, a timestamp, and a
        //           field designation, it returns a formatted string that
        //           contains the STIL field for the tune number (if exists).
        //           If it doesn't exist, returns a NULL.
        //           If a timestamp is specified and the STIL entry is a
        //           multiple cover entry, the field corresponding to the
        //           give timestamp is returned, not the whole entry.
        // NOTE: TIMESTAMP IS NOT IMPLEMENTED, YET!
        // ARGUMENTS:
        //      relPathToEntry = relative to the C64Music base dir, starting with
        //                       a slash
        //      tuneNo         = song number within the song (default=1).
        //                       For single tune entries:
        //                       - tuneNo=0 is equivalent to saying field=all
        //                       - tuneNo != 0, field=all : all of the STIL entry is
        //                         returned.
        //                       - tuneNo != 0, field != all: the given field is returned.
        //                       For multitune entries:
        //                       - tuneNo=0, field=all : all of the STIL entry is returned.
        //                       - tuneNo=0, field=comment : the tune-global comment is
        //                          returned.
        //                       For all other tune numbers 'tuneNo' and 'field' behaves
        //                       just like for single tune entries, but only taking into
        //                       account the specified tune number.
        //      timestamp      = number of seconds within the SID tune (default=0).
        //                       If there are no timestamps in the STIL entry,
        //                       the value of this parameter is ignored.
        //                       NOT IMPLEMENTED, YET!
        //      field          = which field to retrieve (default=all).
        // RETURNS:
        //      NULL - if there's absolutely no STIL entry for the tune
        //      char * - pointer to a printable formatted string containing
        //               the STIL entry
        //
        char *getEntry(const char *relPathToEntry, int tuneNo=1, int timestamp=0, STILField field=all);

        // Same as above, but with an absolute path given
        // given in your machine's format.
        //
        char *getAbsEntry(const char *absPathToEntry, int tuneNo=1, int timestamp=0, STILField field=all);

        //
        // getGlobalComment()
        //
        // FUNCTION: Given an HVSC pathname and tune number it returns a
        //           formatted string that contains the section-global
        //           comment for the tune number (if it exists). If it
        //           doesn't exist, returns a NULL.
        // ARGUMENTS:
        //      relPathToEntry = relative to the C64Music base dir starting with
        //                       a slash
        // RETURNS:
        //      NULL - if there's absolutely no section-global comment
        //             for the tune
        //      char * - pointer to a printable formatted string containing
        //               the section-global comment
        //
        char *getGlobalComment(const char *relPathToEntry);

        // Same as above, but with an absolute path
        // given in your machine's format.
        //
        char *getAbsGlobalComment(const char *absPathToEntry);

        //
        // getBug()
        //
        // FUNCTION: Given an HVSC pathname and tune number it returns a
        //           formatted string that contains the BUG entry for the
        //           tune number (if exists). If it doesn't exist, returns
        //           a NULL.
        // ARGUMENTS:
        //      relPathToEntry = relative to the C64Music base dir starting with
        //                       a slash
        //      tuneNo         = song number within the song (default=1)
        //                       If tuneNo=0, returns all of the BUG entry.
        //                       If there's only one tune in the given SID, the
        //                       value of this parameter is ignored.
        // RETURNS:
        //      NULL - if there's absolutely no BUG entry for the tune
        //      char * - pointer to a printable formatted string containing
        //               the BUG entry
        //
        char *getBug(const char *relPathToEntry, int tuneNo=1);

        // Same as above, but with an absolute path
        // given in your machine's format.
        //
        char *getAbsBug(const char *absPathToEntry, int tuneNo=1);

        //
        // getError()
        //
        // FUNCTION: Returns a specific error number identifying the problem
        //           that happened at the last invoked public method.
        // ARGUMENTS:
        //      NONE
        // RETURNS:
        //      STILerror - an enumerated error value
        //
        inline STILerror getError() {return (lastError);}

        //
        // hasCriticalError()
        //
        // FUNCTION: Returns true if the last error encountered was critical
        //           (ie. not one that the STIL class can recover from).
        // ARGUMENTS:
        //      NONE
        // RETURNS:
        //      bool - true if the last error encountered was critical
        //
        inline bool hasCriticalError() {
            return ( (lastError >= CRITICAL_STIL_ERROR) ? true : false);
        }
        
        //
        // getErrorStr()
        //
        // FUNCTION: Returns an ASCII error string containing the
        //           description of the error that happened at the last
        //           invoked public method.
        // ARGUMENTS:
        //      NONE
        // RETURNS:
        //      char * - pointer to string with the error description
        //
        inline const char *getErrorStr() {return (STIL_ERROR_STR[lastError]);}

    private:

        // Version number/copyright string
        char versionString[2*STIL_MAX_LINE_SIZE];

        // STIL.txt's version number
        float STILVersion;

        // Base dir
        char baseDir[STIL_MAX_PATH_SIZE];
        size_t baseDirLength;

        // File handles
        ifstream stilFile;
        ifstream bugFile;

        // List of sections (subdirs) for easier positioning
        struct dirList {
            char *dirName;
            streampos position;
        } stilDirs[STIL_MAX_DIRS], bugDirs[STIL_MAX_DIRS];

        // This tells us what the line delimiter is in STIL.txt.
        // (It may be two chars!)
        char STIL_EOL;
        char STIL_EOL2;

        // Error number of the last error that happened.
        STILerror lastError;

        // Error strings containing the description of the possible errors in STIL.
        static const char *STIL_ERROR_STR[];

        ////////////////

        // The last retrieved entry
        char entrybuf[STIL_MAX_ENTRY_SIZE];

        // The last retrieved section-global comment
        char globalbuf[STIL_MAX_ENTRY_SIZE];

        // The last retrieved BUGentry
        char bugbuf[STIL_MAX_ENTRY_SIZE];

        // Buffers to hold the resulting strings
        char resultEntry[STIL_MAX_ENTRY_SIZE];
        char resultBug[STIL_MAX_ENTRY_SIZE];

        ////////////////

        //
        // determineEOL()
        //
        // FUNCTION: Determines what the EOL char is (or are) from STIL.txt.
        //           It is assumed that BUGlist.txt will use the same EOL.
        // ARGUMENTS:
        //      NONE
        // RETURNS:
        //      false - something went wrong
        //      true  - everything is okay
        //
        bool determineEOL();

        //
        // getDirs()
        //
        // FUNCTION: Populates the given dirList array with the directories
        //           obtained from 'inFile' for faster positioning within
        //           'inFile'.
        // ARGUMENTS:
        //      inFile - where to read the directories from
        //      dirs   - the dirList array that should be populated with the
        //               directory list
        //      isSTILFile - is this the STIL or the BUGlist we are parsing
        // RETURNS:
        //      false - No entries were found or otherwise failed to process
        //              inFile
        //      true  - everything is okay
        //
        bool getDirs(ifstream& inFile, dirList *dirs, bool isSTILFile);

        //
        // positionToEntry()
        //
        // FUNCTION: Positions the file pointer to the given entry in 'inFile'
        //           using the 'dirs' dirList for faster positioning.
        // ARGUMENTS:
        //      entryStr - the entry to position to
        //      inFile   - position the file pointer in this file
        //      dirs     - the list of dirs in inFile for easier positioning
        // RETURNS:
        //      true - if successful
        //      false - otherwise
        bool positionToEntry(const char *entryStr, ifstream& inFile, dirList *dirs);

        //
        // readEntry()
        //
        // FUNCTION: Reads the entry from 'inFile' into 'buffer'. 'inFile' should
        //           already be positioned to the entry to be read.
        // ARGUMENTS:
        //      inFile   - filehandle of file to read from
        //      entryStr - the entry needed to be read
        //      buffer   - where to put the result to
        // RETURNS:
        //      NONE
        void readEntry(ifstream& inFile, char *buffer);

        //
        // getField()
        //
        // FUNCTION: Given a STIL formatted entry in 'buffer', a tune number, a
        //           timestamp, and a field designation, it returns the requested
        //           STIL field into 'result'.
        //           If field=all, it also puts the tune-global comment (if it exists)
        //           as the first field into 'result'.
        // ARGUMENTS:
        //      result - where to put the resulting string to (if any)
        //      buffer - pointer to the first char of what to search for
        //               the field. Should be a buffer in standard STIL
        //               format.
        //      tuneNo - song number within the song (default=0)
        //               If tuneNo=0, returns all of the entry.
        //               If there's only one tune in the given SID, the
        //               value of this parameter is ignored.
        //      timestamp - number of seconds within the SID tune (default=0).
        //               If there are no timestamps in the buffer,
        //               the value of this parameter is ignored.
        //      field  - which field to retrieve (default=all).
        // RETURNS:
        //      false - if nothing was put into 'result'
        //      true  - 'result' has the resulting field
        bool getField(char *result, char *buffer, int tuneNo=0, int timestamp=0, STILField field=all);

        //
        // getOneField()
        //
        // FUNCTION:
        // ARGUMENTS:
        //      result - where to put the resulting string to (if any)
        //      start  - pointer to the first char of what to search for
        //               the field. Should be a buffer in standard STIL
        //               format.
        //      end    - pointer to the last+1 char of what to search for
        //               the field. ('end-1' should be a '\n'!)
        //      field  - which specific field to retrieve
        // RETURNS:
        //      false - if nothing was put into 'result'
        //      true  - 'result' has the resulting field
        bool getOneField(char *result, char *start, char *end, STILField field);

        //
        // getStilLine()
        //
        // FUNCTION: Extracts one line from 'infile' to 'line[]'. The end of
        //           the line is marked by endOfLineChar. Also eats up
        //           additional EOL-like chars.
        // ARGUMENTS:
        //      infile - filehandle (streampos should already be positioned
        //               to the start of the desired line)
        //      line   - char array to put the line into
        // RETURNS:
        //      NONE
        void getStilLine(ifstream& infile, char *line);

        //
        // addNewLine()
        //
        // FUNCTION: Adds a newline to 'str' and terminates it with NULL
        //           (takes into account the UNIX/DOS difference).
        // ARGUMENTS:
        //      str - string to add newline to
        // RETURNS:
        //      NONE
        //
        inline void addNewline(char *str);
};

#endif // _STIL_H
