
/*
 * DIABLO.C	INTERNET NEWS TRANSIT AGENT, By Matthew Dillon
 *
 * Diablo implements the news transfer portion of the INN command
 * set.  Basically ihave, check, takethis, mode stream, head, and stat.
 * Not much more.  The purpose is to transfer news as quickly and
 * as efficiently as possible.
 *
 * Diablo is a forking server.  It supports both standard non-streaming
 * ihave feeds and streaming check/takethis feeds.  In fact, it supports
 * streaming on its entire command set if you wish to use it that way.
 * It forks on every connection and is able to do history file lookups
 * and article reception in parallel.  History file updates use minimal
 * locking and take advantage of the flexibility of INN's return codes.
 *
 * There is no active file or newsgroups file, and Control's are not 
 * processed in any form (not even cancels).  DIABLO is strictly a
 * backbone redistribution/transit agent.
 *
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 */

#include "defs.h"

#if NEED_TERMIOS
#include <sys/termios.h>
#endif

/* #define SLOT_DEBUG	1 */

typedef struct Feed {
    struct Feed *fe_Next;
    char	*fe_Label;
    int		fe_Fd;
    char	*fe_Buf;
    int		fe_BufIdx;
    int		fe_BufMax;
    int		fe_Failed;
} Feed;

typedef struct Retain {
    struct Retain *re_Next;
    FILE	*re_Fi;
    FILE	*re_Fo;
    int		re_What;
} Retain;

typedef struct Track {
    pid_t	tr_Pid;
    struct in_addr tr_Addr;
} Track;

#define RET_CLOSE	1
#define RET_PAUSE	2
#define RET_LOCK	3

#define REJMSGSIZE	32

void DiabloServer(void);
void DoAccept(int lfd);
void DoPipe(int fd);
void DoSession(int fd, int count);
void LogSession(void);
void LogSession2(void);
void DoCommand(int ufd);
int LoadArticle(Buffer *bi, const char *msgid, int noWrite, int headerOnly, char *refBuf);
void ArticleFileInit(void);
int ArticleFile(History *h, int *pbpos, const char *nglist, const char *msgid);
void ArticleFileCloseAll(void);
void ArticleFileClose(int i);
void ArticleFileTrunc(int artFd, off_t bpos);
void ArticleFileSetSize(int artFd);
void ngAddControl(char *nglist, int ngSize, const char *ctl);

void writeFeedOut(const char *label, const char *file, const char *msgid, const char *offSize, int rt, int headOnly);
void flushFeeds(int justClose);
void flushFeedOut(Feed *fe);

void FeedRSet(FILE *fo);
void FeedCommit(FILE *fo);
void FeedAddDel(FILE *fo, char *gwild, int add);

void FinishRetain(int what);
int QueueRange(const char *label, int *pqnum, int *pqarts, int *pqrun);
const char *ftos(double d);
int countFds(fd_set *rfds);

char	*HName;
char	*NewsPathName;
char	*CommonPathName;
char	HostName[256];
char	HLabel[256];
int	MaxFds;
int	NumForks;
int	ReadOnlyCount = 0;
int	PausedCount;
fd_set	RFds;
Buffer	*PipeAry[MAXFDS];
Feed	*FeBase;
Retain	*ReBase;
Track	PidAry[MAXFDS];
volatile int Exiting = 0;
int	DidFork = 0;
int	NewsPort = 119;
struct in_addr NewsBindHost;
int	NumForksExceeded;
time_t	SessionBeg;
time_t	SessionMark;
int	ArtsEntered;
int	ArtsSpam;
int	ArtsReceived;
int	ArtsIHave;
int	ArtsCheck;
int	ArtsPColl;
int	ArtsRej;
int	ArtsStage2Rej;
int	ArtsStage3Rej;
int	ArtsErr;
int	TxBufSize;
int	RxBufSize;
int	ArtsBytes;
double	TtlArtsReceived;
double	TtlArtsTested;
double	TtlArtsBytes;
double	TtlArtsFed;
int	LogCount;
int	FeedTableReady;
char	*DebugLabel;
char	PeerIpName[64];
hash_t	PeerIpHash;
int	HasStatusLine;
int	RTFileSize = 16 * 1024 * 1024;	/* 16 MBytes */
int	SpamFilterOpt = 1;
int	RejectArtsWithNul;
MemPool	*ParProcMemPool;
char	*MySpoolHome;
int	AllowReadOnly = 0;
int	ReadOnlyMode = 0;
pid_t	HostCachePid = 0;

#if DIABLO_FILTER
    int diablofilterenabled = 1;
#endif

#define REJ_ACCEPTED    0	/* not rejected, accepted		*/
#define REJ_CTLMSG	1	/* not rejected, accepted was control	*/
#define REJ_FAILSAFE    2	/* rejected, failsafe			*/
#define REJ_MISSHDRS    3	/* rejected, missing headers		*/
#define REJ_TOOOLD      4	/* rejected, too old			*/
#define REJ_GRPFILTER   5	/* rejected, incoming group filter	*/
#define REJ_SPAMFILTER  6	/* rejected, spam filter		*/
#define REJ_INSTANTEXP  7	/* rejected, late expire		*/
#define REJ_IOERROR	8	/* rejected, io error			*/
#define REJ_NOTINACTV	9	/* rejected, not in active file		*/
#define REJ_EARLYEXP	10	/* not rejected, but will expire early	*/
#define REJ_NSLOTS	11

int	RejectStats[REJ_NSLOTS];

#define REJECTART(sti,msgid,msg)	{ ++RejectStats[sti]; if (DebugOpt) ddprintf("%d ** %s\t%s", (int)getpid(), (msgid ? msgid : "??"), msg); }

int
main(int ac, char **av)
{
    char *op = NULL;

    /*
     * On many modern UNIX systems, buffers for stdio are not allocated
     * until the first read or write AND, generally, large buffers (like 64K)
     * are allocated.  Since we print to stdout and stderr but do not really
     * need the buffers, we make them smaller.
     */

    LoadDiabloConfig(ac, av);
    (void)hhash("x");		/* prime hash table prior to forks */

    gethostname(HostName, sizeof(HostName));

    rsignal(SIGPIPE, SIG_IGN);
    SessionBeg = SessionMark = time(NULL);

    srandom((int32)SessionBeg ^ (getpid() * 100));
    random();
    random();

    NewsBindHost.s_addr = INADDR_ANY;

    /*
     * Options
     */

    {
	int i;

	for (i = 1; i < ac; ++i) {
	    char *ptr = av[i];

	    if (*ptr != '-') {
		if (op) {
		    fprintf(stderr, "service option specified twice\n");
		    exit(1);
		}
		op = ptr;
		continue;
	    }
	    ptr += 2;
	    switch(ptr[-1]) {
	    case 'C':
		if (*ptr == 0)
		    ++i;
		break;
	    case 'T':
		TxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
		break;
	    case 'R':
		RxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
		if (RxBufSize < 4096)
		    RxBufSize = 4096;
		break;
	    case 'e':
		SetPreCommitExpire(strtol(((*ptr) ? ptr : av[++i]), NULL, 0));
		break;
	    case 's':
		SetStatusLine(ptr - 2, strlen(ptr - 2));
		HasStatusLine = 1;
		break;
	    case 'd':
		if (isdigit((int)(unsigned char)*ptr)) {
		    DebugOpt = strtol(ptr, NULL, 0);
		} else {
		    --ptr;
		    while (*ptr == 'd') {
			++DebugOpt;
			++ptr;
		    }
		}
		break;
	    case 'h':
		ptr = (*ptr) ? ptr : av[++i];
		if (strlen(ptr) < sizeof(HostName))
		    strcpy(HostName, ptr);
		break;
	    case 'p':
		NewsPathName = (*ptr) ? ptr : av[++i];
		if (strcmp(NewsPathName, "0") == 0)	/* bridge */
		    NewsPathName = "";
		break;
	    case 'c':
		CommonPathName = (*ptr) ? ptr : av[++i];
		if (*CommonPathName == 0)	/* 0-length string no good */
		    CommonPathName = NULL;
		break;
	    case 'P':
		if (*ptr == 0)
		    ptr = av[++i];
		if ((NewsPort = strtol(ptr, NULL, 0)) == 0) {
		    struct servent *sen;
		    if ((sen = getservbyname(ptr, "tcp")) != NULL) {
			NewsPort = ntohl(sen->s_port);
		    } else {
			fprintf(stderr, "Unknown service: %s\n", ptr);
			exit(1);
		    }
		}
		break;
	    case 'H':
		if (*ptr == 0)
		    ptr = av[++i];
		if (strtol(ptr, NULL, 0) > 0) {
		    NewsBindHost.s_addr = inet_addr(ptr);
		} else {
		    struct hostent *he;

		    if ((he = gethostbyname(ptr)) != NULL) {
			NewsBindHost = *(struct in_addr *)he->h_addr;
		    } else {
			fprintf(stderr, "Unknown host for bindhost option: %s\n", ptr);
			exit(1);
		    }
		}
		break;
	    case 'Z':
		RejectArtsWithNul = (*ptr) ? strtol(ptr, NULL, 0) : 1;
		break;
	    case 'S':
		/*
		 * Optionally disable just the NNTPPostingHost portion of
		 * the filter.
		 */
		if (*ptr == 'D') {
		    SetFilterTrip(-1, 0);
		    ++ptr;
		}
		/*
		 * Optionally set trip point (-S0 disables the filter
		 * entirely).
		 */
		if (*ptr) {
		    int n;

		    n = strtol(ptr, NULL, 0);
		    if (n) {
			SpamFilterOpt = 1;
			SetFilterTrip(n, -1);
		    } else {
			SpamFilterOpt = 0;
		    }
		} else {
		    SpamFilterOpt = 1;
		}
		break;
	    case 'M':
		if (*ptr == 0)
		    ptr = av[++i];
		MaxPerRemote = strtol(ptr, NULL, 0);
		break;
#if DIABLO_FILTER
	    case 'F':
		diablofilterenabled = 0;
		break;
#endif
	    case 'r':
		if (*ptr == 0)
		    ptr = av[++i];
		RTFileSize = strtol(ptr, &ptr, 0);
		while (*ptr) {
		    if (*ptr == 'k')
			RTFileSize *= 1024;
		    if (*ptr == 'm')
			RTFileSize *= 1024 * 1024;
		    if (*ptr == 'g')
			RTFileSize *= 1024 * 1024 * 1024;
		    ++ptr;
		}
		break;
	    default:
		fprintf(stderr, "unknown option: %s\n", ptr - 2);
		exit(1);
	    }
	}
	if (i > ac) {
	    fprintf(stderr, "expected argument to last option\n");
	    exit(1);
	}
    }

    /*
     * For our Path: insertion
     */

    if (NewsPathName == NULL) {
	fprintf(stderr, "No '-p newspathname' specified\n");
	exit(1);
    }

    /*
     * Extract the spool home pattern into a real path and place
     * in MySpoolHome
     */

    {
	int l = strlen(PatExpand(SpoolHomePat)) + 1;
	MySpoolHome = malloc(l);
	snprintf(MySpoolHome, l, PatExpand(SpoolHomePat));
    }

    /*
     * The chdir is no longer required, but we do it anyway to
     * have a 'starting point' to look for cores and such.
     */

    if (chdir(MySpoolHome) < 0) {
	fprintf(stderr, "%s: chdir('%s'): %s\n", av[0], MySpoolHome, strerror(errno));
	exit(1);
    }

    if (op == NULL) {
	fprintf(stderr, "Must specify service option: (server)\n");
	exit(1);
    } else if (strcmp(op, "server") == 0) {
	DiabloServer();
    } else {
	fprintf(stderr, "unknown service option: %s\n", op);
	exit(1);
    }
    return(0);
}

/*
 * This needs fork.  What we are trying to accomplish
 * is to ensure that all the pipes from the children
 * are flushed back to the parent and the parent
 * writes them out before exiting.  Otherwise
 * feed redistribution might fail.
 */

void
sigHup(int sigNo)
{
    Exiting = 1;
    if (DidFork) {
	if (FeedFo) {
	    fflush(FeedFo);
	    fclose(FeedFo);
	    FeedFo = NULL;
	}
	LogSession();	/* try, may not work */
	exit(1);
    } else {
	if (Exiting == 0) {
	    int i;

	    Exiting = 1;
	    for (i = 0; i < MAXFDS; ++i) {
		if (PidAry[i].tr_Pid) {
		    kill(PidAry[i].tr_Pid, SIGHUP);
		}
	    }
	}
    }
}

void
sigUsr1(int sigNo)
{
    ++DebugOpt;
}

void
sigUsr2(int sigNo)
{
    DebugOpt = 0;
}

void
sigAlrm(int sigNo)
{
    if (AllowReadOnly) {
	ReadOnlyMode = 1;
	if (DidFork) {
	    struct sockaddr_un soun;
	    int ufd;
	    FILE *fo;

	    memset(&soun, 0, sizeof(soun));
	    if((ufd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
		/* Can't make socket; hang ourselves */
		syslog(LOG_INFO, "Unable to create UNIX socket: %s\n",
			strerror(errno));
		sigHup(SIGHUP);
	    }

	    soun.sun_family = AF_UNIX;
	    sprintf(soun.sun_path, "%s", PatExpand(DiabloSocketPat));
	    if (connect(ufd, (struct sockaddr *)&soun, offsetof(struct sockaddr_un, sun_path[strlen(soun.sun_path)+1])) < 0) {
		/* Can't connect; hang ourselves */
		syslog(LOG_INFO, "Unable to connect to master %s: %s\n",
			soun.sun_path, strerror(errno));
		sigHup(SIGHUP);
	    }

	    fo = fdopen(ufd, "w");

	    fprintf(fo, "child-is-readonly\n");
	    fprintf(fo, "quit\n");
	    fflush(fo);

	    fclose(fo);

	    close(ufd);
	}
    } else {
	sigHup(SIGHUP);
    }
}

/*
 * DIABLOSERVER() - The master server process.  It accept()s connections
 *		    and forks off children.
 */

void
DiabloServer(void)
{
    int lfd;
    int ufd;

    /*
     * Detach
     */

    if (DebugOpt == 0) {
	pid_t pid = fork();

	if (pid < 0) {
	    perror("fork");
	    exit(1);
	}
	if (pid > 0) {
	    exit(0);
	}

	/*
	 * Child continues
	 */

	DDUseSyslog = 1;

	freopen("/dev/null", "w", stdout);
	freopen("/dev/null", "w", stderr);
	freopen("/dev/null", "r", stdin);
#if USE_TIOCNOTTY
	{
	    int fd = open("/dev/tty", O_RDWR);
	    if (fd >= 0) {
		ioctl(fd, TIOCNOTTY, 0);
		close(fd);
	    }
	}
#endif
#if USE_SYSV_SETPGRP
	setpgrp();
#else
	setpgrp(0, 0);
#endif
    }

    /*
     * select()/signal() setup
     */

    FD_ZERO(&RFds);

    rsignal(SIGHUP, sigHup);
    rsignal(SIGINT, sigHup);
    rsignal(SIGUSR1, sigUsr1);
    rsignal(SIGUSR2, sigUsr2);
    rsignal(SIGALRM, sigAlrm);

    /*
     * Call InitPreCommit() and InitSpamFilter() to setup any shared memory
     * segments.  These need to be created and mapped prior to any forks that
     * we do.
     */

    InitPreCommit();
    InitSpamFilter();
    InitSeqSpace();

    /*
     * logs, socket setup
     */

    {
	struct sockaddr_in sin;

	memset(&sin, 0, sizeof(sin));

	openlog("diablo", (DebugOpt ? LOG_PERROR : 0)|LOG_PID|LOG_NDELAY, LOG_NEWS);

	/*
	 * listen socket for news
	 */

	if ((lfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	    perror("socket");
	    exit(1);
	}
	sin.sin_addr = NewsBindHost;
	sin.sin_port = htons(NewsPort);
	sin.sin_family = AF_INET;
	{
	    int on = 1;
	    setsockopt(lfd, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on));
	    setsockopt(lfd, SOL_SOCKET, SO_KEEPALIVE, (void *)&on, sizeof(on));
	}
	if (TxBufSize) {
	    setsockopt(lfd, SOL_SOCKET, SO_SNDBUF, (void *)&TxBufSize, sizeof(int));
	}
	if (RxBufSize) {
	    setsockopt(lfd, SOL_SOCKET, SO_RCVBUF, (void *)&RxBufSize, sizeof(int));
	}
	if (bind(lfd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	    perror("bind");
	    exit(1);
	}

#if NONBLK_ACCEPT_BROKEN
	/* HPUX is broken, see lib/config.h */
#else
	fcntl(lfd, F_SETFL, O_NONBLOCK);
#endif

	if (listen(lfd, 10) < 0) {
	    perror("listen");
	    exit(1);
	}

	FD_SET(lfd, &RFds);
	if (MaxFds <= lfd)
	    MaxFds = lfd + 1;
    }

    /*
     * Upward compatibility hack - older versions of diablo create
     * the unix domain socket as root.  We have to make sure the
     * file path is cleared out so we can create the socket as user news.
     */
    remove(PatExpand(DiabloSocketPat));

    /*
     * change my uid/gid
     */

    {
	struct passwd *pw = getpwnam("news");
	struct group *gr = getgrnam("news");
	gid_t gid;

	if (pw == NULL) {
	    perror("getpwnam('news')");
	    exit(1);
	}
	if (gr == NULL) {
	    perror("getgrnam('news')");
	    exit(1);
	}
	gid = gr->gr_gid;
	setgroups(1, &gid);
	setgid(gr->gr_gid);
	setuid(pw->pw_uid);
    }

    /*
     * UNIX domain socket
     */

    {
	struct sockaddr_un soun;

	memset(&soun, 0, sizeof(soun));

	if ((ufd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
	    perror("udom-socket");
	    exit(1);
	}
	soun.sun_family = AF_UNIX;

	sprintf(soun.sun_path, "%s", PatExpand(DiabloSocketPat));
	remove(soun.sun_path);
	if (bind(ufd, (struct sockaddr *)&soun, offsetof(struct sockaddr_un, sun_path[strlen(soun.sun_path)+1])) < 0) {
	    perror("udom-bind");
	    exit(1);
	}
	chmod(soun.sun_path, 0770);

#if NONBLK_ACCEPT_BROKEN
	/* HPUX is broken, see lib/config.h */
#else
	fcntl(ufd, F_SETFL, O_NONBLOCK);
#endif

	if (listen(ufd, 200) < 0) {
	    perror("udom-listen");
	    exit(1);
	}

	FD_SET(ufd, &RFds);
	if (MaxFds <= ufd)
	    MaxFds = ufd + 1;
    }

    /*
     * Initial load of dnewsfeeds file.  We recheck every select
     * (but it doesn't call stat() every time).  The master server
     * requires the entire file to be parsed, so we do not supply
     * a label.
     */

    LoadNewsFeed(0, 1, NULL);
    LoadSlotDir(time(NULL), 1);
    HostCachePid = LoadHostAccess(time(NULL), 1, HostCacheRebuildTime);

    /*
     * Main Loop
     */

    while (NumForks || Exiting == 0) {
	fd_set rfds = RFds;
	int i;
	int n;
	struct timeval tv = { 5, 0 };
	time_t t;

	if ((ReadOnlyMode || PausedCount) && (NumForks == ReadOnlyCount)) {
	    FinishRetain(RET_PAUSE);
	}

	n = select(MaxFds, &rfds, NULL, NULL, &tv);

	t = time(NULL);

	LoadNewsFeed(t, 0, NULL);
	LoadSlotDir(t, 0);
	if (HostCachePid == 0)
	    HostCachePid = LoadHostAccess(time(NULL), 0, HostCacheRebuildTime);

	for (i = 0; i < MaxFds; ++i) {
	    if (FD_ISSET(i, &rfds)) {
		--n;
		if (i == lfd)
		    DoAccept(i);
		else if (i == ufd)
		    DoCommand(i);
		else
		    DoPipe(i);
	    }
	}
	{
	    pid_t pid;

	    while ((pid = wait3(NULL, WNOHANG, NULL)) > 0) {
		int i;

		if (pid == HostCachePid) {
		    HostCachePid = 0;
		    continue;
		}

		for (i = 0; i < MaxFds; ++i) {
		    if (PidAry[i].tr_Pid == pid) {
			bzero(&PidAry[i], sizeof(PidAry[0]));
		    }
		}
	    }
	}
	if (NumForks >= MAXFORKS || Exiting) {
	    if (NumForksExceeded == 0) {
		if (Exiting) {
		    syslog(LOG_WARNING, "Exiting, waiting on children");
		} else {
		    syslog(LOG_WARNING, "NumForks exceeded");
		}
		NumForksExceeded = 1;
	    }
	    if (lfd >= 0)
		FD_CLR(lfd, &RFds);
	    if (Exiting) {
		close(lfd);
		lfd = -1;
	    }
	} else {
	    if (NumForksExceeded) {
		if (NumForksExceeded == 1)
		    syslog(LOG_WARNING, "NumForks ok, reaccepting");
		NumForksExceeded = 0;
	    }
	    FD_SET(lfd, &RFds);
	}
    }
    LogSession2();
    FinishRetain(RET_PAUSE);
    FinishRetain(RET_CLOSE);
    flushFeeds(0);
    TermSpamFilter();
    sleep(2);
}

/*
 * DOACCEPT()	- accept and deal with a new connection
 */

void
DoAccept(int lfd)
{
    int fd;
    struct sockaddr_in asin;
    ACCEPT_ARG3_TYPE alen = sizeof(asin);

    if (Exiting)
	return;

    if ((fd = accept(lfd, (struct sockaddr *)&asin, &alen)) >= 0) {
	int fds[2] = { -1, -1 };
	int ok = 0;
	int count = 0;
	struct in_addr raddr;

	fcntl(fd, F_SETFL, 0);
	bzero(&raddr, sizeof(raddr));

	/*
	 * Handle connection limit.  Note that we assume the TCP write buffer
	 * is large enough to hold our response message so the write() does
	 * not block.
	 */

	{
	    struct sockaddr_in rsin;
	    ACCEPT_ARG3_TYPE rslen = sizeof(rsin);

	    if (getpeername(fd, (struct sockaddr *)&rsin, &rslen) == 0) {
		raddr = rsin.sin_addr;
		if (MaxPerRemote) {
		    int i;

		    for (i = 0; i < MaxFds; ++i) {
			if (PidAry[i].tr_Addr.s_addr == raddr.s_addr)
			    ++count;
		    }
		    if (MaxPerRemote > 0 && count >= MaxPerRemote) {
			char buf[80];

			syslog(
			    LOG_WARNING, 
			    "Connect Limit exceeded (from -M/diablo.config) for %s (%d)",
			    inet_ntoa(raddr),
			    count
			);
			sprintf(
			    buf, 
			    "502 %s DIABLO parallel connection limit is %d\r\n",
			    HostName,
			    MaxPerRemote
			);
			write(fd, buf, strlen(buf));
			ok = -1;
		    }
		}
		sprintf(PeerIpName, "%s", inet_ntoa(raddr));
		bhash(&PeerIpHash, PeerIpName, strlen(PeerIpName));
	    } else if (MaxPerRemote) {
		ok = -1;
	    }
	}

	if (ok == 0 && pipe(fds) == 0 && fds[0] < MAXFDS) {
	    pid_t pid;

	    /*
	     * we MUST clear our FILE stdio so a call to
	     * exit() doesn't flush FILE structures to
	     * the wrong descriptors!
	     *
	     * WARNING WARNING!  There cannot be a single
	     * critical FILE handle open for which we may
	     * close it's underlying descriptor!
	     */

	    fflush(stdout);
	    fflush(stderr);

	    if ((pid = fork()) == 0) {
		int i;

		flushFeeds(1);	/* close feed descriptors without flushing */

		SessionMark = SessionBeg = time(NULL);

		if (HasStatusLine)
		    stprintf("%s", inet_ntoa(asin.sin_addr));

		closelog();

		for (i = 0; i < MaxFds; ++i) {
		    if (i > 2 && i != fds[1] && i != fd && i != ZoneFd && i != FilterFd) {
			if (PipeAry[fd] != NULL) {
			    bclose(PipeAry[fd], 0);
			    PipeAry[fd] = NULL;
			}
			close(i);
		    }
		}

		openlog("diablo", (DebugOpt ? LOG_PERROR : 0)|LOG_PID|LOG_NDELAY, LOG_NEWS);

		FeedFo = fdopen(fds[1], "w");
		if ((HName = Authenticate(fd, &asin, HLabel)) == NULL) {
		    FILE *fo = fdopen(fd, "w");

		    if (fo == NULL) {
			syslog(LOG_CRIT, "fdopen() of socket failed");
			exit(1);
		    }
		    xfprintf(fo, "502 Holy DNS batman, you aren't on my list! (Diablo)\r\n");
		    syslog(LOG_INFO, "Connection %d from %s (no permission)",
			fds[0],
			inet_ntoa(asin.sin_addr)
		    );
		    exit(0);
		}
		if (HLabel[0] == 0) {
		    FILE *fo = fdopen(fd, "w");

		    if (fo == NULL) {
			syslog(LOG_CRIT, "fdopen() of socket failed");
			exit(1);
		    }
		    xfprintf(fo, "502 %s DIABLO Misconfiguration, label missing in diablo.hosts, contact news admin\r\n", HostName);
		    syslog(LOG_CRIT, "diablo.hosts entry for %s missing label",
			HName
		    );
		    exit(0);
		}

		DidFork = 1;

		if (HasStatusLine)
		    stprintf("%s", HName);

		syslog(LOG_INFO, "Connection %d from %s %s",
		    fds[0],
		    HName,
		    inet_ntoa(asin.sin_addr)
		);

		/*
		 * Free the parent process memory pool, which the child does not
		 * use (obviously!)
		 */

		freePool(&ParProcMemPool);

		/*
		 * Simple session debugging support
		 */
		if (DebugLabel && HLabel[0] && strcmp(DebugLabel, HLabel) == 0) {
		    char path[256];
		    int tfd;

		    sprintf(path, "/tmp/diablo.debug.%d", (int)getpid());
		    DebugOpt = 1;
		    remove(path);
		    if ((tfd = open(path, O_EXCL|O_CREAT|O_TRUNC|O_RDWR, 0600)) >= 0) {
			close(tfd);
			freopen(path, "a", stderr);
			freopen(path, "a", stdout);
			printf("Debug label %s pid %d\n", HLabel, (int)getpid());
		    } else {
			printf("Unable to create %s\n", path);
		    }
		}

		DoSession(fd, count);
		LogSession();
		syslog(LOG_INFO, "Disconnect %d from %s %s (%d elapsed)",
		    fds[0],
		    HName,
		    inet_ntoa(asin.sin_addr),
		    (int)(time(NULL) - SessionBeg)
		);
		exit(0);
	    }
	    if (pid < 0) {
		syslog(LOG_EMERG, "fork failed: %s", strerror(errno));
	    } else {
		ok = 1;
		++NumForks;
		PidAry[fds[0]].tr_Pid = pid;
		PidAry[fds[0]].tr_Addr = raddr;
	    }
	}
	close(fd);
	if (fds[1] >= 0)
	    close(fds[1]);

	if (ok > 0) {
	    fcntl(fds[0], F_SETFL, O_NONBLOCK);
	    FD_SET(fds[0], &RFds);
	    if (MaxFds <= fds[0])
		MaxFds = fds[0] + 1;
	} else {
	    if (fds[0] >= 0) {
		syslog(LOG_WARNING, "Maximum file descriptors exceeded");
		close(fds[0]);
	    } else if (ok == 0 && fds[0] < 0) {
		syslog(LOG_EMERG, "pipe() failed: %s", strerror(errno));
	    }
	}
    }
}

/*
 * DOPIPE()	- handle data returned from our children over a pipe
 */

int 
fwCallBack(const char *hlabel, const char *msgid, const char *path, const char *offsize, int plfo, int headOnly)
{
    writeFeedOut(hlabel, path, msgid, offsize, ((plfo > 0) ? 1 : 0), headOnly);
    TtlArtsFed += 1.0;
    return(0);
}

void
DoPipe(int fd)
{
    char *ptr;
    int maxCount = 100;
    int bytes;

    if (PipeAry[fd] == NULL)
	PipeAry[fd] = bopen(fd);

    while ((ptr = egets(PipeAry[fd], &bytes)) != NULL && ptr != (char *)-1) {
	char *s1;

	ptr[bytes - 1] = 0;	/* replace newline with NUL */

	if (DebugOpt > 2)
	    ddprintf("%d << %*.*s", (int)getpid(), bytes, bytes, ptr);

	s1 = strtok(ptr, "\t\n");

	if (s1 == NULL)
	    continue;

	if (strncmp(s1, "SOUT", 4) == 0) {
	    char *path = strtok(NULL, "\t\n");
	    char *offsize = strtok(NULL, "\t\n");
	    const char *msgid = MsgId(strtok(NULL, "\t\n"));
	    char *nglist = strtok(NULL, "\t\n");
	    char *dist = strtok(NULL, "\t\n");
	    char *npath = strtok(NULL, "\t\n");
	    char *headOnly = strtok(NULL, "\t\n");

	    if (DebugOpt > 2) {
		ddprintf(
		    "%d SOUTLINE: %s %s %s %s %s HO=%s\n",
		    (int)getpid(), 
		    path, 
		    offsize,
		    msgid,
		    nglist,
		    npath, 
		    headOnly
		);
	    }

	    if (path && offsize && msgid && nglist && npath && headOnly) {
		int spamArt = 0;

#if DIABLO_FILTER
		if (diablofilterenabled) 
		{
		    char loc[128];
		    time_t startTime;
		    time_t endTime;

		    snprintf(loc, sizeof(loc), "%s:%s", path, offsize);
		    startTime = time(NULL);
		    spamArt = diabfilter(loc);
		    endTime = time(NULL);
		}
#endif

		FeedWrite(fwCallBack, msgid, path, offsize, nglist, npath, dist, headOnly, spamArt);
		{
		    char *p;
		    if ((p = strchr(offsize, ',')) != NULL)
			bytes = strtol(p + 1, NULL, 0);
		    TtlArtsBytes += (double)bytes;
		}
		TtlArtsReceived += 1.0;
		if (++LogCount == 1024) {
		    LogCount = 0;
		    LogSession2();
		}
	    }
	} else if (strncmp(s1, "FLUSH", 5) == 0) {
	    flushFeeds(0);
	}

	if (--maxCount == 0)
	    break;
    }

    bextfree(PipeAry[fd]);	/* don't keep large buffers around */

    if (ptr == (void *)-1) {
	if (PipeAry[fd] != NULL) {
	    bclose(PipeAry[fd], 0);
	    PipeAry[fd] = NULL;
	}
	close(fd);
	FD_CLR(fd, &RFds);
	--NumForks;
    }
}

/*
 * WRITEFEEDOUT()	- the master parent writes to the outgoing feed files
 * FLUSHFEEDOUT()
 */

void
writeFeedOut(const char *label, const char *file, const char *msgid, const char *offSize, int rt, int headOnly)
{
    Feed *fe;

    /*
     * locate feed
     */

    for (fe = FeBase; fe; fe = fe->fe_Next) {
	if (strcmp(label, fe->fe_Label) == 0)
	    break;
    }

    /*
     * allocate feed if not found
     */

    if (fe == NULL) {
	fe = zalloc(&ParProcMemPool, sizeof(Feed) + strlen(label) + 1);
	fe->fe_Label = (char *)(fe + 1);
	strcpy(fe->fe_Label, label);
	fe->fe_Fd = xopen(O_APPEND|O_RDWR|O_CREAT, 0644, "%s/%s", PatExpand(DQueueHomePat), label);
	if (fe->fe_Fd >= 0) {
	    int bsize;

	    fe->fe_Buf = pagealloc(&bsize, 1);
	    fe->fe_BufMax = bsize;
	    fe->fe_BufIdx = 0;
	    fe->fe_Failed = 0;
	}
	fe->fe_Next = FeBase;
	FeBase = fe;
    }

    /*
     * write to buffered feed, flushing buffers if there is not enough
     * room.  If the line is too long, something has gone wrong and we
     * throw it away.
     *
     * note that we cannot fill the buffer to 100%, because the trailing
     * nul (which we do not write) will overrun it.  I temporarily add 4 to
     * l instead of 3 to include the trailing nul in the calculations, but
     * subtract it off after the actual copy operation.
     */

    if (fe->fe_Fd >= 0) {
	int l = strlen(file) + strlen(msgid) + strlen(offSize) + (3 + 32 + 1);

	/*
	 * line would be too long?
	 */
	if (l < fe->fe_BufMax) {
	    /*
	     * line fits in buffer with trailing nul ?
	     */
	    if (l >= fe->fe_BufMax - fe->fe_BufIdx)
		flushFeedOut(fe);

	    sprintf(fe->fe_Buf + fe->fe_BufIdx, "%s %s %s%s\n",
		file, msgid, offSize, 
		(headOnly ? " H" : "")
	    );
	    fe->fe_BufIdx += strlen(fe->fe_Buf + fe->fe_BufIdx);
	    if (rt)
		flushFeedOut(fe);
	}
    }
}

void
flushFeeds(int justClose)
{
    Feed *fe;

    while ((fe = FeBase) != NULL) {
	if (justClose == 0)
	    flushFeedOut(fe);
	if (fe->fe_Buf)
	    pagefree(fe->fe_Buf, 1);
	if (fe->fe_Fd >= 0)
	    close(fe->fe_Fd);
	fe->fe_Fd = -1;
	fe->fe_Buf = NULL;
	FeBase = fe->fe_Next;
	zfree(&ParProcMemPool, fe, sizeof(Feed) + strlen(fe->fe_Label) + 1);
    }
}

void
flushFeedOut(Feed *fe)
{
    if (fe->fe_BufIdx && fe->fe_Buf && fe->fe_Fd >= 0) {
	/*
	 * flush buffer.  If the write fails, we undo it to ensure
	 * that we do not get garbaged feed files.
	 */
	int n = write(fe->fe_Fd, fe->fe_Buf, fe->fe_BufIdx);
	if (n >= 0 && n != fe->fe_BufIdx) {
	    ftruncate(fe->fe_Fd, lseek(fe->fe_Fd, 0L, 1) - n);
	}
	if (n != fe->fe_BufIdx && fe->fe_Failed == 0) {
	    fe->fe_Failed = 1;
	    syslog(LOG_INFO, "failure writing to feed %s", fe->fe_Label);
	}
    }
    fe->fe_BufIdx = 0;
}

/*
 * DOSESSION()	- a child process to handle a diablo connection
 */

void
DoSession(int fd, int count)
{
    Buffer *bi;
    FILE *fo;
    char *buf;
    int streamMode = 0;
    int headerOnly = 0;
    int syntax = 0;
    int unimp = 0;

    /*
     * reinitialize random generator
     */

    srandom((int32)random() ^ (getpid() * 100) ^ (int32)time(NULL));

    /*
     * Fixup pipe
     */

    {
	int nfd = dup(fd);
	if (nfd < 0) {
	    syslog(LOG_CRIT, "DoSession: dup()");
	    exit(1);
	}
	bi = bopen(nfd);
	fo = fdopen(fd, "w");

	if (bi == NULL || fo == NULL) {
	    syslog(LOG_CRIT, "DoSession() bopen failure");
	    exit(1);
	}
    }
    if (PausedCount) {
	xfprintf(fo, "502 %s DIABLO is currently paused\r\n", HostName);
	fflush(fo);
	exit(0);
    }
    if (ReadOnlyMode) {
	xfprintf(fo, "502 %s DIABLO is currently in read-only mode\r\n",
		HostName);
	fflush(fo);
	exit(0);
    }

    HistoryOpen(NULL, 0);
    LoadNewsFeed(0, 1, HLabel);	/* free old memory and load only our label */

    switch(FeedValid(HLabel, &count)) {
    case FEED_VALID:
	break;
    case FEED_MAXCONNECT:
	xfprintf(fo, "502 %s DIABLO parallel connection limit is %d\r\n",
	    HostName,
	    count
	);
	syslog(
	    LOG_WARNING, 
	    "Connect Limit exceeded (from dnewsfeeds) for %s", HName
	);
	fflush(fo);
	exit(0);
	/* not reached */
    case FEED_MISSINGLABEL:
	xfprintf(fo, "502 %s DIABLO misconfigugration, label missing in dnewsfeeds, contact news admin\r\n", HostName);
	syslog(LOG_CRIT, "Diablo misconfiguration, label %s not found in dnewsfeeds", HLabel);
	fflush(fo);
	exit(0);
	/* not reached */
    }

    LoadExpireCtl(1);		/* load entire expire.ctl		   */
    ArticleFileInit();		/* initialize article file cache	   */
    AllowReadOnly = FeedCanReadOnly(HLabel);  /* allow read-only mode?     */

    if (DiabloActiveEnabled && DiabloXRefSlave == 0)
	InitDActive(ServerDActivePat); /* initialize dactive.kp if enabled   */

    /*
     * print header, start processing commands
     */

    xfprintf(fo, "200 %s News server DIABLO %s-%s ready\r\n", HostName, VERS, SUBREV);
    fflush(fo);

    while ((buf = bgets(bi, NULL)) != NULL && buf != (char *)-1) {
	char *cmd;

	if (DebugOpt > 2) {
	    ddprintf("%d << %s", (int)getpid(), buf);
	}

	cmd = strtok(buf, " \t\r\n");

	if (cmd == NULL)
	    continue;

	if (strcasecmp(cmd, "ihave") == 0) {
	    if(ReadOnlyMode) {
	       xfprintf(fo, "436 Spool in read-only mode; try again later\r\n");
	    } else {

	    const char *msgid = MsgId(strtok(NULL, "\r\n"));

	    ++ArtsIHave;
	    TtlArtsTested += 1.0;

	    /*
	     * The PreCommit cache also doubles as a recent history 'hit'
	     * cache, so check it first.
	     */

#if DO_PCOMMIT_POSTCACHE
	    if (strcmp(msgid, "<>") == 0 || PreCommit(msgid, 0) < 0) {
		xfprintf(fo, "435 %s\r\n", msgid);
		++ArtsPColl;
	    } else if (HistoryLookup(msgid, NULL, NULL, NULL, NULL) == 0) {
#if USE_PCOMMIT_RW_MAP
		(void)PreCommit(msgid, 1);
#endif
		xfprintf(fo, "435 %s\r\n", msgid);
#else
	    if (strcmp(msgid, "<>") == 0 || 
		HistoryLookup(msgid, NULL, NULL, NULL, NULL) == 0
	    ) {
		xfprintf(fo, "435 %s\r\n", msgid);
	    } else if (PreCommit(msgid, 0) < 0) {
		xfprintf(fo, "435 %s\r\n", msgid);
		++ArtsPColl;
#endif
	    } else {
		int r;
		char rejMsg[REJMSGSIZE];

		rejMsg[0] = 0;

		xfprintf(fo, "335 %s\r\n", msgid);
		fflush(fo);
		switch((r = LoadArticle(bi, msgid, 0, headerOnly, rejMsg))) {
		case RCOK:
		    xfprintf(fo, "235\r\n");	/* article posted ok	*/
		    break;
		case RCALREADY:
		    /*
		     * see RELEASE_NOTES V1.16-test8.  435 changed to 437.
		     */
		    xfprintf(fo, "437 Duplicate\r\n");	/* already have it */
#ifdef NOTDEF
		    xfprintf(fo, "435\r\n");	/* already have it	*/
#endif
		    break;
		case RCTRYAGAIN:
		    xfprintf(fo, "436 I/O error, try again later\r\n");
		    break;
		case RCREJECT:
		    xfprintf(fo, "437 Rejected%s\r\n", rejMsg);
		    break;
		case RCERROR:
		    ++ArtsErr;
		    /*
		     * protocol error during transfer (e.g. no terminating .)
		     */
		    xfprintf(fo, "436 Protocol error, missing terminator\r\n");
		    break;
		default:
		    ++ArtsErr;
		    /*
		     * An I/O error of some sort (e.g. disk full).
		     */
                    syslog(LOG_ERR, "%-20s %s code 400 File Error: %s",
                        HName,
                        msgid,
                        strerror(-r)
                    ); 
		    LogSession();
		    sleep(30);	/* reduce remote reconnection rate */
		    xfprintf(fo, "400 File Error: %s\r\n", strerror(-r));
		    fflush(fo);
		    ArticleFileCloseAll();
		    sleep(5);
		    exit(1);
		    break;	/* not reached */
		}
	    }
	    } /* ReadOnlyMode */
	} else if (strcasecmp(cmd, "check") == 0) {
	    if(ReadOnlyMode) {
	       xfprintf(fo, "436 Spool in read-only mode; try again later\r\n");
	    } else {

	    const char *msgid = MsgId(strtok(NULL, "\r\n"));

	    ++ArtsCheck;
	    TtlArtsTested += 1.0;

	    /*
	     * The PreCommit cache may also double as a recent history 'hit'
	     * cache, so check it first in that case. 
	     */

#if DO_PCOMMIT_POSTCACHE
	    if (strcmp(msgid, "<>") == 0 || PreCommit(msgid, 0) < 0) {
		xfprintf(fo, "438 %s\r\n", msgid);
		++ArtsPColl;
	    } else if (HistoryLookup(msgid, NULL, NULL, NULL, NULL) == 0) {
#if USE_PCOMMIT_RW_MAP
		(void)PreCommit(msgid, 1);
#endif
		xfprintf(fo, "438 %s\r\n", msgid);
#else
	    if (strcmp(msgid, "<>") == 0 ||
		HistoryLookup(msgid, NULL, NULL, NULL, NULL) == 0
	    ) {
		xfprintf(fo, "438 %s\r\n", msgid);
		(void)PreCommit(msgid, 1);
	    } else if (PreCommit(msgid, 0) < 0) {
		xfprintf(fo, "438 %s\r\n", msgid);
		++ArtsPColl;
#endif
	    } else {
		xfprintf(fo, "238 %s\r\n", msgid);
	    }
	    } /* ReadOnlyMode */
	} else if (strcasecmp(cmd, "takethis") == 0) {
	    if(ReadOnlyMode) {
	       xfprintf(fo, "436 Spool in read-only mode; try again later\r\n");
	    } else {

	    const char *msgid = MsgId(strtok(NULL, "\r\n"));
	    int r;
	    int alreadyResponded = 0;
	    char rejMsg[REJMSGSIZE];

	    rejMsg[0] = 0;

	    TtlArtsTested += 1.0;
	    if (HistoryLookup(msgid, NULL, NULL, NULL, NULL) == 0) {
		xfprintf(fo, "439 %s\r\n", msgid);
		fflush(fo);
		LoadArticle(bi, msgid, 1, headerOnly, NULL);
		r = RCALREADY;
		++ArtsStage2Rej;
		alreadyResponded = 1;
	    } else {
		r = LoadArticle(bi, msgid, 0, headerOnly, rejMsg);
	    }

	    if (alreadyResponded == 0) {
		switch(r) {
		case RCOK:
		    xfprintf(fo, "239 %s\r\n", msgid);	/* thank you */
		    break;
		case RCALREADY:
		    xfprintf(fo, "439 %s%s\r\n", msgid, rejMsg); /* already have it or do not requeue it	*/
		    break;
		case RCTRYAGAIN:
		    xfprintf(fo, "431 %s%s\r\n", msgid, rejMsg);
		    break;
		case RCREJECT:
		    xfprintf(fo, "439 %s%s\r\n", msgid, rejMsg);
		    break;
		case RCERROR:
		    xfprintf(fo, "431 %s%s\r\n", msgid, rejMsg);	/* article failed due to something, do not req */
		    ++ArtsErr;
		    break;
		default:
		    ++ArtsErr;
		    /*
		     * An I/O error of some sort (e.g. disk full).
		     */
		    syslog(LOG_ERR, "%-20s %s code 400 File Error: %s",
			HName,
			msgid,
			strerror(-r)
		    ); 
		    LogSession();
		    sleep(30);	/* reduce remote reconnection rate */
		    xfprintf(fo, "400 File Error: %s%s, %s\r\n", msgid, rejMsg, strerror(-r));
		    fflush(fo);
		    ArticleFileCloseAll();
		    sleep(5);
		    exit(1);
		    break;	/* not reached */
		}
	    }
	    } /* ReadOnlyMode */
	} else if (strcasecmp(cmd, "head") == 0 ||
		   strcasecmp(cmd, "body") == 0 ||
		   strcasecmp(cmd, "article") == 0
	) {
	    const char *msgid = MsgId(strtok(NULL, "\r\n"));
	    char *data = NULL;
	    int32 fsize = 0;
	    int pmart = 0;
	    int headOnly = 0;

	    if (strcmp(msgid, "<>") == 0) {
		xfprintf(fo, "443 Bad Message-ID\r\n");
	    } else if (HistoryLookup(msgid, &data, &fsize, &pmart, &headOnly) == 0) {
		enum ArtState { AS_ARTICLE, AS_BODY, AS_HEAD } as = AS_HEAD;

		switch(cmd[0]) {
		case 'b':
		case 'B':
		    as = AS_BODY;
		    break;
		case 'a':
		case 'A':
		    as = AS_ARTICLE;
		    break;
		default:	/* default, must be AS_HEAD */
		    break;
		}

		if (data && headOnly && as != AS_HEAD) {
		    xfprintf(fo, "430 Article not found\r\n");
		    if (DebugOpt > 2)
			ddprintf(">> (NO DATA: BODY/ARTICLE REQUEST FOR HEADER-ONLY STORE)");
		} else if (data) {
		    int b;
		    int i;
		    int inHeader = 1;
		    int doHead = 0;
		    int doBody = 0;

		    switch(as) {
		    case AS_BODY:
			xfprintf(fo, "222 0 body %s\r\n", msgid);
			doBody = 1;
			break;
		    case AS_ARTICLE:
			xfprintf(fo, "220 0 article %s\r\n", msgid);
			doHead = 1;
			doBody = 1;
			break;
		    default:
			doHead = 1;
			xfprintf(fo, "221 0 head %s\r\n", msgid);
			break;
		    }
		    if (doBody) {
			xadvise(data, fsize, XADV_WILLNEED);
			xadvise(data, fsize, XADV_SEQUENTIAL);
		    }

		    if (DebugOpt > 2)
			ddprintf(">> (DATA)");

		    for (i = b = 0; i < fsize; b = i) {
			int separator = 0;

			/*
			 * find the end of line
			 */
			while (i < fsize && data[i] != '\n')
			    ++i;

			/*
			 * if in the headers, check for a blank line
			 */

			if (inHeader && i - b == 0) {
			    inHeader = 0;
			    separator = 1;
			    if (doBody == 0)
				break;
			}

			/*
			 * if printing the headers and/or the body, do any
			 * appropriate escape, write the line non-inclusive
			 * of the \n, then write a CR+LF.
			 *
			 * the blank line separating the header and body
			 * is only printed for the 'article' command.
			 */

			if ((inHeader && doHead) || (!inHeader && doBody)) {
			    if (separator == 0 || (doHead && doBody)) {
				if (data[b] == '.')
				    fputc('.', fo);
				fwrite(data + b, i - b, 1, fo);
				fwrite("\r\n", 2, 1, fo);
			    }
			}

			++i;	/* skip the nl */

			/*
			 * if i > fsize, we hit the end of the file without
			 * a terminating LF.  We don't have to do anything
			 * since we've already terminated the last line.
			 */
		    }
		    xfprintf(fo, ".\r\n");
		} else {
		    xfprintf(fo, "430 Article not found\r\n");
		    if (DebugOpt > 2)
			ddprintf(">> (NO DATA: UNABLE TO FIND ARTICLE)");
		}
	    } else {
		xfprintf(fo, "430 Article expired\r\n");
	    }
	    if (data)
		xunmap(data, fsize + pmart);
	} else if (strcasecmp(cmd, "stat") == 0) {
	    const char *msgid = MsgId(strtok(NULL, "\r\n"));

	    if (strcmp(msgid, "<>") == 0) {
		xfprintf(fo, "443 Bad Message-ID\r\n");
	    } else if (HistoryLookup(msgid, NULL, NULL, NULL, NULL) == 0) {
		xfprintf(fo, "223 0 %s\r\n", msgid);
	    } else {
		xfprintf(fo, "430\r\n");
	    }
	} else if (strcasecmp(cmd, "feedrset") == 0) {
	    if (HLabel[0]) {
		FeedRSet(fo);
	    } else {
		xfprintf(fo, "490 Operation not allowed\r\n");
	    }
	} else if (strcasecmp(cmd, "feedcommit") == 0) {
	    if (HLabel[0]) {
		if (FeedTableReady)
		    FeedCommit(fo);
		else
		    xfprintf(fo, "491 No feedrset/add/del\r\n");
	    } else {
		xfprintf(fo, "490 Operation not allowed\r\n");
	    }
	} else if (strcasecmp(cmd, "feedadd") == 0) {
	    char *p = strtok(NULL, " \t\r\n");
	    if (p) {
		if (HLabel[0]) {
		    if (FeedTableReady)
			FeedAddDel(fo, p, 1);
		    else
			xfprintf(fo, "491 No feedrset\r\n");
		} else {
		    xfprintf(fo, "490 Operation not allowed\r\n");
		}
	    } else {
		xfprintf(fo, "491 Syntax Error\r\n");
	    }
	} else if (strcasecmp(cmd, "feeddel") == 0) {
	    char *p = strtok(NULL, " \t\r\n");
	    if (p) {
		if (HLabel[0]) {
		    if (FeedTableReady)
			FeedAddDel(fo, p, 0);
		    else
			xfprintf(fo, "491 No feedrset\r\n");
		} else {
		    xfprintf(fo, "490 Operation not allowed\r\n");
		}
	    } else {
		xfprintf(fo, "491 Syntax Error\r\n");
	    }
	} else if (strcasecmp(cmd, "mode") == 0) {
	    char *p = strtok(NULL, " \t\r\n");
	    if (p && strcasecmp(p, "stream") == 0) {
		streamMode = 1;
		xfprintf(fo, "203 StreamOK.\r\n");
	    } else if (p && strcasecmp(p, "headfeed") == 0) {
		headerOnly = 1;
		xfprintf(fo, "250 Mode Command OK.\r\n");
	    } else if (p && strcasecmp(p, "artfeed") == 0) {
		headerOnly = 0;
		xfprintf(fo, "250 Mode Command OK.\r\n");
	    } else if (p && strcasecmp(p, "reader") == 0) {
		unimp = 1;
	    } else {
		syntax = 1;
	    }
	} else if (strcasecmp(cmd, "outq") == 0) {
	    int qnum;
	    int qarts;
	    int qrun;

	    if (HLabel[0] && QueueRange(HLabel, &qnum, &qarts, &qrun) == 0) {
		xfprintf(fo, "290 qfile-backlog=%d arts=%d now-running=%d\r\n", 
		    qnum,
		    qarts,
		    qrun
		);
	    } else {
		xfprintf(fo, "491 No queue info available\r\n");
	    }
	} else if (strcasecmp(cmd, "authinfo") == 0) {
	    xfprintf(fo, "281 Authentication ok, no authentication required\r\n");
	} else if (strcasecmp(cmd, "quit") == 0) {
	    xfprintf(fo, "205 See ya later.\r\n");
	    fflush(fo);
	    break;
	} else if (strcasecmp(cmd, "help") == 0) {
	    xfprintf(fo, "100 Legal commands\r\n");
	    xfprintf(fo, "\tauthinfo\r\n"
			"\thelp\r\n"
			"\tihave\r\n"
			"\tcheck\r\n"
			"\ttakethis\r\n"
			"\tmode\r\n"
			"\tquit\r\n"
			"\thead\r\n"
			"\tstat\r\n"
			"\toutq\r\n"
			"\tfeedrset\r\n"
			"\tfeedadd grpwildcard\r\n"
			"\tfeeddel grpwildcard\r\n"
			"\tfeedcommit\r\n"
	    );
	    xfprintf(fo, ".\r\n");
	} else {
	    syntax = 1;
	    if (strcasecmp(cmd, "list") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "group") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "last") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "newgroups") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "newnews") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "next") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "post") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "slave") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "xhdr") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "xpath") == 0)
		unimp = 1;
	    if (strcasecmp(cmd, "xreplic") == 0)
		unimp = 1;
	    if (unimp)
		syntax = 0;
	}
	if (unimp) {
	    xfprintf(fo, "500 command not implemented\r\n");
	    unimp = 0;
	}
	if (syntax) {
	    xfprintf(fo, "500 Syntax error or bad command\r\n");
	    syntax = 0;
	}
	fflush(fo);
	if (HasStatusLine) {
	    stprintf("ihav=%-4d chk=%-4d rec=%-4d ent=%-4d %s",
		ArtsIHave,
		ArtsCheck,
		ArtsReceived,
		ArtsEntered,
		HName
	    );
	}
    }
    bclose(bi, 1);
    fclose(fo);
    ArticleFileCloseAll();
}

/*
 * LOADARTICLE()	- get an article from the remote
 */

int
LoadArticle(Buffer *bi, const char *msgid, int noWrite, int headerOnly, char *rejBuf)
{
    /*
     * create article file
     */
    int r = RCOK;
    int size = 0;
    int haveValidFile = 1;
    int didActiveDrop = 0;
    int didIOError = 0;
    int32 bpos = -1;
    time_t t = time(NULL);
    /*char path[256];*/
    History h = { 0 };

    h.hv = hhash(msgid);

    /*path[0] = 0;*/

    LoadExpireCtl(0);

    /*
     * Obtain the file used to handle this article.  The file may already
     * be cached... generally, we wind up with one new file every 10 minutes.
     * to the file at the same time, we will be ok.
     */

    errno = 0;	/* in case noWrite is true */

    h.boffset = 1;	/* force new queue file format in ArticleFileName */
    h.bsize = 1;

    r = RCERROR;

    {
	Buffer *b = NULL;
	char *p;
	int inHeader = 1;
	int lastNl = 0;
	int thisNl = 1;
	int skipHeader = 0;
	int haveMsgId = 0;
	int haveBytes = 0;
	int haveValidArticle = 1;
	int artFd = -1;
	int haveDate = 0;
	int haveSubject = 0;
	int bytes;
	int lncount = -1;
	char nglist[MAXLINE];
	char npath[MAXLINE];
	char dateBuf[64];
	char subject[80];
	char nntpPostingHost[64];
	char control[64];
	char dist[80];
	hash_t bh = { 0 };

	struct timespec delay;
	int delay_counter = 0;
	int delay_counter_max;
	delay.tv_sec = 0;

	FeedGetThrottle(HLabel, (int *)&(delay.tv_nsec), &delay_counter_max);

	nglist[0] = 0;
	npath[0] = 0;
	dateBuf[0] = 0;
	nntpPostingHost[0] = 0;
	subject[0] = 0;
	control[0] = 0;
	dist[0] = 0;

	if (noWrite == 0)
	    b = bopen(-1);

	while ((p = bgets(bi, &bytes)) != NULL && p != (char *)-1) {

	    if(delay_counter_max) {
		delay_counter++;
		if(delay_counter == delay_counter_max) {
		    delay_counter = 0;
		    if(delay.tv_nsec)
			nanosleep(&delay, NULL);
		}
	    }

	    size += bytes;

	    lastNl = thisNl;
	    thisNl = (p[bytes-1] == '\n');

	    if (haveValidArticle && RejectArtsWithNul) {
		int i;
		for (i = 0; i < bytes; ++i) {
		    if (p[i] == 0) {
			haveValidArticle = 0;
			break;
		    }
		}
	    }

	    /*
	     * defer lone CR's (only occurs if the buffer overflows)
	     * replace CRLF's with LF's
	     */

	    if (p[bytes-1] == '\r') {
		bunget(bi, 1);
		--bytes;
	    } else if (bytes > 1 && p[bytes-1] == '\n' && p[bytes-2] == '\r') {
		p[bytes-2] = '\n';
		--bytes;
	    }

	    /*
	     * Look for end of article
	     */
	    if (lastNl && bytes == 2 && strncmp(p, ".\n", 2) == 0) {
		if (r == RCERROR)
		    r = RCOK;
		break;
	    }

	    /*
	     * Look for dot escape.  bytes may be 0 after this.  NOTE! 
	     * THERE IS NO TERMINATOR
	     */
	    if (lastNl && *p == '.') {
		++p;
		--bytes;
	    }

	    /*
	     * Extract certain header information
	     */

	    if (inHeader) {
		/*
		 * if skipping a header-in-progress
		 */
		if (skipHeader) {
		    /*
		     * We didn't get the whole line from the previous header,
		     * so this line is part of the previous header.
		     */
		    if (lastNl == 0) 
			continue;

		    /*
		     * Header begins with a space or tab, it is a continuation
		     * from a previous header.
		     */
		    if (bytes && (p[0] == ' ' || p[0] == '\t'))
			continue;

		    /*
		     * header complete, skip terminated
		     */

		    skipHeader = 0;
		}

		/*
		 * If not skipping header and this is a continuation of a
		 * prior header due to a buffer overflow, write it out and 
		 * continue.  (this way we can support long paths independant
		 * of the buffer size).
		 */
		if (lastNl == 0) {
		    if (b && haveValidArticle)
			bwrite(b, p, bytes);
		    continue;
		}

		if (bytes == 1 && thisNl && p[0] == '\n') {
		    /*
		     * end of headers.  Setup artFd, h.exp, and fixup
		     * h.iter, h.gmt.  We have to add the dummy
		     * control.CTLNAME newsgroup before we GetExpire().
		     */
		    if (control[0])
			ngAddControl(nglist, sizeof(nglist), control);
		    h.exp = GetExpire(msgid, nglist, lncount, 0);

		    if (b && haveValidArticle) {
			/*
			 * handle Xref: header.  If 'activedrop yes' is set 
			 * and none of the groups can be found in the active 
			 * file, the article is dropped.
			 */
			if (DiabloActiveEnabled &&
			    DiabloXRefSlave == 0 && 
			    NewsPathName[0]
			) {
			    if (
				GenerateXRef(b, nglist, NewsPathName, control) == 0 &&
				DiabloActiveDrop
			    ) {
				haveValidFile = 0;
				didActiveDrop = 1;
				/* artFd = -1; */
			    }
			}

			/*
			 * If mode headfeed is used, the article MUST contain
			 * a Bytes: header.  We can set haveValidFile = 0
			 * now since we haven't tried to create the article
			 * file yet.
			 */

			if (headerOnly && haveBytes == 0) {
			    haveValidFile = 0;
			    /* artFd = -1; */
			}

			/* sets h.iter,h.gmt */
			if (haveValidFile) {
			    artFd = ArticleFile(&h, &bpos, nglist, msgid);
			    if (artFd >= 0) {
				bsetfd(b, artFd);
				bflush(b);
			    } else {
				haveValidFile = 0;
				didIOError = 1;
			    }
			}
		    }
		    inHeader = 0;
		} else if (bytes >= 6 && strncasecmp(p, "Bytes:", 6) == 0) {
		    haveBytes = 1;
		    /* write thru */
		} else if (bytes >= 6 && strncasecmp(p, "Lines:", 6) == 0) {
		    char lbuf[32];

		    diablo_strlcpynl(lbuf, p + 6, bytes - 6, sizeof(lbuf));
		    lncount = strtol(lbuf, NULL, 10);
		    /* write thru */
		} else if (bytes >= 5 && strncasecmp(p, "Path:", 5) == 0) {
		    /*
		     * Path: line, prepend our path element and store in npath.
		     */
		    p += 5;
		    bytes -= 5;

		    while (bytes && (*p == ' ' || *p == '\t')) {
			++p;
			--bytes;
		    }
		    /*
		     * FeedAdd breaks if Path: header contains a tab.  Drop
		     * the article (the idiots who put the tab in there have
		     * to fix their news system).
		     */
		    {
			int i;

			for (i = 0; i < bytes; ++i) {
			    if (p[i] == '\t' && p[i+1] != '\n') {
				haveValidArticle = 0;
				syslog(LOG_ERR, "%-20s %s Path header contains tab: %*.*s",
				    HName,
				    msgid,
				    bytes, bytes, p
				);
			    }
			}
		    }

		    if (b && haveValidArticle) {
			/*
			 * check first path element against aliases
			 * in dnewsfeeds file.
			 */
			char ipfail[128];
			int idx = 0;
			char *npx = (NewsPathName[0]) ? "!" : "";
			static int ReportedNoMatch = 0;

			if (PathElmMatches(HLabel, p, bytes, &idx) < 0) {
			    sprintf(ipfail, "%s%s.MISMATCH!", npx, PeerIpName);
			    if (ReportedNoMatch == 0) {
				ReportedNoMatch = 1;
				syslog(LOG_ERR, "%-20s Path element fails to match aliases: %*.*s",
				    HName,
				    idx, idx, p
				);
			    }
			} else {
			    strcpy(ipfail, npx);
			}
			/*
			 * write out new Path: line.  NewsPathName can be
			 * an empty string (make news node act as a bridge).
			 *
			 * write out CommonPathName if it was specified and
			 * does not already exist in the Path:
			 */
			bwrite(b, "Path: ", 6);
			bwrite(b, NewsPathName, strlen(NewsPathName));
			bwrite(b, ipfail, strlen(ipfail));
			if (CommonPathName && 
			    CommonElmMatches(CommonPathName, p, bytes) < 0
			) {
			    bwrite(b, CommonPathName, strlen(CommonPathName));
			    bwrite(b, "!", 1);
			}
		    }
		    diablo_strlcpynl(npath, p, bytes, sizeof(npath));
		} else if (bytes >= 5 && strncasecmp(p, "Xref:", 5) == 0) {
		    if (DiabloXRefSlave == 0) {
			skipHeader = 1;
			continue;
		    }
		    /* write thru */
		} else if (bytes >= 13 && strncasecmp(p, "Distribution:", 13) == 0) {
		    diablo_strlcpynl(dist, p + 13, bytes - 13, sizeof(dist));
		    /* write thru */
		} else if (bytes >= 5 && strncasecmp(p, "Bytes:", 6) == 0) {
		    /*
		     * strip Bytes: header, we add our own in dnewslink
		     */
		    skipHeader = 1;
		    continue;
		} else if (bytes >= 11 && strncasecmp(p, "Newsgroups:", 11) == 0) {
		    diablo_strlcpynl(nglist, p + 11, bytes - 11, MAXLINE);
		    /*
		     * FeedAdd breaks if newsgroups: header contains a
		     * tab, but we allow trailing tabs because many systems 
		     * are screwed and add one.  strlcpynl will deal with it.
		     */
		    {
			int i;

			for (i = 11; i < bytes; ++i) {
			    if (p[i] == '\t' && p[i+1] != '\n') {
				haveValidArticle = 0;
				syslog(LOG_ERR, "%-20s %s Newsgroups header contains tab: %*.*s",
				    HName,
				    msgid,
				    bytes, bytes, p
				);
			    }
			}
		    }
		} else if (bytes >= 8 && strncasecmp(p, "Subject:", 8) == 0) {
		    haveSubject = 1;
		    /*
		     * copy & remove newline
		     */
		    diablo_strlcpynl(subject, p + 8, bytes - 8, sizeof(subject));
		    /* 
		     * if the body is the same but the subject is different
		     * (aka: control messages), include them in the body
		     * hash so the SpamFilter does not filter them.
		     */
		    bhash(&bh, p, bytes);
		} else if (bytes >= 5 && strncasecmp(p, "Date:", 5) == 0) {
		    haveDate = 1;

		    diablo_strlcpynl(dateBuf, p + 5, bytes - 5, sizeof(dateBuf));
		} else if (SpamFilterOpt &&
			   bytes >= 18 && 
			   strncasecmp(p, "NNTP-Posting-Host:", 18) == 0
		) {
		    diablo_strlcpynl(nntpPostingHost, p + 18, bytes - 18, sizeof(nntpPostingHost));
		} else if (bytes >= 11 && strncasecmp(p, "Message-ID:", 11) == 0) {
		    char *ps;
		    char *pe;

		    for (ps = p + 11; ps - p < bytes && *ps != '<'; ++ps)
			;
		    for (pe = ps; pe - p < bytes && *pe != '>'; ++pe)
			;

		    if (pe - p < bytes) {
			int l = pe - ps + 1;
			if (strlen(msgid) == l && strncmp(msgid, ps, l) == 0)
			    haveMsgId = 1;
		    }
		    if (haveMsgId == 0) {
			int l = bytes - (ps - p);

			syslog(LOG_ERR, "%-20s message-id mismatch, command: %s, article: %*.*s",
			    HName,
			    msgid,
			    l, l, ps
			);
		    }
		} else if (bytes > 8 && strncasecmp(p, "Control:", 8) == 0) {
		    int i = 8;
		    int j = 0;

		    while (i < bytes && (p[i] == ' ' || p[i] == '\t'))
			++i;
		    while (j < sizeof(control) - 1 && i < bytes && isalpha(p[i]))
			control[j++] = p[i++];
		    control[j] = 0;
		}
	    } /* inHeader */

	    /*
	     * If not in header hash the article body for the spam filter.
	     * This is relatively easy to defeat (the NNTP-Posting-Host:
	     * is less so), but will still catch a shitload of spam since
	     * the article body is usually 100% duplicated.
	     */

	    if (inHeader == 0 && SpamFilterOpt && b) {
		bhash(&bh, p, bytes);
	    }

	    if (b && haveValidFile && haveValidArticle) {
		bwrite(b, p, bytes);
	    }
	} /* while */

	/*
	 * We don't create the file descriptor until we have all our
	 * headers.  If the article had no message body, we have to do
	 * that here. 
	 */

	if (b && inHeader && haveValidArticle) {
	    if (control[0])
		ngAddControl(nglist, sizeof(nglist), control);

	    h.exp = GetExpire(msgid, nglist, lncount, 0);

	    /* sets h.iter, h.gmt */
	    artFd = ArticleFile(&h, &bpos, nglist, msgid);

	    if (artFd >= 0) {
		bsetfd(b, artFd);
	    } else {
		haveValidFile = 0;
		didIOError = 1;		/* only if didn't expire */
	    }
	}
	inHeader = 0;

	/*
	 * Cascade haveValidFile into haveValidArticle.
	 */
	if (haveValidFile == 0)
	    haveValidArticle = 0;

	if (b && haveValidArticle) {
	    char z = 0;

	    h.boffset = bpos;
	    h.bsize = btell(b) - bpos;
	    bwrite(b, &z, 1); 		/* terminator (sanity check) */
	    bflush(b);
	}

	if (DebugOpt > 1) {
	    ddprintf("%s: haveValidFile=%d b=%08lx artFd=%d boff=%d bsize=%d",
		msgid, haveValidFile, (long)b, artFd, (int)h.boffset, (int)h.bsize
	    );
	}

	/*
	 * check output file for error, only if r == RCOK.
	 * If there was a problem, change the return code
	 * to RCTRYAGAIN.
	 */

	if (b != NULL && berror(b)) {
	    if (r == RCOK) {
		sleep(1);		/* failsafe */
		r = -errno;
		if (r >= 0) {
		    REJECTART(REJ_FAILSAFE, msgid, "Failsafe");
		    r = RCTRYAGAIN;
		    if (rejBuf)
			snprintf(rejBuf, REJMSGSIZE, " Failsafe/missing-dir");
		}
	    }
	}

	/*
	 * If the message isn't already error'd out,
	 * reject it.
	 */

	if (nglist[0] == 0 || haveMsgId == 0 || haveValidArticle == 0 ||
	    (headerOnly && haveBytes == 0)
	) {
	    if (r >= 0 && r != RCERROR) {
		if (didActiveDrop) {
		    REJECTART(REJ_NOTINACTV, msgid, "NotInActive");
		    if (rejBuf)
			snprintf(rejBuf, REJMSGSIZE, " NotInActive");
		} else if (didIOError) {
		    REJECTART(REJ_IOERROR, msgid, "IOError");
		    if (rejBuf)
			snprintf(rejBuf, REJMSGSIZE, " IOError");
		} else {
		    REJECTART(REJ_MISSHDRS, msgid, "MissingHdrs");
		    if (rejBuf)
			snprintf(rejBuf, REJMSGSIZE, " MissingHdrs");
		}
		r = RCREJECT;
	    }
	}

	if (DebugOpt > 1) {
	    ddprintf("nglist: %s", nglist);
	}

	/*
	 * If the messsage is too old, reject it.  The standard
	 * is 16 days, but our parsedate() routine is only 
	 * an estimate, so we use 14. 
	 *
	 * XXX reject the article if parsedate doesn't have a clue ?
	 */
	if (r == RCOK && haveDate) {
	    time_t tart = parsedate(dateBuf);

	    if (tart != (time_t)-1) {
		int32 dt = t - tart;
		int32 drdsecs = DiabloRememberDays * 24 * 60 * 60;

		if (dt > drdsecs || dt < -drdsecs) {
		    REJECTART(REJ_TOOOLD, msgid, "TooOld");
		    r = RCREJECT;
		    if (rejBuf)
			snprintf(rejBuf, REJMSGSIZE, " TooOld");
		}
	    }
	}

	if (b) {
	    /*
	     * close our buffered I/O without closing the underlying
	     * descriptor.  We have already flushed it (bclose() does
	     * not flush!).
	     */
	    bclose(b, 0);
	    b = NULL;
	}

	/*
	 * If the article from this incoming feed is filtered out due to
	 * group firewall.  The default, if no filter directives match, is
	 * to not filter (hence > 0 rather then >= 0)
	 */

	if (r == RCOK && IsFiltered(HLabel, nglist) > 0) {
	    REJECTART(REJ_GRPFILTER, msgid, "FilteredGroup");
	    r = RCREJECT;
	    if (rejBuf)
		snprintf(rejBuf, REJMSGSIZE, " GroupFilter");
	}

	/*
	 * XXX put other inbound filters here.  Be careful in regards to
	 *     what gets added to the history file and what does not, the
	 *     message may be valid when received via some other path.
	 */


	/*
	 * If everything is ok, check the spam filter
	 */

	if (r == RCOK && SpamFilterOpt) {
	    int rv = 0;
	    int how = 0;
	    hash_t hv = hhash(msgid);

	    if (nntpPostingHost[0])
		rv = FeedQuerySpam(HLabel, nntpPostingHost);

	    if (rv == 0)
		rv = SpamFilter(hv, nntpPostingHost, &bh, &how);

	    if (rv < 0) {
		/*
		 * Reject the article as being spam, plus write it to the
		 * history file to prevent future dups from getting through.
		 *
		 * h.iter is set to 0xFFFF, a non-existent file, allowing us
		 * to keep the offset/size stored in the history without
		 * having to worry about corruption due to the fact that
		 * the file space is reclaimed.
		 */ 
		h.hv = hv;
		h.iter = (unsigned short)-1;
		h.exp = H_EXP((unsigned short)-1);

		if ((-rv & 31) == 1) {
		    syslog(LOG_INFO, "SpamFilter/%s copy #%d: %s %s (%s)", 
			((how == 0) ? "dnewsfeeds" : ((how == 1) ? "by-post-rate" : "by-dup-body")),
			((how == 0) ? -1 : -rv),
			msgid, nntpPostingHost, subject
		    );
		}
		(void)HistoryAdd(msgid, &h);
		(void)PreCommit(msgid, 1);
		if (DebugOpt > 1) {
		    printf("SpamFilter: %s\t%s\n", msgid, nntpPostingHost);
		} 
		r = RCREJECT;
		REJECTART(REJ_SPAMFILTER, msgid, "IsSpam");
		++ArtsSpam;
		if (rejBuf)
		    snprintf(rejBuf, REJMSGSIZE, " Spam");
	    }
	}

	/*
	 * If the article does not have a distribution, add one.
	 */

	if (dist[0] == 0)
	    strcpy(dist, "world");

	/*
	 * If everything is ok, commit the message
	 */

	if (r == RCOK) {
#if USE_SYSV_SIGNALS
	    sighold(SIGHUP);
	    sighold(SIGALRM);
#else
	    int smask = sigblock((1 << SIGHUP) | (1 << SIGALRM));
#endif
	    /*
	     * Get the expiration.  An expiration of zero
	     * is an indication that we should not take 
	     * the article (usually due to size), but
	     * we need to add it to the history file 
	     * so we can reject it later.
	     *
	     * Use the PreCommit cache as a faster history
	     * cache.
	     *
	     * We must call GetExpire again with a valid size.  While it 
	     * is too late to do slot-assignment for size-based expires ( we 
	     * have already written out the file ), it is not too late to 
	     * handle the maximum-size parameter.
	     */

	    if (GetExpire(msgid, nglist, lncount, size) == 0)
		h.exp = 0;

	    if (headerOnly)
		h.exp |= EXPF_HEADONLY;

	    if (r != RCOK)
		++ArtsStage3Rej;

	    if (H_EXP(h.exp) == 0) {
		r = RCREJECT;
		REJECTART(REJ_INSTANTEXP, msgid, "InstantExpired");
		if (rejBuf)
		    snprintf(rejBuf, REJMSGSIZE, " InstantExpire");
	    }

	    /*
	     * Do not save an article map if the article is bad, we will 
	     * ftruncate or overwrite this space in the file later on!
	     *
	     * (We can differentiate from a spam-filter expired article
	     * and a dexpire.ctl expired article by observing the history
	     * file entry.  A spam-filter expired article uses an iteration
	     * of 0xFFFF)
	     *
	     * Add to the history file in the case of:
	     *
	     *		valid article
	     *		instant expire (handled here)
	     *		spam filter (handled previously)
	     */

	    if (r != RCOK) {
		h.boffset = 0;
		h.bsize = 0;
	    }
	    r = HistoryAdd(msgid, &h);
	    (void)PreCommit(msgid, 1);

	    if (r == RCOK) {
		if (FeedAdd(msgid, t, &h, nglist, npath, dist, headerOnly) < 0) {
		    /*
		     * If we loose our pipe, exit immediately.
		     */
		    ArticleFileCloseAll();
		    exit(1);
		}
	    } else {
		REJECTART(REJ_IOERROR, msgid, "IOError");
		if (rejBuf)
		    snprintf(rejBuf, REJMSGSIZE, " IOError");
	    }
#if USE_SYSV_SIGNALS
	    sigrelse(SIGHUP);
	    sigrelse(SIGALRM);
#else
	    sigsetmask(smask);
#endif
	}
	++ArtsReceived;

	if (r == RCOK) {
	    ++ArtsEntered;
	    ArtsBytes += (int32)size;
	    REJECTART(REJ_ACCEPTED, msgid, "ArticleAccepted");
	    if (control[0])
		++RejectStats[REJ_CTLMSG];
	    if (H_EXP(h.exp) < 5)
		++RejectStats[REJ_EARLYEXP];
	} else {
	    ++ArtsRej;
	}

	if ((ArtsReceived % 1024) == 0)
	    LogSession();

	/*
	 * Record the current append position, possibly
	 * updating the filesize in the article file cache.
	 *
	 * If we created a file but the return code is
	 * not RCOK, truncate the file.
	 */
	if (artFd >= 0)
	    ArticleFileSetSize(artFd);

	if (r != RCOK && haveValidFile) {
	    if (artFd >= 0) {
		ArticleFileTrunc(artFd, bpos);
#ifdef SLOT_DEBUG
		syslog(LOG_INFO, "ftruncate artfd %s to %d", msgid, bpos);
#endif
	    }
	}
    }

    return(r);
}

void
ngAddControl(char *nglist, int ngSize, const char *ctl)
{
    /*
     * If a control message, append 'control.CTLNAME' to grouplist
     */
    int l = strlen(nglist);

    if (l && nglist[l-1] == '\n')
	--l;

    if (strlen(ctl) + l < ngSize - 11)
	sprintf(nglist + l, ",control.%s", ctl);
}


/*
 * LOGSESSION()	- Log statistics for a session
 */

void
LogSession(void)
{
    time_t t = time(NULL);
    int32 dt = t - SessionMark;
    int32 nuse;

    nuse = ArtsCheck + ArtsIHave;
    if (nuse < ArtsReceived)
	nuse = ArtsReceived;

    syslog(LOG_INFO, "%-20s secs=%-4d ihave=%-4d chk=%-4d rec=%-4d rej=%-4d predup=%-4d posdup=%-4d pcoll=%-4d spam=%-4d err=%-4d added=%-4d bytes=%-4d (%d/sec)",
	HName,
	dt,
	ArtsIHave,
	ArtsCheck,
	ArtsReceived,
	ArtsRej,
	ArtsStage2Rej,
	ArtsStage3Rej,
	ArtsPColl,
	ArtsSpam,
	ArtsErr,
	ArtsEntered,
	ArtsBytes,
	nuse / ((dt == 0) ? 1 : dt)
    );
    ArtsBytes = 0;
    ArtsIHave = 0;
    ArtsCheck = 0;
    ArtsReceived = 0;
    ArtsEntered = 0;
    ArtsRej = 0;
    ArtsStage2Rej = 0;
    ArtsStage3Rej = 0;
    ArtsPColl = 0;
    ArtsSpam = 0;
    ArtsErr = 0;
    SessionMark = t;

    syslog(LOG_INFO, "%-20s stats acc=%d ctl=%d failsafe=%d misshdrs=%d tooold=%d grpfilt=%d spamfilt=%d earlyexp=%d instantexp=%d notinactv=%d ioerr=%d",
	HName,
	RejectStats[REJ_ACCEPTED],
	RejectStats[REJ_CTLMSG],
	RejectStats[REJ_FAILSAFE],
	RejectStats[REJ_MISSHDRS],
	RejectStats[REJ_TOOOLD],
	RejectStats[REJ_GRPFILTER],
	RejectStats[REJ_SPAMFILTER],
	RejectStats[REJ_EARLYEXP],
	RejectStats[REJ_INSTANTEXP],
	RejectStats[REJ_NOTINACTV],
	RejectStats[REJ_IOERROR]
    );
    bzero(RejectStats, sizeof(RejectStats));
}

void
LogSession2(void)
{
    int dt = (int)(time(NULL) - SessionMark);

    syslog(LOG_INFO, 
	"DIABLO uptime=%d:%02d arts=%s tested=%s bytes=%s fed=%s",
	dt / 3600,
	dt / 60 % 60,
	ftos(TtlArtsReceived),
	ftos(TtlArtsTested),
	ftos(TtlArtsBytes),
	ftos(TtlArtsFed)
    );
}

/*
 * DOCOMMAND() - handles a control connection
 */

void
DoCommand(int ufd)
{
    int fd;
    int retain = 0;
    struct sockaddr_in asin;
    ACCEPT_ARG3_TYPE alen = sizeof(asin);

    /*
     * This can be a while() or an if(), but apparently some OS's fail to
     * properly handle O_NONBLOCK for accept() on unix domain sockets so...
     */

    if ((fd = accept(ufd, (struct sockaddr *)&asin, &alen)) > 0) {
	FILE *fi;
	FILE *fo;

	fcntl(fd, F_SETFL, 0);
	fo = fdopen(dup(fd), "w");
	fi = fdopen(fd, "r");

	if (fi && fo) {
	    char buf[MAXLINE];

	    while (fgets(buf, sizeof(buf), fi) != NULL) {
		char *s1;

		if (DebugOpt)
		    printf("%d << %s\n", (int)getpid(), buf);

		if ((s1 = strtok(buf, " \t\r\n")) == NULL)
		    continue;
		if (strcmp(s1, "status") == 0) {
		    fprintf(fo, "211 Paused=%d Exiting=%d Forks=%d NFds=%d\r\n",
			PausedCount, Exiting, NumForks, countFds(&RFds)
		    );
		} else if (strcmp(s1, "flush") == 0) {
		    fprintf(fo, "211 Flushing feeds\n");
		    fflush(fo);
		    flushFeeds(0);
		} else if (strcmp(s1, "pause") == 0) {
		    retain = RET_PAUSE;
		    ++PausedCount;
		    ReadOnlyMode = 0;
		    ReadOnlyCount = 0;
		    fprintf(fo, "200 Pause, count %d.\n", PausedCount);
		    if (PausedCount == 1) {
			int i;

			for (i = 0; i < MAXFDS; ++i) {
			    if (PidAry[i].tr_Pid) {
				kill(PidAry[i].tr_Pid, SIGHUP);
			    }
			}
		    }
		} else if (strcmp(s1, "readonly") == 0) {
		    retain = RET_PAUSE;
		    ReadOnlyMode = 1;
		    ReadOnlyCount = 0;
		    fprintf(fo, "200 Read-only mode %d.\n", ReadOnlyMode);
		    {
			int i;

			for (i = 0; i < MAXFDS; ++i) {
			    if (PidAry[i].tr_Pid) {
				kill(PidAry[i].tr_Pid, SIGALRM);
			    }
			}
		    }
		} else if (strcmp(s1, "child-is-readonly") == 0) {
		    ReadOnlyCount++;
		} else if (strcmp(s1, "go") == 0) {
		    int i;

		    if ((PausedCount == 1) && ReadOnlyMode) {
			for (i = 0; i < MAXFDS; ++i) {
			    if (PidAry[i].tr_Pid) {
				kill(PidAry[i].tr_Pid, SIGHUP);
			    }
			}
			ReadOnlyMode = 0;
		    }

		    if (PausedCount)
			--PausedCount;

		    if (ReadOnlyMode && !PausedCount)
			ReadOnlyMode = 0;

		    fprintf(fo, "200 Resume, count %d\n", PausedCount);
		} else if (strcmp(s1, "quit") == 0) {
		    fprintf(fo, "200 Goodbye\n");
		    fprintf(fo, ".\n");
		    break;
		} else if (strcmp(s1, "exit") == 0 || strcmp(s1, "aexit")==0) {
		    int i;

		    if (s1[0] == 'e')
			retain = RET_CLOSE;

		    fprintf(fo, "211 Exiting\n");
		    fflush(fo);

		    if (Exiting == 0) {
			Exiting = 1;
			for (i = 0; i < MAXFDS; ++i) {
			    if (PidAry[i].tr_Pid) {
				kill(PidAry[i].tr_Pid, SIGHUP);
			    }
			}
		    } else {
			fprintf(fo, "200 Exit is already in progress\n");
		    }
		    flushFeeds(0);
		} else if (strcmp(s1, "debug") == 0) {
		    char *s2 = strtok(NULL, " \t\r\n");

		    zfreeStr(&SysMemPool, &DebugLabel);

		    if (s2) {
			DebugLabel = zallocStr(&SysMemPool, s2);
			fprintf(fo, "200 Debugging '%s'\n", DebugLabel);
		    } else {
			DebugLabel = NULL;
			fprintf(fo, "200 Debugging turned off\n");
		    }
		} else {
		    fprintf(fo, "500 Unrecognized command: %s\n", s1);
		}
		if (retain == 0)
		    fprintf(fo, ".\n");
		fflush(fo);
		if (retain)
		    break;
	    }
	}
	if (retain) {
	    Retain *ret = zalloc(&ParProcMemPool, sizeof(Retain));

	    ret->re_Next = ReBase;
	    ret->re_Fi = fi;
	    ret->re_Fo = fo;
	    ret->re_What = retain;
	    ReBase = ret;
	} else {
	    if (fi)
		fclose(fi);
	    if (fo)
		fclose(fo);
	}
    }
}

/*
 * REMOTE FEED CODE
 */

FILE *Ft;
int  FtNumCommit;

void
FeedRSet(FILE *fo)
{
    Ft = xfopen("w", "%s/%s.new", PatExpand(FeedsHomePat), HLabel);
    if (Ft != NULL) {
	FeedTableReady = 1;
	FtNumCommit = 0;
	xfprintf(fo, "290 Label Reset\r\n");
    } else {
	xfprintf(fo, "490 file create failed\r\n");
    }
}

void
FeedCommit(FILE *fo)
{
    char path1[256+64];
    char path2[256+64];

    sprintf(path1, "%s/%s.new", PatExpand(FeedsHomePat), HLabel);
    sprintf(path2, "%s/%s", PatExpand(FeedsHomePat), HLabel);
    fflush(Ft);
    if (ferror(Ft)) {
	remove(path1);
	xfprintf(fo, "490 file write failed\r\n");
    } else {
	if (FtNumCommit == 0) {
	    xfprintf(fo, "290 empty feed list, reverting to initial\r\n");
	    remove(path1);
	    remove(path2);
	} else {
	    xfprintf(fo, "290 feed commit complete\r\n");
	    rename(path1, path2);
	    TouchNewsFeed();
	}
    }
    fclose(Ft);
    FeedTableReady = 0;
}

void
FeedAddDel(FILE *fo, char *gwild, int add)
{
    int i;

    if (strlen(gwild) > MAXGNAME - 8) {
	xfprintf(fo, "490 wildcard too long\r\n");
	return;
    }
    if (FtNumCommit == MAXFEEDTABLE) {
	xfprintf(fo, "490 too many entries, max is %d\r\n", MAXFEEDTABLE);
	return;
    }
    for (i = 0; gwild[i]; ++i) {
	if (isalnum(gwild[i]))
	    continue;
	if (gwild[i] == '*')
	    continue;
	if (gwild[i] == '?')
	    continue;
	if (gwild[i] == '.')
	    continue;
	if (gwild[i] == '-')
	    continue;
	xfprintf(fo, "490 illegal character: %c\r\n", gwild[i]);
	break;
    }
    if (gwild[i] == 0) {
	xfprintf(fo, "290 ok\r\n");
	fprintf(Ft, "%s\t%s\n", ((add) ? "addgroup" : "delgroup"), gwild);
	++FtNumCommit;
    }
}

void
FinishRetain(int what)
{
    Retain *ret;
    Retain **pret;

    for (pret = &ReBase; (ret = *pret) != NULL; ) {
	if (ret->re_What == what) {
	    if (ret->re_Fo)
		fprintf(ret->re_Fo, "200 Operation Complete\n.\n");
	    *pret = ret->re_Next;
	    if (ret->re_Fi)
		fclose(ret->re_Fi);
	    if (ret->re_Fo)
		fclose(ret->re_Fo);
	    zfree(&ParProcMemPool, ret, sizeof(Retain));
	} else {
	    pret = &ret->re_Next;
	}
    }
}

int
QueueRange(const char *label, int *pqnum, int *pqarts, int *pqrun)
{
    FILE *fi = xfopen("r", "%s/.%s.seq", PatExpand(DQueueHomePat), label);
    int r = -1;

    if (fi) {
	int qbeg;
	int qend;
	if (fscanf(fi, "%d %d", &qbeg, &qend) == 2) {
	    r = 0;
	    *pqrun = 0;
	    *pqarts = -1;
	    *pqnum = qend - qbeg;

	    if (*pqnum < 500000) {
		int i;

		*pqarts = 0;
		for (i = qbeg; i < qend; ++i) {
		    FILE *fj = xfopen("r", "%s/%s.S%05d", PatExpand(DQueueHomePat), label, i);
		    if (fj) {
			char buf[256];

			while (fgets(buf, sizeof(buf), fj) != NULL) {
			    ++*pqarts;
			}
			if (xflock(fileno(fj), XLOCK_EX|XLOCK_NB) < 0)
			    ++*pqrun;
			fclose(fj);
		    }
		}
	    }
	}
	fclose(fi);
    }
    return(r);
}

#define ONE_K	1024.0	
#define ONE_M	(1024.0*1024.0)
#define ONE_G	(1024.0*1024.0*1024.0)

const char *
ftos(double d)
{
    static char FBuf[8][32];
    static int FCnt;
    char *p = FBuf[FCnt];

    if (d < 1024.0) {
	sprintf(p, "%d", (int)d);
    } else if (d < ONE_M) {
	sprintf(p, "%d.%03dK", (int)(d / ONE_K), ((int)d % (int)ONE_K) * 1000 / (int)ONE_K);
    } else if (d < ONE_G) {
	sprintf(p, "%d.%03dM", (int)(d / ONE_M), ((int)(d / ONE_K) % (int)ONE_K) * 1000 / (int)ONE_K);
    } else {
	sprintf(p, "%d.%03dG", (int)(d / ONE_G), ((int)(d / ONE_M) % (int)ONE_K) * 1000 / (int)ONE_K);
    }

    FCnt = (FCnt + 1) & 7;
    return(p);
}

int
countFds(fd_set *rfds)
{
    int i;
    int count = 0;

    for (i = 0; i < MAXFDS; ++i) {
	if (FD_ISSET(i, rfds))
	    ++count;
    }
    return(count);
}

/*
 * ArticleFile() - calculate article filename, open, and cache descriptors.
 *		   assigns h->iter, uses h->exp and h->gmt.  Assign *pbpos
 *		   and lseek's the descriptor to the start position for the
 *		   next article write.  The file is exclusively locked.
 *
 *		   NOTE!  We never create 'older' directories.  This could
 *		   lead to an offset,size being stored for a message, the
 *		   spool file getting deleted, then the spool file getting
 *		   recreated and a new message written.  Then an attempt to 
 *		   feed the old message would result in corrupt data.
 *
 *		   There are a number of post-write reject cases, including
 *		   spam cache hits, file too large, in-transit history 
 *		   collisions, and so forth.  Rather then ftruncate() the
 *		   spool file, the feeder now simply lseek's back and 
 *		   overwrites the dead article.  Diablo will do a final
 *		   ftruncate() on the file when the descriptor is finally
 *		   closed.
 */

typedef struct AFCache {
    int		af_Fd;
    off_t	af_AppendOff;	/* cached append offset		*/
    off_t	af_FileSize;	/* calculated current file size	*/
    uint32	af_Slot;	/* 10 minute bounded    	*/
    uint16	af_Iter;
} AFCache;

AFCache AFAry[MAXDIABLOFDCACHE];
int AFNum;

void
ArticleFileInit(void)
{
    int i;

    for (i = 0; i < MAXDIABLOFDCACHE; ++i)
	AFAry[i].af_Fd = -1;
}

int
ArticleFile(History *h, int *pbpos, const char *nglist, const char *msgid)
{
    AFCache	*af = NULL;
    int		rfd = -1;

    /*
     * figure out the gmt
     */

    h->gmt = ExpireToSeq(H_EXP(h->exp));

    /*
     * Look for entry in cache.
     */

    {
	int i;

	for (i = 0; i < AFNum; ++i) {
	    AFCache *raf = &AFAry[i];

	    if (raf->af_Slot == h->gmt) {
		af = raf;
		break;
	    }
	}
    }

    /*
     * Add new entry to cache.  If we hit the wall, close the last entry
     * in the cache.  If we are in feeder mode, the cache is degenerate with
     * only one entry.  If in reader mode, the cache is reasonably-sized.
     */

    if (af == NULL) {
	int cnt;
	int maxdfd = (DiabloExpire == EXPIRE_READER) ? MAXDIABLOFDCACHE : 1;

	/*
	 * blow away LRU if cache is full
	 */

	if (AFNum == maxdfd) {
	    ArticleFileClose(maxdfd - 1);
	    --AFNum;
	}

	/*
	 * our new entry
	 */

	af = &AFAry[AFNum];

	bzero(af, sizeof(AFCache));

	af->af_Iter = (uint16)(PeerIpHash.h1 ^ (PeerIpHash.h1 >> 16));
	af->af_Fd = -1;
	af->af_Slot = h->gmt;

	for (cnt = 0; cnt < 100000; ++cnt) {
	    char path[256];

	    errno = 0;
	    af->af_Iter &= 0x7FFF;

	    sprintf(path, "%s/D.%08x/B.%04x", 
		MySpoolHome, 
		(int)h->gmt,
		(int)af->af_Iter
	    );
	    if ((af->af_Fd = open(cdcache(path), O_RDWR|O_CREAT, 0644)) >= 0) {
		struct stat st;

		if (xflock(af->af_Fd, XLOCK_EX|XLOCK_NB) < 0 || 
		    (fstat(af->af_Fd, &st), st.st_nlink) == 0
		) {
		    close(af->af_Fd);
		    errno = 0;
		    af->af_Fd = -1;
		    ++af->af_Iter;	/* bump iteration */
		    continue;		/* try again 	  */
		}
		++AFNum;
		break;
	    }

	    /*
	     * The intermediate directory is missing, create a current
	     * directory (which expire better not touch)
	     */

	    sleep(1);
	    if (errno == ENOSPC || errno == EIO || errno == EMFILE || 
		errno == ENFILE
	    ) {
		break;
	    }

	    /*
	     * Try to find a more recent directory with tail recursion
	     */

	    if (H_EXP(h->exp) < 100) {
#ifdef SLOT_DEBUG
		syslog(LOG_INFO, "slot %d (%08x) not available", h->exp, h->gmt);
#endif
		h->exp = ((H_EXP(h->exp) + 25) & EXPF_MASK) |
			(h->exp & EXPF_FLAGS);
		if (H_EXP(h->exp) > 100)
		    h->exp = H_EXP(100) | (h->exp & EXPF_FLAGS);
		return(ArticleFile(h, pbpos, nglist, msgid));
	    }
#ifdef SLOT_DEBUG
	    syslog(LOG_INFO, "using last slot");
#endif

	    /*
	     * Attempt to mkdir the current (most recent) directory if it
	     * does not exist and try again.
	     */

	    *strrchr(path, '/') = 0;
	    if (mkdir(path, 0755) < 0) {
		if (errno == ENOSPC || errno == EIO)
		    break;
		/*
		 * directory exists, something odd.. bump iteration
		 */
		++af->af_Iter;
	    }
	} /* for */

	if (af->af_Fd >=0 ) {
	    af->af_FileSize = lseek(af->af_Fd, 0L, 2);
	    af->af_AppendOff = af->af_FileSize;
	}
    }

    /*
     * If we have a good descriptor, set the append position and
     * make sure the entry is at the beginning of the cache.
     */

    if ((rfd = af->af_Fd) >= 0) {
	*pbpos = (int)af->af_AppendOff;
	lseek(rfd, af->af_AppendOff, 0);

#ifdef SLOT_DEBUG
	syslog(LOG_INFO, "using slot %d (%08x) file %04x %s %s @ %d", H_EXP(h->exp), h->gmt, af->af_Iter, msgid, nglist, *pbpos);
#endif

	h->iter = af->af_Iter;
	if (af != &AFAry[0]) {
	    AFCache tmp = *af;
	    memmove(
		&AFAry[1],	/* dest 	*/
		&AFAry[0],	/* source 	*/
		(char *)af - (char *)&AFAry[0]
	    );
	    AFAry[0] = tmp;
	}
    }
    return(rfd);
}

void
ArticleFileCloseAll(void)
{
    int i;

    for (i = 0; i < MAXDIABLOFDCACHE; ++i) {
	if (AFAry[i].af_Fd >= 0)
	    ArticleFileClose(i);
    }
}

void
ArticleFileClose(int i)
{
    AFCache *aftmp = &AFAry[i];

    if (aftmp->af_AppendOff < aftmp->af_FileSize) {
#ifdef NOTDEF
	/*
	 * XXX debug code.  Do not ftruncate, fill with FF's
	 */
	char buf[4096];
	int l = aftmp->af_FileSize - aftmp->af_AppendOff;

	memset(buf, 0xFF, (l > sizeof(buf)) ? sizeof(buf) : l);
	lseek(aftmp->af_Fd, aftmp->af_AppendOff, 0);
	while (l > 0) {
	    int n = (l > sizeof(buf)) ? sizeof(buf) : l;
	    write(aftmp->af_Fd, buf, n);
	    l -= n;
	}
#endif
	ftruncate(aftmp->af_Fd, aftmp->af_AppendOff);
    }
    close(aftmp->af_Fd);
    bzero(aftmp, sizeof(AFCache));
    aftmp->af_Fd = -1;
}

void
ArticleFileTrunc(int artFd, off_t bpos)
{
    AFCache *af =  &AFAry[0];

    if (af->af_Fd != artFd) {
	syslog(LOG_ERR, "internal artFd mismatch %d/%d", artFd, af->af_Fd);
	return;
    }
    if (bpos >= 0)
	af->af_AppendOff = bpos;
}

void
ArticleFileSetSize(int artFd)
{
    AFCache *af =  &AFAry[0];
    off_t bpos = lseek(artFd, 0L, 1);

    if (af->af_Fd != artFd) {
	syslog(LOG_ERR, "internal artFd mismatch %d/%d", artFd, af->af_Fd);
	return;
    }
    if (bpos > af->af_FileSize)
	af->af_FileSize = bpos;
    af->af_AppendOff = bpos;
}

